#include "session_process.h"

namespace NDrivematics {

bool CreateSession(TContext::TPtr context, TUserPermissions::TPtr permissions, const NDrive::IServer& server, NDrive::TEntitySession& tx, const TString& settingName) {
    const auto& carId = context->GetCarId();
    const auto& settings = server.GetSettings();

    const auto offerName = settings.GetValue<TString>(settingName + ".offer_name");
    if (!offerName) {
        context->AddErrorReport("CreateSession", "empty offer Name");
        return false;
    }

    IOffer::TPtr offer;
    {
        TFullPricesContext ctx;

        auto standardOffer = MakeAtomicShared<TStandartOffer>(ctx);
        standardOffer->SetChargableAccounts({ "card" });
        standardOffer->SetName(*offerName);
        standardOffer->SetObjectId(carId);
        standardOffer->SetUserId(permissions->GetUserId());
        offer = std::move(standardOffer);
    }
    context->SetSessionId(offer->GetOfferId());
    TChargableTag::TBookOptions bookOptions;
    bookOptions.MultiRent = true;
    const auto booked = TChargableTag::Book(offer, *permissions, server, tx, bookOptions);
    R_ENSURE(booked, {}, "cannot start session", tx);
    const bool evolveToRiding = settings.GetValue<bool>(settingName + ".evolve_to_riding").GetOrElse(false);
    if (evolveToRiding) {
        if (!TChargableTag::DirectEvolve(booked, TChargableTag::Acceptance, *permissions, server, tx, nullptr)) {
            context->AddErrorReport("CreateSession", "cannot evolve to " + TChargableTag::Acceptance);
            return false;
        }
        if (!TChargableTag::DirectEvolve(booked.GetTagId(), TChargableTag::Riding, *permissions, server, tx, nullptr)) {
            context->AddErrorReport("CreateSession", "cannot evolve to " + TChargableTag::Riding);
            return false;
        }
    }
    tx.Committed().Subscribe([carId, offer](const NThreading::TFuture<void>& c) {
        if (c.HasValue()) {
            NDrive::TEventLog::Log("StartSession", NJson::TMapBuilder
                ("id", carId)
                ("offer", NJson::ToJson(offer))
            );
        }
    });
    return true;
}

bool SwitchToImpl(const TString& tagTo, TContext::TPtr context, TUserPermissions::TPtr permissions, const NDrive::IServer& server, NDrive::TEntitySession& tx) {
    if (!TChargableTag::DirectEvolve(context->GetChargableTag(), tagTo, *permissions, server, tx, nullptr)) {
        context->AddErrorReport("SwitchToImpl", "cannot evolve to " + tagTo);
        return false;
    }
    tx.Committed().Subscribe([tagTo, carId = context->GetCarId(), tagId = context->GetChargableTag().GetTagId()](const NThreading::TFuture<void>& c) {
        if (c.HasValue()) {
            NDrive::TEventLog::Log("SwitchToImpl", NJson::TMapBuilder
                ("id", carId)
                ("tag_id", tagId)
                ("tag_name", tagTo)
            );
        }
    });
    return true;
}

bool SwitchToRiding(TContext::TPtr context, TUserPermissions::TPtr permissions, const NDrive::IServer& server, NDrive::TEntitySession& tx) {
    return SwitchToImpl(TChargableTag::Riding, context, permissions, server, tx);
}

bool SwitchToParking(TContext::TPtr context, TUserPermissions::TPtr permissions, const NDrive::IServer& server, NDrive::TEntitySession& tx) {
    return SwitchToImpl(TChargableTag::Parking, context, permissions, server, tx);
}

bool SwitchToReservation(TContext::TPtr context, TUserPermissions::TPtr permissions, const NDrive::IServer& server, NDrive::TEntitySession& tx) {
    return SwitchToImpl(TChargableTag::Reservation, context, permissions, server, tx);
}

bool IsOldTimestamp(TContext::TPtr context, TInstant timestamp, const TDeviceTagsManager& tagManager, NDrive::TEntitySession& tx) {
    auto queryOptions = IEntityTagsManager::TQueryOptions(1, true);
    queryOptions.SetOrderBy({
        "object_id",
        "history_event_id",
    });
    const auto depth = TDuration::Hours(1);
    const auto threshold = timestamp - depth;
    const auto optionalTagEvents = tagManager.GetEventsByTag(context->GetChargableTag().GetTagId(), tx, {}, threshold, queryOptions);
    R_ENSURE(optionalTagEvents, {}, "cannot GetEventsByTag: " << context->GetChargableTag().GetTagId(), tx);
    context->MutableLastUpdatedAt() = !optionalTagEvents->empty()
        ? optionalTagEvents->front().GetHistoryInstant()
        : threshold
    ;
    if (context->GetLastUpdatedAt() >= timestamp) {
        return true;
    }
    return false;
}

bool ProcessTaxiOffer(TContext::TPtr context, TUserPermissions::TPtr permissions,
                      const NDrive::IServer& server, NDrive::TEntitySession& tx,
                      const TDBTag& restoredTag, bool isTaxiSession, bool engineOn) {
    if (!isTaxiSession && (restoredTag->GetName() == TChargableTag::Riding || restoredTag->GetName() == TChargableTag::Parking)) {
        R_ENSURE(
            SwitchToReservation(context, permissions, server, tx),
            {},
            context->GetErrorReport(),
            tx
        );
        return true;
    }
    if (isTaxiSession) {
        if (!engineOn && restoredTag->GetName() == TChargableTag::Riding) {
            R_ENSURE(
                SwitchToParking(context, permissions, server, tx),
                {},
                context->GetErrorReport(),
                tx
            );
            return true;
        }
        if (engineOn && restoredTag->GetName() == TChargableTag::Parking) {
            R_ENSURE(
                SwitchToRiding(context, permissions, server, tx),
                {},
                context->GetErrorReport(),
                tx
            );
            return true;
        }
    }
    return false;
}

TDBTag GetRestoredTag(const TString& carId, TConstArrayRef<TString> tagNames, const TDeviceTagsManager& tagManager, NDrive::TEntitySession& tx) {
    const auto restoredTags = tagManager.RestoreTags(TVector<TString>{ carId }, std::move(tagNames), tx);
    R_ENSURE(restoredTags, {}, "cannot RestoreTags", tx);
    R_ENSURE(restoredTags->size() == 1, {}, "restoredTags->size() != 1", tx);

    return restoredTags->front();
}

bool CloseSignalqSession(const ISession::TPtr session, const TString& userId, const NDrive::IServer& server, bool dryRun) {
    const auto api = Yensured(server.GetDriveAPI());
    const auto& tagManager = api->GetTagsManager().GetDeviceTags();

    const auto permissions = api->GetUserPermissions(userId);
    R_ENSURE(permissions, HTTP_INTERNAL_SERVER_ERROR, "cannot GetUserPermissions for " << userId);
    const auto& sessionId = session->GetSessionId();
    const auto& carId = session->GetObjectId();
    auto tx = tagManager.BuildSession(/*readOnly=*/false, /*repeatableRead=*/true);
    tx.SetOriginatorId(userId);

    if (!TChargableTag::DirectEvolve(session, TChargableTag::Reservation, *permissions, server, tx, nullptr)) {
        ERROR_LOG << "Cannot evolve to " << TChargableTag::Reservation << " on carId: " << carId << Endl;
        return false;
    }
    if (dryRun) {
        Y_UNUSED(tx.Rollback());
        NOTICE_LOG << "Dry run 'CloseSession' for sessionId: " << sessionId << " with carId: " << carId << " and userId: " << userId << Endl;
        return true;
    }
    tx.Committed().Subscribe([sessionId, carId, userId](const NThreading::TFuture<void>& c) {
        if (c.HasValue()) {
            NDrive::TEventLog::Log("FinishSignalqSession", NJson::TMapBuilder
                ("session_id", sessionId)
                ("car_id", carId)
                ("user_id", userId)
            );
        }
    });
    if (!tx.Commit()) {
        ERROR_LOG << "Cannot Commit for " << sessionId << ": " << tx.GetStringReport() << Endl;
        return false;
    }
    INFO_LOG << "Evolved to 'Reservation' for carId: " << carId << " with session: "  << sessionId << Endl;
    return true;
}

TUserPermissions::TPtr GetLinkedOperatorPermissions(const TString& carId, const NDrive::IServer& server, NDrive::TEntitySession& session) {
    const auto driveApi = Yensured(server.GetDriveAPI());
    TVector<TString> performerTagNames = SplitString(
        server.GetSettings().GetValueDef<TString>("telematics.settings.linked_operator_tag_name", ""),
        ","
    );
    if (!performerTagNames) {
        return nullptr;
    }

    TVector<TDBTag> tags;
    {
        const auto& tagManager = driveApi->GetTagsManager().GetDeviceTags();
        auto optionalTags = tagManager.RestoreEntityTags(carId, performerTagNames, session);
        R_ENSURE(optionalTags, HTTP_INTERNAL_SERVER_ERROR, "failed to restore performer tag");
        tags = std::move(*optionalTags);
    }

    auto evlog = NDrive::GetThreadEventLogger();
    if (tags.empty()) {
        if (evlog) {
            evlog->AddEvent(NJson::TMapBuilder
                ("event", "NoPerformerTag")
                ("source", "NDrivematics::GetActualPerformer")
                ("id", carId)
            );
        }
        return nullptr;
    }

    TString performerId = "";
    for (const auto& tag : tags) {
        if (!performerId) {
            performerId = tag->GetPerformer();
            continue;
        }

        if (performerId != tag->GetPerformer()) {
            if (evlog) {
                evlog->AddEvent(NJson::TMapBuilder
                    ("event", "MultiplePerformers")
                    ("source", "NDrivematics::GetActualPerformer")
                    ("id", carId)
                );
            }
            return nullptr;
        }
    }

    if (!performerId) {
        return nullptr;
    }

    return driveApi->GetUserPermissions(performerId, TUserPermissionsFeatures());
}

}

