#include "signalq_resolution_set.h"

#include <drive/backend/processors/signalq/processor.h>
#include <drive/backend/signalq/signals/helpers.h>
#include <drive/backend/signalq/signals/tag.h>

#include <drive/library/cpp/taxi/signalq_drivematics_api/client.h>
#include <drive/library/cpp/taxi/signalq_drivematics_api/definitions.h>


void TSignalqDrivematicsTraceTagResolutionSet::ProcessServiceRequest(TJsonReport::TGuard& g,
                                                  TUserPermissions::TPtr permissions,
                                                  const NJson::TJsonValue& requestData){
    const auto resolution = GetValue<NDrive::NSignalq::EEventResolution>(requestData, "resolution", true).GetRef();
    if (resolution == NDrive::NSignalq::EEventResolution::SupportApprove) {
        ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::Tag, TSignalqEventTraceTag::TypeName);
    }

    const auto& traceTagManager = DriveApi->GetTagsManager().GetTraceTags();
    const auto tagId = GetUUID(requestData, "tag_id", true);

    auto tx = BuildTx<NSQL::Writable>();
    auto signalqOptionalTag = traceTagManager.RestoreTag(tagId, tx);
    R_ENSURE(signalqOptionalTag, HTTP_NOT_FOUND, "cannot find tag_id " << tagId, tx);
    auto tag = signalqOptionalTag->MutableTagAs<TSignalqEventTraceTag>();
    R_ENSURE(tag, HTTP_BAD_REQUEST, "tag type is not supported for resolutions, tag_id = " << tagId, tx);

    NDrive::NSignalq::ProcessTagResolution(
        *signalqOptionalTag,
        resolution,
        Server,
        permissions,
        tx
    );

    if (resolution != NDrive::NSignalq::EEventResolution::SupportApprove) {
        const auto signalqDrivematicsApiClient = Server->GetTaxiSignalqDrivematicsApiClient();
        R_ENSURE(signalqDrivematicsApiClient, HTTP_INTERNAL_SERVER_ERROR, "No signalq-drivematics-api client", tx);

        NSignalq::TV1EventResolutionRequest resuolutionRequest;
        resuolutionRequest.SetResolution(std::move(resolution));
        resuolutionRequest.SetSerialNumber(std::move(tag->MutableSerialNumber()));
        resuolutionRequest.SetEventId(std::move(tag->MutableEvent().MutableId()));
        const auto responseFuture = signalqDrivematicsApiClient->SetEventResolutionInFleet(resuolutionRequest);
        responseFuture.Wait(TDuration::Seconds(1));
        if (!responseFuture.HasValue() || responseFuture.HasException()) {
            g.AddReportElement("error", NThreading::GetExceptionInfo(responseFuture));
            g.SetCode(HTTP_INTERNAL_SERVER_ERROR);
            return;
        }
    }

    R_ENSURE(tx.Commit(), HTTP_INTERNAL_SERVER_ERROR, "cannot Commit", tx);
    g.SetCode(HTTP_OK);
}
