#include "signalq_status_history.h"

#include "common.h"

#include <drive/backend/cars/car.h>
#include <drive/backend/cars/hardware.h>

#include <drive/library/cpp/taxi/signalq_drivematics_api/client.h>
#include <drive/library/cpp/taxi/signalq_drivematics_api/definitions.h>

namespace NDrivematics {
    void TSignalqStatusHistoryProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
        const auto carId = GetString(requestData, "car_id", true);
        ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::ObserveStructure, TAdministrativeAction::EEntity::Car, carId);
        R_ENSURE(DriveApi->GetCarsData()->FetchInfo(carId).size() != 0, HTTP_FORBIDDEN, "no such car");

        TCarGenericAttachment attachment;
        R_ENSURE(DriveApi->GetCarAttachmentAssignments().TryGetAttachmentOfType(carId, EDocumentAttachmentType::CarSignalDevice, attachment), HTTP_BAD_REQUEST, "no SignalQ attached");

        const auto carSignalDevicePtr = dynamic_cast<const TCarSignalDevice*>(attachment.Get());
        R_ENSURE(carSignalDevicePtr, HTTP_INTERNAL_SERVER_ERROR, "Cannot cast carId " + carId + "'s attachment " + attachment.GetId() + " to TCarSignalDevice");

        const auto signalqDrivematicsApiClient = Server->GetTaxiSignalqDrivematicsApiClient();
        R_ENSURE(signalqDrivematicsApiClient, {}, "No signalq-drivematics-api client");

        const auto& serialNumber = carSignalDevicePtr->GetSerialNumber();
        NDrive::NSignalq::TV1DeviceStatusHistoryIntervalsRequest deviceStatusHistoryParams;
        deviceStatusHistoryParams.SetSerialNumber(serialNumber);
        if (requestData.Has("cursor")) {
            deviceStatusHistoryParams.SetCursor(GetString(requestData, "cursor", true));
        }
        if (requestData.Has("period")) {
            const auto periodFrom = GetTimestamp(requestData["period"], "from", true);
            R_ENSURE(periodFrom, HTTP_BAD_REQUEST, "Period is missing required field 'from'");
            const auto periodTo = GetTimestamp(requestData["period"], "to", true);
            R_ENSURE(periodTo, HTTP_BAD_REQUEST, "Period is missing required field 'to'");
            NDrive::NSignalq::TPeriod period;
            period.SetFrom(*periodFrom);
            period.SetTo(*periodTo);
            deviceStatusHistoryParams.SetPeriod(period);
        }
        if (requestData.Has("statuses_intervals_limit")) {
            deviceStatusHistoryParams.SetStatusesIntervalsLimit(GetValue<ui32>(requestData, "statuses_intervals_limit", true));
        }
        const auto deviceStatusHistoryResponseJson = signalqDrivematicsApiClient->GetSignalqStatusHistoryIntervalsJson(deviceStatusHistoryParams);
        const auto start = Now();
	    const auto deviceStatusHistoryFuture = deviceStatusHistoryResponseJson
            .Apply([start, report = g.GetReport()](const NThreading::TFuture<NJson::TJsonValue>& response) {
                TJsonReport::TGuard g(report);
                if (response.HasValue()) {
                    const auto finish = Now();
                    const auto duration = finish - start;
                    INFO_LOG << "Fetched device's status history, duration: " << duration << Endl;
                    auto responseJson = response.GetValue();
                    g.SetExternalReport(std::move(responseJson));
                    g.SetCode(HTTP_OK);
                } else {
                    auto code = HTTP_INTERNAL_SERVER_ERROR;
                    auto jsonReport = NThreading::GetExceptionInfo(response);
                    try {
                        const auto value = response.GetValue();
                    } catch (const NDrive::NSignalq::TBadRequestException&) {
                        code = HTTP_BAD_REQUEST;
                    }
                    g.SetExternalReport(std::move(jsonReport));
                    g.SetCode(code);
                }
            }
        );
        deviceStatusHistoryFuture.Wait();
    }
}
