#include "taxi_companies.h"
#include "common.h"

#include <library/cpp/http/misc/httpcodes.h>

#include <drive/backend/data/leasing/leasing.h>
#include <drive/backend/data/telematics.h>
#include <drive/backend/tags/tags_manager.h>

namespace {
    auto GetPageInfo(const TMap<TString, TSet<TString>>& taxiCompaniesStats,
                    ui64 pageSize, ui64 pageNumber) {
        const ui64 taxiCompaniesCount = taxiCompaniesStats.size();
        ui64 leftBound = 0;
        ui64 rightBound = taxiCompaniesCount;
        bool canGetMorePages = false;
        if (pageSize) {
            leftBound = Min(pageSize * (pageNumber - 1), taxiCompaniesCount);
            rightBound = Min(pageSize * pageNumber, taxiCompaniesCount);
            canGetMorePages = rightBound < taxiCompaniesCount;
        }
        auto pageBegin = taxiCompaniesStats.begin();
        std::advance(pageBegin, leftBound);
        auto pageEnd = taxiCompaniesStats.begin();
        std::advance(pageEnd, rightBound);
        return std::make_tuple(pageBegin, pageEnd, canGetMorePages);
    }

    NJson::TJsonValue BuildParksReport(TMap<TString, TSet<TString>>::const_iterator begin,
                                        TMap<TString, TSet<TString>>::const_iterator end,
                                        ui32 daysCount, TMaybe<ui32> daySince, TMaybe<ui32> dayUntil,
                                        TMaybe<ui32> oldDaySince, TMaybe<ui32> oldDayUntil,
                                        ui32 noSHThreshold,
                                        const IDriveTagsManager& tagsManager,
                                        const TTaggedObjectsSnapshot& objectsSnapshot,
                                        bool reportScores
                                        ) {
        NJson::TJsonValue parks;
        for (auto it = begin; it != end; ++it) {
            auto description = std::dynamic_pointer_cast<const NDrivematics::TTaxiCompanyTag::TDescription>(tagsManager.GetTagsMeta().GetDescriptionByName(it->first));
            Y_ENSURE(description);
            NJson::TJsonValue value;
            value["name"] = description->GetTaxiCompanyName();
            value["city"] = description->GetCity();
            value["park_id"] = it->first;
            {
                auto& allCarsStatsJson = value["all_cars"];
                if (reportScores) {
                    allCarsStatsJson["score"] = NJson::TMapBuilder("value", description->GetScore())
                                                            ("type", ToString(NDrivematics::ELeasingStatsType::Numeric));
                }
                {
                    auto allCarsStats = NDrivematics::CalculateStats(it->second, objectsSnapshot, daySince, dayUntil, daysCount, noSHThreshold, NDrivematics::TLeasingStatsTag::CalculateTaxiCompanyStats, [](const TTaggedDevice& /* taggedDevice */) {
                        return true;
                    });
                    ForEach(allCarsStats.GetFields(), [&allCarsStatsJson] (auto&& field) {
                        if (!(field.GetTraits() & NDrivematics::ReportInTaxiCompanies)) {
                            return;
                        }
                        allCarsStatsJson[field.GetName()] = NJson::TMapBuilder("value", field.Value)
                                                                        ("type", ToString(field.GetStatsType()));
                    });
                }
                if (daySince && dayUntil && oldDaySince && oldDayUntil) {
                    auto oldAllCarsStats = NDrivematics::CalculateStats(it->second, objectsSnapshot, oldDaySince, oldDayUntil, daysCount, noSHThreshold, NDrivematics::TLeasingStatsTag::CalculateTaxiCompanyStats, [](const TTaggedDevice& /* taggedDevice */) {
                        return true;
                    });
                    ForEach(oldAllCarsStats.GetFields(), [&allCarsStatsJson] (auto&& field) {
                        if (!(field.GetTraits() & NDrivematics::ReportInTaxiCompanies)) {
                            return;
                        }
                        allCarsStatsJson[field.GetName()]["old_value"] = field.Value;
                    });
                }
            }
            {
                auto& telematicsCarsStatsJson = value["telematics_cars"];
                if (reportScores) {
                    telematicsCarsStatsJson["score"] = NJson::TMapBuilder("value", description->GetTelematicsCarsScore())
                                                            ("type", ToString(NDrivematics::ELeasingStatsType::Numeric));
                }
                {
                    auto telematicsCarsStats = NDrivematics::CalculateStats(it->second, objectsSnapshot, daySince, dayUntil, daysCount, noSHThreshold, NDrivematics::TLeasingStatsTag::CalculateTaxiCompanyStats, [](const TTaggedDevice& taggedDevice) {
                        return taggedDevice.GetTag(NDrivematics::THasTelematicsTag::TypeName);
                    });
                    ForEach(telematicsCarsStats.GetFields(), [&telematicsCarsStatsJson](auto&& field) {
                        if (!(field.GetTraits() & NDrivematics::ReportInTaxiCompanies)) {
                            return;
                        }
                        telematicsCarsStatsJson[field.GetName()] = NJson::TMapBuilder("value", field.Value)
                                                                        ("type", ToString(field.GetStatsType()));
                    });
                }
                if (daySince && dayUntil && oldDaySince && oldDayUntil) {
                    auto oldTelematicsCarsStats = NDrivematics::CalculateStats(it->second, objectsSnapshot, oldDaySince, oldDayUntil, daysCount, noSHThreshold, NDrivematics::TLeasingStatsTag::CalculateTaxiCompanyStats, [](const TTaggedDevice& taggedDevice) {
                        return taggedDevice.GetTag(NDrivematics::THasTelematicsTag::TypeName);
                    });
                    ForEach(oldTelematicsCarsStats.GetFields(), [&telematicsCarsStatsJson](auto&& field) {
                        if (!(field.GetTraits() & NDrivematics::ReportInTaxiCompanies)) {
                            return;
                        }
                        telematicsCarsStatsJson[field.GetName()]["old_value"] = field.Value;
                    });
                }
            }
            parks.AppendValue(std::move(value));
        }
        return parks;
    }
}

void TTaxiCompaniesListProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    const TCgiParameters& cgi = Context->GetCgiParameters();
    const auto defaultPageSize = GetHandlerSetting<ui32>("page_size").GetOrElse(0);
    const auto pageSize = GetValue<ui64>(cgi, "page_size", false).GetOrElse(defaultPageSize);
    const auto pageNumber = GetValue<ui64>(cgi, "page_number", false).GetOrElse(1);
    const auto daySince = GetValue<ui32>(requestData, "dates.since", false);
    const auto dayUntil = GetValue<ui32>(requestData, "dates.until", false);
    const auto oldDaySince = GetValue<ui32>(requestData, "old_dates.since", false);
    const auto oldDayUntil = GetValue<ui32>(requestData, "old_dates.until", false);
    auto session = BuildTx<NSQL::ReadOnly>();
    const auto& api = *Yensured(Server->GetDriveAPI());

    auto observableTagsNames = permissions->GetTagNamesByAction(TTagAction::ETagAction::Observe);
    auto taxiCompanyTagsNames = api.GetTagsManager().GetTagsMeta().GetRegisteredTagNames({NDrivematics::TTaxiCompanyTag::TypeName});
    auto observableTaxiCompanyTagsNames = MakeIntersection(observableTagsNames, taxiCompanyTagsNames);
    TVector<TDBTag> dbTags;

    R_ENSURE(
        api.GetTagsManager().GetDeviceTags().RestoreTagsRobust({}, std::cref(observableTaxiCompanyTagsNames), dbTags, session),
        {},
        "can not RestoreTags",
        session
    );

    TSet<TString> objectIds;
    TMap<TString, TString> carIdToTaxiCompany;
    for (const auto& dbTag : dbTags) {
        if (!dbTag.Is<NDrivematics::TTaxiCompanyTag>()) {
            continue;
        }
        objectIds.insert(dbTag.GetObjectId());
        carIdToTaxiCompany[dbTag.GetObjectId()] = dbTag->GetName();
    }

    TTaggedObjectsSnapshot objectsSnapshot;
    {
        auto eg = g.BuildEventGuard("restore_cars");
        R_ENSURE(api.GetTagsManager().GetDeviceTags().GetObjectsFromCacheByIds(objectIds, objectsSnapshot, TInstant::Zero()), HTTP_INTERNAL_SERVER_ERROR, "can't restore cars from cache");
    }


    TMap<TString, TSet<TString>> taxiCompanies;
    for (const auto& carId : objectIds) {
        taxiCompanies[carIdToTaxiCompany[carId]].insert(carId);
    }

    auto [pageBegin, pageEnd, canGetMorePages] = GetPageInfo(taxiCompanies, pageSize, pageNumber);
    auto daysCount = Server->GetSettings().GetValueDef<ui32>("leasing_cabinet.aggregation.days_count", 60);
    auto noSHThreshold = Server->GetSettings().GetValueDef<ui32>("leasing_cabinet.supply_hours_threshold", 320);
    bool reportScores = CheckAdmActions(permissions, TAdministrativeAction::EAction::ObserveStructure, TAdministrativeAction::EEntity::Tag, NDrivematics::TTaxiCompanyTag::TypeName);
    auto parks = BuildParksReport(pageBegin, pageEnd, daysCount, daySince, dayUntil, oldDaySince, oldDayUntil, noSHThreshold, api.GetTagsManager(), objectsSnapshot, reportScores);
    g.MutableReport().AddReportElement("parks", std::move(parks));

    if (pageSize) {
        g.MutableReport().AddReportElement("page_number", pageNumber);
        g.MutableReport().AddReportElement("can_get_more_pages", canGetMorePages);
    }
    g.SetCode(HTTP_OK);
}
