#include "processor.h"

#include <drive/backend/data/leasing/acl/acl.h>
#include <drive/backend/data/leasing/company.h>

#include <rtline/util/types/uuid.h>

namespace  {
    using namespace NDrivematics;

    NJson::TJsonValue GetReport(const TMap<TString, TArea>& areaContainer) {
        NJson::TJsonValue result = NJson::JSON_MAP;
        for (const auto& area : areaContainer) {
            NJson::TJsonValue areaReport = NJson::JSON_MAP;
            areaReport = area.second.SerializeToReport();
            R_ENSURE(areaReport.Has("area_details") && areaReport["area_details"].Has("group"), {}, "can't get group id");
            if (!result.Has(areaReport["area_details"]["group"].GetString())) {
                result
                    .InsertValue(areaReport["area_details"]["group"].GetString(), NJson::JSON_ARRAY)
                    .AppendValue(areaReport);
            } else {
                result[areaReport["area_details"]["group"].GetString()]
                    .AppendValue(areaReport);
            }
        }
        return result;
    }

    NJson::TJsonValue GetFullReport(const NDrivematics::TZone& zone, const TMap<TString, TArea>& areaContainer) {
        NJson::TJsonValue zoneReport = NJson::JSON_MAP;
        zoneReport = zone.SerializeToJson();

        if (zoneReport.Has("meta")) {
            if (NJson::TJsonValue areaIds; zoneReport.GetValueByPath("meta.area_ids", areaIds)) {
                if (NJson::TJsonValue::TArray arrayAreaIds; areaIds.GetArray(&arrayAreaIds)) {
                    for (auto& areaId : arrayAreaIds) {
                        if (auto areaIt = areaContainer.find(areaId.GetString()); areaIt != areaContainer.end()) {
                            zoneReport["meta"]["areas"].AppendValue(areaIt->second.SerializeToReport());
                        }
                    }
                }
            }
            zoneReport["meta"].EraseValue("area_ids");
        }
        return zoneReport;
    }

    void AddNewZone(NDrivematics::TZone::TPtr zone, NDrivematics::TACLTag::TDescription::TConstPtrImpl aclCompany, TUserPermissions::TPtr permissions, const NDrive::IServer& server, NDrive::TEntitySession& tx, bool force) {
        auto affiliatedCompanyTagDescription = TUserOrganizationAffiliationTag::GetAffiliatedCompanyTagDescription(permissions->GetUserId(), server, tx);

        zone->SetOwner(affiliatedCompanyTagDescription->GetName());
        TSet<TString> companyGroup = aclCompany->GetEntityObjects(TAdministrativeAction::EEntity::Zone, permissions, false);

        if (!force) {
            if (companyGroup.contains(zone->GetInternalId())) {
                Y_UNUSED(tx.Rollback());
                R_ENSURE(false, HTTP_BAD_REQUEST, "zone already exist: " << zone->GetName(), NDrive::MakeError("zone.already_exist"), tx);
            }
        }

        R_ENSURE(
            server.GetDriveAPI()->GetZoneDB()->UpsertObject(*zone, permissions->GetUserId(), tx)
            , HTTP_INTERNAL_SERVER_ERROR
            , "cannot add zone"
            , NDrive::MakeError("zone.add")
            , tx
        );
    }
}

void TZoneAddProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    auto tx = BuildTx<NSQL::Writable | NSQL::RepeatableRead>();

    R_ENSURE(
          requestData.Has("object") && requestData["object"].IsArray() && !requestData["object"].GetArray().empty()
        , HTTP_INTERNAL_SERVER_ERROR
        , "incorrect json, must be array 'object'"
        ,  NDrive::MakeError("zone.incorrect_request.add")
        , tx
    );
    auto force = GetValue<bool>(Context->GetCgiParameters(), "force", false).GetOrElse(false);

    auto acl = NDrivematics::TACLTag::GetACLTag(permissions, tx, *Server);
    auto aclCompany = acl->GetCompanyTagDescription(permissions->GetUserId(), *Server);

    auto zone = NDrivematics::TZone::Construct(requestData, permissions->GetUserId(), tx, *Server, force);
    if (requestData.Has("revision")) {
        zone->SetRevision(requestData["revision"].GetInteger());
    }
    AddNewZone(zone, aclCompany, permissions, *Server, tx, force);

    TMessagesCollector errors;
    if (!force) {
        R_ENSURE(
              aclCompany->AddEntityObject(GetEntityType(), zone->GetInternalId(), permissions, errors)
            , HTTP_FORBIDDEN
            , "cannot add object to company, " << errors.GetStringReport()
            , NDrive::MakeError("zone.acl.add")
            , tx
        );
    }
    R_ENSURE(
          Server->GetDriveAPI()->GetTagsManager().GetTagsMeta().RegisterTag(aclCompany, permissions->GetUserId(), tx)
        , {}
        , "cannot update acl tag"
        , NDrive::MakeError("acl.update")
        , tx
    );
    R_ENSURE(tx.Commit(), {}, "cannot commit", NDrive::MakeError("tx.save"), tx);

    g.SetCode(HTTP_OK);
}

void TZoneRemoveProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    auto tx = BuildTx<NSQL::Writable | NSQL::RepeatableRead>();

    auto zoneNameToRemove = MakeSet(GetStrings(requestData, "ids", true));
    R_ENSURE(zoneNameToRemove, HTTP_INTERNAL_SERVER_ERROR, "'ids' must be present", NDrive::MakeError("zone.incorrect_request.remove"), tx);

    auto acl = NDrivematics::TACLTag::GetACLTag(permissions, tx, *Server);
    auto aclCompany = acl->GetCompanyTagDescription(permissions->GetUserId(), *Server);

    NDrive::TZoneIds companyZoneIds = aclCompany->GetEntityObjects(GetEntityType(), permissions);
    NDrive::TZoneIds companyZoneIdsToRemove;
    auto action = [&zoneNameToRemove, &companyZoneIdsToRemove](const TZone& zone) -> void {
        if (zoneNameToRemove.contains(zone.GetName())) {
            companyZoneIdsToRemove.insert(zone.GetInternalId());
        }
    };
    R_ENSURE(Server->GetDriveAPI()->GetZoneDB()->ForObjectsList(action, Now(), &companyZoneIds), HTTP_INTERNAL_SERVER_ERROR, "cannot restore zones", NDrive::MakeError("zone.acl.object_not_found"), tx);
    R_ENSURE(companyZoneIdsToRemove.size() == zoneNameToRemove.size(), HTTP_INTERNAL_SERVER_ERROR, "cannot restore zones: " << companyZoneIdsToRemove.size() << " != " << zoneNameToRemove.size(), NDrive::MakeError("zone.acl.object_not_found"), tx);

    R_ENSURE(aclCompany->RemoveEntityObjects(GetEntityType(), companyZoneIdsToRemove, permissions, Server, tx), {}, "cannot remove objects", NDrive::MakeError("zone.acl.remove"), tx);
    R_ENSURE(
          Server->GetDriveAPI()->GetTagsManager().GetTagsMeta().RegisterTag(aclCompany, permissions->GetUserId(), tx)
        , {}
        , "cannot update acl tag"
        , NDrive::MakeError("zone.acl.update")
        , tx
    );
    R_ENSURE(tx.Commit(), {}, "cannot commit", NDrive::MakeError("tx.save"), tx);

    g.SetCode(HTTP_OK);
}

void TZoneGetProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    bool newVersion = GetValue<bool>(Context->GetCgiParameters(), "new", false).GetOrElse(false);
    auto dropAllow = GetValue<bool>(Context->GetCgiParameters(), "drop_allow", false);
    auto forMobile = GetValue<bool>(Context->GetCgiParameters(), "for_mobile", false);

    TSet<TString> zoneNamesRequest = MakeSet(GetStrings(Context->GetCgiParameters(), "id", false));

    auto tx = BuildTx<NSQL::ReadOnly>();

    auto acl = NDrivematics::TACLTag::GetACLTag(permissions, tx, *Server);
    auto aclCompany = acl->GetCompanyTagDescription(permissions->GetUserId(), *Server);

    TMaybe<TMap<TString, NDrivematics::TZone>> allZoneObjectsCompany;
    TMap<TString, NDrivematics::TZone> zoneObjectsCompany;
    TMap<TString, TArea> areaToReport;

    NDrive::TZoneIds companyZoneIds = aclCompany->GetEntityObjects(GetEntityType(), permissions, true);
    allZoneObjectsCompany = Server->GetDriveAPI()->GetZoneDB()->GetCachedObjectsMap(companyZoneIds);
    if (allZoneObjectsCompany || allZoneObjectsCompany->size() != companyZoneIds.size()) {
        for (const auto& companyId : companyZoneIds) {
            if (!allZoneObjectsCompany->contains(companyId)) {
                ALERT_LOG << "not all zones can be restored: " << companyId << Endl;
            }
        }
    }
    R_ENSURE(allZoneObjectsCompany || companyZoneIds.empty(), HTTP_INTERNAL_SERVER_ERROR, "cannot restore zones", NDrive::MakeError("zone.acl.object_not_found"), tx);

    ForEach(allZoneObjectsCompany->begin(), allZoneObjectsCompany->end(), [&zoneObjectsCompany, &zoneNamesRequest](const std::pair<TString, NDrivematics::TZone>& zone){
        if (zoneNamesRequest.empty() || zoneNamesRequest.contains(zone.second.GetName())) {
            zoneObjectsCompany.insert(zone);
        }
    });
    R_ENSURE(zoneNamesRequest.empty() || zoneObjectsCompany.size() == zoneNamesRequest.size(), HTTP_FORBIDDEN, "not all zone found: ", NDrive::MakeError("zone.acl.object_not_found"), tx);
    {
        auto action = [&areaToReport, &dropAllow](const TString& key, const TArea& entity) -> void {
            if (dropAllow) {
                if (entity.GetTags().contains("drop_allow") ^ *dropAllow) {
                    return;
                }
            }
            areaToReport[key] = entity;
        };
        for (auto&& [zoneId, zone] : zoneObjectsCompany) {
            if (!zone.IsEmpty()) {
                if (forMobile && zone.IsForMobile() ^ *forMobile) {
                    continue;
                }
                R_ENSURE(
                    Server->GetDriveAPI()->GetAreasDB()->ForObjectsMap(action, Context->GetRequestStartTime(), zone.GetAreaIdsPtr())
                    , {}
                    , "cannot get objects from cache"
                    , NDrive::MakeError("zone.area.object_not_found")
                    , tx
                );
            }
        }
    }

    NJson::TJsonValue result = NJson::JSON_MAP;
    if (newVersion) {
            for (const auto& [_, zone] : zoneObjectsCompany) {
                if (dropAllow || forMobile) {
                    auto fullReport = GetFullReport(zone, areaToReport);
                    if (fullReport["meta"].Has("areas")) {
                        result.InsertValue(zone.GetName(), std::move(fullReport));
                    }
                    continue;
                }
                result.InsertValue(zone.GetName(),  GetFullReport(zone, areaToReport));
            }
        g.MutableReport().AddReportElement("company_zones", std::move(result));
    } else {
        NJson::TJsonValue result = NJson::JSON_MAP;
        result.InsertValue("company_zones", GetReport(areaToReport));
        g.MutableReport().AddReportElement("objects", std::move(result));
    }
    g.SetCode(HTTP_OK);
}

void TZoneUpsertProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    auto tx = BuildTx<NSQL::Writable>();

    R_ENSURE(requestData.IsMap(), HTTP_INTERNAL_SERVER_ERROR, "incorrect json", NDrive::MakeError("zone.incorrect_request.update"), tx);

    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::Zone);

    auto acl = NDrivematics::TACLTag::GetACLTag(permissions, tx, *Server);
    auto aclCompany = acl->GetCompanyTagDescription(permissions->GetUserId(), *Server);

    TZone zoneUpsert;
    TMessagesCollector errors;
    R_ENSURE(
          TBaseDecoder::DeserializeFromJsonVerbose(zoneUpsert, requestData, errors)
        , HTTP_INTERNAL_SERVER_ERROR
        , "incorrect object json: " + errors.GetStringReport()
        , NDrive::MakeError("zone.deserialize")
        , tx
    );

    auto existedZoneOpt = Server->GetDriveAPI()->GetZoneDB()->GetCustomObject(zoneUpsert.GetInternalId());
    R_ENSURE(existedZoneOpt, {}, "error when restore zone", NDrive::MakeError("zone.restore"), tx);
    auto& existedZone = *existedZoneOpt;

    TVector<TArea> newAreas;
    TVector<TArea> oldAreas;
    NJson::TJsonValue areasJson;
    R_ENSURE(requestData.GetValueByPath("meta.areas", areasJson), {}, "error when get areas", NDrive::MakeError("zone.incorrect_request.update"), tx);
    if (NJson::TJsonValue::TArray arrayAreas; areasJson.GetArray(&arrayAreas)) {
        zoneUpsert.SetAreaIds(NDrive::TAreaIds{});
        for (auto& areaJson : arrayAreas) {
            TArea* area;
            if (!areaJson.Has("area_id")) {
                area = &newAreas.emplace_back(TArea());
                area->SetIdentifier(NUtil::CreateUUID());
            } else {
                zoneUpsert.MutableAreaIdsRef().insert(areaJson["area_id"].GetString());
                area = &oldAreas.emplace_back(TArea());
            }
            R_ENSURE(TBaseDecoder::DeserializeFromJson(*area, areaJson), {}, "incorrect object json", NDrive::MakeError("zone.incorrect_request.update"), tx);
            R_ENSURE(existedZone.GetName() == area->GetZoneName(), HTTP_BAD_REQUEST, "incorrect zone, all area objects must have one group", NDrive::MakeError("zone.incorrect_request.update"), tx);
        }
    }
    if (!existedZone.IsEmpty()) {
        NDrive::TAreaIds toRemove;
        for (auto areaIdIt = existedZone.GetAreaIdsRef().begin(); areaIdIt != existedZone.GetAreaIdsRef().end();) {
            if (!zoneUpsert.GetAreaIdsRef().contains(*areaIdIt)) {
                toRemove.insert(*areaIdIt);
                areaIdIt = existedZone.OptionalAreaIds()->erase(areaIdIt);
            } else {
                ++areaIdIt;
            }
        }
        R_ENSURE(Server->GetDriveAPI()->GetAreasDB()->RemoveObject(toRemove, permissions->GetUserId(), tx), {}, "cannot remove unused area", NDrive::MakeError("zone.area.remove"), tx);
    }
    if (!newAreas.empty()) {
        for (auto&& newArea : newAreas) {
            R_ENSURE(
                Server->GetDriveAPI()->GetAreasDB()->UpsertObject(newArea, permissions->GetUserId(), tx)
                , HTTP_INTERNAL_SERVER_ERROR
                , "cannot add areas"
                , NDrive::MakeError("zone.area.add")
                , tx
            );
            zoneUpsert.MutableAreaIds()->insert(newArea.GetInternalId());
        }
    }
    for (const auto& area : oldAreas) {
        R_ENSURE(Server->GetDriveAPI()->GetAreasDB()->UpsertObject(area, permissions->GetUserId(), tx), HTTP_INTERNAL_SERVER_ERROR, "cannot update area", NDrive::MakeError("zone.area.update"), tx);
    }

    R_ENSURE(Server->GetDriveAPI()->GetZoneDB()->UpsertObject(zoneUpsert, permissions->GetUserId(), tx), {}, "cannot update zone", NDrive::MakeError("zone.update"), tx);

    R_ENSURE(tx.Commit(), {}, "cannot commit", NDrive::MakeError("tx.save"), tx);
    g.SetCode(HTTP_OK);
}
