#include "processor.h"

#include <library/cpp/charset/wide.h>
#include <library/cpp/json/json_reader.h>

#include <util/generic/string.h>
#include <util/string/split.h>

namespace {

IRequestProcessorConfig::TFactory::TRegistrator<TLocalizationProposeProcessor::THandlerConfig> ProposeRegistrator(TLocalizationProposeProcessor::GetTypeName() + "_registrator");
IRequestProcessorConfig::TFactory::TRegistrator<TLocalizationConfirmProcessor::THandlerConfig> ConfirmRegistrator(TLocalizationConfirmProcessor::GetTypeName() + "_registrator");
IRequestProcessorConfig::TFactory::TRegistrator<TLocalizationRejectProcessor::THandlerConfig> RejectRegistrator(TLocalizationRejectProcessor::GetTypeName() + "_registrator");
IRequestProcessorConfig::TFactory::TRegistrator<TLocalizationPropositionsProcessor::THandlerConfig> ListRegistrator(TLocalizationPropositionsProcessor::GetTypeName() + "_registrator");

}

void TLocalizationInfoProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::Localization);
    auto localization = Server->GetLocalizationAs<NLocalization::TLocalizationDB>();
    R_ENSURE(localization, ConfigHttpStatus.ServiceUnavailable, "no localization info");
    const TVector<TString> ids = GetStrings(Context->GetCgiParameters(), "id", false);
    TVector<NLocalization::TResource> result;
    R_ENSURE(localization->GetInfo(MakeSet(ids), result, Context->GetRequestStartTime()), ConfigHttpStatus.UnknownErrorStatus, "cannot fetch data about localization");
    NJson::TJsonValue jsonResult = NJson::JSON_ARRAY;
    for (auto&& object : result) {
        jsonResult.AppendValue(object.SerializeToJson());
    }
    g.MutableReport().AddReportElement("resources", std::move(jsonResult));
    g.SetCode(HTTP_OK);
}

void TLocalizationUpsertProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::Localization);

    NLocalization::TResource resource;
    R_ENSURE(resource.DeserializeFromJson(requestData), ConfigHttpStatus.SyntaxErrorStatus, "incorrect format");

    auto session = BuildTx<NSQL::Writable>();
    if (!Server->GetLocalizationAs<NLocalization::TLocalizationDB>()->UpsertObject(resource, permissions->GetUserId(), session) || !session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }
    g.SetCode(HTTP_OK);
}

void TLocalizationRemoveProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Remove, TAdministrativeAction::EEntity::Localization);

    const TSet<TString> ids = MakeSet(GetStrings(requestData, "id", true));

    auto session = BuildTx<NSQL::Writable>();
    if (!Server->GetLocalizationAs<NLocalization::TLocalizationDB>()->RemoveObject(ids, permissions->GetUserId(), session) || !session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }
    g.SetCode(HTTP_OK);
}
