#include "processor.h"

#include <drive/backend/cars/car.h>
#include <drive/backend/major/client.h>

void TMaintenanceRemoveProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Remove, TAdministrativeAction::EEntity::Maintenance);
    const TString vin = GetString(Context->GetCgiParameters(), "vin");
    auto session = BuildTx<NSQL::Writable>();
    R_ENSURE(
        Server->GetDriveAPI()->GetMaintenanceDB().RemoveObjects({ vin }, permissions->GetUserId(), session),
        {},
        "RemoveMaintenanceInfoError",
        session
    );
    R_ENSURE(session.Commit(), {}, "CommitError", session);
    g.SetCode(HTTP_OK);
}

void TMaintenanceUpsertProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::Maintenance);
    TMaintenanceInfo info;
    ReqCheckCondition(info.DeserializeFromJson(requestData), ConfigHttpStatus.SyntaxErrorStatus, EDriveLocalizationCodes::SyntaxUserError);
    auto session = BuildTx<NSQL::Writable>();
    R_ENSURE(
        Server->GetDriveAPI()->GetMaintenanceDB().UpsertMaintenanceInfo(info, permissions->GetUserId(), /* force = */ false, session),
        {},
        "UpsertMaintenanceInfoError",
        session
    );
    R_ENSURE(session.Commit(), {}, "CommitError", session);
    g.SetCode(HTTP_OK);
}

void TMaintenanceInfoProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::Maintenance);
    const TString vin = GetString(Context->GetCgiParameters(), "vin", false);
    auto session = BuildTx<NSQL::ReadOnly>();
    auto info = vin
        ? Server->GetDriveAPI()->GetMaintenanceDB().GetObjects({ vin }, session)
        : Server->GetDriveAPI()->GetMaintenanceDB().GetObjects(session);
    if (!info) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }
    NJson::TJsonValue infoJson = NJson::JSON_ARRAY;
    for (auto&& i : *info) {
        infoJson.AppendValue(i.SerializeToJson());
    }
    g.MutableReport().AddReportElement("maintenance", std::move(infoJson));
    g.SetCode(HTTP_OK);
}

template<class TTypeContainer, class TEntityContainer>
NJson::TJsonValue GetJsonReport(const TTypeContainer& types, const TEntityContainer& entities, bool addDetails = true) {
    NJson::TJsonValue result;
    for (const auto& type : types) {
        NJson::TJsonValue typeJson;
        typeJson.InsertValue("type_info", type.ToJson());
        NJson::TJsonValue statusesArray;
        auto itStatuses = entities.find(type.GetId());
        if (itStatuses != entities.end()) {
            for (const auto& status : itStatuses->second) {
                NJson::TJsonValue statusJson;
                statusJson.InsertValue("status", status.first);
                if (status.second.size() == 1 && NJson::ToJson(status.second[0]).Has("count")) {
                    statusJson.InsertValue("count", NJson::ToJson(status.second[0])["count"]);
                } else {
                    statusJson.InsertValue("count", status.second.size());
                }
                if (addDetails) {
                    statusJson.InsertValue("entities", NJson::ToJson(status.second));
                } else {
                    statusJson.InsertValue("entities", NJson::JSON_NULL);
                }
                statusesArray.AppendValue(std::move(statusJson));
            }
        }
        typeJson.InsertValue("statuses", statusesArray);
        result.AppendValue(typeJson);
    }
    return result;
}

void TTyreServiceInfoProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::ObserveStructure, TAdministrativeAction::EEntity::Tyre);
    ReqCheckCondition(Server->GetDriveAPI()->HasMajorClient(), ConfigHttpStatus.UnknownErrorStatus, EDriveLocalizationCodes::InternalServerError);
    const TInstant since = GetTimestamp(Context->GetCgiParameters(), "since", TInstant::Zero());
    const TInstant until = GetTimestamp(Context->GetCgiParameters(), "until", TInstant::Max());
    NMajorClient::TTyreServiceInfoRequest::TTyreServiceInfo info;
    TMessagesCollector errors;
    ReqCheckCondition(Server->GetDriveAPI()->GetMajorClient().GetTyreServiceInfo(info, since, until, errors), ConfigHttpStatus.UnknownErrorStatus, errors.GetStringReport());

    TMap<ui64, TMap<TString, NMajorClient::TTyreServiceInfoRequest::TTyres>> tyresByType;
    TMap<ui64, TMap<TString, NMajorClient::TTyreServiceInfoRequest::TRims>> rimsByType;

    for (const auto& query : info.GetQueries()) {
        for (const auto& tyre : query.GetTyres()) {
            tyresByType[tyre.GetTypeId()][tyre.GetStatus()].emplace_back(std::move(tyre));
        }
        for (const auto& rim : query.GetRims()) {
            rimsByType[rim.GetTypeId()][rim.GetStatus()].emplace_back(std::move(rim));
        }
    }

    g.MutableReport().AddReportElement("tyres", GetJsonReport(info.GetTyreTypes(), tyresByType));
    g.MutableReport().AddReportElement("rims", GetJsonReport(info.GetRimTypes(), rimsByType));

    g.SetCode(HTTP_OK);
}

void TTyreStorageInfoProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::ObserveStructure, TAdministrativeAction::EEntity::Tyre);
    ReqCheckCondition(Server->GetDriveAPI()->HasMajorClient(), ConfigHttpStatus.UnknownErrorStatus, EDriveLocalizationCodes::InternalServerError);
    const TVector<NMajorClient::ETyreRegion> regions = GetValues<NMajorClient::ETyreRegion>(Context->GetCgiParameters(), "regions", true);
    const NMajorClient::ETyreStorageType storageType = GetValue<NMajorClient::ETyreStorageType>(Context->GetCgiParameters(), "storage", true).GetOrElse(NMajorClient::ETyreStorageType::Service);

    NMajorClient::TTyreStorageInfoRequest::TInfo info;
    TMessagesCollector errors;
    ReqCheckCondition(Server->GetDriveAPI()->GetMajorClient().GetTyreStorageInfo(regions, storageType, info, errors), ConfigHttpStatus.UnknownErrorStatus, errors.GetStringReport());

    for (size_t i = 0; i < regions.size(); ++i) {
        TMap<ui64, TMap<TString, NMajorClient::TTyreStorageInfoRequest::TTyres>> tyresByType;
        TMap<ui64, TMap<TString, NMajorClient::TTyreStorageInfoRequest::TRims>> rimsByType;

        for (const auto& tyre : info.GetStorageTyre()) {
            if (tyre.GetRegion() == regions[i]) {
                tyresByType[tyre.GetId()][tyre.GetStatus()].emplace_back(std::move(tyre));
            }
        }

        for (const auto& rim : info.GetStorageRim()) {
            if (rim.GetRegion() == regions[i]) {
                rimsByType[rim.GetId()][rim.GetStatus()].emplace_back(std::move(rim));
            }
        }
        NJson::TJsonValue regionResult;
        regionResult.InsertValue("tyres", GetJsonReport(info.GetTyreTypes(), tyresByType, false));
        regionResult.InsertValue("rims", GetJsonReport(info.GetRimTypes(), rimsByType, false));
        g.MutableReport().AddReportElement(ToString(regions[i]), std::move(regionResult));
    }
    g.SetCode(HTTP_OK);
}

void TLastTyreServiceProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::Tyre);
    ReqCheckCondition(Server->GetDriveAPI()->HasMajorClient(), ConfigHttpStatus.UnknownErrorStatus, EDriveLocalizationCodes::InternalServerError);
    const TVector<TString> ids = GetStrings(Context->GetCgiParameters(), "ids", true);

    auto gCars = Server->GetDriveAPI()->GetCarsData()->GetCachedOrFetch(ids);

    TMap<TString, TString> carVins;
    for (const auto& car : gCars) {
        if (!car.second.GetVin().empty()) {
            carVins.emplace(car.second.GetVin(), car.first);
        }
    }

    TVector<NMajorClient::TLastTyreServiceRequest::TInfo> infos;
    TMessagesCollector errors;
    ReqCheckCondition(Server->GetDriveAPI()->GetMajorClient().GetLastTyreService(MakeVector(NContainer::Keys(carVins)), infos, errors), ConfigHttpStatus.UnknownErrorStatus, errors.GetStringReport());

    NJson::TJsonValue result;
    for (const auto& info : infos) {
        auto it = carVins.find(info.GetVin());
        if (it != carVins.end()) {
            result.InsertValue(it->second, info.GetDate().Seconds());
        }
    }

    g.MutableReport().AddReportElement("result", std::move(result));
    g.SetCode(HTTP_OK);
}

void TTyreFilesProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::Tyre);
    ReqCheckCondition(Server->GetDriveAPI()->HasMajorClient(), ConfigHttpStatus.UnknownErrorStatus, EDriveLocalizationCodes::InternalServerError);
    const TString id = GetString(Context->GetCgiParameters(), "id", true);
    const NMajorClient::ETyreFileStatus status = GetValue<NMajorClient::ETyreFileStatus>(Context->GetCgiParameters(), "status", false).GetOrElse(NMajorClient::ETyreFileStatus::Undefined);

    auto gCars = Server->GetDriveAPI()->GetCarsData()->GetCached(id);
    auto carPtr = gCars.GetResultPtr(id);
    ReqCheckCondition(!!carPtr, ConfigHttpStatus.SyntaxErrorStatus, "incorrect car id");

    TVector<NMajorClient::TTyreFilesRequest::TFile> files;
    TMessagesCollector errors;
    ReqCheckCondition(Server->GetDriveAPI()->GetMajorClient().GetTyreFiles(carPtr->GetVin(), status, files, errors), ConfigHttpStatus.UnknownErrorStatus, errors.GetStringReport());

    NJson::TJsonValue result(NJson::JSON_ARRAY);
    for (const auto& info : files) {
        result.AppendValue(info.ToJson());
    }

    g.MutableReport().AddReportElement("files", std::move(result));
    g.SetCode(HTTP_OK);
}
