#include "processor.h"

#include <drive/backend/data/leasing/company.h>
#include <drive/backend/logging/events.h>
#include <drive/backend/report/json.h>
#include <drive/backend/support_center/ifaces.h>

#include <drive/library/cpp/image_transformation/recognize.h>
#include <drive/library/cpp/mds/client.h>

#include <util/generic/string.h>

#include <rtline/util/types/uuid.h>

namespace {
constexpr ui64 megabytesToBytes = 1024 * 1024;
}

void TMDSBucketInfoProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    R_ENSURE(DriveApi->HasMDSClient(), ConfigHttpStatus.UnknownErrorStatus, "No MDS configured available");

    const TString& bucketName = GetString(Context->GetCgiParameters(), "bucket", true);
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::ObserveStructure, TAdministrativeAction::EEntity::MDS, bucketName);
    auto bucket = DriveApi->GetMDSClient().GetBucket(bucketName);
    R_ENSURE(bucket, ConfigHttpStatus.UnknownErrorStatus, "Internal error: bucket doesn't exist");

    const TString& path = GetString(Context->GetCgiParameters(), "path", false);

    TVector<TS3Client::TBucketElement> keys;
    const ui32 code = bucket->GetKeys(path, keys);
    R_ENSURE(code == HTTP_OK, (IsHttpCode(code) ? code : ConfigHttpStatus.UnknownErrorStatus), "MDS server error");

    const TString& reportFormat = GetString(Context->GetCgiParameters(), "report", false);

    NJson::TJsonValue jsonResult;
    for (auto&& key : keys) {
        NJson::TJsonValue report;
        if (reportFormat == "objects") {
            report = key.GetReport();
        } else {
            report = bucket->GetTmpFilePath(key.Key);
        }
        jsonResult.AppendValue(std::move(report));
    }

    g.MutableReport().AddReportElement("keys", std::move(jsonResult));
    g.SetCode(HTTP_OK);
}

namespace {

    class TSimpleReportSubsciber {
    public:
        TSimpleReportSubsciber(IServerReportBuilder::TPtr&& report, NDrive::TEventLog::TState&& eventLogState)
            : Report(std::move(report))
            , EventLogState(std::move(eventLogState))
        {
        }

    void operator()(const NThreading::TFuture<NUtil::THttpReply>& r) {
        NDrive::TEventLog::TStateGuard stateGuard(EventLogState);
        TJsonReport::TGuard g(Report, HttpCodes::HTTP_INTERNAL_SERVER_ERROR);
        try {
            g.SetCode(r.GetValue().Code());
        } catch (const yexception& e) {
            g.MutableReport().AddReportElement("status", "error");
            g.MutableReport().AddReportElement("error", e.what());
        }
    }

    private:
        IServerReportBuilder::TPtr Report;
        NDrive::TEventLog::TState EventLogState;
    };

}

void TMDSUploadProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    R_ENSURE(DriveApi->HasMDSClient(), ConfigHttpStatus.UnknownErrorStatus, "No MDS configured available");

    const TString& bucketName = GetString(Context->GetCgiParameters(), "bucket", true);
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Add, TAdministrativeAction::EEntity::MDS, bucketName);
    auto bucket = DriveApi->GetMDSClient().GetBucket(bucketName);
    R_ENSURE(bucket, ConfigHttpStatus.UnknownErrorStatus, "Internal error: bucket doesn't exist");

    TString key = GetString(Context->GetCgiParameters(), "key", false);
    TString contentType(Context->GetRequestData().HeaderInOrEmpty("Content-Type"));
    TBlob file = Context->GetBuf();
    R_ENSURE(!file.Empty(), ConfigHttpStatus.SyntaxErrorStatus, "Blob is empty");

    if (Config.HasMaxFileSize()) {
        auto fileSizeBytes = file.size() * sizeof(TBlob::value_type);
        R_ENSURE(fileSizeBytes <= Config.GetMaxFileSizeRef() * megabytesToBytes, ConfigHttpStatus.UnknownErrorStatus, "File size exceed limit", NDrive::MakeError("mds.file_size_exceed_limit"));
    }

    if (Config.GetCheckIsImage()) {
        TMemoryInput postData(file.Data(), file.Size());
        auto postDataStr = postData.ReadAll();
        R_ENSURE(NImageTransformation::TRecognize::CheckFileIsImage(postDataStr), ConfigHttpStatus.UnknownErrorStatus, "Post data is not photo", NDrive::MakeError("mds.post_data_is_not_image"));
    }

    if (Config.GetIsExternalApi()) {
        auto tx = BuildTx<NSQL::ReadOnly>();
        auto organizationAffiliationTagPtr = NDrivematics::TUserOrganizationAffiliationTag::GetAffiliatedCompanyTagDescription(permissions->GetUserId(), *Server, tx);
        Y_ENSURE(organizationAffiliationTagPtr);
        if (!key) {
            key = organizationAffiliationTagPtr->GetName() + "/" + NUtil::CreateUUID();
        }
        g.MutableReport().AddReportElement("url", bucket->GetTmpFilePath(key));
    }

    R_ENSURE(!key.empty(), ConfigHttpStatus.SyntaxErrorStatus, "Key is empty");

    bucket->PutKey(key, TString(file.AsCharPtr(), file.Size()), contentType).Subscribe(TSimpleReportSubsciber(g.GetReport(), NDrive::TEventLog::CaptureState()));
    g.Release();
}

void TMDSRemoveProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    R_ENSURE(DriveApi->HasMDSClient(), ConfigHttpStatus.UnknownErrorStatus, "No MDS configured available");

    const TString& bucketName = GetString(Context->GetCgiParameters(), "bucket", true);
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Remove, TAdministrativeAction::EEntity::MDS, bucketName);
    auto bucket = DriveApi->GetMDSClient().GetBucket(bucketName);
    R_ENSURE(bucket, ConfigHttpStatus.UnknownErrorStatus, "Internal error: bucket " + bucketName + " doesn't exist");

    const TString& key = GetString(Context->GetCgiParameters(), "key", true);

    bucket->DeleteKey(key).Subscribe(TSimpleReportSubsciber(g.GetReport(), NDrive::TEventLog::CaptureState()));
    g.Release();
}

void TMDSProxyProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    const auto supportMds = Server->GetSupportCenterManager() ? Server->GetSupportCenterManager()->GetMDSClient() : nullptr;
    R_ENSURE(DriveApi->HasMDSClient() || supportMds, ConfigHttpStatus.UnknownErrorStatus, "No MDS configured available");

    const TString& bucketName = GetString(Context->GetCgiParameters(), "bucket", true);
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::MDS, bucketName);
    const auto driveBucket = DriveApi->HasMDSClient() ? DriveApi->GetMDSClient().GetBucket(bucketName) : nullptr;
    const TS3Client::TBucket* bucket = driveBucket
        ? driveBucket
        : (supportMds ? supportMds->GetBucket(bucketName) : nullptr);
    R_ENSURE(bucket, ConfigHttpStatus.UnknownErrorStatus, "Internal error: bucket doesn't exist");

    TString encryption = GetString(Context->GetCgiParameters(), "encryption", false);
    R_ENSURE(!encryption || bucket->HasEncryption(encryption), ConfigHttpStatus.SyntaxErrorStatus, "Internal error: encryption is not configured");
    if (encryption.empty()) {
        encryption = bucket->GetDefEncryptionName();
    }

    auto subscriber = [
        report = g.GetReport()
        , defType = GetHandlerSettingDef<TString>("content_type", "")
    ](const NThreading::TFuture<NS3::TFile>& file) {
        try {
            if (!report) {
                throw yexception() << "incorrect report value";
            }
            const auto& data = file.GetValue();
            report->Finish(
                HTTP_OK,
                data.GetContentType().empty() ? defType : data.GetContentType(),
                TBuffer(data.GetContent().Data(), data.GetContent().Size())
            );
        } catch (const yexception& e) {
            TJsonReport::TGuard g(report, HttpCodes::HTTP_INTERNAL_SERVER_ERROR);
            g.MutableReport().AddReportElement("status", "error");
            g.MutableReport().AddReportElement("error", e.what());
        }
    };

    const TString& key = GetString(Context->GetCgiParameters(), "key", true);
    if (encryption) {
        bucket->GetDecrypted(key, encryption).Subscribe(subscriber);
    } else {
        bucket->GetFile(key, GetHandlerSettingDef<TString>("content_type", "")).Subscribe(subscriber);
    }
    g.Release();
}
