#include "processor.h"

#include <drive/library/cpp/threading/future_cast.h>

namespace {
    NThreading::TFuture<NDrive::TMediaBillingClient::TOrderInfo> MakeFakeOrderInfo(ui64 orderId) {
        NDrive::TMediaBillingClient::TOrderInfo orderInfo;
        orderInfo.Id = orderId;
        orderInfo.Status = "success";
        orderInfo.Raw["fake"] = true;
        return NThreading::MakeFuture(std::move(orderInfo));
    }
}

void TGetOrderInfoProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    const auto dryRun = GetHandlerSetting<bool>("dry_run").GetOrElse(false);
    const auto orderId = GetValue<ui64>(requestData, "orderId").GetRef();
    const auto& uid = permissions ? permissions->GetUserFeatures().GetUid() : TString{};
    const auto& mediabilling = GetClient();

    auto orderInfo = dryRun ? MakeFakeOrderInfo(orderId) : mediabilling.GetOrderInfo(orderId, uid);
    {
        auto eg = g.BuildEventGuard("wait_order_info");
        R_ENSURE(orderInfo.Initialized(), ConfigHttpStatus.UnknownErrorStatus, "uninitialized async OrderInfo");
        R_ENSURE(orderInfo.Wait(Context->GetRequestDeadline()), ConfigHttpStatus.TimeoutStatus, "async OrderInfo timeout");
        R_ENSURE(orderInfo.HasValue(), ConfigHttpStatus.UnknownErrorStatus, "async OrderInfo exception: " << NThreading::GetExceptionMessage(orderInfo));
    }
    g.AddReportElement("result", NJson::ToJson(orderInfo));
    g.SetCode(HTTP_OK);
}

void TSubmitOrderProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    const auto& clientIp = GetClientIp();
    const auto& uid = permissions ? permissions->GetUserFeatures().GetUid() : TString{};
    const auto& mediabilling = GetClient();
    auto card = GetValue<TString>(requestData, "payment_method.card", false);
    auto dryRun = GetHandlerSetting<bool>("dry_run").GetOrElse(false);
    auto origin = GetHandlerSetting<TString>("mediabilling.origin").GetOrElse("drive");
    auto source = GetHandlerSetting<TString>("mediabilling.source").GetOrElse("drive");
    auto target = GetHandlerSetting<TString>("mediabilling.target").GetOrElse("drive");
    auto productId = GetHandlerSetting<TString>("mediabilling.product_id").GetOrElse({});

    g.AddEvent(NJson::TMapBuilder
        ("event", "input")
        ("card", NJson::ToJson(card))
        ("target", target)
        ("product_id", productId)
        ("uid", uid)
    );

    auto products = mediabilling.GetProducts(clientIp, uid, target);
    {
        auto eg = g.BuildEventGuard("wait_products");
        R_ENSURE(products.Initialized(), ConfigHttpStatus.UnknownErrorStatus, "uninitialized async Products");
        R_ENSURE(products.Wait(Context->GetRequestDeadline()), ConfigHttpStatus.TimeoutStatus, "async Products timeout");
        R_ENSURE(products.HasValue(), ConfigHttpStatus.UnknownErrorStatus, "async Products exception: " << NThreading::GetExceptionMessage(products));
    }
    g.AddReportElement("products", NJson::ToJson(products));

    TMaybe<NDrive::TMediaBillingClient::TProductDescription> product;
    if (productId) {
        for (auto&& description : products.GetValue()) {
            if (description.Product.Id == productId) {
                product = description;
                break;
            }
        }
    } else {
        auto descriptions = products.GetValue();
        std::sort(descriptions.begin(), descriptions.end());
        std::reverse(descriptions.begin(), descriptions.end());
        for (auto&& description : descriptions) {
            product = description;
            break;
        }
    }
    R_ENSURE(product, ConfigHttpStatus.EmptySetStatus, "product " << productId << " is not found");
    productId = product->Product.Id;
    g.AddReportElement("product", NJson::ToJson(product));

    NThreading::TFuture<NDrive::TMediaBillingClient::TOrderInfo> order;
    if (card && !dryRun) {
        order = mediabilling.SubmitOrder(productId, clientIp, uid, *card, source, origin);
        auto eg = g.BuildEventGuard("wait_order");
        R_ENSURE(order.Initialized(), ConfigHttpStatus.UnknownErrorStatus, "uninitialized async SubmitOrder");
        R_ENSURE(order.Wait(Context->GetRequestDeadline()), ConfigHttpStatus.TimeoutStatus, "async SubmitOrder timeout");
        R_ENSURE(order.HasValue(), ConfigHttpStatus.UnknownErrorStatus, "async SubmitOrder exception: " << NThreading::GetExceptionMessage(order));
    } else if (card && dryRun) {
        order = MakeFakeOrderInfo(Seconds());
    }
    g.AddReportElement("result", NJson::ToJson(order));
    g.SetCode(HTTP_OK);
}
