#include "processor.h"

#include <drive/backend/models/storage.h>
#include <drive/backend/offers/ranking/model.h>
#include <drive/backend/proto/models.pb.h>

#include <library/cpp/json/json_reader.h>

#include <util/string/cast.h>

void TAddOfferModelProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    Y_UNUSED(permissions);
    NDrive::TModelsStorage* storage = Server->GetModelsStorage();
    R_ENSURE(storage, ConfigHttpStatus.ServiceUnavailable, "ModelsStorage is not available");
    const TCgiParameters& cgi = Context->GetCgiParameters();
    const TBlob& post = Context->GetBuf();
    const TString& format = GetString(cgi, "format", false);
    const TString& separator = GetString(cgi, "separator", false);

    THolder<NDrive::IOfferModel> model;
    if (format == "proto") {
        NDrive::NProto::TOfferModel proto;
        R_ENSURE(proto.ParseFromArray(post.Data(), post.Size()), ConfigHttpStatus.UserErrorState, "cannot parse NDrive::NProto::TOfferModel from POST data");
        auto name = GetString(cgi, "name", false);
        if (name) {
            proto.SetName(name);
        }
        model = NDrive::IOfferModel::Construct(proto);
    } else {
        NJson::TJsonValue source;
        if (cgi.Has("meta")) {
            source = NJson::ReadJsonFastTree(cgi.Get("meta"));
            if (!post.Empty()) {
                R_ENSURE(!source.Has("data"), ConfigHttpStatus.SyntaxErrorStatus, "meta should not have data");
                TStringBuf data(post.AsCharPtr(), post.Size());
                if (separator) {
                    for (auto&& i : StringSplitter(data).SplitByString(separator)) {
                        source["data"].AppendValue(i.Token());
                    }
                } else {
                    source["data"] = ToString(data);
                }
            }
        } else {
            source = requestData;
        }
        model = NDrive::IOfferModel::Construct(source);
    }
    R_ENSURE(model, ConfigHttpStatus.SyntaxErrorStatus, "cannot construct model");
    R_ENSURE(storage->AddOfferModel(std::move(model)), ConfigHttpStatus.UnknownErrorStatus, "cannot save model");
    g.SetCode(HTTP_OK);
}

void TGetOfferModelProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    Y_UNUSED(permissions);
    Y_UNUSED(requestData);
    NDrive::TModelsStorage* storage = Server->GetModelsStorage();
    R_ENSURE(storage, ConfigHttpStatus.ServiceUnavailable, "ModelsStorage is not available");

    const TCgiParameters& cgi = Context->GetCgiParameters();
    auto format = GetString(cgi, "format", false);
    auto name = GetString(cgi, "name");
    auto model = storage->GetOfferModel(name);
    R_ENSURE(model, ConfigHttpStatus.EmptySetStatus, "model " << name << " is not found");
    if (format == "proto") {
        auto serialized = model->Serialize<TString>();
        auto buffer = TBuffer(serialized.data(), serialized.size());
        auto report = g.Release();
        report->Finish(HTTP_OK, "application/octet-stream", buffer);
    } else {
        auto result = model->Serialize<NJson::TJsonValue>();
        g.MutableReport().SetExternalReport(std::move(result));
        g.SetCode(HTTP_OK);
    }
}

void TListOfferModelProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    Y_UNUSED(permissions);
    NDrive::TModelsStorage* storage = Server->GetModelsStorage();
    R_ENSURE(storage, ConfigHttpStatus.ServiceUnavailable, "ModelsStorage is not available");

    NJson::TJsonValue models = NJson::JSON_ARRAY;
    for (auto&& i : storage->ListOfferModels()) {
        models.AppendValue(i);
    }
    g.MutableReport().AddReportElement("models", std::move(models));
    g.SetCode(HTTP_OK);
}

void TRemoveOfferModelProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    Y_UNUSED(permissions);
    NDrive::TModelsStorage* storage = Server->GetModelsStorage();
    R_ENSURE(storage, ConfigHttpStatus.ServiceUnavailable, "ModelsStorage is not available");

    auto name = GetString(requestData, "name", true);
    R_ENSURE(storage->RemoveOfferModel(name), ConfigHttpStatus.UnknownErrorStatus, "cannot remove model " << name);
    g.SetCode(HTTP_OK);
}
