#include "processor.h"

#include <drive/backend/data/transformation.h>
#include <drive/backend/data/user_tags.h>
#include <drive/backend/history_iterator/history_iterator.h>
#include <drive/backend/localization/localization.h>
#include <drive/backend/offers/actions/standart.h>
#include <drive/backend/offers/context.h>
#include <drive/backend/offers/manager.h>
#include <drive/backend/processors/user_app/sessions_context.h>
#include <drive/backend/processors/user_app/user_context.h>
#include <drive/backend/processors/warning_screen/processor.h>
#include <drive/backend/sessions/manager/billing.h>
#include <drive/library/cpp/mtapi/api.h>

namespace {

    void SetOfferPriceRange(NDrive::NMtApi::TOfferRangePriceEntity& range, ui32 price, TMaybe<ui32> blurRange) {
        if (!blurRange || *blurRange == 0) {
            range.Min = price;
            range.Max = price;
        } else {
            range.Min = (price / *blurRange) * *blurRange;
            range.Max = range.Min + *blurRange;
        }
    }

    NDrive::NMtApi::TOfferRangeEntity BuildOfferReport(
        ICommonOffer::TPtr offer,
        const TRolesManager& roles,
        const ILocalization* localization,
        ELocalization locale,
        TMaybe<ui32> priceBlurRange = {}
    ) {
        Y_ENSURE(offer);
        NDrive::NMtApi::TOfferRangeEntity report;
        report.Id = offer->GetOfferId();
        report.Name = offer->GetName();
        if (auto&& builderAction = roles.GetAction(offer->GetBehaviourConstructorId()); builderAction) {
            if (auto&& builder = builderAction->GetAs<TStandartOfferConstructor>(); builder) {
                report.Description = offer->FormDescriptionElement(builder->GetDetailedDescription(), locale, localization);
            }
        }
        if (auto&& standard = std::dynamic_pointer_cast<const TStandartOffer>(offer); standard) {
            report.HoldAmount = standard->GetDepositAmount();
            report.FreeWaitingTime = standard->GetFreeDuration(TChargableTag::Reservation, false);
            SetOfferPriceRange(report.Parking, standard->GetPublicDiscountedParking(), priceBlurRange);
            SetOfferPriceRange(report.Riding, standard->GetPublicDiscountedRiding(), priceBlurRange);
            SetOfferPriceRange(report.Inspection, standard->GetPublicDiscountedAcceptance(), priceBlurRange);
            SetOfferPriceRange(report.Waiting, standard->GetPublicDiscountedParking(), priceBlurRange);
            report.FreeInspectionTime = standard->GetFreeDuration(TChargableTag::Acceptance, false);
            report.ValidUntil = standard->GetDeadline();
        }
        return report;
    }

    NDrive::NMtApi::TLocationEntity BuildLocationReport(const NDrive::TLocation& location) {
        NDrive::NMtApi::TLocationEntity report;
        report.Lat = location.Latitude;
        report.Lon = location.Longitude;
        report.Address = location.Name;
        return report;
    }

    NDrive::NMtApi::TFuelCapacity BuildFuelCapacityReport(
        const TRTDeviceSnapshot& snapshot,
        const TDriveCarInfo& carData,
        const TDriveModelData& modelData
    ) {
        NDrive::NMtApi::TFuelCapacity report;
        if (auto&& level = snapshot.GetFuelLevel(); level) {
            report.Value = *level;
            report.Type = "percent";
        } else if (auto&& distance = snapshot.GetFuelDistance(); distance) {
            report.Value = *distance * 0.001;
            report.Type = "km";
        } else if (auto&& volume = snapshot.GetFuelVolume(); volume) {
            report.Value = *volume * 100.0 / NDrive::GetFuelTankVolume(&carData, &modelData);
            report.Type = "percent";
        }
        return report;
    }

    NDrive::NMtApi::TRideCarEntity BuildRideCarEntityReport(const TCarsFetcher::TOrderedCarInfo& carInfo) {
        NDrive::NMtApi::TRideCarEntity report;
        report.Id = carInfo.Info.GetIMEI();
        report.Number = carInfo.Info.GetNumber();
        report.Model = carInfo.Info.GetModel();
        const auto& snapshot = *Yensured(carInfo.Snapshot);
        if (auto&& location = snapshot.GetPreciseLocation(); location) {
            report.Position = BuildLocationReport(*location);
        }
        const auto& modelInfo = *Yensured(carInfo.ModelInfo);
        report.FuelCapacity = BuildFuelCapacityReport(snapshot, carInfo.Info, modelInfo);
        return report;
    }

    NDrive::NMtApi::TRideWaitingInfoEntity BuildRideWaitingInfoEntityReport(IOffer::TPtr offer) {
        Y_ENSURE(offer);
        NDrive::NMtApi::TRideWaitingInfoEntity report;
        if (auto&& standard = std::dynamic_pointer_cast<const TStandartOffer>(offer); standard) {
            report.PricePerMin = standard->GetPublicDiscountedParking();
            report.FreeUntil = standard->GetTimestamp() + standard->GetFreeDuration(TChargableTag::Reservation);
        }
        return report;
    }

    NDrive::NMtApi::TRideInspectionInfoEntity BuildRideInspectionInfoEntityReport(IOffer::TPtr offer) {
        Y_ENSURE(offer);
        NDrive::NMtApi::TRideInspectionInfoEntity report;
        if (auto&& standard = std::dynamic_pointer_cast<const TStandartOffer>(offer); standard) {
            report.PricePerMin = standard->GetPublicDiscountedAcceptance();
        }
        return report;
    }

    NDrive::NMtApi::TRideDrivingInfoEntity BuildRideDrivingInfoEntityReport(IOffer::TPtr offer) {
        Y_ENSURE(offer);
        NDrive::NMtApi::TRideDrivingInfoEntity report;
        if (auto&& standard = std::dynamic_pointer_cast<const TStandartOffer>(offer); standard) {
            report.PricePerMin = standard->GetPublicDiscountedRiding();
        }
        return report;
    }

    NDrive::NMtApi::TRideParkingInfoEntity BuildRideParkingInfoEntityReport(IOffer::TPtr offer) {
        Y_ENSURE(offer);
        NDrive::NMtApi::TRideParkingInfoEntity report;
        if (auto&& standard = std::dynamic_pointer_cast<const TStandartOffer>(offer); standard) {
            report.PricePerMin = standard->GetPublicDiscountedParking();
        }
        return report;
    }

    NDrive::NMtApi::TRideCreatedEntity BuildRideCreatedEntityReport(
        IOffer::TPtr offer,
        const TCarsFetcher::TOrderedCarInfo& carInfo
    ) {
        Y_ENSURE(offer);
        NDrive::NMtApi::TRideCreatedEntity report;
        report.Id = offer->GetOfferId();
        report.DeepLink = carInfo.Info.GetDeepLink();
        report.Car = BuildRideCarEntityReport(carInfo);
        report.WaitingInfo = BuildRideWaitingInfoEntityReport(offer);
        report.InspectionInfo = BuildRideInspectionInfoEntityReport(offer);
        const auto& snapshot = *Yensured(carInfo.Snapshot);
        if (auto&& location = snapshot.GetPreciseLocation(); location) {
            report.PointStart = BuildLocationReport(*location);
        }
        report.CreatedAt = offer->GetTimestamp();
        return report;
    }

    void FillRideReport(NDrive::NMtApi::TRideEntity& report, IOffer::TPtr offer) {
        Y_ENSURE(offer);
        report.WaitingInfo = BuildRideWaitingInfoEntityReport(offer);
        report.InspectionInfo = BuildRideInspectionInfoEntityReport(offer);
        report.DrivingInfo = BuildRideDrivingInfoEntityReport(offer);
        report.ParkingInfo = BuildRideParkingInfoEntityReport(offer);
    }

    void FillRideReport(NDrive::NMtApi::TRideEntity& report, const TVector<TCompiledLocalEvent>& events) {
        TMaybe<TCompiledLocalEvent> prevEvent;
        for (auto&& event : events) {
            if (prevEvent) {
                auto duration = event.GetInstant() - prevEvent->GetInstant();
                auto&& tag = prevEvent->GetTagName();
                if (tag == TChargableTag::Reservation) {
                    report.WaitingInfo.TotalDuration += duration;
                } else if (tag == TChargableTag::Acceptance) {
                    report.InspectionInfo.TotalDuration += duration;
                } else if (tag == TChargableTag::Riding) {
                    report.DrivingInfo.TotalDuration += duration;
                } else if (tag == TChargableTag::Parking) {
                    report.ParkingInfo.TotalDuration += duration;
                }
            }
            prevEvent = event;
        }
    }

    void FillRideReport(NDrive::NMtApi::TRideEntity& report, const TBill& bill) {
        for (auto&& record : bill.GetRecords()) {
            auto&& id = record.GetId();
            if (id == TChargableTag::Reservation) {
                report.WaitingInfo.TotalDuration = record.GetDuration();
                report.WaitingInfo.TotalPrice = record.GetCost();
            } else if (id == TChargableTag::Acceptance) {
                report.InspectionInfo.TotalDuration = record.GetDuration();
                report.InspectionInfo.TotalPrice = record.GetCost();
            } else if (id == TChargableTag::Riding) {
                report.DrivingInfo.TotalDuration = record.GetDuration();
                report.DrivingInfo.TotalPrice = record.GetCost();
            } else if (id == TChargableTag::Parking) {
                report.ParkingInfo.TotalDuration = record.GetDuration();
                report.ParkingInfo.TotalPrice = record.GetCost();
            }
        }
    }

    TString GetRideStatus(const TString& stage) {
        if (stage == TChargableTag::Reservation) {
            return NDrive::NMtApi::WaitingRide;
        } else if (stage == TChargableTag::Acceptance) {
            return NDrive::NMtApi::InspectionRide;
        } else if (stage == TChargableTag::Riding) {
            return NDrive::NMtApi::ActiveRide;
        } else if (stage == TChargableTag::Parking) {
            return NDrive::NMtApi::ParkingRide;
        }
        return "unknown";
    }

    TString GetRideStatus(const THistoryRideObject& ride) {
        if (ride.IsActive()) {
            return GetRideStatus(ride.GetStage());
        }
        if (auto&& compiledRide = ride.GetFullCompiledRiding(); compiledRide && !compiledRide->OptionalRidingDuration()) {
            return NDrive::NMtApi::CanceledRide;
        }
        return NDrive::NMtApi::FinishedRide;
    }

    NDrive::NMtApi::TRideEntity BuildRideReport(
        const TUserSessionsContext::TSessionContext& session,
        const TRolesManager& roles,
        const ILocalization* localization,
        ELocalization locale,
        const TCarsFetcher::TOrderedCarInfo& carInfo,
        const NDrive::IServer* server
    ) {
        NDrive::NMtApi::TRideEntity report;
        report.DeepLink = carInfo.Info.GetDeepLink();
        report.Car = BuildRideCarEntityReport(carInfo);
        if (auto offer = session.GetOffer()) {
            report.Id = offer->GetOfferId();
            report.Offer = BuildOfferReport(offer, roles, localization, locale);
            report.CreatedAt = offer->GetTimestamp();
            FillRideReport(report, offer);
        }
        if (auto compilation = session.GetBillingCompilation()) {
            FillRideReport(report, compilation->GetLocalEvents());
            FillRideReport(report, compilation->GetBill(locale, server, nullptr));
            report.SummaryInfo.TotalPrice = compilation->GetReportSumPrice();
            report.SummaryInfo.TotalDuration = compilation->GetSumDuration();
        }
        report.Status = NDrive::NMtApi::WaitingRide;
        if (auto eSession = session.GetEventsSession()) {
            if (auto lastEvent = eSession->GetLastEvent()) {
                report.Status = GetRideStatus(Yensured(*lastEvent)->GetName());
            }
        }
        return report;
    }

    NDrive::NMtApi::TRideEntity BuildRideReport(
        const THistoryRideObject& ride,
        const TRolesManager& roles,
        const ILocalization* localization,
        ELocalization locale,
        const TCarsFetcher::TOrderedCarInfo& carInfo
    ) {
        NDrive::NMtApi::TRideEntity report;
        report.DeepLink = carInfo.Info.GetDeepLink();
        report.Car = BuildRideCarEntityReport(carInfo);
        if (auto offer = ride.GetOffer()) {
            report.Id = offer->GetOfferId();
            report.Offer = BuildOfferReport(offer, roles, localization, locale);
            FillRideReport(report, offer);
        }
        if (auto compiledRide = ride.GetFullCompiledRiding()) {
            FillRideReport(report, compiledRide->GetLocalEvents());
            if (compiledRide->HasBill()) {
                FillRideReport(report, compiledRide->GetBillUnsafe());
            }
        }
        if (auto location = ride.GetStartLocation()) {
            report.PointStart = BuildLocationReport(*location);
        }
        if (auto location = ride.GetLastLocation()) {
            report.PointFinish = BuildLocationReport(*location);
        }
        report.SummaryInfo.TotalPrice = ride.GetSumPrice();
        report.SummaryInfo.TotalDuration = ride.GetLastTS() - ride.GetStartTS();
        report.Status = GetRideStatus(ride);
        report.CreatedAt = ride.GetStartTS();
        if (report.Status == NDrive::NMtApi::FinishedRide) {
            report.FinishedAt = ride.GetLastTS();
        } else if (report.Status == NDrive::NMtApi::CanceledRide) {
            report.CanceledAt = ride.GetLastTS();
        }
        return report;
    }

    NGeoJson::TFeature BuildAreaFeature(const TArea& area) {
        NGeoJson::TFeature feature;
        NGeoJson::TGeometry geometry;
        geometry.GetCoordinates() = area.GetCoords();
        feature.SetGeometry(std::move(geometry));
        return feature;
    }

    NDrive::NMtApi::TZoneEntity BuildZoneEntityReport(const TArea& area) {
        NDrive::NMtApi::TZoneEntity report;
        report.Name = area.GetTitle();
        report.GeoJson.GetFeatures().push_back(BuildAreaFeature(area));
        return report;
    }

    NDrive::NMtApi::TCarZonesEntity BuildCarZonesEntityReport(
        const NDrive::IServer& server,
        const TString& carId,
        const TConstAreasSnapshot::TPtr areaSnapshot,
        IOfferReport::TPtr offerReport
    ) {
        Y_ENSURE(areaSnapshot);
        NDrive::NMtApi::TCarZonesEntity report;
        for (auto&& i : Reversed(*areaSnapshot)) {
            auto context = TCarLocationContext::BuildByArea(carId, i.GetArea(), server);
            auto features = context.GetCarAreaFeatures(false, offerReport ? dynamic_cast<const IOffer*>(offerReport->GetOffer().Get()) : nullptr, &server);
            auto dropAbility = features.GetAllowDropDef(EDropAbility::NotAllow);
            if (dropAbility == EDropAbility::Allow) {
                auto zone = BuildZoneEntityReport(i.GetArea());
                report.AllowComplete.push_back(zone);
            } else if (dropAbility == EDropAbility::Deny || dropAbility == EDropAbility::DenyIncorrectData) {
                auto zone = BuildZoneEntityReport(i.GetArea());
                report.ForbiddenComplete.push_back(zone);
            } else if (features.GetAllowRiding()) {
                report.AvailableForDriving.GetFeatures().push_back(BuildAreaFeature(i.GetArea()));
            }
        }
        return report;
    }

    NDrive::NMtApi::TCarEntity BuildCarEntityReport(
        const NDrive::IServer& server,
        const TVector<IOfferReport::TPtr>& offers,
        const TCarsFetcher::TOrderedCarInfo& carInfo,
        const TRolesManager& roles,
        const TConstAreasSnapshot::TPtr areaSnapshot,
        const ILocalization* localization,
        ELocalization locale,
        TMaybe<ui32> priceBlurRange
    ) {
        NDrive::NMtApi::TCarEntity report;
        report.Id = carInfo.Info.GetIMEI();
        report.DeepLink = carInfo.Info.GetDeepLink();
        report.Number = carInfo.Info.GetNumber();
        report.Model = carInfo.Info.GetModel();
        report.Zones = BuildCarZonesEntityReport(server, carInfo.Info.GetId(), areaSnapshot, !offers.empty() ? offers.front() : nullptr);
        if (carInfo.Snapshot) {
            if (auto location = carInfo.Snapshot->GetPreciseLocation()) {
                report.Location = BuildLocationReport(*location);
            }
            if (carInfo.ModelInfo) {
                report.FuelCapacity = BuildFuelCapacityReport(*carInfo.Snapshot, carInfo.Info, *carInfo.ModelInfo);
            }
        }
        for (auto&& offerReport : offers) {
            auto&& offer = Yensured(offerReport)->GetOffer();
            report.Offers.push_back(BuildOfferReport(offer, roles, localization, locale, priceBlurRange));
        }
        return report;
    }

}

namespace NDrive {

    template<typename THandler, typename THandlerConfig>
    void TMtProcessor<THandler, THandlerConfig>::ProcessException(TJsonReport::TGuard& g, const TCodedException& exception) const {
        NMtApi::TErrorEntity report;
        report.ErrorCode = exception.GetErrorCode() ? exception.GetErrorCode() : NMtApi::OtherErrorCode;
        report.Message = exception.GetLocalizedMessage();
        const auto locale = TBase::GetLocale();
        report.DeepLink = TBase::GetHandlerLocalization("error.deeplink", TString{}, locale);
        g.SetCode(exception.GetCode());
        g.SetExternalReport(NJson::ToJson(report));
    }

    template<typename THandler, typename THandlerConfig>
    TString TMtProcessor<THandler, THandlerConfig>::GetOrigin() const {
        return "maas";
    }

    TString TMtTokenProcessor::GetTypeName() {
        return "mtapi/token";
    }

    void TMtTokenProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
        auto tx = BuildTx<NSQL::ReadOnly>();
        auto fetchUserData = Server->GetDriveAPI()->GetUsersData()->FetchInfo(permissions->GetUserId(), tx);
        ReqCheckCondition(!!fetchUserData, HTTP_BAD_REQUEST, NMtApi::OtherErrorCode);
        auto userData = fetchUserData.GetResultPtr(permissions->GetUserId());
        ReqCheckCondition(!!userData, HTTP_BAD_REQUEST, NMtApi::OtherErrorCode);
        ReqCheckCondition(userData->GetStatus() != UserStatusBlocked, HTTP_BAD_REQUEST, NMtApi::BlockedErrorCode);
        ReqCheckCondition(userData->GetStatus() == UserStatusActive, HTTP_BAD_REQUEST, NMtApi::RegistrationNotCompletedErrorCode);
        NMtApi::TTokenEntity report;
        // Our API returns empty report.
        g.SetCode(HTTP_OK);
        g.SetExternalReport(NJson::ToJson(report));
    };

    TString TMtBookOfferProcessor::GetTypeName() {
        return "mtapi/book_offer";
    }

    void TMtBookOfferProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
        const auto locale = GetLocale();
        NMtApi::TRideDto data;
        ReqCheckCondition(NJson::TryFromJson(requestData, data), HTTP_BAD_REQUEST, NMtApi::OtherErrorCode);
        auto tx = BuildTx<NSQL::Writable | NSQL::RepeatableRead>();
        ReqCheckCondition(TUserProblemTag::EnsureNotBlocked(permissions->GetUserId(), *Server, tx), HTTP_FORBIDDEN, NMtApi::BlockedErrorCode);
        auto fetchUserData = Server->GetDriveAPI()->GetUsersData()->FetchInfo(permissions->GetUserId(), tx);
        ReqCheckCondition(!!fetchUserData, HTTP_BAD_REQUEST, NMtApi::OtherErrorCode);
        auto userData = fetchUserData.GetResultPtr(permissions->GetUserId());
        ReqCheckCondition(!!userData, HTTP_BAD_REQUEST, NMtApi::OtherErrorCode);
        const bool skipEmailVerification = permissions->GetSetting<bool>("user.skip_email_verification", false);
        ReqCheckCondition(skipEmailVerification || !userData || !!userData->GetEmail(), HTTP_FORBIDDEN, NMtApi::RegistrationNotCompletedErrorCode);
        auto asyncOffer = Server->GetOffersStorage()->RestoreOffer(data.OfferId, permissions->GetUserId(), tx);
        ReqCheckCondition(asyncOffer.Initialized(), HTTP_INTERNAL_SERVER_ERROR, NMtApi::OtherErrorCode);
        ReqCheckCondition(asyncOffer.Wait(Context->GetRequestDeadline()), HTTP_GATEWAY_TIME_OUT, NMtApi::OtherErrorCode);
        auto baseOffer = asyncOffer.GetValue();
        ReqCheckCondition(!!baseOffer, HTTP_INTERNAL_SERVER_ERROR, NMtApi::OtherErrorCode);
        ReqCheckCondition(CheckUserCards(permissions, baseOffer), HTTP_FORBIDDEN, NMtApi::OtherErrorCode);
        auto offer = std::dynamic_pointer_cast<IOffer>(baseOffer);
        ReqCheckCondition(!!offer, HTTP_INTERNAL_SERVER_ERROR, NMtApi::OtherErrorCode);
        ReqCheckCondition(offer->GetOrigin() == GetOrigin(), HTTP_NOT_FOUND, NMtApi::OtherErrorCode);
        const auto carId = offer->GetObjectId();
        TCarsFetcher fetcher(*Server, NDeviceReport::ReportAll);
        fetcher.SetCheckVisibility(false);
        fetcher.SetIsRealtime(true);
        fetcher.SetLocale(locale);
        ReqCheckCondition(!!fetcher.FetchData(permissions, {carId}), HTTP_INTERNAL_SERVER_ERROR, NMtApi::OtherErrorCode);
        auto carInfo = fetcher.GetOrderedCarInfo(carId);
        ReqCheckCondition(!!carInfo, HTTP_INTERNAL_SERVER_ERROR, NMtApi::OtherErrorCode);
        TChargableTag::TBookOptions bookOptions;
        bookOptions.Futures = false;
        bookOptions.CheckBlocked = false;
        auto booked = TChargableTag::Book(offer, *permissions, *Server, tx, bookOptions);
        ReqCheckCondition(!!booked, HTTP_INTERNAL_SERVER_ERROR, NMtApi::OtherErrorCode);
        if (!tx.Commit()) {
            tx.DoExceptionOnFail(ConfigHttpStatus);
        }
        NMtApi::TRideCreatedEntity report = BuildRideCreatedEntityReport(offer, *carInfo);
        g.SetCode(HTTP_OK);
        g.SetExternalReport(NJson::ToJson(report));
    };

    TString TMtCurrentRideProcessor::GetTypeName() {
        return "mtapi/current_ride";
    }

    void TMtCurrentRideProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
        const auto locale = GetLocale();
        const TInstant requestTimestamp = Context->GetRequestStartTime();
        NDriveSession::TReportTraits reportTraits = 0;
        NDeviceReport::TReportTraits deviceReportTraits = 0;
        TUserCurrentContext userCurrentContext(Server, permissions, reportTraits, GetOrigin(), {});
        TUserSessionsContext userSessionsContext(Server, permissions, locale, deviceReportTraits);
        {
            auto tx = BuildTx<NSQL::ReadOnly>();
            if (!userCurrentContext.Initialize(tx, g.MutableReport())) {
                tx.DoExceptionOnFail(ConfigHttpStatus);
            }
            if (!userSessionsContext.Initialize(tx, g.MutableReport(), requestTimestamp, true, "", &userCurrentContext)) {
                tx.DoExceptionOnFail(ConfigHttpStatus);
            }
        }
        ReqCheckCondition(!userSessionsContext.GetSessions().empty(), HTTP_NOT_FOUND, NMtApi::OtherErrorCode);
        auto&& session = userSessionsContext.GetSessions().front();
        auto offer = Yensured(session.GetOffer());
        NMtApi::TActiveRideInfoEntity report;
        if (offer->GetOrigin() != GetOrigin()) {
            report.Status = "serviceRide";
            report.DeepLink = TBase::GetHandlerLocalization("serviceRide.deeplink", TString{}, locale);
        } else {
            report.Status = "active";
        }
        const auto carId = offer->GetObjectId();
        TCarsFetcher fetcher(*Server, NDeviceReport::ReportAll);
        fetcher.SetCheckVisibility(false);
        fetcher.SetIsRealtime(true);
        fetcher.SetLocale(locale);
        ReqCheckCondition(!!fetcher.FetchData(permissions, {carId}), HTTP_INTERNAL_SERVER_ERROR, NMtApi::OtherErrorCode);
        auto carInfo = fetcher.GetOrderedCarInfo(carId);
        ReqCheckCondition(!!carInfo, HTTP_INTERNAL_SERVER_ERROR, NMtApi::OtherErrorCode);
        const auto& rolesManager = *Yensured(Server->GetDriveAPI()->GetRolesManager());
        report.Ride = BuildRideReport(
            session,
            rolesManager,
            Server->GetLocalization(),
            locale,
            *carInfo,
            Server
        );
        g.SetCode(HTTP_OK);
        g.SetExternalReport(NJson::ToJson(report));
    }

    TString TMtRideProcessor::GetTypeName() {
        return "mtapi/ride";
    }

    void TMtRideProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
        const auto& cgi = Context->GetCgiParameters();
        const auto locale = GetLocale();
        const auto sessionId = GetString(cgi, "session_id", true);
        ReqCheckCondition(!!sessionId, HTTP_BAD_REQUEST, NMtApi::OtherErrorCode);
        TString userId = Yensured(permissions)->GetUserId();
        THistoryRidesContext context(*Server, TInstant::Zero(), false, false);
        auto tx = BuildTx<NSQL::ReadOnly>();
        auto ydbTx = BuildYdbTx<NSQL::ReadOnly | NSQL::Deferred>("mt_ride");
        ReqCheckCondition(
            context.InitializeSession(sessionId, tx, ydbTx),
            HTTP_INTERNAL_SERVER_ERROR,
            "history_iterator.initialization.error",
            tx.GetStringReport()
        );
        bool hasMore = false;
        TVector<THistoryRideObject> sessions = context.GetSessions(TInstant::Max(), 1, &hasMore, false);
        ReqCheckCondition(!sessions.empty(), HTTP_NOT_FOUND, NMtApi::OtherErrorCode);
        auto&& session = sessions.front();
        ReqCheckCondition(session.GetUserId() == userId, HTTP_NOT_FOUND, NMtApi::OtherErrorCode);
        auto offer = Yensured(session.GetOffer());
        const auto carId = offer->GetObjectId();
        TCarsFetcher fetcher(*Server, NDeviceReport::ReportAll);
        fetcher.SetCheckVisibility(false);
        fetcher.SetIsRealtime(true);
        fetcher.SetLocale(locale);
        ReqCheckCondition(!!fetcher.FetchData(permissions, {carId}), HTTP_INTERNAL_SERVER_ERROR, NMtApi::OtherErrorCode);
        auto carInfo = fetcher.GetOrderedCarInfo(carId);
        ReqCheckCondition(!!carInfo, HTTP_INTERNAL_SERVER_ERROR, NMtApi::OtherErrorCode);
        const auto& rolesManager = *Yensured(Server->GetDriveAPI()->GetRolesManager());
        NMtApi::TRideEntity report = BuildRideReport(
            session,
            rolesManager,
            Server->GetLocalization(),
            locale,
            *carInfo
        );
        g.SetCode(HTTP_OK);
        g.SetExternalReport(NJson::ToJson(report));
    }

    TString TMtCancelRideProcessor::GetTypeName() {
        return "mtapi/cancel_ride";
    }

    void TMtCancelRideProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
        const auto& cgi = Context->GetCgiParameters();
        const auto locale = GetLocale();
        const auto action = GetString(cgi, "action", true);
        ReqCheckCondition(action == "request" || action == "confirm", HTTP_NOT_FOUND, NMtApi::OtherErrorCode);
        const auto sessionId = GetString(cgi, "session_id", true);
        ReqCheckCondition(!!sessionId, HTTP_BAD_REQUEST, NMtApi::OtherErrorCode);
        TString userId = permissions->GetUserId();
        auto tx = BuildTx<NSQL::Writable | NSQL::RepeatableRead>();
        const auto& sessionManager = DriveApi->GetSessionManager();
        auto optionalSession = sessionManager.GetSession(sessionId, tx);
        ReqCheckCondition(!!optionalSession, HTTP_INTERNAL_SERVER_ERROR, NMtApi::OtherErrorCode, tx.GetStringReport());
        auto session = *optionalSession;
        ReqCheckCondition(!!session, HTTP_NOT_FOUND, NMtApi::OtherErrorCode, "unable to get session");
        ReqCheckCondition(session->Compile(), HTTP_INTERNAL_SERVER_ERROR, NMtApi::OtherErrorCode, "unable to compile session");
        ReqCheckCondition(session->GetUserId() == userId, HTTP_NOT_FOUND, NMtApi::OtherErrorCode);
        auto lastEvent = session->GetLastEvent();
        ReqCheckCondition(!!lastEvent, HTTP_INTERNAL_SERVER_ERROR, NMtApi::OtherErrorCode, "empty session");
        const auto& tagsManager = Server->GetDriveAPI()->GetTagsManager();
        const auto& deviceTagsManager = tagsManager.GetDeviceTags();
        const auto& tagsMeta = tagsManager.GetTagsMeta();
        TVector<TDBTag> tags;
        ReqCheckCondition(
            !!deviceTagsManager.RestoreTags({ lastEvent->GetTagId() }, tags, tx),
            HTTP_INTERNAL_SERVER_ERROR,
            NMtApi::OtherErrorCode,
            "cannot RestoreTags: " + tx.GetStringReport()
        );
        ReqCheckCondition(tags.size() != 0, HTTP_INTERNAL_SERVER_ERROR, NMtApi::OtherErrorCode, "cannot restore tags");
        ReqCheckCondition(tags.size() < 2, HTTP_INTERNAL_SERVER_ERROR, NMtApi::OtherErrorCode, "too many restored tags");
        auto&& tagCurrent = tags.front();
        ReqCheckCondition(tagCurrent->GetName() != TTransformationTag::TypeName, HTTP_FORBIDDEN, NMtApi::OtherErrorCode, "car is in transformation");
        const auto carId = tagCurrent.GetObjectId();
        TCarsFetcher fetcher(*Server, NDeviceReport::ReportAll);
        fetcher.SetCheckVisibility(false);
        fetcher.SetIsRealtime(true);
        fetcher.SetLocale(locale);
        ReqCheckCondition(!!fetcher.FetchData(permissions, {carId}), HTTP_INTERNAL_SERVER_ERROR, NMtApi::OtherErrorCode);
        auto carInfo = fetcher.GetOrderedCarInfo(carId);
        ReqCheckCondition(!!carInfo, HTTP_INTERNAL_SERVER_ERROR, NMtApi::OtherErrorCode);
        TString confirmMessage;
        {
            auto checkers = NDrive::GetWarningScreenCheckers("finish_rental", locale, permissions);
            for (auto&& checker : checkers) {
                auto landing = Yensured(checker)->Call(*Server, Context, permissions);
                if (landing && landing->IsDefined()) {
                    confirmMessage = GetHandlerLocalization("drop_session." + checker->GetName() + ".message", "", locale);
                    if (confirmMessage) {
                        break;
                    }
                }
            }
        }
        if (action == "request") {
            if (!confirmMessage) {
                g.SetCode(HTTP_METHOD_NOT_ALLOWED);
                g.SetExternalReport(NJson::JSON_MAP);
                return;
            }
            NMtApi::TRideCancelConfirmEntity report;
            report.Id = sessionId;
            report.Message = confirmMessage;
            g.SetCode(HTTP_OK);
            g.SetExternalReport(NJson::ToJson(report));
            return;
        }
        NMtApi::TRideCancelConfirmDto data;
        ReqCheckCondition(NJson::TryFromJson(requestData, data), HTTP_BAD_REQUEST, NMtApi::OtherErrorCode);
        if (confirmMessage) {
            ReqCheckCondition(data.ConfirmationRequestId == sessionId, HTTP_UNAUTHORIZED, "invalid_confirmation_request_id");
            // Mark that user accepted conditions.
            tx.MutableRequestContext().SetUserChoice("accept");
        }
        ITag::TPtr newTag = tagsMeta.CreateTag(TChargableTag::Reservation);
        ReqCheckCondition(!!newTag, HTTP_INTERNAL_SERVER_ERROR, NMtApi::OtherErrorCode, "cannot create tag " + TChargableTag::Reservation);
        auto self = Self();
        auto timeout = GetDuration(cgi, "timeout", Config.GetDefaultTimeout());
        auto taskTimeout = timeout * Config.GetTaskTimeoutFraction();
        auto rollbackTimeout = std::max(timeout * Config.GetKfTimeoutRollbackByRobot() * Config.GetTaskTimeoutFraction(), TDuration::Minutes(1));
        const EEvolutionMode eMode = EEvolutionMode::Default;
        ProcessEvolveTag(self, g, tx, tagCurrent, newTag, permissions, locale, false, {}, taskTimeout, rollbackTimeout, eMode, &deviceTagsManager, ConfigHttpStatus, Server);
        const auto& rolesManager = *Yensured(Server->GetDriveAPI()->GetRolesManager());
        THistoryRideObject ride(session, {}, {}, {}, Server);
        NMtApi::TRideEntity report = BuildRideReport(
            ride,
            rolesManager,
            Server->GetLocalization(),
            locale,
            *carInfo
        );
        g.SetCode(HTTP_OK);
        g.SetExternalReport(NJson::ToJson(report));
    }

    TString TMtCarCardProcessor::GetTypeName() {
        return "mtapi/car_card";
    }

    void TMtCarCardProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
        const auto& cgi = Context->GetCgiParameters();
        const auto locale = GetLocale();
        const auto imei = GetString(cgi, "imei", true);
        ReqCheckCondition(!!imei, HTTP_BAD_REQUEST, NMtApi::OtherErrorCode);
        const auto carId = Yensured(Server->GetDriveAPI())->GetCarIdByIMEI(imei);
        ReqCheckCondition(!!carId, HTTP_NOT_FOUND, NMtApi::OtherErrorCode);
        TCarsFetcher fetcher(*Server, NDeviceReport::ReportAll);
        fetcher.SetCheckVisibility(false);
        fetcher.SetIsRealtime(true);
        fetcher.SetLocale(locale);
        ReqCheckCondition(!!fetcher.FetchData(permissions, {carId}), HTTP_INTERNAL_SERVER_ERROR, NMtApi::OtherErrorCode);
        auto carInfo = fetcher.GetOrderedCarInfo(carId);
        ReqCheckCondition(!!carInfo, HTTP_INTERNAL_SERVER_ERROR, NMtApi::OtherErrorCode);
        TUserOfferContext uoc(Server, permissions, nullptr);
        uoc.SetFilterAccounts(permissions->GetSetting<bool>(Server->GetSettings(), "billing.filter_accounts", true));
        uoc.SetAccountName("card");
        uoc.SetLocale(locale);
        uoc.SetOrigin(GetOrigin());
        ReqCheckCondition(uoc.Prefetch(), HTTP_INTERNAL_SERVER_ERROR, NMtApi::OtherErrorCode);
        NDrive::TEntitySession tx = BuildTx<NSQL::Writable | NSQL::Deferred>();
        TOffersBuildingContext context(Server);
        context.SetUserHistoryContext(uoc);
        context.SetCarId(carId);
        context.Prefetch();
        TSet<TString> offerBuilderTypes;
        {
            auto typesString = permissions->GetSetting<TString>(Server->GetSettings(), "maas.offer_builder_types", TStandartOfferConstructor::GetTypeStatic());
            StringSplitter(typesString).Split(',').SkipEmpty().Collect(&offerBuilderTypes);
        }
        TSet<TString> offerTypes;
        {
            auto typesString = permissions->GetSetting<TString>(Server->GetSettings(), "maas.offer_types", TStandartOffer::GetTypeNameStatic());
            StringSplitter(typesString).Split(',').SkipEmpty().Collect(&offerTypes);
        }
        TVector<IOfferReport::TPtr> allOffers;
        for (auto&& userAction : permissions->GetOfferBuilders()) {
            const IOfferBuilderAction* offerBuilder = dynamic_cast<const IOfferBuilderAction*>(userAction.Get());
            if (!offerBuilder) {
                continue;
            }
            // Ignore default for carsharing offers.
            if (offerBuilder->GetOrigins().empty()) {
                continue;
            }
            if (!offerBuilderTypes.contains(offerBuilder->GetType())) {
                continue;
            }
            TVector<IOfferReport::TPtr> offers;
            auto result = offerBuilder->BuildOffers(*permissions, permissions->GetOfferCorrections(), offers, context, Server, tx);
            R_ENSURE(result != EOfferCorrectorResult::Problems, {}, "cannot build offers for builder " << offerBuilder->GetName(), tx);
            for (auto&& offerReport : offers) {
                auto offer = Yensured(offerReport)->GetOffer();
                if (!offerTypes.contains(Yensured(offer)->GetTypeName())) {
                    continue;
                }
                allOffers.push_back(std::move(offerReport));
            }
        }
        if (!allOffers.empty()) {
            TEventsGuard eg(g.MutableReport(), "store_offers");
            auto asyncOffers = Server->GetOffersStorage()->StoreOffers(allOffers, tx);
            ReqCheckCondition(asyncOffers.Initialized(), HTTP_INTERNAL_SERVER_ERROR, NMtApi::OtherErrorCode);
        }
        const auto& rolesManager = *Yensured(Server->GetDriveAPI()->GetRolesManager());
        const auto& areaDb = DriveApi->GetAreasDB();
        auto areaSnapshot = Yensured(areaDb)->GetSnapshot();
        auto priceBlurRange = permissions->GetSetting<ui32>(Server->GetSettings(), "maas.offer_price_blur_range");
        NMtApi::TCarEntity report = BuildCarEntityReport(
            *Server,
            allOffers,
            *carInfo,
            rolesManager,
            areaSnapshot,
            Server->GetLocalization(),
            locale,
            priceBlurRange
        );
        ReqCheckCondition(tx.Commit(), HTTP_INTERNAL_SERVER_ERROR, NMtApi::OtherErrorCode, tx.GetStringReport());
        g.SetCode(HTTP_OK);
        g.SetExternalReport(NJson::ToJson(report));
    }

    TString TMtZonesProcessor::GetTypeName() {
        return "mtapi/zones";
    }

    // This method returns stub empty report.
    //
    // Zones are returned in car card report.
    void TMtZonesProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr /*permissions*/, const NJson::TJsonValue& /*requestData*/) {
        const auto& cgi = Context->GetCgiParameters();
        const auto setId = GetString(cgi, "set_id", true);
        Y_UNUSED(setId);
        NMtApi::TZoneSetEntity report;
        g.SetCode(HTTP_OK);
        g.SetExternalReport(NJson::ToJson(report));
    }

    TString TMtPolicyProcessor::GetTypeName() {
        return "mtapi/policy";
    }

    void TMtPolicyProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr /*permissions*/, const NJson::TJsonValue& /*requestData*/) {
        auto locale = GetLocale();
        auto localization = Server->GetLocalizationAs<NLocalization::TLocalizationDB>();
        ReqCheckCondition(!!localization, HTTP_INTERNAL_SERVER_ERROR, NMtApi::OtherErrorCode);
        TVector<NLocalization::TResource> resources;
        ReqCheckCondition(localization->GetInfo({"maas.policy"}, resources, Context->GetRequestStartTime()), HTTP_INTERNAL_SERVER_ERROR, NMtApi::OtherErrorCode);
        ReqCheckCondition(!resources.empty(), HTTP_NOT_FOUND, NMtApi::OtherErrorCode);
        auto&& resource = resources.front();
        NMtApi::TPolicyEntity report;
        ReqCheckCondition(resource.HasRevision(), HTTP_NOT_FOUND, NMtApi::OtherErrorCode);
        ReqCheckCondition(!!resource.GetLocalization(locale), HTTP_NOT_FOUND, NMtApi::OtherErrorCode);
        report.Content = *resource.GetLocalization(locale);
        report.Version = ToString(resource.GetRevision());
        g.SetCode(HTTP_OK);
        g.SetExternalReport(NJson::ToJson(report));
    }

}
