#include "processor.h"

#include <drive/backend/data/leasing/company.h>
#include <drive/backend/data/leasing/acl/acl.h>
#include <drive/backend/database/drive/named_filters.h>
#include <drive/backend/cars/car.h>

namespace {
    using namespace NDrivematics;

    enum class EFilterExistence: ui8 {
        NotExist = 0,
        ExistInOtherObject = 1,
        ExistInCurrentObject = 2,
        InconsistentValue = 8
    };

    EFilterExistence CheckObjectExistence(const TNamedFilter& namedFilter, const TSet<TString>& objectIds, TMessagesCollector& errors, const NDrive::IServer& server) {
        TMap<TString, TVector<TNamedFilter>> filterToEntity;
        const auto actionSelf = [&filterToEntity](const TString& /*key*/, const TNamedFilter& entity) {
            filterToEntity[entity.GetFilter().ToString()].push_back(entity);
        };
        R_ENSURE(server.GetDriveAPI()->GetNamedFiltersDB()->ForObjectsMap(actionSelf, TInstant::Now(), &objectIds), HTTP_INTERNAL_SERVER_ERROR, "cannot get object from cache");

        size_t countFilters = filterToEntity[namedFilter.GetFilter().ToString()].size();
        if (countFilters == 0) {
            return EFilterExistence::NotExist;
        }
        if (countFilters == 1) {
            if (filterToEntity[namedFilter.GetFilter().ToString()].begin()->GetId() != namedFilter.GetId()) {
                errors.AddMessage("CheckObjectExistence", TStringBuilder() << "object already exist: " << filterToEntity[namedFilter.GetFilter().ToString()].begin()->GetId());
                return EFilterExistence::ExistInOtherObject;
            } else {
                return EFilterExistence::ExistInCurrentObject;
            }
        }
        errors.AddMessage("CheckObjectExistence", "inconsistent filters data");
        return EFilterExistence::InconsistentValue;
    }

    bool CheckUpsert(const TNamedFilter& namedFilter, const TSet<TString>& namedFiltersIds, NDrive::TEntitySession& tx, const NDrive::IServer& server) {
        TMessagesCollector errors;
        TMaybe<EFilterExistence> exists = CheckObjectExistence(namedFilter, namedFiltersIds, errors, server);
        if (!exists.Defined()) {
            tx.AddErrorMessage("CheckUpsert", TStringBuilder() << "cannot find object " << namedFilter.GetId());
            return false;
        }
        if (*exists != EFilterExistence::ExistInCurrentObject && *exists != EFilterExistence::NotExist) {
            tx.AddErrorMessage("CheckUpsert", TStringBuilder() << errors.GetStringReport());
            return false;
        }
        return true;
    }
}

void TNamedFiltersAddProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    const bool toOrganization = this->template GetValue<bool>(Context->GetCgiParameters(), "to_organization", false).GetOrElse(false);
    auto tx = BuildTx<NSQL::Writable>();

    auto acl = NDrivematics::TACLTag::GetACLTag(permissions, tx, *Server);
    auto aclCompany = acl->GetCompanyTagDescription(permissions->GetUserId(), *Server);

    R_ENSURE(requestData.Has("object"), HTTP_INTERNAL_SERVER_ERROR, "incorrect json, must have 'object'");
    TNamedFilter object;
    TMessagesCollector errors;
    R_ENSURE(TBaseDecoder::DeserializeFromJsonVerbose(object, requestData["object"], errors), HTTP_INTERNAL_SERVER_ERROR, "incorrect object json: " << errors.GetStringReport());

    NStorage::TObjectRecordsSet<TNamedFilter> records;
    R_ENSURE(GetEntitiesManager(Server)->AddObjects({object}, permissions->GetUserId(), tx, &records), HTTP_INTERNAL_SERVER_ERROR, "cannot add object ", tx);
    R_ENSURE(!records.empty(), {}, "cannot add object to table");
    object.MutableId() = records.front().GetId();

    if (toOrganization) {
        TSet<TString> companyNamedFilters = aclCompany->GetEntityObjects(GetEntityType(), permissions, true);
        R_ENSURE(CheckObjectExistence(object, companyNamedFilters, errors, *Server) == EFilterExistence::NotExist
            , HTTP_CONFLICT
            , errors.GetStringReport()
        );
        R_ENSURE(aclCompany->AddEntityObject(GetEntityType(), object.GetId(), permissions, errors), {}, "cannot add object to company, " << errors.GetStringReport());
        R_ENSURE(Server->GetDriveAPI()->GetTagsManager().GetTagsMeta().RegisterTag(aclCompany, permissions->GetUserId(), tx), {}, "cannot update object for company", tx);
    } else {
        TSet<TString> userNamedFilters = acl->GetEntityObjects(GetEntityType(), permissions, true);
        R_ENSURE(CheckObjectExistence(object, userNamedFilters, errors, *Server) == EFilterExistence::NotExist
            , HTTP_CONFLICT
            , errors.GetStringReport()
        );
        R_ENSURE(acl->AddEntityObject(GetEntityType(), object.GetId(), errors, permissions), HTTP_FORBIDDEN, "cannot add object to user, " << errors.GetStringReport());
        R_ENSURE(
            DriveApi->GetTagsManager().GetUserTags().UpdateTagData(acl, permissions->GetUserId(), tx)
            , HTTP_INTERNAL_SERVER_ERROR
            , "cannot update tag"
            , tx
        );
    }

    R_ENSURE(tx.Commit(), {}, "cannot commit", tx);

    g.SetCode(HTTP_OK);
}

template <class T>
void TRemoveACLObjectsProcessor<T>::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    auto ids = TBase::GetStrings(requestData, "ids");

    auto tx = TBase::template BuildTx<NSQL::Writable>();

    auto acl = NDrivematics::TACLTag::GetACLTag(permissions, tx, *TBase::Server);
    auto aclCompany = acl->GetCompanyTagDescription(permissions->GetUserId(), *TBase::Server);

    TSet<TString> userObjects = acl->GetEntityObjects(GetEntityType(), permissions, true);
    TSet<TString> companyObjects = aclCompany->GetEntityObjects(GetEntityType(), permissions, true);

    TSet<TString> userToRemove;
    TSet<TString> companyToRemove;
    for (const auto& id : ids) {
        if (userObjects.contains(id)) {
            userToRemove.insert(id);
        } else if (companyObjects.contains(id)) {
            companyToRemove.insert(id);
        } else {
            R_ENSURE(false, {}, "not found object");
        }
    }

    if (!userToRemove.empty()) {
        R_ENSURE(acl->RemoveEntityObjects(GetEntityType(), userToRemove, permissions, TBase::Server, tx), {}, "cannot remove objects", tx);
        R_ENSURE(
            TBase::DriveApi->GetTagsManager().GetUserTags().UpdateTagData(acl, permissions->GetUserId(), tx)
            , HTTP_INTERNAL_SERVER_ERROR
            , "cannot update tag: " << tx.GetStringReport()
        );
    }
    if (!companyToRemove.empty()) {
        R_ENSURE(aclCompany->RemoveEntityObjects(GetEntityType(), companyToRemove, permissions, TBase::Server, tx), {}, "cannot remove objects", tx);
        R_ENSURE(TBase::DriveApi->GetTagsManager().GetTagsMeta().RegisterTag(aclCompany, permissions->GetUserId(), tx), {}, "cannot update object for company", tx);
    }
    R_ENSURE(tx.Commit(), {}, "cannot commit", tx);

    g.SetCode(HTTP_OK);
}

template class TRemoveACLObjectsProcessor<TNamedFiltersRemoveProcessor>;
template class TRemoveACLObjectsProcessor<TStaticFiltersRemoveProcessor>;

void TNamedFiltersUpsertProcessor::UpsertObjects(const bool force, const TVector<TNamedFilter>& containers, TUserPermissions::TPtr permissions, NDrive::TEntitySession& session) const {
    for (auto&& container : containers) {
        if (force) {
            if (!GetEntitiesManager(Server)->ForceUpsertObject(container, permissions->GetUserId(), session)) {
                session.DoExceptionOnFail(ConfigHttpStatus);
            }
        } else {
            if (!GetEntitiesManager(Server)->UpsertObject(container, permissions->GetUserId(), session)) {
                session.DoExceptionOnFail(ConfigHttpStatus);
            }
        }
    }
}

void TNamedFiltersUpsertProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    R_ENSURE(requestData["objects"].IsArray(), HTTP_INTERNAL_SERVER_ERROR, "'objects' is not array");
    const bool force = IsTrue(Context->GetCgiParameters().Get("force"));

    TVector<TNamedFilter> objectContainer;
    for (auto&& i : requestData["objects"].GetArraySafe()) {
        TNamedFilter container;
        TMessagesCollector errors;
        R_ENSURE(TBaseDecoder::DeserializeFromJsonVerbose(container, i, errors), HTTP_INTERNAL_SERVER_ERROR, "incorrect object json: " + errors.GetStringReport());
        R_ENSURE(container.GetFilterType() == "car", HTTP_INTERNAL_SERVER_ERROR, "NamedFilter type must be 'car'");
        objectContainer.emplace_back(std::move(container));
    }

    auto tx = BuildTx<NSQL::Writable>();

    auto acl = NDrivematics::TACLTag::GetACLTag(permissions, tx, *Server);
    auto aclCompany = acl->GetCompanyTagDescription(permissions->GetUserId(), *Server);

    TSet<TString> userNamedFilters = acl->GetEntityObjects(GetEntityType(), permissions, true);
    TSet<TString> companyNamedFilters = aclCompany->GetEntityObjects(GetEntityType(), permissions, true);

    bool needCheckRight = true;
    for (auto& object : objectContainer) {
        if (companyNamedFilters.contains(object.GetId())) {
            R_ENSURE(
                CheckUpsert(object, companyNamedFilters, tx, *Server)
                , HTTP_INTERNAL_SERVER_ERROR
                , "cannot upsert object"
                , tx
            );
            if (needCheckRight) {
                R_ENSURE(
                    aclCompany->CheckPermission(GetEntityType(), TAdministrativeAction::EAction::Modify, permissions)
                    , HTTP_FORBIDDEN
                    , "no_permissions." + ToString(GetEntityType()) + "::Modify"
                );
                needCheckRight = false;
            }
        } else if (CheckUpsert(object, userNamedFilters, tx, *Server)) {
            R_ENSURE(false, {}, "not found object");
        }
    }
    UpsertObjects(force, objectContainer, permissions, tx);

    R_ENSURE(tx.Commit(), {}, "cannot commit", tx);

    g.SetCode(HTTP_OK);
}

void TNamedFiltersGetProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    const TSet<TString> ids = MakeSet(GetStrings(Context->GetCgiParameters(), "id", false));

    auto tx = BuildTx<NSQL::ReadOnly>();

    auto acl = NDrivematics::TACLTag::GetACLTag(permissions, tx, *Server);
    auto aclCompany = acl->GetCompanyTagDescription(permissions->GetUserId(), *Server);

    TSet<TString> userNamedFilters = acl->GetEntityObjects(GetEntityType(), permissions, true);
    TSet<TString> companyNamedFilters = aclCompany->GetEntityObjects(GetEntityType(), permissions, true);

    TSet<TString> selfIds;
    TSet<TString> companyIds;
    if (!ids.empty()) {
        selfIds = MakeIntersection(userNamedFilters, ids);
        if (!selfIds.empty()) {
            R_ENSURE(
                acl->CheckPermission(GetEntityType(), TAdministrativeAction::EAction::Observe, permissions)
                , HTTP_FORBIDDEN
                , "no_permissions." + ToString(GetEntityType()) + "::Observe"
                , NDrive::MakeError("acl.access.named_filters.self")
                , tx
            );
        }
        companyIds = MakeIntersection(companyNamedFilters, ids);
        if (!companyIds.empty()) {
            R_ENSURE(
                aclCompany->CheckPermission(GetEntityType(), TAdministrativeAction::EAction::Observe, permissions)
                , HTTP_FORBIDDEN
                , "no_permissions." + ToString(GetEntityType()) + "::Observe"
                , NDrive::MakeError("acl.access.named_filters.company")
                , tx
            );
        }
        R_ENSURE(ids.size() == companyIds.size() + selfIds.size(), HTTP_BAD_REQUEST, "cannot found all objects", NDrive::MakeError("named_filters.incorrect_request"));
    } else {
        selfIds = acl->CheckPermission(GetEntityType(), TAdministrativeAction::EAction::Observe, permissions) ? userNamedFilters : TSet<TString>();
        companyIds = aclCompany->CheckPermission(GetEntityType(), TAdministrativeAction::EAction::Observe, permissions) ? companyNamedFilters : TSet<TString>();
    }

    TVector<TNamedFilter> objectSelf;
    TVector<TNamedFilter> objectCompany;
    if (!selfIds.empty()) {
        R_ENSURE(Server->GetDriveAPI()->GetNamedFiltersDB()->GetCustomObjectsFromCache(objectSelf, selfIds, Context->GetRequestStartTime()), {}, "cannot get objects from cache");
    }
    if (!companyIds.empty()) {
        R_ENSURE(Server->GetDriveAPI()->GetNamedFiltersDB()->GetCustomObjectsFromCache(objectCompany, companyIds, Context->GetRequestStartTime()), {}, "cannot get objects from cache");
    }

    NJson::TJsonValue result = NJson::JSON_MAP;
    result.InsertValue("company_named_filters", TNamedFilter::GetReport(std::move(objectCompany)));
    result.InsertValue("self_named_filters", TNamedFilter::GetReport(std::move(objectSelf)));
    g.MutableReport().AddReportElement("objects", std::move(result));

    g.SetCode(HTTP_OK);
}

void TStaticFiltersAddProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    const bool toOrganization = this->template GetValue<bool>(Context->GetCgiParameters(), "to_organization", false).GetOrElse(false);
    auto tx = BuildTx<NSQL::Writable>();

    auto acl = TACLTag::GetACLTag(permissions, tx, *Server);
    auto aclCompany = acl->GetCompanyTagDescription(permissions->GetUserId(), *Server);

    R_ENSURE(requestData.Has("object"), HTTP_INTERNAL_SERVER_ERROR, "incorrect json, must have 'object'");

    NStorage::TTableRecord tRecord;
    R_ENSURE(tRecord.DeserializeFromJson(requestData["object"]), ConfigHttpStatus.SyntaxErrorStatus, "table record cannot be parsed from json", EDriveSessionResult::IncorrectRequest);

    TTagDescription::TPtr tDescription = TTagDescription::ConstructFromTableRecord(tRecord);
    R_ENSURE(tDescription, ConfigHttpStatus.SyntaxErrorStatus, "tag description cannot be constructed", EDriveSessionResult::IncorrectRequest);

    TTagDescription::TConstPtr oldDescription = GetEntitiesManager(Server).GetDescriptionByName(tDescription->GetName(), Now());
    R_ENSURE(!oldDescription, HTTP_CONFLICT, "already exists", EDriveSessionResult::IncorrectRequest);

    R_ENSURE(GetEntitiesManager(Server).RegisterTag(tDescription, permissions->GetUserId(), tx), HTTP_INTERNAL_SERVER_ERROR, "cannot add object", tx);

    TMessagesCollector errors;
    if (toOrganization) {
        R_ENSURE(aclCompany->AddEntityObject(GetEntityType(), tDescription->GetName(), permissions, errors), {}, "cannot add object to company, " << errors.GetStringReport());
        R_ENSURE(Server->GetDriveAPI()->GetTagsManager().GetTagsMeta().RegisterTag(aclCompany, permissions->GetUserId(), tx), {}, "cannot update object for company", tx);
    } else {
        R_ENSURE(acl->AddEntityObject(GetEntityType(), tDescription->GetName(), errors, permissions), HTTP_FORBIDDEN, "cannot add object to user, " << errors.GetStringReport());
        R_ENSURE(
            DriveApi->GetTagsManager().GetUserTags().UpdateTagData(acl, permissions->GetUserId(), tx)
            , HTTP_INTERNAL_SERVER_ERROR
            , "cannot update tag"
            , tx
        );
    }

    R_ENSURE(tx.Commit(), {}, "cannot commit", tx);

    g.SetCode(HTTP_OK);
}

void TStaticFiltersUpsertProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    R_ENSURE(requestData["objects"].IsArray(), HTTP_INTERNAL_SERVER_ERROR, "'objects' is not array");

    auto tx = BuildTx<NSQL::Writable>();

    auto acl = TACLTag::GetACLTag(permissions, tx, *Server);
    auto aclCompany = acl->GetCompanyTagDescription(permissions->GetUserId(), *Server);

    TSet<TString> userObjects = acl->GetEntityObjects(GetEntityType(), permissions, true);
    TSet<TString> companyObjects = aclCompany->GetEntityObjects(GetEntityType(), permissions, true);

    bool needCheckRight = true;
    for (auto&& i : requestData["objects"].GetArraySafe()) {
        NStorage::TTableRecord tRecord;
        R_ENSURE(tRecord.DeserializeFromJson(i), ConfigHttpStatus.SyntaxErrorStatus, "table record cannot be parsed from json", EDriveSessionResult::IncorrectRequest);

        TTagDescription::TPtr object = TTagDescription::ConstructFromTableRecord(tRecord);
        R_ENSURE(object, ConfigHttpStatus.SyntaxErrorStatus, "tag description cannot be constructed", EDriveSessionResult::IncorrectRequest);

        if (companyObjects.contains(object->GetName())) {
            if (needCheckRight) {
                R_ENSURE(
                    aclCompany->CheckPermission(GetEntityType(), TAdministrativeAction::EAction::Modify, permissions)
                    , HTTP_FORBIDDEN
                    , "no_permissions." + ToString(GetEntityType()) + "::Modify"
                );
                needCheckRight = false;
            }
        } else if (userObjects.contains(object->GetName())) {

        } else {
            R_ENSURE(false, {}, "not found object");
        }

        R_ENSURE(GetEntitiesManager(Server).RegisterTag(object, permissions->GetUserId(), tx), HTTP_INTERNAL_SERVER_ERROR, "cannot add object", tx);
    }

    R_ENSURE(tx.Commit(), {}, "cannot commit", tx);

    g.SetCode(HTTP_OK);
}

void TStaticFiltersGetProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    const TSet<TString> ids = MakeSet(GetStrings(Context->GetCgiParameters(), "id", false));
    auto locale = GetLocale();
    NJson::TJsonValue objectSelfReport(NJson::JSON_ARRAY);
    NJson::TJsonValue objectCompanyReport(NJson::JSON_ARRAY);

    auto tx = BuildTx<NSQL::ReadOnly>();

    auto acl = TACLTag::GetACLTag(permissions, tx, *Server);
    auto aclCompany = acl->GetCompanyTagDescription(permissions->GetUserId(), *Server);

    TSet<TString> userObjects = acl->GetEntityObjects(GetEntityType(), permissions, true);
    TSet<TString> companyObjects = aclCompany->GetEntityObjects(GetEntityType(), permissions, true);

    TSet<TString> selfIds;
    TSet<TString> companyIds;
    if (!ids.empty()) {
        selfIds = MakeIntersection(userObjects, ids);
        if (!selfIds.empty()) {
            R_ENSURE(
                acl->CheckPermission(GetEntityType(), TAdministrativeAction::EAction::Observe, permissions)
                , HTTP_FORBIDDEN
                , "no_permissions." + ToString(GetEntityType()) + "::Observe"
                , NDrive::MakeError("acl.access.static_filters.self")
                , tx
            );
        }
        companyIds = MakeIntersection(companyObjects, ids);
        if (!companyIds.empty()) {
            R_ENSURE(
                aclCompany->CheckPermission(GetEntityType(), TAdministrativeAction::EAction::Observe, permissions)
                , HTTP_FORBIDDEN
                , "no_permissions." + ToString(GetEntityType()) + "::Observe"
                , NDrive::MakeError("acl.access.static_filters.company")
                , tx
            );
        }
        R_ENSURE(ids.size() == companyIds.size() + selfIds.size(), HTTP_BAD_REQUEST, "cannot found all objects", NDrive::MakeError("static_filters.incorrect_request"));
    } else {
        selfIds = acl->CheckPermission(GetEntityType(), TAdministrativeAction::EAction::Observe, permissions) ? userObjects : TSet<TString>();
        companyIds = aclCompany->CheckPermission(GetEntityType(), TAdministrativeAction::EAction::Observe, permissions) ? companyObjects : TSet<TString>();
    }

    for (const auto& id : selfIds) {
        auto desc = GetEntitiesManager(Server).GetDescriptionByName(id);
        if (desc) {
            objectSelfReport.AppendValue(desc->BuildJsonReport(locale));
        }
    }
    for (const auto& id : companyIds) {
        auto desc = GetEntitiesManager(Server).GetDescriptionByName(id);
        if (desc) {
            objectCompanyReport.AppendValue(desc->BuildJsonReport(locale));
        }
    }

    NJson::TJsonValue result = NJson::JSON_MAP;
    result.InsertValue("company_static_filters", objectSelfReport);
    result.InsertValue("self_static_filters", objectCompanyReport);
    g.MutableReport().AddReportElement("objects", std::move(result));

    g.SetCode(HTTP_OK);
}

void TStaticFiltersGetObjectsProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    const auto id = GetString(Context->GetCgiParameters(), "id");
    auto locale = GetLocale();

    auto tx = BuildTx<NSQL::ReadOnly>();

    auto acl = TACLTag::GetACLTag(permissions, tx, *Server);
    auto aclCompany = acl->GetCompanyTagDescription(permissions->GetUserId(), *Server);

    TSet<TString> userObjects = acl->GetEntityObjects(GetEntityType(), permissions, true);
    TSet<TString> companyObjects = aclCompany->GetEntityObjects(GetEntityType(), permissions, true);

    R_ENSURE(userObjects.contains(id) || (companyObjects.contains(id) && aclCompany->CheckPermission(GetEntityType(), TAdministrativeAction::EAction::Observe, permissions)), HTTP_FORBIDDEN, "no_permissions." + ToString(GetEntityType()) + "::Observe");

    auto dbTags = DriveApi->GetTagsManager().GetDeviceTags().RestoreTagsRobust({}, {id}, tx);
    R_ENSURE(dbTags, {}, "cannot fetch tags", tx);
    TSet<TString> carIds;
    Transform(dbTags->begin(), dbTags->end(), std::inserter(carIds, carIds.begin()), [](const auto& item) { return item.GetObjectId(); });
    auto fetchResult = DriveApi->GetCarsData()->FetchInfo(carIds, tx);
    R_ENSURE(fetchResult, {}, "cannot fetch cars", tx);

    NJson::TJsonValue result(NJson::JSON_ARRAY);
    for (const auto& car : fetchResult) {
        result.AppendValue(car.second.GetReport(locale, permissions->GetDeviceReportTraits()));
    }
    g.MutableReport().AddReportElement("objects", std::move(result));
    g.SetCode(HTTP_OK);
}

void TStaticFiltersAddObjectsProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    const auto id = GetString(requestData, "id");
    const TSet<TString> objectIds = MakeSet(GetStrings(requestData, "object_ids", false));

    auto tx = BuildTx<NSQL::Writable>();

    auto acl = TACLTag::GetACLTag(permissions, tx, *Server);
    auto aclCompany = acl->GetCompanyTagDescription(permissions->GetUserId(), *Server);

    TSet<TString> userObjects = acl->GetEntityObjects(GetEntityType(), permissions, true);
    TSet<TString> companyObjects = aclCompany->GetEntityObjects(GetEntityType(), permissions, true);

    R_ENSURE(userObjects.contains(id) || (companyObjects.contains(id) && aclCompany->CheckPermission(GetEntityType(), TAdministrativeAction::EAction::Modify, permissions)), HTTP_FORBIDDEN, "no_permissions." + ToString(GetEntityType()) + "::Observe");
    auto tag = GetEntitiesManager(Server).CreateTag(id);
    R_ENSURE(tag, {}, "incorrect tag name " + id, tx);

    for (const auto& car : objectIds) {
        auto taggedObject = DriveApi->GetTagsManager().GetDeviceTags().GetCachedOrRestoreObject(car, tx);
        R_ENSURE(taggedObject, {}, "cannot GetCachedOrRestoreObject " << id, tx);
        R_ENSURE(permissions->GetVisibility(*taggedObject, NEntityTagsManager::EEntityType::Car) != TUserPermissions::EVisibility::NoVisible, HTTP_FORBIDDEN, "no permissions for " + car, tx);
        R_ENSURE(DriveApi->GetTagsManager().GetDeviceTags().AddTag(tag, permissions->GetUserId(), car, Server, tx), {}, "cannot add tag", tx);
    };

    R_ENSURE(tx.Commit(), {}, "cannot commit", tx);
    g.SetCode(HTTP_OK);
}
