#include "processor.h"

#include <drive/backend/chat_robots/robot_manager.h>
#include <drive/backend/chat_robots/configuration/config.h>
#include <drive/backend/notifications/manager.h>

#include <util/system/compiler.h>

namespace {
    IRequestProcessorConfig::TFactory::TRegistrator<TNotifiersInfoProcessor::THandlerConfig> Registrator1(TNotifiersInfoProcessor::GetTypeName() + "~");
    IRequestProcessorConfig::TFactory::TRegistrator<TNotifiersUpsertProcessor::THandlerConfig> Registrator2(TNotifiersUpsertProcessor::GetTypeName() + "~");
    IRequestProcessorConfig::TFactory::TRegistrator<TNotifiersRemoveProcessor::THandlerConfig> Registrator3(TNotifiersRemoveProcessor::GetTypeName() + "~");
    IRequestProcessorConfig::TFactory::TRegistrator<TNotifiersProposeProcessor::THandlerConfig> Registrator4(TNotifiersProposeProcessor::GetTypeName() + "~");
    IRequestProcessorConfig::TFactory::TRegistrator<TNotifiersConfirmProcessor::THandlerConfig> Registrator5(TNotifiersConfirmProcessor::GetTypeName() + "~");
    IRequestProcessorConfig::TFactory::TRegistrator<TNotifiersRejectProcessor::THandlerConfig> Registrator6(TNotifiersRejectProcessor::GetTypeName() + "~");
    IRequestProcessorConfig::TFactory::TRegistrator<TNotifiersListProcessor::THandlerConfig> Registrator7(TNotifiersListProcessor::GetTypeName() + "~");
}

bool TNotifiersUpsertProcessor::PreprocessContainer(const TNotifierContainer& container, TUserPermissions::TPtr permissions, TMessagesCollector& errors) const {
    if (container.GetNotifierConfigPtr()) {
        auto chatConfig = std::dynamic_pointer_cast<TChatRobotConfig>(container.GetNotifierConfigPtr());
        const TSimpleUnencryptedMediaStorage* mediaStorage = nullptr;
        if (Server->GetChatRobotsManager()) {
            mediaStorage = Server->GetChatRobotsManager()->GetChatRobotsMediaStorage();
        }
        if (chatConfig && mediaStorage) {
            auto resources = chatConfig->GetChatScript().GetLinkedResources();
            TVector<TMediaResourceDescription> uploadedDescriptions;
            for (auto&& resource : resources) {
                auto description = mediaStorage->UploadResourceToMdsByLink(resource.Name, resource.Link, resource.ContentType, permissions->GetUserId(), true);
                if (!description) {
                    errors.AddMessage("NotifiersPreprocessContainer", description.GetError());
                    return false;
                } else {
                    uploadedDescriptions.emplace_back(std::move(*description));
                }
            }

            if (!uploadedDescriptions.empty()) {
                auto chatSession = BuildChatSession(false);
                for (auto&& descr : uploadedDescriptions) {
                    if (!mediaStorage->AddResourceDescription(descr, chatSession)) {
                        errors.AddMessage("NotifiersPreprocessContainer", chatSession.GetStringReport());
                        return false;
                    }
                }
                if (!chatSession.Commit()) {
                    errors.AddMessage("NotifiersPreprocessContainer", chatSession.GetStringReport());
                    return false;
                }
            }
        }
    }
    return true;
}

void TTestNotifierProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    const auto& notifierId = GetString(requestData, "notifier_id", true);
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::Notifier, notifierId);
    const auto userId = GetString(requestData, "user_id", false);
    const auto notifier = Server->GetNotifier(notifierId);
    R_ENSURE(notifier, HTTP_NOT_FOUND, "cannot find notifier " << notifierId);

    NDrive::TNotifierContext context;
    if (userId) {
        auto optionalUser = Server->GetDriveDatabase().GetUserManager().GetCachedObject(userId);
        R_ENSURE(optionalUser, HTTP_NOT_FOUND, "cannot find user " << userId);
        context.MutableRecipients().push_back(*optionalUser);
    }

    NDrive::TNotifierMessage message(TString{});
    R_ENSURE(message.DeserializeFromJson(requestData), HTTP_BAD_REQUEST, "cannot deserialize Message from POST data");

    auto result = notifier->Notify(message, context);
    g.AddReportElement("result", result ? result->SerializeToJson() : NJson::JSON_NULL);
    g.SetCode(HTTP_OK);
}
