#include "standard_with_discount_area_offer.h"

#include <drive/backend/offers/actions/standard_with_discount_area.h>
#include <drive/backend/offers/offers/standart.h>
#include <drive/backend/offers/context.h>

void BuildStandardWithDiscountAreaOfferProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    const auto& cgi = Context->GetCgiParameters();
    const TString baseOfferId = GetString(cgi, "base_offer_id");
    const auto locale = GetLocale();
    NDrive::TEntitySession tx = BuildTx<NSQL::Writable | NSQL::Deferred>();
    auto asyncOffer = Server->GetOffersStorage()->RestoreOffer(baseOfferId, permissions->GetUserId(), tx);
    R_ENSURE(asyncOffer.Initialized(), {}, "cannot RestoreOffer " << baseOfferId, tx);

    TUserOfferContext uoc(Server, permissions, Context);
    uoc.SetLocale(locale);
    ReqCheckCondition(uoc.FillData(this, requestData), ConfigHttpStatus.UnknownErrorStatus, "user_offer_context_fill_data_error");
    ReqCheckCondition(uoc.Prefetch(), ConfigHttpStatus.UnknownErrorStatus, "user_offer_context_prefetch_error");
    TOffersBuildingContext offersBuildingContext(std::move(uoc));
    if (!offersBuildingContext.Parse(Context, *permissions, requestData, tx)) {
        tx.DoExceptionOnFail(ConfigHttpStatus);
    }
    R_ENSURE(asyncOffer.Wait(Context->GetRequestDeadline()), ConfigHttpStatus.TimeoutStatus, "RestoreOffer timeout");
    auto stOffer = std::dynamic_pointer_cast<TStandartOffer>(asyncOffer.GetValue());
    R_ENSURE(stOffer, ConfigHttpStatus.UnknownErrorStatus, "Cannot get standart offer");
    R_ENSURE(stOffer->GetTypeName() == TStandartOffer::GetTypeNameStatic(), ConfigHttpStatus.UnknownErrorStatus, "Incorrect oofer type. Base offer type can't be " + stOffer->GetTypeName());
    offersBuildingContext.SetCarId(stOffer->GetObjectId());
    auto builders = permissions->GetOfferBuilders();
    TVector<IOfferReport::TPtr> offers;
    for (auto&& builder : builders) {
        auto defaultConstructor = std::dynamic_pointer_cast<const IOfferBuilderAction>(builder);
        auto constructor = std::dynamic_pointer_cast<const TStandardWithDiscountAreaOfferBuilder>(defaultConstructor);
        if (!constructor) {
            continue;
        }
        EOfferCorrectorResult result = constructor->BuildStandardWithDiscountAreaOffers(*stOffer, offers, offersBuildingContext, Server, tx, *permissions);
        R_ENSURE(result != EOfferCorrectorResult::Problems, {}, "cannot BuildStandardWithDiscountAreaOffers", tx);
    }
    // Do not forget to store offers in offers storage.
    if (!offers.empty()) {
        TEventsGuard eg(g.MutableReport(), "store_offers");
        auto asyncOffers = Server->GetOffersStorage()->StoreOffers(offers, tx);
        R_ENSURE(asyncOffers.Initialized(), {}, "cannot store_offers", tx);
    }
    // Build offers report.
    {
        auto reportOptions = ICommonOffer::TReportOptions(locale, NDriveSession::ReportUserApp);
        NJson::TJsonValue offersJson(NJson::JSON_ARRAY);
        for (const auto& discountedOfferReport : offers) {
            offersJson.AppendValue(discountedOfferReport->GetOffer()->BuildJsonReport(reportOptions, *Server));
        }
        g.AddReportElement("offers", std::move(offersJson));
    }
    g.SetCode(HTTP_OK);
}
