#include "processor.h"

#include <drive/backend/processors/common_app/fetcher.h>

#include <drive/backend/billing/manager.h>
#include <drive/backend/data/area_tags.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/data/delegation.h>
#include <drive/backend/data/dictionary_tags.h>
#include <drive/backend/data/model.h>
#include <drive/backend/data/user_tags.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/logging/events.h>
#include <drive/backend/offers/action.h>
#include <drive/backend/offers/context.h>
#include <drive/backend/offers/manager.h>
#include <drive/backend/offers/actions/additional_service.h>
#include <drive/backend/offers/actions/distributing_block.h>
#include <drive/backend/offers/actions/fix_point.h>
#include <drive/backend/offers/actions/flexipack.h>
#include <drive/backend/offers/actions/long_term.h>
#include <drive/backend/offers/actions/pack.h>
#include <drive/backend/offers/actions/standard_with_discount_area.h>
#include <drive/backend/offers/actions/standart.h>
#include <drive/backend/roles/manager.h>

#include <drive/library/cpp/compression/simple.h>
#include <drive/library/cpp/threading/future_cast.h>

#include <library/cpp/iterator/concatenate.h>
#include <library/cpp/protobuf/json/proto2json.h>

#include <rtline/util/algorithm/ptr.h>

namespace {
    NJson::TJsonValue CreateDelegationLandingBlock(const TString& blockName, const TString& title, const TString& body) {
        NJson::TJsonValue result;
        result["name"] = blockName;
        result["value"] = title;
        result["details"] = body;
        return result;
    }

    void SetGlobalFullInsurance(TUserPermissions::TPtr permissions, const NDrive::IServer& server, NDrive::TEntitySession& session) {
        TUserDictionaryTag::SetSettings(NDrive::UserSettingsTagNameSetting, {{"insurance_type", "full"}}, permissions, "", server, session);
    }
};

void TCreateOffersProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    TEventsGuard egProcess(g.MutableReport(), "build_offer_processor");
    const auto& cgi = Context->GetCgiParameters();
    const auto report = GetString(cgi, "report", false);
    auto forcedOfferName = GetString(cgi, "forced_offer_name", false);
    auto offerName = GetString(cgi, "offer_name", false);
    auto offerBuilderTypes = MakeSet(GetStrings(cgi, "offer_builder_type", false));
    auto offerTypes = MakeSet(GetStrings(cgi, "offer_type", false));
    if (offerTypes) {
        if (!offerBuilderTypes && offerTypes.contains(TFixPointOffer::GetTypeNameStatic())) {
            offerBuilderTypes.insert(TFixPointOfferConstructor::GetTypeStatic());
        }
    }
    auto setGlobalFullInsurance = GetValue<bool>(requestData, "set_global_full_insurance", false).GetOrElse(false);

    const ISettings& settings = Server->GetSettings();
    const ILocalization* localization = Server->GetLocalization();
    ELocalization locale = GetLocale();

    NDrive::TEntitySession tx = BuildTx<NSQL::Writable | NSQL::Deferred>();
    TOffersBuildingContext offersBuildingContext(Server);
    R_ENSURE(offersBuildingContext.Parse(Context, *permissions, requestData, tx), {}, "cannot Parse OffersBuildingContext", tx);

    {
        TUserOfferContext uoc(Server, permissions, Context);
        ReqCheckCondition(uoc.FillData(this, requestData), ConfigHttpStatus.UnknownErrorStatus, "user_offer_context_fill_data_error");
        uoc.SetNeedHistoryFreeTimeFees(true);
        uoc.SetNeedTaxiPrice(GetHandlerSettingDef("need_taxi_price", true));
        uoc.SetEnableTaxiSuggest(GetHandlerSetting<bool>("offer.taxi_suggest.enabled").GetOrElse(false));
        uoc.SetExternalUserId(GetExternalUserId());
        uoc.SetLocale(locale);
        uoc.SetOrigin(GetOrigin());
        uoc.SetNeedYandexPaymentMethod(true);
        uoc.SetNeedDefaultAccount(false);
        uoc.SetNeedPaymentMethods(GetHandlerSettingDef("need_payment_methods", true));
        bool allowDistributingBlock = GetValue<bool>(requestData, TDistributingBlockAction::AllowDistributingBlockFlag, false).GetOrElse(true);
        allowDistributingBlock &= uoc.GetSetting<bool>(TDistributingBlockAction::AllowDistributingBlockInOffers).GetOrElse(true);
        uoc.SetNeedDistributingBlockShowsRestriction(uoc.GetSetting<bool>(TDistributingBlockAction::EnableShowsRestrictionSetting).GetOrElse(false) && allowDistributingBlock);
        uoc.SetAllowDistributingBlock(allowDistributingBlock);
        ReqCheckCondition(uoc.Prefetch(), ConfigHttpStatus.UnknownErrorStatus, "user_offer_context_prefetch_error");
        offersBuildingContext.SetUserHistoryContext(std::move(uoc));
        offersBuildingContext.SetNeedTaxiSurge(GetHandlerSetting<bool>("offer.need_taxi_surge").GetOrElse(false));
    }
    TUserOfferContext& uoc = offersBuildingContext.MutableUserHistoryContextRef();
    uoc.SetFilterAccounts(permissions->GetSetting<bool>(settings, "billing.filter_accounts", false));
    if (!uoc.GetRequestAccountIds() && offerBuilderTypes.size() == 1 && offerBuilderTypes.contains(TLongTermOfferBuilder::GetTypeName())) {
        g.AddEvent("OverrideAccountName");
        uoc.SetAccountName("card");
    }
    if (!uoc.GetRequestAccountIds() && !offersBuildingContext.HasCarId() && !offersBuildingContext.GetPreviousOfferId()) {
        g.AddEvent("OverrideAccountName2");
        uoc.SetAccountName("card");
    }

    if (!offersBuildingContext.HasCarId() && offersBuildingContext.GetPreviousOfferId()) {
        auto eg = g.BuildEventGuard("restore_previous_offer");
        auto asyncPreviousOffer = NThreading::Initialize(offersBuildingContext.GetPreviousOffer());
        auto previousOffer = asyncPreviousOffer.Wait(Context->GetRequestDeadline()) && asyncPreviousOffer.HasValue()
            ? std::dynamic_pointer_cast<IOffer>(asyncPreviousOffer.GetValue())
            : nullptr;
        auto previousFixPointOffer = std::dynamic_pointer_cast<TFixPointOffer>(previousOffer);
        if (previousOffer) {
            if (previousOffer->GetObjectId()) {
                offersBuildingContext.InitializeCar(previousOffer->GetObjectId());
            }
            if (previousOffer->GetBehaviourConstructorId()) {
                offerName = previousOffer->GetBehaviourConstructorId();
            }
            if (!uoc.GetRequestAccountIds() && previousOffer->GetSelectedCharge()) {
                g.AddEvent("RestorePreviousOfferSelectedCharge");
                uoc.SetAccountName(previousOffer->GetSelectedCharge());
            }
        } else {
            g.AddEvent(NJson::TMapBuilder
                ("event", "RestorePreviousOfferError")
                ("previous_offer_id", offersBuildingContext.GetPreviousOfferId())
                ("previous_offer", NJson::ToJson(asyncPreviousOffer))
            );
        }
        if (previousFixPointOffer) {
            TUserOfferContext::TDestinationDescription destination;
            destination.SetCoordinate(previousFixPointOffer->GetFinish());
            destination.SetContextClient(previousFixPointOffer->GetDestinationContext());
            destination.SetDescription(previousFixPointOffer->GetDestinationDescription());
            destination.SetName(previousFixPointOffer->GetDestinationName());
            uoc.SetUserDestination(std::move(destination));
        }
        R_ENSURE(
            previousOffer,
            ConfigHttpStatus.EmptySetStatus,
            "offer " << offersBuildingContext.GetPreviousOfferId() << " is not found"
        );
        R_ENSURE(
            previousOffer->GetUserId() == permissions->GetUserId(),
            ConfigHttpStatus.UnauthorizedStatus,
            "offer " << offersBuildingContext.GetPreviousOfferId() << " belongs to another user"
        );
    }

    if (offersBuildingContext.HasCarId()) {
        TMaybe<TGeoCoord> c = offersBuildingContext.GetStartPosition();
        if (!!c) {
            TCarLocationContext clc = TCarLocationContext::BuildByCoord(offersBuildingContext.GetCarIdUnsafe(), *c, *Server);
            TCarLocationFeatures clf = clc.GetCarAreaFeatures(false, nullptr, Server);
            if (clf.HasAllowDrop()) {
                g.MutableReport().AddReportElement("drop_car_policy", ::ToString(clf.GetAllowDropUnsafe()));
            }
            if (clf.HasAllowDrop() && clf.OptionalAllowDrop() != EDropAbility::Allow) {
                auto notification = MakeHolder<NDrive::TOfferNotification>("ride_away");
                offersBuildingContext.MutableNotifications().AddNotification(std::move(notification));
            }
        }
    }

    TMaybe<TSet<TCiString>> offerFilterTags;

    IOffer::TPtr currentOffer;
    const TBillingSession* bSession = nullptr;
    TMaybe<TBillingSession::TBillingCompilation> billingCompilation;
    if (offersBuildingContext.HasCarId()) {
        TEventsGuard eg(g.MutableReport(), "enrich_car_data");
        TMaybe<TTaggedObject> deviceTags = offersBuildingContext.GetTaggedCar();
        ReqCheckCondition(!!deviceTags, ConfigHttpStatus.UnknownErrorStatus, "incorrect_car_id");

        auto bSessions = DriveApi->GetTagsManager().GetDeviceTags().GetHistoryManager().GetSessionsBuilder("billing", TInstant::Zero());
        auto objectSession = bSessions->GetLastObjectSession(offersBuildingContext.GetCarIdUnsafe());
        bSession = dynamic_cast<const TBillingSession*>(objectSession.Get());
        if (bSession && !bSession->GetClosed() && !!bSession->GetCurrentOffer()) {
            billingCompilation = bSession->GetCompilationAs<TBillingSession::TBillingCompilation>();
            currentOffer = bSession->GetCurrentOffer();
            if (currentOffer->GetUserId() == permissions->GetUserId()) {
                offersBuildingContext.SetEnableAcceptanceCost(false);
                offersBuildingContext.SetSwitching(true);
                offersBuildingContext.SetProlonging(true);
                offersBuildingContext.SetBillingSessionId(bSession->GetCurrentOffer()->GetOfferId());
                if (currentOffer->HasOriginalRidingStart()) {
                    offersBuildingContext.SetOriginalRidingStart(currentOffer->GetOriginalRidingStartUnsafe());
                }
                auto action = DriveApi->GetRolesManager()->GetAction(currentOffer->GetBehaviourConstructorId());
                auto actionBuilder = action ? action->GetAs<IOfferBuilderAction>() : nullptr;
                if (actionBuilder) {
                    offerFilterTags = actionBuilder->GetSwitchOfferTagsList();
                    if (!permissions->GetSetting<bool>(settings, "offers.switching.enabled", false)) {
                        offerFilterTags->clear();
                    }
                }

                auto currentOfferType = currentOffer->GetTypeName();
                bool packOfferTraits =
                    currentOfferType == TPackOffer::GetTypeNameStatic() ||
                    currentOfferType == TLongTermOffer::GetTypeNameStatic() ||
                    currentOfferType == TFlexiblePackOffer::GetTypeNameStatic();
                auto packOffer = packOfferTraits ? dynamic_cast<const TPackOffer*>(currentOffer.Get()) : nullptr;
                if (packOffer && billingCompilation) {
                    auto packOfferState = billingCompilation->GetCurrentOfferStateAs<TPackOfferState>();
                    if (packOfferState) {
                        auto name = localization ? localization->GetLocalString(locale, "offers.switching.pack_offer_name_override", TString()) : TString();
                        if (name) {
                            offersBuildingContext.SetPackOfferNameOverride(name);
                        }
                        offersBuildingContext.SetExtraDistance(packOfferState->GetRemainingDistance());
                        offersBuildingContext.SetExtraDuration(packOfferState->GetPackRemainingTime());
                    }
                    auto longTermOfferState = billingCompilation->GetCurrentOfferStateAs<TLongTermOfferState>();
                    if (longTermOfferState) {
                        offersBuildingContext.SetLongTermSince(ModelingNow() + packOfferState->GetPackRemainingTime());
                    }
                    g.AddEvent(NJson::TMapBuilder
                        ("event", "CurrentSessionInfo")
                        ("state", packOfferState ? packOfferState->GetReport(locale, *Server) : NJson::JSON_NULL)
                    );
                }
            } else if (Server->GetSurgeConstructor()) {
                TEventsGuard eg(g.MutableReport(), "check_futures");
                TMaybe<TCarRTFactors> carFactors = Server->GetSurgeConstructor()->GetInfoActual(offersBuildingContext.GetCarIdUnsafe());
                const TFixPointOffer* fpOffer = dynamic_cast<const TFixPointOffer*>(bSession->GetCurrentOffer().Get());
                if (fpOffer && carFactors && carFactors->HasFinishDuration()) {
                    auto notification = MakeHolder<NDrive::TOfferNotification>("pre_order");
                    offersBuildingContext.MutableNotifications().AddNotification(std::move(notification));
                    offersBuildingContext.SetCarWaitingDuration(carFactors->GetFinishDurationUnsafe());
                    offersBuildingContext.SetCarFuturesStart(fpOffer->GetFinish());
                    offersBuildingContext.SetAvailableStartArea(fpOffer->GetFinishArea());
                    offersBuildingContext.SetLocationTags(DriveApi->GetTagsInPoint(fpOffer->GetFinish()));
                    g.MutableReport().AddReportElement("car_waiting_duration", offersBuildingContext.GetCarWaitingDurationUnsafe());
                }
            }
        }

        TEventsGuard eg1(g.MutableReport(), "check_device_tags");

        TDBTag delegationTag = deviceTags->GetFirstTagByClass<IDelegationTag>();
        if (!!delegationTag) {
            ReqCheckCondition(!delegationTag->GetPerformer() || delegationTag->GetPerformer() == permissions->GetUserId(), ConfigHttpStatus.UserErrorState, "car_is_busy");
        } else if (offersBuildingContext.HasCarWaitingDuration()) {
            TDBTag fReservationTag = deviceTags->GetFirstTagByClass<TTagReservationFutures>();
            ReqCheckCondition(!fReservationTag || !fReservationTag->GetPerformer(), ConfigHttpStatus.UserErrorState, "car_is_busy");
        } else {
            if (currentOffer) {
                auto action = DriveApi->GetRolesManager()->GetAction(currentOffer->GetBehaviourConstructorId());
                auto actionBuilder = action ? action->GetAs<IOfferBuilderAction>() : nullptr;
                if (actionBuilder) {
                    ReqCheckCondition(actionBuilder->CheckDeviceTagsPerformer(*deviceTags, *permissions), ConfigHttpStatus.UserErrorState, "car_is_busy");
                }
            } else {
                ReqCheckCondition(ICommonOfferBuilderAction::CheckDeviceTagsPerformerDefault(*deviceTags, *permissions), ConfigHttpStatus.UserErrorState, "car_is_busy");
            }
        }
    }

    if (offersBuildingContext.HasCarId()) {
        offersBuildingContext.SetPreviousOffersCount(Server->GetOffersStorage()->GetOffersCount(offersBuildingContext.GetCarIdUnsafe(), permissions->GetUserId(), tx));
    }

    if (offersBuildingContext.IsReplacingCar()) {
        auto session = Yensured(offersBuildingContext.GetBillingSession());
        auto currentOffer = Yensured(session->GetCurrentOffer());
        R_ENSURE(currentOffer->GetUserId() == permissions->GetUserId(), ConfigHttpStatus.UnknownErrorStatus, "current session does not belong to current user");
        offerName = currentOffer->GetBehaviourConstructorId();
    }

    if (offerName) {
        offersBuildingContext.SetBuildingFromGroupOffer(true);
    }

    auto delegationCarTagImpl = offersBuildingContext.HasDelegationCarTag() ? offersBuildingContext.GetDelegationCarTagRef().GetTagAs<TP2PDelegationTag>() : nullptr;
    auto currentPackOffer = currentOffer ? std::dynamic_pointer_cast<TPackOffer>(currentOffer) : nullptr;
    if (delegationCarTagImpl) {
        offersBuildingContext.SetEnableAcceptanceCost(false);
    }
    if (delegationCarTagImpl && delegationCarTagImpl->IsPreserveOffer() && currentPackOffer) {
        offersBuildingContext.SetInsuranceType(currentPackOffer->GetInsuranceType());
        offersBuildingContext.SetForceInsuranceType(true);
    }

    if (!uoc.GetRequestAccountIds() && currentOffer && currentOffer->GetSelectedCharge()
        && (delegationCarTagImpl || currentOffer->GetUserId() == permissions->GetUserId())) {
        g.AddEvent("RestoreCurrentOfferSelectedCharge");
        auto chargeAccount = (currentOffer->GetUserId() == permissions->GetUserId() || !TBillingGlobals::SupportedCashbackMethods.contains(currentOffer->GetSelectedCharge())) ? currentOffer->GetSelectedCharge() : "card";
        for (const auto& account : uoc.GetUserAccounts()) {
            if (account && account->GetUniqueName() == chargeAccount && account->IsActual(Context->GetRequestStartTime())) {
                uoc.SetAccountName(chargeAccount);
                break;
            }
        }
    }

    uoc.SetNeedDefaultAccount(true);
    offersBuildingContext.Prefetch();

    auto builders = permissions->GetOfferBuilders();
    auto correctors = permissions->GetOfferCorrections();

    auto buildOffers = [&](TUserAction::TConstPtr builder) -> TVector<IOfferReport::TPtr> {
        auto action = std::dynamic_pointer_cast<const ICommonOfferBuilderAction>(builder);
        if (!action || !action->GetIsPublish()) {
            return {};
        }
        // Additional service offers are not switch offers.
        if (std::dynamic_pointer_cast<const IOfferBuilderAction>(action)) {
            if (offerFilterTags && IsIntersectionEmpty(*offerFilterTags, action->GetGrouppingTags())) {
                return {};
            }
        }
        if (!offerBuilderTypes.empty() && !offerBuilderTypes.contains(action->GetType())) {
            return {};
        }

        TEventsGuard egProcessBuildOffer(g.MutableReport(), action->GetName());
        TVector<IOfferReport::TPtr> offersCurrent;
        R_ENSURE(
            action->BuildOffers(*permissions, correctors, offersCurrent, offersBuildingContext, Server, tx) != EOfferCorrectorResult::Problems,
            {},
            "cannot BuildOffers for " << action->GetName(),
            tx
        );
        if (offerTypes) {
            auto pred = [&offerTypes](const IOfferReport::TPtr& offerReport) -> bool {
                return !offerTypes.contains(offerReport->GetOffer()->GetTypeName());
            };
            offersCurrent.erase(std::remove_if(offersCurrent.begin(), offersCurrent.end(), pred), offersCurrent.end());
        }
        return offersCurrent;
    };
    auto compareOffers = [](IOfferReport::TPtr left, IOfferReport::TPtr right) {
        if (!left) {
            return true;
        }
        if (!right) {
            return false;
        }
        return
            std::tuple(left->GetListPriorityDef(0), left->GetInternalPriorityDef(0), left->GetOffer()->GetGroupName(), left->GetOffer()->GetName()) <
            std::tuple(right->GetListPriorityDef(0), right->GetInternalPriorityDef(0), right->GetOffer()->GetGroupName(), right->GetOffer()->GetName());
    };

    TVector<IOfferReport::TPtr> offersAll;
    {
        auto eg = g.BuildEventGuard("build_offers");
        for (auto&& builder : builders) {
            const auto& name = builder->GetName();
            if (offerName && offerName != name) {
                continue;
            }
            if (forcedOfferName == name) {
                continue;
            }
            auto offers = buildOffers(builder);
            offersAll.insert(offersAll.end(), offers.begin(), offers.end());
        }
    }
    if (forcedOfferName) {
        auto eg = g.BuildEventGuard("forced_build_offers");
        auto action = DriveApi->GetRolesManager()->GetAction(forcedOfferName);
        if (action) {
            auto offers = buildOffers(action->Impl());
            offersAll.insert(offersAll.end(), offers.begin(), offers.end());
        }
    }
    if (offersBuildingContext.HasDelegationCarTag()) {
        R_ENSURE(
            delegationCarTagImpl,
            HTTP_INTERNAL_SERVER_ERROR,
            "cannot cast " << offersBuildingContext.GetDelegationCarTagRef().GetTagId() << " as delegation tag"
        );

        auto disableLocationTagsOnTransfer = GetHandlerSetting<bool>("offer.disable_location_tags_on_transfer").GetOrElse(false);
        bool found = false;
        for (auto&& offer : offersAll) {
            auto o = offer->GetOffer();
            if (!o) {
                continue;
            }
            if (delegationCarTagImpl->IsPreserveOffer()) {
                if (o->GetName() == currentOffer->GetName() && o->GetBehaviourConstructorId() == currentOffer->GetBehaviourConstructorId()) {
                    found = true;
                    break;
                }
            } else {
                auto enforcedOfferName = offersBuildingContext.GetSetting<TString>("delegation.p2p.enforce_offer_name");
                if (enforcedOfferName) {
                    if (o->GetName() == *enforcedOfferName) {
                        found = true;
                        break;
                    }
                } else {
                    if (o->GetTypeName() == TStandartOffer::GetTypeNameStatic()) {
                        found = true;
                        break;
                    }
                }
            }
        }
        if (!found && disableLocationTagsOnTransfer) {
            auto eg = g.BuildEventGuard("disable_location_tags_on_transfer_build_offers");
            offersBuildingContext.ClearGeoConditionsCheckResults();
            offersBuildingContext.ClearPricesForObject();
            offersBuildingContext.SetUseLocationTags(false);
            for (auto&& builder : builders) {
                const auto& name = builder->GetName();
                if (offerName && offerName != name) {
                    continue;
                }
                auto offers = buildOffers(builder);
                offersAll.insert(offersAll.end(), offers.begin(), offers.end());
            }
        }
    }
    std::sort(offersAll.begin(), offersAll.end(), compareOffers);

    TVector<IOfferReport::TPtr> offersComplementary;
    TSet<TString> complementaryInsuranceTypes = offersBuildingContext.GetComplementaryInsuranceTypes().GetOrElse({});
    for (auto&& insuranceType : complementaryInsuranceTypes) {
        auto eg = g.BuildEventGuard("build_complementary_" + insuranceType);
        auto ignoreUnavailableInsuranceType = GetHandlerSetting<bool>("offer.complementary.ignore_unavailable_insurance_type").GetOrElse(true);
        offersBuildingContext.SetIgnoreUnavailableInsuranceType(ignoreUnavailableInsuranceType);
        offersBuildingContext.SetInsuranceType(insuranceType);
        for (auto&& builder : builders) {
            const auto& name = builder->GetName();
            if (offerName && offerName != name) {
                continue;
            }
            auto offers = buildOffers(builder);
            offersComplementary.insert(offersComplementary.end(), offers.begin(), offers.end());
        }
    }
    std::sort(offersComplementary.begin(), offersComplementary.end(), compareOffers);

    {
        if (uoc.HasUserDestination() && offersAll.empty() && !!offersBuildingContext.GetOfferConstructionProblemFirst()) {
            ReqCheckCondition(false, ConfigHttpStatus.UserErrorState, offersBuildingContext.GetOfferConstructionProblemFirst());
        }
    }

    if (offersBuildingContext.HasDelegationCarTag()) {
        ConstructDelegationLandingReply(g, permissions, offersBuildingContext, offersAll, currentOffer, bSession, tx);
        R_ENSURE(tx.Commit(), {}, "cannot Commit", tx);
        return;
    }

    if (Config.GetIsExternal()) {
        TEventsGuard egProcessFetch(g.MutableReport(), "fetch");
        NJson::TJsonValue offersJson(NJson::JSON_ARRAY);

        TEventsGuard egProcessReport(g.MutableReport(), "report");
        for (auto&& report : offersAll) {
            const TStandartOffer* stOffer = report->GetOfferAs<TStandartOffer>();
            if (stOffer && stOffer->GetTypeName() == TStandartOffer::GetTypeNameStatic()) {
                NJson::TJsonValue offerJson;
                auto createPriceRange = [](ui32 originalPrice) {
                    NJson::TJsonValue result;
                    result["lower_bound"] = static_cast<ui32>(floor(0.75 * originalPrice));
                    result["upper_bound"] = static_cast<ui32>(ceil(1.25 * originalPrice));
                    return result;
                };
                ui32 ridingPrice = stOffer->GetRiding().GetPrice();
                ui32 parkingPrice = stOffer->GetParking().GetPrice();

                offerJson.InsertValue("riding", createPriceRange(ridingPrice));
                offerJson.InsertValue("parking", createPriceRange(parkingPrice));
                offersJson.AppendValue(offerJson);
            }
        }
        R_ENSURE(offersBuildingContext.HasCarId(), ConfigHttpStatus.UserErrorState, "no such car");
        auto maybeCarInfo = DriveApi->GetCarsData()->GetObject(offersBuildingContext.GetCarIdUnsafe());
        ReqCheckCondition(!!maybeCarInfo, ConfigHttpStatus.ServiceUnavailable, "internal server error");
        g.MutableReport().AddReportElement("offers", std::move(offersJson));
        auto deeplinkTemplate = GetHandlerSetting<TString>("mosgorpas.offer_deeplink_template").GetOrElse({});
        TString deeplink;
        if (deeplinkTemplate) {
            deeplink = std::move(deeplinkTemplate);
            SubstGlobal(deeplink, "NUMBER", CGIEscapeRet(maybeCarInfo->GetNumber()));
        } else {
            deeplink = maybeCarInfo->GetDeepLink();
        }
        g.MutableReport().AddReportElement("deeplink", std::move(deeplink));
        g.SetCode(HTTP_OK);
        return;
    } else {
        {
            TEventsGuard egProcessStoreOffers(g.MutableReport(), "store_offers");
            auto asyncOffers = Server->GetOffersStorage()->StoreOffers(offersAll, tx);
            R_ENSURE(asyncOffers.Initialized(), {}, "cannot store_offers", tx);
        }
        if (!offersComplementary.empty()) {
            auto eg = g.BuildEventGuard("store_complementary_offers");
            auto asyncOffers = Server->GetOffersStorage()->StoreOffers(offersComplementary, tx);
            R_ENSURE(asyncOffers.Initialized(), {}, "cannot store_complementary_offers", tx);
        }

        auto walkingDuration = offersBuildingContext.GetWalkingDuration(settings.GetValueDef<double>("offers.pedestrian_speed_km/h", 4.2));
        FillOffersReport(
            g.MutableReport(),
            offersAll,
            offersComplementary,
            &offersBuildingContext,
            permissions,
            walkingDuration,
            uoc.GetRequestAccountIds(),
            uoc.GetRequestCreditCard()
        );
    }
    if (auto insuranceTypes = offersBuildingContext.GetInsuranceTypes()) {
        g.AddReportElement("insurance_types", NJson::ToJson(NJson::Nullable(insuranceTypes)));
    }
    if (auto action = offersBuildingContext.GetInsuranceDescription()) {
        auto priceOption = action->Build(*permissions);
        if (priceOption) {
            g.AddReportElement("insurance_description", priceOption->GetPublicReport(locale));
        }
    }
    if (tx.GetMessages().HasMessages()) {
        g.MutableReport().AddReportElement("debug_session_info", tx.GetReport());
    }
    for (auto&& surgeType : offersBuildingContext.GetSurgeTypes()) {
        auto notification = MakeHolder<NDrive::TSurgeOfferNotification>(surgeType);
        offersBuildingContext.MutableNotifications().AddNotification(std::move(notification));
    }
    auto notifications = offersBuildingContext.GetNotifications().GetReport(offersBuildingContext);
    g.AddReportElement("notifications", std::move(notifications));

    const TDistributingBlockAction* bestAction = nullptr;
    for (const auto& action : permissions->GetActionsActual()) {
        auto distributingBlockAction = action.GetAs<TDistributingBlockAction>();
        if (distributingBlockAction && distributingBlockAction->IsRelevant(offersBuildingContext, offersAll, TDistributingBlockAction::ReportInOffers)) {
            if (!bestAction || distributingBlockAction->GetPriority() < bestAction->GetPriority()) {
                bestAction = distributingBlockAction;
            }
        }
    }
    if (bestAction) {
        g.AddReportElement("distributing_block", bestAction->BuildJsonReport(*Server, offersBuildingContext, offersAll, TDistributingBlockAction::EPlaceReportTraits::ReportInOffers));
    }
    if (setGlobalFullInsurance) {
        SetGlobalFullInsurance(permissions, *Server, tx);
    }
    R_ENSURE(tx.Commit(), {}, "cannot Commit", tx);
    g.SetCode(HTTP_OK);
}

void TOffersHandlerBase::ConstructDelegationLandingReply(
    TJsonReport::TGuard& g,
    TUserPermissions::TPtr permissions,
    TOffersBuildingContext& offersBuildingContext,
    const TVector<IOfferReport::TPtr>& offersAll,
    IOffer::TPtr currentOffer,
    const TBillingSession* bSession,
    NDrive::TInfoEntitySession& session
) {
    TEventsGuard egProcessReport(g.MutableReport(), "report");

    auto locale = GetLocale();
    auto objectId = offersBuildingContext.GetCarIdUnsafe();
    TString reportOfferTitle;
    TString reportOfferBody;
    TString reportLocation;

    auto delegationCarTagImpl = offersBuildingContext.GetDelegationCarTagUnsafe().GetTagAs<TP2PDelegationTag>();
    R_ENSURE(delegationCarTagImpl, ConfigHttpStatus.UnknownErrorStatus, "no p2p tag to work with");

    // Accept params
    if (!delegationCarTagImpl->IsPreserveOffer()) {
        TEventsGuard egProcessReport(g.MutableReport(), "accept_params");
        TVector<IOfferReport::TPtr> offersForStoring;
        auto enforceOfferName = Server->GetSettings().GetValueDef<TString>("delegation.p2p.enforce_offer_name", "");
        for (auto&& offer : offersAll) {
            TString offerDescription;
            TStandartOffer* stOffer = offer->GetOfferAs<TStandartOffer>();
            if (stOffer && stOffer->GetTypeName() == TStandartOffer::GetTypeNameStatic()) {
                if (enforceOfferName && enforceOfferName != stOffer->GetName()) {
                    continue;
                }

                reportOfferTitle = Server->GetLocalization()->FormatDelegatedStandartOfferTitle(locale, stOffer->GetName(), stOffer->GetRiding().GetPrice());
                reportOfferBody = Server->GetLocalization()->FormatDelegatedStandartOfferBody(locale, stOffer->GetParking().GetPrice(), stOffer->GetFreeDuration("old_state_reservation"));

                NJson::TJsonValue acceptParamsPost;
                acceptParamsPost.InsertValue("offer_id", stOffer->GetOfferId());
                NJson::TJsonValue acceptParams;
                acceptParams["post"] = std::move(acceptParamsPost);
                acceptParams["cgi"] = NJson::JSON_MAP;
                g.MutableReport().AddReportElement("accept_params", std::move(acceptParams));

                stOffer->SetDeadline(delegationCarTagImpl->GetSLAInstant());
                stOffer->SetAreaInfos(currentOffer->GetAreaInfos());
                offersForStoring.push_back(offer);
                break;
            }
        }
        Y_ENSURE_EX(
            !offersForStoring.empty(),
            TCodedException(ConfigHttpStatus.UserErrorState)
                .SetErrorCode(ToString(TDriveAPI::EDelegationAbility::NoTariffAccess))
                .SetLocalizedTitle(Server->GetLocalization()->GetLocalString(locale, "delegation.p2p.errors.delegatee.title." + ToString(TDriveAPI::EDelegationAbility::NoTariffAccess)))
                .SetLocalizedMessage(Server->GetLocalization()->GetLocalString(locale, "delegation.p2p.errors.delegatee.message." + ToString(TDriveAPI::EDelegationAbility::NoTariffAccess)))
        );
        TVector<IOfferReport::TPtr> offers;
        {
            TEventsGuard egProcessStoreOffers(g.MutableReport(), "store_offers");
            auto asyncOffers = Server->GetOffersStorage()->StoreOffers(offersForStoring, session);
            if (!asyncOffers.Initialized()) {
                session.DoExceptionOnFail(ConfigHttpStatus);
            }
            offers = offersForStoring;
        }
    } else {
        R_ENSURE(bSession, ConfigHttpStatus.UnknownErrorStatus, "no billing session");
        auto billingCompilation = bSession->GetCompilationAs<TBillingSession::TBillingCompilation>();

        TEventsGuard egProcessReport(g.MutableReport(), "accept_params");
        TVector<IOfferReport::TPtr> offersForStoring;
        R_ENSURE(billingCompilation.Defined(), ConfigHttpStatus.SyntaxErrorStatus, "no billing compilation");
        R_ENSURE(currentOffer, ConfigHttpStatus.SyntaxErrorStatus, "no active offer");
        R_ENSURE(
            currentOffer->GetTypeName() == TPackOffer::GetTypeNameStatic() || currentOffer->GetTypeName() == TFlexiblePackOffer::GetTypeNameStatic(),
            ConfigHttpStatus.SyntaxErrorStatus,
            "current offer is not pack offer"
        );

        auto packOffer = dynamic_cast<const TPackOffer*>(currentOffer.Get());
        R_ENSURE(packOffer, ConfigHttpStatus.UnknownErrorStatus, "current offer is not pack offer");

        offersBuildingContext.SetExtraDistance(0);
        offersBuildingContext.SetExtraDuration(TDuration::Zero());

        double delegatedDistance = 0;
        TDuration delegatedDuration = TDuration::Zero();
        ICommonOffer::TPtr newOffer;
        if (packOffer) {
            auto packOfferState = billingCompilation->GetCurrentOfferStateAs<TPackOfferState>();
            R_ENSURE(packOfferState, ConfigHttpStatus.UnknownErrorStatus, "no pack offer state");
            delegatedDistance = packOfferState->GetRemainingDistance();
            delegatedDuration = packOfferState->GetPackRemainingTime();
        }

        for (auto&& offerReport : offersAll) {
            auto offer = offerReport->GetOffer();
            if (!offer) {
                continue;
            }
            if (offer->GetName() == currentOffer->GetName() && offer->GetBehaviourConstructorId() == currentOffer->GetBehaviourConstructorId()) {
                newOffer = offer;
                break;
            }
        }

        Y_ENSURE_EX(
            !!newOffer,
            TCodedException(ConfigHttpStatus.UserErrorState)
                .SetErrorCode(ToString(TDriveAPI::EDelegationAbility::NoTariffAccess))
                .SetLocalizedTitle(Server->GetLocalization()->GetLocalString(locale, "delegation.p2p.errors.delegatee.title." + ToString(TDriveAPI::EDelegationAbility::NoTariffAccess)))
                .SetLocalizedMessage(Server->GetLocalization()->GetLocalString(locale, "delegation.p2p.errors.delegatee.message." + ToString(TDriveAPI::EDelegationAbility::NoTariffAccess)))
        );

        auto newPackOffer = dynamic_cast<TPackOffer*>(newOffer.Get());
        R_ENSURE(newPackOffer, ConfigHttpStatus.SyntaxErrorStatus, "current offer is not pack offer");
        R_ENSURE(delegationCarTagImpl->ShrinkPackOfferOnDelegation(newOffer, *bSession), ConfigHttpStatus.SyntaxErrorStatus, "could not shrink offer on delegation");

        TVector<IOfferReport::TPtr> offers;
        {
            TVector<IOfferReport::TPtr> offersForStoring;
            TEventsGuard egProcessStoreOffers(g.MutableReport(), "store_offers");
            offersForStoring.emplace_back(new TPackOfferReport(newOffer, nullptr));
            auto asyncOffers = Server->GetOffersStorage()->StoreOffers(offersForStoring, session);
            if (!asyncOffers.Initialized()) {
                session.DoExceptionOnFail(ConfigHttpStatus);
            }
            offers = offersForStoring;
        }

        {
            bool isIntercity = false;
            {
                auto action = Server->GetDriveAPI()->GetRolesManager()->GetAction(newPackOffer->GetBehaviourConstructorId());
                auto offerBuilder = action ? action->GetAs<IOfferBuilderAction>() : nullptr;
                if (offerBuilder) {
                    const auto& tags = offerBuilder->GetGrouppingTags();
                    for (auto&& tag : tags) {
                        if (tag == IntercityOfferTag) {
                            isIntercity = true;
                            break;
                        }
                    }
                }
            }

            reportOfferTitle = Server->GetLocalization()->FormatDelegatedPackOfferTitle(locale, newOffer->GetName());
            if (!isIntercity) {
                reportOfferBody = Server->GetLocalization()->FormatDelegatedPackOfferBody(
                    locale,
                    delegatedDistance,
                    delegatedDuration,
                    newPackOffer->GetParking().GetPrice(),
                    newPackOffer->GetFreeDuration("old_state_reservation"),
                    newPackOffer->GetDiscountedOvertimePrice(),
                    newPackOffer->GetDiscountedOverrunPrice(),
                    newPackOffer->GetInsuranceType()
                );
            } else {
                reportOfferBody = Server->GetLocalization()->FormatDelegatedIntercityOfferBody(
                    locale,
                    delegatedDistance,
                    delegatedDuration,
                    newPackOffer->GetParking().GetPrice(),
                    newPackOffer->GetFreeDuration("old_state_reservation"),
                    newPackOffer->GetDiscountedOvertimePrice(),
                    newPackOffer->GetDiscountedOverrunPrice(),
                    newPackOffer->GetInsuranceType()
                );
            }

            NJson::TJsonValue acceptParamsPost;
            acceptParamsPost.InsertValue("offer_id", newOffer->GetOfferId());
            NJson::TJsonValue acceptParams;
            acceptParams["post"] = std::move(acceptParamsPost);
            acceptParams["cgi"] = NJson::JSON_MAP;
            g.MutableReport().AddReportElement("accept_params", std::move(acceptParams));
        }
    }

    // Reject params
    {
        TEventsGuard egProcessReport(g.MutableReport(), "reject_params");
        NJson::TJsonValue rejectParams;

        NJson::TJsonValue rejectParamsCgi;
        if (offersBuildingContext.HasIncomingDelegationTag()) {
            rejectParamsCgi["tag_id"] = offersBuildingContext.GetIncomingDelegationTagRef().GetTagId();
        } else {
            rejectParamsCgi["car_tag_id"] = offersBuildingContext.GetDelegationCarTagRef().GetTagId();
        }
        rejectParams["cgi"] = std::move(rejectParamsCgi);
        rejectParams["post"] = NJson::JSON_MAP;

        g.MutableReport().AddReportElement("reject_params", std::move(rejectParams));
    }

    // Car
    {
        TEventsGuard egProcessFetch(g.MutableReport(), "car");
        R_ENSURE(offersBuildingContext.HasCarId(), ConfigHttpStatus.UnknownErrorStatus, "offers context doesn't have car id");

        TCarsFetcher fetcher(*Server, NDeviceReport::ReportOffers);
        fetcher.SetIsRealtime(true);
        fetcher.SetLocale(locale);

        auto delegatorPermissions = DriveApi->GetUserPermissions(delegationCarTagImpl->GetBaseUserId(), TUserPermissionsFeatures());
        R_ENSURE(delegatorPermissions, ConfigHttpStatus.UnknownErrorStatus, "could not restore delegator permissions");
        R_ENSURE(fetcher.FetchData(delegatorPermissions, {objectId}), ConfigHttpStatus.UnknownErrorStatus, "cannot fetch cars info");

        NJson::TJsonValue carInfo;
        R_ENSURE(fetcher.BuildOneCarInfoReport(objectId, permissions->GetFilterActions(), carInfo), ConfigHttpStatus.UnknownErrorStatus, "cannot build car report");
        g.MutableReport().AddReportElement("car", std::move(carInfo));

        const TRTDeviceSnapshot* snapshot = fetcher.GetSnapshots().GetIterator().SkipTo(objectId);
        if (snapshot) {
            auto geocoded = snapshot->GetGeocoded(TDuration::Max(), fetcher.GetLocationOptions());
            reportLocation = geocoded ? geocoded->Content : TString();
        }

        {
            auto car = Server->GetDriveAPI()->GetCarsData()->GetObject(objectId);
            if (car.Defined()) {
                auto modelCode = car->GetModel();
                auto modelFetchResult = Server->GetDriveAPI()->GetModelsData()->GetCachedOrFetch(modelCode);
                auto modelPtr = modelFetchResult.GetResultPtr(modelCode);
                if (modelPtr) {
                    g.MutableReport().AddReportElement("model", modelPtr->GetReport(locale, NDriveModelReport::ReportAll));
                }
            }
        }

        // Info
        {
            TEventsGuard egProcessFetch(g.MutableReport(), "info");
            NJson::TJsonValue landingBlocks = NJson::JSON_ARRAY;
            landingBlocks.AppendValue(CreateDelegationLandingBlock("тариф", reportOfferTitle, reportOfferBody));
            if (reportLocation) {
                landingBlocks.AppendValue(CreateDelegationLandingBlock("место", reportLocation, ""));
            }
            if (delegationCarTagImpl) {
                const auto delegatorUserId = delegationCarTagImpl->GetBaseUserId();
                auto delegatorFetchResult = DriveApi->GetUsersData()->FetchInfo(delegatorUserId);
                auto delegator = delegatorFetchResult.GetResultPtr(delegatorUserId);
                R_ENSURE(delegator, ConfigHttpStatus.UnknownErrorStatus, "cannot FetchInfo for user " << delegatorUserId);
                auto shortDelegatorName = delegator->GetShortName();
                landingBlocks.AppendValue(CreateDelegationLandingBlock("водитель", shortDelegatorName, ""));
            }
            g.MutableReport().AddReportElement("info", std::move(landingBlocks));
        }
    }

    g.SetCode(HTTP_OK);
}

void TOffersHandlerBase::FillOffersReport(
    TJsonReport& report,
    const TVector<IOfferReport::TPtr>& offers,
    const TVector<IOfferReport::TPtr>& complementary,
    const TOffersBuildingContext* context,
    const TUserPermissions::TPtr permissions,
    const TMaybe<TDuration> walkingDuration,
    const TSet<TString>& accountNames,
    const TMaybe<TString> creditCard
) {
    auto locale = GetLocale();
    auto reportRelevanceInfo = GetHandlerSetting<bool>("offer.report_relevance_info").GetOrElse(false);
    TEventsGuard egProcessFetch(report, "fetch");
    auto carsFetcherTraits = NDeviceReport::ReportOffers;
    if (!GetHandlerSetting<bool>("fetch_clusters").GetOrElse(true)) {
        carsFetcherTraits &= ~EReportTraits::ReportClusters;
    }
    TCarsFetcher fetcher(*Server, carsFetcherTraits);
    fetcher.SetCheckVisibility(false);
    fetcher.SetIsRealtime(true);
    fetcher.SetLocale(locale);
    TSet<TString> cars;
    TSet<TString> models;
    if (context && context->HasCarId()) {
        cars.insert(context->GetCarIdRef());
    }
    for (auto&& r : Concatenate(offers, complementary)) {
        if (!r || !r->GetVisible()) {
            continue;
        }
        auto offer = r->GetOfferPtrAs<IOffer>();
        if (!offer) {
            continue;
        }
        const auto& objectId = offer->GetObjectId();
        const auto& objectModel = offer->GetObjectModel();
        if (objectId) {
            cars.insert(objectId);
        }
        if (objectModel) {
            models.insert(objectModel);
        }
    }
    fetcher.SetSelectedModels(models);
    R_ENSURE(fetcher.FetchData(permissions, cars), ConfigHttpStatus.UnknownErrorStatus, "cannot fetch cars info");

    NJson::TJsonValue transportation(NJson::JSON_MAP);
    for (auto&& objectId : cars) {
        TEventsGuard eventsGuard(report, "transportation");
        const TRTDeviceSnapshot* snapshot = fetcher.GetSnapshots().GetIterator().SkipTo(objectId);
        if (snapshot) {
            auto geocoded = snapshot->GetGeocoded(TDuration::Max(), fetcher.GetLocationOptions());
            auto location = snapshot->GetLocation(TDuration::Max(), fetcher.GetLocationOptions());
            if (location && geocoded && geocoded->Content) {
                TTransportationTag::TFirstMileReportOptions fmro;
                fmro.Locale = locale;
                fmro.Location = location->GetCoord();
                fmro.App = GetUserApp();
                fmro.Geocoded = geocoded ? geocoded->Content : TString();
                fmro.SetBeaconsInfo(Server->GetSettings(), *snapshot);
                if (walkingDuration) {
                    fmro.WalkingDuration = *walkingDuration;
                }
                transportation.InsertValue("first_mile", TTransportationTag::GetFirstMileReport(
                    fmro,
                    *Server
                ));
            }
        }
        break;
    }

    NJson::TJsonValue localizations(NJson::JSON_MAP);
    if (fetcher.HasFutures()) {
        auto localization = Server->GetLocalization();
        localizations.InsertValue("futures_caption_title", localization->GetLocalString(locale, "futures.caption.title"));
        localizations.InsertValue("futures_caption_description", localization->GetLocalString(locale, "futures.caption.description"));
    }

    TEventsGuard egProcessReport(report, "report");
    bool hasFails = false;
    ui32 countOffers = 0;

    const auto& cgi = Context->GetCgiParameters();
    const auto reportTraitsCgi = GetString(cgi, "report", false);
    const auto traits = GetValues<NDriveSession::EReportTraits>(cgi, "traits", false);
    NDriveSession::TReportTraits reportTraits = NDriveSession::GetReportTraits(reportTraitsCgi);
    for (auto&& trait : traits) {
        reportTraits |= trait;
    }

    NDrive::TAreaIds destinationAreaIds;
    TSet<TFilterAction::TSequentialId> explicitFilters;

    auto uoc = context && context->HasUserHistoryContext() ? context->GetUserHistoryContextPtr() : nullptr;
    TVector<NDrive::NBilling::IBillingAccount::TPtr> accounts;
    TMaybe<TVector<TPaymentMethod>> cards;
    {
        TEventsGuard eg(report, "fetch_payment_methods");
        if (uoc) {
            accounts = uoc->GetUserAccounts();
            cards = uoc->GetPaymentMethods();
        } else if (DriveApi->HasBillingManager()) {
            accounts = DriveApi->GetBillingManager().GetAccountsManager().GetSortedUserAccounts(permissions->GetUserId());
            cards = DriveApi->GetUserPaymentMethodsSync(*permissions, *Server, true);
        }
    }
    auto paymentCards = TBillingManager::GetUserPaymentCards(cards.Get(), false);
    auto yandexAccount = permissions->UseYandexPaymentMethod(Server->GetSettings()) ? TBillingManager::GetYandexAccount(cards.Get()) : Nothing();

    auto makeOfferReport = [&](IOfferReport::TPtr r) -> NJson::TJsonValue {
        if (!r->GetVisible()) {
            return {};
        }
        auto offer = r->GetOfferPtrAs<ICommonOffer>();
        if (!offer) {
            return {};
        }
        auto standartOffer = std::dynamic_pointer_cast<TStandartOffer>(offer);
        auto fixPointOffer = std::dynamic_pointer_cast<TFixPointOffer>(offer);
        if (fixPointOffer) {
            auto areaIds = DriveApi->GetAreaIds(fixPointOffer->GetFinish());
            if (destinationAreaIds.empty()) {
                destinationAreaIds = std::move(areaIds);
            } else {
                destinationAreaIds.insert(areaIds.begin(), areaIds.end());
            }
        }

        NJson::TJsonValue offerReport = r->BuildJsonReport(locale, reportTraits, *Server, *permissions);
        R_ENSURE(
            offerReport.IsMap(),
            ConfigHttpStatus.UnknownErrorStatus,
            "incorrect report from offer " << offer->GetTypeName() << ' ' << offer->GetOfferId()
        );

        if (auto rideOffer = r->GetOfferPtrAs<IOffer>(); !!rideOffer) {
            if (const TString& objectId = rideOffer->GetObjectId(); objectId) {
                NJson::TJsonValue carInfo;
                if (fetcher.BuildOneCarInfoReport(objectId, permissions->GetFilterActions(), carInfo)) {
                    auto driveCarInfo = fetcher.GetCarInfo(objectId);
                    if (driveCarInfo) {
                        offerReport.InsertValue("car_number", driveCarInfo->GetNumber());
                    }
                    offerReport.InsertValue("car_info", carInfo);
                } else {
                    report.AddEvent(NJson::TMapBuilder
                        ("event", "BuildOneCarInfoReportError")
                        ("object_id", objectId)
                        ("offer_id", rideOffer->GetOfferId())
                    );
                    hasFails = true;
                    return {};
                }
            }
        }

        if (reportTraits & NDriveSession::EReportTraits::ReportPaymentMethods) {
            offerReport.InsertValue("payment_methods", TBillingManager::GetPaymentMethodsReport(
                permissions->GetSetting<bool>("show_bonus_payment_method").GetOrElse(false),
                locale,
                offer->GetTimestamp(),
                accounts,
                MakeSet(offer->GetChargableAccounts()),
                paymentCards,
                yandexAccount,
                *Server,
                permissions->GetMobilePaymentMethod(Server->GetSettings())
            ));
        }

        if (r->HasTags()) {
            auto filters = fetcher.CalcFilters(r->GetTagsRef());
            explicitFilters.insert(filters.begin(), filters.end());
            offerReport["filters"] = NJson::ToJson(filters);
        }

        if (reportRelevanceInfo) {
            TStringBuilder relevanceInfo;
            if (offer) {
                relevanceInfo << "Correctors:" << Endl;
                for (auto&& corrector : offer->GetCorrectors()) {
                    relevanceInfo << '\t' << corrector << Endl;
                }
            }
            if (standartOffer) {
                relevanceInfo << "RidingPriceInfo:" << Endl;
                for (auto&& i : standartOffer->GetRiding().GetPriceModelInfos()) {
                    relevanceInfo << '\t' << i.Name << '\t' << i.Before << '\t' << i.After << Endl;
                }
            }
            if (standartOffer) {
                relevanceInfo << "ParkingPriceInfo:" << Endl;
                for (auto&& i : standartOffer->GetParking().GetPriceModelInfos()) {
                    relevanceInfo << '\t' << i.Name << '\t' << i.Before << '\t' << i.After << Endl;
                }
            }
            offerReport["relevance_info"] = relevanceInfo;
        }
        if (standartOffer && standartOffer->GetTypeName() == TStandartOffer::GetTypeNameStatic()) {
            if (context && !context->GetDestinations().empty()){
                auto predictedPoint = context->GetDestinations()[0].GetDestination().GetCoordinate();
                offerReport.InsertValue("predicted_point", predictedPoint.ToString());
            }
            bool enablePointB = false;
            auto builders = permissions->GetOfferBuilders();
            for (auto&& builder : builders) {
                auto constructor = std::dynamic_pointer_cast<const TStandardWithDiscountAreaOfferBuilder>(builder);
                if (constructor && context && constructor->CheckOfferConditions(*context, *permissions) == EOfferCorrectorResult::Success) {
                    enablePointB = true;
                    break;
                }
            }
            offerReport.InsertValue("enable_point_b", enablePointB);
        }
        if (GetHandlerSetting<bool>("report_proto").GetOrElse(false)) {
            offerReport["proto"] = *Yensured(NDrive::Compress(offer->SerializeToProto().SerializeAsString()));
        }
        ++countOffers;
        return offerReport;
    };
    NJson::TJsonValue offerReports = NJson::JSON_ARRAY;
    for (auto&& offer : offers) {
        auto offerReport = makeOfferReport(offer);
        if (offerReport.IsDefined()) {
            offerReports.AppendValue(std::move(offerReport));
        }
    }
    NJson::TJsonValue complementaryReports = NJson::JSON_ARRAY;
    for (auto&& offer : complementary) {
        auto offerReport = makeOfferReport(offer);
        if (offerReport.IsDefined()) {
            complementaryReports.AppendValue(std::move(offerReport));
        }
    }
    R_ENSURE(!hasFails || countOffers, ConfigHttpStatus.UnknownErrorStatus, "cannot fetch cars info");

    report.AddReportElementString("cars", fetcher.GetAvailableCarsReportSafe(permissions->GetFilterActions()));
    report.AddReportElement("transportation", std::move(transportation));
    report.AddReportElement("offers", std::move(offerReports));
    if (complementary) {
        report.AddReportElement("complementary", std::move(complementaryReports));
    }
    if (countOffers == 0) {
        auto localization = Server->GetLocalization();
        localizations.InsertValue("empty_title", localization->GetLocalString(locale, "offers.empty.title"));
        localizations.InsertValue("empty_subtitle", localization->GetLocalString(locale, "offers.empty.subtitle"));
        localizations.InsertValue("empty_button", localization->GetLocalString(locale, "offers.empty.button"));
        localizations.InsertValue("empty_fast", localization->GetLocalString(locale, "offers.empty.fast"));
    }
    if (!explicitFilters.empty()) {
        report.AddReportElement("filters", fetcher.GetFiltersReportSafe(&explicitFilters));
    }
    report.AddReportElement("localizations", std::move(localizations));

    NJson::TJsonValue accountsJson(NJson::JSON_ARRAY);
    for (auto&& account : accounts) {
        if (!account->IsActive() || !account->IsSelectable()) {
            continue;
        }
        NJson::TJsonValue accJson;
        accJson["name"] = account->GetName();
        accJson["id"] = account->GetUniqueName();
        accJson["icon"] = "";
        if (!account->GetHiddenBalance()) {
            accJson["balance"] = account->GetBalance();
        }
        accountsJson.AppendValue(accJson);
    }
    report.AddReportElement("accounts", std::move(accountsJson));
    report.AddReportElement("payment_methods", TBillingManager::GetPaymentMethodsReport(
        permissions->GetSetting<bool>("show_bonus_payment_method").GetOrElse(false),
        locale,
        Context->GetRequestStartTime(),
        accounts,
        accountNames,
        paymentCards,
        yandexAccount,
        *Server,
        permissions->GetMobilePaymentMethod(Server->GetSettings()),
        creditCard.GetOrElse(""),
        accountNames
    ));

    auto destinationAreas = DriveApi->GetAreasDB()->GetCachedObjectsVector(destinationAreaIds);
    report.AddReportElement("poi", NDrive::GetPoiReport(destinationAreas, NDrive::DefaultPoiLocationTags, {}, locale, *Server));

    NJson::TJsonValue defaultsJson(NJson::JSON_MAP);
    if (accountNames) {
        defaultsJson["account_id"] = *accountNames.begin();
    }
    report.AddReportElement("defaults", std::move(defaultsJson));
    fetcher.BuildReportMeta(report);
}

void TRestoreOfferProcessor::ProcessServiceRequest(TJsonReport::TGuard& g,
                                                   TUserPermissions::TPtr permissions,
                                                   const NJson::TJsonValue& /*requestData*/) {
    const auto& cgi = Context->GetCgiParameters();
    const TString offerId = GetString(cgi, "offer_id");

    auto tx = BuildTx<NSQL::Deferred | NSQL::ReadOnly>();
    auto asyncOffer = Server->GetOffersStorage()->RestoreOffer(offerId, permissions->GetUserId(), tx);
    R_ENSURE(asyncOffer.Initialized(), {}, "cannot RestoreOffer " << offerId, tx);
    R_ENSURE(asyncOffer.Wait(Context->GetRequestDeadline()), HTTP_GATEWAY_TIME_OUT, "RestoreOffer timeout", tx);
    R_ENSURE(asyncOffer.HasValue(), HTTP_INTERNAL_SERVER_ERROR, NThreading::GetExceptionMessage(asyncOffer), tx);
    auto offer = asyncOffer.GetValue();
    R_ENSURE(offer, HTTP_NOT_FOUND, "cannot find offer " << offerId, tx);
    {
        FillOffersReport(
            g.MutableReport(),
            {MakeAtomicShared<TFakeOfferReport>(offer)},
            {},
            nullptr,
            permissions,
            {},
            { offer->GetSelectedCharge() },
            offer->GetSelectedCreditCard()
        );
        g.SetCode(HTTP_OK);
    }
}
