#include "processor.h"

#include <drive/backend/cars/car.h>
#include <drive/backend/data/rental/rental_service_mode_tag.h>
#include <drive/backend/data/rental/timetable_builder.h>
#include <drive/backend/offers/actions/long_term.h>
#include <drive/backend/offers/actions/rental_offer.h>
#include <drive/backend/roles/permissions.h>

namespace {
constexpr auto week{ TDuration::Days(7) };
constexpr auto year{ TDuration::Days(365) };
}

void TOffersTimetableProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    Y_UNUSED(requestData);
    TEventsGuard egProcess(g.MutableReport(), "offers/timetable");
    const auto& cgi = Context->GetCgiParameters();
    const auto defaultUntilTime = Context->GetRequestStartTime();
    const auto defaultSinceTime = defaultUntilTime - week;
    const auto sinceTimetable = GetTimestamp(cgi, "since", defaultSinceTime);
    const auto untilTimetable = GetTimestamp(cgi, "until", defaultUntilTime);

    const auto& api = *Yensured(Server->GetDriveAPI());
    auto tx = BuildTx<NSQL::ReadOnly>();
    const auto tagsDescriptions = permissions->GetTagsByAction(NTagActions::ETagAction::Observe);

    TSet<TString> userTagNames;
    TSet<TString> carTagNames;
    const TString targetHolderTagType = Config.GetIsRentalOffers() ? TRentalOfferHolderTag::Type() : TLongTermOfferHolderTag::Type();
    for (const auto& tagDescription: tagsDescriptions) {
        if (tagDescription->GetType() == targetHolderTagType) {
            userTagNames.insert(tagDescription->GetName());
        } else if (tagDescription->GetType() == TRentalServiceModeTag::Type()) {
            carTagNames.insert(tagDescription->GetName());
        }
    }

    TTagEventsManager::TQueryOptions queryOptions;
    queryOptions.SetTags(userTagNames);
    queryOptions.SetOrderBy(TVector<TString>{"history_timestamp"});

    TMap<TString, TTaggedObject> allCars;
    TTaggedObjectsSnapshot carsDevicesSnapshot;
    {
        auto eg = g.BuildEventGuard("restore_cars");
        R_ENSURE(api.GetTagsManager().GetDeviceTags().GetObjectsFromCacheByIds({}, carsDevicesSnapshot, TInstant::Zero()), HTTP_INTERNAL_SERVER_ERROR, "can't restore cars from cache");
    }

    {
        auto eg = g.BuildEventGuard("get_visibility");
        for (const auto& [carId, taggedObject] : carsDevicesSnapshot) {
            if (permissions->GetVisibility(taggedObject, NEntityTagsManager::EEntityType::Car) == TUserPermissions::EVisibility::Visible) {
                allCars[carId] = taggedObject;
            }
        }
    }
    TOptionalTagHistoryEvents userTagEvents, carTagEvents;

    userTagEvents = api.GetTagsManager().GetUserTags().GetEvents({}, {sinceTimetable - year}, tx, queryOptions);
    R_ENSURE(userTagEvents, ConfigHttpStatus.UnknownErrorStatus, "cannot get events", tx);

    queryOptions.SetTags(carTagNames);
    carTagEvents = api.GetTagsManager().GetDeviceTags().GetEvents({}, {sinceTimetable - year}, tx, queryOptions);
    R_ENSURE(carTagEvents, ConfigHttpStatus.UnknownErrorStatus, "cannot get events", tx);

    TMap<TString, TMap<TString, TTimetableEventMetadata>> carsTimetable;
    if (Config.GetIsRentalOffers()) {
        R_ENSURE(TTimetableBuilder::Instance().BuildTimetable<TTimetableBuilder::ETimetableType::Rental>(carsTimetable,
                                                                                                         *userTagEvents,
                                                                                                         *carTagEvents,
                                                                                                         sinceTimetable,
                                                                                                         untilTimetable,
                                                                                                         *permissions,
                                                                                                         Server,
                                                                                                         tx,
                                                                                                         false),
                                                                                                         ConfigHttpStatus.UnknownErrorStatus,
                                                                                                         "cannot build timetable",
                                                                                                         tx);
    } else {
        R_ENSURE(TTimetableBuilder::Instance().BuildTimetable<TTimetableBuilder::ETimetableType::LongTerm>(carsTimetable,
                                                                                                 *userTagEvents,
                                                                                                 *carTagEvents,
                                                                                                 sinceTimetable,
                                                                                                 untilTimetable,
                                                                                                 *permissions,
                                                                                                 Server,
                                                                                                 tx,
                                                                                                 false),
                                                                                                 ConfigHttpStatus.UnknownErrorStatus,
                                                                                                 "cannot build timetable",
                                                                                                 tx);
    }

    NJson::TJsonMap jTimetable;
    TSet<TString> userIds;
    TSet<TString> carIds;

    for (const auto& [carId, carTaggedObject]: allCars) {
        jTimetable[carId] = NJson::JSON_ARRAY;
    }
    auto localization = Server->GetLocalization();
    auto locale = GetLocale();
    for (const auto& [carId, offers]: carsTimetable) {
        NJson::TJsonArray jOffersMetadata;
        TVector<TTimetableEventMetadata> sortedEvents;
        for (const auto& [until, offerMetadata]: offers) {
            sortedEvents.push_back(std::move(offerMetadata));
        }
        std::sort(sortedEvents.begin(), sortedEvents.end(), [](const TTimetableEventMetadata& lhs, const TTimetableEventMetadata& rhs)->bool {
            return lhs.Until < rhs.Until;
        });
        for (const auto& event: sortedEvents) {
            NJson::TJsonMap jOfferMetadata;
            jOfferMetadata["since"] = event.Since.Seconds();
            jOfferMetadata["until"] = event.Until.Seconds();
            jOfferMetadata["offer_id"] = event.OfferId;
            jOfferMetadata["user_id"] = event.UserId;
            jOfferMetadata["tag_id"] = event.TagId;
            const auto status = ToLowerUTF8(event.Status);
            jOfferMetadata["status"] = status;
            const auto statusTitleId = "rental.status." + status + ".title";
            jOfferMetadata["status_title"] = localization ? localization->GetLocalString(locale, statusTitleId) : statusTitleId;
            jOfferMetadata["stage"] = event.Stage;
            if (!event.UserId.empty()) {
                userIds.insert(event.UserId);
                if (event.ActualSince) {
                    jOfferMetadata["actual_since"] = event.ActualSince->Seconds();
                }
                if (event.ActualUntil) {
                    jOfferMetadata["actual_until"] = event.ActualUntil->Seconds();
                }
            }
            jOffersMetadata.AppendValue(jOfferMetadata);
        }
        jTimetable[carId] = std::move(jOffersMetadata);
        carIds.insert(carId);
    }

    const auto usersInfo = api.GetUsersData()->FetchInfo(userIds, tx).GetResult();
    const auto carsInfo = api.GetCarsData()->FetchInfo(carIds, tx).GetResult();

    NJson::TJsonMap jUsers;
    for (const auto& [userId, userInfo]: usersInfo) {
        NJson::TJsonMap jUserInfo;
        jUserInfo["first_name"] = userInfo.GetFirstName();
        jUserInfo["last_name"] = userInfo.GetLastName();
        jUserInfo["p_name"] = userInfo.GetPName();
        jUserInfo["phone"] = userInfo.GetPhone();
        jUserInfo["status"] = userInfo.GetStatus();
        jUserInfo["email"] = userInfo.GetEmail();
        jUsers[userId] = std::move(jUserInfo);
    }

    NJson::TJsonMap jCars;
    for (const auto& [carId, carInfo]: carsInfo) {
        NJson::TJsonMap jCarInfo;
        jCarInfo["number"] = carInfo.GetNumber();
        jCarInfo["model"] = carInfo.GetModel();
        jCars[carId] = std::move(jCarInfo);
    }

    g.AddReportElement("offers_timetable", std::move(jTimetable));
    g.AddReportElement("users", std::move(jUsers));
    g.AddReportElement("cars", std::move(jCars));
    g.SetCode(HTTP_OK);
}
