#include "processor.h"

#include <drive/backend/cars/car.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/data/user_tags.h>
#include <drive/backend/offers/manager.h>
#include <drive/backend/roles/permissions.h>
#include <drive/backend/sessions/manager/billing.h>

namespace {
const TString activeRidingNotFound{ "offer.update.active_riding_not_found" };
constexpr bool checkRidingConflict{ false };
}

void TOffersUpdateProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    Y_UNUSED(permissions, requestData);
    TEventsGuard egProcess(g.MutableReport(), "offers/update");
    const TString offerHolderTagId = GetUUID(requestData, "tag_id", false);

    TDBTag tag;
    auto tx = BuildTx<NSQL::Writable>();
    if (!offerHolderTagId.empty()) {
        auto& manager = Server->GetDriveAPI()->GetTagsManager().GetUserTags();
        auto optionalTag = manager.RestoreTag(offerHolderTagId, tx);
        R_ENSURE(optionalTag, HTTP_INTERNAL_SERVER_ERROR, "cannot get tag", tx);
        tag = *optionalTag;
    }

    if (tag) {
        UpdateOfferInBooking(g, permissions, requestData, tx, tag);
    } else {
        UpdateOfferInSession(g, permissions, requestData, tx);
    }
}

void TOffersUpdateProcessor::UpdateOfferInBooking(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions,
                                                  const NJson::TJsonValue& requestData, NDrive::TEntitySession& tx,
                                                  TDBTag& tag) {
    R_ENSURE(permissions->GetTagNamesByAction(TTagAction::ETagAction::Update).contains(tag->GetName()), ConfigHttpStatus.PermissionDeniedStatus, "No permissions for update this type of tag", tx);
    ITag::TPtr data;
    auto serializedData = requestData;
    serializedData["tag"] = tag->GetName();
    data = IJsonSerializableTag::BuildFromJson(DriveApi->GetTagsManager(), serializedData, &tx.MutableMessages());
    R_ENSURE(data, ConfigHttpStatus.SyntaxErrorStatus, "cannot build tag", tx);
    R_ENSURE(data->GetName() == tag->GetName(), ConfigHttpStatus.SyntaxErrorStatus, "tag name inconsistency: " << tag->GetName(), tx);
    auto& manager = Server->GetDriveAPI()->GetTagsManager().GetUserTags();
    if (!manager.UpdateTagData(tag, data, permissions->GetUserId(), Server, tx) || !tx.Commit()) {
        tx.DoExceptionOnFail(ConfigHttpStatus);
    }

    g.SetCode(HTTP_OK);
}

void TOffersUpdateProcessor::UpdateOfferInSession(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions,
                                                  const NJson::TJsonValue& requestData, NDrive::TEntitySession& tx) {
    const TString offerId = GetUUID(requestData, "offer_id", true);
    auto ridingSession = DriveApi->GetSessionManager().GetSession(offerId, tx);
    R_ENSURE(ridingSession, HTTP_INTERNAL_SERVER_ERROR, "cannot get session " << offerId, tx);
    auto pRidingSession = *ridingSession;
    if (!pRidingSession) {
        tx.SetErrorInfo("OffersUpdateProcessor::UpdateOfferInSession", "session not found", NDrive::MakeError(activeRidingNotFound));
        tx.DoExceptionOnFail(ConfigHttpStatus);
    }

    const auto& tagId = pRidingSession->GetInstanceId();
    auto& manager = DriveApi->GetTagsManager().GetDeviceTags();
    auto optionalTag = manager.RestoreTag(tagId, tx);
    R_ENSURE(optionalTag, HTTP_INTERNAL_SERVER_ERROR, "cannot get chargable tag " << tagId, tx);
    auto tag = *optionalTag;
    if (!tag) {
        tx.SetErrorInfo("OffersUpdateProcessor::UpdateOfferInSession", "chargable tag not found", NDrive::MakeError(activeRidingNotFound));
        tx.DoExceptionOnFail(ConfigHttpStatus);
    }
    auto chargableTag = std::dynamic_pointer_cast<TChargableTag>(tag.GetData());
    R_ENSURE(chargableTag, HTTP_INTERNAL_SERVER_ERROR, "cannot get tag as chargable tag " << tagId, tx);

    auto offer = chargableTag->GetVehicleOffer();
    if (!offer) {
        TAtomicSharedPtr<const ISession> offerSessison;
        R_ENSURE(DriveApi->GetUserSession(permissions->GetUserId(), offerSessison, offerId, Now()), HTTP_INTERNAL_SERVER_ERROR, "cannot GetRequestUserSession");
        R_ENSURE(offerSessison, HTTP_NOT_FOUND, "no session found");
        auto billingSession = std::dynamic_pointer_cast<const TBillingSession>(offerSessison);
        R_ENSURE(billingSession, HTTP_INTERNAL_SERVER_ERROR, "cannot cast session " << offerSessison->GetSessionId() << " to BillingSession");
        offer = billingSession->GetCurrentOffer();
    }

    R_ENSURE(offer, HTTP_INTERNAL_SERVER_ERROR, "offer is nullptr " << offerId, tx);

    if (!offer->PatchOffer(requestData, tx, Server, checkRidingConflict)) {
        tx.DoExceptionOnFail(ConfigHttpStatus);
    }

    chargableTag->SetOffer(offer);

    if (!manager.UpdateTagData(tag, chargableTag, permissions->GetUserId(), Server, tx) || !tx.Commit()) {
        tx.DoExceptionOnFail(ConfigHttpStatus);
    }

    g.SetCode(HTTP_OK);
}
