#include "processor.h"

#include <drive/backend/data/leasing/company.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/data/user_tags.h>
#include <drive/backend/tags/tags_manager.h>

#include <drive/library/cpp/raw_text/phone_number.h>

void TOrganizationListProcessor::DoAuthProcess(TJsonReport::TGuard& g, IAuthInfo::TPtr authInfo) {
    TEventsGuard egProcess(g.MutableReport(), "organization/list");
    auto server = NDrive::GetServer().GetAsPtrSafe<NDrive::IServer>();
    auto driveApi = server->GetDriveAPI();
    auto tx = driveApi->GetUsersData()->BuildTx<NSQL::ReadOnly>();

    R_ENSURE(driveApi, ConfigHttpStatus.UnknownErrorStatus, "drive api is nullptr", NDrive::MakeError("internal_error"), tx);
    auto usersData = driveApi->GetUsersData();
    R_ENSURE(usersData, ConfigHttpStatus.UnknownErrorStatus, "usersData is nullptr", NDrive::MakeError("internal_error"), tx);

    TVector<TString> userIds;
    if (const auto& userId = authInfo->GetUserId(); !userId.empty()) {
        userIds.push_back(userId);
    } else {
        const auto& phone = authInfo->GetPhone();
        R_ENSURE(phone, ConfigHttpStatus.UnknownErrorStatus, "phone not found",  NDrive::MakeError("rental.token.phone_not_found"), tx);

        TPhoneNormalizer phoneNormalizer;
        const auto phoneNumber = phoneNormalizer.TryNormalize(phone);

        auto optionalUsers = usersData->GetUsersByPhone(phoneNumber, tx);
        R_ENSURE(optionalUsers, ConfigHttpStatus.UnknownErrorStatus, "cannot get users", NDrive::MakeError("internal_error"), tx);

        for (const auto& user: *optionalUsers) {
            if (NDrive::UserStatusDeleted != user.GetStatus()) {
                userIds.push_back(user.GetUserId());
            }
        }
    }

    NJson::TJsonArray jArr;
    for (const auto& userId: userIds) {
        auto optionalObject = server->GetDriveDatabase().GetTagsManager().GetUserTags().GetCachedOrRestoreObject(userId, tx);
        R_ENSURE(optionalObject, ConfigHttpStatus.UnknownErrorStatus, "cannot restore object " + userId, NDrive::MakeError("internal_error"), tx);

        const auto& taggedObject = *optionalObject;
        auto dbTag = taggedObject.GetFirstTagByClass<NDrivematics::TUserOrganizationAffiliationTag>();
        if (!dbTag) {
            continue;
        }

        auto description = server->GetDriveDatabase().GetTagsManager().GetTagsMeta().GetDescriptionByName(dbTag->GetName());
        R_ENSURE(description, ConfigHttpStatus.UnknownErrorStatus, "cannot find description " + dbTag->GetName(), NDrive::MakeError("internal_error"), tx);

        auto affilationTagDescription = std::dynamic_pointer_cast<const NDrivematics::TUserOrganizationAffiliationTag::TDescription>(description);
        R_ENSURE(affilationTagDescription, ConfigHttpStatus.UnknownErrorStatus, "cannot cast affilation tag description", NDrive::MakeError("internal_error"), tx);

        NJson::TJsonValue jVal;
        jVal["user_id"] = userId;
        jVal["company_name"] = affilationTagDescription->GetCompanyName();

        const auto locale = GetLocale();
        auto& shortDescription = affilationTagDescription->GetShortDescription();
        auto localization = server->GetLocalization();
        jVal["short_description"] = localization ? localization->GetLocalString(locale, shortDescription) : shortDescription;

        auto& icon = affilationTagDescription->GetIconMobileLogin();
        jVal["icon"] = localization ? localization->GetLocalString(locale, icon) : icon;

        jArr.AppendValue(jVal);
    }

    if (jArr.GetArray().empty() && !userIds.empty()) {
        NJson::TJsonValue jVal;
        jVal["user_id"] = userIds.front();
        jArr.AppendValue(jVal);
    }

    g.SetExternalReport(std::move(jArr));
    g.SetCode(HTTP_OK);
}

TSimpleProcessorRegistrator<TOrganizationListProcessor> TOrganizationListProcessorRegistrator("organization/list");
