#include "processor.h"

#include <drive/backend/billing/manager.h>
#include <drive/backend/chat_robots/abstract.h>
#include <drive/backend/common/localization.h>
#include <drive/backend/data/billing_tags.h>
#include <drive/backend/data/dictionary_tags.h>
#include <drive/backend/data/user_tags.h>
#include <drive/backend/notifications/native_chat/chat.h>
#include <drive/backend/offers/context.h>
#include <drive/backend/offers/actions/correctors.h>
#include <drive/backend/promo_codes/entities.h>
#include <drive/backend/promo_codes/common/manager.h>
#include <drive/backend/roles/manager.h>
#include <drive/backend/tags/tags_manager.h>

#include <drive/library/cpp/clear_web/client.h>
#include <drive/library/cpp/clck/client.h>

#include <rtline/library/json/adapters.h>
#include <rtline/util/algorithm/container.h>

void TGetUserPromoCodeProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    auto generator = GetHandlerSetting<TString>("promo.generator");
    ReqCheckCondition(!generator.Empty(), ConfigHttpStatus.UnknownErrorStatus, "undefined_promo_generator");
    g.AddEvent(NJson::TMapBuilder("generator", *generator));

    TPromoCodesSearchContext context;
    context.SetActiveOnly(true);
    context.SetGenerator(*generator);
    context.SetGivenOut("__FILLED");
    TVector<IPromoCodeMetaReport::TPtr> codesMeta;
    auto session = BuildTx<NSQL::ReadOnly>();
    ReqCheckCondition(
        Server->GetPromoCodesManager()->GetPromoCodes(context, permissions, codesMeta, session),
        ConfigHttpStatus.UnknownErrorStatus,
        "get_promo_codes_failure"
    );
    ReqCheckCondition(
        !codesMeta.empty(),
        ConfigHttpStatus.UnknownErrorStatus,
        "promo_codes_missing"
    );
    g.AddEvent(NJson::TMapBuilder("promo_codes_count", codesMeta.size()));

    auto hash = FnvHash<ui32>(permissions->GetUserFeatures().GetUid());
    g.AddEvent(NJson::TMapBuilder("hash", hash));

    auto codeReport = codesMeta[hash % codesMeta.size()]->GetReport(*Server, true);
    g.SetExternalReport(std::move(codeReport));
    g.SetCode(HTTP_OK);
}

void TGetPromoCodesProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    ReqCheckCondition(Server->GetPromoCodesManager(), ConfigHttpStatus.ServiceUnavailable, EDriveLocalizationCodes::InternalServerError);

    TPromoCodesSearchContext context;
    ReqCheckCondition(context.ReadFrom(Context->GetCgiParameters()), ConfigHttpStatus.SyntaxErrorStatus, EDriveLocalizationCodes::SyntaxUserError);

    TVector<IPromoCodeMetaReport::TPtr> codesMeta;
    auto withCode = GetValue<bool>(Context->GetCgiParameters(), "with_code", false).GetOrElse(false);
    auto session = BuildTx<NSQL::ReadOnly>();
    ReqCheckCondition(Server->GetPromoCodesManager()->GetPromoCodes(context, permissions, codesMeta, session), ConfigHttpStatus.UnknownErrorStatus, ToString(EDriveLocalizationCodes::InternalServerError), session.GetStringReport());

    NJson::TJsonValue report = NJson::JSON_ARRAY;
    for (auto&& i : codesMeta) {
        report.AppendValue(i->GetReport(*Server, withCode));
    }

    g.MutableReport().AddReportElement("codes", std::move(report));
    g.SetCode(HTTP_OK);
}

void TCheckPromoCodeProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    ReqCheckCondition(Server->GetPromoCodesManager(), ConfigHttpStatus.ServiceUnavailable, EDriveLocalizationCodes::InternalServerError);
    const TString code = Strip(GetString(Context->GetCgiParameters(),"code", true));

    auto session = BuildTx<NSQL::ReadOnly>();
    auto promoMeta = Server->GetPromoCodesManager()->GetMetaByCode(code, session);
    if (!promoMeta) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }
    ReqCheckCondition(!promoMeta->empty(), ConfigHttpStatus.EmptySetStatus, EDriveLocalizationCodes::IncorrectPromoCode);
    const TString id = promoMeta->front().GetId();

    TPromoCodesSearchContext context;
    context.SetIds({ id });
    context.SetUsageHistory(IsTrue(Context->GetCgiParameters().Get("usage_history")));

    TVector<IPromoCodeMetaReport::TPtr> codesMeta;
    ReqCheckCondition(Server->GetPromoCodesManager()->GetPromoCodes(context, permissions, codesMeta, session), ConfigHttpStatus.UnknownErrorStatus, EDriveLocalizationCodes::InternalServerError);
    ReqCheckCondition(codesMeta.size() == 1, ConfigHttpStatus.UserErrorState, EDriveLocalizationCodes::NoPermissions);

    const auto& meta = codesMeta.front();

    NJson::TJsonValue report = NJson::JSON_ARRAY;
    report.AppendValue(meta->GetReport(*Server));

    g.MutableReport().AddReportElement("codes", std::move(report));
    g.SetCode(HTTP_OK);
}

void TUserHistoryPromoCodeProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    ReqCheckCondition(Server->GetPromoCodesManager(), ConfigHttpStatus.ServiceUnavailable, EDriveLocalizationCodes::InternalServerError);
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::ObserveStructure, TAdministrativeAction::EEntity::PromoCodes);
    const TString userId = GetString(Context->GetCgiParameters(), "user_id", true);
    const TInstant since = GetTimestamp(Context->GetCgiParameters(), "since", TInstant::Zero());

    TVector<IPromoCodeMetaReport::TPtr> codesMeta;
    auto session = BuildTx<NSQL::ReadOnly>();
    ReqCheckCondition(Server->GetPromoCodesManager()->GetUserActivationsHistory(userId, since, codesMeta, session), ConfigHttpStatus.UnknownErrorStatus, EDriveLocalizationCodes::InternalServerError);

    NJson::TJsonValue report = NJson::JSON_ARRAY;
    for (auto&& i : codesMeta) {
        report.AppendValue(i->GetReport(*Server));
    }

    g.MutableReport().AddReportElement("codes", std::move(report));
    g.SetCode(HTTP_OK);
}

void TMarkPromoGivenOutProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    ReqCheckCondition(Server->GetPromoCodesManager(), ConfigHttpStatus.ServiceUnavailable, EDriveLocalizationCodes::InternalServerError);
    ReqCheckCondition(!!requestData["given_out_info"].GetString(), ConfigHttpStatus.SyntaxErrorStatus, EDriveLocalizationCodes::SyntaxUserError);
    const TSet<TString> codeIds = ::MakeSet(GetStrings(requestData, "ids", false));
    const TString generator = GetString(requestData, "generator", codeIds.empty());
    const ui32 count = GetValue<ui32>(requestData, "count", !generator.empty()).GetOrElse(0);
    ReqCheckCondition(!codeIds.empty() || count, ConfigHttpStatus.SyntaxErrorStatus, EDriveLocalizationCodes::SyntaxUserError);

    auto session = BuildTx<NSQL::Writable | NSQL::RepeatableRead>();
    TVector<TPromoCodeMeta> codes;

    if (codeIds) {
        if (!Server->GetPromoCodesManager()->GiveOutCodes(codeIds, requestData["given_out_info"].GetString(), permissions, codes, session)) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
    } else {
        if (!Server->GetPromoCodesManager()->GiveOutCodes(generator, count, requestData["given_out_info"].GetString(), permissions, codes, session)) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
    }
    if (!session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }
    NJson::TJsonValue report = NJson::JSON_ARRAY;
    for (const auto& i : codes) {
        report.AppendValue(i.GetCode());
    }
    g.MutableReport().AddReportElement("codes", std::move(report));
    g.SetCode(HTTP_OK);
}

void TRemovePromocodeProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    ReqCheckCondition(Server->GetPromoCodesManager(), ConfigHttpStatus.ServiceUnavailable, EDriveLocalizationCodes::InternalServerError);
    CheckAdmActions(permissions, TAdministrativeAction::EAction::Remove, TAdministrativeAction::EEntity::PromoCodes);

    const TSet<TString> codeIds = ::MakeSet(GetStrings(requestData, "ids"));
    const bool force = GetValue<bool>(requestData, "force", false).GetOrElse(false);

    auto session = BuildTx<NSQL::Writable | NSQL::RepeatableRead>();
    if (!Server->GetPromoCodesManager()->RemoveCodes(codeIds, force, permissions->GetUserId(), session) || !session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }

    g.SetCode(HTTP_OK);
}

void TCreateReferralCodeProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    ReqCheckCondition(Server->GetPromoCodesManager(), ConfigHttpStatus.ServiceUnavailable, EDriveLocalizationCodes::InternalServerError);

    const TString generator = GetHandlerSettingDef<TString>("generator", "");
    const TString settingsKey = Server->GetSettings().GetValueDef("referral_program.user_settings_field", GetHandlerSettingDef<TString>("settings_key", ""));
    ReqCheckCondition(!!generator && !!settingsKey, ConfigHttpStatus.UnknownErrorStatus, EDriveLocalizationCodes::InternalServerError);

    auto session = BuildTx<NSQL::Writable | NSQL::RepeatableRead>();

    auto stantartContext = GetHandlerSetting<TString>(generator + "_standart_context");
    ReqCheckCondition(stantartContext.Defined(), ConfigHttpStatus.EmptySetStatus, EDriveLocalizationCodes::ObjectNotFound);
    TGeneratorContext context;
    {
        NJson::TJsonValue jsonContext;
        ReqCheckCondition(ReadJsonTree(*stantartContext, &jsonContext) && context.DeserializeFromJson(jsonContext), ConfigHttpStatus.UnknownErrorStatus, EDriveLocalizationCodes::InternalServerError);
    }
    context.SetGivenOut(permissions->GetUserId());

    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Add, TAdministrativeAction::EEntity::PromoCodes, generator);
    auto action = Server->GetDriveAPI()->GetRolesManager()->GetAction(generator);
    ReqCheckCondition(action.Defined(), ConfigHttpStatus.UnknownErrorStatus, EDriveLocalizationCodes::InternalServerError);
    const IPromoCodeGenerator* generatorAction = (*action).GetAs<IPromoCodeGenerator>();
    ReqCheckCondition(generatorAction, ConfigHttpStatus.UnknownErrorStatus, EDriveLocalizationCodes::InternalServerError);

    TDBTag settingsTag = Server->GetDriveAPI()->GetUserSettings(permissions->GetUserId(), session);
    R_ENSURE(settingsTag.HasData(), ConfigHttpStatus.UnknownErrorStatus, "incorrect settings configuration");
    const TString tagName = GetHandlerSettingDef<TString>("tag_setting", settingsTag->GetName());

    const TString customCodeSettingsKey = Server->GetSettings().GetValueDef("referral_program.user_settings_custom_code_field", GetHandlerSettingDef<TString>("custom_code_settings_key", "custom_referral_code"));

    const TString custom = GetString(requestData, "custom_code", /* required = */ false);
    if (custom) {
        {
            NJson::TJsonValue customErrorts;
            const TString customErrortsParam = GetHandlerSettingDef<TString>("custom_errors", "{}");
            ReqCheckCondition(NJson::ReadJsonFastTree(customErrortsParam, &customErrorts), ConfigHttpStatus.UnknownErrorStatus, EDriveLocalizationCodes::InternalServerError);
            TRegExMatch re;
            for (auto&& [error, pattern] : customErrorts.GetMap()) {
                ReqCheckCondition(pattern.IsString(), ConfigHttpStatus.UnknownErrorStatus, EDriveLocalizationCodes::InternalServerError);
                re.Compile(pattern.GetString(), REG_UTF8);
                ReqCheckCondition(re.IsCompiled(), ConfigHttpStatus.UnknownErrorStatus, EDriveLocalizationCodes::InternalServerError);
                ReqCheckCondition(!re.Match(custom.data()), ConfigHttpStatus.SyntaxErrorStatus, error);
            }
        }
        const TSet<char> allowedSymbols = MakeSet(GetHandlerSettingDef<TString>("allowed_symbols", "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789"));
        for (const auto i : custom) {
            ReqCheckCondition(allowedSymbols.contains(i), ConfigHttpStatus.SyntaxErrorStatus, "wrong_symbols");
        }
        auto maxSize = GetHandlerSetting<ui32>("max_custom_code_length").OrElse(10);
        ReqCheckCondition(custom.size() <= maxSize, ConfigHttpStatus.SyntaxErrorStatus, "too_long_code");
        auto minSize = GetHandlerSetting<ui32>("min_custom_code_length").OrElse(5);
        ReqCheckCondition(custom.size() >= minSize, ConfigHttpStatus.SyntaxErrorStatus, "too_short_code");
        if (GetHandlerSettingDef<bool>("use_clear_web", false)) {
            ReqCheckCondition(Server->GetClearWebClient(), ConfigHttpStatus.ServiceUnavailable, EDriveLocalizationCodes::InternalServerError);
            bool clearWebVerdict = false;
            TMessagesCollector errors;
            const bool isReplyOk = Server->GetClearWebClient()->CheckText(custom, clearWebVerdict, errors) / 100 == 2;
            R_ENSURE(isReplyOk || GetHandlerSettingDef<bool>("skip_clear_web_errors", true), ConfigHttpStatus.UnknownErrorStatus, "Fail to check text: " + errors.GetStringReport());
            if (isReplyOk) {
                ReqCheckCondition(clearWebVerdict, ConfigHttpStatus.SyntaxErrorStatus, "wrong_code_content");
            }
        }
        context.MutableCodeBuilder().SetPrefix(custom);
        context.MutableCodeBuilder().SetCountSymbols(0);
        auto sameCodes = Server->GetPromoCodesManager()->GetMetaByCode(custom, session);
        if (!sameCodes) {
            session.DoExceptionOnFail(ConfigHttpStatus, Server->GetLocalization());
        }
        if (FindIfPtr(*sameCodes, [&userId = permissions->GetUserId()](const auto& meta) { return meta.GetGivenOut() == userId; })) {
            if (custom != settingsTag.GetTagAs<TUserDictionaryTag>()->GetField(customCodeSettingsKey).OrElse("")) {
                TUserDictionaryTag::SetSettings(tagName, {std::make_pair(customCodeSettingsKey, custom)}, permissions, "", *Server, session);
                R_ENSURE(session.Commit(), {}, "cannot Commit", session);
            }
            g.SetCode(HTTP_OK);
            return;
        }
    } else {
        if (settingsTag.Is<TUserDictionaryTag>()) {
            const auto customField = settingsTag.GetTagAs<TUserDictionaryTag>()->GetField(customCodeSettingsKey);
            const bool deleteCustom = customField.Defined() && *customField;
            if (deleteCustom) {
                TUserDictionaryTag::SetSettings(tagName, {std::make_pair(customCodeSettingsKey, "")}, permissions, "", *Server, session);
            }
            auto settingsField = settingsTag.GetTagAs<TUserDictionaryTag>()->GetField(settingsKey);
            if (settingsField && *settingsField) {
                R_ENSURE(!deleteCustom || session.Commit(), {}, "cannot Commit", session);
                g.SetCode(HTTP_OK);
                return;
            }
        }
    }

    TVector<TPromoCodeMeta> codes;
    if (!Server->GetPromoCodesManager()->GenerateCodes(1, *generatorAction, context, permissions->GetUserId(), codes, session)) {
        session.DoExceptionOnFail(ConfigHttpStatus, Server->GetLocalization());
    }
    ReqCheckCondition(!!codes.size(), ConfigHttpStatus.EmptySetStatus, EDriveLocalizationCodes::ObjectNotFound);

    const TString& codeKey = custom.empty() ? settingsKey : customCodeSettingsKey;
    TUserDictionaryTag::SetSettings(tagName, {std::make_pair(codeKey, codes.begin()->GetCode())}, permissions, "", *Server, session);

    TString chatId;
    TString chatTopic;
    IChatRobot::ParseTopicLink(GetHandlerSettingDef<TString>("chat_topic_link", ""), chatId, chatTopic);
    IChatRobot::TPtr chatRobot = Server->GetChatRobot(chatId);

    if (!!chatRobot) {
        if (Server->GetChatEngine()->GetDatabaseName() != Server->GetDriveAPI()->GetDatabaseName()) {
            auto chatSession = BuildChatSession();
            if (!chatRobot->EnsureChat(permissions->GetUserId(), chatTopic, chatSession) || chatSession.Commit()) {
                chatSession.DoExceptionOnFail(ConfigHttpStatus);
            }
        } else {
            chatRobot->EnsureChat(permissions->GetUserId(), chatTopic, session);
        }
    }

    R_ENSURE(session.Commit(), {}, "cannot Commit", session);

    if (Server->GetUserDevicesManager()) {
        Y_UNUSED(Server->GetUserDevicesManager()->RegisterEvent(IUserDevicesManager::EEventType::ReferralCodeGeneration, permissions->GetUserId(), Context->GetRequestStartTime()));
    }
    g.SetCode(HTTP_OK);
}

void TGetReferralProgramInfoProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    ReqCheckCondition(Server->GetLocalization(), ConfigHttpStatus.UnknownErrorStatus, EDriveLocalizationCodes::InternalServerError);
    const ILocalization& localization = *(Server->GetLocalization());
    const auto locale = GetLocale();

    TString referralCode;
    bool isCustomCode = false;
    bool enableReferralProgram = false;

    auto referralType = permissions->GetSetting<EReferralType>("referral_type").GetOrElse(EReferralType::Bonuses);

    ui32 totalBonuses = 0;
    {
        auto session = BuildTx<NSQL::ReadOnly>();
        TString customReferralCode;
        enableReferralProgram = Server->GetDriveAPI()->CheckReferralProgramParticipation(referralCode, customReferralCode, *permissions, session);
        if (customReferralCode) {
            referralCode = customReferralCode;
            isCustomCode = true;
        }

        TSet<TString> bonusTags;
        switch (referralType) {
            case EReferralType::Bonuses:
                bonusTags = StringSplitter(GetHandlerSettingDef<TString>("bonus_tags", "")).Split(',').SkipEmpty();
                break;
            case EReferralType::YandexPlus:
                bonusTags = StringSplitter(GetHandlerSettingDef<TString>("referral_bonus_tags", "")).Split(',').SkipEmpty();
                break;
            default:
                break;
        }
        if (bonusTags) {
            TTagEventsManager::TQueryOptions options;
            options.SetTags(bonusTags);
            options.SetActions({EObjectHistoryAction::Remove});
            auto tags = Server->GetDriveAPI()->GetTagsManager().GetUserTags().GetEventsByObject(permissions->GetUserId(), session, 0, TInstant::Zero(), options);
            R_ENSURE(tags, {}, "cannot restore bonus tags", session);
            for (const auto& tag : *tags) {
                auto billingTag = tag.GetTagAs<TBillingTag>();
                if (!billingTag) {
                    continue;
                }
                totalBonuses += billingTag->GetAmount();
            }
        }
    }
    TString reportTemplateStr = GetHandlerSettingDef<TString>("report_template", "{}");

    if (enableReferralProgram) {
        constexpr TStringBuf promoTemplate = "_PROMOCODE_";
        if (referralCode) {
            SubstGlobal(reportTemplateStr, promoTemplate, referralCode);
        }
        constexpr TStringBuf hasReferralTemplate = "_HAS_REFERRAL_CODE_";
        SubstGlobal(reportTemplateStr, hasReferralTemplate, !!referralCode ? "true" : "false");
        constexpr TStringBuf isCustomCodeTemplate = "_IS_CUSTOM_REFERRAL_CODE_";
        SubstGlobal(reportTemplateStr, isCustomCodeTemplate, isCustomCode ? "true" : "false");

        NJson::TJsonValue chatDescription(NJson::JSON_MAP);
        constexpr TStringBuf chatDescriptionTemplate = "_REFERRAL_CHAT_";
        if (referralCode) {
            auto notifier = dynamic_cast<const TNativeChatNotifier*>(Server->GetNotifier(GetHandlerSettingDef<TString>("referral_chat_topic_link", "referral")).Get());
            if (notifier) {
                chatDescription = notifier->GetDescriptor(permissions->GetUserId(), TString());
            }
        }
        SubstGlobal(reportTemplateStr, chatDescriptionTemplate, chatDescription.GetStringRobust());

        constexpr TStringBuf referralTypeTemplate = "_REFERRAL_TYPE_";
        SubstGlobal(reportTemplateStr, referralTypeTemplate, ::ToString(referralType));

        constexpr TStringBuf totalBonusesTemplate = "_TOTAL_BONUSES_";
        TString substBonuses = ::ToString(totalBonuses / 100);
        SubstGlobal(reportTemplateStr, totalBonusesTemplate, substBonuses);

        NJson::TJsonValue report;
        ReqCheckCondition(NJson::ReadJsonFastTree(reportTemplateStr, &report), ConfigHttpStatus.UnknownErrorStatus, EDriveLocalizationCodes::InternalServerError);
        localization.ApplyResourcesForJson(report, locale);
        reportTemplateStr = report.GetStringRobust();

        if (referralCode) {
            SubstGlobal(reportTemplateStr, promoTemplate, referralCode);
        }

        TString linkTemplateStr = GetHandlerSettingDef<TString>("link_template", "yandexdrive://promo/_PROMOCODE_");
        if (referralCode) {
            SubstGlobal(linkTemplateStr, promoTemplate, referralCode);
        }
        if (Server->GetClckClient() && GetHandlerSettingDef<bool>("short_deeplink", true)) {
            TMessagesCollector errors;
            Y_UNUSED(Server->GetClckClient()->Compress(linkTemplateStr, linkTemplateStr, errors));
        }
        SubstGlobal(reportTemplateStr, "_REFERRALLINK_", linkTemplateStr);

        SubstGlobal(reportTemplateStr, totalBonusesTemplate, substBonuses);

        report = NJson::JSON_NULL;
        ReqCheckCondition(NJson::ReadJsonFastTree(reportTemplateStr, &report), ConfigHttpStatus.UnknownErrorStatus, EDriveLocalizationCodes::InternalServerError);

        g.SetExternalReport(std::move(report));
    }
    g.SetCode(HTTP_OK);
}

void TPromoGenerationSchemeReportingProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr /* permissions */, const NJson::TJsonValue& /* requestData */) {
    g.SetExternalReport(NPromoCodes::GetGeneratorContextScheme(*Server).SerializeToJson());
    g.SetCode(HTTP_OK);
}

void TPromoGenerationProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    ReqCheckCondition(Server->GetPromoCodesManager(), ConfigHttpStatus.ServiceUnavailable, EDriveLocalizationCodes::InternalServerError);

    const TString generatorName = GetString(Context->GetCgiParameters(), "generator", true);
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Add, TAdministrativeAction::EEntity::PromoCodes, generatorName);
    auto action = Server->GetDriveAPI()->GetRolesManager()->GetAction(generatorName);
    ReqCheckCondition(action.Defined(), ConfigHttpStatus.UnknownErrorStatus, EDriveLocalizationCodes::InternalServerError);
    const IPromoCodeGenerator* generatorAction = (*action).GetAs<IPromoCodeGenerator>();
    ReqCheckCondition(generatorAction, ConfigHttpStatus.UnknownErrorStatus, EDriveLocalizationCodes::InternalServerError);

    auto codesCount = GetValue<ui32>(Context->GetCgiParameters(), "count", true);
    ReqCheckCondition(codesCount.Defined(), ConfigHttpStatus.SyntaxErrorStatus, EDriveLocalizationCodes::SyntaxUserError);

    TGeneratorContext context;
    auto stantartContext = GetHandlerSetting<TString>(generatorName + "_standart_context");
    if (stantartContext.Defined()) {
        NJson::TJsonValue contextArray;
        for (const auto& param : requestData.GetMap()) {
            contextArray.AppendValue(NJson::TMapBuilder("key", param.first)("value", param.second.GetStringRobust()));
        }
        NJson::TJsonValue jsonContext;
        ReqCheckCondition(ReadJsonTree(*stantartContext, &jsonContext) && context.DeserializeFromJson(jsonContext) && context.SetTagDictionaryContext(contextArray), ConfigHttpStatus.UnknownErrorStatus, EDriveLocalizationCodes::InternalServerError);
    } else {
        ReqCheckCondition(context.DeserializeFromJson(requestData), ConfigHttpStatus.SyntaxErrorStatus, EDriveLocalizationCodes::SyntaxUserError);
    }

    TVector<TPromoCodeMeta> codes;
    auto session = BuildTx<NSQL::Writable | NSQL::RepeatableRead>();
    if (!Server->GetPromoCodesManager()->GenerateCodes(*codesCount, *generatorAction, context, permissions->GetUserId(), codes, session) || !session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus, Server->GetLocalization());
    }

    if (context.GetGivenOut()) {
        NJson::TJsonValue report = NJson::JSON_ARRAY;
        for (const auto& i : codes) {
            report.AppendValue(i.GetCode());
        }
        g.MutableReport().AddReportElement("codes", std::move(report));
    }

    g.SetCode(HTTP_OK);
}

void TAcceptPromoCodeProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    ReqCheckCondition(Server->GetPromoCodesManager(), ConfigHttpStatus.ServiceUnavailable, EDriveLocalizationCodes::InternalServerError);
    ReqCheckCondition(!!Server->GetChatEngine(), ConfigHttpStatus.ServiceUnavailable, EDriveLocalizationCodes::InternalServerError);
    TString code = Strip(Context->GetCgiParameters().Get("code"));
    TString objectId = GetString(Context->GetCgiParameters(), "object_id", false);
    auto sum = GetValue<ui32>(Context->GetCgiParameters(), "sum", false).GetOrElse(0);


    auto session = BuildTx<NSQL::Writable | NSQL::RepeatableRead>();
    auto chatSession = BuildChatSession(false);

    NJson::TJsonValue report;
    IPromoCodesManager::TApplyContext context(code);
    if (sum) {
        context.PaymentSum = sum;
    }
    if (!Server->GetPromoCodesManager()->ApplyCode(context, objectId ? objectId : permissions->GetUserId(), permissions, report, session, chatSession) || !session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus, Server->GetLocalization());
    }
    bool chatSessionCommitted = chatSession.Commit();
    Y_UNUSED(chatSessionCommitted); // TODO: handle error

    g.MutableReport().SetExternalReport(std::move(report));
    g.SetCode(HTTP_OK);
}

TString TGetUserDiscountsProcessor::GetLimitsReport(const TString& localKey, const ITemporaryActionTag& tag, ELocalization locale, const ILocalization& localization) const {
    ui32 shift = 10800;
    TString limits = localization.GetLocalString(locale, GetHandlerLocalization(localKey, "Количество поездок: _attempts__max_attempts_string_\nДействует_since__until_", locale));
    TString maxStr = localization.GetLocalString(locale, GetHandlerLocalization(localKey + ".max_attempts", " из _max_attempts_", locale));
    if (tag.GetMaxAttempts() > 0) {
        SubstGlobal(limits, "_max_attempts_string_", maxStr);
    } else {
        SubstGlobal(limits, "_max_attempts_string_", "");
    }
    SubstGlobal(limits, "_max_attempts_", ToString(tag.GetMaxAttempts()));

    SubstGlobal(limits, "_attempts_", ToString(tag.GetAttempts()));
    TString sinceStr;
    if (tag.GetSince() > Now()) {
        sinceStr = localization.GetLocalString(locale, GetHandlerLocalization(localKey + ".since", " c _date_", locale));
        SubstGlobal(sinceStr, "_date_", localization.FormatInstantWithYear(locale, tag.GetSince() + TDuration::Seconds(shift)));
    }
    SubstGlobal(limits, "_since_", sinceStr);

    TString untilStr;
    if (tag.GetUntil() != TInstant::Max()) {
        untilStr = localization.GetLocalString(locale, GetHandlerLocalization(localKey + ".until", " до _date_", locale));
        SubstGlobal(untilStr, "_date_", localization.FormatInstantWithYear(locale, tag.GetUntil() + TDuration::Seconds(shift)));
    }
    SubstGlobal(limits, "_until_", untilStr);
    return limits;
}

TString TGetUserDiscountsProcessor::CreateDescription(const TString& description, const TString& descriptionKey, const TString& limitsKey, ELocalization locale, const ILocalization& localization, const ITemporaryActionTag* tag) const {
    TString fullDescription = localization.GetLocalString(locale, GetHandlerLocalization(descriptionKey, "_description_", locale));
    SubstGlobal(fullDescription, "_description_", description);
    if (tag) {
        if (fullDescription) {
            fullDescription += "\n";
        }
        fullDescription += GetLimitsReport(limitsKey, *tag, locale, localization);
    }
    return fullDescription;
}

TString TGetUserDiscountsProcessor::CreateDiscountDescription(const TString& description, ELocalization locale, const ILocalization& localization, const ITemporaryActionTag* tag) const {
    return CreateDescription(description, "discount_description", "discount_limits", locale, localization, tag);
}

TString TGetUserDiscountsProcessor::CreatePromoCodeDescription(const TString& description, ELocalization locale, const ILocalization& localization, const ITemporaryActionTag* tag) const {
    return CreateDescription(description, "promo_codes_description", "promo_codes_limits", locale, localization, tag);
}


struct TBonusHistoryEvent {
    ui32 Amount;
    TBillingTagDescription::EAction Type;
    TString Description;
    TInstant Timestamp;
};

template <>
NJson::TJsonValue NJson::ToJson(const TBonusHistoryEvent& object) {
    NJson::TJsonValue json;
    NJson::InsertField(json, "amount", object.Amount);
    NJson::InsertField(json, "type", ToString(object.Type));
    NJson::InsertNonNull(json, "description", object.Description);
    NJson::InsertField(json, "timestamp", NJson::Seconds(object.Timestamp));
    return json;
}

void TGetUserDiscountsProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    NJson::TJsonValue promoCodesReport(NJson::JSON_ARRAY);

    ReqCheckCondition(Server->GetLocalization(), ConfigHttpStatus.UnknownErrorStatus, EDriveLocalizationCodes::InternalServerError);
    const ILocalization& localization = *(Server->GetLocalization());
    const auto locale = GetLocale();

    NJson::TJsonValue discountsReport(NJson::JSON_ARRAY);
    NJson::TJsonValue promoCards(NJson::JSON_ARRAY);

    auto session = BuildTx<NSQL::ReadOnly>();
    TOffersBuildingContext offerContext(Server);
    if (!offerContext.Parse(Context, *permissions, requestData, session)) {
        session.Check();
    }

    {
        TUserOfferContext uoc(Server, permissions, Context);
        uoc.SetLocale(locale);
        offerContext.SetUserHistoryContext(std::move(uoc));
    }

    auto additionalActions = DriveApi->GetRolesManager()->GetUserAdditionalActions(permissions->GetUserId(), DriveApi->GetTagsManager(), true, session);
    R_ENSURE(additionalActions, ConfigHttpStatus.UnknownErrorStatus, "cannot GetUserAdditionalActions", session);

    if (DriveApi->HasBillingManager()) {
        auto accounts = DriveApi->GetBillingManager().GetAccountsManager().GetUserAccounts(permissions->GetUserId(), session);
        R_ENSURE(accounts, ConfigHttpStatus.UnknownErrorStatus, "cannot GetUserAccounts", session);
        auto walletAdditionalActions = DriveApi->GetRolesManager()->GetAccountsAdditionalActions(*accounts, DriveApi->GetTagsManager(), true, session);
        R_ENSURE(walletAdditionalActions, ConfigHttpStatus.UnknownErrorStatus, "cannot GetWalletAdditionalActions", session);
        additionalActions->Join(std::move(*walletAdditionalActions));
    }

    TMap<TString, TSet<TString>> groupOffers;
    for (auto&& i : *additionalActions) {
        const ITemporaryActionTag* taTag = i.GetTag().GetTagAs<ITemporaryActionTag>();
        if (!taTag || !i.GetAction()->GetEnabled()) {
            continue;
        }
        auto correctorPtr = i.GetAction().GetAs<TDiscountOfferCorrector>();
        bool isPublicCorrector = correctorPtr && correctorPtr->GetVisible() && correctorPtr->CheckVisibleBoons(offerContext);

        auto offerPtr = i.GetAction().GetAs<IOfferBuilderAction>();
        bool isPublicOfferBuilder = offerPtr && offerPtr->GetIsPublish() && offerPtr->CheckVisibleBoons(offerContext);

        if (isPublicOfferBuilder && !groupOffers[i.GetTag().GetTagId()].emplace(offerPtr->GetGroupName()).second) {
            continue;
        }

        if (isPublicCorrector || isPublicOfferBuilder) {
            NJson::TJsonValue report = i.GetAction()->GetPublicReport(locale, localization);
            if (taTag->IsPromoCode()) {
                report.InsertValue("full_description", CreatePromoCodeDescription(report["full_description"].GetStringRobust(), locale, localization, taTag));
                promoCodesReport.AppendValue(report);
            } else {
                report.InsertValue("full_description", CreateDiscountDescription(report["full_description"].GetStringRobust(), locale, localization, taTag));
                discountsReport.AppendValue(report);
            }
        }
    }

    TSet<TString> userDiscounts;
    for (const auto& corrector : permissions->GetOfferCorrections()) {
        if (!corrector || !corrector->GetEnabled()) {
            continue;
        }
        userDiscounts.insert(corrector->GetName());
        const TDiscountOfferCorrector* correctorPtr = dynamic_cast<const TDiscountOfferCorrector*>(corrector.Get());
        if (correctorPtr && correctorPtr->CheckVisibleBoons(offerContext)) {
            auto acceptableFlag = permissions->CheckActionAcceptable(*corrector, *Server);
            if (acceptableFlag == IDynamicActionChecker::ECheckResult::EOk) {
                NJson::TJsonValue report = corrector->GetPublicReport(locale, localization);
                report.InsertValue("full_description", CreateDiscountDescription(report["full_description"].GetStringRobust(), locale, localization));
                discountsReport.AppendValue(report);
            } else if (acceptableFlag == IDynamicActionChecker::ECheckResult::EUserFails && correctorPtr->GetHowToGetBlock().GetIsActive()) {
                promoCards.AppendValue(correctorPtr->GetHowToGetBlock().GetPublicReport(locale, localization));
            }
        }
    }

    TSet<TString> availablePromoCards;
    auto availablePromoCardsStr = GetHandlerSetting<TString>("available_promo_cards");
    if (availablePromoCardsStr) {
        availablePromoCards = MakeSet(StringSplitter(*availablePromoCardsStr).Split(',').ToList<TString>());
    }

    for (const auto& corrector : Server->GetDriveAPI()->GetRolesManager()->GetActionsDB().template GetActionsWithType<TDiscountOfferCorrector>()) {
        if (!availablePromoCards.contains(corrector.GetName()) || userDiscounts.contains(corrector.GetName()) || !corrector.GetEnabled() || !corrector.CheckVisibleBoons(offerContext) || !corrector.GetHowToGetBlock().GetIsActive()) {
            continue;
        }
        auto acceptableFlag = permissions->CheckActionAcceptable(corrector, *Server);
        if (acceptableFlag == IDynamicActionChecker::ECheckResult::EOk) {
            promoCards.AppendValue(corrector.GetHowToGetBlock().GetPublicReport(locale, localization));
        }
    }

    g.MutableReport().AddReportElement("discounts", std::move(discountsReport));
    g.MutableReport().AddReportElement("promo_codes", std::move(promoCodesReport));
    g.MutableReport().AddReportElement("promo_cards", std::move(promoCards));

    ReqCheckCondition(Server->GetDriveAPI()->HasBillingManager(), ConfigHttpStatus.UnknownErrorStatus, EDriveLocalizationCodes::InternalServerError);
    g.MutableReport().AddReportElement("bonus_amount", Server->GetDriveAPI()->GetBillingManager().GetAccountsManager().GetBonuses(permissions->GetUserId(), Context->GetRequestStartTime()));

    auto limitedBonuses = Server->GetDriveAPI()->GetBillingManager().GetAccountsManager().GetLimitedBonuses(permissions->GetUserId(), Context->GetRequestStartTime());
    NJson::TJsonValue jsonLimits(NJson::JSON_ARRAY);
    for (const auto& limit : limitedBonuses) {
        const TBillingTagDescription* billingTagDescription = nullptr;
        if (limit.GetSource()) {
            auto desc = Server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetDescriptionByName(limit.GetSource());
            if (!desc) {
                continue;
            }
            billingTagDescription = desc->GetAs<TBillingTagDescription>();
        }
        if (limit.GetDeadline() == TInstant::Max() && !(billingTagDescription && billingTagDescription->GetAmountDescription())) {
            continue;
        }

        NJson::TJsonValue json;
        json.InsertValue("amount", limit.GetBalance());
        if (limit.GetDeadline() != TInstant::Max()) {
            json.InsertValue("expiration_date", limit.GetDeadline().Seconds());
        }
        if (billingTagDescription && billingTagDescription->GetAmountDescription()) {
            json.InsertValue("description", localization.ApplyResources(billingTagDescription->GetAmountDescription(), locale));
        }
        jsonLimits.AppendValue(std::move(json));
    }
    g.MutableReport().AddReportElement("promo_bonuses", std::move(jsonLimits));

    TVector<TBonusHistoryEvent> bonusEvents;
    TSet<TString> bonusAccounts;
    auto registeredAccounts = Server->GetDriveAPI()->GetBillingManager().GetAccountsManager().GetRegisteredAccounts();
    for (auto&& description : registeredAccounts) {
        if (TBillingGlobals::BonusTypes.contains(description.GetType())) {
            bonusAccounts.insert(description.GetName());
        }
    }

    auto bonusesHistoryDeep = GetHandlerSettingDef<TDuration>("bonuses_history_deep", TDuration::Days(30));
    auto bonusesSinceTs = Now() - bonusesHistoryDeep;
    auto bonusTagDescriptions = Server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetRegisteredTags(NEntityTagsManager::EEntityType::User, { TFixedSumTag::TypeName, TOperationTag::TypeName });
    if (bonusTagDescriptions) {
        TTagEventsManager::TQueryOptions options;
        options.SetTags(MakeSet(NContainer::Keys(bonusTagDescriptions)));
        options.SetActions({EObjectHistoryAction::Remove});
        auto tags = Server->GetDriveAPI()->GetTagsManager().GetUserTags().GetEventsByObject(permissions->GetUserId(), session, 0, bonusesSinceTs, options);
        R_ENSURE(tags, {}, "cannot restore bonus tags", session);
        for (const auto& tag : *tags) {
            auto billingTag = tag.GetTagAs<TBillingTag>();
            if (!billingTag) {
                continue;
            }
            auto billingTagDescriptionPtr = bonusTagDescriptions.FindPtr(billingTag->GetName());
            if (!billingTagDescriptionPtr) {
                continue;
            }
            auto billingTagDescription = std::dynamic_pointer_cast<const TBillingTagDescription>(*billingTagDescriptionPtr);
            if (!billingTagDescription || !billingTagDescription->GetAvailableAccounts()) {
                continue;
            }

            if (!bonusAccounts.contains(*billingTagDescription->GetAvailableAccounts().begin())) {
                continue;
            }

            TBonusHistoryEvent bonusEvent{billingTag->GetAmount(), billingTagDescription->GetAction(), billingTagDescription->GetAmountDescription(), tag.GetHistoryTimestamp()};
            bonusEvents.emplace_back(std::move(bonusEvent));
        }
    }

    {
        NSQL::TQueryOptions paymentOptions;
        auto ridingDescriptionStr = localization.GetLocalString(locale, "bonuses_history_riding_description", "Поездка");
        paymentOptions.AddGenericCondition("user_id", permissions->GetUserId());
        TSet<TString> typesStr;
        for(const auto& id : TBillingGlobals::BonusTypes) {
            typesStr.emplace(::ToString(id));
        }
        paymentOptions.SetGenericCondition("payment_type", typesStr);
        paymentOptions.SetGenericCondition("created_at_ts", TRange<ui64>(bonusesSinceTs.Seconds()));
        auto payments = Server->GetDriveAPI()->GetBillingManager().GetPaymentsManager().GetPayments(session, paymentOptions);
        R_ENSURE(payments, {}, "cannot restore payments", session);
        for (const auto& payment : *payments) {
            if (payment.GetCleared() > 0) {
                TBonusHistoryEvent bonusEvent{payment.GetCleared(), TBillingTagDescription::EAction::Debit, ridingDescriptionStr, payment.GetCreatedAt()};
                bonusEvents.emplace_back(std::move(bonusEvent));
            }
        }
    }
    std::sort(bonusEvents.begin(), bonusEvents.end(), [](TBonusHistoryEvent a, TBonusHistoryEvent b) -> bool {
        return a.Timestamp > b.Timestamp;
    });

    auto bonusEventsJson = NJson::ToJson(bonusEvents);
    localization.ApplyResourcesForJson(bonusEventsJson, locale);
    g.MutableReport().AddReportElement("bonuses_history", std::move(bonusEventsJson));
    auto bonusesBanner = GetHandlerSettingDef<TString>("bonuses_history_banner", "");
    if (bonusesBanner) {
        g.MutableReport().AddReportElement("bonuses_history_banner", localization.ApplyResources(bonusesBanner, locale));
    }

    auto additionalFieldsStr = GetHandlerSetting<TString>("additional_fields");
    TMap<TString, TVector<TString>> fieldPaths;
    if (additionalFieldsStr) {
        NJson::TJsonValue additionalFieldsJson;
        if (NJson::ReadJsonFastTree(*additionalFieldsStr, &additionalFieldsJson)) {
            for (auto&& field : additionalFieldsJson["fields"].GetArray()) {
                TString fieldName;
                TVector<TString> fieldParts;
                if (!NJson::ParseField(field, "name", fieldName, true) ||
                    !NJson::ParseField(field, "values", fieldParts, true)) {
                    ERROR_LOG << "GetUserDiscountsProcessor skipping additional field " << field.GetStringRobust() << Endl;
                    continue;
                }
                fieldPaths[fieldName] = std::move(fieldParts);
            }
        }
    }
    for (auto&& [name, parts] : fieldPaths) {
        NJson::TJsonValue section = NJson::JSON_ARRAY;
        for (auto&& part : parts) {
            auto actionFlag = GetHandlerSetting<TString>(name + "." + part + ".flag");
            if (!actionFlag || permissions->GetFlagsReport()[*actionFlag].GetBooleanRobust()) {
                auto descriptionStr = GetHandlerSetting<TString>(name + "." + part + ".description");
                NJson::TJsonValue description;
                if (descriptionStr && NJson::ReadJsonFastTree(*descriptionStr, &description)) {
                    auto chatName = GetHandlerSetting<TString>(name + "." + part + ".chat");
                    if (chatName) {
                        NJson::TJsonValue chatDescription = NJson::JSON_MAP;
                        auto notifier = dynamic_cast<const TNativeChatNotifier*>(Server->GetNotifier(*chatName).Get());
                        if (notifier) {
                            NJson::TJsonValue chatDescription = notifier->GetDescriptor(permissions->GetUserId(), TString());
                            description["chat"] = chatDescription;
                        }
                    }
                    section.AppendValue(description);
                }
            }
        }
        if (section.GetArray().size() > 0) {
            g.MutableReport().AddReportElement(name, std::move(section));
        }
    }
    g.SetCode(HTTP_OK);
}

class TUsageExternalPromo {
public:
    void SetUsage(bool reserved, ui64 count) {
        if (reserved) {
            Used = count;
        } else {
            Unused = count;
        }
    }

    NJson::TJsonValue ToJson() const {
        NJson::TJsonValue json;
        json.InsertValue("reserved", Used);
        json.InsertValue("all", (ui64)Unused + Used);
        return json;
    }

private:
    ui32 Used = 0;
    ui32 Unused = 0;
};

template <>
NJson::TJsonValue NJson::ToJson<TUsageExternalPromo>(const TUsageExternalPromo& object) {
    return object.ToJson();
}

void TGetExternalPromoReportProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::ExternalPromo);

    auto session = BuildTx<NSQL::ReadOnly>();
    auto transaction = session.GetTransaction();
    TRecordsSet recordSet;
    TQueryResultPtr queryResult = transaction->Exec("SELECT type, reserved, COUNT(1) as count FROM " + Config.GetTableName() + " group by type, reserved", &recordSet);
    ReqCheckCondition(queryResult->IsSucceed(), ConfigHttpStatus.UnknownErrorStatus, transaction->GetErrors().GetStringReport());
    TMap<TString, TUsageExternalPromo> externalPromo;
    for (const auto& record : recordSet) {
        bool reserved = false;
        ui64 count = 0;
        TString type;
        ReqCheckCondition(record.TryGet("type", type) && record.TryGet("reserved", reserved) && record.TryGet("count", count), ConfigHttpStatus.UnknownErrorStatus, "incorrect record " + record.BuildSet(*transaction));
        externalPromo[type].SetUsage(reserved, count);
    }

    g.MutableReport().AddReportElement("info", NJson::ToJson(NJson::Dictionary(externalPromo)));
    g.SetCode(HTTP_OK);
}

void TUploadExternalPromoProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Add, TAdministrativeAction::EEntity::ExternalPromo);
    const auto& codesArray = requestData["codes"].GetArray();
    const TString deeplinkPrefix = GetString(requestData, "deeplink_prefix", false);
    const TString type = GetString(requestData, "type", false);

    TRecordsSet recordSet;
    for (const auto& codeJson : codesArray) {
        NStorage::TTableRecord record;
        ReqCheckCondition(record.DeserializeFromJson(codeJson), ConfigHttpStatus.UnknownErrorStatus, "incorrect row " + codeJson.GetStringRobust());
        const TString& code = record.Get("code");
        ReqCheckCondition(!!code, ConfigHttpStatus.SyntaxErrorStatus, "incorrect code value " + codeJson.GetStringRobust());
        if (type) {
            record.ForceSet("type", type);
        }
        if (deeplinkPrefix) {
            record.ForceSet("deeplink", deeplinkPrefix + code);
        }
        recordSet.AddRow(std::move(record));
    }

    auto session = BuildTx<NSQL::Writable>();
    auto promoTable = session->GetDatabase().GetTable(Config.GetTableName());
    ReqCheckCondition(!!promoTable, ConfigHttpStatus.UnknownErrorStatus, "incorrect external promo table");

    auto transaction = session.GetTransaction();

    TQueryResultPtr queryResult = promoTable->AddRows(recordSet, transaction);
    ReqCheckCondition(queryResult->IsSucceed(), ConfigHttpStatus.UnknownErrorStatus, transaction->GetErrors().GetStringReport());

    if (!session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }

    g.SetCode(HTTP_OK);
}
