#include "processor.h"

#include <drive/backend/clickhouse/client.h>
#include <rtline/util/algorithm/type_traits.h>

ui64 GetAggregationSeconds(const TString& aggregation) {
    static constexpr ui64 secondsInDay = TDuration::Days(1).Seconds();
    static constexpr ui64 secondsInWeek = TDuration::Days(7).Seconds();
    static constexpr ui64 secondsInMonth = TDuration::Days(30).Seconds();
    if (aggregation == "week") {
        return secondsInWeek;
    } else if (aggregation == "month") {
        return secondsInMonth;
    }
    return secondsInDay;
}

void TOrdersSummaryProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    const TCgiParameters& cgi = Context->GetCgiParameters();
    const auto deadline = Context->GetRequestDeadline();
    const auto maxCarsOutput = GetHandlerSetting<int>("max_cars_can_be_extracted").GetOrElse(5000);
    const auto checkVisibility = GetHandlerSetting<bool>("check_visibility").GetOrElse(true);
    const auto mainTariff = GetHandlerSetting<TString>("main_tariff").GetOrElse("");
    static constexpr ui64 secondsInHours = TDuration::Hours(1).Seconds();

    const auto numdoc = GetValue<ui32>(cgi, "limit", false).GetOrElse(maxCarsOutput);
    const auto objectId = GetString(cgi, "car_id", false);
    const auto now = Context->GetRequestStartTime();
    const auto since = GetTimestamp(cgi, "since", now - TDuration::Days(15));
    const auto until = GetTimestamp(cgi, "until", now - TDuration::Days(1));
    const auto tagsFilter = GetString(cgi, "tags_filter", false);
    const auto aggregation = GetString(cgi, "aggregation", false);
    const auto tariff = GetValue<TString>(cgi, "tariff", false).GetOrElse(mainTariff);
    const auto delta = until.Seconds() - since.Seconds();
    auto timestamp = MakeRange<TInstant>(
        since,
        until + TDuration::Seconds(1)
    );

    auto ridesSummaryClient = Server->GetRidesInfoClient();
    R_ENSURE(ridesSummaryClient, HTTP_INTERNAL_SERVER_ERROR, "ridesSummaryClient is not configured");
    TReportTraits reportTraits = permissions->GetDeviceReportTraits();
    TVector<TString> carIds;
    if (!objectId) {
        TCarsFetcher carsFetcher(*Server, reportTraits & NDeviceReport::ReportLeasing, Context->GetRequestDeadline());
        carsFetcher.SetLimit(numdoc);
        carsFetcher.SetIsRealtime(true);
        carsFetcher.SetCheckVisibility(checkVisibility);
        if (tagsFilter) {
            carsFetcher.MutableCarTagsFilter() = TTagsFilter();
            R_ENSURE(
                carsFetcher.MutableCarTagsFilterRef().DeserializeFromString(tagsFilter),
                ConfigHttpStatus.SyntaxErrorStatus,
                "incorrect tags filter"
            );
        }
        R_ENSURE(carsFetcher.FetchData(permissions, nullptr), ConfigHttpStatus.UnknownErrorStatus, "cannot fetch data for report");
        g.AddReportElement("statuses", std::move(carsFetcher.GetStatusesReportSafe()));
        carIds = MakeVector(carsFetcher.GetCarsIds());
    } else {
        carIds.push_back(objectId);
    }
    const size_t carIdsSize = carIds.size();
    g.AddReportElement("car_count", carIdsSize);

    auto ordersInfoFuture = ridesSummaryClient->Get(NDrive::TClickHouseBackendClient::TOrdersSummaryQuery()
        .SetObjectIds(carIds)
        .SetTimestamp(timestamp)
        .SetTariff(tariff)
        .SetAggregationType(aggregation)
        , deadline
    );

    R_ENSURE(ordersInfoFuture.Wait(deadline), HTTP_GATEWAY_TIME_OUT, "wait timeout");
    auto ordersInfoResult = ordersInfoFuture.GetValue();

    TMap<ui64, NDrive::TOrdersSumaryBase> summaryByTimeStatistic;
    NDrive::TOrdersSumaryBase totalOrdersSummary;
    TMap<TString, NDrive::TOrdersSumaryBase> resultForCar;
    TMap<TString, int> adonSales;
    TMap<TString, ui64> InsuranceSales;
    TString currency;
    const auto secondsInDayWeekMonth = GetAggregationSeconds(aggregation);
    for (auto&& [carId, orders] : ordersInfoResult) {
        for (const auto& order : orders) {
            const auto secondsTimestamp = order.Timestamp.Seconds();
            auto nextTimePeriod = secondsTimestamp + secondsInDayWeekMonth;
            auto realTimePeriod = secondsTimestamp;
            if (nextTimePeriod >= order.FinishLongOrderTimestamp) {
                summaryByTimeStatistic[secondsTimestamp] += order;
            } else {
                summaryByTimeStatistic[secondsTimestamp].Duration += nextTimePeriod - order.StartLongOrderTimestamp;
                realTimePeriod = nextTimePeriod;
                nextTimePeriod += secondsInDayWeekMonth;
                while(nextTimePeriod < order.FinishLongOrderTimestamp && realTimePeriod <= until.Seconds() ) {
                    summaryByTimeStatistic[realTimePeriod].Duration += secondsInDayWeekMonth;
                    realTimePeriod = nextTimePeriod;
                    nextTimePeriod += secondsInDayWeekMonth;
                }
                if (realTimePeriod <= until.Seconds()) {
                    summaryByTimeStatistic[realTimePeriod].Duration += order.FinishLongOrderTimestamp - realTimePeriod;
                }
                summaryByTimeStatistic[secondsTimestamp] += order;
                summaryByTimeStatistic[secondsTimestamp].Duration -= order.Duration;
            }
            totalOrdersSummary += order;
            resultForCar[std::move(carId)] += order;

            if (!order.InsuranceType.empty()) {
                InsuranceSales[order.InsuranceType] += order.Count;
            }
            currency = order.Currency;
        }
    }

    NJson::TJsonValue summaryByTimeStatisticList = NJson::JSON_ARRAY;
    for (auto&& [time, result] : summaryByTimeStatistic) {
        NJson::TJsonValue timeStatisticJson;
        timeStatisticJson["timestamp"] = time;
        NJson::TJsonValue valueJson;
        valueJson["revenue"] = result.TotalPayment;
        valueJson["bookings"] = result.Count;
        valueJson["total_mileage"] = result.Mileage;
        const double ridingDuration = static_cast<double>(result.RidingDuration);
        const double duration = static_cast<double>(result.Duration);
        valueJson["total_engine_hours"] = ridingDuration / secondsInHours;
        valueJson["total_duration_hours"] = duration / secondsInHours;
        if (!carIds.empty()) {
            valueJson["rental_utilization"] = CalculateUtilization(duration, secondsInDayWeekMonth, carIdsSize);
            valueJson["fleet_utilization"] = CalculateUtilization(ridingDuration, secondsInDayWeekMonth, carIdsSize);
        }
        if (result.Count > 0) {
            valueJson["average_check"] = static_cast<double>(result.TotalPayment) / result.Count;
        }
        timeStatisticJson["value"] = std::move(valueJson);
        summaryByTimeStatisticList.AppendValue(std::move(timeStatisticJson));
    }
    g.AddReportElement("gistogram_value", std::move(summaryByTimeStatisticList));

    const double totalDuration = static_cast<double>(totalOrdersSummary.Duration);
    const double totalRidingDuration = static_cast<double>(totalOrdersSummary.RidingDuration);
    NJson::TJsonValue totalStatisticJson;
    auto locale = GetLocale();
    totalStatisticJson["currency"] = Server->GetLocalization()->GetLocalString(locale, currency);
    totalStatisticJson["revenue_total_for_the_period"] = totalOrdersSummary.TotalPayment;
    totalStatisticJson["bookings_total_for_the_period"] = totalOrdersSummary.Count;
    totalStatisticJson["mileage_total_for_the_period"] = totalOrdersSummary.Mileage;
    totalStatisticJson["engine_hours_total_for_the_period"] = totalRidingDuration / secondsInHours;
    totalStatisticJson["duration_hours_total_for_the_period"] = totalDuration / secondsInHours;
    totalStatisticJson["rentals"] = totalDuration / secondsInDayWeekMonth;
    if (delta > 0) {
        const auto timeInterval = (static_cast<double>(delta) / secondsInDayWeekMonth);
        totalStatisticJson["mileage_avg_per_time"] = totalOrdersSummary.Mileage / timeInterval;
        totalStatisticJson["bookings_avg_per_time"] = totalOrdersSummary.Count / timeInterval;
        totalStatisticJson["engine_hours_avg_per_time"] = (totalRidingDuration / timeInterval) / secondsInHours;
        totalStatisticJson["duration_hours_avg_per_time"] = (totalDuration / timeInterval) / secondsInHours;
        totalStatisticJson["revenue_avg_per_time"] = totalOrdersSummary.TotalPayment / timeInterval;
        if (!carIds.empty()) {
            totalStatisticJson["fleet_utilization_total_for_the_period"] = CalculateUtilization(totalRidingDuration, delta, carIdsSize);
            totalStatisticJson["rental_utilization_total_for_the_period"] = CalculateUtilization(totalDuration, delta, carIdsSize);
        }
    }
    if (totalOrdersSummary.Count > 0) {
        totalStatisticJson["rentals_avg_per_time"] = (totalDuration / secondsInDayWeekMonth) / totalOrdersSummary.Count;
        totalStatisticJson["average_check"] = static_cast<double>(totalOrdersSummary.TotalPayment) / totalOrdersSummary.Count;
    }
    g.AddReportElement("total_values", std::move(totalStatisticJson));

    NJson::TJsonValue addonSalesJson;
    addonSalesJson["entry_to_eco_zones_in_germany"] = totalOrdersSummary.AddonSalesCount.EcoGermanyZones;
    addonSalesJson["snow_chains"] = totalOrdersSummary.AddonSalesCount.SnowChains;
    addonSalesJson["gps"] = totalOrdersSummary.AddonSalesCount.Gps;
    addonSalesJson["roof_rack"] = totalOrdersSummary.AddonSalesCount.RoofRack;
    addonSalesJson["child_seat"] = totalOrdersSummary.AddonSalesCount.ChildSeat;
    g.AddReportElement("addon_sales", std::move(addonSalesJson));

    NJson::TJsonValue insuranceSalesList = NJson::JSON_ARRAY;
    for (auto&& [type, count] : InsuranceSales) {
        NJson::TJsonValue InsuranceSalesJson;
        InsuranceSalesJson[type] = count;
        insuranceSalesList.AppendValue(std::move(InsuranceSalesJson));
    }
    g.AddReportElement("insurance_sales", std::move(insuranceSalesList));

    NJson::TJsonValue oneCarList = NJson::JSON_ARRAY;
    auto modelNameCache = Server->GetDriveDatabase().GetModelsDB().GetCached();
    for (auto&& [carId, result] : resultForCar) {
        NJson::TJsonValue oneCarJson;
        auto carObject = Yensured(Server)->GetDriveDatabase().GetCarManager().GetObject(carId);
        oneCarJson["object_id"] = carId;
        auto modelId = carObject->GetModel();
        oneCarJson["model_id"] = modelId;
        oneCarJson["number"] = carObject->GetNumber();
        auto modelNameResult = modelNameCache.GetResult();
        oneCarJson["name"] = modelNameResult[modelId].GetName();
        NJson::TJsonValue value;
        value["mileage"] = result.Mileage;
        value["revenue"] = result.TotalPayment;
        if (result.Count > 0) {
            value["average_check"] = static_cast<double>(result.TotalPayment) / result.Count;
        }
        value["riding_duration_hours"] = static_cast<double>(result.RidingDuration) / secondsInHours;
        value["duration_hours"] = static_cast<double>(result.RidingDuration) / secondsInHours;
        if (delta > 0) {
            value["rental_utilization"] = static_cast<double>(result.Duration) / delta;
            value["fleet_utilization"] = static_cast<double>(result.RidingDuration) / delta;
        }
        oneCarJson["values"] = std::move(value);
        oneCarList.AppendValue(std::move(oneCarJson));
    }
    g.AddReportElement("result_by_car", std::move(oneCarList));

    g.SetCode(HTTP_OK);
}

double TOrdersSummaryProcessor::CalculateUtilization(const double duration, const ui64 realTime, const ui64 count) const {
    R_ENSURE(realTime != 0, HTTP_INTERNAL_SERVER_ERROR, "Division by zero");
    R_ENSURE(count != 0, HTTP_INTERNAL_SERVER_ERROR, "Division by zero");
    return (duration / realTime) / count;
}

void TMileageSummaryProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    const TCgiParameters& cgi = Context->GetCgiParameters();
    const auto deadline = Context->GetRequestDeadline();
    const auto maxCarsOutput = GetHandlerSetting<int>("max_cars_can_be_extracted").GetOrElse(5000);

    const auto numdoc = GetValue<ui32>(cgi, "limit", false).GetOrElse(maxCarsOutput);
    const auto objectId = GetString(cgi, "car_id", false);
    const auto now = Context->GetRequestStartTime();
    const auto since = GetTimestamp(cgi, "since", now - TDuration::Days(15));
    const auto until = GetTimestamp(cgi, "until", now - TDuration::Days(1));
    const auto tagsFilter = GetString(cgi, "tags_filter", false);
    const auto aggregation = GetString(cgi, "aggregation", false);
    auto timestamp = MakeRange<TInstant>(
        since,
        until + TDuration::Seconds(1)
    );

    auto ridesSummaryClient = Server->GetRidesInfoClient();
    R_ENSURE(ridesSummaryClient, HTTP_INTERNAL_SERVER_ERROR, "ridesSummaryClient is not configured");
    TReportTraits reportTraits = permissions->GetDeviceReportTraits();
    TVector<TString> carIds;
    if (!objectId) {
        TCarsFetcher carsFetcher(*Server, reportTraits & NDeviceReport::ReportLeasing, Context->GetRequestDeadline());
        carsFetcher.SetLimit(numdoc);
        carsFetcher.SetIsRealtime(true);
        carsFetcher.SetCheckVisibility(true);
        if (tagsFilter) {
            carsFetcher.MutableCarTagsFilter() = TTagsFilter();
            R_ENSURE(
                carsFetcher.MutableCarTagsFilterRef().DeserializeFromString(tagsFilter),
                ConfigHttpStatus.SyntaxErrorStatus,
                "incorrect tags filter"
            );
        }
        R_ENSURE(carsFetcher.FetchData(permissions, nullptr), ConfigHttpStatus.UnknownErrorStatus, "cannot fetch data for report");
        carIds = MakeVector(carsFetcher.GetCarsIds());
    } else {
        carIds.push_back(objectId);
    }

    auto mileageSummaryFuture = ridesSummaryClient->Get(NDrive::TClickHouseBackendClient::TMileageSummaryQuery()
        .SetObjectIds(carIds)
        .SetTimestamp(timestamp)
        .SetAggregationType(aggregation)
        , deadline
    );
    R_ENSURE(mileageSummaryFuture.Wait(deadline), HTTP_GATEWAY_TIME_OUT, "wait timeout");
    auto mileageSummaryResult = mileageSummaryFuture.GetValue();

    TMap<ui64, float> mileageByTime;
    float totalMileage = 0.0;
    TMap<TString, float> resultForCar;
    for (auto&& [carId, records] : mileageSummaryResult) {
        for (const auto& record : records) {
            const auto secondsTimestamp = record.Timestamp.Seconds();
            mileageByTime[secondsTimestamp] += record.Mileage;
            totalMileage += record.Mileage;
            resultForCar[std::move(carId)] += record.Mileage;
        }
    }

    NJson::TJsonValue mileageByTimeList = NJson::JSON_ARRAY;
    for (auto&& [time, result] : mileageByTime) {
        NJson::TJsonValue timeStatisticJson;
        timeStatisticJson["timestamp"] = time;
        timeStatisticJson["total_mileage"] = result;
        mileageByTimeList.AppendValue(std::move(timeStatisticJson));
    }
    g.AddReportElement("gistogram_value", std::move(mileageByTimeList));

    const auto secondsInDayWeekMonth = GetAggregationSeconds(aggregation);
    NJson::TJsonValue totalMileageJson;
    totalMileageJson["mileage_total_for_the_period"] = totalMileage;
    if (since != until) {
        const auto timeInterval = (static_cast<double>(until.Seconds() - since.Seconds()) / secondsInDayWeekMonth);
        totalMileageJson["mileage_avg_per_time"] = totalMileage / timeInterval;
    }
    g.AddReportElement("total", std::move(totalMileageJson));

    NJson::TJsonValue oneCarList = NJson::JSON_ARRAY;
    auto modelNameCache = Server->GetDriveDatabase().GetModelsDB().GetCached();
    for (auto&& [carId, result] : resultForCar) {
        NJson::TJsonValue oneCarJson;
        auto carObject = Yensured(Server)->GetDriveDatabase().GetCarManager().GetObject(carId);
        oneCarJson["object_id"] = carId;
        auto modelId = carObject->GetModel();
        oneCarJson["model_id"] = modelId;
        oneCarJson["number"] = carObject->GetNumber();
        auto modelNameResult = modelNameCache.GetResult();
        oneCarJson["name"] = modelNameResult[modelId].GetName();
        oneCarJson["mileage"] = result;
        oneCarList.AppendValue(std::move(oneCarJson));
    }
    g.AddReportElement("result_by_car", std::move(oneCarList));

    g.SetCode(HTTP_OK);
}
