#include "landmarks.h"

#include "helpers.h"

#include <drive/backend/offers/context.h>
#include <drive/backend/saas/api.h>
#include <drive/library/cpp/maps_router/router.h>

#include <library/cpp/string_utils/quote/quote.h>
#include <library/cpp/yconf/conf.h>

#include <rtline/library/geometry/polyline.h>
#include <rtline/library/geometry/rect.h>
#include <rtline/library/graph/protos/report.pb.h>
#include <rtline/protos/proto_helper.h>

bool TLandmarkObject::operator==(const TLandmarkObject& other) const {
    return (Coord.GetLengthTo(other.Coord) < 1e-5) && Name == other.Name;
}

void TLandmarkObject::ToJson(NJson::TJsonValue& json) const {
    json["place"]["category"] = Category;
    json["place"]["point"] = TGeoCoord::SerializeVector(TVector<TGeoCoord>({ Coord }));
    json["place"]["name"] = Name;
}

template <class T>
bool FillFromReport(const NMetaProtocol::TReport& rtyReport, const THttpStatusManagerConfig& configHttpStatus, T& output, TJsonReport::TGuard& g) {
    TJsonReport& report = g.MutableReport();

    if (rtyReport.GroupingSize() != 1) {
        report.AddReportElement("error", "Incorrect reply (grouping size != 1)");
        return false;
    }

    const ::NMetaProtocol::TGrouping& grouping = rtyReport.GetGrouping(0);
    if (grouping.GroupSize() == 0) {
        g.SetCode(configHttpStatus.EmptySetStatus);
        return true;
    }

    for (auto&& grouping : rtyReport.GetGrouping()) {
        for (auto&& group : grouping.GetGroup()) {
            for (auto&& d : group.GetDocument()) {
                TLandmarkObject object;

                TReadSearchProtoHelper helper(d);
                TString coord;
                if (!helper.GetProperty("entrance_coord", coord) || !ParsePoint(coord, object.Coord)) {
                    report.AddReportElement("error", "Incorrect reply (incorrect 'entrance_coord')");
                    return false;
                }

                if (!helper.GetProperty("Name", object.Name)) {
                    report.AddReportElement("error", "Incorrect reply (incorrect 'name')");
                    return false;
                }

                helper.GetProperty("category", object.Category);
                output.AddPoint(object);
            }
        }
    }
    return true;
}

void TLandmarksProcessorConfig::InitFeatures(const TYandexConfig::Section* section) {
    RTLineAPIName = section->GetDirectives().Value("RTLineAPIName", RTLineAPIName);
    DefaultDocsCount = section->GetDirectives().Value("DefaultDocsCount", DefaultDocsCount);
    DefaultSearchDistance = section->GetDirectives().Value("DefaultSearchDistance", DefaultSearchDistance);
    DynamicSort = section->GetDirectives().Value("DynamicSort", DynamicSort);
    FilterDistance = section->GetDirectives().Value("FilterDistance", FilterDistance);
    SubwayRouting = section->GetDirectives().Value("SubwayRouting", SubwayRouting);;
}

void TLandmarksProcessorConfig::ToStringFeatures(IOutputStream& os) const {
    os << "RTLineAPIName: " << RTLineAPIName << Endl;
    os << "DefaultDocsCount: " << DefaultDocsCount << Endl;
    os << "DefaultSearchDistance: " << DefaultSearchDistance << Endl;
    os << "FilterDistance: " << FilterDistance << Endl;
    os << "DynamicSort: " << DynamicSort << Endl;
    os << "SubwayRouting: " << SubwayRouting << Endl;
}

void TMarkedRouteProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    Y_UNUSED(permissions);
    const auto& configHttpStatus = Server->GetHttpStatusManagerConfig();
    TJsonReport& report = g.MutableReport();
    const TCgiParameters& cgi = Context->GetCgiParameters();
    TMaybe<double> fromLon = GetValue<double>(cgi, "from_lon", true);
    TMaybe<double> fromLat = GetValue<double>(cgi, "from_lat", true);
    TMaybe<double> toLon = GetValue<double>(cgi, "to_lon", true);
    TMaybe<double> toLat = GetValue<double>(cgi, "to_lat", true);
    R_ENSURE(fromLon, HTTP_BAD_REQUEST, "invalid value of 'from_lon'");
    R_ENSURE(fromLat, HTTP_BAD_REQUEST, "invalid value of 'from_lat'");
    R_ENSURE(toLon, HTTP_BAD_REQUEST, "invalid value of 'to_lon'");
    R_ENSURE(toLat, HTTP_BAD_REQUEST, "invalid value of 'to_lat'");

    const TString& subwayStr = cgi.Get("use_subway");
    bool useSubwayRouting = subwayStr ? IsTrue(subwayStr) : Config.UseSubwayRouting();

    const auto& sClient = Server->GetRTLineAPI(Config.GetRTLineAPIName())->GetSearchClient();
    NRTLine::TQuery query;
    TStringBuilder qText;
    qText << "type:marked_route;d:100;from:" << fromLon << " " << fromLat << ";to:" << toLon << " " << toLat;
    if (useSubwayRouting) {
        qText << ";perm:Transport;algorithm:layer_p2p";
    } else {
        qText << ";perm:ptPedestrian";
    }
    DEBUG_LOG << qText << Endl;
    query.SetText(qText);
    query.AddExtraParams("&component=Graph");

    if(useSubwayRouting) {
        query.AddExtraParams("&pron=packed_report10");
    }
    auto router = Server->GetPedestrianRouter();
    if (router && permissions->GetSetting<bool>("pedestrian_router.enabled", false)) {
        TGeoCoord from(*fromLon, *fromLat);
        TGeoCoord to(*toLon, *toLat);
        auto asyncRoute = router->GetRoute(from, to);
        asyncRoute.Wait(Context->GetRequestDeadline());
        R_ENSURE(!asyncRoute.HasException(), HTTP_INTERNAL_SERVER_ERROR, NThreading::GetExceptionMessage(asyncRoute));
        R_ENSURE(asyncRoute.HasValue(), HTTP_INTERNAL_SERVER_ERROR, "unable to fetch route");
        auto route = asyncRoute.GetValue();
        R_ENSURE(route, HTTP_NOT_FOUND, "no route found");
        TStringBuilder trace;
        for (auto&& element : route->Elements) {
            for (auto&& point : element.Points) {
                if (!trace.empty()) {
                    trace << " ";
                }
                trace << point.X << " " << point.Y;
            }
        }
        report.AddReportElement("route", trace);
        report.AddReportElement("time", route->Time);
        g.SetCode(HTTP_OK);
        return;
    }

    NRTLine::TSearchReply reply = sClient.SendQuery(query);

    if (reply.GetCode() == configHttpStatus.EmptySetStatus) {
        g.SetCode(configHttpStatus.EmptySetStatus);
        return;
    }

    if (reply.GetCode() != 200) {
        report.AddReportElement("error", "Incorrect response from rtline cluster " + ::ToString(reply.GetCode()) + ": " + reply.GetRawReport());
        return;
    }

    if (reply.GetReport().GroupingSize() < 1) {
        report.AddReportElement("error", "Incorrect reply (grouping size < 1)");
        return;
    }

    TFilteredObjects<TLandmarkObject> landmarks(Config.GetFilterDistance());

    const auto pedestrianSpeedKmh = permissions->GetSetting<double>("offers.pedestrian_speed_km/h", 4.2);
    for (auto&& grouping : reply.GetReport().GetGrouping()) {
        for (auto&& group : grouping.GetGroup()) {
            if (group.GetCategoryName() == "Landmarks") {
                for (auto&& d : group.GetDocument()) {
                    TLandmarkObject object;

                    TReadSearchProtoHelper helper(d);
                    TString coord;
                    if (!helper.GetProperty("entrance_coord", coord) || !ParsePoint(coord, object.Coord)) {
                        report.AddReportElement("error", "Incorrect reply (incorrect 'entrance_coord')");
                        return;
                    }

                    if (!helper.GetProperty("Name", object.Name)) {
                        report.AddReportElement("error", "Incorrect reply (incorrect 'name')");
                        return;
                    }

                    helper.GetProperty("category", object.Category);
                    landmarks.AddPoint(object);
                }
            } else if (group.GetCategoryName() == "Routes") {
                CHECK_WITH_LOG(group.DocumentSize() > 0);
                auto&& doc = group.GetDocument(0);
                TReadSearchProtoHelper helper(doc);


                if (useSubwayRouting) {
                    TString packedReport;
                    if (!helper.GetProperty("packed_segments", packedReport)) {
                        report.AddReportElement("error", "Incorrect reply (incorrect 'packed_segments')");
                        continue;
                    }

                    NRTYGeometryGraph::TReportSegment reportSegment;
                    if (!reportSegment.ParseFromString(packedReport)) {
                        report.AddReportElement("error", "cannot deserialize NRTYGeometryGraph::TReportSegment");
                        continue;
                    }

                    ui32 accessType = 0;
                    NJson::TJsonValue jsonReport;

                    TVector<TGeoCoord> path;
                    ui32 duration = 0;
                    for (ui32 cIdx = 0; cIdx < reportSegment.CoordSize(); ++cIdx) {
                        auto&& segment = reportSegment.GetCoord(cIdx);
                        CHECK_WITH_LOG(segment.HasCoord());
                        ui32 featuresPack = segment.GetFeaturesPack();
                        TGeoCoord c;
                        c.Deserialize(segment.GetCoord());
                        if ((featuresPack & NRTYGeometryGraph::ESpecialFeatures::ACCESS_TYPE_SWITCH) != 0
                            || cIdx == reportSegment.CoordSize() - 1) {
                            if (!path.empty()) {
                                NJson::TJsonValue pathElement;
                                pathElement["type"] = accessType;
                                if (segment.HasTimestamp()) {
                                    pathElement["time"] = TOffersBuildingContext::CorrectWalkingDuration(segment.GetTimestamp() - duration, pedestrianSpeedKmh).Seconds();
                                } else {
                                    pathElement["time"] = 0;
                                }
                                pathElement["path"] = TGeoCoord::SerializeVector(path);
                                jsonReport.AppendValue(pathElement);
                            }
                            accessType = segment.GetAccessType();
                            path.clear();
                            if (segment.HasTimestamp()) {
                                duration = segment.GetTimestamp();
                            }
                        }
                        path.push_back(c);
                    }
                    report.AddReportElement("route_segments", std::move(jsonReport));

                } else {
                    double fullTime = 0.;
                    if (!helper.GetProperty("Time", fullTime)) {
                        report.AddReportElement("error", "Incorrect reply (incorrect 'Time')");
                        continue;
                    }

                    TString trace;
                    if (!helper.GetProperty("Path", trace)) {
                        report.AddReportElement("error", "Incorrect reply (incorrect 'Path')");
                        continue;
                    }
                    report.AddReportElement("route", trace);
                    report.AddReportElement("time", TOffersBuildingContext::CorrectWalkingDuration(fullTime, pedestrianSpeedKmh).Seconds());
                }
            }
        }
    }

    NJson::TJsonValue landmarksJson;
    for (auto& mark : landmarks.GetObjects()) {
        NJson::TJsonValue place;
        mark.ToJson(place);
        landmarksJson.AppendValue(place);
    }

    report.AddReportElement("places", std::move(landmarksJson));
    g.SetCode(HTTP_OK);
}

