#include "processor.h"


namespace {
    IRequestProcessorConfig::TFactory::TRegistrator<TRTBackgroundProposeProcessor::THandlerConfig> ProposeRegistrator(TRTBackgroundProposeProcessor::GetTypeName() + "_registrator");
    IRequestProcessorConfig::TFactory::TRegistrator<TRTBackgroundConfirmProcessor::THandlerConfig> ConfirmRegistrator(TRTBackgroundConfirmProcessor::GetTypeName() + "_registrator");
    IRequestProcessorConfig::TFactory::TRegistrator<TRTBackgroundRejectPropositionProcessor::THandlerConfig> RejectRegistrator(TRTBackgroundRejectPropositionProcessor::GetTypeName() + "_registrator");
}


void TRTBackgroundRemoveProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    R_ENSURE(Server->GetRTBackgroundManager(), ConfigHttpStatus.NotImplementedState, "not configured");
    R_ENSURE(requestData["ids"].IsArray(), ConfigHttpStatus.SyntaxErrorStatus, "'ids' is not array");

    auto session = BuildTx<NSQL::Writable>();
    TMap<TString, TRTBackgroundProcessContainer> settings;
    R_ENSURE(Server->GetRTBackgroundManager()->GetSettingsInfo(&session, settings), ConfigHttpStatus.UnknownErrorStatus, "cannot_fetch_infos");

    TVector<TString> ids;
    for (auto&& i : requestData["ids"].GetArraySafe()) {
        auto it = settings.find(i.GetString());
        R_ENSURE(it != settings.end(), ConfigHttpStatus.UserErrorState, "unknown id " + i.GetString());
        if (it->second->GetOwners().empty()) {
            ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Remove, TAdministrativeAction::EEntity::RTBackground, it->first, it->second->GetGrouppingTags());
        } else if (!it->second->GetOwners().contains(permissions->GetUserId())) {
            ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Remove, TAdministrativeAction::EEntity::RTBackground, it->first, TSet<TString>({"__ROOT_ACCESS__"}));
        }
        ids.emplace_back(i.GetString());
    }
    R_ENSURE(Server->GetRTBackgroundManager()->RemoveBackground(ids, permissions->GetUserId(), session), ConfigHttpStatus.UnknownErrorStatus, "cannot remove options");
    if (!session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }

    g.SetCode(HTTP_OK);
}

void TRTBackgroundUpsertProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    R_ENSURE(Server->GetRTBackgroundManager(), ConfigHttpStatus.NotImplementedState, "not configured");
    R_ENSURE(requestData["backgrounds"].IsArray(), ConfigHttpStatus.SyntaxErrorStatus, "'backgrounds' is not array");

    auto session = BuildTx<NSQL::Writable>();
    TMap<TString, TRTBackgroundProcessContainer> settings;
    R_ENSURE(Server->GetRTBackgroundManager()->GetSettingsInfo(&session, settings), ConfigHttpStatus.UnknownErrorStatus, "cannot_fetch_infos");
    const bool force = IsTrue(Context->GetCgiParameters().Get("force"));
    TVector<TSetting> backgrounds;
    for (auto&& i : requestData["backgrounds"].GetArraySafe()) {
        TRTBackgroundProcessContainer container;
        R_ENSURE(TBaseDecoder::DeserializeFromJson(container, i), ConfigHttpStatus.SyntaxErrorStatus, "incorrect background json");

        auto it = settings.find(container.GetName());
        if (it != settings.end()) {
            if (it->second->GetOwners().empty()) {
                ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::RTBackground, it->first, it->second->GetGrouppingTags());
            } else if (!it->second->GetOwners().contains(permissions->GetUserId())) {
                ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::RTBackground, it->first, TSet<TString>({"__ROOT_ACCESS__"}));
            }
        } else {
            ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Add, TAdministrativeAction::EEntity::RTBackground, container->GetRTProcessName(), container->GetGrouppingTags());
        }

        if (force) {
            if (!Server->GetRTBackgroundManager()->ForceUpsertObject(container, permissions->GetUserId(), session)) {
                session.DoExceptionOnFail(ConfigHttpStatus);
            }
        } else {
            if (!Server->GetRTBackgroundManager()->UpsertObject(container, permissions->GetUserId(), session)) {
                session.DoExceptionOnFail(ConfigHttpStatus);
            }
        }

    }

    if (!session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }

    g.SetCode(HTTP_OK);
}

void TRTBackgroundInfoProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    R_ENSURE(Server->GetRTBackgroundManager(), ConfigHttpStatus.NotImplementedState, "not configured");

    const TCgiParameters& cgi = Context->GetCgiParameters();
    auto ids = MakeSet(GetStrings(cgi, "ids", false));
    if (ids.empty()) {
        ids = MakeSet(GetStrings(cgi, "bp_name", false));
    }
    const auto types = MakeSet(GetStrings(cgi, "bp_type", false));
    const auto envs = MakeSet(GetStrings(cgi, "ctype", false));
    const auto reportType = GetString(cgi, "report", false);

    bool compactSettings = false;
    bool serializeState = true;
    if (reportType == "compact") {
        compactSettings = true;
        serializeState = false;
    }

    auto session = BuildTx<NSQL::ReadOnly>();
    TMap<TString, TRTBackgroundProcessContainer> settings;
    {
        auto eg = g.BuildEventGuard("GetSettingsInfo");
        NSQL::TQueryOptions options;
        if (!ids.empty()) {
            options.SetGenericCondition("bp_name", ids);
        }
        if (!types.empty()) {
            options.SetGenericCondition("bp_type", types);
        }
        if (!envs.empty()) {
            options.SetGenericCondition("bp_settings::json->'host_filter'->>'ctype'", envs);
        }
        R_ENSURE(Server->GetRTBackgroundManager()->GetSettingsInfo(&session, settings, options), ConfigHttpStatus.UnknownErrorStatus, "cannot_fetch_infos", session);
    }
    TMap<TString, TRTBackgroundProcessStateContainer> states;
    if (serializeState) {
        auto eg = g.BuildEventGuard("GetStatesInfo");
        R_ENSURE(Server->GetRTBackgroundManager()->GetStatesInfo(&session, states, MakeSet(NContainer::Keys(settings))), ConfigHttpStatus.UnknownErrorStatus, "cannot_fetch_states", session);
    }
    NJson::TJsonValue bgJson(NJson::JSON_ARRAY);
    for (auto&& [id, container] : settings) {
        if (!ids.empty() && !ids.contains(id)) {
            continue;
        }
        if (!container) {
            continue;
        }
        if (container->GetOwners().empty()) {
            if (!CheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::RTBackground, id, container->GetGrouppingTags())) {
                continue;
            }
        } else if (!container->GetOwners().contains(permissions->GetUserId())) {
            if (!CheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::RTBackground, id, TSet<TString>({"__ROOT_ACCESS__"}))) {
                continue;
            }
        }
        NJson::TJsonValue reportItem = container.GetReport(compactSettings);
        if (serializeState) {
            auto it = states.find(id);
            if (it != states.end()) {
                reportItem.InsertValue("background_process_state", it->second.GetReport());
            }
        }
        if (Server->GetRTBackgroundManager()->GetConfig().GetBlockedProcesses().contains(id)) {
            reportItem.InsertValue("manual_block", true);
        }
        bgJson.AppendValue(reportItem);
    }
    g.MutableReport().AddReportElement("rt_backgrounds", std::move(bgJson));

    TSet<TString> users;
    NJson::TJsonValue propositionsReport = NJson::JSON_ARRAY;
    {
        auto propositions = Yensured(Server->GetRTBackgroundManager()->GetPropositions())->Get(session);
        R_ENSURE(propositions, ConfigHttpStatus.UnknownErrorStatus, "cannot_fetch_propositions", session);
        for (auto&& i : *propositions) {
            if (ids.size() && !ids.contains(i.second.GetName())) {
                continue;
            }
            propositionsReport.AppendValue(i.second.BuildJsonReport());
            users.emplace(i.second.GetPropositionAuthor());
        }
    }
    g.MutableReport().AddReportElement("propositions", std::move(propositionsReport));

    NJson::TJsonValue usersReport = NJson::JSON_MAP;
    {
        auto userData = DriveApi->GetUsersData()->FetchInfo(users, session);
        R_ENSURE(userData, ConfigHttpStatus.UnknownErrorStatus, "cannot_fetch_users", session);
        for (auto&& [_, i] : userData.GetResult()) {
            usersReport.InsertValue(i.GetUserId(), i.GetReport(permissions->GetUserReportTraits()));
        }
    }
    g.MutableReport().AddReportElement("users", std::move(usersReport));

    g.SetCode(HTTP_OK);
}

NDrive::TScheme TRTBackgroundInfoProcessor::GetCgiParametersScheme(const IServerBase* /* server */, const TCgiParameters& /* schemeCgi */) {
    NDrive::TScheme scheme;
    scheme.Add<TFSVariants>("report", "Вид").SetVariants({"full", "compact"});
    scheme.Add<TFSString>("bp_name", "Идентификаторы");
    scheme.Add<TFSString>("bp_type", "Тип робота");
    scheme.Add<TFSString>("ctype", "Окружение");
    return scheme;
}
