#include "processor.h"

#include "search.h"

#include <drive/backend/processors/common_app/fetcher.h>

#include <drive/backend/areas/areas.h>
#include <drive/backend/billing/manager.h>
#include <drive/backend/common/localization.h>
#include <drive/backend/data/area_tags.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/data/device_tags.h>
#include <drive/backend/data/telematics.h>
#include <drive/backend/database/config.h>
#include <drive/backend/database/entity/search_index.h>
#include <drive/backend/device_snapshot/image.h>
#include <drive/backend/insurance/task/history.h>
#include <drive/backend/logging/events.h>
#include <drive/backend/registrar/ifaces.h>
#include <drive/backend/tags/tags_filter.h>
#include <drive/backend/user_document_photos/manager.h>

#include <drive/library/cpp/raw_text/phone_number.h>
#include <drive/telematics/protocol/errors.h>
#include <drive/telematics/server/tasks/lite.h>

#include <kernel/daemon/common/time_guard.h>

#include <library/cpp/json/json_reader.h>

#include <rtline/library/geometry/polyline.h>
#include <rtline/util/blob_with_header.h>
#include <rtline/util/algorithm/type_traits.h>

#include <util/generic/adaptor.h>
#include <util/generic/serialized_enum.h>
#include <util/string/subst.h>
#include <util/string/vector.h>

class TCarCommand : public TServiceAppProcessorBase::IUserCommand {
private:
    const TString Command;

protected:
    NJson::TJsonValue DoGetReport() const override {
        NJson::TJsonValue result;
        result["command"] = Command;
        return result;
    }
    TString GetType() const override {
        return "car_command";
    }

public:
    bool Compare(IUserCommand::TPtr commandExt) const override {
        const TCarCommand* command = dynamic_cast<const TCarCommand*>(commandExt.Get());
        if (!command) {
            return false;
        }
        return command->Command == Command;
    }

    TCarCommand(const TString& command)
        : Command(command)
    {
    }
};

class TTagEvolveCommand : public TServiceAppProcessorBase::IUserCommand {
private:
    const TString TagId;
    const TString ToName;
    const TString ActionName;

protected:
    NJson::TJsonValue DoGetReport() const override {
        NJson::TJsonValue result;
        result["from_id"] = TagId;
        result["to_name"] = ToName;
        result["action_name"] = ActionName;
        return result;
    }
    TString GetType() const override {
        return "evolve_tag";
    }

public:
    bool Compare(IUserCommand::TPtr commandExt) const override {
        const TTagEvolveCommand* command = dynamic_cast<const TTagEvolveCommand*>(commandExt.Get());
        if (!command) {
            return false;
        }
        return (command->TagId == TagId) && (command->ToName == ToName);
    }

    TTagEvolveCommand(const TString& fromTagId, const TString& toName, const TString& name)
        : TagId(fromTagId)
        , ToName(toName)
        , ActionName(name)
    {
    }
};

bool TServiceAppProcessorBase::GetListAvailableCommands(TUserPermissions::TPtr permissions, const TString& carId, TMap<TString, TVector<IUserCommand::TPtr>>& result, NDrive::TEntitySession& session) const {
    TMap<TString, TVector<TDBTag>> tagsByObject;
    const TString& userId = permissions->GetUserId();
    if (!DriveApi->GetTagsManager().GetDeviceTags().GetPerformObjects(userId, tagsByObject, session)) {
        return false;
    }

    auto registeredTags = DriveApi->GetTagsManager().GetTagsMeta().GetRegisteredTags();

    for (auto&& i : tagsByObject) {
        if (carId != "" && i.first != carId)
            continue;
        TVector<IUserCommand::TPtr> actions;
        for (auto&& t : i.second) {
            auto it = registeredTags.find(t->GetName());
            if (it == registeredTags.end()) {
                continue;
            }
            auto td = it->second;
            auto actionsByTag = td->GetAvailableCarActions();
            for (auto&& i : actionsByTag) {
                actions.push_back(new TCarCommand(i));
            }
            auto* variants = permissions->GetEvolutions(t->GetName());
            if (variants) {
                for (auto&& eVariant : *variants) {
                    actions.push_back(new TTagEvolveCommand(t.GetTagId(), eVariant.second.GetTagNameTo(), eVariant.second.GetName()));
                }
            }
        }
        result.emplace(i.first, actions);
    }
    return true;
}

void TServiceAppProcessorBase::Wait(TUserPermissions::TPtr permissions, const NThreading::TFuture<NDrive::TCommonCommandResponse>& r, ELocalization locale, bool ignoreTimeout) const {
    R_ENSURE(
        r.Wait(Context->GetRequestDeadline()),
        ConfigHttpStatus.IncompleteStatus,
        "telematics timeout"
    );
    const auto& response = r.GetValue();
    switch (response.Status) {
    case NDrive::TTelematicsClient::EStatus::Success:
        break;
    case NDrive::TTelematicsClient::EStatus::Processing:
    case NDrive::TTelematicsClient::EStatus::Timeouted:
        if (ignoreTimeout) {
            break;
        } else {
            [[fallthrough]];
        }
    default: {
        NDrive::TEntitySession tx;
        TString developerMessage = response.Message;
        NDrive::ETelematicsNotification error = NDrive::ParseError(developerMessage);
        if (error != NDrive::ETelematicsNotification::Unknown) {
            tx.SetLocalizedMessageKey(TTelematicsCommandTag::GetUserErrorDescription(error, locale, *permissions, *Server));
        }
        R_ENSURE(
            response.Status == NDrive::TTelematicsClient::EStatus::Success,
            ConfigHttpStatus.ConflictRequest,
            developerMessage,
            tx
        );
    }
    }
}

void TCarCommandProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    const auto& configHttpStatus = Server->GetHttpStatusManagerConfig();
    const TCgiParameters& cgi = Context->GetCgiParameters();

    TString carId = GetUUID(requestData, "car_id", false);
    if (!carId) {
        carId = GetUUID(cgi, "car_id", false);
    }
    R_ENSURE(carId, configHttpStatus.SyntaxErrorStatus, "car_id is missing in both request data and CGI parameters");

    auto actionFromPost = GetString(requestData, "action", false);
    auto actionFromCgi = GetString(cgi, "action", false);
    auto action = actionFromPost ? actionFromPost : actionFromCgi;
    R_ENSURE(action, configHttpStatus.SyntaxErrorStatus, "action is missing in both request data and CGI parameters");

    auto locale = GetLocale();

    {
        auto session = BuildTx<NSQL::ReadOnly>();
        TMap<TString, TVector<IUserCommand::TPtr>> availableCommands;
        if (!GetListAvailableCommands(permissions, carId, availableCommands, session)) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }

        auto it = availableCommands.find(carId);
        R_ENSURE(it != availableCommands.end(), configHttpStatus.UserErrorState, "not performed car", NDrive::MakeError("not_available_command_for_performed_tasks"), session);
        IUserCommand::TPtr currentCommand = new TCarCommand(action);
        bool found = false;
        for (auto&& i : it->second) {
            if (i->Compare(currentCommand)) {
                found = true;
                break;
            }
        }
        R_ENSURE(found, configHttpStatus.PermissionDeniedStatus, "have not permissions for this action", EDriveSessionResult::NoUserPermissions);
    }

    auto command = actionFromPost
        ? TTelematicsCommandTag::ParseCommand(requestData, carId, *Server)
        : TTelematicsCommandTag::ParseCommand(action, carId, *Server);
    ReqCheckCondition(command, ConfigHttpStatus.SyntaxErrorStatus, EDriveLocalizationCodes::SyntaxUserError);
    auto session = BuildTx<NSQL::Writable>();
    auto timeout = GetDuration(cgi, "timeout", TDuration::Seconds(30));
    auto response = TTelematicsCommandTag::Command(carId, *command, timeout, *permissions, Server, session);
    R_ENSURE(response.Initialized(), {}, "cannot Command", session);
    R_ENSURE(session.Commit(), {}, "cannot Commit", session);
    Wait(permissions, response, locale);
    g.SetCode(HTTP_OK);
}

void TStartServiceProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    const auto& configHttpStatus = Server->GetHttpStatusManagerConfig();
    auto tagIds = GetUUIDs(requestData, "tag_id", false);
    auto tagNames = GetStrings(requestData, "tag_name", false);
    R_ENSURE(!tagIds.empty() || !tagNames.empty(), configHttpStatus.SyntaxErrorStatus, "Either 'tag_id' or 'tag_name' is required");

    TDuration transactionTime;
    {
        TTimeGuard tg(transactionTime);
        NDrive::TEntitySession session = BuildTx<NSQL::Writable | NSQL::RepeatableRead>();

        TSet<TString> tagIdsSelected(tagIds.begin(), tagIds.end());
        if (!tagNames.empty()) {
            auto carId = GetUUID(requestData, "car_id");
            TVector<TDBTag> selectedTagNames;
            if (!DriveApi->GetTagsManager().GetDeviceTags().RestoreEntityTags(carId, tagNames, selectedTagNames, session)) {
                session.DoExceptionOnFail(ConfigHttpStatus);
            } else {
                for (auto&& i : selectedTagNames) {
                    tagIdsSelected.emplace(i.GetTagId());
                }
            }
        }

        TVector<TDBTag> tags;
        const IEntityTagsManager* tagsManager = nullptr;
        if (!DriveApi->DetectObjectTagsOperator(tagsManager, tagIdsSelected, tags, session)) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        } else {
            ReqCheckCondition(tagsManager, ConfigHttpStatus.UserErrorState, EDriveLocalizationCodes::SyntaxUserError);
        }

        if (!tagsManager->InitPerformer(tags, *permissions, Server, session)) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
        if (!session.Commit()) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
    }
    g.SetCode(HTTP_OK);
}

void TFinishServiceProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    const TCgiParameters& cgi = Context->GetCgiParameters();
    const auto locale = GetLocale();

    const NJson::TJsonValue::TArray& linkedTags = requestData["linked_tags"].GetArray();

    TString carIdAddTags;
    TVector<ITag::TPtr> serviceTags;
    if (!linkedTags.empty()) {
        carIdAddTags = GetUUID(requestData, "car_id", false);
        R_ENSURE(carIdAddTags, ConfigHttpStatus.SyntaxErrorStatus, "Need in 'car_id'");
        for (auto&& tagJson : linkedTags) {
            TMessagesCollector errors;
            ITag::TPtr tag = IJsonSerializableTag::BuildFromJson(DriveApi->GetTagsManager(), tagJson, &errors);
            R_ENSURE(tag, ConfigHttpStatus.SyntaxErrorStatus, errors.GetReport());
            serviceTags.push_back(tag);
        }
    }

    TSet<TString> freeTags;
    TSet<TString> removeTags;
    {
        auto tags = GetUUIDs(requestData, "tag_id", false);
        removeTags.insert(tags.begin(), tags.end());
        freeTags.insert(removeTags.begin(), removeTags.end());
    }

    TSet<TString> dropTags;
    {
        auto tags = GetUUIDs(requestData, "drop_tag_ids", false);
        dropTags.insert(tags.begin(), tags.end());
        freeTags.insert(dropTags.begin(), dropTags.end());
    }

    const IEntityTagsManager* tagsManager = nullptr;
    {
        NDrive::TEntitySession session = BuildTx<NSQL::ReadOnly>();
        if (!DriveApi->DetectObjectTagsOperator(tagsManager, freeTags, session)) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        } else {
            ReqCheckCondition(tagsManager, ConfigHttpStatus.UserErrorState, EDriveLocalizationCodes::SyntaxUserError);
        }
    }

    TDuration transactionTime;
    {
        TTimeGuard tg(transactionTime);
        if (dynamic_cast<const TDeviceTagsManager*>(tagsManager)) {
            TVector<TTaggedObject> devices;
            {
                NDrive::TEntitySession session = BuildTx<NSQL::ReadOnly>();
                if (!tagsManager->GetObjectsByTagIds(freeTags, devices, session)) {
                    session.DoExceptionOnFail(ConfigHttpStatus);
                }
            }
            TSet<TString> carsForLock;
            for (auto&& car : devices) {
                bool forLock = false;
                TVector<ITag::TPtr> tagsForCheckActions = (car.GetId() == carIdAddTags) ? serviceTags : TVector<ITag::TPtr>();
                for (auto&& tag : car.MutableTags()) {
                    if (tag->GetPerformer() == permissions->GetUserId()) {
                        forLock = true;
                        if (!freeTags.contains(tag.GetTagId())) {
                            forLock = false;
                            break;
                        }
                    }
                    if (!removeTags.contains(tag.GetTagId())) {
                        tagsForCheckActions.push_back(tag.GetData());
                    }
                }
                if (!forLock) {
                    continue;
                }
                for (auto&& tag : tagsForCheckActions) {
                    auto actionsLocal = DriveApi->GetActions(tag);
                    if (actionsLocal.contains("ignore-telematics")) {
                        forLock = false;
                        break;
                    }
                }
                if (forLock) {
                    carsForLock.emplace(car.GetId());
                }
            }
            for (auto&& carWithTags : carsForLock) {
                if (!DriveApi->GetIMEI(carWithTags)) {
                    continue;
                }
                auto command = NDrive::NVega::ECommandCode::YADRIVE_END_OF_LEASE;
                auto timeout = GetDuration(cgi, "timeout", TDuration::Seconds(30));
                NDrive::TEntitySession session = BuildTx<NSQL::Writable>();
                auto response = TTelematicsCommandTag::Command(carWithTags, command, timeout, *permissions, Server, session);
                R_ENSURE(response.Initialized(), {}, "cannot Command", session);
                R_ENSURE(session.Commit(), {}, "cannot Commit", session);
                Wait(permissions, response, locale, /*ignoreTimeout=*/true);
            }
        }
        {
            NDrive::TEntitySession session = BuildTx<NSQL::Writable>();
            R_ENSURE(tagsManager->DropPerformer(removeTags, dropTags, *permissions, Server, session), {}, "cannot DropPerformer", session);
            R_ENSURE(tagsManager->AddTags(serviceTags, permissions->GetUserId(), carIdAddTags, Server, session), {}, "cannot AddTags", session);
            R_ENSURE(session.Commit(), {}, "cannot Commit", session);
        }
    }

    g.SetCode(HTTP_OK);
}

void TGetCarInfoProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr /*permissions*/, const NJson::TJsonValue& /*requesData*/) {
    const TCgiParameters& cgi = Context->GetCgiParameters();
    const auto locale = GetLocale();
    const auto traits = EReportTraits::ReportIMEI |
        EReportTraits::ReportCarId |
        EReportTraits::ReportModels
    ;

    NJson::TJsonValue cars(NJson::JSON_ARRAY);
    if (cgi.Has("car_id")) {
        TVector<TString> selectedCars = GetUUIDs(cgi, "car_id");
        TCarsDB::TFetchResult fetchResult = DriveApi->GetCarsData()->GetCachedOrFetch(selectedCars);
        for (auto&& car : fetchResult.GetResult()) {
            cars.AppendValue(car.second.GetReport(locale, traits));
        }
    } else if (cgi.Has("imei")) {
        TVector<TString> selectedImeis = GetStrings(cgi, "imei");
        TDBImei::TFetchResult fetchResult = DriveApi->GetCarsImei()->GetCachedOrFetch(selectedImeis);
        for (auto&& car : fetchResult.GetResult()) {
            cars.AppendValue(car.second.GetReport(locale, traits));
        }
    } else {
        throw TCodedException(ConfigHttpStatus.SyntaxErrorStatus) << "either 'car_id' or 'imei' should be present";
    }
    g.MutableReport().AddReportElement("cars", std::move(cars));
    g.SetCode(HTTP_OK);
}

namespace {
    TReportTraits InferReportTraits(TString report) {
        TReportTraits reportTraits = 0;
        if (report == "service_app") {
            reportTraits = NDeviceReport::ReportServiceApp;
        }
        if (report == "service_app_stat") {
            reportTraits = NDeviceReport::ReportServiceAppStat;
        }
        if (report == "service_app_details") {
            reportTraits = NDeviceReport::ReportServiceAppDetails;
        }
        if (report == "external") {
            reportTraits = NDeviceReport::ReportExternal;
        }
        if (report == "user_app") {
            reportTraits = NDeviceReport::ReportUserApp;
        }
        if (report == "admin_app") {
            reportTraits = NDeviceReport::ReportAdminIFace;
        }
        if (report == "models") {
            reportTraits = NDeviceReport::ReportAvailableModels;
        }
        if (report == "leasing") {
            reportTraits = NDeviceReport::ReportLeasing;
        }
        if (report == "dedicated_fleet") {
            reportTraits = NDeviceReport::ReportDedicatedFleet;
        }
        if (report == "taxi_parks") {
            reportTraits = NDeviceReport::ReportTaxiParks;
        }
        return reportTraits;
    }

    NJson::TJsonValue ReportPOIAreasOnScreen(const TAreasDB& areasDB, const TGeoRect& devicesRect, ELocalization locale, const NDrive::IServer& server) {
        TVector<TArea> poiAreas;
        const auto poiAreasFilter = [&poiAreas](const TFullAreaInfo& areaInfo) {
            if (areaInfo.GetArea().GetType() == "poi") {
                poiAreas.push_back(areaInfo.GetArea());
            }
            return true;
        };
        areasDB.ProcessAreasIntersectingPolyline(TPolyLine<TGeoCoord>::FromRect(devicesRect), poiAreasFilter);
        return NDrive::GetPoiReport(poiAreas, NDrive::DefaultPoiLocationTags, {}, locale, server);
    }
}

void TGetCarsProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    const TCgiParameters& cgi = Context->GetCgiParameters();
    const auto clusterId = GetValue<i32>(cgi, "cluster", false);
    const auto defaultLimit = GetHandlerSetting<ui32>("limit").GetOrElse(Max<ui32>());
    const auto defaultPreLimit = GetHandlerSetting<ui32>("pre_limit").GetOrElse(Max<ui32>());
    const auto defaultTagsFilter = GetHandlerSetting<TString>("tags_filter").GetOrElse({});
    const auto defaultEnableRelevance = GetHandlerSetting<bool>("enable_relevance").GetOrElse(false);
    const auto relevanceModelWeight = GetHandlerSetting<double>("relevance_model_weight");
    const auto defaultPageSize = GetHandlerSetting<ui32>("page_size").GetOrElse(0);
    const auto reportPOI = GetHandlerSetting<bool>("report_poi").GetOrElse(false);
    const auto reportCreationTime = GetHandlerSetting<bool>("report_creation_time").GetOrElse(false);
    const auto useCache = GetHandlerSetting<bool>("use_cache");

    const auto numdoc = GetValue<ui32>(cgi, "limit", false).GetOrElse(defaultLimit);
    const auto preNumdoc = GetValue<ui32>(cgi, "pre_limit", false).GetOrElse(defaultPreLimit);
    const auto enableRelevance = GetValue<bool>(cgi, "enable_relevance", false).GetOrElse(defaultEnableRelevance);
    const auto recalculateStatuses = GetValue<bool>(cgi, "recalculate_statuses", false);
    const auto tagsFilterExternal = GetString(cgi, "tags_filter", false);
    const auto tagsFilterString = Coalesce(tagsFilterExternal, defaultTagsFilter);
    const auto pageSize = GetValue<ui32>(cgi, "page_size", false).GetOrElse(defaultPageSize);
    const auto pageNumber = GetValue<ui32>(cgi, "page_number", false).GetOrElse(1);

    TSet<TString> selectedCars = MakeSet(GetUUIDs(cgi, "car_id", false));

    TVector<TString> selectedCarNumbers = GetStrings(cgi, "car_number", false);
    TVector<TString> selectedImeis = GetStrings(cgi, "imei", false);
    if (selectedCarNumbers) {
        auto ids = DriveApi->GetCarIdByNumbers(selectedCarNumbers);
        for (size_t i = 0; i < ids.size(); ++i) {
            R_ENSURE(i < selectedCarNumbers.size(), ConfigHttpStatus.UnknownErrorStatus, "incorrect GetCarIdByNumbers size:" << ids.size());
            R_ENSURE(ids[i], ConfigHttpStatus.EmptySetStatus, "cannot find device with number " << selectedCarNumbers[i]);
        }
        selectedCars.insert(ids.begin(), ids.end());
    }
    if (selectedImeis) {
        auto ids = DriveApi->GetCarIdByIMEIs(selectedImeis);
        for (size_t i = 0; i < ids.size(); ++i) {
            R_ENSURE(i < selectedImeis.size(), ConfigHttpStatus.UnknownErrorStatus, "incorrect GetCarIdByIMEIs size:" << ids.size());
            R_ENSURE(ids[i], ConfigHttpStatus.EmptySetStatus, "cannot find device with IMEI " << selectedImeis[i]);
        }
        selectedCars.insert(ids.begin(), ids.end());
    }

    TMaybe<::TRect<TGeoCoord>> devicesRect;
    TMaybe<TGeoCoord> nativeUserLocation;
    TMaybe<TGeoCoord> userLocation;
    if (selectedCars.empty()) {
        bool useBoundingBox = GetHandlerSetting<bool>("use_bbox").GetOrElse(true);
        auto bboxGrowDistance = GetHandlerSetting<double>("bbox_grow_distance");
        if (!bboxGrowDistance) {
            bboxGrowDistance = GetHandlerSetting<double>("window_growing(m)");
        }
        auto bbox = GetValue<TGeoRect>(cgi, "bbox", false);
        nativeUserLocation = GetUserLocation();
        if (bbox && useBoundingBox) {
            devicesRect = bbox;
            userLocation = bbox->GetCenter();
            if (bboxGrowDistance) {
                devicesRect->GrowDistance(*bboxGrowDistance);
            }
        } else if (cgi.Has("ne") && cgi.Has("sw") && useBoundingBox) {
            TGeoCoord ne;
            TGeoCoord sw;
            R_ENSURE(ne.DeserializeFromString(cgi.Get("ne")) && sw.DeserializeFromString(cgi.Get("sw")), ConfigHttpStatus.SyntaxErrorStatus, "incorrect ne/sw cgi fields", EDriveSessionResult::IncorrectRequest);
            devicesRect = ::TRect<TGeoCoord>(ne, sw);
            userLocation = devicesRect->GetCenter();
            devicesRect->GrowDistance(bboxGrowDistance.GetOrElse(0));
            g.MutableReport().AddReportElement("req_rect", devicesRect->ToString());
        } else {
            userLocation = nativeUserLocation;
        }
    }

    auto bboxPermissionsRecalcDistance = GetHandlerSetting<double>("bbox_permissions_recalc_distance").GetOrElse(100 * 1000);
    auto bboxDistance = nativeUserLocation.GetOrElse({}).GetLengthTo(userLocation.GetOrElse({}));
    auto originalPermissions = permissions;
    if (bboxDistance >= bboxPermissionsRecalcDistance) {
        auto ev = g.BuildEventGuard("BboxRecalcPermissions");
        auto userId = permissions->GetUserId();
        auto upf = permissions->GetUserFeatures();
        upf.SetUserLocation(userLocation);
        permissions = DriveApi->GetUserPermissions(userId, upf, TInstant::Zero(), Context);
        R_ENSURE(permissions, ConfigHttpStatus.UnknownErrorStatus, "cannot recreate permissions for " << userId);
    }

    TVector<NDeviceReport::EReportTraits> antitraits = GetValuesIgnoreErrors<NDeviceReport::EReportTraits>(cgi, "antitraits", false);
    TVector<NDeviceReport::EReportTraits> traits = GetValuesIgnoreErrors<NDeviceReport::EReportTraits>(cgi, "traits", false);
    TSet<NDrive::TSensorId> sensors = MakeSet(GetValues<NDrive::TSensorId>(cgi, "sensors", false));
    TSet<TString> statuses = MakeSet(GetStrings(cgi, "status", false));
    TMaybe<TCarsFetcher::ETagsStatSortMethod> tagsSortMethod = GetValue<TCarsFetcher::ETagsStatSortMethod>(cgi, "tags_sort_method", false);
    bool allTags = GetValue<bool>(cgi, "tags_all", false).GetOrElse(true);

    TString report = GetString(cgi, "report", false);
    TReportTraits reportTraits = InferReportTraits(report);
    for (auto&& trait : traits) {
        reportTraits |= trait;
    }
    for (auto&& trait : antitraits) {
        reportTraits &= (~trait);
    }
    reportTraits &= permissions->GetDeviceReportTraits();

    auto modelTraits = GetValuesIgnoreErrors<NDriveModelReport::EReportTraits>(cgi, "model_traits", false);
    auto modelReportTraits = NDriveModelReport::UserReport;
    for (auto&& trait : modelTraits) {
        modelReportTraits |= trait;
    }

    auto sort = GetValue<TCarsFetcher::ESortMethod>(cgi, "sort", false);
    if (!sort && clusterId) {
        sort = TCarsFetcher::ESortMethod::Rank;
    }
    if (!sort && userLocation) {
        sort = TCarsFetcher::ESortMethod::Proximity;
    }

    TCarsFetcher::TLSSortParams LSOptions;
    if (sort && sort == TCarsFetcher::ESortMethod::LeasingStats) {
        LSOptions.SetLSOrderField(GetValue<TString>(cgi, "order_field", false).GetOrElse("utilization"));
        LSOptions.SetLSOrderDesc(GetValue<bool>(cgi, "order_desc", false).GetOrElse(true));
    }

    auto modelSpecificationFilters = NJson::FromJson<TMaybe<TCarsFetcher::TModelSpecificationFilters>>(requestData["model_specification_filters"]);

    if (!!userLocation && (reportTraits & NDeviceReport::EReportTraits::ReportUserRectOnly)) {
        const auto rectDetermination = [&devicesRect](const TFullAreaInfo& areaInfo)->bool {
            devicesRect = areaInfo.GetArea().GetPolyline().GetRectSafe();
            return false;
        };
        DriveApi->GetAreasDB()->ProcessTagsInPoint(*userLocation, rectDetermination, {"global_area"}, DriveApi->GetConfig().GetGeoAreasFreshness());
    }

    auto organizationIds = MakeSet(GetValues<ui64>(cgi, "account_ids", reportTraits & NDeviceReport::EReportTraits::ReportOrganizationOnly));
    if (organizationIds) {
        CheckOrganizationsAccess(permissions, TAdministrativeAction::EAction::Observe, organizationIds);
    }

    auto locale = GetLocale();
    {
        TEventsGuard egFetch(g.MutableReport(), "fetch");
        TCarsFetcher fetcher(*Server, reportTraits, Context->GetRequestDeadline(), modelReportTraits);
        fetcher.SetEnableRelevance(enableRelevance);
        fetcher.SetNoSelected(cgi.Has("no_selected", "1"));
        fetcher.SetWithClusters(!cgi.Has("no_clusters", "1"));
        fetcher.SetLimit(numdoc);
        fetcher.SetPreLimit(preNumdoc);
        fetcher.SetPageSize(pageSize);
        fetcher.SetPageNumber(pageNumber);
        fetcher.SetReportCreationTime(reportCreationTime);
        fetcher.SetFleetsFilter(organizationIds);
        fetcher.SetLSSortParams(std::move(LSOptions));
        fetcher.SetModelSpecificationFilters(std::move(modelSpecificationFilters));
        fetcher.SetRecalculateStatuses(recalculateStatuses.GetOrElse(!selectedCars.empty()));
        fetcher.SetReportUserObservableTags(allTags);
        fetcher.SetLocale(locale);
        if (tagsFilterString) {
            fetcher.MutableCarTagsFilter() = TTagsFilter();
            R_ENSURE(
                fetcher.MutableCarTagsFilterRef().DeserializeFromString(tagsFilterString),
                ConfigHttpStatus.SyntaxErrorStatus,
                "incorrect tags filter"
            );
        } else if (cgi.Has("composite_filter")) {
            TString filter = GetString(cgi, "composite_filter");
            TString original = filter;
            TVector<TString> filterIds = StringSplitter(filter).SplitBySet(",*()").SkipEmpty();

            size_t offset = 0;
            for (auto&& i : filterIds) {
                auto pos = filter.find(i, offset);
                R_ENSURE(pos != filter.npos, HTTP_INTERNAL_SERVER_ERROR, "cannot find " << i << " in " << filter);

                const TFilterAction* filterAction = permissions->GetFilterAction(i);
                if (filterAction) {
                    auto filterString = "(" + filterAction->GetTagsFilter().ToString() + ")";
                    filter.replace(pos, i.size(), filterString);
                    offset = pos + filterString.size();
                } else {
                    offset = pos + i.size();
                }
            }
            g.AddEvent(NJson::TMapBuilder
                ("event", "composite_filter")
                ("original", original)
                ("unpacked", filter)
            );
            if (!filter.empty()) {
                TTagsFilter tagsFilter;
                R_ENSURE(tagsFilter.DeserializeFromString(filter), HTTP_BAD_REQUEST, "cannot parse filter from " << filter);
                fetcher.MutableCarTagsFilter() = std::move(tagsFilter);
            }
        }
        if (cgi.Has("signalq_status_filter")) {
            const auto cgiSignalqStatusFilter = GetValues<TString>(cgi, "signalq_status_filter");
            TSet<TCarsFetcher::ESignalqStatus> signalqStatusFilter;
            for (const auto& cgiStatus : cgiSignalqStatusFilter) {
                TCarsFetcher::ESignalqStatus status;
                if (TryFromString(cgiStatus, status)) {
                    signalqStatusFilter.insert(std::move(status));
                }
            }
            fetcher.SetSignalqStatusFilter(std::move(signalqStatusFilter));
        }
        if (cgi.Has("area_tags_filter")) {
            fetcher.MutableAreaTagsFilter() = TTagsFilter();
            R_ENSURE(fetcher.MutableAreaTagsFilterRef().DeserializeFromString(cgi.Get("area_tags_filter")), ConfigHttpStatus.SyntaxErrorStatus, "incorrect tags filter");
        }
        if (cgi.Has("location_type_filter")) {
            auto types = GetValues<NDrive::TLocation::EType>(cgi, "location_type_filter");
            fetcher.SetLocationTypeFilter(MakeSet(types));
        }
        if (clusterId) {
            fetcher.GetClusters().SetSelectedClusterId(*clusterId);
        }
        if (!!devicesRect) {
            fetcher.SetRect(*devicesRect);
        }
        if (relevanceModelWeight) {
            fetcher.SetRelevanceModelWeight(*relevanceModelWeight);
        }
        if (!sensors.empty()) {
            fetcher.SetSensorIds(sensors);
        }
        if (sort) {
            fetcher.SetSortMethod(*sort);
        }
        if (!statuses.empty()) {
            fetcher.SetStatusFilter(statuses);
        }
        fetcher.SetIsRealtime(true);
        fetcher.SetCheckVisibility(GetHandlerSetting<bool>("fetcher.check_visiability").GetOrElse(true));
        if (userLocation) {
            fetcher.SetUserLocation(*userLocation);
        }
        if (useCache) {
            fetcher.SetUseCache(*useCache);
        }
        fetcher.SetTagsSortMethod(tagsSortMethod.GetOrElse(TCarsFetcher::ETagsStatSortMethod::CarsCount));

        if (!(reportTraits & EReportTraits::ReportServiceSessions) && !organizationIds.empty()) {
            TEventsGuard egAccounts(g.MutableReport(), "create_charge_accounts_filter");
            TSet<TString> availableChargeAccounts;
            auto accounts = DriveApi->GetBillingManager().GetAccountsManager().GetAccountsChildren(MakeSet(organizationIds), Context->GetRequestStartTime());
            for (const auto& accountName : accounts)
                availableChargeAccounts.insert(accountName.first);
            fetcher.SetChargeAccountsFilter(std::move(availableChargeAccounts));
        }

        R_ENSURE(
            fetcher.FetchData(permissions, selectedCars.empty() ? nullptr : &selectedCars),
            ConfigHttpStatus.UnknownErrorStatus,
            "cannot fetch data for report"
        );
        if (reportTraits & EReportTraits::ReportCars) {
            TEventsGuard egFetchReport(g.MutableReport(), "report_cars");
            g.MutableReport().AddReportElementString("cars", fetcher.GetAvailableCarsReportSafe(permissions->GetFilterActions()));
            if (fetcher.GetPageSize()) {
                g.AddReportElement("can_get_more_pages", fetcher.GetCanGetMorePages());
                g.AddReportElement("page_number", fetcher.GetPageNumber());
            }
        }
        fetcher.BuildReportMeta(g.MutableReport());

        if (reportTraits & EReportTraits::ReportFilters) {
            TEventsGuard egFetchReport(g.MutableReport(), "report_filters");
            g.MutableReport().AddReportElement("filters", fetcher.GetFiltersReportSafe());
        }
        auto debugEventsEnabled = GetHandlerSetting<bool>("debug_events.enabled").GetOrElse(false);
        if (debugEventsEnabled) {
            TEventsGuard eg(g.MutableReport(), "debug_events");
            auto debugEventsCarLimit = GetHandlerSetting<size_t>("debug_events.car_limit").GetOrElse(1000);
            NDrive::TEventLog::Log("CarList", NJson::TMapBuilder
                ("cars", fetcher.GetAvailableCarsDebugReport(debugEventsCarLimit))
            );
        }
        if (reportTraits & EReportTraits::ReportLocationDetails) {
            if (auto debugAreaTags = GetHandlerSetting<TString>("debug_area_tags").GetOrElse("")) {
                NDrive::TLocationTags areaTags;
                StringSplitter(debugAreaTags).SplitBySet(", ").SkipEmpty().Collect(&areaTags);
                NDrive::TEventLog::Log("CarListAreaTags", fetcher.GetCarIdsByAreaTags(areaTags));
            }
        }
    }
    if (devicesRect) {
        g.AddReportElement("bbox", NJson::ToJson(devicesRect));
        if (reportPOI) {
            g.AddReportElement("poi", ReportPOIAreasOnScreen(*Yensured(DriveApi->GetAreasDB()), *devicesRect, locale, *Server));
        }
    }
    g.MutableReport().AddReportElement("timestamp", Context->GetRequestStartTime().Seconds());
    g.SetCode(HTTP_OK);
}

void TListTagsOperationsProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*jsonValue*/) {
    NJson::TJsonValue results;
    auto session = BuildTx<NSQL::ReadOnly>();

    NJson::TJsonValue reportTags(NJson::JSON_MAP);
    for (auto&& i : GetEnumNames<TTagAction::ETagAction>()) {
        const TSet<TString> tagNames = permissions->GetTagNamesByAction(i.first);
        NJson::TJsonValue tagsJson(NJson::JSON_ARRAY);
        for (auto&& i : tagNames) {
            tagsJson.AppendValue(i);
        }
        reportTags.InsertValue(i.second, tagsJson);
    }

    g.MutableReport().AddReportElement("actions", std::move(reportTags));
    g.SetCode(HTTP_OK);
}

void TGetActionsProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue&) {
    NDrive::TEntitySession session = BuildTx<NSQL::ReadOnly>();

    NJson::TJsonValue reportTags(NJson::JSON_MAP);
    for (auto&& i : GetEnumNames<TTagAction::ETagAction>()) {
        const auto& tagNames = permissions->GetTagsByAction(i.first);
        NJson::TJsonValue& tagsJson = reportTags.InsertValue(i.second, NJson::JSON_ARRAY);
        for (auto&& i : tagNames) {
            NJson::TJsonValue& description = tagsJson.AppendValue(NJson::JSON_MAP);
            description["tag"] = i->GetName();
            description["display_name"] = i->GetDisplayName();
        }
    }

    TMap<TString, TVector<IUserCommand::TPtr>> availableCommands;
    R_ENSURE(
        GetListAvailableCommands(permissions, "", availableCommands, session),
        {},
        "cannot GetListAvailableCommands",
        session
    );

    NJson::TJsonValue report(NJson::JSON_MAP);
    for (auto&& i : availableCommands) {
        NJson::TJsonValue reportByCar(NJson::JSON_MAP);
        NJson::TJsonValue jsonActions(NJson::JSON_ARRAY);
        for (auto&& a : i.second) {
            jsonActions.AppendValue(a->GetReport());
        }
        reportByCar["actions"] = jsonActions;
        report[i.first] = reportByCar;
    }
    g.MutableReport().AddReportElement("cars", std::move(report));
    g.MutableReport().AddReportElement("tags", std::move(reportTags));
    g.SetCode(HTTP_OK);
}

template<typename Manager>
TSet<TString> Search(
    const TSearchProcessor& processor,
    const TDriveAPI* DriveApi,
    const NDrive::IServer* Server,
    const TSearchRequest& searchRequest,
    const Manager& manager,
    const IEntityTagsManager& tagsManager,
    TUserPermissions::TPtr permissions,
    NDrive::TEntitySession& session
) {
    auto entityType = tagsManager.GetEntityType();
    const bool forceShowObjectsWithoutTags = processor.GetHandlerSetting<bool>("force_show_objects_without_tags").GetOrElse(true);
    auto entityFilter = [&](const TString& id) -> bool {
        auto taggedObject = tagsManager.GetCachedOrRestoreObject(id, session);
        R_ENSURE(taggedObject, {}, "cannot GetCachedOrRestoreObject " << id, session);
        return permissions->GetVisibility(*taggedObject, entityType, nullptr, forceShowObjectsWithoutTags) != TUserPermissions::EVisibility::NoVisible;
    };

    auto matchedIds = MakeSet(manager.GetMatchingIds(searchRequest, entityFilter));

    if (entityType == NEntityTagsManager::EEntityType::Car && matchedIds.size() < searchRequest.Limit && searchRequest.IsOneTermSearch() && (searchRequest.Traits & NDeviceReport::ReportAttachments)) {
        auto id = DriveApi->GetCarAttachmentAssignments().GetAttachmentOwnerByServiceAppSlug(searchRequest.GetOriginalSearchTerm());
        if (id && entityFilter(id)) {
            matchedIds.insert(id);
        }
    }
    // for user, there is an additional option of searching by deviceid
    auto userDevicesManager = Server->GetUserDevicesManager();
    if (userDevicesManager && entityType == NEntityTagsManager::EEntityType::User && matchedIds.size() < searchRequest.Limit && searchRequest.IsOneTermSearch()) {
        TSet<TString> usersWithDevice;
        TSet<TString> userIds;
        if (userDevicesManager->GetUsersByDeviceIds({searchRequest.GetSearchTerm()}, userIds, session)) {
            usersWithDevice = std::move(userIds);
            userIds.clear();
        }
        if (userDevicesManager->GetUsersByDeviceIds({ToUpperUTF8(searchRequest.GetSearchTerm())}, userIds, session)) {
            for (auto&& userId : userIds) {
                usersWithDevice.emplace(userId);
            }
        }
        for (auto&& userId : usersWithDevice) {
            if (entityFilter(userId)) {
                matchedIds.emplace(userId);
            }
            if (matchedIds.size() == searchRequest.Limit) {
                break;
            }
        }
    }
    if (entityType == NEntityTagsManager::EEntityType::User && matchedIds.size() < searchRequest.Limit && searchRequest.IsOneTermSearch() && DriveApi->HasDocumentPhotosManager() && searchRequest.GetSearchTerm().size() == 36) {
        auto fr = DriveApi->GetDocumentPhotosManager().GetDocumentVerificationAssignments().FetchInfo(searchRequest.GetSearchTerm(), session);
        for (auto&& assignmentIt : fr) {
            const auto& userId = assignmentIt.second.GetUserId();
            if (entityFilter(userId)) {
                matchedIds.emplace(userId);
            }
            if (matchedIds.size() == searchRequest.Limit) {
                break;
            }
        }
    }
    if (entityType == NEntityTagsManager::EEntityType::User && matchedIds.size() < searchRequest.Limit && searchRequest.IsOneTermSearch() && Server->GetUserRegistrationManager()) {
        auto hash = Server->GetUserRegistrationManager()->GetDocumentNumberHash(searchRequest.GetSearchTerm());
        auto optionalUserIds = DriveApi->GetUsersData()->GetDocumentOwnersByHash(hash, session);
        R_ENSURE(optionalUserIds, {}, "cannot GetDocumentOwnersByHash " << hash, session);
        for (auto&& userId : *optionalUserIds) {
            if (entityFilter(userId)) {
                matchedIds.emplace(userId);
            }
            if (matchedIds.size() == searchRequest.Limit) {
                break;
            }
        }
    }
    return matchedIds;
}

void TSearchProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    R_ENSURE(DriveApi->GetUsersData()->IsSearchEnabled() || DriveApi->GetCarsData()->IsSearchEnabled(), ConfigHttpStatus.UnknownErrorStatus, "Search was not enabled in config, therefore no index was built");
    auto searchRequest = TSearchRequest();
    auto locale = GetLocale();

    // parse required and optional tokens for search
    const TCgiParameters& cgi = Context->GetCgiParameters();
    auto requiredParams = GetStrings(cgi, "has_all_of", false);
    auto optionalParams = GetStrings(cgi, "has_one_of", false);
    R_ENSURE(requiredParams || optionalParams, HTTP_BAD_REQUEST, "missing has_all_of and has_one_of");
    for (auto param : requiredParams) {
        searchRequest.AddMatchingCondition(TMatchCondition(param, true));
    }
    for (auto param : optionalParams) {
        searchRequest.AddMatchingCondition(TMatchCondition(param, false));
    }

    // parse docs limit
    const ui32 maxLimit = 100;
    auto limit = GetValue<ui32>(cgi, "limit", false).GetOrElse(maxLimit);
    R_ENSURE(limit <= maxLimit, ConfigHttpStatus.SyntaxErrorStatus, "Limit should be equal or less than " + ToString(maxLimit));
    searchRequest.SetLimit(limit);

    // do search
    NJson::TJsonValue result;
    bool searchEverything = false;
    TVector<TString> searchObjectTypes;
    if (!cgi.Has("what")) {
        searchEverything = true;
    } else {
        searchObjectTypes = GetStrings(cgi, "what");
    }

    auto session = BuildTx<NSQL::ReadOnly>();
    auto lastActivityInterval = GetHandlerSetting<TDuration>("last_activity_interval");
    if (lastActivityInterval) {
        const auto& userId = permissions->GetUserId();
        const auto& userTagManager = DriveApi->GetTagsManager().GetUserTags();
        bool active = false;
        if (!active) {
            auto eg = g.BuildEventGuard("RestorePerformedUserTags");
            auto optionalPerformedTags = userTagManager.RestorePerformedTags({}, NContainer::Scalar(userId), session);
            R_ENSURE(optionalPerformedTags, {}, "cannot RestorePerformedTags for " << userId, session);
            active = !optionalPerformedTags->empty();
        }
        if (!active) {
            auto eg = g.BuildEventGuard("GetDropPerformEvents");
            auto now = Now();
            auto since = now - *lastActivityInterval;
            auto optionalEvents = userTagManager.GetEvents(since, session, IEntityTagsManager::TQueryOptions(1)
                .SetActions({ EObjectHistoryAction::DropTagPerformer })
                .SetUserIds({ userId })
            );
            R_ENSURE(optionalEvents, {}, "cannot GetEvents since " << since, session);
            active = !optionalEvents->empty();
        }
        R_ENSURE(active, HTTP_FORBIDDEN, "no activity for " << lastActivityInterval->Seconds() << " seconds", session);
    }

    bool usePermittedSearchTraits = GetHandlerSetting<bool>("use_permitted_search_traits").GetOrElse(false);
    bool searchUsers = std::find(searchObjectTypes.begin(), searchObjectTypes.end(), "users") != searchObjectTypes.end();
    if ((searchUsers || searchEverything) && permissions->CheckAdministrativeActions(TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::User)) {
        if (usePermittedSearchTraits) {
            searchRequest.SetTraits(permissions->GetUserSearchTraits());
        }
        const auto& manager = DriveApi->GetUserManager();
        const auto& matchedIds = Search(*this, DriveApi, Server, searchRequest, manager, DriveApi->GetTagsManager().GetUserTags(), permissions, session);
        auto matchedObjects = manager.FetchInfo(matchedIds, session);
        NJson::TJsonValue matchedObjectReport = NJson::JSON_ARRAY;
        for (auto&& [id, object] : matchedObjects) {
            matchedObjectReport.AppendValue(object.GetSearchReport());
        }
        result.InsertValue("user_search_traits", NJson::ToJson(NJson::BitMask<NUserReport::EReportTraits>(searchRequest.Traits)));
        result.InsertValue("users", std::move(matchedObjectReport));
    }

    TSet<TString> matchedModelIds;
    bool searchCars = std::find(searchObjectTypes.begin(), searchObjectTypes.end(), "cars") != searchObjectTypes.end();
    if (searchCars || searchEverything) {
        if (usePermittedSearchTraits) {
            searchRequest.SetTraits(permissions->GetDeviceSearchTraits());
        }
        const auto& manager = DriveApi->GetCarManager();
        const auto& matchedIds = Search(*this, DriveApi, Server, searchRequest, manager, DriveApi->GetTagsManager().GetDeviceTags(), permissions, session);
        auto matchedObjects = manager.FetchInfo(matchedIds, session);
        NJson::TJsonValue matchedObjectReport = NJson::JSON_ARRAY;
        for (auto&& [id, object] : matchedObjects) {
            auto objectReport = object.GetSearchReport();
            if (permissions->GetDeviceReportTraits() & NDeviceReport::ReportFormerNumbers) {
                auto optionalFormerNumbers = manager.FetchFormerNumbers(id, session);
                R_ENSURE(optionalFormerNumbers, {}, "cannot FetchFormerNumbers for " << id, session);
                objectReport.InsertValue("former_numbers", NJson::ToJson(optionalFormerNumbers));
            }
            if (permissions->GetDeviceReportTraits() & NDeviceReport::ReportModels) {
                matchedModelIds.insert(object.GetModel());
            }
            matchedObjectReport.AppendValue(std::move(objectReport));
        }
        result.InsertValue("car_search_traits", NJson::ToJson(NJson::BitMask<NDeviceReport::EReportTraits>(searchRequest.Traits)));
        result.InsertValue("cars", std::move(matchedObjectReport));
    }
    if ((searchCars || searchEverything) && permissions->GetDeviceReportTraits() & NDeviceReport::ReportModels) {
        auto matchedModels = DriveApi->GetModelsDB().FetchInfo(matchedModelIds, session);
        NJson::TJsonValue matchedModelReport = NJson::JSON_MAP;
        for (auto&& [id, object] : matchedModels) {
            matchedModelReport.InsertValue(id, object.GetReport(locale, NDriveModelReport::ReportNames));
        }
        result.InsertValue("models", std::move(matchedModelReport));
    }

    g.MutableReport().AddReportElement("objects", std::move(result));
    g.SetCode(HTTP_OK);
}


void TRestoreBillingTasksProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::ObserveStructure, TAdministrativeAction::EEntity::Payment);
    const TInstant now = Now();
    auto bSessions = DriveApi->GetTagsManager().GetDeviceTags().GetHistoryManager().GetSessionsBuilder("billing", now);
    auto vSessions = bSessions->GetSessionsActual(now - TDuration::Seconds(1), now + TDuration::Seconds(1));
    TVector<TString> sessionIds;
    for (auto&& u : vSessions) {
        for (auto&& i : u.second) {
            if (!i->GetClosed()) {
                sessionIds.push_back(i->GetSessionId());
            }
        }
    }
    if (sessionIds.empty()) {
        g.SetCode(ConfigHttpStatus.EmptySetStatus);
        return;
    }

    const TString cond = "'" + JoinSeq("','", sessionIds) + "'";

    TSet<TString> actualTasks;
    {
        TRecordsSet records;
        auto session = BuildTx<NSQL::ReadOnly>();
        R_ENSURE(
            session->Exec("SELECT * FROM billing_tasks WHERE session_id IN (" + cond + ")", &records)->IsSucceed(),
            ConfigHttpStatus.UnknownErrorStatus,
            "cannot select actual tasks",
            session
        );
        for (auto&& i : records) {
            actualTasks.emplace(i.Get("session_id"));
        }
    }

    auto pred = [&actualTasks](const TString& s) -> bool {
        return actualTasks.contains(s);
    };

    sessionIds.erase(std::remove_if(sessionIds.begin(), sessionIds.end(), pred), sessionIds.end());

    actualTasks.clear();
    {
        TRecordsSet records;
        auto session = BuildTx<NSQL::ReadOnly>();
        R_ENSURE(
            session->Exec("SELECT * FROM billing_tasks_history WHERE session_id IN (" + cond + ") AND history_timestamp > " + ToString((Now() - TDuration::Minutes(1)).Seconds()), &records)->IsSucceed(),
            ConfigHttpStatus.UnknownErrorStatus,
            "cannot select actual tasks 2",
            session
        );
        for (auto&& i : records) {
            actualTasks.emplace(i.Get("session_id"));
        }
    }

    sessionIds.erase(std::remove_if(sessionIds.begin(), sessionIds.end(), pred), sessionIds.end());

    if (!sessionIds.size()) {
        g.SetCode(ConfigHttpStatus.EmptySetStatus);
        return;
    }

    NJson::TJsonValue brokenTasks;
    for (auto&& i : sessionIds) {
        brokenTasks.AppendValue(i);
    }

    g.MutableReport().AddReportElement("broken_tasks", std::move(brokenTasks));
    g.SetCode(HTTP_OK);
}

void TRemapUserUid::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::User);
    TVector<TString> passportUids = GetStrings(Context->GetCgiParameters(), "uids", false);
    TVector<TString> phones = GetStrings(Context->GetCgiParameters(), "phones", false);
    TVector<TString> dls = GetStrings(Context->GetCgiParameters(), "driver_licenses", false);
    bool withDeleted = GetValue<bool>(Context->GetCgiParameters(), "with_deleted", false).GetOrElse(false);
    if (dls.empty()) {
        dls = GetStrings(requestData, "driver_licenses", false);
    }

    R_ENSURE(phones || passportUids || dls, HTTP_BAD_REQUEST, "no uids, phones, driver_licenses");

    NJson::TJsonValue report(NJson::JSON_ARRAY);

    if (phones) {
        TSet<TString> phoneNumbers(phones.begin(), phones.end());
        auto userList = Server->GetDriveAPI()->GetUsersData()->FetchUsersByPhone(phoneNumbers);
        for (auto&& userIt : userList) {
            if (userIt.second.IsPhoneVerified() && userIt.second.GetStatus() == NDrive::UserStatusActive) {
                NJson::TJsonValue& item = report.AppendValue(NJson::JSON_MAP);
                item["phone"] = userIt.second.GetPhone();
                item["id"] = userIt.second.GetUserId();
            }
        }
    } else if (passportUids) {
        R_ENSURE(passportUids.size() < Config.GetMaxItems(), ConfigHttpStatus.UserErrorState, "too many items");
        for (auto&& uid : passportUids) {
            auto userId = Server->GetDriveAPI()->GetUsersData()->GetUserIdByUid(uid);
            if (!!userId) {
                NJson::TJsonValue& item = report.AppendValue(NJson::JSON_MAP);
                item[uid] = userId;
            }
        }
    } else {
        R_ENSURE(Server->GetUserRegistrationManager(), HTTP_INTERNAL_SERVER_ERROR, "no registration manager");
        auto session = BuildTx<NSQL::ReadOnly>();
        for (auto&& value : dls) {
            auto hash = Server->GetUserRegistrationManager()->GetDocumentNumberHash(value);
            auto users = DriveApi->GetUsersData()->GetDocumentOwnersObjectsByHash(hash, session);
            if (!users) {
                session.Check();
            }
            for (auto&& user : *users) {
                if (!withDeleted && user.GetStatus() == NDrive::UserStatusDeleted) {
                    continue;
                }
                if (user.GetDrivingLicenseNumberHash() == hash) {
                    NJson::TJsonValue& item = report.AppendValue(NJson::JSON_MAP);
                    item["driver_license"] = value;
                    item["id"] = user.GetUserId();
                    item["first_name"] = user.GetFirstName();
                    item["last_name"] = user.GetLastName();
                    item["p_name"] = user.GetPName();
                    item["status"] = user.GetStatus();
                }
            }
        }
    }
    g.MutableReport().AddReportElement("users", std::move(report));
    g.SetCode(HTTP_OK);
}

void TFindUserByPhone::OnFail(const TString& errorCode, NDrive::TEntitySession& session, const int errorState) {
    if (errorCode) {
        session.SetError(NDrive::MakeError(errorCode));
        session.SetLocalizedMessageKey("find_user_by_phone.messages." + errorCode);
    } else {
        session.SetLocalizedMessageKey("find_user_by_phone.messages.unknown_code");
    }
    R_ENSURE(false, errorState, errorCode, session);
}

void TFindUserByPhone::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    TString phone = GetString(requestData, "phone", true);
    TString userName = GetString(requestData, "name", false);
    TString configName = GetString(requestData, "config", false);
    NJson::TJsonValue patchedRequestData = requestData;
    if (configName) {
        TString configStr = GetHandlerSettingDef<TString>("config." + configName, "");
        NJson::TJsonValue json;
        R_ENSURE(NJson::ReadJsonFastTree(configStr, &patchedRequestData), ConfigHttpStatus.UserErrorState, "can't parse config from gvars " + configName);
    }
    TVector<TString> allowedStatuses = GetStrings(patchedRequestData, "allowed_statuses", false);
    bool checkNoDebt = GetValue<bool>(patchedRequestData, "check_no_debt", false).GetOrElse(true);
    auto phoneVerified = GetValue<bool>(patchedRequestData, "phone_verified", false);
    auto getUserReport = GetValue<bool>(patchedRequestData, "get_user_report", false);
    bool checkLastPhoneVerification = GetValue<bool>(patchedRequestData, "check_last_phone_verification", false).GetOrElse(false);
    R_ENSURE(!phoneVerified || !checkLastPhoneVerification, ConfigHttpStatus.UserErrorState, "only one of {phone_verified, check_last_phone_verification} is allowed");

    auto session = BuildTx<NSQL::ReadOnly>();

    TPhoneNormalizer phoneNormalizer;
    auto normalizedPhones = phoneNormalizer.Normalize(phone);
    if (normalizedPhones.empty()) {
        OnFail("bad_number", session);
        return;
    } else {
        phone = normalizedPhones.front();
    }

    auto usersDataMaybe = Server->GetDriveAPI()->GetUsersData()->FindUsersByPhone(phone, MakeSet(allowedStatuses), phoneVerified, session);
    if (!usersDataMaybe) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }
    auto usersData = std::move(usersDataMaybe.GetRef());
    NJson::TJsonValue report(NJson::JSON_MAP);
    report["phone"] = phone;
    report["name"] = userName;
    TMaybe<TDriveUserData> result;
    if (usersData.size() == 0) {
        OnFail("not_found", session);
        return;
    } else {
        if (checkLastPhoneVerification && Server->GetUserDevicesManager()) {
            TMap<TString, TDriveUserData> idsToUsers;
            for (auto&& user : usersData) {
                if (user.GetUserId() != permissions->GetUserId()) {
                    idsToUsers[user.GetUserId()] = user;
                }
            }
            TMaybe<TObjectEvent<TUserDevice>> lastDeviceEvent;
            if (!Server->GetUserDevicesManager()->GetLastDeviceEvent(MakeSet(NContainer::Keys(idsToUsers)), { EObjectHistoryAction::Confirmation }, {}, session, lastDeviceEvent)) {
                session.DoExceptionOnFail(ConfigHttpStatus);
            }
            if (!lastDeviceEvent) {
                OnFail("not_found", session);
                return;
            }
            TDriveUserData deviceUser = idsToUsers[lastDeviceEvent->GetUserId()];
            usersData = { deviceUser };
        }
        if (checkNoDebt && Server->GetDriveAPI()->HasBillingManager()) {
            TVector<TDriveUserData> debtResult;
            for (auto&& user : usersData) {
                auto debt = Server->GetDriveAPI()->GetBillingManager().GetDebt(user.GetUserId(), session);
                if (!debt) {
                    session.DoExceptionOnFail(ConfigHttpStatus);
                }
                if (*debt == 0) {
                    debtResult.emplace_back(std::move(user));
                    break;
                }
            }
            if (debtResult.empty()) {
                OnFail("not_found", session);
                return;
            }
            usersData = std::move(debtResult);
        }
        if (!result) {
            result = std::move(usersData.front());
        }
        report["user_id"] = result->GetUserId();
        report["login"] = result->GetObfuscatedLogin();
    }
    if (!session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }
    g.MutableReport().SetExternalReport(std::move(report));
    if (result && result->GetUserId() && getUserReport) {
        auto user = Server->GetDriveAPI()->GetUsersData()->GetCachedObject(result->GetUserId());
        auto userReportFuture = user->GetFullReport(NUserReport::EReportTraits::ReportYPassport, *Server);
        auto gReport = g.GetReport();
        userReportFuture.Subscribe([gReport](const NThreading::TFuture<NJson::TJsonValue>& result) {
            TJsonReport::TGuard g(gReport, HTTP_OK);
            TJsonReport& r = g.MutableReport();
            if (result.HasValue()) {
                auto userData = result.GetValue();
                r.AddReportElement("user_report", std::move(userData));
            } else if (result.HasException()) {
                r.AddReportElement("error", NThreading::GetExceptionInfo(result));
            }
            g.SetCode(HTTP_OK);
        });
        g.Release();
    } else {
        g.SetCode(HTTP_OK);
    }
}

void TRegisterInsuranceTask::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Control, TAdministrativeAction::EEntity::Payment);

    NStorage::TTableRecord record;
    R_ENSURE(record.DeserializeFromJson(requestData), ConfigHttpStatus.UserErrorState, "incorrect json format");
    TInsuranceNotification insuranceTask;
    R_ENSURE(insuranceTask.DeserializeFromTableRecord(record, nullptr), ConfigHttpStatus.UserErrorState, "incorrect json data");
    auto session = BuildTx<NSQL::Writable>();
    session.SetComment("manual insurance task");
    bool result = DriveApi->InsuranceNotification(insuranceTask.GetSessionId(),
        insuranceTask.GetStart(), insuranceTask.GetFinish(),
        insuranceTask.GetUserId(), insuranceTask.GetCarId(),
        permissions->GetUserId(), session);

    if (!result || !session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }
    g.SetCode(HTTP_OK);
}
