#pragma once

#include "config.h"

#include <drive/backend/processors/common_app/processor.h>

#include <drive/backend/cars/car.h>
#include <drive/backend/common/localization.h>
#include <drive/backend/tags/tags_manager.h>

#include <drive/telematics/api/server/interop.h>

struct TSearchRequest;

class TServiceAppProcessorBase: public TCommonServiceAppProcessorBase {
public:
    class IUserCommand {
    public:
        using TPtr = TAtomicSharedPtr<IUserCommand>;

    protected:
        virtual TString GetType() const = 0;
        virtual NJson::TJsonValue DoGetReport() const = 0;

    public:
        virtual ~IUserCommand() = default;

        virtual bool Compare(IUserCommand::TPtr command) const = 0;

        virtual NJson::TJsonValue GetReport() const {
            NJson::TJsonValue result;
            result["type"] = GetType();
            result["meta"] = DoGetReport();
            return result;
        }
    };

protected:
    const TDriveServiceAppConfig& AppConfig;

public:
    TServiceAppProcessorBase(const TDriveServiceAppConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TCommonServiceAppProcessorBase(config, context, authModule, server)
        , AppConfig(config) {}

protected:
    bool GetListAvailableCommands(TUserPermissions::TPtr permissions, const TString& carId, TMap<TString, TVector<IUserCommand::TPtr>>& result, NDrive::TEntitySession& session) const;
    void Wait(TUserPermissions::TPtr permissions, const NThreading::TFuture<NDrive::TCommonCommandResponse>& r, ELocalization locale, bool ignoreTimeout = false) const;
};

class TCarCommandProcessor: public TServiceAppProcessorBase {
public:
    TCarCommandProcessor(const TDriveServiceAppConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TServiceAppProcessorBase(config, context, authModule, server) {}

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requesData) override;
};

class TProxyCommandProcessor: public TServiceAppProcessorBase {
public:
    using TServiceAppProcessorBase::TServiceAppProcessorBase;
    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requesData) override;
};

class TStartServiceProcessor: public TServiceAppProcessorBase {
public:
    using TServiceAppProcessorBase::TServiceAppProcessorBase;
    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requesData) override;
};

class TFinishServiceProcessor: public TServiceAppProcessorBase {
public:
    TFinishServiceProcessor(const TDriveServiceAppConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TServiceAppProcessorBase(config, context, authModule, server) {}

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requesData) override;
};

class TGetCarInfoProcessor: public TServiceAppProcessorBase {
public:
    using TServiceAppProcessorBase::TServiceAppProcessorBase;

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requesData) override;
};

class TGetCarsProcessor: public TAppCommonProcessor<TGetCarsProcessor> {
private:
    using TBase = TAppCommonProcessor<TGetCarsProcessor>;

public:
    TGetCarsProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server)
    {
    }

    static TString GetTypeName() {
        return "service_app_list";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requesData) override;
};

class TListTagsOperationsProcessor: public TServiceAppProcessorBase {
public:
    TListTagsOperationsProcessor(const TDriveServiceAppConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TServiceAppProcessorBase(config, context, authModule, server)
    {
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requesData) override;
};

class TGetActionsProcessor: public TServiceAppProcessorBase {
public:
    TGetActionsProcessor(const TDriveServiceAppConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TServiceAppProcessorBase(config, context, authModule, server)
    {
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requesData) override;
};

class TSearchProcessor: public TAppCommonProcessor<TSearchProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TSearchProcessor, TEmptyConfig>;

public:
    TSearchProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server)
    {
    }

    static TString GetTypeName() {
        return "service_app_search";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TCurrentFlowStateProcessor: public TAppCommonProcessor<TCurrentFlowStateProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TCurrentFlowStateProcessor, TEmptyConfig>;

public:
    TCurrentFlowStateProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server)
    {
    }

    static TString GetTypeName() {
        return "current_flow_state";
    }

    template <class T>
    NJson::TJsonValue GetAvailableTagActionsReport(const TDBTag& tag, const T& manager, TUserPermissions::TPtr permissions) const {
        NJson::TJsonValue result(NJson::JSON_MAP);
        if (!tag) {
            return result;
        }
        auto locale = GetLocale();
        auto session = BuildTx<NSQL::ReadOnly>();
        const auto& evolutions = *permissions->GetEvolutions();
        const auto it = evolutions.find(tag->GetName());
        auto obj = manager.RestoreObject(tag.GetObjectId(), session);
        if (!obj) {
            session.Check();
        }
        if (it != evolutions.end()) {
            NJson::TJsonValue& evJson = result.InsertValue("evolutions", NJson::JSON_ARRAY);
            for (auto&& [to, evolution] : it->second) {
                TString errorInfo;
                NJson::TJsonValue& evolutionInfo = evJson.AppendValue(evolution.SerializeSpecialsToJson());
                auto td = DriveApi->GetTagsManager().GetTagsMeta().GetDescriptionByName(to);
                if (td) {
                    NJson::TJsonValue& tagDescriptionReport = evolutionInfo.InsertValue("tag_to_description", td->BuildJsonReport(locale));
                    tagDescriptionReport.InsertValue("type", td->GetType());
                }

                evolutionInfo.InsertValue("id", evolution.GetName());
                if (evolution.CheckEvolution(manager, tag, *obj, session, &errorInfo)) {
                    evolutionInfo.InsertValue("available", true);
                } else {
                    evolutionInfo.InsertValue("available", false);
                    evolutionInfo.InsertValue("ev_problem", errorInfo);
                }
            }
        }
        {
            NJson::TJsonValue& actJson = result.InsertValue("non_conditional_actions", NJson::JSON_ARRAY);
            TTagAction::TTagActions avActions = permissions->GetActionsByTagIdx(tag->GetDescriptionIndex());
            for (auto e : GetEnumAllValues<TTagAction::ETagAction>()) {
                if (avActions & (TTagAction::TTagActions)e) {
                    actJson.AppendValue(::ToString(e));
                }
            }
        }
        {
            NJson::TJsonValue& actJson = result.InsertValue("conditional_actions", NJson::JSON_ARRAY);
            TMap<TString, const TTagAction*> conditionalActions;
            for (auto&& [_, actions] : permissions->GetConditionalTagActions()) {
                for (auto&& action : actions) {
                    conditionalActions.emplace(action->GetName(), action);
                }
            }
            auto optionalEvents = manager.GetEventsByTag(tag.GetTagId(), session);
            if (!optionalEvents) {
                return result;
            }
            for (auto&& [_, e] : conditionalActions) {
                if (!e->Match(tag->GetName())) {
                    continue;
                }
                TString errorInfo;
                NJson::TJsonValue& actionInfo = actJson.AppendValue(e->SerializeSpecialsToJson());
                if (e->CheckObjectStatus(tag, *permissions, obj->GetTags(), *optionalEvents, &errorInfo) == TTagAction::ECheckResult::Accept) {
                    actionInfo.InsertValue("available", true);
                } else {
                    actionInfo.InsertValue("available", false);
                    actionInfo.InsertValue("act_problem", errorInfo);
                }
            }
        }
        return result;
    }

    template <class T>
    void FillReport(NJson::TJsonValue& result, const TString& tagId, const T& tagsManager, TSet<TString>& objectIds, TUserPermissions::TPtr permissions, TSet<TString>& tagNames) const {
        TVector<TDBTag> actualTags;
        if (tagId) {
            auto tx = BuildTx<NSQL::ReadOnly>();
            auto optionalTag = tagsManager.RestoreTag(tagId, tx);
            R_ENSURE(optionalTag, {}, "cannot RestoreTag " << tagId, tx);
            auto tag = std::move(*optionalTag);
            if (tag) {
                bool performable = false;
                if (!performable) {
                    performable = tag->GetPerformer() == permissions->GetUserId();
                }
                if (!performable) {
                    auto performableTags = permissions->GetTagNamesByAction(NTagActions::ETagAction::Perform);
                    performable = performableTags.contains(tag->GetName());
                }
                if (!performable) {
                    auto forcePerformableTags = permissions->GetTagNamesByAction(NTagActions::ETagAction::ForcePerform);
                    performable = forcePerformableTags.contains(tag->GetName());
                }
                R_ENSURE(performable, HTTP_FORBIDDEN, "tag " << tagId << " is not performed", tx);
                actualTags.push_back(std::move(tag));
            }
        } else {
            auto session = BuildTx<NSQL::ReadOnly>();
            R_ENSURE(tagsManager.RestorePerformerTags({permissions->GetUserId()}, actualTags, session),
                ConfigHttpStatus.UnknownErrorStatus, "cannot restore tags");
        }

        TMap<TString, TVector<TDBTag>> tagsByFlow;

        for (auto&& t : actualTags) {
            objectIds.emplace(t.TConstDBTag::GetObjectId());
            tagNames.emplace(t->GetName());

        }

        TMap<TString, TSet<TString>> objectActions;
        auto tagDescriptions = DriveApi->GetTagsManager().GetTagsMeta().GetRegisteredTags();
        for (auto&& t : actualTags) {
            auto itDescription = tagDescriptions.find(t->GetName());
            TString tagFlow;
            if (itDescription != tagDescriptions.end()) {
                tagFlow = itDescription->second->GetTagFlow();
                auto actions = itDescription->second->GetAvailableCarActions();
                objectActions[t.GetObjectId()].insert(actions.begin(), actions.end());
            }
            tagsByFlow[tagFlow].emplace_back(t);
        }

        NJson::TJsonValue& actionsByObjectJson = result["actions_by_object"];
        for (auto&& i : objectActions) {
            NJson::TJsonValue& objectActionsJson = actionsByObjectJson.InsertValue(i.first, NJson::JSON_ARRAY);
            for (auto&& act : i.second) {
                objectActionsJson.AppendValue(act);
            }
        }

        NJson::TJsonValue& tagFlowsJson = result["flows"];
        tagFlowsJson.SetType(NJson::JSON_MAP);
        for (auto&& f : tagsByFlow) {
            NJson::TJsonValue& tagFlowJson = tagFlowsJson.InsertValue(f.first, NJson::JSON_ARRAY);
            const auto predCompareNames = [&tagDescriptions](const TDBTag& l, const TDBTag& r) {
                auto itL = tagDescriptions.find(l->GetName());
                auto itR = tagDescriptions.find(r->GetName());
                i32 pLeft = (itL == tagDescriptions.end()) ? 0 : itL->second->GetTagFlowPriority();
                i32 pRight = (itR == tagDescriptions.end()) ? 0 : itR->second->GetTagFlowPriority();
                return std::tie(pLeft, l->GetName()) < std::tie(pRight, r->GetName());
            };
            std::sort(f.second.begin(), f.second.end(), predCompareNames);
            for (auto&& i : f.second) {
                NJson::TJsonValue& flowCurrentJson = tagFlowJson.AppendValue(NJson::JSON_MAP);
                flowCurrentJson.InsertValue("tag_name", i->GetName());
                flowCurrentJson.InsertValue("tag", i->SerializeToJson());
                flowCurrentJson.InsertValue("actions", GetAvailableTagActionsReport(i, tagsManager, permissions));
            }
        }
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) override {
        const TCgiParameters& cgi = Context->GetCgiParameters();
        const TString& tagId = GetString(cgi, "tag_id", false);
        const auto locale = GetLocale();
        const bool reportAbilities = GetValue<bool>(cgi, "report_abilities", false).GetOrElse(true);
        const bool reportDescriptions = GetValue<bool>(cgi, "report_descriptions", false).GetOrElse(true);
        NJson::TJsonValue tagsReport = NJson::JSON_MAP;
        TSet<TString> tagNames;
        {
            TSet<TString> objectIds;
            FillReport(tagsReport, tagId, Server->GetDriveAPI()->GetTagsManager().GetDeviceTags(), objectIds, permissions, tagNames);
            auto gObjects = DriveApi->GetCarsData()->FetchInfo(objectIds, Context->GetRequestStartTime());
            NJson::TJsonValue& objectsJson = tagsReport.InsertValue("cars", NJson::JSON_MAP);
            for (auto&& i : gObjects.GetResult()) {
                objectsJson.InsertValue(i.first, i.second.GetReport(locale, permissions->GetDeviceReportTraits()));
            }
        }
        g.MutableReport().AddReportElement("tags", std::move(tagsReport));

        auto tagDescriptions = DriveApi->GetTagsManager().GetTagsMeta().GetRegisteredTags();
        if (reportAbilities) {
            const auto actions = {TTagAction::ETagAction::Add, TTagAction::ETagAction::Perform};
            TMap<TTagAction::ETagAction, TMap<TString, TVector<TTagDescription::TConstPtr>>> descriptionsByFlows;
            for (auto&& i : tagDescriptions) {
                for (auto&& act : actions) {
                    if (permissions->GetActionsByTagIdx(i.second->GetIndex()) & (TTagAction::TTagActions)act) {
                        descriptionsByFlows[act][i.second->GetTagFlow()].emplace_back(i.second);
                    }
                }
            }
            NJson::TJsonValue tagFlowStart = NJson::JSON_MAP;
            NJson::TJsonValue abilities = NJson::JSON_MAP;
            for (auto&& act : descriptionsByFlows) {
                NJson::TJsonValue& byActJson = abilities.InsertValue(::ToString(act.first), NJson::JSON_MAP);
                for (auto&& i : act.second) {
                    NJson::TJsonValue& descriptions = byActJson.InsertValue(i.first, NJson::JSON_ARRAY);
                    for (auto&& t : i.second) {
                        tagNames.emplace(t->GetName());
                        descriptions.AppendValue(t->GetName());
                    }
                }
                if (act.first == TTagAction::ETagAction::Add) {
                    tagFlowStart = byActJson;
                }
            }
            g.MutableReport().AddReportElement("flow_start_abilities", std::move(tagFlowStart));
            g.MutableReport().AddReportElement("tag_abilities", std::move(abilities));
        }

        if (reportDescriptions) {
            NJson::TJsonValue tagDescriptionsJson = NJson::JSON_MAP;
            for (auto&& i : tagDescriptions) {
                if (tagNames.contains(i.second->GetName())) {
                    tagDescriptionsJson.InsertValue(i.second->GetName(), i.second->BuildJsonReport(locale));
                }
            }
            g.MutableReport().AddReportElement("tag_descriptions", std::move(tagDescriptionsJson));
        }

        g.SetCode(HTTP_OK);
    }
};

class TRestoreBillingTasksProcessor: public TAppCommonProcessor<TRestoreBillingTasksProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TRestoreBillingTasksProcessor, TEmptyConfig>;

public:
    TRestoreBillingTasksProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {
    }

    static TString GetTypeName() {
        return "restore_billing_tasks";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TRemapUserUid : public TAppCommonProcessor<TRemapUserUid, TPackProcessorConfig> {
private:
    using TBase = TAppCommonProcessor<TRemapUserUid, TPackProcessorConfig>;

public:
    TRemapUserUid(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server)
    {}

    static TString GetTypeName() {
        return "remap_user_uid";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};


class TRegisterInsuranceTask : public TAppCommonProcessor<TRegisterInsuranceTask, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TRegisterInsuranceTask, TEmptyConfig>;

public:
    TRegisterInsuranceTask(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {}

    static TString GetTypeName() {
        return "register_insurance_task";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TFindUserByPhone : public TAppCommonProcessor<TFindUserByPhone, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TFindUserByPhone, TEmptyConfig>;

public:
    TFindUserByPhone(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {}

    static TString GetTypeName() {
        return "find_user_by_phone";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;

private:
    bool CheckAndUpdateAccessCount(TUserPermissions::TPtr permissions, NDrive::TEntitySession& session) const;
    void OnFail(const TString& errorCode, NDrive::TEntitySession& session, const int errorState = HTTP_BAD_REQUEST);
};
