#include "route_builder.h"

#include <drive/backend/processors/common_app/fetcher.h>

#include <drive/backend/data/user_tags.h>

#include <drive/library/cpp/searchserver/http_status_config.h>

#include <library/cpp/json/json_reader.h>

#include <rtline/library/json/cast.h>


NJson::TJsonValue TGetServiceRouteProcessor::BuildOneTaskJsonReport(const TDBTag& tag, TTagDescription::TConstPtr tagDescription, const TServiceTask& task) {
    NJson::TJsonValue taskInfo = NJson::ToJson(task);
    tag.DoBuildReportItem(taskInfo);
    if (tagDescription) {
        taskInfo["tag_details"]["display_name"] = tagDescription->GetDisplayName();
    }
    return taskInfo;
}

void TGetServiceRouteProcessor::BuildJsonReport(TJsonReport::TGuard& g, const TServiceRoute& route, const TSet<TString>& droppedTasks, TUserPermissions::TPtr permissions,
                                    NDrive::TEntitySession& session) {
    const auto tags = [&]() -> TVector<TDBTag> {
        TVector<TDBTag> result;
        TSet<TString> tagIds;
        TMap<TString, i32> tagIdToPos;
        i32 idx = 0;
        for (const auto& task : route.GetTasks()) {
            if (task.IsGarage() || droppedTasks.contains(task.GetTagId())) {
                continue;
            }
            tagIds.insert(task.GetTagId());
            tagIdToPos[task.GetTagId()] = idx++;
        }
        R_ENSURE(Server->GetDriveAPI()->GetTagsManager().GetDeviceTags().RestoreTags(tagIds, result, session),
                        ConfigHttpStatus.UnknownErrorStatus, "can't restore tags");
        std::sort(result.begin(), result.end(), [&tagIdToPos](const TDBTag& lhs, const TDBTag& rhs) {
            return tagIdToPos[lhs.GetTagId()] < tagIdToPos[rhs.GetTagId()];
        });
        return result;
    }();

    NJson::TJsonValue tasks = NJson::JSON_ARRAY;
    TVector<TTaggedObject> objects;
    TSet<TString> necessaryCarsIds;
    auto taskIterator = route.GetTasks().begin();
    for (const auto& tag : tags) {
        R_ENSURE(Server->GetDriveAPI()->GetTagsManager().GetDeviceTags().GetObjectsByTagIds({tag.GetTagId()}, objects, session), ConfigHttpStatus.UnknownErrorStatus,
                        "can't get objects by id for tag " << tag.GetTagId() << ": " << session.GetStringReport());
        if (objects.size() != 1) {
            g.AddEvent(NJson::TMapBuilder
                ("event", "IncorrectObjectsCount")
                ("tag_id", tag.GetTagId()));
            continue;
        }
        const auto& carId = objects[0].GetId();
        necessaryCarsIds.insert(carId);
        taskIterator = std::find_if(taskIterator, route.GetTasks().end(), [&tag](const TServiceTask& task) {
            return task.GetTagId() == tag.GetTagId();
        });
        R_ENSURE(taskIterator != route.GetTasks().end(), ConfigHttpStatus.UnknownErrorStatus,
                        "can't find task with tag_id " << tag.GetTagId() << " in route");
        auto tagDescription = Server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetDescriptionByName(tag->GetName());
        tasks.AppendValue(BuildOneTaskJsonReport(tag, tagDescription, *taskIterator));
        ++taskIterator;
    }
    TCarsFetcher carsFetcher(*Server, NDeviceReport::ReportServiceApp);
    carsFetcher.SetIsRealtime(true);
    carsFetcher.SetCheckVisibility(false);
    R_ENSURE(carsFetcher.FetchData(permissions, necessaryCarsIds), ConfigHttpStatus.UnknownErrorStatus, "cannot fetch data for report");
    g.MutableReport().AddReportElement("routing_task_id", route.GetRoutingTaskId());
    g.MutableReport().AddReportElement("tasks", std::move(tasks));
    g.MutableReport().AddReportElementString("cars", carsFetcher.GetAvailableCarsReportSafe(permissions->GetFilterActions()));
    g.MutableReport().AddReportElement("models", carsFetcher.GetModelsReportSafe());
}

void TGetServiceRouteProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /* requestData */) {
    const TCgiParameters& cgi = Context->GetCgiParameters();
    TString userId = GetString(cgi, "user_id", false);
    if (!userId) {
        userId = permissions->GetUserId();
    } else if (permissions->GetUserId() != userId) {
        ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::ServiceRoute);
    }

    auto session = BuildTx<NSQL::ReadOnly>();
    const auto& api = *Yensured(Server->GetDriveAPI());
    auto optionalTaggedUser = api.GetTagsManager().GetUserTags().RestoreObject(userId, session);

    TMaybe<TDBTag> optionalTag = optionalTaggedUser ? optionalTaggedUser->GetTag(TServiceRouteTag::ServiceRouteTagName) : TMaybe<TDBTag>{};
    if (!optionalTag) {
        g.MutableReport().AddReportElement("started_at", 0);
        g.SetCode(HTTP_OK);
        return;
    }

    const auto& tag = *optionalTag;
    auto queryOptions = IEntityTagsManager::TQueryOptions{};
    queryOptions.SetActions({ EObjectHistoryAction::Add, EObjectHistoryAction::UpdateData }).SetObjectIds({ userId }).SetLimit(1).SetDescending(true);
    auto optionalEvents = api.GetTagsManager().GetUserTags().GetEventsByTag(tag.GetTagId(), session, 0, TInstant::Zero(), queryOptions);
    auto updatedAt = optionalEvents && !optionalEvents->empty() ? optionalEvents->front().GetHistoryTimestamp().Get() : TInstant::Zero();
    g.MutableReport().AddReportElement("started_at", updatedAt.Seconds());

    auto serviceRouteTag = tag.GetTagAs<TServiceRouteTag>();
    R_ENSURE(serviceRouteTag, ConfigHttpStatus.UnknownErrorStatus, "can't cast tag " << tag.GetTagId() << " as TServiceRouteTag");

    TSet<TString> droppedTasks;
    bool reportDroppedTasks = Server->GetSettings().GetValueDef<bool>("routing.report_dropped_tasks", true);
    TSet<TString> tagIds;
    for (const auto& task : serviceRouteTag->GetRoute().GetTasks()) {
        if (task.IsGarage()) {
            continue;
        }
        tagIds.insert(task.GetTagId());
    }
    if (!reportDroppedTasks) {
        auto now = Now();
        auto dropTasksHistorySize = Server->GetSettings().GetValueDef<TDuration>("routing.drop_tasks_history_size", TDuration::Days(1));
        auto optionalEvents = DriveApi->GetTagsManager().GetDeviceTags().GetEvents({ now - dropTasksHistorySize, now }, session, TTagEventsManager::TQueryOptions()
            .SetTagIds(std::move(tagIds)).SetUserIds({ userId }).SetActions({EObjectHistoryAction::DropTagPerformer})
        );
        R_ENSURE(optionalEvents, {}, "cannot fetch history events", session);

        for (const auto& ev : *optionalEvents) {
            droppedTasks.insert(ev.GetTagId());
        }
    }
    BuildJsonReport(g, serviceRouteTag->GetRoute(), droppedTasks, permissions, session);
    g.SetCode(HTTP_OK);
}
