#include "worker.h"

#include <drive/backend/data/user_tags.h>
#include <drive/backend/tags/tags_manager.h>
#include <drive/library/cpp/user_events_api/client.h>
#include <drive/library/cpp/user_events_api/realtime_user_data.h>

#include <library/cpp/http/misc/httpcodes.h>


void TGetSubordinateWorkersProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /* requestData */) {
    TString userId = permissions->GetUserId();
    auto subordinatesTagsNames = permissions->GetTagNamesByAction(TTagAction::ETagAction::UpdateObject);
    auto session = BuildTx<NSQL::ReadOnly>();
    const auto& api = *Yensured(Server->GetDriveAPI());
    const auto& userEventsApi = *Yensured(Server->GetUserEventsApi());

    TVector<TString> workersIds;
    {
        TVector<TDBTag> dbTags;
        R_ENSURE(api.GetTagsManager().GetUserTags().RestoreTags({}, MakeVector(subordinatesTagsNames), dbTags, session), {}, "cannot RestoreTags", session);
        for (const auto& tag : dbTags) {
            workersIds.push_back(tag.GetObjectId());
        }
    }

    TVector<NThreading::TFuture<NDrive::TRealtimeUserData>> futures;
    futures.reserve(workersIds.size());
    for (const auto& id : workersIds) {
        futures.push_back(userEventsApi.GetRealtimeUserInfo(id, true, false));
    }

    auto fetchResult = api.GetUsersData()->FetchInfo(workersIds, session);
    TMap<TString, TDriveUserData> workersInfo = std::move(fetchResult.MutableResult());

    TString tagName = Server->GetSettings().GetValue<TString>(TServiceZonesTag::AvailableZonesTagNameSettings).GetOrElse(TServiceZonesTag::DefaultAvailableZonesTagName);
    TMap<TString, TVector<TString>> workerToZones;
    {
        TVector<TDBTag> dbTags;
        R_ENSURE(api.GetTagsManager().GetUserTags().RestoreTags(MakeSet(workersIds), {tagName}, dbTags, session), {}, "cannot fetch service tags", session);
        for (auto& dbTag : dbTags) {
            auto tag = dbTag.MutableTagAs<TServiceZonesTag>();
            if (!tag) {
                continue;
            }
            workerToZones[dbTag.GetObjectId()] = std::move(tag->MutableAvailableZones());
        }
    }

    TMap<TString, TGeoCoord> workerToCoords;
    auto workerActivenessThreshold = GetHandlerSetting<TDuration>("worker_activeness_threshold").GetOrElse(TDuration::Max());
    auto now = Now();
    for (size_t i = 0; i < futures.size(); ++i) {
        try {
            auto value = futures[i].ExtractValueSync();
            if (now - workerActivenessThreshold < value.GetLastLocationUpdate()) {
                workerToCoords[workersIds[i]] = value.GetLocation();
            }
        } catch(const std::exception& e) {
            NDrive::TEventLog::Log("RealtimeUserInfoFetchError", NJson::TMapBuilder
                ("message", FormatExc(e))
            );
        }
    }

    NJson::TJsonValue result;
    for (const auto& workerId : workersIds) {
        auto& workerReport = result.AppendValue(NJson::JSON_MAP);
        workerReport["full_name"] = workersInfo[workerId].GetFullName();
        workerReport["available_zones"] = NJson::ToJson(workerToZones[workerId]);
        workerReport["id"] = workerId;
        auto it = workerToCoords.find(workerId);
        if (it != workerToCoords.end()) {
            workerReport["location"] = workerToCoords[workerId].SerializeLatLonToJson();
        }
    }
    g.MutableReport().AddReportElement("workers", std::move(result));
    g.SetCode(HTTP_OK);
}

void TPerformForSubordinateWorkerProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    auto workerId = GetUUID(requestData, "worker_id");
    auto tagId = GetUUID(requestData, "tag_id");
    const auto& api = *Yensured(Server->GetDriveAPI());
    const auto& deviceTagsManager = api.GetTagsManager().GetDeviceTags();
    const auto& userTagsManager = api.GetTagsManager().GetUserTags();
    auto session = BuildTx<NSQL::Writable>();
    session.SetOriginatorId(permissions->GetUserId());
    auto subordinatesTagsNames = permissions->GetTagNamesByAction(TTagAction::ETagAction::UpdateObject);
    TVector<TDBTag> dbTags;
    R_ENSURE(userTagsManager.RestoreTags({ workerId }, MakeVector(subordinatesTagsNames), dbTags, session), {}, "can not RestoreTags", session);
    R_ENSURE(!dbTags.empty(), HTTP_FORBIDDEN, "cannot set " << workerId << " as performer", session);
    auto workerPermissions = api.GetUserPermissions(workerId);
    R_ENSURE(deviceTagsManager.InitPerformer({ tagId }, *Yensured(workerPermissions.Get()), Server, session), {}, "cannot InitPerformer", session);
    R_ENSURE(session.Commit(), {}, "cannot Commit", session);
    g.SetCode(HTTP_OK);
}
