#include "zones.h"

#include <drive/backend/areas/areas.h>
#include <drive/backend/data/user_tags.h>

#include <rtline/library/json/adapters.h>

const TString TServiceZonesListProcessors::CityToZonesIdsSettings = "city_to_zones_mapping";

void TServiceZonesListProcessors::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr /* permissions */, const NJson::TJsonValue& /* requestData */) {
    const auto& cgi = Context->GetCgiParameters();
    TSet<TString> ids = MakeSet(GetStrings(cgi, "ids", false));
    const TString city = GetString(cgi, "city", false);
    R_ENSURE(ids || city, HTTP_BAD_REQUEST, "expected ids or city parameter in input");
    if (!ids) {
        TMaybe<TString> cityToZonesIdsStr = GetHandlerSetting<TString>(CityToZonesIdsSettings);
        R_ENSURE(cityToZonesIdsStr, HTTP_INTERNAL_SERVER_ERROR, "can't find city to zones config");
        NJson::TJsonValue jsonValue;
        TMap<TString, TSet<TString>> result;
        R_ENSURE(NJson::ReadJsonFastTree(*cityToZonesIdsStr, &jsonValue), HTTP_INTERNAL_SERVER_ERROR, "can't parse settings json");
        R_ENSURE(NJson::TryFromJson(jsonValue, NJson::Dictionary(result)), HTTP_INTERNAL_SERVER_ERROR, "can't convert json to settings");

        auto it = result.find(city);
        R_ENSURE(it != result.end(), HTTP_BAD_REQUEST, "no city " << city << " in config");
        ids = it->second;
    }
    TConstAreasSnapshot::TPtr areas = Server->GetDriveAPI()->GetAreasDB()->GetSnapshot(Context->GetRequestStartTime());
    R_ENSURE(areas, HTTP_INTERNAL_SERVER_ERROR, "can't get areas snapshot");
    NJson::TJsonValue report;
    for (const auto& areaInfo : *areas) {
        if (areaInfo.GetArea().GetType() != TServiceZonesTag::ServiceZoneAreaType) {
            continue;
        }
        if (!ids.contains(areaInfo.GetArea().GetIdentifier())) {
            continue;
        }
        report[areaInfo.GetArea().GetIdentifier()] = areaInfo.SerializeToReport();
    }
    g.MutableReport().AddReportElement("areas", std::move(report));
    g.SetCode(HTTP_OK);
}
