#include "processor.h"

#include <drive/backend/processors/common_app/processor.h>

#include <drive/backend/billing/manager.h>
#include <drive/backend/cars/hardware.h>
#include <drive/backend/compiled_riding/manager.h>
#include <drive/backend/context_fetcher/json.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/data/feedback.h>
#include <drive/backend/device_snapshot/image.h>
#include <drive/backend/history_iterator/history_iterator.h>
#include <drive/backend/images/database.h>
#include <drive/backend/major/client.h>
#include <drive/backend/sessions/manager/billing.h>

#include <rtline/library/json/merge.h>
#include <rtline/util/algorithm/ptr.h>

#include <util/string/builder.h>
#include <util/string/split.h>
#include <util/string/vector.h>

void TRemoveTagProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*jsonValue*/) {
    const TCgiParameters& cgi = Context->GetCgiParameters();
    const TString& tagName = GetString(cgi, "tag_name");
    const TString& userId = permissions->GetUserId();

    if (IsTrue(Context->GetCgiParameters().Get("real_remove"))) {
        ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Remove, TAdministrativeAction::EEntity::Tag);

        NDrive::TEntitySession session = BuildTx<NSQL::Writable>();
        if (DriveApi->GetTagsManager().GetTagsMeta().UnregisterTag(tagName, userId, session) && session.Commit()) {
            g.SetCode(HTTP_OK);
        } else {
            g.MutableReport().AddReportElement("error", session.GetReport());
            g.SetCode(ConfigHttpStatus.UnknownErrorStatus);
        }
    } else {
        ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::Tag);

        auto td = DriveApi->GetTagsManager().GetTagsMeta().GetDescriptionByName(tagName);
        R_ENSURE(td, HTTP_BAD_REQUEST, "cannot get description for tag " << tagName);
        TTagDescription::TPtr tdCopy = td->Clone();
        R_ENSURE(tdCopy, HTTP_INTERNAL_SERVER_ERROR, "cannot copy description");
        if (tdCopy->GetDeprecated()) {
            g.SetCode(HTTP_OK);
            return;
        }
        tdCopy->SetDeprecated(true);
        NDrive::TEntitySession session = BuildTx<NSQL::Writable>();
        if (DriveApi->GetTagsManager().GetTagsMeta().RegisterTag(tdCopy, userId, session) && session.Commit()) {
            g.SetCode(HTTP_OK);
        } else {
            g.MutableReport().AddReportElement("error", session.GetReport());
            g.SetCode(ConfigHttpStatus.UnknownErrorStatus);
        }
    }
}

TMaybe<TTaggedObject> TListEntityTagsProcessor::GetTaggedObject(const TString& objectId, NDrive::TEntitySession& session, NDrive::TEntitySession& sessionForYdb) const {
    Y_UNUSED(sessionForYdb);
    auto taggedObject = GetEntityTagsManager().RestoreObject(objectId, session);
    return taggedObject;
}

void TListEntityTagsProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*jsonValue*/) {
    TString objectId = Context->GetCgiParameters().Has("object_id") ? Context->GetCgiParameters().Get("object_id") : Context->GetCgiParameters().Get("car_id");

    if (!objectId) {
        // For yaauto server requests get car data from cgi
        TString headDeviceId = Context->GetCgiParameters().Get("head_id");
        if (!headDeviceId) {
            // For yaauto requests get car data from header
            headDeviceId = Context->GetRequestData().HeaderInOrEmpty("DeviceId");
        }
        objectId = DriveApi->GetCarAttachmentAssignments().GetAttachmentOwnerByServiceAppSlug(headDeviceId);
    }

    if (IsObjectIdUUID()) {
        Y_ENSURE_EX(!GetUuid(objectId).IsEmpty(), TCodedException(ConfigHttpStatus.SyntaxErrorStatus) << "incorrect 'object_id' / 'car_id' in parameters (no uuid)");
    }

    auto session = BuildTx<NSQL::ReadOnly>();
    auto ydbTx = BuildYdbTx<NSQL::Deferred | NSQL::ReadOnly>("list_entity_tags");

    auto taggedObject = GetTaggedObject(objectId, session, ydbTx);
    R_ENSURE(taggedObject, {}, "cannot GetTaggedObject " << objectId, session);

    TUserPermissions::TUnvisibilityInfoSet invisibilityInfo = 0;
    R_ENSURE(
        permissions->GetVisibility(*taggedObject, GetEntityTagsManager().GetEntityType(), &invisibilityInfo) != TUserPermissions::EVisibility::NoVisible,
        HTTP_FORBIDDEN,
        "no permissions for object observe: " << TUserPermissions::ExplainInvisibility(invisibilityInfo),
        session
    );

    const TSet<TString>* tagsForObserve = permissions->GetTagNamesByActionPtr(TTagAction::ETagAction::Observe);
    bool observeDetails = permissions->CheckAdministrativeActions(TAdministrativeAction::EAction::ObserveStructure, TAdministrativeAction::EEntity::Tag, TypeName);

    TSet<TString> userIds;
    for (auto&& tag : taggedObject->GetTags()) {
        if (tag && tag->GetPerformer()) {
            userIds.emplace(tag->GetPerformer());
        }
    }

    NJson::TJsonValue results = NJson::JSON_ARRAY;
    auto gUsers = DriveApi->GetUsersData()->FetchInfo(userIds, session);
    for (auto&& carTag : taggedObject->GetTags()) {
        if (!carTag) {
            continue;
        }
        if (!!tagsForObserve && !tagsForObserve->contains(carTag->GetName())) {
            continue;
        }
        NJson::TJsonValue tagJson = carTag.SerializeToJson();
        auto description = DriveApi->GetTagsManager().GetTagsMeta().GetDescriptionByName(carTag->GetName());
        if (description) {
            tagJson["display_name"] = description->GetDisplayName();
        } else {
            tagJson["display_name"] = carTag->GetName();
        }
        if (observeDetails && !!carTag->GetPerformer()) {
            const TDriveUserData* userInfo = gUsers.GetResultPtr(carTag->GetPerformer());
            if (!!userInfo) {
                tagJson.InsertValue("performer_details", userInfo->GetPublicReport());
            }
        }
        results.AppendValue(tagJson);
    }
    g.MutableReport().AddReportElement("records", std::move(results));

    if (observeDetails && GetEntityTagsManager().GetPropositionsPtr()) {
        NJson::TJsonValue propositionsJson = NJson::JSON_ARRAY;
        auto optionalPropositions = GetEntityTagsManager().GetPropositionsPtr()->GetByObjectId({objectId}, session);
        R_ENSURE(optionalPropositions, {}, "cant get propositions", session);
        auto propositions = *optionalPropositions;

        TSet<TString> propositionUserIds;
        for (auto&& i : propositions) {
            if (!!tagsForObserve && !tagsForObserve->contains(i.second->GetName())) {
                continue;
            }
            propositionsJson.AppendValue(i.second.BuildJsonReport());
            i.second.FillUsers(propositionUserIds);
        }
        g.MutableReport().AddReportElement("propositions", std::move(propositionsJson));
        auto gPropositionUsers = DriveApi->GetUsersData()->FetchInfo(propositionUserIds, session);
        NJson::TJsonValue usersJson = NJson::JSON_ARRAY;
        for (auto&& user : gPropositionUsers) {
            usersJson.AppendValue(user.second.GetReport(permissions->GetUserReportTraits()));
        }
        if (gPropositionUsers.size()) {
            g.MutableReport().AddReportElement("proposition_users", std::move(usersJson));
        }
    }

    g.SetCode(HTTP_OK);
}

TSet<TString> TRemoveEntityTagProcessor::GetEntityObjectIds(const NJson::TJsonValue& jsonValue) const {
    return MakeSet<TString>(GetStrings(jsonValue, "object_ids", false));
}

TSet<TString> TRemoveCarTagProcessor::GetEntityObjectIds(const NJson::TJsonValue& jsonValue) const {
    auto objectIds = TRemoveEntityTagProcessor::GetEntityObjectIds(jsonValue);
    if (objectIds.empty()) {
        objectIds = MakeSet<TString>(GetStrings(jsonValue, "car_ids", false));
    }
    if (objectIds.empty()) {
        const auto& attachments = DriveApi->GetCarAttachmentAssignments();
        auto headIds = GetStrings(jsonValue, "head_id", false);
        for (auto&& headId : headIds) {
            auto objectId = attachments.GetCarByHeadId(headId);
            R_ENSURE(!objectId.empty(), ConfigHttpStatus.EmptySetStatus, "cannot find object with head_id " << headId);
            objectIds.emplace(objectId);
        }
    }
    return objectIds;
}

void TRemoveEntityTagProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& jsonValue) {
    const TString& userId = permissions->GetUserId();
    const TCgiParameters& cgi = Context->GetCgiParameters();
    const TManagers& managers = GetEntityTagsManagers();
    for (auto&& manager : managers) {
        R_ENSURE(manager, ConfigHttpStatus.UnknownErrorStatus, "null manager");
    }

    TVector<TString> tagNames;
    if (tagNames.empty()) {
        tagNames = GetStrings(jsonValue, "tag_names", /*required=*/false);
    }
    if (tagNames.empty()) {
        tagNames = GetStrings(cgi, "tag_names", /*required=*/false);
    }
    auto force = GetValue<bool>(cgi, "force", false).GetOrElse(false);
    auto tryRemove = GetValue<bool>(cgi, "try_remove", false).GetOrElse(false);

    TMap<TManagerPtr, TVector<TDBTag>> managerTags;
    if (!tagNames.empty()) {
        TSet<TString> objectIds = GetEntityObjectIds(jsonValue);
        if (objectIds.empty()) {
            auto session = GetCurrentUserSession(permissions, TInstant::Zero());
            if (session) {
                objectIds.emplace(session->GetObjectId());
            }
        }
        R_ENSURE(!objectIds.empty(), ConfigHttpStatus.UserErrorState, "no object_ids provided");
        {
            NDrive::TEntitySession session = BuildTx<NSQL::ReadOnly>();
            for (auto&& manager : managers) {
                if (!manager->RestoreTags(objectIds, tagNames, managerTags[manager], session)) {
                    session.DoExceptionOnFail(ConfigHttpStatus);
                }
            }
        }
    } else {
        TVector<TString> tagIds;
        if (tagIds.empty()) {
            tagIds = GetUUIDs(cgi, "tag_id", false);
        }
        if (tagIds.empty()) {
            tagIds = GetUUIDs(jsonValue, "tag_id");
        }
        NDrive::TEntitySession session = BuildTx<NSQL::ReadOnly>();
        for (auto&& manager : managers) {
            if (!manager->RestoreTags(MakeSet(tagIds), managerTags[manager], session)) {
                session.DoExceptionOnFail(ConfigHttpStatus);
            }
        }
    }

    auto removePerformSet = permissions->GetTagNamesByAction(TTagAction::ETagAction::RemovePerform);
    auto removeSet = permissions->GetTagNamesByAction(TTagAction::ETagAction::Remove);
    for (auto&&[manager, tags] : managerTags) {
        for (auto&& i : tags) {
            const TString& name = i->GetName();
            const TString& performer = i->GetPerformer();
            R_ENSURE(removeSet.contains(name), ConfigHttpStatus.PermissionDeniedStatus, "no_permissions_remove_tag:" + name, NDrive::MakeError("no_permissions_to_remove_tag"));
            if (performer) {
                R_ENSURE(removePerformSet.contains(name), ConfigHttpStatus.PermissionDeniedStatus, "no_permissions_remove_perform_tag:" + name, NDrive::MakeError("no_permissions_to_remove_tag"));
            }
        }
        for (size_t i = 0; i < tags.size(); ) {
            size_t next = Min<size_t>(i + 50, tags.size());
            TVector<TDBTag> localTags(tags.begin() + i, tags.begin() + next);
            NDrive::TEntitySession session = BuildTx<NSQL::Writable>();
            if (!manager->RemoveTags(localTags, userId, Server, session, force, tryRemove) || !session.Commit()) {
                session.DoExceptionOnFail(ConfigHttpStatus);
            }
            i = next;
        }
    }
    g.SetCode(HTTP_OK);
}

TRemoveEntityTagProcessor::TManagers TRemoveGenericTagProcessor::GetEntityTagsManagers() const {
    const TCgiParameters& cgi = Context->GetCgiParameters();
    const auto entities = GetValues<NEntityTagsManager::EEntityType>(cgi, "entities");
    TManagers result;
    for (auto&& entity : entities) {
        result.push_back(&DriveApi->GetEntityTagsManager(entity));
    }
    return result;
}

TSet<TString> TAddEntityTagProcessor::GetEntityObjectIds(const NJson::TJsonValue& jsonValue, NDrive::TEntitySession& /*session*/, TUserPermissions::TPtr /*permissions*/) const {
    TSet<TString> objectIds;
    if (objectIds.empty()) {
        objectIds = MakeSet(GetStrings(jsonValue, "object_ids", false));
    }
    if (objectIds.empty()) {
        auto objectId = GetString(jsonValue, "object_id", false);
        if (objectId) {
            objectIds.insert(objectId);
        }
    }
    return objectIds;
}

TSet<TString> TAddCarTagProcessor::GetEntityObjectIds(const NJson::TJsonValue& jsonValue, NDrive::TEntitySession& session, TUserPermissions::TPtr permissions) const {
    TSet<TString> objectIds = TAddEntityTagProcessor::GetEntityObjectIds(jsonValue, session);
    if (objectIds.empty()) {
        auto objectId = GetString(jsonValue, "car_id", false);
        if (objectId) {
            objectIds.insert(objectId);
        }
    }
    if (objectIds.empty()) {
        auto carNumber = GetString(jsonValue, "car_number", false);
        if (carNumber) {
            auto objectId = DriveApi->GetCarIdByNumber(carNumber);
            R_ENSURE(objectId, ConfigHttpStatus.EmptySetStatus, "cannot find object with car_number " << carNumber);
            objectIds.insert(objectId);
        }
    }
    if (objectIds.empty()) {
        const auto& attachments = DriveApi->GetCarAttachmentAssignments();
        auto headIds = GetStrings(jsonValue, "head_id", false);
        for (auto&& headId : headIds) {
            auto objectId = attachments.GetCarByHeadId(headId);
            R_ENSURE(!objectId.empty(), ConfigHttpStatus.EmptySetStatus, "cannot find object with head_id " << headId);
            objectIds.insert(objectId);
        }
    }
    if (objectIds.empty()) {
        auto session = GetCurrentUserSession(permissions, TInstant::Zero());
        if (session) {
            objectIds.emplace(session->GetObjectId());
        }
    }
    return objectIds;
}

TSet<TString> TAddUserTagProcessor::GetEntityObjectIds(const NJson::TJsonValue& jsonValue, NDrive::TEntitySession& session, TUserPermissions::TPtr permissions) const {
    TSet<TString> objectIds = TAddEntityTagProcessor::GetEntityObjectIds(jsonValue, session);
    if (objectIds.empty()) {
        auto uids = GetStrings(jsonValue, "uids", false);
        if (uids) {
            ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Add, TAdministrativeAction::EEntity::User);
        }
        for (auto&& uid : uids) {
            NDrive::TExternalUser externalUser;
            externalUser.SetUid(uid);
            auto user = DriveApi->GetUsersData()->FindOrRegisterExternal(permissions->GetUserId(), session, externalUser);
            R_ENSURE(user, {}, "cannot FindOrRegisterExternal by uid " << uid, session);
            objectIds.emplace(user->GetUserId());
        }
    }

    if (objectIds.empty() && permissions) {
        objectIds.insert(permissions->GetUserId());
    }
    return objectIds;
}

void TAddEntityTagProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& jsonValue) {
    const TCgiParameters& cgi = Context->GetCgiParameters();
    const auto uniquePolicy = GetValue<EUniquePolicy>(cgi, "unique_policy", false).GetOrElse(EUniquePolicy::Undefined);

    NDrive::TEntitySession session = BuildTx<NSQL::Writable>();

    TSet<TString> objectIds = GetEntityObjectIds(jsonValue, session, permissions);
    R_ENSURE(!objectIds.empty(), ConfigHttpStatus.EmptyRequestStatus, "no object_ids provided");

    const ui32 objectsForTagsLimit = Server->GetSettings().GetHandlerValueDef(TypeName, "limit_objects_for_tags", 1000);
    R_ENSURE(objectsForTagsLimit >= objectIds.size(), HTTP_BAD_REQUEST, "incorrect 'object_ids' (limit is " << objectsForTagsLimit << ")");
    if (IsObjectIdUUID()) {
        for (auto&& i : objectIds) {
            ValidateUUID(i);
        }
    }

    auto customTag = GetString(cgi, "tag_name", false);
    TVector<ITag::TPtr> tags;

    if (jsonValue.Has("add_tags")) {
        const NJson::TJsonValue::TArray* arr = nullptr;
        ReqCheckCondition(jsonValue["add_tags"].GetArrayPointer(&arr), ConfigHttpStatus.SyntaxErrorStatus, EDriveLocalizationCodes::SyntaxUserError);
        for (auto&& i : *arr) {
            TMessagesCollector errors;
            NJson::TJsonValue jsonTag = i;
            if (customTag) {
                jsonTag["tag"] = customTag;
            }

            auto tag = IJsonSerializableTag::BuildFromJson(DriveApi->GetTagsManager(), jsonTag, &errors);
            tags.emplace_back(std::move(tag));
            ReqCheckCondition(
                !!tags.back(),
                ConfigHttpStatus.SyntaxErrorStatus,
                TStringBuilder() << "cannot_build_tag from " << i.GetStringRobust() << ": " << errors.GetStringReport()
            );
            tags.back()->SetPerformer("");
        }
    } else {
        TMessagesCollector errors;
        NJson::TJsonValue jsonTag = jsonValue;
        if (customTag) {
            jsonTag["tag"] = customTag;
        }
        auto tag = IJsonSerializableTag::BuildFromJson(DriveApi->GetTagsManager(), jsonTag, &errors);
        tags.emplace_back(std::move(tag));
        ReqCheckCondition(
            !!tags.back(),
            ConfigHttpStatus.SyntaxErrorStatus,
            TStringBuilder() << "cannot_build_tag from " << jsonValue.GetStringRobust() << ": " << errors.GetStringReport()
        );
        tags.back()->SetPerformer("");
    }

    const auto reportTagInfo = GetValue<bool>(cgi, "report_tag_info", false).GetOrElse(false);
    const auto performerId = GetString(jsonValue, "performer", false);
    NJson::TJsonValue taggedObjectsJson = NJson::JSON_ARRAY;
    TSet<TString> addedTagIds;

    if (GetHandlerSettingDef<bool>("check_object_tag_action", false)) {
        TMap<TString, TTaggedObject> taggedObjects;
        R_ENSURE(
            GetEntityTagsManager().RestoreObjects(objectIds, taggedObjects, session),
            ConfigHttpStatus.UnknownErrorStatus,
            "can't restore objects",
            session
        );
        for (const auto& tagPtr : tags) {
            TDBTag dbTag;
            dbTag.SetData(tagPtr);
            for (const auto& [_, taggedObject] : taggedObjects) {
                R_ENSURE(
                    permissions->CheckObjectTagAction(TTagAction::ETagAction::Add, dbTag, taggedObject, {}),
                    ConfigHttpStatus.PermissionDeniedStatus,
                    "no permissions to add " << dbTag->GetName() << " tag on object " << taggedObject.GetId()
                );
            }
        }
    } else {
        for (const auto& tagPtr : tags) {
            R_ENSURE(
                permissions->GetTagNamesByAction(TTagAction::ETagAction::Add).contains(tagPtr->GetName()),
                ConfigHttpStatus.PermissionDeniedStatus,
                "no permissions for add " << tags.back()->GetName() << " tag"
            );
        }
    }

    for (auto&& i : objectIds) {
        auto optionalAddedTags = GetEntityTagsManager().AddTags(tags, permissions->GetUserId(), i, Server, session, uniquePolicy);
        R_ENSURE(optionalAddedTags, {}, "cannot AddTags", session);

        NJson::TJsonValue taggedObject;
        taggedObject.InsertValue("object_id", i);

        NJson::TJsonValue tagIds = NJson::JSON_ARRAY;
        for (auto&& t : *optionalAddedTags) {
            tagIds.AppendValue(t.GetTagId());
            if (reportTagInfo || performerId) {
                addedTagIds.emplace(t.GetTagId());
            }
        }
        taggedObject.InsertValue("tag_id", std::move(tagIds));
        taggedObjectsJson.AppendValue(std::move(taggedObject));
    }

    if (reportTagInfo || performerId) {
        NJson::TJsonValue addedTagsJson = NJson::JSON_ARRAY;
        auto addedTags = GetEntityTagsManager().RestoreTags(addedTagIds, session);
        R_ENSURE(addedTags, {}, "cannot RestoreTags", session);

        if (performerId) {
            const auto permissions = DriveApi->GetUserPermissions(performerId, TUserPermissionsFeatures());
            R_ENSURE(
                permissions && GetEntityTagsManager().InitPerformer(*addedTags, *permissions, Server, session),
                ConfigHttpStatus.PermissionDeniedStatus,
                "failed to init performer " + performerId,
                session
            );
        }

        if (reportTagInfo) {
            for (auto&& addedTag : *addedTags) {
                addedTagsJson.AppendValue(addedTag.BuildJsonReport());
            }
            g.MutableReport().AddReportElement("added_tags_data", std::move(addedTagsJson));
        }

    }

    if (!session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }

    g.MutableReport().AddReportElement("tagged_objects", std::move(taggedObjectsJson));
    g.SetCode(HTTP_OK);
}

void TUpdateEntityTagProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& jsonValue) {
    const TString& tagId = GetUUID(jsonValue, "tag_id", true);

    const IEntityTagsManager& manager = GetEntityTagsManager();
    auto session = BuildTx<NSQL::Writable>();
    auto optionalTag = manager.RestoreTag(tagId, session);
    R_ENSURE(optionalTag, {}, "cannot RestoreTag " << tagId, session);
    auto tag = *optionalTag;
    R_ENSURE(tag, ConfigHttpStatus.EmptySetStatus, "cannot find tag_id " << tagId);

    TTagAction::ETagAction needAction;
    if (tag->GetPerformer().empty()) {
        needAction = TTagAction::ETagAction::Update;
    } else {
        needAction = ((tag->GetPerformer() == permissions->GetUserId()) ? TTagAction::ETagAction::UpdateOnPerform : TTagAction::ETagAction::UpdateOnPerformByAnother);
    }
    Y_ENSURE_EX(permissions->GetTagNamesByAction(needAction).contains(tag->GetName()),
        TCodedException(ConfigHttpStatus.PermissionDeniedStatus) << "No permissions for update this type of tag");

    ITag::TPtr data;
    if (jsonValue.Has("tag") || jsonValue.Has("tag_name")) {
        data = IJsonSerializableTag::BuildFromJson(DriveApi->GetTagsManager(), jsonValue, &session.MutableMessages());
    } else {
        auto serializedData = jsonValue;
        serializedData["tag"] = tag->GetName();
        data = IJsonSerializableTag::BuildFromJson(DriveApi->GetTagsManager(), serializedData, &session.MutableMessages());
    }
    R_ENSURE(data, ConfigHttpStatus.SyntaxErrorStatus, "cannot build tag", session);
    R_ENSURE(data->GetName() == tag->GetName(), ConfigHttpStatus.SyntaxErrorStatus, "tag name inconsistency: " << tag->GetName(), session);

    if (!GetEntityTagsManager().UpdateTagData(tag, data, permissions->GetUserId(), Server, session) || !session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }
    g.SetCode(HTTP_OK);
}

void TDetailEntityTagProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*jsonValue*/) {
    const TCgiParameters& cgi = Context->GetCgiParameters();
    const TString& tagId = GetUUID(cgi, "tag_id");

    const IEntityTagsManager& manager = GetEntityTagsManager();
    auto session = BuildTx<NSQL::ReadOnly>();
    auto optionalTag = manager.RestoreTag(tagId, session);
    R_ENSURE(optionalTag, {}, "cannot RestoreTag " << tagId, session);
    const auto& tag = *optionalTag;
    R_ENSURE(tag, ConfigHttpStatus.EmptySetStatus, "cannot find tag_id " << tagId);

    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::ObserveStructure, TAdministrativeAction::EEntity::Tag, tag->GetName());
    g.MutableReport().AddReportElement("tag", tag ? tag->SerializeToJson() : NJson::JSON_NULL);

    auto optionalEvents = manager.GetEventsByTag(tag.GetTagId(), session);
    R_ENSURE(optionalEvents, {}, "cannot GetEventsByTag " << tag.GetTagId(), session);
    {
        NJson::TJsonValue history(NJson::JSON_ARRAY);
        for (auto&& ev : *optionalEvents) {
            if (ev) {
                history.AppendValue(ev.BuildReportItem());
            } else {
                history.AppendValue(NJson::JSON_NULL);
            }
        }
        g.MutableReport().AddReportElement("history", std::move(history));
    }

    auto snapshot = tag ? tag->GetObjectSnapshot() : nullptr;
    g.MutableReport().AddReportElement("snapshot", snapshot ? snapshot->SerializeToJson() : NJson::JSON_NULL);
    g.SetCode(HTTP_OK);
}

TString TSetCarTagProcessor::GetDeviceBySession(const TString& sessionId, TUserPermissions::TPtr permissions, NDrive::TEntitySession& tx) const {
    auto optionalSession = DriveApi->GetSessionManager().GetSession(sessionId, tx);
    R_ENSURE(optionalSession, {}, "cannot GetSession " << sessionId, tx);
    auto session = *optionalSession;
    if (session) {
        if (session->GetUserId() != permissions->GetUserId()) {
            ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Delegation, TAdministrativeAction::EEntity::Car);
        }
        return session->GetObjectId();
    }
    const auto& compiledSessionManager = DriveApi->GetMinimalCompiledRides();
    auto ydbTx = BuildYdbTx<NSQL::ReadOnly>("set_car_tag");
    auto compiledSessions = compiledSessionManager.Get<TMinimalCompiledRiding>({ sessionId }, tx, ydbTx);
    if (!compiledSessions) {
        tx.Check();
    }
    for (auto&& compiledSession : *compiledSessions) {
        if (compiledSession.GetHistoryUserId() != permissions->GetUserId()) {
            ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Delegation, TAdministrativeAction::EEntity::Car);
        }
        return compiledSession.GetObjectId();
    }
    return {};
}

void TSetCarTagProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& jsonValue) {
    const auto& cgi = Context->GetCgiParameters();
    auto objectId = GetUUID(cgi, "car_id", false);
    auto objectType = NEntityTagsManager::EEntityType::Car;
    auto locale = GetLocale();

    const TString& userId = permissions->GetUserId();
    auto session = BuildTx<NSQL::Writable>();
    TString deviceId;
    if (objectId) {
        ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Delegation, TAdministrativeAction::EEntity::Car);
        deviceId = objectId;
    } else if (jsonValue.Has("session_id")) {
        objectType = NEntityTagsManager::EEntityType::Trace;
        objectId = GetString(jsonValue, "session_id", false);
        deviceId = GetDeviceBySession(objectId, permissions, session);
    } else if (Context->GetCgiParameters().Has("session_id")) {
        auto sessionId = GetString(cgi, "session_id", false);
        deviceId = GetDeviceBySession(sessionId, permissions, session);
        objectId = deviceId;
    } else {
        TVector<TDBTag> perfTags;
        if (!Server->GetDriveAPI()->GetTagsManager().GetDeviceTags().RestorePerformerTags({permissions->GetUserId()}, perfTags, session)) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
        for (auto&& i : perfTags) {
            if (i.GetTagAs<TChargableTag>()) {
                R_ENSURE(!objectId || objectId == i.GetObjectId(), ConfigHttpStatus.UserErrorState, "incorrect_cars_selected", NDrive::MakeError("incorrect_cars_for_user"), session);
                objectId = i.GetObjectId();
            }
        }
        if (!objectId) {
            for (auto&& i : perfTags) {
                R_ENSURE(!objectId || objectId == i.GetObjectId(), ConfigHttpStatus.UserErrorState, "incorrect_cars_selected", NDrive::MakeError("incorrect_cars_for_user"), session);
                objectId = i.GetObjectId();
            }
        }
        deviceId = objectId;
    }
    const IEntityTagsManager& tagsManager = Server->GetDriveAPI()->GetEntityTagsManager(objectType);

    R_ENSURE(objectId && !!deviceId, ConfigHttpStatus.UserErrorState, "cannot_detect_car", NDrive::MakeError("incorrect_cars_for_user"), session);

    auto addSet = permissions->GetTagNamesByAction(TTagAction::ETagAction::Add);
    auto removeSet = permissions->GetTagNamesByAction(TTagAction::ETagAction::Remove);
    auto totalResponse = TMaybe<TFeedbackTraceTag::TResponse>();
    {
        if (jsonValue.Has("remove")) {
            const NJson::TJsonValue& removeJson = jsonValue["remove"];
            TVector<TString> tagNames;
            if (removeJson.Has("names")) {
                const NJson::TJsonValue& removeJsonNames = removeJson["names"];
                R_ENSURE(removeJsonNames.IsArray(), ConfigHttpStatus.SyntaxErrorStatus, "incorrect_request");
                for (auto&& i : removeJsonNames.GetArraySafe()) {
                    R_ENSURE(i.IsString(), ConfigHttpStatus.SyntaxErrorStatus, "incorrect_request");
                    tagNames.emplace_back(i.GetString());
                    R_ENSURE(removeSet.contains(i.GetString()), ConfigHttpStatus.PermissionDeniedStatus, "Have not permissions for remove tag " + i.GetString());
                }
                if (tagNames.size()) {
                    TVector<TDBTag> tagsDB;
                    if (!tagsManager.RestoreTags({objectId}, tagNames, tagsDB, session)) {
                        session.DoExceptionOnFail(ConfigHttpStatus);
                    }
                    tagsDB.erase(std::remove_if(tagsDB.begin(), tagsDB.end(), [permissions](const TDBTag& tag) {
                        if (!tag) {
                            return true;
                        }
                        const TTagAction::TTagActions actions = permissions->GetActionsByTagIdx(tag->GetDescriptionIndex());
                        if (!tag->GetPerformer()) {
                            return (actions & (ui64)TTagAction::ETagAction::Remove) == 0;
                        } else if (tag->GetPerformer() == permissions->GetUserId()) {
                            return (actions & (ui64)TTagAction::ETagAction::RemovePerform) == 0;
                        } else {
                            return ((actions & (ui64)TTagAction::ETagAction::ForcePerform) && (actions & (ui64)TTagAction::ETagAction::RemovePerform)) == false;
                        }
                    }), tagsDB.end());

                    if (!tagsManager.RemoveTags(tagsDB, permissions->GetUserId(), Server, session, false)) {
                        session.DoExceptionOnFail(ConfigHttpStatus);
                    }
                }
            }
        }

        if (jsonValue.Has("add")) {
            const NJson::TJsonValue& addJson = jsonValue["add"];
            TVector<TString> tagNames;
            R_ENSURE(addJson.IsArray(), ConfigHttpStatus.SyntaxErrorStatus, "incorrect_request");
            TVector<ITag::TPtr> newTags;
            TVector<TAtomicSharedPtr<TFeedbackTraceTag>> feedbackTags;

            auto getPhotoSnapshot = [&](const NJson::TJsonValue& jsonValue) -> NDrive::IObjectSnapshot::TPtr {
                if (!jsonValue.Has("photos")) {
                    return nullptr;
                }
                NDrive::IObjectSnapshot::TPtr snapshot;
                snapshot = DriveApi->GetImagesDB().BuildSnapshot(
                    jsonValue,
                    deviceId,
                    NEntityTagsManager::EEntityType::Car,
                    "feedback",
                    permissions->GetUserId(),
                    Config.GetGeneratePreviewPath(),
                    session,
                    objectId
                );
                R_ENSURE(
                    !!snapshot,
                    ConfigHttpStatus.SyntaxErrorStatus,
                    TStringBuilder() << "cannot build snapshot from " << jsonValue["photos"].GetStringRobust() << ": " + session.GetStringReport()
                );
                return snapshot;
            };

            NDrive::IObjectSnapshot::TPtr commonPhotoSnapshot = getPhotoSnapshot(jsonValue);
            for (auto&& i : addJson.GetArraySafe()) {
                TMessagesCollector errors;
                ITag::TPtr tag = IJsonSerializableTag::BuildFromJson(DriveApi->GetTagsManager(), i, &errors);
                R_ENSURE(tag, ConfigHttpStatus.SyntaxErrorStatus, "Cannot build tag: " + errors.GetStringReport() + "/" + i.GetStringRobust());

                auto photoSnapshot = getPhotoSnapshot(i);
                tag->SetObjectSnapshot(photoSnapshot ? photoSnapshot : commonPhotoSnapshot);

                const TTagAction::TTagActions actions = permissions->GetActionsByTagIdx(tag->GetDescriptionIndex());
                ReqCheckCondition(actions & (ui64)TTagAction::ETagAction::Add, ConfigHttpStatus.PermissionDeniedStatus, EDriveLocalizationCodes::NoPermissions);
                R_ENSURE(
                    addSet.contains(tag->GetName()),
                    ConfigHttpStatus.PermissionDeniedStatus,
                    "Have not permissions for add tag " + tag->GetName(),
                    EDriveSessionResult::NoUserPermissions
                );

                auto feedbackTag = std::dynamic_pointer_cast<TFeedbackTraceTag>(tag);
                if (feedbackTag) {
                    R_ENSURE(
                        feedbackTag->Set(deviceId, userId, Context->GetRequestStartTime(), Server, session),
                        ConfigHttpStatus.UnknownErrorStatus,
                        "cannot set feedback tag: " << session.GetStringReport()
                    );
                    feedbackTags.push_back(feedbackTag);

                    auto response = feedbackTag->GetResponse(Server);
                    totalResponse = TFeedbackTraceTag::MergeResponses(totalResponse.Get(), response);
                }

                newTags.push_back(tag);
            }
            for (auto&& feedbackTag : feedbackTags) {
                if (!feedbackTag) {
                    continue;
                }
                const TString& bonusTag = feedbackTag->GetBonusTag();
                if (bonusTag && bonusTag != totalResponse->BonusTag) {
                    feedbackTag->CancelBonus();
                }
            }
            if (!tagsManager.AddTags(newTags, userId, objectId, Server, session, EUniquePolicy::NoUnique)) {
                session.DoExceptionOnFail(ConfigHttpStatus);
            }
        }
        if (!session.Commit()) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
    }
    if (totalResponse && totalResponse->Show) {
        auto notification = TFeedbackTraceTag::FormatResponse(*totalResponse, locale, Server);
        g.AddReportElement("notification", std::move(notification));
    } else {
        g.AddReportElement("skipped_notification", totalResponse.Defined());
    }
    g.SetCode(HTTP_OK);
}

void TListTagsProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*jsonValue*/) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::Tag);
    NJson::TJsonValue results = NJson::JSON_ARRAY;
    NJson::TJsonValue deprecated = NJson::JSON_ARRAY;
    const TCgiParameters& cgi = Context->GetCgiParameters();
    auto tagsDescriptions = DriveApi->GetTagsManager().GetTagsMeta().GetRegisteredTags(Now());

    TSet<NEntityTagsManager::EEntityType> entitiesSet;
    if (cgi.Has("entities")) {
        StringSplitter(cgi.Get("entities")).SplitBySet(", ").SkipEmpty().ParseInto(&entitiesSet);
    }

    TSet<TString> descriptionNames;
    if (cgi.Has("descriptions")) {
        StringSplitter(cgi.Get("descriptions")).SplitBySet(", ").ParseInto(&descriptionNames);
    }

    auto availableTagNames = permissions->GetTagNamesByAction(TTagAction::ETagAction::Observe);
    for (auto&& tDescr : tagsDescriptions) {
        if (!descriptionNames.empty() && !descriptionNames.contains(tDescr.second->GetName())) {
            continue;
        }

        if (availableTagNames.contains(tDescr.second->GetName())) {
            NJson::TJsonValue tagJson = tDescr.second->SerializeToJson();
            NDrive::ITag::TPtr tag = NDrive::ITag::TFactory::Construct(tDescr.second->GetType());
            const TSet<NEntityTagsManager::EEntityType> tagObjects = !!tag ? tag->GetObjectType() : TSet<NEntityTagsManager::EEntityType>({NEntityTagsManager::EEntityType::Undefined});
            TJsonProcessor::WriteContainerArrayStrings(tagJson, "entities", tagObjects);
            if (entitiesSet.size() && IsIntersectionEmpty(entitiesSet, tagObjects)) {
                continue;
            }
            if (!!tag) {
                tagJson.InsertValue("unique_policy", ToString(tag->GetUniquePolicy()));
            }
            if (tDescr.second->GetDeprecated()) {
                deprecated.AppendValue(tagJson);
            } else {
                results.AppendValue(tagJson);
            }
        }
    }

    NJson::TJsonValue propositionsJson = NJson::JSON_ARRAY;
    TSet<TString> users;
    {
        const auto& propositionManager = DriveApi->GetTagsManager().GetTagsMeta().GetPropositionManager();
        auto tx = BuildTx<NSQL::ReadOnly>();

        auto optionalTagDescPropose = propositionManager.Get(tx);
        R_ENSURE(optionalTagDescPropose, {}, "cannot get propositions", tx);
        auto& tagDescPropose = *optionalTagDescPropose;

        for (auto&& [idPropose, proposeReport] : tagDescPropose) {
            auto objectReport = proposeReport.BuildJsonReport();
            NDrive::ITag::TPtr tag = NDrive::ITag::TFactory::Construct(proposeReport->GetType());
            const TSet<NEntityTagsManager::EEntityType> tagObjects = !!tag ? tag->GetObjectType() : TSet<NEntityTagsManager::EEntityType>({NEntityTagsManager::EEntityType::Undefined});
            TJsonProcessor::WriteContainerArrayStrings(objectReport, "entities", tagObjects);
            if (entitiesSet.size() && IsIntersectionEmpty(entitiesSet, tagObjects)) {
                continue;
            }
            if (!!tag) {
                objectReport.InsertValue("unique_policy", ToString(tag->GetUniquePolicy()));
            }
            propositionsJson.AppendValue(objectReport);
            users.emplace(proposeReport.GetPropositionAuthor());
        }
    }
    NJson::TJsonValue usersJson = NJson::JSON_MAP;
    {
        auto userData = DriveApi->GetUsersData()->FetchInfo(users);
        for (auto&& [_, i] : userData) {
            usersJson.InsertValue(i.GetUserId(), i.GetReport(permissions->GetUserReportTraits()));
        }
    }

    TSet<TString> types;
    ITag::TFactory::GetRegisteredKeys(types);

    g.MutableReport().AddReportElement("records", std::move(results));
    g.MutableReport().AddReportElement("deprecated", std::move(deprecated));
    g.MutableReport().AddReportElement("types", NJson::ToJson(types));
    g.MutableReport().AddReportElement("propositions", std::move(propositionsJson));
    g.MutableReport().AddReportElement("users", std::move(usersJson));

    g.SetCode(HTTP_OK);
}

void TRegisterTagProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& jsonValue) {
    const auto& cgi = Context->GetCgiParameters();
    const auto force = GetValue<bool>(cgi, "force", false).GetOrElse(false);
    const TString& userId = permissions->GetUserId();

    TMessagesCollector errors;

    NStorage::TTableRecord tRecord;
    R_ENSURE(tRecord.DeserializeFromJson(jsonValue), ConfigHttpStatus.SyntaxErrorStatus, "table record cannot be parsed from json", EDriveSessionResult::IncorrectRequest);

    TTagDescription::TPtr tDescription = TTagDescription::ConstructFromTableRecord(tRecord);
    R_ENSURE(tDescription, ConfigHttpStatus.SyntaxErrorStatus, "tag description cannot be constructed", EDriveSessionResult::IncorrectRequest);
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Add, TAdministrativeAction::EEntity::Tag, tDescription->GetType());
    if (force) {
        ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::ModifyStructure, TAdministrativeAction::EEntity::Tag, tDescription->GetType());
    }

    R_ENSURE(tDescription->GetName().find(" ") == TString::npos, ConfigHttpStatus.SyntaxErrorStatus, "tag name contains whitespace", EDriveSessionResult::IncorrectRequest);
    R_ENSURE(tDescription->GetName(), ConfigHttpStatus.SyntaxErrorStatus, "empty tag name", EDriveSessionResult::IncorrectRequest);

    auto tx = BuildTx<NSQL::Writable>();
    R_ENSURE(
        DriveApi->GetTagsManager().GetTagsMeta().RegisterTag(tDescription, userId, tx, force),
        {},
        "cannot RegisterTag",
        tx
    );
    R_ENSURE(tx.Commit(), {}, "cannot Commit", tx);
    g.SetCode(HTTP_OK);
}

NJson::TJsonValue TEntityHistoryProcessor::GetUsersReport(const TSet<TString>& users, TUserPermissions::TPtr permissions, NDrive::TEntitySession& session) const {
    auto usersData = DriveApi->GetUsersData()->FetchInfo(users, session);
    NJson::TJsonValue report(NJson::JSON_MAP);
    for (auto&& i : usersData) {
        report.InsertValue(i.first, i.second.GetReport(permissions->GetUserReportTraits()));
    }
    return report;
}

NJson::TJsonValue TBaseCarTagsHistoryProcessor::GetObjectsReport(const TSet<TString>& objectIds, TUserPermissions::TPtr permissions, NDrive::TEntitySession& session) const {
    auto objectFetchResult = DriveApi->GetCarsData()->FetchInfo(objectIds, session);
    NJson::TJsonValue report(NJson::JSON_MAP);
    for (auto&& i : objectFetchResult) {
        report.InsertValue(i.first, i.second.GetReport(GetLocale(), permissions->GetDeviceReportTraits()));
    }
    return report;
}

NJson::TJsonValue TAccountTagsHistoryProcessor::GetObjectsReport(const TSet<TString>& objectIds, TUserPermissions::TPtr /*permissions*/, NDrive::TEntitySession& /*session*/) const {
    TSet<ui32> accountIds;
    for (const auto& objectId : objectIds) {
        ui32 id = 0;
        if (TryFromString(objectId, id)) {
            accountIds.insert(id);
        }
    }
    auto objectFetchResult = DriveApi->GetBillingManager().GetAccountsManager().GetAccountsByIds(accountIds);
    NJson::TJsonValue report(NJson::JSON_MAP);
    if (!objectFetchResult) {
        return report;
    }
    for (auto&& i : *objectFetchResult) {
        if (!i) {
            continue;
        }
        report.InsertValue(::ToString(i->GetId()), i->GetReport());
    }
    return report;
}

void TEntityHistoryProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*jsonValue*/) {
    const TCgiParameters& cgi = Context->GetCgiParameters();
    const TString& objectId = cgi.Has("object_id") ? cgi.Get("object_id") : cgi.Get("car_id");
    const TString& tagId = cgi.Has("tag_id") ? cgi.Get("tag_id") : Default<TString>();
    const auto actionTypes = MakeSet(GetValues<EObjectHistoryAction>(cgi, "action_types", false));
    TSet<TString> availableTags;
    StringSplitter(cgi.Get("tags")).SplitBySet(",").SkipEmpty().Collect(&availableTags);
    auto pageSize = GetValue<ui32>(cgi, "page_size", false);
    auto pageNumber = GetValue<ui32>(cgi, "page_number", false).GetOrElse(1);
    if (!!tagId) {
        Y_ENSURE_EX(!GetUuid(tagId).IsEmpty(), TCodedException(ConfigHttpStatus.SyntaxErrorStatus) << "incorrect 'tag_id' in parameters");
    } else if(!!objectId) {
        if (IsObjectIdUUID()) {
            Y_ENSURE_EX(!GetUuid(objectId).IsEmpty(), TCodedException(ConfigHttpStatus.SyntaxErrorStatus) << "incorrect 'object_id' in parameters (no uuid)");
        }
    } else {
        ReqCheckCondition(false, ConfigHttpStatus.SyntaxErrorStatus, "incorrect_object_or_tag");
    }

    const auto how = GetString(cgi, "how", false);
    TInstant since = ModelingNow() - TDuration::Days(3);
    TInstant until = ModelingNow();
    TDuration requestPeriod;
    if (cgi.Has("duration")) {
        R_ENSURE(TDuration::TryParse(cgi.Get("duration"), requestPeriod), ConfigHttpStatus.SyntaxErrorStatus, "incorrect duration");
        since = Now() - requestPeriod;
    } else {
        if (cgi.Has("since")) {
            ui32 sinceTs;
            R_ENSURE(TryFromString(cgi.Get("since"), sinceTs), ConfigHttpStatus.SyntaxErrorStatus, "incorrect since");
            since = TInstant::Seconds(sinceTs);
        }
        if (cgi.Has("until")) {
            ui32 untilTs;
            R_ENSURE(TryFromString(cgi.Get("until"), untilTs), ConfigHttpStatus.SyntaxErrorStatus, "incorrect until");
            until = TInstant::Seconds(untilTs);
        }
    }

    const auto& tagManager = GetEntityTagsManager();
    bool descending = !how || how == "asc";
    auto session = BuildTx<NSQL::ReadOnly>();
    auto events = TOptionalTagHistoryEvents();
    if (!!tagId) {
        events = tagManager.GetEventsByTag(tagId, session, 0, since);
    } else if (!!objectId) {
        events = tagManager.GetEventsByObject(objectId, session, 0, since);
    } else {
        events = tagManager.GetEvents({since, until}, session, TTagEventsManager::TQueryOptions());
    }
    R_ENSURE(events, {}, "cannot GetEvents", session);
    if (descending) {
       std::reverse(events->begin(), events->end());
    }
    auto& objectHistory = *events;

    const TSet<TString> observableTags = permissions->GetTagNamesByAction(TTagAction::ETagAction::Observe);
    std::function<bool(const TTagHistoryEvent&)> customPredicate = GetCustomPredicate();

    const auto removePred = [&availableTags, &observableTags, &actionTypes, until, customPredicate](const TTagHistoryEvent& item) {
        return !observableTags.contains(item->GetName())
            || (item.GetHistoryInstant() >= until)
            || (!availableTags.empty() && !availableTags.contains(item->GetName()))
            || (!actionTypes.empty() && !actionTypes.contains(item.GetHistoryAction()))
            || !customPredicate(item);
    };
    objectHistory.erase(std::remove_if(objectHistory.begin(), objectHistory.end(), removePred), objectHistory.end());
    auto begin = objectHistory.begin();
    auto end = objectHistory.end();
    if (pageSize) {
        auto beginIdx = Min<ui32>(*pageSize * (pageNumber - 1), objectHistory.size());
        auto endIdx = Min<ui32>(*pageSize * pageNumber, objectHistory.size());
        begin = objectHistory.begin() + beginIdx;
        end = objectHistory.begin() + endIdx;
        g.MutableReport().AddReportElement("can_get_more_pages", end != objectHistory.end());
        g.MutableReport().AddReportElement("page_number", pageNumber);
    }

    auto tagDescriptions = DriveApi->GetTagsManager().GetTagsMeta().GetRegisteredTags();


    TSet<TString> users;
    TSet<TString> objects;
    for (auto iter = begin; iter != end; ++iter) {
        const auto& objectTag = *iter;
        const TString& userId = objectTag.GetHistoryUserId();
        if (!!userId && !GetUuid(userId).IsEmpty()) {
            users.emplace(userId);
        }
        const TString& originatorUserId = objectTag.GetHistoryOriginatorId();
        if (!!originatorUserId && !GetUuid(originatorUserId).IsEmpty()) {
            users.emplace(originatorUserId);
        }
        objects.emplace(objectTag.TConstDBTag::GetObjectId());
    }

    NJson::TJsonValue report;
    auto usersData = DriveApi->GetUsersData()->FetchInfo(users, session);
    for (auto iter = begin; iter != end; ++iter) {
        const auto& objectTag = *iter;

        const TString& userId = objectTag.GetHistoryUserId();
        NJson::TJsonValue historyItemInfo = objectTag.BuildReportItem();
        {
            const TDriveUserData* userData = usersData.GetResultPtr(userId);
            if (!!userData) {
                historyItemInfo.InsertValue("user_data_full", userData->GetReport());
            }
        }
        if (!!objectTag.GetHistoryOriginatorId()) {
            const TDriveUserData* userData = usersData.GetResultPtr(objectTag.GetHistoryOriginatorId());
            if (!!userData) {
                historyItemInfo.InsertValue("originator_data", userData->GetReport());
            }
        }
        auto itTagDescription = tagDescriptions.find(objectTag->GetName());
        if (itTagDescription != tagDescriptions.end()) {
            historyItemInfo.InsertValue("tag_display_name", itTagDescription->second->GetDisplayName());
        }
        report.AppendValue(historyItemInfo);
    }
    g.MutableReport().AddReportElement("records", std::move(report));
    g.MutableReport().AddReportElement("objects", GetObjectsReport(objects, permissions, session));
    g.MutableReport().AddReportElement("users", GetUsersReport(users, permissions, session));
    g.SetCode(HTTP_OK);
}

void TEntityTagHistoryDetailsProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*jsonValue*/) {
    const TCgiParameters& cgi = Context->GetCgiParameters();
    const auto eventId = GetValue<TCarTagHistoryEvent::TEventId>(cgi, "event_id").GetRef();

    auto session = BuildTx<NSQL::ReadOnly>();
    auto optionalEvent = GetEntityTagsManager().GetEvent(eventId, session);
    R_ENSURE(optionalEvent, {}, "cannot GetEvent " << eventId, session);

    const auto eventPtr = *optionalEvent;
    R_ENSURE(
        eventPtr && *eventPtr,
        ConfigHttpStatus.EmptySetStatus,
        "event with id " << eventId << " is not found",
        EDriveSessionResult::IncorrectRequest,
        session
    );
    const auto& ev = *eventPtr;
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::ObserveStructure, TAdministrativeAction::EEntity::Tag, ev->GetName());

    NJson::TJsonValue report = ev.BuildReportItem();
    report["tag"] = ev->SerializeToJson();
    NDrive::IObjectSnapshot::TPtr snapshot = ev->GetObjectSnapshot();
    if (snapshot) {
        report["snapshot"] = snapshot->SerializeToJson();
    } else {
        report["snapshot"] = NJson::JSON_NULL;
    }
    g.MutableReport().SetExternalReport(std::move(report));
    g.SetCode(HTTP_OK);
}

void TUserHistoryProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr /*permissions*/, const NJson::TJsonValue& /*jsonValue*/) {
    const TCgiParameters& cgi = Context->GetCgiParameters();
    const auto& userId = GetUUID(cgi, "user_id");
    const auto duration = GetDuration(cgi, "duration", TDuration::Days(3));
    const auto until = GetTimestamp(cgi, "until", ModelingNow());
    const auto since = GetTimestamp(cgi, "since", until - duration);

    const auto& tagManager = DriveApi->GetTagsManager().GetDeviceTags();
    const auto& userTagManager = DriveApi->GetTagsManager().GetUserTags();
    auto session = BuildTx<NSQL::ReadOnly>();
    auto optionalEvents = tagManager.GetEvents({ since, until }, session, TTagEventsManager::TQueryOptions()
        .SetUserIds({ userId })
    );
    R_ENSURE(optionalEvents, {}, "cannot GetEvents", session);

    auto optionalUserTagEvents = userTagManager.GetEvents({ since, until }, session, TTagEventsManager::TQueryOptions()
        .SetUserIds({ userId })
    );
    R_ENSURE(optionalUserTagEvents, {}, "cannot GetEvents", session);

    auto& carTagHistory = *optionalEvents;
    auto& userTagHistory = *optionalUserTagEvents;

    auto howSort = GetString(cgi, "how", false);
    if (!howSort || howSort == "asc") {
        std::reverse(carTagHistory.begin(), carTagHistory.end());
        std::reverse(userTagHistory.begin(), userTagHistory.end());
    }

    TSet<TString> carIds;
    for (auto&& i : carTagHistory) {
        carIds.emplace(i.TConstDBTag::GetObjectId());
    }

    TSet<TString> userIds;
    for (auto&& tag : userTagHistory) {
        userIds.insert(tag.GetObjectId());
    }
    auto usersData = DriveApi->GetUsersData()->FetchInfo(userIds, session);

    NJson::TJsonValue report = NJson::JSON_ARRAY;
    auto carsData = DriveApi->GetCarsData()->FetchInfo(carIds, session);
    for (auto carTag : carTagHistory) {
        NJson::TJsonValue reportHistoryItem = carTag.BuildReportItem();
        auto* carInfo = carsData.GetResultPtr(carTag.TConstDBTag::GetObjectId());
        if (!!carInfo) {
            reportHistoryItem.InsertValue("car", carInfo->GetSearchReport());
        }
        report.AppendValue(reportHistoryItem);
    }
    g.MutableReport().AddReportElement("records", std::move(report));

    NJson::TJsonValue userTagReport = NJson::JSON_ARRAY;
    for (auto&& tag : userTagHistory) {
        auto report = tag.BuildReportItem();
        auto userData = usersData.GetResultPtr(tag.GetObjectId());
        if (userData) {
            report.InsertValue("user", userData->GetSearchReport());
        }
        userTagReport.AppendValue(std::move(report));
    }
    g.AddReportElement("user_tags", std::move(userTagReport));

    g.SetCode(HTTP_OK);
}

void ICarRepairTagBaseProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& jsonValue) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::MajorApi);
    R_ENSURE(DriveApi->HasMajorClient(), ConfigHttpStatus.UnknownErrorStatus, "No MajorClient configured available");

    const TCgiParameters& cgi = Context->GetCgiParameters();
    const TString queryId = GetString(cgi, "query_id", false);
    if (!queryId.empty()) {
        ProcessRequest(nullptr, queryId, g, permissions, jsonValue);
        return;
    }
    const TString tagId = GetUUID(cgi, "id", true);

    const TDeviceTagsManager& manager = DriveApi->GetTagsManager().GetDeviceTags();
    IEntityTagsManager::TOptionalTag optionalTag;
    {
        auto session = BuildTx<NSQL::ReadOnly>();
        optionalTag = manager.RestoreTag(tagId, session);
        R_ENSURE(optionalTag, {}, "cannot RestoreTag " << tagId, session);
    }
    const TDBTag& tag = *optionalTag;
    R_ENSURE(tag, ConfigHttpStatus.EmptySetStatus, "cannot find tag_id " << tagId);

    const auto repairTag = tag.GetTagAs<TRepairTagRecord>();
    R_ENSURE(repairTag, ConfigHttpStatus.EmptySetStatus, "There is not repair tag " << tagId);
    R_ENSURE(repairTag->GetQueryId(), ConfigHttpStatus.EmptySetStatus, "QueryId doesn't set for tag " << tagId);

    ProcessRequest(&tag, repairTag->GetQueryId(), g, permissions, jsonValue);
}

void TCarRepairTagDetailsProcessor::ProcessRequest(const TDBTag* tag, const TString& queryId, TJsonReport::TGuard& g, TUserPermissions::TPtr /*permissions*/, const NJson::TJsonValue& /*jsonValue*/) const {
    TMessagesCollector errors;
    NMajorClient::TQueryInfoRequest::TFullQueryInfo info;
    R_ENSURE(DriveApi->GetMajorClient().GetQueryInfo(queryId, info, errors) && !errors.HasMessages(), ConfigHttpStatus.UnknownErrorStatus, "Internal major api error: " << errors.GetStringReport());
    if (!!tag) {
        g.MutableReport().AddReportElement(tag->GetTagId(), info.GetJsonReport());
    } else {
        g.MutableReport().AddReportElement(queryId, info.GetJsonReport());
    }
    g.SetCode(HTTP_OK);
}

void TCarRepairTagCancelProcessor::ProcessRequest(const TDBTag* tag, const TString& queryId, TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*jsonValue*/) const {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Reject, TAdministrativeAction::EEntity::MajorApi);
    TMessagesCollector errors;
    R_ENSURE(DriveApi->GetMajorClient().CancelQuery(queryId, errors) && !errors.HasMessages(), ConfigHttpStatus.UnknownErrorStatus, "Internal major api error: " << errors.GetStringReport());

    const TDeviceTagsManager& manager = DriveApi->GetTagsManager().GetDeviceTags();
    auto session = BuildTx<NSQL::Writable>();
    if (!!tag && !manager.RemoveTag(*tag, permissions->GetUserId(), Server, session, true) || !session.Commit()) {
        session.AddErrorMessage("cancel_major_query", "Major query was canceled but tag isn't remove");
        session.DoExceptionOnFail(ConfigHttpStatus);
    }

    g.SetCode(HTTP_OK);
}

void TCurrentMajorQueriesProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*jsonValue*/) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::MajorApi);
    R_ENSURE(DriveApi->HasMajorClient(), ConfigHttpStatus.UnknownErrorStatus, "No MajorClient configured available");

    TMessagesCollector errors;
    TVector<NMajorClient::TGetQueriesRequest::TQuery> queries;
    if (IsTrue(Context->GetCgiParameters().Get("all"))) {
        R_ENSURE(DriveApi->GetMajorClient().GetAllQueryList(queries, errors) && !errors.HasMessages(), ConfigHttpStatus.UnknownErrorStatus, "Internal major api error: " << errors.GetStringReport());
    } else {
        R_ENSURE(DriveApi->GetMajorClient().GetQueryList(false, queries, errors) && !errors.HasMessages(), ConfigHttpStatus.UnknownErrorStatus, "Internal major api error: " << errors.GetStringReport());
    }

    R_ENSURE(DriveApi->GetCarVins(), ConfigHttpStatus.UnknownErrorStatus, "cannot take car ids by vins");
    TSet<TString> vins;
    for (auto&& query : queries) {
        vins.insert(query.GetVIN());
    }
    auto session = BuildTx<NSQL::ReadOnly>();
    auto byVinResult = DriveApi->GetCarVins()->FetchInfo(vins, session);

    NJson::TJsonValue json;
    for (auto&& query : queries) {
        NJson::TJsonValue report = query.GetJsonReport();
        auto carByVin = byVinResult.GetResultPtr(query.GetVIN());
        if (!!carByVin) {
            report.InsertValue("car_id", carByVin->GetId());
            report.InsertValue("car_number", carByVin->GetNumber());
        }
        json.AppendValue(report);
    }

    g.MutableReport().AddReportElement("queries", std::move(json));
    g.SetCode(HTTP_OK);
}

void TListTagsByPerformerProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*jsonValue*/) {
    const auto& cgi = Context->GetCgiParameters();
    auto performerId = GetUUID(cgi, "performer", false);
    if (!performerId) {
        performerId = permissions->GetUserId();
    } else {
        ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::User);
    }

    const auto& tagsManager = DriveApi->GetTagsManager();

    auto locale = GetLocale();
    NDrive::TEntitySession session = BuildTx<NSQL::ReadOnly>();
    TVector<TDBTag> tags;
    R_ENSURE(
        tagsManager.GetDeviceTags().RestorePerformerTags({performerId}, tags, session),
        {},
        "cannot RestorePerformerDeviceTags for " << performerId,
        session
    );

    TVector<TDBTag> userTags;
    R_ENSURE(
        tagsManager.GetUserTags().RestorePerformerTags({performerId}, userTags, session),
        {},
        "cannot RestorePerformerDeviceTags for " << performerId,
        session
    );

    TSet<TString> objectIds;
    for (auto&& i : tags) {
        objectIds.emplace(i.GetObjectId());
    }
    TSet<TString> userIds;
    for (auto&& tag : userTags) {
        userIds.insert(tag.GetObjectId());
    }

    auto carsInfo = DriveApi->GetCarsData()->FetchInfo(objectIds, session);
    auto usersInfo = DriveApi->GetUsersData()->FetchInfo(userIds, session);

    NJson::TJsonValue results = NJson::JSON_ARRAY;
    for (auto&& i : tags) {
        auto description = DriveApi->GetTagsManager().GetTagsMeta().GetDescriptionByName(i->GetName());
        NJson::TJsonValue& tagJson = results.AppendValue(i.SerializeToJson());
        if (description) {
            tagJson["display_name"] = description->GetDisplayName();
        } else {
            tagJson["display_name"] = i->GetName();
        }
        auto* carInfo = carsInfo.GetResultPtr(i.GetObjectId());
        if (carInfo) {
            tagJson.InsertValue("object_info", carInfo->GetReport(locale, permissions->GetDeviceReportTraits()));
        }
    }
    g.MutableReport().AddReportElement("records", std::move(results));

    NJson::TJsonValue userTagReport = NJson::JSON_ARRAY;
    for (auto&& tag : userTags) {
        auto description = tagsManager.GetTagsMeta().GetDescriptionByName(tag->GetName());
        auto& report = userTagReport.AppendValue(tag.SerializeToJson());
        report.InsertValue("display_name", description ? description->GetDisplayName() : tag->GetName());
        auto userInfo = usersInfo.GetResultPtr(tag.GetObjectId());
        if (userInfo) {
            report.InsertValue("object_info", userInfo->GetReport(permissions->GetUserReportTraits()));
        }
    }
    g.AddReportElement("user_tags", std::move(userTagReport));

    g.SetCode(HTTP_OK);
}

void TProposeTagProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    ReqCheckCondition(GetEntityTagsManager().GetPropositionsPtr(), ConfigHttpStatus.ServiceUnavailable, "no propositions in server configuration");
    const TString& objectId = Context->GetCgiParameters().Get("object_id");
    ReqCheckCondition(!!objectId, ConfigHttpStatus.SyntaxErrorStatus, "no_object_id");
    ReqCheckCondition(!GetUuid(objectId).IsEmpty(), ConfigHttpStatus.SyntaxErrorStatus, "incorrect_object_id");

    TMessagesCollector errors;
    ITag::TPtr tag = IJsonSerializableTag::BuildFromJson(DriveApi->GetTagsManager(), requestData, &errors);
    ReqCheckCondition(!!tag, ConfigHttpStatus.SyntaxErrorStatus, "Cannot build tag: " + errors.GetReport().GetStringRobust());
    Y_ENSURE_EX(permissions->GetTagNamesByAction(TTagAction::ETagAction::Propose).contains(tag->GetName()),
        TCodedException(ConfigHttpStatus.PermissionDeniedStatus) << "No permissions for propose this tag type: " << tag->GetName());

    NDrive::TEntitySession session = BuildTx<NSQL::Writable>();
    session.SetComment(Context->GetCgiParameters().Get("comment"));
    TDBTag dbTag;
    dbTag.SetData(tag);
    dbTag.SetObjectId(objectId);

    if (!GetEntityTagsManager().ProposeTag(dbTag, permissions->GetUserId(), session) || !session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }

    g.SetCode(HTTP_OK);
}

void TConfirmTagPropositionProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    ReqCheckCondition(GetEntityTagsManager().GetPropositionsPtr(), ConfigHttpStatus.ServiceUnavailable, "no propositions in server configuration");
    const TVector<TString> propositionIds = GetStrings(requestData, "proposition_ids");

    auto session = BuildTx<NSQL::Writable>();

    auto optionalSessions = GetEntityTagsManager().GetPropositionsPtr()->Get(propositionIds, session);
    R_ENSURE(optionalSessions, {}, "cannot get propositions", session);
    auto sessions = *optionalSessions;

    for (auto&& i : propositionIds) {
        auto it = sessions.find(i);
        R_ENSURE(it != sessions.end(), ConfigHttpStatus.UserErrorState, "incorrect propose_id " + i, EDriveSessionResult::IncorrectRequest);
        R_ENSURE(
            permissions->GetTagNamesByAction(TTagAction::ETagAction::Confirm).contains(it->second->GetName()),
            ConfigHttpStatus.PermissionDeniedStatus,
            "no_permissions for confirm object",
            EDriveSessionResult::NoUserPermissions
        );
    }

    session.SetComment(Context->GetCgiParameters().Get("comment"));

    if (!GetEntityTagsManager().ConfirmPropositions(MakeSet(propositionIds), permissions->GetUserId(), Server, session) || !session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }
    g.SetCode(HTTP_OK);
}

void TRejectTagPropositionProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    ReqCheckCondition(GetEntityTagsManager().GetPropositionsPtr(), ConfigHttpStatus.ServiceUnavailable, "no propositions in server configuration");
    const TVector<TString> propositionIds = GetStrings(requestData, "proposition_ids");
    auto session = BuildTx<NSQL::Writable>();

    auto optionalSessions = GetEntityTagsManager().GetPropositionsPtr()->Get(propositionIds, session);
    R_ENSURE(optionalSessions, {}, "cannot get propositions", session);
    auto sessions = *optionalSessions;
    for (auto&& i : propositionIds) {
        auto it = sessions.find(i);
        R_ENSURE(it != sessions.end(), ConfigHttpStatus.UserErrorState, "incorrect propose_id " + i, EDriveSessionResult::IncorrectRequest);
        R_ENSURE(
            permissions->GetTagNamesByAction(TTagAction::ETagAction::Reject).contains(it->second->GetName()),
            ConfigHttpStatus.PermissionDeniedStatus,
            "no_permissions for confirm object",
            EDriveSessionResult::NoUserPermissions
        );
    }

    session.SetComment(Context->GetCgiParameters().Get("comment"));

    if (!GetEntityTagsManager().RejectPropositions(MakeSet(propositionIds), permissions->GetUserId(), session) || !session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }

    g.SetCode(HTTP_OK);
}

TSet<TString> TAddServiceWorkerTagProcessor::GetEntityObjectIds(const NJson::TJsonValue& jsonValue, NDrive::TEntitySession& session, TUserPermissions::TPtr permissions) const {
    auto subordinatesTagsNames = permissions->GetTagNamesByAction(TTagAction::ETagAction::UpdateObject);
    TVector<TDBTag> dbTags;
    R_ENSURE(
        Server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({}, MakeVector(subordinatesTagsNames), dbTags, session),
        HTTP_INTERNAL_SERVER_ERROR,
        "cannot RestoreTags",
        session
    );
    auto workerId = GetUUID(jsonValue, "worker_id");
    auto it = std::find_if(dbTags.begin(), dbTags.end(), [&workerId](const TDBTag& dbTag) {
        return dbTag.GetObjectId() == workerId;
    });
    R_ENSURE(it != dbTags.end(), HTTP_FORBIDDEN, "not allowed to update tags of " << workerId);
    return {workerId};
}

void TServiceCarTagDetailsProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /* requestData */) {
    const TCgiParameters& cgi = Context->GetCgiParameters();
    const TString& tagId = GetUUID(cgi, "tag_id");
    const auto locale = GetLocale();
    const auto photosFetchPeriod = GetHandlerSettingDef<TDuration>("photos_fetch_period", TDuration::Days(15));
    const auto reportDescription = GetValue<bool>(cgi, "report_description", false).GetOrElse(false);
    const auto& api = *Yensured(DriveApi);
    const auto& usersData = *Yensured(api.GetUsersData());
    const auto& manager = api.GetTagsManager().GetDeviceTags();
    auto session = BuildTx<NSQL::ReadOnly>();
    auto optionalTag = manager.RestoreTag(tagId, session);
    R_ENSURE(optionalTag, {}, "cannot RestoreTag " << tagId, session);
    const auto& tag = *optionalTag;
    R_ENSURE(tag, ConfigHttpStatus.EmptySetStatus, "cannot find tag_id " << tagId);

    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::ObserveStructure, TAdministrativeAction::EEntity::Tag, tag->GetName());

    TInstant now = Now();
    auto since = now - photosFetchPeriod;
    auto until = now;

    auto optionalEvents = manager.GetEventsByTag(tag.GetTagId(), session, 0);
    R_ENSURE(optionalEvents, ConfigHttpStatus.UnknownErrorStatus, "can not get tag events");

    TMaybe<TInstant> creationTime;
    TMaybe<TString> authorId;
    {
        NJson::TJsonValue history(NJson::JSON_ARRAY);
        for (auto&& ev : *optionalEvents) {
            if (ev) {
                auto jsonReport = ev.BuildReportItem();
                if (auto userDataPtr = usersData.GetCachedObject(ev.GetHistoryUserId()); userDataPtr) {
                    jsonReport["user_full_name"] = userDataPtr->GetFullNameOrLogin();
                }
                history.AppendValue(std::move(jsonReport));
                if (ev.GetHistoryAction() == EObjectHistoryAction::Add && !authorId) {
                    authorId = ev.GetHistoryUserId();
                    creationTime = ev.GetHistoryTimestamp();
                }
            } else {
                history.AppendValue(NJson::JSON_NULL);
            }
        }
        g.MutableReport().AddReportElement("history", std::move(history));
    }

    auto tagDescription = Server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetDescriptionByName(tag->GetName());
    R_ENSURE(tagDescription, ConfigHttpStatus.UnknownErrorStatus, "can not get tag description");
    if (reportDescription) {
        g.AddReportElement("description", tagDescription->BuildJsonReport(locale));
    }

    NJson::TJsonValue tagReport = tag->SerializeToJson();
    if (tag->GetPerformer()) {
        auto userDataPtr = usersData.GetCachedObject(tag->GetPerformer());
        if (userDataPtr) {
            tagReport["performer_full_name"] = userDataPtr->GetFullNameOrLogin();
            tagReport["performer_phone_number"] = userDataPtr->GetPhone();
        }
    }

    tagReport["display_name"] = tagDescription->GetDisplayName();
    tagReport["tag_id"] = tag.GetTagId();
    tagReport["object_id"] = tag.GetObjectId();
    if (authorId) {
        tagReport["author_id"] = *authorId;
        tagReport["creation_time"] = creationTime->Seconds();
        auto userDataPtr = usersData.GetCachedObject(*authorId);
        if (userDataPtr) {
            tagReport["author_full_name"] = userDataPtr->GetFullNameOrLogin();
            tagReport["author_phone_number"] = userDataPtr->GetPhone();
        }
    }
    g.MutableReport().AddReportElement("tag", std::move(tagReport));

    auto snapshot = tag->GetObjectSnapshot();
    g.MutableReport().AddReportElement("snapshot", snapshot ? snapshot->SerializeToJson() : NJson::JSON_NULL);


    auto hostByImageSourceMapping = DriveApi->HasMDSClient()
        ? TCommonImageData::GetHostByImageSourceMapping(DriveApi->GetMDSClient())
        : TMap<TString, TString>();

    TMap<ui64, TCommonImageData> imagesInfoMapping;
    {
        auto optionalImages = DriveApi->GetImagesDB().Get(tag.GetObjectId(), {since, until}, session);
        if (!optionalImages) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }

        for (auto&& image : *optionalImages) {
            imagesInfoMapping.emplace(image.GetImageId(), std::move(image));
        }
    }

    auto permittedTags = permissions->GetTagNamesByAction(TTagAction::ETagAction::Observe);
    if (permittedTags.contains(tag->GetName())) {
        NJson::TJsonValue imagesReport;
        for (auto&& ev : *optionalEvents) {
            if (!ev) {
                continue;
            }
            if (ev.GetHistoryTimestamp() < since) {
                continue;
            }
            if (ev.GetHistoryAction() != EObjectHistoryAction::AddSnapshot) {
                continue;
            }

            auto snapshot = ev->GetObjectSnapshotAs<TImagesSnapshot>();
            if (!snapshot) {
                continue;
            }

            NJson::TJsonValue rawImages;
            for (auto&& image: snapshot->GetImages()) {
                auto imageInfoPtr = imagesInfoMapping.FindPtr(*image.ImageId);
                auto imageReport = NJson::ToJson(image);
                if (imageInfoPtr) {
                    NJson::MergeJson(imageInfoPtr->BuildReport(hostByImageSourceMapping), imageReport);
                }
                rawImages.AppendValue(std::move(imageReport));
            }
            if (rawImages.GetArray().empty()) {
                continue;
            }

            NJson::TJsonValue imageSnapshotReport;
            imageSnapshotReport["timestamp"] = ev.GetHistoryInstant().Seconds();
            imageSnapshotReport["images"] = std::move(rawImages);
            imagesReport.AppendValue(std::move(imageSnapshotReport));
        }
        g.MutableReport().AddReportElement("images", std::move(imagesReport));
    }
    g.SetCode(HTTP_OK);
}

std::function<bool(const TTagHistoryEvent&)> TServiceAppCarTagsHistoryProcessor::GetCustomPredicate() const {
    return [](const TTagHistoryEvent& event) -> bool {
        return !SessionTags.contains(event->GetName());
    };
}

TMaybe<TTaggedObject> TListTraceTagsProcessor::GetTaggedObject(const TString& objectId, NDrive::TEntitySession& session, NDrive::TEntitySession& sessionForYdb) const {
    TTaggedObjectsSnapshot objectsSnapshot;
    if (!NDrive::GetFullEntityTags(GetEntityTagsManager(), { objectId }, objectsSnapshot, session, sessionForYdb)) {
        return {};
    }
    auto taggedObject = objectsSnapshot.Get(objectId);
    if (!taggedObject) {
        return {};
    }
    return *taggedObject;
}

namespace {
    IRequestProcessorConfig::TFactory::TRegistrator<TTagsMetaProposeProcessor::THandlerConfig> ProposeRegistrator(TTagsMetaProposeProcessor::GetTypeName() + "_registrator");
    IRequestProcessorConfig::TFactory::TRegistrator<TTagsMetaConfirmProcessor::THandlerConfig> ConfirmRegistrator(TTagsMetaConfirmProcessor::GetTypeName() + "_registrator");
    IRequestProcessorConfig::TFactory::TRegistrator<TTagsMetaRejectProcessor::THandlerConfig> RejectRegistrator(TTagsMetaRejectProcessor::GetTypeName() + "_registrator");
}
