#pragma once

#include "config.h"

#include <drive/backend/processors/common_app/db_entities.h>
#include <drive/backend/processors/service_app/processor.h>
#include <drive/backend/processors/user_app/processor.h>

#include <drive/backend/areas/areas.h>
#include <drive/backend/data/device_tags.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/tags/tags_manager.h>


class TTagsMetaProposeProcessor: public TObjectsProposeProcessor<TDBTagMeta, TTagsMetaProposeProcessor> {
private:
    using TBase = TObjectsProposeProcessor<TDBTagMeta, TTagsMetaProposeProcessor>;

public:
    using TBase::TBase;

    const IDBEntitiesWithPropositionsManager<TDBTagMeta>* GetEntitiesManager(const NDrive::IServer* server) const override {
        return server ? &server->GetDriveDatabase().GetTagsManager().GetTagsMeta().GetEntityManager() : nullptr;
    }

    TAdministrativeAction::EEntity GetEntityType() const override {
        return TAdministrativeAction::EEntity::Tag;
    }
};

class TTagsMetaConfirmProcessor: public TObjectsConfirmProcessor<TDBTagMeta, TTagsMetaConfirmProcessor> {
private:
    using TBase = TObjectsConfirmProcessor<TDBTagMeta, TTagsMetaConfirmProcessor>;

public:
    using TBase::TBase;

    const IDBEntitiesWithPropositionsManager<TDBTagMeta>* GetEntitiesManager(const NDrive::IServer* server) const override {
        return server ? &server->GetDriveDatabase().GetTagsManager().GetTagsMeta().GetEntityManager() : nullptr;
    }

    TAdministrativeAction::EEntity GetEntityType() const override {
        return TAdministrativeAction::EEntity::Tag;
    }
};

class TTagsMetaRejectProcessor: public TObjectsRejectProcessor<TDBTagMeta, TTagsMetaRejectProcessor> {
private:
    using TBase = TObjectsRejectProcessor<TDBTagMeta, TTagsMetaRejectProcessor>;

public:
    using TBase::TBase;

    const IDBEntitiesWithPropositionsManager<TDBTagMeta>* GetEntitiesManager(const NDrive::IServer* server) const override {
        return server ? &server->GetDriveDatabase().GetTagsManager().GetTagsMeta().GetEntityManager() : nullptr;
    }

    TAdministrativeAction::EEntity GetEntityType() const override {
        return TAdministrativeAction::EEntity::Tag;
    }
};

class TRejectTagPropositionProcessor: public TAppCommonProcessor<TRejectTagPropositionProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TRejectTagPropositionProcessor, TEmptyConfig>;

public:
    TRejectTagPropositionProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server)
    {
    }

    const IEntityTagsManager& GetEntityTagsManager() const {
        return DriveApi->GetTagsManager().GetDeviceTags();
    }

    static TString GetTypeName() {
        return "tags/propositions/reject";
    }

protected:
    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& jsonValue) override;
};

class TConfirmTagPropositionProcessor: public TAppCommonProcessor<TConfirmTagPropositionProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TConfirmTagPropositionProcessor, TEmptyConfig>;

public:
    TConfirmTagPropositionProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server)
    {
    }

    const IEntityTagsManager& GetEntityTagsManager() const {
        return DriveApi->GetTagsManager().GetDeviceTags();
    }

    static TString GetTypeName() {
        return "tags/propositions/confirm";
    }

protected:
    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& jsonValue) override;
};

class TProposeTagProcessor: public TAppCommonProcessor<TProposeTagProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TProposeTagProcessor, TEmptyConfig>;

public:
    TProposeTagProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server)
    {
    }

    const IEntityTagsManager& GetEntityTagsManager() const {
        return DriveApi->GetTagsManager().GetDeviceTags();
    }

    static TString GetTypeName() {
        return "tags/propositions/propose";
    }

protected:
    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& jsonValue) override;
};

class TRegisterTagProcessor: public TAppCommonProcessor<TRegisterTagProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TRegisterTagProcessor, TEmptyConfig>;

public:
    TRegisterTagProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {}

    static TString GetTypeName() {
        return "service_tags/add";
    }

protected:
    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& jsonValue) override;
};

class TListTagsProcessor: public TAppCommonProcessor<TListTagsProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TListTagsProcessor, TEmptyConfig>;

public:
    TListTagsProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {}

    static TString GetTypeName() {
        return "service_tags/list";
    }

protected:
    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& jsonValue) override;
};

class TRemoveTagProcessor: public TAppCommonProcessor<TRemoveTagProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TRemoveTagProcessor, TEmptyConfig>;

public:
    TRemoveTagProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {}

    static TString GetTypeName() {
        return "service_tags/remove";
    }

protected:
    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& jsonValue) override;
};

class TListTagsByPerformerProcessor: public TAppCommonProcessor<TListTagsByPerformerProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TListTagsByPerformerProcessor, TEmptyConfig>;

public:
    using TBase::TBase;

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*jsonValue*/) override;

    static TString GetTypeName() {
        return "tags/by_performer";
    }
};

class TListEntityTagsProcessor: public TCommonServiceAppProcessorBase {
private:
    using TBase = TCommonServiceAppProcessorBase;

public:
    TListEntityTagsProcessor(const TCommonAppConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server)
    {
    }
    virtual bool IsObjectIdUUID() const {
        return true;
    }

protected:
    void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*jsonValue*/) override;
    virtual const IEntityTagsManager& GetEntityTagsManager() const = 0;
    virtual TMaybe<TTaggedObject> GetTaggedObject(const TString& objectId, NDrive::TEntitySession& session, NDrive::TEntitySession& sessionForYdb) const;
};

class TListAreaTagsProcessor
    : public TAppCommonRegistrator<TListAreaTagsProcessor, TEmptyConfig>
    , public TListEntityTagsProcessor
{
private:
    using TBase = TAppCommonRegistrator<TListAreaTagsProcessor, TEmptyConfig>;
protected:
    virtual const IEntityTagsManager& GetEntityTagsManager() const override {
        return DriveApi->GetAreasDB()->GetTagsManager();
    }
    virtual bool IsObjectIdUUID() const override {
        return false;
    }

public:
    TListAreaTagsProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config)
        , TListEntityTagsProcessor(config, context, authModule, server)
    {
    }

    static TString GetTypeName() {
        return "area_tags/list";
    }
};

class TListCarTagsProcessor
    : public TAppCommonRegistrator<TListCarTagsProcessor, TEmptyConfig>
    , public TListEntityTagsProcessor
{
private:
    using TBase = TAppCommonRegistrator<TListCarTagsProcessor, TEmptyConfig>;

protected:
    virtual const IEntityTagsManager& GetEntityTagsManager() const override {
        return DriveApi->GetTagsManager().GetDeviceTags();
    }

public:
    TListCarTagsProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config)
        , TListEntityTagsProcessor(config, context, authModule, server)
    {
    }

    static TString GetTypeName() {
        return "car_tags/list";
    }
};

class TListTraceTagsProcessor
    : public TAppCommonRegistrator<TListTraceTagsProcessor, TEmptyConfig>
    , public TListEntityTagsProcessor
{
private:
    using TBase = TAppCommonRegistrator<TListTraceTagsProcessor, TEmptyConfig>;

protected:
    virtual const IEntityTagsManager& GetEntityTagsManager() const override {
        return DriveApi->GetTagsManager().GetTraceTags();
    }
    virtual bool IsObjectIdUUID() const override {
        return false;
    }
    TMaybe<TTaggedObject> GetTaggedObject(const TString& objectId, NDrive::TEntitySession& session, NDrive::TEntitySession& sessionForYdb) const override;

public:
    TListTraceTagsProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config)
        , TListEntityTagsProcessor(config, context, authModule, server)
    {
    }

    static TString GetTypeName() {
        return "trace_tags/list";
    }
};

class TListUserTagsProcessor
    : public TAppCommonRegistrator<TListUserTagsProcessor, TEmptyConfig>
    , public TListEntityTagsProcessor
{
private:
    using TBase = TAppCommonRegistrator<TListUserTagsProcessor, TEmptyConfig>;

protected:
    virtual const IEntityTagsManager& GetEntityTagsManager() const override {
        return DriveApi->GetTagsManager().GetUserTags();
    }

public:
    TListUserTagsProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config)
        , TListEntityTagsProcessor(config, context, authModule, server)
    {
    }

    static TString GetTypeName() {
        return "user_tags/list";
    }
};

class TListAccountTagsProcessor
    : public TAppCommonRegistrator<TListAccountTagsProcessor, TEmptyConfig>
    , public TListEntityTagsProcessor
{
private:
    using TBase = TAppCommonRegistrator<TListAccountTagsProcessor, TEmptyConfig>;

protected:
    virtual const IEntityTagsManager& GetEntityTagsManager() const override {
        return DriveApi->GetTagsManager().GetAccountTags();
    }

public:
    TListAccountTagsProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config)
        , TListEntityTagsProcessor(config, context, authModule, server)
    {
    }

    virtual bool IsObjectIdUUID() const override {
        return false;
    }

    static TString GetTypeName() {
        return "account_tags/list";
    }
};

class TRemoveEntityTagProcessor: public TCommonServiceAppProcessorBase {
private:
    using TBase = TCommonServiceAppProcessorBase;

public:
    TRemoveEntityTagProcessor(const TCommonAppConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server)
    {
    }

protected:
    using TManagerPtr = const IEntityTagsManager*;
    using TManagers = TVector<TManagerPtr>;

protected:
    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr /*permissions*/, const NJson::TJsonValue& /*jsonValue*/) override;
    virtual TSet<TString> GetEntityObjectIds(const NJson::TJsonValue& jsonValue) const;
    virtual TManagers GetEntityTagsManagers() const = 0;
};

class TRemoveGenericTagProcessor
    : public TAppCommonRegistrator<TRemoveGenericTagProcessor, TEmptyConfig>
    , public TRemoveEntityTagProcessor
{
private:
    using TBase = TAppCommonRegistrator<TRemoveGenericTagProcessor, TEmptyConfig>;

public:
    TRemoveGenericTagProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config)
        , TRemoveEntityTagProcessor(config, context, authModule, server)
    {
    }

    static TString GetTypeName() {
        return "entity_tags/remove";
    }

protected:
    virtual TManagers GetEntityTagsManagers() const override;
};

class TRemoveAreaTagProcessor
    : public TAppCommonRegistrator<TRemoveAreaTagProcessor, TEmptyConfig>
    , public TRemoveEntityTagProcessor
{
private:
    using TBase = TAppCommonRegistrator<TRemoveAreaTagProcessor, TEmptyConfig>;

protected:
    virtual TManagers GetEntityTagsManagers() const override {
        return { &DriveApi->GetAreasDB()->GetTagsManager() };
    }

public:
    TRemoveAreaTagProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config)
        , TRemoveEntityTagProcessor(config, context, authModule, server)
    {
    }

    static TString GetTypeName() {
        return "area_tags/remove";
    }
};

class TRemoveCarTagProcessor
    : public TAppCommonRegistrator<TRemoveCarTagProcessor, TEmptyConfig>
    , public TRemoveEntityTagProcessor
{
private:
    using TBase = TAppCommonRegistrator<TRemoveCarTagProcessor, TEmptyConfig>;

protected:
    virtual TSet<TString> GetEntityObjectIds(const NJson::TJsonValue& jsonValue) const override;
    virtual TManagers GetEntityTagsManagers() const override {
        return { &DriveApi->GetTagsManager().GetDeviceTags() };
    }

public:
    TRemoveCarTagProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config)
        , TRemoveEntityTagProcessor(config, context, authModule, server)
    {
    }

    static TString GetTypeName() {
        return "car_tags/remove";
    }
};

class TRemoveUserTagProcessor
    : public TAppCommonRegistrator<TRemoveUserTagProcessor, TEmptyConfig>
    , public TRemoveEntityTagProcessor
{
private:
    using TBase = TAppCommonRegistrator<TRemoveUserTagProcessor, TEmptyConfig>;

protected:
    virtual TManagers GetEntityTagsManagers() const override {
        return { &DriveApi->GetTagsManager().GetUserTags() };
    }

public:
    TRemoveUserTagProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config)
        , TRemoveEntityTagProcessor(config, context, authModule, server)
    {
    }

    static TString GetTypeName() {
        return "user_tags/remove";
    }
};

class TRemoveTraceTagProcessor
    : public TAppCommonRegistrator<TRemoveTraceTagProcessor, TEmptyConfig>
    , public TRemoveEntityTagProcessor
{
private:
    using TBase = TAppCommonRegistrator<TRemoveTraceTagProcessor, TEmptyConfig>;

protected:
    virtual TManagers GetEntityTagsManagers() const override {
        return { &DriveApi->GetTagsManager().GetTraceTags() };
    }

public:
    TRemoveTraceTagProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config)
        , TRemoveEntityTagProcessor(config, context, authModule, server)
    {
    }

    static TString GetTypeName() {
        return "trace_tags/remove";
    }
};

class TRemoveAccountTagProcessor
    : public TAppCommonRegistrator<TRemoveAccountTagProcessor, TEmptyConfig>
    , public TRemoveEntityTagProcessor
{
private:
    using TBase = TAppCommonRegistrator<TRemoveAccountTagProcessor, TEmptyConfig>;

protected:
    virtual TManagers GetEntityTagsManagers() const override {
        return { &DriveApi->GetTagsManager().GetAccountTags() };
    }

public:
    TRemoveAccountTagProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config)
        , TRemoveEntityTagProcessor(config, context, authModule, server)
    {
    }

    static TString GetTypeName() {
        return "account_tags/remove";
    }
};

class TAddEntityTagProcessor: public TCommonServiceAppProcessorBase {
private:
    using TBase = TCommonServiceAppProcessorBase;

public:
    TAddEntityTagProcessor(const TCommonAppConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TCommonServiceAppProcessorBase(config, context, authModule, server) {}

protected:
    virtual bool IsObjectIdUUID() const {
        return true;
    }
    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& jsonValue) override;
    virtual TSet<TString> GetEntityObjectIds(const NJson::TJsonValue& jsonValue, NDrive::TEntitySession& /*session*/, TUserPermissions::TPtr permissions = nullptr) const;
    virtual const IEntityTagsManager& GetEntityTagsManager() const = 0;
};

class TAddAreaTagProcessor
    : public TAppCommonRegistrator<TAddAreaTagProcessor, TEmptyConfig>
    , public TAddEntityTagProcessor
{
private:
    using TBase = TAppCommonRegistrator<TAddAreaTagProcessor, TEmptyConfig>;

protected:
    virtual bool IsObjectIdUUID() const override {
        return false;
    }
    virtual const IEntityTagsManager& GetEntityTagsManager() const override {
        return DriveApi->GetAreasDB()->GetTagsManager();
    }

public:
    TAddAreaTagProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config)
        , TAddEntityTagProcessor(config, context, authModule, server)
    {
    }

    static TString GetTypeName() {
        return "area_tags/add";
    }
};

class TAddCarTagProcessor
    : public TAppCommonRegistrator<TAddCarTagProcessor, TEmptyConfig>
    , public TAddEntityTagProcessor
{
private:
    using TBase = TAppCommonRegistrator<TAddCarTagProcessor, TEmptyConfig>;

protected:
    virtual TSet<TString> GetEntityObjectIds(const NJson::TJsonValue& jsonValue, NDrive::TEntitySession& session, TUserPermissions::TPtr permissions = nullptr) const override;
    virtual const IEntityTagsManager& GetEntityTagsManager() const override {
        return DriveApi->GetTagsManager().GetDeviceTags();
    }

public:
    TAddCarTagProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config)
        , TAddEntityTagProcessor(config, context, authModule, server)
    {
    }

    static TString GetTypeName() {
        return "car_tags/add";
    }
};

class TAddUserTagProcessor
    : public TAppCommonRegistrator<TAddUserTagProcessor, TEmptyConfig>
    , public TAddEntityTagProcessor
{
private:
    using TBase = TAppCommonRegistrator<TAddUserTagProcessor, TEmptyConfig>;

protected:
    virtual TSet<TString> GetEntityObjectIds(const NJson::TJsonValue& jsonValue, NDrive::TEntitySession& session, TUserPermissions::TPtr permissions = nullptr) const override;
    virtual const IEntityTagsManager& GetEntityTagsManager() const override {
        return DriveApi->GetTagsManager().GetUserTags();
    }

public:
    TAddUserTagProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config)
        , TAddEntityTagProcessor(config, context, authModule, server)
    {
    }

    static TString GetTypeName() {
        return "user_tags/add";
    }
};

class TAddTraceTagProcessor
    : public TAppCommonRegistrator<TAddTraceTagProcessor, TEmptyConfig>
    , public TAddEntityTagProcessor
{
private:
    using TBase = TAppCommonRegistrator<TAddTraceTagProcessor, TEmptyConfig>;

protected:
    virtual bool IsObjectIdUUID() const override {
        return false;
    }

    virtual const IEntityTagsManager& GetEntityTagsManager() const override {
        return DriveApi->GetTagsManager().GetTraceTags();
    }

public:
    TAddTraceTagProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config)
        , TAddEntityTagProcessor(config, context, authModule, server)
    {
    }

    static TString GetTypeName() {
        return "trace_tags/add";
    }
};

class TAddAccountTagProcessor
    : public TAppCommonRegistrator<TAddAccountTagProcessor, TEmptyConfig>
    , public TAddEntityTagProcessor
{
private:
    using TBase = TAppCommonRegistrator<TAddAccountTagProcessor, TEmptyConfig>;

protected:
    virtual const IEntityTagsManager& GetEntityTagsManager() const override {
        return DriveApi->GetTagsManager().GetAccountTags();
    }

public:
    TAddAccountTagProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config)
        , TAddEntityTagProcessor(config, context, authModule, server)
    {
    }

    virtual bool IsObjectIdUUID() const override {
        return false;
    }

    static TString GetTypeName() {
        return "account_tags/add";
    }
};

class TUpdateEntityTagProcessor : public TCommonServiceAppProcessorBase {
private:
    using TBase = TCommonServiceAppProcessorBase;

public:
    TUpdateEntityTagProcessor(const TCommonAppConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TCommonServiceAppProcessorBase(config, context, authModule, server) {}

protected:
    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& jsonValue) override;
    virtual const IEntityTagsManager& GetEntityTagsManager() const = 0;
};

class TUpdateAreaTagProcessor
    : public TAppCommonRegistrator<TUpdateAreaTagProcessor, TEmptyConfig>
    , public TUpdateEntityTagProcessor
{
private:
    using TBase = TAppCommonRegistrator<TUpdateAreaTagProcessor, TEmptyConfig>;

protected:
    virtual const IEntityTagsManager& GetEntityTagsManager() const override {
        return DriveApi->GetAreasDB()->GetTagsManager();
    }

public:
    TUpdateAreaTagProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config)
        , TUpdateEntityTagProcessor(config, context, authModule, server)
    {
    }

    static TString GetTypeName() {
        return "area_tags/update";
    }
};

class TUpdateCarTagProcessor
    : public TAppCommonRegistrator<TUpdateCarTagProcessor, TEmptyConfig>
    , public TUpdateEntityTagProcessor
{
private:
    using TBase = TAppCommonRegistrator<TUpdateCarTagProcessor, TEmptyConfig>;

protected:
    virtual const IEntityTagsManager& GetEntityTagsManager() const override {
        return DriveApi->GetTagsManager().GetDeviceTags();
    }

public:
    TUpdateCarTagProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config)
        , TUpdateEntityTagProcessor(config, context, authModule, server)
    {
    }

    static TString GetTypeName() {
        return "car_tags/update";
    }
};

class TUpdateUserTagProcessor
    : public TAppCommonRegistrator<TUpdateUserTagProcessor, TEmptyConfig>
    , public TUpdateEntityTagProcessor
{
private:
    using TBase = TAppCommonRegistrator<TUpdateUserTagProcessor, TEmptyConfig>;

protected:
    virtual const IEntityTagsManager& GetEntityTagsManager() const override {
        return DriveApi->GetTagsManager().GetUserTags();
    }

public:
    TUpdateUserTagProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config)
        , TUpdateEntityTagProcessor(config, context, authModule, server)
    {
    }

    static TString GetTypeName() {
        return "user_tags/update";
    }
};

class TUpdateTraceTagProcessor
    : public TAppCommonRegistrator<TUpdateTraceTagProcessor, TEmptyConfig>
    , public TUpdateEntityTagProcessor
{
private:
    using TBase = TAppCommonRegistrator<TUpdateTraceTagProcessor, TEmptyConfig>;

protected:
    virtual const IEntityTagsManager& GetEntityTagsManager() const override {
        return DriveApi->GetTagsManager().GetTraceTags();
    }

public:
    TUpdateTraceTagProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config)
        , TUpdateEntityTagProcessor(config, context, authModule, server)
    {
    }

    static TString GetTypeName() {
        return "trace_tags/update";
    }
};

class TUpdateAccountTagProcessor
    : public TAppCommonRegistrator<TUpdateAccountTagProcessor, TEmptyConfig>
    , public TUpdateEntityTagProcessor
{
private:
    using TBase = TAppCommonRegistrator<TUpdateAccountTagProcessor, TEmptyConfig>;

protected:
    virtual const IEntityTagsManager& GetEntityTagsManager() const override {
        return DriveApi->GetTagsManager().GetAccountTags();
    }

public:
    TUpdateAccountTagProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config)
        , TUpdateEntityTagProcessor(config, context, authModule, server)
    {
    }

    static TString GetTypeName() {
        return "account_tags/update";
    }
};

class TDetailEntityTagProcessor : public TCommonServiceAppProcessorBase {
private:
    using TBase = TCommonServiceAppProcessorBase;

public:
    using TBase::TBase;

protected:
    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr /*permissions*/, const NJson::TJsonValue& /*jsonValue*/) override;
    virtual const IEntityTagsManager& GetEntityTagsManager() const = 0;
};

class TDetailCarTagProcessor
    : public TAppCommonRegistrator<TDetailCarTagProcessor, TEmptyConfig>
    , public TDetailEntityTagProcessor
{
private:
    using TRegistrator = TAppCommonRegistrator<TDetailCarTagProcessor, TEmptyConfig>;
    using TBase = TDetailEntityTagProcessor;

protected:
    virtual const IEntityTagsManager& GetEntityTagsManager() const override {
        return DriveApi->GetTagsManager().GetDeviceTags();
    }

public:
    TDetailCarTagProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TRegistrator(config)
        , TBase(config, context, authModule, server)
    {
    }

    static TString GetTypeName() {
        return "car_tags/details";
    }
};

class TDetailUserTagProcessor
    : public TAppCommonRegistrator<TDetailUserTagProcessor, TEmptyConfig>
    , public TDetailEntityTagProcessor
{
private:
    using TRegistrator = TAppCommonRegistrator<TDetailUserTagProcessor, TEmptyConfig>;
    using TBase = TDetailEntityTagProcessor;

protected:
    virtual const IEntityTagsManager& GetEntityTagsManager() const override {
        return DriveApi->GetTagsManager().GetUserTags();
    }

public:
    TDetailUserTagProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TRegistrator(config)
        , TBase(config, context, authModule, server)
    {
    }

    static TString GetTypeName() {
        return "user_tags/details";
    }
};

class TDetailAccountTagProcessor
    : public TAppCommonRegistrator<TDetailAccountTagProcessor, TEmptyConfig>
    , public TDetailEntityTagProcessor
{
private:
    using TRegistrator = TAppCommonRegistrator<TDetailAccountTagProcessor, TEmptyConfig>;
    using TBase = TDetailEntityTagProcessor;

protected:
    virtual const IEntityTagsManager& GetEntityTagsManager() const override {
        return DriveApi->GetTagsManager().GetAccountTags();
    }

public:
    TDetailAccountTagProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TRegistrator(config)
        , TBase(config, context, authModule, server)
    {
    }

    static TString GetTypeName() {
        return "account_tags/details";
    }
};

class TDetailTraceTagProcessor
    : public TAppCommonRegistrator<TDetailTraceTagProcessor, TEmptyConfig>
    , public TDetailEntityTagProcessor
{
private:
    using TRegistrator = TAppCommonRegistrator<TDetailTraceTagProcessor, TEmptyConfig>;
    using TBase = TDetailEntityTagProcessor;

protected:
    virtual const IEntityTagsManager& GetEntityTagsManager() const override {
        return DriveApi->GetTagsManager().GetTraceTags();
    }

public:
    TDetailTraceTagProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TRegistrator(config)
        , TBase(config, context, authModule, server)
    {
    }

    static TString GetTypeName() {
        return "trace_tags/details";
    }
};

class TSetCarTagProcessor: public TAppCommonProcessor<TSetCarTagProcessor, TSetCarTagProcessorConfig> {
    using TBase = TAppCommonProcessor<TSetCarTagProcessor, TSetCarTagProcessorConfig>;

public:
    using TBase::TBase;

    static TString GetTypeName() {
        return "tags_actualization";
    }

protected:
    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr /*permissions*/, const NJson::TJsonValue& /*jsonValue*/) override;

private:
    TString GetDeviceBySession(const TString& sessionId, TUserPermissions::TPtr permissions, NDrive::TEntitySession& tx) const;
};

class TEntityHistoryProcessor: public TCommonServiceAppProcessorBase {
private:
    using TBase = TCommonServiceAppProcessorBase;

protected:
    virtual const IEntityTagsManager& GetEntityTagsManager() const = 0;
    virtual NJson::TJsonValue GetObjectsReport(const TSet<TString>& objects, TUserPermissions::TPtr permissions, NDrive::TEntitySession& session) const = 0;
    virtual NJson::TJsonValue GetUsersReport(const TSet<TString>& users, TUserPermissions::TPtr permissions, NDrive::TEntitySession& session) const;
    virtual std::function<bool(const TTagHistoryEvent&)> GetCustomPredicate() const {
        return [](const TTagHistoryEvent&) -> bool {
            return true;
        };
    }
    virtual bool IsObjectIdUUID() const {
        return true;
    }

public:
    using TBase::TBase;

protected:
    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr /*permissions*/, const NJson::TJsonValue& /*jsonValue*/) override;
};

class TBaseCarTagsHistoryProcessor : public TEntityHistoryProcessor {
private:
    using TBase = TEntityHistoryProcessor;

public:
    using TBase::TBase;

    virtual const IEntityTagsManager& GetEntityTagsManager() const override {
        return DriveApi->GetTagsManager().GetDeviceTags();
    }

    virtual NJson::TJsonValue GetObjectsReport(const TSet<TString>& objects, TUserPermissions::TPtr permissions, NDrive::TEntitySession& session) const override;
};

class TAreaTagsHistoryProcessor
    : public TAppCommonRegistrator<TAreaTagsHistoryProcessor, TEmptyConfig>
    , public TEntityHistoryProcessor
{
private:
    using TBase = TAppCommonRegistrator<TAreaTagsHistoryProcessor, TEmptyConfig>;

protected:
    virtual bool IsObjectIdUUID() const override {
        return false;
    }
    virtual const IEntityTagsManager& GetEntityTagsManager() const override {
        return DriveApi->GetAreasDB()->GetTagsManager();
    }
    virtual NJson::TJsonValue GetObjectsReport(const TSet<TString>& /*objectIds*/, TUserPermissions::TPtr /*permissions*/, NDrive::TEntitySession& /*session*/) const override {
        return NJson::JSON_ARRAY;
    }

public:
    TAreaTagsHistoryProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config)
        , TEntityHistoryProcessor(config, context, authModule, server)
    {
    }

    static TString GetTypeName() {
        return "area_tags_history";
    }
};

class TCarTagsHistoryProcessor
    : public TAppCommonRegistrator<TCarTagsHistoryProcessor, TEmptyConfig>
    , public TBaseCarTagsHistoryProcessor
{
private:
    using TBase = TAppCommonRegistrator<TCarTagsHistoryProcessor, TEmptyConfig>;

public:
    TCarTagsHistoryProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config)
        , TBaseCarTagsHistoryProcessor(config, context, authModule, server)
    {
    }

    static TString GetTypeName() {
        return "car_history";
    }
};

class TUserTagsHistoryProcessor
    : public TAppCommonRegistrator<TUserTagsHistoryProcessor, TEmptyConfig>
    , public TEntityHistoryProcessor
{
private:
    using TBase = TAppCommonRegistrator<TUserTagsHistoryProcessor, TEmptyConfig>;

protected:
    virtual const IEntityTagsManager& GetEntityTagsManager() const override {
        return DriveApi->GetTagsManager().GetUserTags();
    }
    virtual NJson::TJsonValue GetObjectsReport(const TSet<TString>& objects, TUserPermissions::TPtr permissions, NDrive::TEntitySession& session) const override {
        return GetUsersReport(objects, permissions, session);
    }

public:
    TUserTagsHistoryProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config)
        , TEntityHistoryProcessor(config, context, authModule, server)
    {
    }

    static TString GetTypeName() {
        return "user_tags_history";
    }
};

class TAccountTagsHistoryProcessor
    : public TAppCommonRegistrator<TAccountTagsHistoryProcessor, TEmptyConfig>
    , public TEntityHistoryProcessor
{
private:
    using TBase = TAppCommonRegistrator<TAccountTagsHistoryProcessor, TEmptyConfig>;

protected:
    virtual const IEntityTagsManager& GetEntityTagsManager() const override {
        return DriveApi->GetTagsManager().GetAccountTags();
    }
    virtual NJson::TJsonValue GetObjectsReport(const TSet<TString>& objects, TUserPermissions::TPtr permissions, NDrive::TEntitySession& session) const override;

public:
    TAccountTagsHistoryProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config)
        , TEntityHistoryProcessor(config, context, authModule, server)
    {
    }

    virtual bool IsObjectIdUUID() const override {
        return false;
    }

    static TString GetTypeName() {
        return "account_tags_history";
    }
};

class TEntityTagHistoryDetailsProcessor: public TCommonServiceAppProcessorBase {
private:
    using TBase = TCommonServiceAppProcessorBase;

protected:
    virtual const IEntityTagsManager& GetEntityTagsManager() const = 0;

public:
    using TBase::TBase;

protected:
    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*jsonValue*/) override;
};

class TCarHistoryDetailsProcessor
    : public TAppCommonRegistrator<TCarHistoryDetailsProcessor, TEmptyConfig>
    , public TEntityTagHistoryDetailsProcessor
{
private:
    using TBase = TAppCommonRegistrator<TCarHistoryDetailsProcessor, TEmptyConfig>;

public:
    TCarHistoryDetailsProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config)
        , TEntityTagHistoryDetailsProcessor(config, context, authModule, server)
    {
    }

    static TString GetTypeName() {
        return "car_history_details";
    }

protected:
    virtual const IEntityTagsManager& GetEntityTagsManager() const override {
        return DriveApi->GetTagsManager().GetDeviceTags();
    }
};

class TUserHistoryDetailsProcessor
    : public TAppCommonRegistrator<TUserHistoryDetailsProcessor, TEmptyConfig>
    , public TEntityTagHistoryDetailsProcessor
{
private:
    using TBase = TAppCommonRegistrator<TUserHistoryDetailsProcessor, TEmptyConfig>;

public:
    TUserHistoryDetailsProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config)
        , TEntityTagHistoryDetailsProcessor(config, context, authModule, server)
    {
    }

    static TString GetTypeName() {
        return "user_tags_history_details";
    }

protected:
    virtual const IEntityTagsManager& GetEntityTagsManager() const override {
        return DriveApi->GetTagsManager().GetUserTags();
    }
};

class TAccountHistoryDetailsProcessor
    : public TAppCommonRegistrator<TAccountHistoryDetailsProcessor, TEmptyConfig>
    , public TEntityTagHistoryDetailsProcessor
{
private:
    using TBase = TAppCommonRegistrator<TAccountHistoryDetailsProcessor, TEmptyConfig>;

public:
    TAccountHistoryDetailsProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config)
        , TEntityTagHistoryDetailsProcessor(config, context, authModule, server)
    {
    }

    static TString GetTypeName() {
        return "account_tags_history_details";
    }

protected:
    virtual const IEntityTagsManager& GetEntityTagsManager() const override {
        return DriveApi->GetTagsManager().GetAccountTags();
    }
};

class TUserHistoryProcessor: public TAppCommonProcessor<TUserHistoryProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TUserHistoryProcessor, TEmptyConfig>;

public:
    using TBase::TBase;

    static TString GetTypeName() {
        return "user_history";
    }

protected:
    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr /*permissions*/, const NJson::TJsonValue& /*jsonValue*/) override;
};

class ICarRepairTagBaseProcessor : public TCommonServiceAppProcessorBase {
private:
    using TBase = TCommonServiceAppProcessorBase;

public:
    using TBase::TBase;

protected:
    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& jsonValue) override;

private:
    virtual void ProcessRequest(const TDBTag* tag, const TString& queryId, TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& jsonValue) const = 0;
};

class TCarRepairTagDetailsProcessor : public TAppCommonRegistrator<TCarRepairTagDetailsProcessor, TEmptyConfig>, public ICarRepairTagBaseProcessor {
private:
    using TBase = TAppCommonRegistrator<TCarRepairTagDetailsProcessor, TEmptyConfig>;

public:
    TCarRepairTagDetailsProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config)
        , ICarRepairTagBaseProcessor(config, context, authModule, server)
    {}

    static TString GetTypeName() {
        return "repair_tag_details";
    }

private:
    virtual void ProcessRequest(const TDBTag* tag, const TString& queryId, TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& jsonValue) const override;
};

class TCarRepairTagCancelProcessor : public TAppCommonRegistrator<TCarRepairTagCancelProcessor, TEmptyConfig>, public ICarRepairTagBaseProcessor {
private:
    using TBase = TAppCommonRegistrator<TCarRepairTagCancelProcessor, TEmptyConfig>;

public:
    TCarRepairTagCancelProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config)
        , ICarRepairTagBaseProcessor(config, context, authModule, server)
    {}

    static TString GetTypeName() {
        return "repair_tag_cancel";
    }

private:
    virtual void ProcessRequest(const TDBTag* tag, const TString& queryId, TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& jsonValue) const override;
};

class TCurrentMajorQueriesProcessor : public TAppCommonProcessor<TCurrentMajorQueriesProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TCurrentMajorQueriesProcessor, TEmptyConfig>;

public:
    using TBase::TBase;

    static TString GetTypeName() {
        return "current_major_queries";
    }

protected:
    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr /*permissions*/, const NJson::TJsonValue& /*jsonValue*/) override;
};

class TAddServiceWorkerTagProcessor
    : public TAppCommonRegistrator<TAddServiceWorkerTagProcessor, TEmptyConfig>
    , public TAddEntityTagProcessor
{
private:
    using TBase = TAppCommonRegistrator<TAddServiceWorkerTagProcessor, TEmptyConfig>;

protected:
    virtual TSet<TString> GetEntityObjectIds(const NJson::TJsonValue& jsonValue, NDrive::TEntitySession& session, TUserPermissions::TPtr permissions = nullptr) const override;
    virtual const IEntityTagsManager& GetEntityTagsManager() const override {
        return DriveApi->GetTagsManager().GetUserTags();
    }

public:
    TAddServiceWorkerTagProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config)
        , TAddEntityTagProcessor(config, context, authModule, server)
    {
    }

    static TString GetTypeName() {
        return "service_worker_tags/add";
    }
};

class TServiceCarTagDetailsProcessor : public TAppCommonProcessor<TServiceCarTagDetailsProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TServiceCarTagDetailsProcessor, TEmptyConfig>;

public:
    using TBase::TBase;

    static TString GetTypeName() {
        return "service/car_tags/details";
    }

protected:
    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& jsonValue) override;
};

class TServiceAppCarTagsHistoryProcessor
    : public TAppCommonRegistrator<TServiceAppCarTagsHistoryProcessor, TEmptyConfig>
    , public TBaseCarTagsHistoryProcessor
{
private:
    using TBase = TAppCommonRegistrator<TServiceAppCarTagsHistoryProcessor, TEmptyConfig>;

private:
    virtual std::function<bool(const TTagHistoryEvent&)> GetCustomPredicate() const override;

public:
    TServiceAppCarTagsHistoryProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config)
        , TBaseCarTagsHistoryProcessor(config, context, authModule, server)
    {
    }

    static TString GetTypeName() {
        return "service/car_tags/history";
    }

    static inline TSet<TString> SessionTags{ "old_state_reservation", "old_state_acceptance", "old_state_riding", "old_state_parking" };
};
