#include "aggression_helper.h"

#include <saas/util/types/cast.h>

namespace NDrive {

    TAggressionHelper::TAggressiveEvent::TAggressiveEvent(const TBase& base)
        : TBase(base)
    {
    }

    NJson::TJsonValue TAggressionHelper::TAggressiveEvent::GetReport(TInstant deadline, const TMaybe<NDrive::TTrackProjector>& trackProjector) const
    {
        NJson::TJsonValue json;
        NJson::InsertField(json, "ts", Timestamp.Seconds());
        if (trackProjector) {
            NJson::InsertField(json, "center", trackProjector->Project(Location, Timestamp));
        } else {
            NJson::InsertField(json, "center", Location);
        }
        if (Kind != TEvent::EKind::Unknown) {
            NJson::InsertField(json, "kind", NJson::Stringify(Kind));
        }
        const auto resp = NThreading::Initialize(GeocoderResponse);
        if (GeocoderResponse.Initialized()) {
            if (GeocoderResponse.Wait(deadline) && GeocoderResponse.HasValue()) {
                NJson::InsertField(json, "description", GeocoderResponse.GetValue().Title);
                TUnistatSignalsCache::SignalAdd("aggressive_event-geocoder", "response-success", 1);
            } else {
                NDrive::TEventLog::Log("GetAggressiveEventGeocoderError", NJson::TMapBuilder
                    ("location", NJson::ToJson(Location.ToString()))
                    ("exception", NThreading::GetExceptionInfo(GeocoderResponse))
                );
                TUnistatSignalsCache::SignalAdd("aggressive_event-geocoder", "response-error", 1);
            }
        }
        return json;
    }

    TAggressionHelper::TAggressionHelper(
        const TSet<TString>& aggressiveTagNames,
        const NDrive::TGeocoder* geocoder /*= nullptr*/
    )
        : AggressiveTagNames(aggressiveTagNames)
        , Geocoder(geocoder)
    {
    }

    const TScoringTraceTag* TAggressionHelper::GetAggressiveTag(const THistoryRideObject& ride) const {
        const auto& tags = ride.GetTraceTags();
        if (!tags) {
            return nullptr;
        }
        for (auto&& tag : tags->GetTags()) {
            if (!tag) {
                continue;
            }
            if (AggressiveTagNames.contains(tag->GetName())) {
                return tag.GetTagAs<TScoringTraceTag>();
            }
        }
        return nullptr;
    }

    TVector<TAggressionHelper::TAggressiveEvent> TAggressionHelper::GetAggressiveEvents(const TScoringTraceTag& tag, ELocalization locale) const {
        TVector<TAggressiveEvent> events;
        for (auto&& tagEvent : tag.GetEvents()) {
            TAggressiveEvent event(tagEvent);
            if (Geocoder) {
                event.GeocoderResponse = Geocoder->Decode(event.Location, enum_cast<ELanguage>(locale));
                TUnistatSignalsCache::SignalAdd("aggressive_event-geocoder", "request", 1);
            }
            events.push_back(std::move(event));
        }
        std::stable_sort(events.begin(), events.end(), [](const TAggressiveEvent& lhs, const TAggressiveEvent& rhs) {
            return lhs.Timestamp < rhs.Timestamp;
        });
        return events;
    }

}
