#pragma once
#include "aggression_helper.h"
#include "violations_helper.h"

#include <drive/backend/fines/manager.h>
#include <drive/backend/processors/common_app/fetcher.h>
#include <drive/backend/processors/common_app/processor.h>

#include <drive/backend/billing/manager.h>
#include <drive/backend/tracks/client.h>
#include <drive/backend/tracks/report.h>

#include <drive/library/cpp/geocoder/api/client.h>

namespace NDeviceReport {
    constexpr TReportTraits ReportTaxiHistory =
        EReportTraits::ReportCars |
        EReportTraits::ReportPatches |
        EReportTraits::ReportViews |
        EReportTraits::ReportModels;
    constexpr TReportTraits OrganizationExport =
        EReportTraits::ReportCars |
        EReportTraits::ReportViews;

    inline TReportTraits GetReportTraits(TStringBuf report) {
        TReportTraits reportTraits;
        if (report == "organization_export") {
            reportTraits = OrganizationExport;
        } else {
            reportTraits = ReportTaxiHistory;
        }
        return reportTraits;
    }
}

class THistoryRideObject;

template <class T>
class TSessionProcessorTraits {
public:
    using TRideAggressions = TMap<TString, TVector<NDrive::TAggressionHelper::TAggressiveEvent>>;
    using TFinesMap = TMap<TString, NDrive::NFine::TFineDB::TEntities>;

public:
    TSessionProcessorTraits(IReplyContext::TPtr context, const NDrive::IServer* server);

protected:
    const TCarsFetcher& GetFetcher(const TSet<TString>& objectIds, TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NDeviceReport::TReportTraits deviceTraits = NDeviceReport::ReportTaxiHistory, const NDriveModelReport::TReportTraits modelTraits = NDriveModelReport::UserReport);
    const TCarsFetcher& GetFetcher(TConstArrayRef<THistoryRideObject> sessions, TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NDeviceReport::TReportTraits deviceTraits = NDeviceReport::ReportTaxiHistory, const NDriveModelReport::TReportTraits modelTraits = NDriveModelReport::UserReport);
    NJson::TJsonValue GetReport(const THistoryRideObject& session, TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, bool detailed, const NDriveSession::TReportTraits traits = NDriveSession::ReportUserApp);

    const TMap<TString, NThreading::TFuture<NDrive::TTracksLinker::TResults>>& GetTracks() const {
        return Tracks;
    }
    const TRideAggressions& GetRealtimeRideAggressions() const {
        return RealtimeRideAggressions;
    }
    const TRideAggressions& GetRideAggressions() const {
        return RideAggressions;
    }

    void PrefetchFullCompiledRiding(TArrayRef<THistoryRideObject> sessions, TJsonReport::TGuard& g);
    void PrefetchGeocodedLocations(TConstArrayRef<THistoryRideObject> sessions, TJsonReport::TGuard& g);
    void PrefetchPayments(TConstArrayRef<THistoryRideObject> sessions, NDrive::TEntitySession& tx, TJsonReport::TGuard& g);
    void PrefetchTracks(TConstArrayRef<THistoryRideObject> sessions, TJsonReport::TGuard& g, bool needGeocodedLocation = false, bool useRawTracks = false);
    void PrefetchUsers(TConstArrayRef<THistoryRideObject> sessions, NDrive::TEntitySession& tx, TJsonReport::TGuard& g);
    void PrefetchAggressiveEvents(TArrayRef<THistoryRideObject> sessions, TJsonReport::TGuard& g);
    void PrefetchFineAttachments(TConstArrayRef<THistoryRideObject> sessions, NDrive::TEntitySession& tx, TJsonReport::TGuard& g);

    void SetLocale(ELocalization locale) {
        Locale = locale;
    }
    void SetReportOffer(bool value) {
        ReportOffer = value;
    }
    void SetTrackFormat(const TString& value) {
        TrackFormat = value;
    }
    void SetTrackStatus(NDrive::ECarStatus value) {
        TrackStatus = value;
    }

    const TFinesMap& GetFines() const {
        return Fines;
    }

    using TFilter = std::function<bool(const THistoryRideObject& ride)>;
    TFilter ConstructFilter(const TSet<TString>& filters) const;

    NDrive::TTracksLinker::TResults FilterTrack(NDrive::TTracksLinker::TResults&& results) const;
    bool IsSpeedLimitExceeded(const THistoryRideObject& session, double speedLimit = 0) const;

    TMaybe<double> GetRideAggressionScore(const THistoryRideObject& session) const;

    TString GetGeocodedStart(const TString& sessionId, TJsonReport::TGuard& g) const;
    TString GetGeocodedFinish(const TString& sessionId, TJsonReport::TGuard& g) const;

private:
    T* Get() {
        return static_cast<T*>(this);
    }
    const T* Get() const {
        return static_cast<const T*>(this);
    }

protected:
    TMaybe<TCarsFetcher> Fetcher;

private:
    TMap<TString, TCachedPayments> CachedPayments;
    TMap<TString, NThreading::TFuture<NDrive::TGeocoder::TResponse>> Starts;
    TMap<TString, NThreading::TFuture<NDrive::TGeocoder::TResponse>> Finishes;
    TMap<TString, NThreading::TFuture<NDrive::TTracksLinker::TResults>> Tracks;
    TMap<TString, NThreading::TFuture<NDrive::TTracks>> RawTracks;

    TMap<TString, TDriveUserData> Users;
    TMap<TString, NDrive::NTrustClient::TPaymentMethod> UserPaymentCards;
    THolder<NDrive::ITrackClient> TracksClient;
    THolder<NDrive::ITrackClient> SignalqTracksClient;
    TFinesMap Fines;
    TMap<TString, NDrive::NFine::TFineAttachmentManager::TEntities> FineAttachments;
    THolder<NDrive::TTracksLinker> TracksLinker;
    TString TrackFormat = "geojson";
    TMaybe<NDrive::ECarStatus> TrackStatus;
    ELocalization Locale = DefaultLocale;

    THolder<NDrive::TAggressionHelper> AggressionHelper;
    THolder<NDrive::TAggressionHelper> RealtimeAggressionHelper;
    TRideAggressions RealtimeRideAggressions;
    TRideAggressions RideAggressions;

    TAtomicSharedPtr<NDrive::TViolationsHelper> ViolationsHelper;
    TAtomicSharedPtr<TYDBTracksClient> YDBTracksClient;

    TSet<TString> SpeedingTagNames;
    TSet<NDrive::ECarStatus> HiddenTrackStatuses;
    bool ReportOffer = false;
    double HardSpeedThreshold = 0;
};
