#pragma once

#include "base.h"

#include <drive/backend/common/localization.h>
#include <drive/backend/data/area_tags.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/fines/autocode/entries.h>
#include <drive/backend/fines/constructors/fine_attachment_constructor.h>
#include <drive/backend/fines/constructors/fine_constructor.h>
#include <drive/backend/fines/constructors/tag_constructor.h>
#include <drive/backend/fines/context_fetchers/fine_context.h>
#include <drive/backend/history_iterator/history_iterator.h>
#include <drive/backend/saas/api.h>
#include <drive/backend/tracks/client.h>

#include <drive/library/cpp/threading/future_cast.h>
#include <drive/library/cpp/tracks/quality.h>

#include <rtline/util/types/cast.h>

namespace {
    THolder<NDrive::TAggressionHelper> CreateAggressionHelper(TStringBuf typeName, const TString& key, const NDrive::IServer& server) {
        const auto geocoder = server.GetDriveAPI()->HasGeocoderClient() ? &server.GetDriveAPI()->GetGeocoderClient() : nullptr;
        const auto& settings = server.GetSettings();
        const auto aggressiveTagsFilter = settings.GetHandlerValueDef<TString>(typeName, key, settings.GetHandlerValueDef<TString>("default", key, ""));
        const TSet<TString> aggressiveTagNames = StringSplitter(aggressiveTagsFilter).Split(',');
        if (aggressiveTagNames.empty()) {
            return nullptr;
        }
        return MakeHolder<NDrive::TAggressionHelper>(aggressiveTagNames, geocoder);
    }

    TMap<TString, TString> StatusRemap = {
        { TChargableTag::Reservation,   "reservation" },
        { TChargableTag::Acceptance,    "acceptance" },
        { TChargableTag::Riding,        "riding" },
        { TChargableTag::Parking,       "parking" },
    };
}

template <class T>
TSessionProcessorTraits<T>::TSessionProcessorTraits(IReplyContext::TPtr context, const NDrive::IServer* server) {
    Y_ENSURE(context);
    Y_ENSURE(server);
    TStringBuf typeName = context->GetUri();
    typeName.SkipPrefix("/");
    const auto& settings = server->GetSettings();
    AggressionHelper = CreateAggressionHelper(typeName, "aggressive_filter_tags", *server);
    RealtimeAggressionHelper = CreateAggressionHelper(typeName, "realtime_aggressive_filter_tags", *server);
    ViolationsHelper = MakeAtomicShared<NDrive::TViolationsHelper>(*server);

    const TString speedingTagsFilter = settings.GetHandlerValueDef<TString>(typeName, "speeding_filter_tags", settings.GetHandlerValueDef<TString>("default", "speeding_filter_tags", ""));
    SpeedingTagNames = StringSplitter(speedingTagsFilter).Split(',');
}

template <class T>
const TCarsFetcher& TSessionProcessorTraits<T>::GetFetcher(const TSet<TString>& objectIds, TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NDeviceReport::TReportTraits deviceTraits /* = NDeviceReport::ReportTaxiHistory */, const NDriveModelReport::TReportTraits modelTraits /* = NDriveModelReport::UserReport */) {
    R_ENSURE(!Fetcher, HTTP_INTERNAL_SERVER_ERROR, "fetcher has already been initialized");
    Fetcher.ConstructInPlace(*Get()->GetServer(), deviceTraits, /* deadline = */ TInstant::Max(), modelTraits);
    {
        auto eg = g.BuildEventGuard("FetchCarsData");
        Fetcher->SetIsRealtime(false);
        if (!Fetcher->FetchData(permissions, objectIds)) {
            eg.AddEvent("FetchDataFailure");
        }
    }
    return *Fetcher;
}

template <class T>
const TCarsFetcher& TSessionProcessorTraits<T>::GetFetcher(TConstArrayRef<THistoryRideObject> sessions, TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NDeviceReport::TReportTraits deviceTraits /* = NDeviceReport::ReportTaxiHistory */, const NDriveModelReport::TReportTraits modelTraits /* = NDriveModelReport::UserReport */) {
    TSet<TString> objectIds;
    for (auto&& i : sessions) {
        objectIds.insert(i.GetObjectId());
    }
    return GetFetcher(objectIds, g, permissions, deviceTraits, modelTraits);
}

template <class T>
NJson::TJsonValue TSessionProcessorTraits<T>::GetReport(const THistoryRideObject& session, TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, bool detailed, const NDriveSession::TReportTraits traits) {
    const auto& sessionId = session.GetSessionId();
    auto guard = g.BuildEventGuard("GetReport:" + sessionId);
    auto fullCompiledRiding = session.GetFullCompiledRiding();
    auto minimalCompiledRiding = fullCompiledRiding
        ? static_cast<TAtomicSharedPtr<TMinimalCompiledRiding>>(fullCompiledRiding)
        : session.GetMinimalCompiledRiding();
    auto server = Yensured(Get()->GetServer());
    auto localization = server->GetLocalization();

    NJson::TJsonValue bill = NJson::JSON_NULL;
    if (fullCompiledRiding && fullCompiledRiding->HasBill() && detailed) {
        bill = fullCompiledRiding->GetBillUnsafe().GetReport(Locale, traits, *server);
    }

    NJson::TJsonValue fines = NJson::JSON_NULL;
    NJson::TJsonValue finesTotal = NJson::JSON_NULL;
    if (Get()->GetDriveApi()->HasFinesManager()) {
        auto eg = g.BuildEventGuard("Fines");

        const auto& finesManager = Get()->GetDriveApi()->GetFinesManager();
        double finesTotalCost = 0;
        for (auto&& fine : session.GetFines()) {
            finesTotalCost += fine.GetSumToPay();
            if (!detailed) {
                continue;
            }

            TString articleCode = fine.GetArticleCode();
            TString readableArticle;

            auto articleMatcherPtr = finesManager.GetFineArticleMatcherPtr(/* ensureActual = */ false);
            if (!!articleMatcherPtr) {
                if (!articleCode) {
                    articleMatcherPtr->DetermineArticle(fine.GetArticleKoap(), articleCode);
                }

                readableArticle = articleMatcherPtr->GetReadableArticleDefault(fine.GetArticleKoap(), articleCode);
            }

            auto fineReport = fine.BuildReport(NDrive::NFine::NFineTraits::NoTraits);
            fineReport.InsertValue("article_code", articleCode);
            fineReport.InsertValue("article_koap_human_readable", readableArticle);
            if (const auto& currentFinePhotos = FineAttachments.FindPtr(fine.GetId())) {
                for (const auto& finePhotoEntry : *currentFinePhotos) {
                    fineReport["photos"].AppendValue(finePhotoEntry.BuildReport());
                }
            }
            fines.AppendValue(std::move(fineReport));
        }
        if (finesTotalCost >= 0.01) {
            finesTotal["cost"] = finesTotalCost;
            finesTotal["currency_code"] = "RUB";
        }
    }

    NJson::TJsonValue payment = NJson::JSON_NULL;
    if (fullCompiledRiding && fullCompiledRiding->HasBill()) {
        payment.InsertValue("cashback", 0.01 * fullCompiledRiding->GetBillUnsafe().GetCashbackSum());
    }
    if (minimalCompiledRiding) {
        payment.InsertValue("cost", 0.01 * minimalCompiledRiding->GetSumPrice());
        payment.InsertValue("currency_code", "RUB");
    }

    NJson::TJsonValue offer = NJson::JSON_NULL;
    if (auto o = session.GetOffer()) {
        ICommonOffer::TReportOptions reportOptions(Locale, traits);
        offer = o->BuildJsonReport(reportOptions, *server);
    }

    NJson::TJsonValue payments = NJson::JSON_NULL;
    for (auto&& payment : CachedPayments[sessionId].GetPayments()) {
        if (payment.GetBillingType() != EBillingType::CarUsage) {
            continue;
        }
        if (payment.GetPaymentType() != NDrive::NBilling::EAccount::Trust && payment.GetPaymentType() != NDrive::NBilling::EAccount::MobilePayment) {
            continue;
        }
        const auto card = UserPaymentCards.find(payment.GetPayMethod());
        const TString& id = payment.GetPaymentId();
        NJson::TJsonValue paymentMethod;
        paymentMethod["card_number"] = payment.GetCardMask();
        paymentMethod["type"] = "card";
        if (card != UserPaymentCards.end()) {
            paymentMethod["system"] = card->second.GetPSystem();
            paymentMethod["title"] = card->second.GetPSystem() + ' ' + payment.GetCardMask();
        } else {
            paymentMethod["title"] = payment.GetCardMask();
        }
        NJson::TJsonValue paymentReport;
        paymentReport["payment_id"] = id;
        paymentReport["payment_method"] = std::move(paymentMethod);
        paymentReport["reciept"] = TStringBuilder() << "https://trust.yandex.ru/checks/" << id << "/receipts/" << id << "?mode=pdf";
        paymentReport["sum"] = 0.01 * payment.GetSum();
        paymentReport["updated_at"] = payment.GetLastUpdate().ToStringUpToSeconds();
        payments.AppendValue(std::move(paymentReport));
    }

    bool isSpeeding = IsSpeedLimitExceeded(session);
    NJson::TJsonValue route = NJson::JSON_NULL;
    TMaybe<NDrive::TTrackProjector> trackProjector;
    {
        auto startLocation = session.GetStartLocation();
        if (startLocation) {
            TString geocoded = GetGeocodedStart(sessionId, g);
            NJson::TJsonValue source;
            if (auto regionTag = NDrive::GetRegionTag(startLocation->GetCoord(), *server)) {
                auto regionKey = TStringBuilder() << "areas.region." << regionTag << ".name";
                source["region"] = localization ? localization->GetLocalString(Locale, regionKey) : regionKey;
            }
            source["point"] = NJson::ToJson(startLocation->GetCoord());
            source["short_text"] = geocoded;
            route.InsertValue("source", std::move(source));
        }
        auto finishLocation = session.GetLastLocation();
        if (finishLocation) {
            TString geocoded = GetGeocodedFinish(sessionId, g);
            NJson::TJsonValue destination;
            if (auto regionTag = NDrive::GetRegionTag(finishLocation->GetCoord(), *server)) {
                auto regionKey = TStringBuilder() << "areas.region." << regionTag << ".name";
                destination["region"] = localization ? localization->GetLocalString(Locale, regionKey) : regionKey;
            }
            destination["point"] = NJson::ToJson(finishLocation->GetCoord());
            destination["short_text"] = geocoded;
            route.InsertValue("destinations", NJson::TArrayBuilder(std::move(destination)));
        }
        auto track = Tracks[sessionId];
        if (track.Initialized()) {
            auto eg = g.BuildEventGuard("WaitTrack");
            if (track.Wait(Get()->GetContext()->GetRequestDeadline()) && track.HasValue()) {
                NDrive::TTracksLinker::TResults trackValue = track.GetValue();
                trackValue = FilterTrack(std::move(trackValue));
                isSpeeding &= ::IsSpeedLimitExceeded(trackValue);
                auto trackReport = NDrive::GetAnalyzerReport<NJson::TJsonValue>(trackValue, TrackFormat);
                route.InsertValue("track", std::move(trackReport));
                trackProjector = NDrive::TTrackProjector(trackValue);
            } else {
                eg.AddEvent(NJson::TMapBuilder
                    ("event", "GetTrackError")
                    ("async_track", NJson::ToJson(track))
                );
            }
        }

        auto rawTrack = RawTracks[sessionId];
        if (rawTrack.Initialized()) {
            auto eg = g.BuildEventGuard("WaitTrack");
            if (rawTrack.Wait(Get()->GetContext()->GetRequestDeadline()) && rawTrack.HasValue()) {
                const auto& rawTrackValue = rawTrack.GetValue();
                const auto result = NDrive::ToGeoJsonFormat(rawTrackValue);
                route.InsertValue("track", std::move(result));
            } else {
                eg.AddEvent(NJson::TMapBuilder
                    ("event", "GetTrackError")
                    ("async_track", NJson::ToJson(rawTrack))
                );
            }
        }

        if (auto distance = session.GetDistance()) {
            route.InsertValue("distance", *distance);
        }
    }

    NJson::TJsonValue user = NJson::JSON_NULL;
    if (auto u = Users.find(session.GetUserId()); u != Users.end()) {
        user = u->second.GetReport(NUserReport::ReportPublicId & permissions->GetUserReportTraits(), true);
    }

    NJson::TJsonValue vehicle = NJson::JSON_NULL;
    if (Fetcher && !Fetcher->BuildOneCarInfoReport(session.GetObjectId(), permissions->GetFilterActions(), vehicle)) {
        g.AddEvent(NJson::TMapBuilder
            ("event", "GetVehicleReportFailure")
            ("object_id", session.GetObjectId())
        );
    }
    if (vehicle.IsDefined()) {
        vehicle["car_number"] = vehicle["number"];
    }

    auto duration = session.GetLastTS() - session.GetStartTS();
    bool cancelled = fullCompiledRiding && !session.IsActive()
        ? fullCompiledRiding->OptionalRidingDuration().GetOrElse(TDuration::Zero()) == TDuration::Zero()
        : false;

    NJson::TJsonValue aggressive = NJson::JSON_NULL;
    NJson::TJsonValue aggressiveEvents = NJson::JSON_NULL;
    auto aggressiveEventsIt = RideAggressions.find(session.GetSessionId());
    bool isRealtimeAggressive = false;
    if (aggressiveEventsIt != RideAggressions.end()) {
        aggressive = true;
        aggressiveEvents = NJson::JSON_ARRAY;
        for (auto&& event : aggressiveEventsIt->second) {
            aggressiveEvents.AppendValue(event.GetReport(Get()->GetContext()->GetRequestDeadline(), trackProjector));
        }
    } else if (auto i = RealtimeRideAggressions.find(session.GetSessionId()); i != RealtimeRideAggressions.end()) {
        aggressive = true;
        aggressiveEvents = NJson::JSON_ARRAY;
        for (auto&& event : i->second) {
            aggressiveEvents.AppendValue(event.GetReport(Get()->GetContext()->GetRequestDeadline(), trackProjector));
        }
        isRealtimeAggressive = true;
    } else if (AggressionHelper) {
        aggressive = false;
    }

    NJson::TJsonValue result;
    result.InsertValue("cancelled", cancelled);
    result.InsertValue("created_at", session.GetStartTS().ToStringUpToSeconds());
    result.InsertValue("finished_at", session.GetLastTS().ToStringUpToSeconds());
    result.InsertValue("is_active", session.IsActive());
    result.InsertValue("is_speeding", isSpeeding);
    result.InsertValue("order_id", sessionId);
    if (auto status = session.GetStage()) {
        auto p = StatusRemap.find(status);
        const auto& remapped = p != StatusRemap.end() ? p->second : status;
        const auto& localizationKey = "offers.status." + remapped + ".title";
        result.InsertValue("status", remapped);
        result.InsertValue("localized_status", localization ? localization->GetLocalString(Locale, localizationKey) : localizationKey);
    }

    if (aggressive.IsDefined() && traits & NDriveSession::ReportAggression) {
        result.InsertValue("is_aggressive", std::move(aggressive));
    }
    if (aggressiveEvents.IsDefined()) {
        result.InsertValue("aggressive_events", std::move(aggressiveEvents));
    }
    if (isRealtimeAggressive) {
        result.InsertValue("is_realtime_aggressive", isRealtimeAggressive);
    }
    if (detailed) {
        result.InsertValue("formatted_duration", localization ? localization->FormatDuration(Locale, duration) : ToString(duration));
    }
    if (bill.IsDefined()) {
        result.InsertValue("bill", std::move(bill));
    }
    if (fines.IsDefined()) {
        result.InsertValue("fines", std::move(fines));
    }
    if (finesTotal.IsDefined()) {
        result.InsertValue("fines_total", std::move(finesTotal));
    }
    if (offer.IsDefined()) {
        result.InsertValue("offer", std::move(offer));
    }
    if (route.IsDefined()) {
        result.InsertValue("route", std::move(route));
    }
    if (user.IsDefined()) {
        result.InsertValue("user", std::move(user));
    }
    if (vehicle.IsDefined()) {
        result.InsertValue("vehicle", std::move(vehicle));
    }
    if (payment.IsDefined()) {
        result.InsertValue("payment", std::move(payment));
    }
    if (payments.IsDefined()) {
        result.InsertValue("payments", std::move(payments));
    }
    return result;
}

template <class T>
NDrive::TTracksLinker::TResults TSessionProcessorTraits<T>::FilterTrack(NDrive::TTracksLinker::TResults&& results) const {
    auto filter = [this](const NDrive::TTracksLinker::TResult& result) {
        return HiddenTrackStatuses.contains(result.Track.Status);
    };
    auto it = std::remove_if(results.begin(), results.end(), filter);
    results.erase(it, results.end());
    return results;
}

template <class T>
bool TSessionProcessorTraits<T>::IsSpeedLimitExceeded(const THistoryRideObject& ride, double speedLimit /*= 0*/) const {
    {
        const auto& tags = ride.GetTraceTags();
        if (!tags) {
            return false;
        }
        for (auto&& tag : tags->GetTags()) {
            if (!tag) {
                continue;
            }
            if (SpeedingTagNames.contains(tag->GetName())) {
                auto data = tag.GetTagAs<TScoringTraceTag>();
                if (data && data->GetValue() >= speedLimit) {
                    return true;
                }
            }
        }
        return false;
    }
}

template <class T>
typename TSessionProcessorTraits<T>::TFilter TSessionProcessorTraits<T>::ConstructFilter(const TSet<TString>& filterNames) const {
    return [this, filterNames](const THistoryRideObject& ride) {
        if (!filterNames) {
            return true;
        }
        if (filterNames.contains("aggressive") && AggressionHelper && AggressionHelper->GetAggressiveTag(ride)) {
            if (AggressionHelper && AggressionHelper->GetAggressiveTag(ride)) {
                return true;
            }
            if (RealtimeAggressionHelper && RealtimeAggressionHelper->GetAggressiveTag(ride)) {
                return true;
            }
        }
        if (filterNames.contains("speeding") && IsSpeedLimitExceeded(ride)) {
            auto linked = Tracks.FindPtr(ride.GetSessionId());
            if (!linked || !linked->HasValue()) {
                return true;
            }
            if (::IsSpeedLimitExceeded(linked->GetValue())) {
                return true;
            }
        }
        if (filterNames.contains("hard_speeding") && IsSpeedLimitExceeded(ride, HardSpeedThreshold)) {
            auto linked = Tracks.FindPtr(ride.GetSessionId());
            if (!linked || !linked->HasValue()) {
                return true;
            }
            if (::IsHardSpeedLimitExceeded(linked->GetValue())) {
                return true;
            }
        }
        if (filterNames.contains("fined") && !ride.GetFines().empty()) {
            return true;
        }
        return false;
    };
}

template <class T>
void TSessionProcessorTraits<T>::PrefetchFullCompiledRiding(TArrayRef<THistoryRideObject> sessions, TJsonReport::TGuard& g) {
    auto eg = g.BuildEventGuard("PrefetchFullCompiledRiding");
    THistoryRideObject::FetchFullRiding(Get()->GetServer(), sessions);
}

template <class T>
void TSessionProcessorTraits<T>::PrefetchGeocodedLocations(TConstArrayRef<THistoryRideObject> sessions, TJsonReport::TGuard& g) {
    auto language = enum_cast<ELanguage>(Locale);
    auto geocoder = Get()->GetDriveApi()->HasGeocoderClient() ? &Get()->GetDriveApi()->GetGeocoderClient() : nullptr;
    if (!geocoder) {
        g.AddEvent("GeocoderMissing");
        return;
    }
    for (auto&& session : sessions) {
        auto sessionId = session.GetSessionId();
        auto start = session.GetStartLocation();
        if (start && geocoder) {
            Starts[sessionId] = geocoder->Decode(start->GetCoord(), language);
        }
        auto finish = session.GetLastLocation();
        if (finish && geocoder) {
            Finishes[sessionId] = geocoder->Decode(finish->GetCoord(), language);
        }
    }
}

template <class T>
void TSessionProcessorTraits<T>::PrefetchPayments(TConstArrayRef<THistoryRideObject> sessions, NDrive::TEntitySession& tx, TJsonReport::TGuard& g) {
    auto eg = g.BuildEventGuard("PrefetchPayments");
    const auto& billingManager = Get()->GetDriveApi()->GetBillingManager();
    const auto& paymentsManager = billingManager.GetPaymentsManager();
    auto userIds = TSet<TString>();
    for (auto&& session : sessions) {
        const auto& sessionId = session.GetSessionId();
        if (!paymentsManager.GetPayments(CachedPayments[sessionId], sessionId, tx)) {
            eg.AddEvent(NJson::TMapBuilder
                ("event", "GetPaymentsError")
                ("session_id", sessionId)
            );
        }
        userIds.insert(session.GetUserId());
    }
    for (auto&& userId : userIds) {
        auto permissions = Get()->GetDriveApi()->GetUserPermissions(userId, TUserPermissionsFeatures());
        if (!permissions) {
            continue;
        }
        auto methods = Get()->GetDriveApi()->GetUserPaymentMethodsSync(*permissions, *(Get()->GetServer()), true);
        auto userPaymentCards = TBillingManager::GetUserPaymentCards(methods.Get());
        if (userPaymentCards) {
            for (auto&& card : *userPaymentCards) {
                auto id = card.GetId();
                UserPaymentCards[std::move(id)] = std::move(card);
            }
        } else {
            eg.AddEvent(NJson::TMapBuilder
                ("event", "GetUserPaymentCardsError")
                ("user_id", userId)
            );
        }
    }
}

template <class T>
void TSessionProcessorTraits<T>::PrefetchTracks(TConstArrayRef<THistoryRideObject> sessions, TJsonReport::TGuard& g, bool needGeocodedLocation, bool useRawTracks) {
    if (const TString hiddenStatuses = Get()->template GetHandlerSetting<TString>("tracks.hidden_statuses").GetOrElse("")) {
        HiddenTrackStatuses.clear();
        for (TStringBuf status: StringSplitter(hiddenStatuses).Split(',').SkipEmpty()) {
            HiddenTrackStatuses.insert(NDrive::GetStatus(status));
        }
    }
    if (!TracksClient) {
        auto tracksApiName = Get()->template GetHandlerSetting<TString>("tracks_api").GetOrElse("drive_graph");
        TracksClient = CreateTrackClient(tracksApiName, *Get()->GetServer());
    }
    if (!SignalqTracksClient) {
        SignalqTracksClient = CreateCameraTrackClient(*Get()->GetServer());
    }
    if (!TracksLinker) {
        auto linkerApiName = Get()->template GetHandlerSetting<TString>("linker_api").GetOrElse("default-linker");
        auto linker = Get()->GetServer()->GetLinker(linkerApiName);
        if (!linker) {
            g.AddEvent(NJson::TMapBuilder
                ("event", "LinkerApiMissing")
                ("name", linkerApiName)
            );
        }
        NDrive::TTracksLinker::TOptions tracksLinkerOptions;
        auto speedThresholdsString = Get()->template GetHandlerSetting<TString>("analyzer.speed_thresholds");
        auto speedThresholdsJson = speedThresholdsString ? NJson::ToJson(NJson::JsonString(*speedThresholdsString)) : NJson::JSON_NULL;
        auto speedThresholds = NJson::TryFromJson<TSpeedThresholds>(speedThresholdsJson);
        if (speedThresholds) {
            tracksLinkerOptions.ViolationOptions.SpeedThresholds = *speedThresholds;
        }
        if (auto hardSpeedThreshold = Get()->template GetHandlerSetting<double>("analyzer.hard_speed_threshold")) {
            tracksLinkerOptions.ViolationOptions.HardSpeedThreshold = *hardSpeedThreshold;
            // Convert m/s to km/h.
            HardSpeedThreshold = *hardSpeedThreshold * 3.6;
        }
        TracksLinker = linker ? MakeHolder<NDrive::TTracksLinker>(linker, tracksLinkerOptions) : nullptr;
        YDBTracksClient = MakeAtomicShared<TYDBTracksClient>(*Get()->GetServer(), tracksLinkerOptions);
    }
    const auto ignoreLargeTracks = Get()->template GetValue<bool>(Get()->GetContext()->GetCgiParameters(), "ignore_large_tracks", false).GetOrElse(false);
    auto largeTrackDistanceThreshold = Get()->template GetHandlerSetting<double>("large_track_distance_threshold").GetOrElse(1500);
    auto largeTrackDurationThreshold = Get()->template GetHandlerSetting<TDuration>("large_track_duration_threshold").GetOrElse(TDuration::Days(7));
    if ((TracksClient || SignalqTracksClient) && (TracksLinker || useRawTracks)) {
        for (auto&& session : sessions) {
            if (ignoreLargeTracks) {
                auto duration = session.GetLastTS() - session.GetStartTS();
                if (duration > largeTrackDurationThreshold) {
                    g.AddEvent(NJson::TMapBuilder
                        ("event", "SkipLargeTrack")
                        ("type", "duration")
                        ("duration", duration.Seconds())
                    );
                    continue;
                }
                auto distance = session.GetDistance().GetOrElse(0);
                if (distance > largeTrackDistanceThreshold) {
                    g.AddEvent(NJson::TMapBuilder
                        ("event", "SkipLargeTrack")
                        ("type", "distance")
                        ("distance", distance)
                    );
                    continue;
                }
            }
            NThreading::TFuture<NDrive::TTracks> tracks;

            NDrive::TTrackQuery query;
            query.SessionId = session.GetSessionId();

            NDrive::TTracksLinker::TResults linkedTracks;
            auto useYDB = Get()->template GetHandlerSetting<bool>("tracks.use_ydb_client").GetOrElse(false);
            if (useYDB) {
                linkedTracks = Yensured(YDBTracksClient)->GetYDBTracks(query);
            }
            if (auto prelinkedRange = NDrive::TTrackOps::GetLinkedTracksRange(linkedTracks); prelinkedRange) {
                query.Since = prelinkedRange->To.GetRef();
            }

            if (!Get()->GetServer()->GetDriveAPI()->GetIMEI(session.GetObjectId())) {
                if (!SignalqTracksClient) {
                    g.AddEvent(NJson::TMapBuilder
                        ("event", "CantGetSignalqTrackClient")
                        ("message", "dissabledFromGvars")
                    );
                    continue;
                }
                tracks = SignalqTracksClient->GetTracks(query, Get()->GetContext()->GetRequestDeadline() - Now());
            } else {
                if (!TracksClient) {
                    g.AddEvent(NJson::TMapBuilder
                        ("event", "CantGetTracksClient")
                    );
                    continue;
                }
                query.Status = TrackStatus;

                tracks = TracksClient->GetTracks(query, Get()->GetContext()->GetRequestDeadline() - Now());
                if (useRawTracks) {
                    g.AddEvent("RawTracks");
                    RawTracks.emplace(query.SessionId, std::move(tracks));
                    continue;
                }
            }
            auto linked = TracksLinker->Link(std::move(tracks)).Apply([
                &server = *Get()->GetServer(),
                locale = Locale,
                deadline = Get()->GetContext()->GetRequestDeadline(),
                prelinked = std::move(linkedTracks),
                &needGeocodedLocation,
                helper = ViolationsHelper
            ](const NThreading::TFuture<NDrive::TTracksLinker::TResults> results) mutable -> NDrive::TTracksLinker::TResults {
                auto correctedTracks = results.GetValue();

                TSpeedLimitCorrectionAreaTag::CorrectSpeedLimitRanges(correctedTracks, server);
                if (needGeocodedLocation) {
                    helper->FetchGeocodedRangeCenters(correctedTracks, locale, deadline);
                }

                prelinked.insert(
                    prelinked.end(),
                    std::make_move_iterator(correctedTracks.begin()),
                    std::make_move_iterator(correctedTracks.end())
                );
                NDrive::TTrackOps::SortLinkedTracks(prelinked);

                return prelinked;
            });
            Tracks.emplace(query.SessionId, std::move(linked));
        }
    }
}

template <class T>
void TSessionProcessorTraits<T>::PrefetchUsers(TConstArrayRef<THistoryRideObject> sessions, NDrive::TEntitySession& tx, TJsonReport::TGuard& g) {
    auto eg = g.BuildEventGuard("PrefetchUsers");
    TSet<TString> userIds;
    for (auto&& session : sessions) {
        userIds.insert(session.GetUserId());
    }
    auto userData = Get()->GetDriveApi()->GetUsersData();
    auto users = Yensured(userData)->FetchInfo(userIds, tx);
    Users = std::move(users.MutableResult());
}

template <class T>
void TSessionProcessorTraits<T>::PrefetchAggressiveEvents(TArrayRef<THistoryRideObject> sessions, TJsonReport::TGuard& g) {
    if (!AggressionHelper && !RealtimeAggressionHelper) {
        return;
    }
    auto eg = g.BuildEventGuard("PrefetchAggression");
    for (auto&& session : sessions) {
        if (auto tag = AggressionHelper ? AggressionHelper->GetAggressiveTag(session) : nullptr) {
            RideAggressions[session.GetSessionId()] = AggressionHelper->GetAggressiveEvents(*tag, Locale);
        }
        if (auto tag = RealtimeAggressionHelper ? RealtimeAggressionHelper->GetAggressiveTag(session) : nullptr) {
            RealtimeRideAggressions[session.GetSessionId()] = RealtimeAggressionHelper->GetAggressiveEvents(*tag, Locale);
        }
    }
}

template <class T>
TMaybe<double> TSessionProcessorTraits<T>::GetRideAggressionScore(const THistoryRideObject& session) const {
    if (!AggressionHelper) {
        return Nothing();
    }
    const auto* tag = AggressionHelper->GetAggressiveTag(session);
    return tag ? MakeMaybe(tag->GetValue()) : Nothing();
}

template <class T>
void TSessionProcessorTraits<T>::PrefetchFineAttachments(TConstArrayRef<THistoryRideObject> sessions, NDrive::TEntitySession& tx, TJsonReport::TGuard& g) {
    auto eg = g.BuildEventGuard("PrefetchFineAttachments");
    if (!Get()->GetDriveApi()->HasFinesManager()) {
        eg.AddEvent(NJson::TMapBuilder
            ("event", "NoFineSettings")
        );
        return;
    }
    TSet<TString> fineIds;
    for (const auto& session : sessions) {
        for (const auto& fine : session.GetFines()) {
            fineIds.insert(fine.GetId());
        }
    }
    if (fineIds.empty()) {
        return;
    }
    const auto& finesManager = Get()->GetDriveApi()->GetFinesManager();
    auto attachments = finesManager.GetFineAttachments(fineIds, NDrive::NFine::TFinesManager::EFineAttachmentType::Photo, tx);
    if (!attachments) {
        tx.Check();
        return;
    }
    for (auto&& item : *attachments) {
        FineAttachments[item.GetFineId()].push_back(std::move(item));
    }
}

template <class T>
TString TSessionProcessorTraits<T>::GetGeocodedStart(const TString& sessionId, TJsonReport::TGuard& g) const {
    auto asyncGeocoded = Starts.find(sessionId);
    if (asyncGeocoded != Starts.end() && asyncGeocoded->second.Initialized()) {
        auto eg = g.BuildEventGuard("WaitStartGeocodedLocation");
        if (asyncGeocoded->second.Wait(Get()->GetContext()->GetRequestDeadline()) && asyncGeocoded->second.HasValue()) {
            return asyncGeocoded->second.GetValue().Title;
        } else {
            eg.AddEvent(NJson::TMapBuilder
                ("event", "GetStartGeocodedLocationError")
                ("async_geocoded", NJson::ToJson(asyncGeocoded->second))
            );
        }
    }
    return "";
}

template <class T>
TString TSessionProcessorTraits<T>::GetGeocodedFinish(const TString& sessionId, TJsonReport::TGuard& g) const {
    auto asyncGeocoded = Finishes.find(sessionId);
    if (asyncGeocoded != Finishes.end() && asyncGeocoded->second.Initialized()) {
        auto eg = g.BuildEventGuard("WaitFinishGeocodedLocation");
        if (asyncGeocoded->second.Wait(Get()->GetContext()->GetRequestDeadline()) && asyncGeocoded->second.HasValue()) {
            return asyncGeocoded->second.GetValue().Title;
        } else {
            eg.AddEvent(NJson::TMapBuilder
                ("event", "GetFinishGeocodedLocationError")
                ("async_geocoded", NJson::ToJson(asyncGeocoded->second))
            );
        }
    }
    return "";
}
