#include "violations_helper.h"

#include <drive/backend/database/drive_api.h>

#include <rtline/library/geometry/polyline.h>

#include <rtline/util/types/cast.h>

NDrive::TViolationsHelper::TViolationsHelper(const NDrive::IServer& server) {
    Geocoder = server.GetDriveAPI()->HasGeocoderClient() ? &server.GetDriveAPI()->GetGeocoderClient() : nullptr;
}

void NDrive::TViolationsHelper::FetchGeocodedRangeCenters(NDrive::TTracksLinker::TResults& results, ELocalization locale, TInstant deadline) const{
    TVector<TResponses> responses(results.size());
    for (size_t i = 0; i < results.size(); ++i) {
        responses[i] = std::move(GetTResponsesVector(results[i], locale));
    }
    for (size_t i = 0; i < results.size(); ++i) {
        ParseTResponsesVector(responses[i], results[i], deadline);
    }
}

void NDrive::TViolationsHelper::FetchGeocodedRangeCenter(NDrive::TTracksLinker::TResult& result, ELocalization locale, TInstant deadline) const {
    TResponses responses;
    responses = std::move(GetTResponsesVector(result, locale));
    ParseTResponsesVector(responses, result, deadline);
}

TVector<TVector<NThreading::TFuture<NDrive::TGeocoder::TResponse>>> NDrive::TViolationsHelper::GetTResponsesVector(NDrive::TTracksLinker::TResult& result, ELocalization locale) const {
    TResponses responses(result.Segments.size());
    for (size_t i = 0; i < result.Segments.size(); ++i) {
        responses[i] = std::move(GetTResponseVector(result.Segments[i].Processed, locale));
    }
    return responses;
}

TVector<NThreading::TFuture<NDrive::TGeocoder::TResponse>> NDrive::TViolationsHelper::GetTResponseVector(const TSpeedLimitRanges& ranges, ELocalization locale) const {
    TResponse response(ranges.size());
    for (size_t i = 0; i < ranges.size(); ++i) {
        response[i] = std::move(GetGeocodedCenter(ranges[i], locale));
    }
    return response;
}

NThreading::TFuture<NDrive::TGeocoder::TResponse> NDrive::TViolationsHelper::GetGeocodedCenter(const TSpeedLimitRange& range, ELocalization locale) const {
    if (Geocoder && range.IsSpeedLimitExceeded()) {
        TGeoPolyLine polyline(range.Points);
        auto center = polyline.GetCoordByLength(polyline.GetLength() / 2);
        return Geocoder->Decode(center, enum_cast<ELanguage>(locale));
    }
    return NThreading::MakeFuture(NDrive::TGeocoder::TResponse());
}

void NDrive::TViolationsHelper::ParseTResponsesVector(const TResponses& responses, NDrive::TTracksLinker::TResult& result, TInstant deadline) const {
    for (size_t i = 0; i < result.Segments.size(); ++i) {
        ParseTResponseVector(responses[i], result.Segments[i].Processed, deadline);
    }
}

void NDrive::TViolationsHelper::ParseTResponseVector(const TResponse& response, TSpeedLimitRanges& ranges, TInstant deadline) const {
    for (size_t i = 0; i < ranges.size(); ++i) {
        FetchGeocodedCenter(ranges[i], response[i], deadline);
    }
}

void NDrive::TViolationsHelper::FetchGeocodedCenter(TSpeedLimitRange& range, NThreading::TFuture<NDrive::TGeocoder::TResponse> geocodedResponse, TInstant deadline) const {
    if (geocodedResponse.Initialized() && geocodedResponse.Wait(deadline) && geocodedResponse.HasValue()) {
        range.GeocodedCenter = geocodedResponse.GetValue().Title;
    }
}
