#include "processor.h"

#include <drive/backend/settings/settings.h>
#include <drive/library/cpp/compression/simple.h>

void TSettingsRemoveProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    R_ENSURE(requestData["settings"].IsArray(), ConfigHttpStatus.SyntaxErrorStatus, "'settings' is not array");

    TVector<TString> settings;
    for (auto&& i : requestData["settings"].GetArraySafe()) {
        R_ENSURE(i["setting_key"].IsString(), ConfigHttpStatus.SyntaxErrorStatus, "incorrect settings list");
        settings.emplace_back(i["setting_key"].GetString());
        ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Remove, TAdministrativeAction::EEntity::Settings, settings.back());
    }
    R_ENSURE(Server->GetSettings().RemoveKeys(settings, permissions->GetUserId()), ConfigHttpStatus.UnknownErrorStatus, "cannot remove options");

    g.SetCode(HTTP_OK);
}

void TSettingsUpsertProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    R_ENSURE(requestData["settings"].IsArray(), ConfigHttpStatus.SyntaxErrorStatus, "'settings' is not array");
    const TString& commonPrefix = GetHandlerSettingDef<TString>("common_prefix", "");

    TVector<TSetting> settings;
    TSet<TString> keys;
    for (auto&& i : requestData["settings"].GetArraySafe()) {
        TSetting setting;
        R_ENSURE(setting.DeserializeFromJson(i), ConfigHttpStatus.SyntaxErrorStatus, "incorrect settings record");
        setting.SetKey(commonPrefix + setting.GetKey());
        keys.emplace(commonPrefix + setting.GetKey());
        settings.emplace_back(setting);
    }
    TSet<TString> existKeys;
    ReqCheckCondition(Server->GetSettings().HasValues(keys, existKeys, TInstant::Zero()), ConfigHttpStatus.UnknownErrorStatus, EDriveLocalizationCodes::InternalServerError);
    for (auto&& i : settings) {
        if (existKeys.contains(i.GetKey())) {
            ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::Settings, i.GetKey());
        } else {
            ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Add, TAdministrativeAction::EEntity::Settings, i.GetKey());
        }
    }

    R_ENSURE(Server->GetSettings().SetValues(settings, permissions->GetUserId()), ConfigHttpStatus.UnknownErrorStatus, "incorrect settings processing");

    g.SetCode(HTTP_OK);
}

void TSettingsProposeProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    const TString& comment = Context->GetCgiParameters().Get("comment");
    R_ENSURE(comment, ConfigHttpStatus.SyntaxErrorStatus, "comment must be present", EDriveSessionResult::IncorrectRequest);
    R_ENSURE(requestData["settings"].IsArray(), ConfigHttpStatus.SyntaxErrorStatus, "'settings' is not array");
    const TString& commonPrefix = GetHandlerSettingDef<TString>("common_prefix", "");

    TVector<TSetting> settings;
    TSet<TString> keys;
    for (auto&& i : requestData["settings"].GetArraySafe()) {
        TSetting setting;
        R_ENSURE(setting.DeserializeFromJson(i), ConfigHttpStatus.SyntaxErrorStatus, "incorrect settings record");
        setting.SetKey(commonPrefix + setting.GetKey());
        keys.emplace(commonPrefix + setting.GetKey());
        settings.emplace_back(setting);
    }
    TSet<TString> existKeys;
    ReqCheckCondition(Server->GetSettings().HasValues(keys, existKeys, TInstant::Zero()), ConfigHttpStatus.UnknownErrorStatus, EDriveLocalizationCodes::InternalServerError);
    for (const auto& setting : settings) {
        if (existKeys.contains(setting.GetKey())) {
            R_ENSURE(setting.HasRevision(), ConfigHttpStatus.UnknownErrorStatus, "existed key: " + setting.GetKey() + " without revision");
        }
    }
    auto tx = Server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
    for (auto&& setting : settings) {
        ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Propose, TAdministrativeAction::EEntity::Settings, setting.GetKey());
        TObjectProposition<TSetting> proposition(setting
            , Server->GetSettingsManager()->GetPropositions()->GetConfig().GetDefaultConfirmationsNeed()
        );
        proposition.SetDescription(comment);
        R_ENSURE(Server->GetSettingsManager()->GetPropositions()->Propose(proposition, permissions->GetUserId(), tx) != EPropositionAcceptance::Problems, {}, "can't propose", tx);
    }
    R_ENSURE(tx.Commit(), {}, "cannot Commit", tx);
    g.SetCode(HTTP_OK);
}

void TSettingsConfirmProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    const TString& comment = Context->GetCgiParameters().Get("comment");
    R_ENSURE(comment, ConfigHttpStatus.SyntaxErrorStatus, "comment must be present", EDriveSessionResult::IncorrectRequest);

    auto tx = BuildTx<NSQL::Writable>();

    const TSet<TString> propositionIds = MakeSet(GetStrings(requestData, "proposition_ids"));
    auto optionalReports = Server->GetSettingsManager()->GetPropositions()->Get(propositionIds, tx);
    R_ENSURE(optionalReports, {}, "cannot get propositions", tx);
    auto reports = *optionalReports;

    for (auto&& i : propositionIds) {
        auto it = reports.find(i);
        R_ENSURE(it != reports.end(), ConfigHttpStatus.UserErrorState, "incorrect propose_id " + i, EDriveSessionResult::IncorrectRequest);
        ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Confirm, TAdministrativeAction::EEntity::Settings, it->second.GetKey());
    }
    tx.SetComment(comment);

    for (auto&& i : propositionIds) {
        auto it = reports.find(i);
        const EPropositionAcceptance confirmResult = Server->GetSettingsManager()->GetPropositions()->Confirm(it->first, permissions->GetUserId(), tx);
        if (confirmResult == EPropositionAcceptance::ConfirmWaiting) {
        } else if (confirmResult == EPropositionAcceptance::ReadyForCommit) {
            const TSetting temp = static_cast<const TSetting>(it->second);
            R_ENSURE(temp && Server->GetSettingsManager()->SetValueWithRevision(temp, permissions->GetUserId(), tx), ConfigHttpStatus.UnknownErrorStatus, "incorrect upsert processing", tx);
        } else {
            tx.DoExceptionOnFail(ConfigHttpStatus);
        }
    }
    R_ENSURE(tx.Commit(), {}, "cannot Commit", tx);
    g.SetCode(HTTP_OK);
}

void TSettingsRejectProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    const TString& comment = Context->GetCgiParameters().Get("comment");
    R_ENSURE(comment, ConfigHttpStatus.SyntaxErrorStatus, "comment must being", EDriveSessionResult::IncorrectRequest);

    auto tx = BuildTx<NSQL::Writable>();

    const TSet<TString> propositionIds = MakeSet(GetStrings(requestData, "proposition_ids"));
    auto optionalReports = Server->GetSettingsManager()->GetPropositions()->Get(propositionIds, tx);
    R_ENSURE(optionalReports, {}, "cannot get propositions", tx);
    auto reports = *optionalReports;

    for (auto&& i : propositionIds) {
        auto it = reports.find(i);
        R_ENSURE(it != reports.end(), ConfigHttpStatus.UserErrorState, "incorrect propose_id " + i, EDriveSessionResult::IncorrectRequest);
    }

    tx.SetComment(comment);
    for (auto&& i : propositionIds) {
        auto it = reports.find(i);
        ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Reject, TAdministrativeAction::EEntity::Settings, it->second.GetKey());
        R_ENSURE(Server->GetSettingsManager()->GetPropositions()->Reject(it->first, permissions->GetUserId(), tx) == EPropositionAcceptance::Rejected
            , ConfigHttpStatus.UnknownErrorStatus
            , "incorrect reject processing", tx
        );
    }

    R_ENSURE(tx.Commit(), {}, "cannot Commit", tx);
    g.SetCode(HTTP_OK);
}

void TSettingsHistoryProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    TVector<TAtomicSharedPtr<TObjectEvent<TSetting>>> result;
    R_ENSURE(Server->GetSettings().GetHistory(TInstant::Zero(), result), ConfigHttpStatus.UnknownErrorStatus, "cannot_fetch_history");
    NJson::TJsonValue settingsHistory(NJson::JSON_ARRAY);
    for (auto it = result.rbegin(); it != result.rend(); ++it) {
        if (CheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::Settings, (*it)->GetKey())) {
            settingsHistory.AppendValue((*it)->SerializeToJson());
        }
    }
    g.MutableReport().AddReportElement("settings_history", std::move(settingsHistory));

    g.SetCode(HTTP_OK);
}

void TSettingsInfoProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    const TCgiParameters& cgi = Context->GetCgiParameters();
    const auto actuality = GetTimestamp(cgi, "actuality", Context->GetRequestStartTime());
    const auto prefix = GetString(cgi, "prefix", false);

    NJson::TJsonValue propositionsJson = NJson::JSON_ARRAY;
    TSet<TString> users;
    {
        auto tx = BuildTx<NSQL::ReadOnly>();

        TMaybe<TMap<TString, TReportObjectProposition<TSetting>>> optionalSettingsPropose;
        optionalSettingsPropose = Server->GetSettingsManager()->GetPropositions()->Get(tx);
        R_ENSURE(optionalSettingsPropose, {}, "cannot get propositions", tx);
        auto& settingsPropose = *optionalSettingsPropose;

        for (auto&& [idPropose, proposeReport] : settingsPropose) {
            if (prefix && !proposeReport.GetKey().StartsWith(prefix)) {
                continue;
            }
            propositionsJson.AppendValue(proposeReport.BuildJsonReport());
            users.emplace(proposeReport.GetPropositionAuthor());
        }
    }
    NJson::TJsonValue usersJson = NJson::JSON_MAP;
    {
        auto userData = DriveApi->GetUsersData()->FetchInfo(users);
        for (auto&& [_, i] : userData) {
            usersJson.InsertValue(i.GetUserId(), i.GetReport(permissions->GetUserReportTraits()));
        }
    }
    NJson::TJsonValue settingsJson(NJson::JSON_ARRAY);
    {
        TVector<TSetting> settings;
        R_ENSURE(Server->GetSettings().GetAllSettings(settings, actuality), ConfigHttpStatus.UnknownErrorStatus, "cannot_fetch_settings");
        for (auto&& i : settings) {
            if (prefix && !i.GetKey().StartsWith(prefix)) {
                continue;
            }
            if (CheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::Settings, i.GetKey())) {
                settingsJson.AppendValue(i.SerializeToJson());
            }
        }
    }
    g.MutableReport().AddReportElement("settings", std::move(settingsJson));
    g.MutableReport().AddReportElement("propositions", std::move(propositionsJson));
    g.MutableReport().AddReportElement("users", std::move(usersJson));
    g.SetCode(HTTP_OK);
}

void TClientSettingsInfoProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    const TCgiParameters& cgi = Context->GetCgiParameters();
    TVector<TSetting> settings;
    R_ENSURE(Server->GetSettings().GetAllSettings(settings, TInstant::Zero()), ConfigHttpStatus.UnknownErrorStatus, "cannot_fetch_settings");
    const ILocalization* localization = Server->GetLocalization();
    NJson::TJsonValue errorsJson = NJson::JSON_NULL;
    NJson::TJsonValue optionsJson = NJson::JSON_MAP;
    const bool defaultNeedTreeStructure = GetHandlerSetting<bool>("need_tree").GetOrElse(true);
    const bool needTreeStructure = GetValue<bool>(cgi, "need_tree", false).GetOrElse(defaultNeedTreeStructure);
    const auto keys = MakeSet(GetStrings(cgi, "key", false));
    const auto requiredPrefix = GetString(cgi, "prefix", false);
    const auto locale = GetLocale();
    for (auto&& i : settings) {
        if (!CheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::Settings, i.GetKey())) {
            continue;
        }
        const TStringBuf key = i.GetKey();
        const TString& prefix = Config.GetCommonPrefix() ? Config.GetCommonPrefix() : GetHandlerSettingDef<TString>("common_prefix", "");
        constexpr TStringBuf decompressSuffix = ".decompress";
        constexpr TStringBuf typeNameSuffix = ".type";
        const bool propertie =
            key.EndsWith(decompressSuffix) ||
            key.EndsWith(typeNameSuffix);
        if (key.StartsWith(prefix) && !propertie) {
            TStringBuf name = key.SubStr(prefix.size());
            if (!keys.empty() && !keys.contains(name)) {
                continue;
            }
            if (requiredPrefix && !name.StartsWith(requiredPrefix)) {
                continue;
            }

            TString value = i.GetValue();
            if (!permissions->GetSettings().GetValueStr(key, value)) {
                g.AddEvent(NJson::TMapBuilder
                    ("event", "BadGetValueStr")
                    ("key", key)
                );
            }

            auto decompress = Server->GetSettings().GetValue<bool>(i.GetKey() + decompressSuffix).GetOrElse(false);
            if (decompress) try {
                value = NDrive::Decompress(value).GetRef();
            } catch (const std::exception& e) {
                errorsJson[key] = FormatExc(e);
            }

            NJson::TJsonValue jsonValue;
            if (!NJson::ReadJsonFastTree(value, &jsonValue)) {
                jsonValue = value;
            }
            localization->ApplyResourcesForJson(jsonValue, locale);

            auto type = Server->GetSettings().GetValue<NJson::EJsonValueType>(i.GetKey() + typeNameSuffix).GetOrElse(NJson::JSON_UNDEFINED);
            switch (type) {
            case NJson::JSON_STRING:
                if (needTreeStructure) {
                    optionsJson.SetValueByPath(name, jsonValue.GetStringRobust());
                } else {
                    optionsJson.InsertValue(name, jsonValue.GetStringRobust());
                }
                break;
            default:
                if (needTreeStructure) {
                    optionsJson.SetValueByPath(name, std::move(jsonValue));
                } else {
                    optionsJson.InsertValue(name, std::move(jsonValue));
                }
                break;
            }
        }
    }
    if (errorsJson.IsDefined()) {
        g.MutableReport().AddReportElement("errors", std::move(errorsJson));
    }
    g.MutableReport().AddReportElement("options", std::move(optionsJson));
    g.SetCode(HTTP_OK);
}
