#include "processor.h"

#include <drive/backend/processors/leasing/get_signals.h>
#include <drive/backend/processors/leasing/session_process.h>

#include <drive/backend/cars/hardware.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/data/support_tags.h>
#include <drive/backend/database/history/cache.h>
#include <drive/backend/database/transaction/tx.h>
#include <drive/backend/offers/offers/standart.h>
#include <drive/backend/roles/permissions.h>
#include <drive/backend/sessions/matcher/session_matcher.h>
#include <drive/backend/signalq/signals/helpers.h>
#include <drive/backend/signalq/signals/tag.h>
#include <drive/backend/tags/tags.h>
#include <drive/backend/tags/tags_manager.h>

#include <drive/library/cpp/taxi/signalq_drivematics_api/client.h>
#include <drive/library/cpp/taxi/signalq_drivematics_api/definitions.h>
#include <drive/library/cpp/taxi/signalq_drivematics_api/utils.h>

#include <rtline/library/json/cast.h>
#include <rtline/library/json/field.h>
#include <rtline/library/json/parse.h>

#include <util/generic/map.h>
#include <util/generic/string.h>
#include <util/string/subst.h>

#include <algorithm>

namespace {
    struct TNotificationSettingsItem {
        TSet<TString> EventTypes;
        TVector<TString> CarTags;
        TVector<TString> NotifierIds;
    };

    void SendNotification(const NDrive::IServer* server, TJsonReport::TGuard& g,
                          const TVector<TNotificationSettingsItem>& notificationSettings, const TString& message,
                          const TString& eventType, const TTaggedObject& cachedObject) {
        for (const auto& [eventTypes, carTags, notifierIds] : notificationSettings) {
            if (!eventTypes.contains(eventType)) {
                continue;
            }
            bool cachedObjectNeedsNotification = false;
            for (const auto& carTag : carTags) {
                if (cachedObject.HasTag(carTag)) {
                    cachedObjectNeedsNotification = true;
                    break;
                }
            }
            if (!cachedObjectNeedsNotification) {
                continue;
            }

            for (const auto& notifierId: notifierIds) {
                const auto notifier = server->GetNotifier(notifierId);
                if (!notifier) {
                    g.AddEvent(NJson::TMapBuilder
                        ("event", "CantFindNotifier")
                        ("notifier_id", notifierId)
                    );
                    continue;
                }
                NDrive::INotifier::Notify(notifier, message);
            }
        }
    }

    TMaybe<TVector<TNotificationSettingsItem>> GetMaybeNotificationSettings(const NDrive::ISettingGetter& settings) {
        const auto notificationSettingsJson = settings.GetJsonValue("signalq.signalq_events.notification_settings");
        TVector<TNotificationSettingsItem> notificationSettings;

        if (!notificationSettingsJson.IsNull()) {
            if (!notificationSettingsJson.IsArray()) {
                ALERT_LOG <<  "Value of signalq.signalq_events.notification_settings is not array" << Endl;
                return NothingObject;
            }
            const auto& jsonArray = notificationSettingsJson.GetArray();
            notificationSettings.reserve(jsonArray.size());
            for (const auto& json : jsonArray) {
                if (!json.IsMap()) {
                    ALERT_LOG <<  "Values of signalq.signalq_events.notification_settings array are not maps" << Endl;
                    return NothingObject;
                }
                TNotificationSettingsItem item;
                auto jsonMap = json.GetMap();
                NJson::TryFromJson(jsonMap["event_types"], item.EventTypes);
                NJson::TryFromJson(jsonMap["car_tags"], item.CarTags);
                NJson::TryFromJson(jsonMap["notifier_ids"], item.NotifierIds);
                notificationSettings.push_back(std::move(item));
            }
        }
        return notificationSettings;
    }

    TMaybe<TMap<TString, TString>> GetMaybeEventTypesToNotificationMessages(const NDrive::ISettingGetter& settings) {
        const auto eventTypesToNotificationMessagesJson = settings.GetJsonValue("signalq.signalq_events.event_types_to_notification_messages");
        TMap<TString, TString> eventTypesToNotificationMessages;

        if (!eventTypesToNotificationMessagesJson.IsNull()) {
            if (!eventTypesToNotificationMessagesJson.IsMap()) {
                ALERT_LOG <<  "Value of signalq.signalq_events.event_types_to_notification_messages is not map" << Endl;
                return NothingObject;
            }
            const auto& jsonMap = eventTypesToNotificationMessagesJson.GetMap();
            for (const auto& [key, value] : jsonMap) {
                eventTypesToNotificationMessages[key] = value.GetString();
            }
        }
        return eventTypesToNotificationMessages;
    }

    TVector<TString> GetCarToPreapproveTags(const NDrive::ISettingGetter& settings) {
        const auto carToPreapproveTagsStr = settings.GetValue<TString>("signalq.signalq_events.car_to_preapprove_tags");
        TVector<TString> carToPreapproveTags;
        if (carToPreapproveTagsStr) {
            NJson::TJsonValue value;
            R_ENSURE(NJson::ReadJsonFastTree(*carToPreapproveTagsStr, &value), HTTP_INTERNAL_SERVER_ERROR, "Can't parse cars' to-preapprove tags json");
            R_ENSURE(NJson::ParseField(value, carToPreapproveTags), HTTP_INTERNAL_SERVER_ERROR, "Can't convert json into cars' to-preapprove tags");
        }
        return carToPreapproveTags;
    }
}

TMaybe<TString> StartSessionIfNotStarted(const NDrive::IServer* server,
                                        TJsonReport::TGuard& g,
                                        TUserPermissions::TPtr permissions,
                                        NDrive::TEntitySession& tx,
                                        const TString& carId,
                                        TInstant unixTimestamp) {
    const auto& settings = server->GetSettings();
    const auto enableSignalqSessions = settings.GetValue<bool>("signalq.signalq_session.enable");
    if (!enableSignalqSessions.GetOrElse(false)) {
        return {};
    }

    const auto driveApi = server->GetDriveAPI();
    const auto& tagManager = driveApi->GetTagsManager().GetDeviceTags();
    const auto cachedObject = tagManager.GetObject(carId);
    R_ENSURE(cachedObject, HTTP_INTERNAL_SERVER_ERROR, "cannot get cached object");

    const auto enableEngineSessionsTagName = settings.GetValue<TString>("telematics.engine_session.tag");
    const auto isEngineSession = enableEngineSessionsTagName && cachedObject->HasTag(*enableEngineSessionsTagName);
    if (isEngineSession) {
        return {};
    }

    const auto enableTaxiSessionsTagName = settings.GetValue<TString>("taxi.taxi_session.tag");
    const auto enableSignalqSessionsTagName = settings.GetValue<TString>("signalq.signalq_session.tag");
    const auto isSignalqSession = enableSignalqSessionsTagName && cachedObject->HasTag(*enableSignalqSessionsTagName);
    const auto isTaxiSession = enableTaxiSessionsTagName && cachedObject->HasTag(*enableTaxiSessionsTagName);
    if (!isSignalqSession && !isTaxiSession) {
        g.AddEvent("NeitherSignalqNorTaxiSession");
        return {};
    }

    auto contextSession = MakeAtomicShared<NDrivematics::TContext>(carId, cachedObject->GetFirstTagByClass<TChargableTag>());
    R_ENSURE(contextSession->GetChargableTag(), HTTP_INTERNAL_SERVER_ERROR, "cached object has no chargable tag");

    contextSession->SetChargableTag(
        NDrivematics::GetRestoredTag(carId, {
            TChargableTag::Reservation,
            TChargableTag::Riding,
            TChargableTag::Parking,
        }, tagManager, tx)
    );

    if (NDrivematics::IsOldTimestamp(contextSession, unixTimestamp, tagManager, tx)) {
        g.AddEvent(NJson::TMapBuilder
            ("event", "OldSignalFromCamera")
            ("last_update", NJson::ToJson(contextSession->GetLastUpdatedAt()))
            ("unix_timestamp", NJson::ToJson(unixTimestamp))
        );
        return {};
    }

    const auto offer = driveApi->GetCurrentCarOffer(carId);
    TString offerName;
    if (offer) {
        offerName = offer->GetName();
    }

    if (offerName.empty()) {
        NDrive::TEventLog::Log("EmptyOfferName", NJson::TMapBuilder
            ("car_id", carId)
            ("tag_id", contextSession->GetChargableTag().GetTagId())
        );
    } else if (offerName == settings.GetValue<TString>("telematics.engine_session.offer_name")) {
        g.AddEvent(NJson::TMapBuilder
            ("event", "SignalqSessionStarterProcessorDropEngineSession")
            ("car_id", carId)
            ("tag_id", contextSession->GetChargableTag().GetTagId())
        );
        const auto performerPermissions = driveApi->GetUserPermissions(contextSession->GetChargableTag()->GetPerformer());
        R_ENSURE(performerPermissions, {}, "performerPermissions are missing", tx);
        R_ENSURE(
            NDrivematics::SwitchToReservation(contextSession, performerPermissions, *server, tx),
            {},
            contextSession->GetErrorReport(),
            tx
        );
        contextSession->SetChargableTag(
            NDrivematics::GetRestoredTag(carId, {
                TChargableTag::Reservation,
                TChargableTag::Riding,
                TChargableTag::Parking,
            }, tagManager, tx)
        );

    }

    if (offerName == settings.GetValue<TString>("taxi.taxi_session.offer_name").GetOrElse("taxi_session")) {
        g.AddEvent(NJson::TMapBuilder
            ("event", "SignalqSessionStarterProcessorContinueWithTaxiOffer")
            ("car_id", carId)
            ("tag_id", contextSession->GetChargableTag().GetTagId())
        );
        if (contextSession->GetChargableTag()->GetPerformer() != permissions->GetUserId()) {
            if (!isTaxiSession && (contextSession->GetChargableTag()->GetName() == TChargableTag::Riding || contextSession->GetChargableTag()->GetName() == TChargableTag::Parking)) {
                const auto performerPermissions = driveApi->GetUserPermissions(contextSession->GetChargableTag()->GetPerformer());
                R_ENSURE(performerPermissions, {}, "performerPermissions are missing", tx);
                R_ENSURE(
                    NDrivematics::SwitchToReservation(contextSession, performerPermissions, *server, tx),
                    {},
                    contextSession->GetErrorReport(),
                    tx
                );
            }
        } else {
            g.AddEvent("NoMatchedUserPerformer");
        }
        return {};
    }

    if (!isSignalqSession) {
        g.AddEvent("NoSignalqSessionsTagName");
        return {};
    }

    if (contextSession->GetChargableTag()->GetName() == TChargableTag::Riding) {
        return {};
    }

    if (contextSession->GetChargableTag()->GetPerformer().empty()) {
        R_ENSURE(
            NDrivematics::CreateSession(contextSession, permissions, *server, tx, "signalq.signalq_session"),
            {},
            contextSession->GetErrorReport(),
            tx
        );
        return contextSession->GetSessionId();
    }

    if (contextSession->GetChargableTag()->GetPerformer() == permissions->GetUserId() && contextSession->GetChargableTag()->GetName() == TChargableTag::Parking) {
        R_ENSURE(
            NDrivematics::SwitchToRiding(contextSession, permissions, *server, tx),
            {},
            contextSession->GetErrorReport(),
            tx
        );
        const auto offer = driveApi->GetCurrentCarOffer(carId);
        return offer->GetOfferId();
    }

    return {};
}

void TSignalqSessionStarterProcessor::ProcessServiceRequest(TJsonReport::TGuard& g,
                                                            TUserPermissions::TPtr permissions,
                                                            const NJson::TJsonValue& requestData) {
    const auto serialNumber = GetString(requestData, "serial_number");
    const auto carId = DriveApi->GetCarAttachmentAssignments().GetAttachmentOwnerByServiceAppSlug(serialNumber);
    R_ENSURE(carId, HTTP_NOT_FOUND, "cannot find car by camera's serial number " << serialNumber);
    const auto optionalUnixTimestamp = GetTimestamp(requestData, "unix_timestamp", true);
    R_ENSURE(optionalUnixTimestamp, HTTP_BAD_REQUEST, "missing required field unix_timestamp");
    const auto unixTimestamp = *optionalUnixTimestamp;

    g.AddEvent(NJson::TMapBuilder
        ("event", "input")
        ("car_id", carId)
        ("serial_number", serialNumber)
        ("unix_timestamp", unixTimestamp.TimeT())
    );

    auto tx = BuildTx<NSQL::Writable | NSQL::Deferred | NSQL::RepeatableRead>();
    const auto changedSessionState = !!StartSessionIfNotStarted(Server, g, permissions, tx, carId, unixTimestamp);

    R_ENSURE(tx.Commit(), {}, "cannot Commit", tx);
    g.SetCode(changedSessionState ? HTTP_ACCEPTED : HTTP_OK);
}

void TSignalqSignalCreaterProcessor::ProcessServiceRequest(TJsonReport::TGuard& g,
                                                           TUserPermissions::TPtr permissions,
                                                           const NJson::TJsonValue& requestData) {
    NDrive::NSignalq::TV1SignalqSignalCreateRequestData requestEvents;
    R_ENSURE(NJson::TryFromJson(requestData, requestEvents), HTTP_BAD_REQUEST, "CantParseRequestData");

    auto tx = BuildTx<NSQL::Writable | NSQL::RepeatableRead | NSQL::Deferred>();
    const auto& settings = GetSettings();
    const auto& tagManager = DriveApi->GetTagsManager().GetDeviceTags();
    const auto& traceTagManager = DriveApi->GetTagsManager().GetTraceTags();
    const auto signalqEventTraceTagName = settings.GetValue<TString>("signalq.signalq_events.trace_tag").GetOrElse("signalq_event_trace_tag");
    const auto enableSignalqEventsTagName = settings.GetValue<TString>("signalq.signalq_events.enable_tag");
    const auto allowedfutureEventAtDelta = settings.GetValue<TDuration>("signalq.signalq_events.allowed_future_event_at_delta").GetOrElse(TDuration::Zero());
    const auto isSkippingFutureEvents = settings.GetValue<bool>("signalq.signalq_events.is_skipping_future_events").GetOrElse(false);
    const auto isSkippingEventNonExistingCar = settings.GetValue<bool>("signalq.signalq_events.is_skipping_event_non_existing_car").GetOrElse(false);
    const auto enableEngineSessionsTagName = settings.GetValue<TString>("telematics.engine_session.tag");
    const auto enableSignalqSessionsTagName = settings.GetValue<TString>("signalq.signalq_session.tag");
    const auto actualRideGap = settings.GetValue<TDuration>("signalq.signalq_events.actual_ride_gap").GetOrElse(TDuration::Minutes(1));
    const auto maybeEventTypesToNotificationMessages = GetMaybeEventTypesToNotificationMessages(settings);
    const auto maybeNotificationSettings = maybeEventTypesToNotificationMessages ? GetMaybeNotificationSettings(settings) : NothingObject;
    const auto carToPreapproveTags = GetCarToPreapproveTags(settings);
    const auto eventToTagName = NDrive::NSignalq::GetEventToTagName();
    const auto signalqSupportPreapprovedTag = NDrive::NSignalq::GetSupportPreapprovedTagName();

    TMap<TString, TString> carIdToNewSessionIds;

    for (auto& requestEvent: requestEvents.MutableEvents()) {
        auto& serialNumber = requestEvent.MutableSerialNumber();
        auto& event = requestEvent.MutableEvent();

        if (event.GetAt() > Now() + allowedfutureEventAtDelta) {
            R_ENSURE(isSkippingFutureEvents, HTTP_BAD_REQUEST, "SignalqEventFromFuture_" << serialNumber);
            g.AddEvent(NJson::TMapBuilder
                ("event", "SignalqEventFromFutureSkipped")
                ("serial_number", serialNumber)
                ("event_at", NJson::ToJson(event.GetAt()))
            );
            continue;
        }

        const auto carId = DriveApi->GetCarAttachmentAssignments().GetAttachmentOwnerByServiceAppSlug(serialNumber);
        if (!carId) {
            R_ENSURE(isSkippingEventNonExistingCar, HTTP_NOT_FOUND, "cannot find car by signalq serial number " << serialNumber);
            g.AddEvent(NJson::TMapBuilder
                ("event", "SignalqEventWithoutCarSkipped")
                ("serial_number", serialNumber)
            );
            continue;
        }
        auto cachedObject = tagManager.GetObject(carId);
        if (!cachedObject) {
            R_ENSURE(isSkippingEventNonExistingCar, HTTP_NOT_FOUND, "NoCachedObject" << carId);
            g.AddEvent(NJson::TMapBuilder
                ("event", "SignalqEventNoCarCachedObjectSkipped")
                ("serial_number", serialNumber)
            );
            continue;
        }

        const auto eventsEnabled = enableSignalqEventsTagName && cachedObject->HasTag(*enableSignalqEventsTagName);
        if (!eventsEnabled) {
            g.AddEvent(NJson::TMapBuilder
                ("event", "SignalqEventsNotEnabled")
                ("object_id", carId)
            );
            continue;
        }

        const auto isEngineSession = enableEngineSessionsTagName && cachedObject->HasTag(*enableEngineSessionsTagName);
        const auto isSignalqSession = enableSignalqSessionsTagName && cachedObject->HasTag(*enableSignalqSessionsTagName);
        const auto signalqCanStartSession = !isEngineSession && isSignalqSession && event.GetType() != "shutdown";
        const auto isActualEvent = event.GetAt() >= Now() - actualRideGap;

        TString sessionId;
        if (signalqCanStartSession && isActualEvent) {
            const auto offer = DriveApi->GetCurrentCarOffer(carId);
            if (offer == nullptr && carIdToNewSessionIds.find(carId) == carIdToNewSessionIds.end()) {
                auto newSessionId = StartSessionIfNotStarted(Server, g, permissions, tx, carId, Now());
                R_ENSURE(newSessionId, HTTP_INTERNAL_SERVER_ERROR, "Failed to start session for signal device  " << serialNumber);
                sessionId = *newSessionId;
                carIdToNewSessionIds[carId] = sessionId;
            } else {
                if (offer == nullptr) {
                    sessionId = carIdToNewSessionIds[carId];
                } else {
                    sessionId = offer->GetOfferId();
                }
            }
        } else {
            auto matcher = NDrive::NSession::TMatchingConstraintsGroup::Construct(Server, {}, { "violation_during_or_after_session" });
            R_ENSURE(matcher, HTTP_INTERNAL_SERVER_ERROR, "cannot create matcher", tx);
            const auto& matched = matcher->MatchSession(carId, event.GetAt());
            if (!matched) {
                g.AddEvent(NJson::TMapBuilder
                    ("event", "UnmatchedSignalqEvent")
                    ("timestamp", NJson::ToJson(event.GetAt()))
                );
                continue;
            } else {
                sessionId = matched->GetSessionId();
                if (!sessionId) {
                    g.AddEvent(NJson::TMapBuilder
                        ("event", "OrphanSignalqEvent")
                        ("matched", NJson::ToJson(matched))
                        ("timestamp", NJson::ToJson(event.GetAt()))
                    );
                    continue;
                }
            }
        }

        auto taggedSession = traceTagManager.RestoreObject(sessionId, tx);
        const auto tags = taggedSession->GetTagsByClass<TSignalqEventTraceTag>();
        bool sessionHasThisEvent = false;
        const auto reqSerialNumberEventId = NDrive::NSignalq::MakeSerialNumberEventId(serialNumber, event.GetId());
        for (const auto& tag: tags) {
            const auto signalTag = tag.GetTagAs<TSignalqEventTraceTag>();
            const auto dbSerialNumberEventId = signalTag->MakeSerialNumberEventId();
            if (reqSerialNumberEventId == dbSerialNumberEventId) {
                g.AddEvent(NJson::TMapBuilder
                    ("event", "SignalqEventDuplicate")
                    ("serial_number_event_id", NJson::ToJson(reqSerialNumberEventId))
                );
                sessionHasThisEvent = true;
                break;
            }
        }
        if (sessionHasThisEvent) {
            continue;
        }

        bool preapproveRequired = false;
        TString tagName;
        if (auto it = eventToTagName.find(event.GetType()); it != eventToTagName.end()) {
            tagName = [&] {
                for (const auto& carToPreapproveTag : carToPreapproveTags) {
                    if (cachedObject->HasTag(carToPreapproveTag)) {
                        preapproveRequired = true;
                        return signalqSupportPreapprovedTag;
                    }
                }
                return it->second;
            }();
        } else {
            tagName = signalqEventTraceTagName;
        }

        auto tag = DriveApi->GetTagsManager().GetTagsMeta().CreateTag(tagName);
        if (!tag && tagName != signalqEventTraceTagName) {
            ALERT_LOG << "Cant create SignalqEventTraceTag: " << tagName << ". Will create base tag" << Endl;
            tag = DriveApi->GetTagsManager().GetTagsMeta().CreateTag(signalqEventTraceTagName);
        }
        R_ENSURE(tag, HTTP_INTERNAL_SERVER_ERROR, "cannot CreateTag " << signalqEventTraceTagName, tx);
        auto impl = std::dynamic_pointer_cast<TSignalqEventTraceTag>(tag);
        R_ENSURE(impl, HTTP_INTERNAL_SERVER_ERROR, "cannot Cast " << tag->GetName() << " as SignalqEventTraceTag", tx);
        const auto eventType = event.GetType();
        impl->SetEvent(std::move(event));
        impl->SetSerialNumber(std::move(serialNumber));
        if (preapproveRequired) {
            impl->SetVisible(false);
        }

        const auto added = DriveApi->GetTagsManager().GetTraceTags().AddTag(tag, permissions->GetUserId(), sessionId, Server, tx);
        R_ENSURE(added && !added->empty(), HTTP_INTERNAL_SERVER_ERROR, "cannot Add SignalqEventTraceTag", tx);
        if (maybeNotificationSettings) {
            const auto messageIt = maybeEventTypesToNotificationMessages->find(eventType);
            if (messageIt != maybeEventTypesToNotificationMessages->end()) {
                auto message = messageIt->second;
                SubstGlobal(message, "_SignalId_", added->front().GetTagId());
                SendNotification(Server, g, *maybeNotificationSettings, message, eventType, *cachedObject);
            }
        }
    }
    R_ENSURE(tx.Commit(), HTTP_INTERNAL_SERVER_ERROR, "cannot Commit", tx);
    g.SetCode(HTTP_OK);
}


void TSignalqNotifySupport::ProcessServiceRequest(TJsonReport::TGuard& g,
                                                  TUserPermissions::TPtr permissions,
                                                  const NJson::TJsonValue& requestData) {
    TSet<TString> vinsForFetch;
    for (const auto& item: requestData["items"].GetArray()) {
        vinsForFetch.insert(ToUpperUTF8(StripString(item["vin"].GetString())));
    }
    auto tx = BuildTx<NSQL::Writable>();
    auto cars = DriveApi->GetCarManager().FetchCarsByVIN(vinsForFetch, tx);
    TMap<TString, TString> carIdByVIN;

    for (auto&& car : cars) {
        carIdByVIN[car.second.GetVin()] = std::move(car.first);
    }

    uint32_t addedTagsAmount = 0;
    const auto matcher = NDrive::NSession::TMatchingConstraintsGroup::Construct(Server, {}, { "violation_during_or_after_session" });
    R_ENSURE(matcher, HTTP_INTERNAL_SERVER_ERROR, "cannot create matcher", tx);

    const auto tagName = GetSettings().GetValue<TString>("signalq.signalq_events.notify_support_tag").GetOrElse("leasing_support_notify");

    for (const auto& item: requestData["items"].GetArray()) {
        auto it = carIdByVIN.find(ToUpperUTF8(StripString(item["vin"].GetString())));
        if (it == carIdByVIN.end()) {
            continue;
        }
        if (!item.Has("event") || !item["event"].Has("at")) {
            continue;
        }
        auto timestamp = NJson::TryFromJson<TInstant>(item["event"]["at"]);
        if (!timestamp) {
            continue;
        }

        const auto matched = matcher->MatchSession(it->second, *timestamp);
        if (!matched) {
            continue;
        }

        const auto userId = matched->GetUserId();
        if (userId.empty()) {
            continue;
        }

        auto tag = DriveApi->GetTagsManager().GetTagsMeta().CreateTag(tagName);
        R_ENSURE(tag, HTTP_INTERNAL_SERVER_ERROR, "cannot CreateTag " << tagName, tx);
        auto impl = std::dynamic_pointer_cast<TSupportOutgoingCommunicationTag>(tag);
        R_ENSURE(impl, HTTP_INTERNAL_SERVER_ERROR, "cannot Cast " << tag->GetName() << " as TSupportOutgoingCommunicationTag", tx);
        impl->SetComment(item["event"].GetStringRobust());

        const auto added = DriveApi->GetTagsManager().GetUserTags().AddTag(std::move(tag), permissions->GetUserId(), userId, Server, tx);
        R_ENSURE(added, HTTP_INTERNAL_SERVER_ERROR, "cannot Add TSupportOutgoingCommunicationTag", tx);
        ++addedTagsAmount;
    }

    R_ENSURE(tx.Commit(), HTTP_INTERNAL_SERVER_ERROR, "cannot Commit", tx);

    g.MutableReport().AddReportElement("added_tags_amount", addedTagsAmount);
    g.SetCode(HTTP_OK);
}


void TSignalqFleetTraceTagResolutionSet::ProcessServiceRequest(TJsonReport::TGuard& g,
                                                  TUserPermissions::TPtr permissions,
                                                  const NJson::TJsonValue& requestData) {
    const auto resolution = GetValue<NDrive::NSignalq::EEventResolution>(requestData, "resolution", true).GetRef();

    const auto serialNumber = GetValue<TString>(requestData, "serial_number", true).GetRef();
    const auto carId = DriveApi->GetCarAttachmentAssignments().GetAttachmentOwnerByServiceAppSlug(serialNumber);
    R_ENSURE(carId, HTTP_NOT_FOUND, "cannot find car by camera's serial number " << serialNumber);

    const auto optionalEventAt = GetTimestamp(requestData, "event_at", true);
    R_ENSURE(optionalEventAt, HTTP_BAD_REQUEST, "missing required field event_at");
    const auto eventAt = *optionalEventAt;

    const auto matcher = NDrive::NSession::TMatchingConstraintsGroup::Construct(Server, {}, { "violation_during_or_after_session" });
    auto tx = BuildTx<NSQL::Writable | NSQL::Deferred>();
    R_ENSURE(matcher, HTTP_INTERNAL_SERVER_ERROR, "cannot create matcher", tx);
    const auto matched = matcher->MatchSession(carId, eventAt);
    R_ENSURE(matched, HTTP_INTERNAL_SERVER_ERROR, "cant match session", tx);

    const auto& sessionId = matched->GetSessionId();
    R_ENSURE(sessionId, HTTP_INTERNAL_SERVER_ERROR, "cant get session id", tx);

    const auto& traceTagsManager = DriveApi->GetTagsManager().GetTraceTags();
    auto taggedSession = traceTagsManager.RestoreObject(sessionId, tx);
    bool resolutionUpdated = false;
    auto tags = taggedSession->GetTagsByClass<TSignalqEventTraceTag>();
    const auto eventId = GetValue<TString>(requestData, "event_id", true).GetRef();

    for (auto& tag : tags) {
        auto signalTag = tag.MutableTagAs<TSignalqEventTraceTag>();
        if (signalTag && signalTag->GetEvent().GetId() == eventId) {
            NDrive::NSignalq::ProcessTagResolution(
                tag,
                resolution,
                Server,
                permissions,
                tx
            );

            resolutionUpdated = true;
            break;
        }
    }

    R_ENSURE(
        resolutionUpdated,
        HTTP_NOT_FOUND,
        "cant find event_id = " << eventId << " and serial_number = " << serialNumber << " and event_at = " << eventAt,
        tx
    );
    R_ENSURE(tx.Commit(), HTTP_INTERNAL_SERVER_ERROR, "cannot Commit", tx);
    g.SetCode(HTTP_OK);
}
