#include "processor.h"

void TStateFiltersRemoveProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Remove, TAdministrativeAction::EEntity::StateFilters);
    R_ENSURE(requestData["ids"].IsArray(), ConfigHttpStatus.SyntaxErrorStatus, "'ids' is not array");

    TVector<TString> ids;
    for (auto&& i : requestData["ids"].GetArraySafe()) {
        R_ENSURE(i.IsString(), ConfigHttpStatus.SyntaxErrorStatus, "incorrect object description");
        ids.emplace_back(i.GetString());
    }
    NDrive::TEntitySession session = BuildTx<NSQL::Writable>();
    if (!Server->GetDriveAPI()->GetStateFiltersDB()->RemoveKeys(ids, permissions->GetUserId(), session)) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }

    if (!session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }

    g.SetCode(HTTP_OK);
}

void TStateFiltersUpsertProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::StateFilters);
    TVector<TStateFilter> objects;
    const NJson::TJsonValue::TArray* arr;
    if (requestData["objects"].GetArrayPointer(&arr)) {
        for (auto&& i : *arr) {
            TStateFilter object;
            R_ENSURE(object.DeserializeFromJson(i), ConfigHttpStatus.SyntaxErrorStatus, "incorrect area record");
            objects.emplace_back(object);
        }
    } else {
        R_ENSURE(false, ConfigHttpStatus.SyntaxErrorStatus, "no arrays with objects");
    }

    NDrive::TEntitySession session = BuildTx<NSQL::Writable>();

    for (auto&& i : objects) {
        if (!Server->GetDriveAPI()->GetStateFiltersDB()->Upsert(i, permissions->GetUserId(), session)) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
    }

    if (!session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }

    g.SetCode(HTTP_OK);
}

void TStateFiltersHistoryProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::StateFilters);
    TVector<TAtomicSharedPtr<TObjectEvent<TStateFilter>>> result;
    R_ENSURE(Server->GetDriveAPI()->GetStateFiltersDB()->GetHistory(TInstant::Zero(), result, Context->GetRequestStartTime()), ConfigHttpStatus.UnknownErrorStatus, "cannot_fetch_history");
    NJson::TJsonValue historyJson(NJson::JSON_ARRAY);
    for (auto it = result.rbegin(); it != result.rend(); ++it) {
        historyJson.AppendValue((*it)->SerializeToJson());
    }
    g.MutableReport().AddReportElement("history", std::move(historyJson));

    g.SetCode(HTTP_OK);
}

void TStateFiltersInfoProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::StateFilters);
    TVector<TStateFilter> objects;
    ReqCheckCondition(Server->GetDriveAPI()->GetStateFiltersDB()->GetAllObjectsFromCache(objects, Context->GetRequestStartTime()), ConfigHttpStatus.UnknownErrorStatus, EDriveLocalizationCodes::InternalServerError);

    std::sort(objects.begin(), objects.end());
    NJson::TJsonValue objectsJson(NJson::JSON_ARRAY);
    for (auto&& i : objects) {
        objectsJson.AppendValue(i.SerializeToReport());
    }
    g.MutableReport().AddReportElement("objects", std::move(objectsJson));

    g.SetCode(HTTP_OK);
}
