#include "processor.h"

#include <drive/backend/billing/manager.h>
#include <drive/backend/data/user_tags.h>
#include <drive/backend/tags/tags_manager.h>

#include <drive/backend/processors/common_app/fetcher.h>

void TSubscriptionInfoProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    const auto userId = permissions->GetUserId();
    const auto locale = GetLocale();
    const auto localization = Server->GetLocalization();
    Y_UNUSED(requestData);

    auto tx = BuildTx<NSQL::ReadOnly>();
    TVector<TDBTag> tags;
    auto defaultTagName = Server->GetSettings().GetValue<TString>("subscription.default_tag_name").GetOrElse("");
    auto tagNames = Server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetRegisteredTagNames({ TUserSubscriptionTag::TypeName });
    R_ENSURE(Server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTagsRobust({ userId }, tagNames, tags, tx), HTTP_INTERNAL_SERVER_ERROR, "cannot restore tags with type " + TUserSubscriptionTag::TypeName + " . UserId: " + userId, tx);

    TUserSubscriptionTag activeTag;
    TString activeSubsriptionTagName;

    if (!tags.empty()) {
        R_ENSURE(tags.size() == 1, HTTP_INTERNAL_SERVER_ERROR, "User with id " + userId + " has more than 1 TUserSubscriptionTag.");
        activeTag = *Yensured(tags.front().MutableTagAs<TUserSubscriptionTag>());
        R_ENSURE(activeTag.GetStatus() != TUserSubscriptionTag::ESubscriptionStatus::Undefined, HTTP_INTERNAL_SERVER_ERROR, "Subscription tag has undefined status. Tag id: " + tags.front().GetTagId() + ". User Id: " + userId);
        activeSubsriptionTagName = activeTag.GetName();
    }

    if (!activeSubsriptionTagName) {
        activeSubsriptionTagName = defaultTagName;
    }
    NJson::TJsonValue report = Server->GetSettings().GetJsonValue("handlers.subscription");
    R_ENSURE(report.IsDefined(), HTTP_INTERNAL_SERVER_ERROR, "subscription is not defined");

    TVector<TDBTag> dbTags;
    auto availableTagNames = Server->GetSettings().GetValue<TString>("subscription.available_tags").GetOrElse("");
    auto splitedNames = MakeSet(StringSplitter(availableTagNames).SplitBySet(",").SkipEmpty().ToList<TString>());

    auto descriptionsPtr = Server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetTagsByType(TUserSubscriptionTag::TypeName);
    TVector<TUserSubscriptionTag::TDescription> descriptions;
    for (const auto& description : descriptionsPtr) {
        auto tagDescription = description->GetAs<TUserSubscriptionTag::TDescription>();
        if (tagDescription && splitedNames.contains(tagDescription->GetName())) {
            descriptions.push_back(*tagDescription);
        }
    }
    std::sort(descriptions.begin(), descriptions.end(), [](const TUserSubscriptionTag::TDescription& left, const TUserSubscriptionTag::TDescription& right){return left.GetPriority() < right.GetPriority();});

    NJson::TJsonValue subscriptions;
    NJson::TJsonValue status;
    for (const auto& description : descriptions) {
        NJson::TJsonValue subscription;

        auto type = description.GetName();
        subscription = NJson::ToJson(description.GetDescription());
        subscription.InsertValue("price_hr", localization->FormatPrice(locale, description.GetDescription().Price, { "units.short.rub" }));
        subscription.InsertValue("type", type);
        subscription.InsertValue("active", type == activeSubsriptionTagName ? true : false);
        if (type != defaultTagName) {
            subscription.InsertValue("more_info", NJson::ToJson(description.GetMoreInfo()));
        }
        subscriptions.AppendValue(subscription);

        if (type == activeSubsriptionTagName) {
            status = NJson::ToJson(description.GetStatus());
            status.InsertValue("type", type);
            report["autopayment"].InsertValue("is_visible", description.GetIsAutoPaymentVisible());
            if (type != defaultTagName) {
                report["info"].InsertValue("tag_id", tags.front().GetTagId());
                report["autopayment"].InsertValue("enabled", activeTag.GetAutoPayment());
                if (activeTag.GetAutoPayment()) {
                    report["autopayment"]["next_payment"]["ts"] = activeTag.GetSLAInstant().Seconds();
                }

                auto paymentStatus = activeTag.CheckPayment(tags.front(), userId, Server, tx);
                switch (activeTag.GetStatus()) {
                case TUserSubscriptionTag::ESubscriptionStatus::Active : {
                    auto activeStatus = localization->GetLocalString(locale, "subscription.status.active");
                    SubstGlobal(activeStatus, "_DATE_", localization->FormatMonthDayWithYear(locale, activeTag.GetSLAInstant()));
                    status.InsertValue("status",activeStatus);

                    if (description.GetMaxTaxiRides() > 0) {
                        NJson::TJsonValue taxi = NJson::ToJson(description.GetTaxiDescription());
                        auto ridesLeft = description.GetMaxTaxiRides() - activeTag.GetUsedTaxiRidesCount();
                        auto rides = TStringBuilder() << ToString(ridesLeft) << "/" << description.GetMaxTaxiRides();
                        taxi.InsertValue("rides", rides);
                        report["info"].InsertValue("taxi", taxi);
                    }
                    if (activeTag.GetAutoPayment() && paymentStatus && *paymentStatus == TUserSubscriptionTag::EPaymentStatus::Failed) {
                        report["info"]["error"].InsertValue("description", localization->GetLocalString(locale, "subscription.error.autoreneval"));
                        report["info"]["error"].InsertValue("icon", localization->GetLocalString(locale, "subscription.error.autoreneval.icon"));
                    }
                    break;
                }
                case TUserSubscriptionTag::ESubscriptionStatus::Pending : {
                    status.InsertValue("status", localization->GetLocalString(locale, "subscription.status.pending"));
                    if (paymentStatus) {
                        switch(*paymentStatus) {
                            case TUserSubscriptionTag::EPaymentStatus::Failed : {
                                report["info"]["error"].InsertValue("description", localization->GetLocalString(locale, "subscription.error.buy_failed"));
                                report["info"]["error"].InsertValue("icon", localization->GetLocalString(locale, "subscription.error.buy_failed.icon"));
                                break;
                            }
                            case TUserSubscriptionTag::EPaymentStatus::Processing : {
                                report["info"]["error"].InsertValue("description", localization->GetLocalString(locale, "subscription.error.processing"));
                                report["info"]["error"].InsertValue("icon", localization->GetLocalString(locale, "subscription.error.processing.icon"));
                                break;
                            }
                            default:
                                break;
                        }
                    }
                }
                default:
                    break;
                }
            } else {
                status.InsertValue("status", localization->GetLocalString(locale, "subscription.status.default"));
            }
        }
    }
    report.InsertValue("subscriptions", subscriptions);
    report["info"].InsertValue("status", status);

    auto bonuses = Server->GetDriveAPI()->GetBillingManager().GetAccountsManager().GetBonuses(permissions->GetUserId(), Context->GetRequestStartTime());
    report["info"]["balance"]["bonuses"].InsertValue("cash", bonuses);

    localization->ApplyResourcesForJson(report, locale);

    g.AddReportElement("subscription", std::move(report));
    g.SetCode(HTTP_OK);
}

void TSubscriptionUpdateProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    const auto userId = permissions->GetUserId();
    const auto& cgi = Context->GetCgiParameters();
    const auto& subscriptionName = GetString(cgi, "type", false);
    bool autoRenewal = GetValue<bool>(cgi, "auto_renewal").GetOrElse(false);

    auto tx = BuildTx<NSQL::Writable | NSQL::RepeatableRead>();
    TVector<TDBTag> tags;
    auto tagNames = Server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetRegisteredTagNames({ TUserSubscriptionTag::TypeName });
    R_ENSURE(Server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTagsRobust({ userId }, tagNames, tags, tx), HTTP_INTERNAL_SERVER_ERROR, "cannot restore tags with type " + TUserSubscriptionTag::TypeName + " . UserId: " + userId, tx);

    if (!subscriptionName) {
        R_ENSURE(!tags.empty(), HTTP_INTERNAL_SERVER_ERROR, "User with id " + userId + " doesn't have TUserSubscriptionTag.");
        R_ENSURE(tags.size() == 1, HTTP_INTERNAL_SERVER_ERROR, "User with id " + userId + " has more than 1 TUserSubscriptionTag.");
        auto userTag = tags.front().MutableTagAs<TUserSubscriptionTag>();
        R_ENSURE(userTag, HTTP_INTERNAL_SERVER_ERROR, "Coundn't get tag as TUserSubscriptionTag. Tag id: " + tags.front().GetTagId());
        R_ENSURE(userTag->GetStatus() != TUserSubscriptionTag::ESubscriptionStatus::Undefined, HTTP_INTERNAL_SERVER_ERROR, "Subscription tag has undefined status. Tag id: " + tags.front().GetTagId() + ". User Id: " + userId);

        userTag->SetAutoPayment(autoRenewal);
        R_ENSURE(Server->GetDriveAPI()->GetTagsManager().GetUserTags().UpdateTagData(tags.front(), userTag, userId, Server, tx), HTTP_INTERNAL_SERVER_ERROR, "Couldn't update tag  data. Tag id: " + tags.front().GetTagId() + ". User Id: " + userId, tx);
        R_ENSURE(tx.Commit(), HTTP_INTERNAL_SERVER_ERROR, "Couldn'commit tx. Error: " + tx.GetStringReport());
        g.SetCode(HTTP_OK);
        return;
    }

    R_ENSURE(tags.empty(), HTTP_BAD_REQUEST, "User with id " + userId + " already has at least one TUserSubscriptionTag.");
    R_ENSURE(tagNames.contains(subscriptionName), HTTP_BAD_REQUEST, "TUserSubscriptionTag with name " + subscriptionName + " doesn't exist.");
    NDrive::ITag::TPtr tag = Server->GetDriveAPI()->GetTagsManager().GetTagsMeta().CreateTag(subscriptionName);
    auto subscriptionTag = std::dynamic_pointer_cast<TUserSubscriptionTag>(tag);
    R_ENSURE(subscriptionTag, HTTP_INTERNAL_SERVER_ERROR, "Incorrect tag type: " +  ::ToString(TUserSubscriptionTag::TypeName));

    auto tagDescription = Server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetDescriptionByName(subscriptionTag->GetName());
    auto subscriptionDescription = Yensured(tagDescription)->GetAs<TUserSubscriptionTag::TDescription>();

    subscriptionTag->SetAutoPayment(autoRenewal);
    subscriptionTag->SetStatus(TUserSubscriptionTag::ESubscriptionStatus::Pending);
    subscriptionTag->SetSLAInstant(TInstant::Now() + subscriptionDescription->GetPendingDuration());

    auto dbTags = Server->GetDriveAPI()->GetTagsManager().GetUserTags().AddTag(subscriptionTag, userId, userId, Server, tx, EUniquePolicy::SkipIfExists);
    R_ENSURE(dbTags, HTTP_INTERNAL_SERVER_ERROR, "coundn't add TUserSubscriptionTag. UserId: " + userId, tx);
    R_ENSURE(dbTags->size() == 1, HTTP_INTERNAL_SERVER_ERROR, "Add tag TUserSubscriptionTag, Added more than 1 tag. UserId: " + userId);
    auto dbTag = dbTags->front();
    R_ENSURE(subscriptionTag->ApplyPayment(dbTag, userId, Server, tx), HTTP_INTERNAL_SERVER_ERROR, "ApplyPayment failed. UserId: " + userId + ". Tag id: " + dbTag.GetTagId(), tx);
    R_ENSURE(tx.Commit(), HTTP_INTERNAL_SERVER_ERROR, "Couldn'commit tx. Error: " + tx.GetStringReport());
    g.SetCode(HTTP_OK);
}

void TSubscriptionTaxiScreenProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    const auto userId = permissions->GetUserId();
    const auto& cgi = Context->GetCgiParameters();
    const auto& needGenerate = GetValue<bool>(cgi, "generate", false).GetOrElse(false);

    TVector<TDBTag> tags;
    auto tx = BuildTx<NSQL::ReadOnly>();
    auto tagNames = Server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetRegisteredTagNames({ TUserSubscriptionTag::TypeName });
    R_ENSURE(Server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTagsRobust({ userId }, tagNames, tags, tx), HTTP_INTERNAL_SERVER_ERROR, "cannot restore tags with type " + TUserSubscriptionTag::TypeName + ". UserId: " + userId, tx);

    TMaybe<TUserSubscriptionTag> activeTag;
    TDBTag dbTag;
    for (const auto& tag : tags) {
        auto userTag = tag.GetTagAs<TUserSubscriptionTag>();
        if (userTag && userTag->GetStatus() == TUserSubscriptionTag::ESubscriptionStatus::Active) {
            activeTag = *userTag;
            dbTag = tag;
        }
    }
    if (!activeTag) {
        if (!CheckCarsAvailability(g, permissions)) {
            NJson::TJsonValue report = Server->GetSettings().GetJsonValue("handlers.subscription.taxi_screen.no_rides");
            R_ENSURE(report.IsDefined(), HTTP_INTERNAL_SERVER_ERROR, "subscription taxi screen with no rides is not defined");
            report["available"] = false;
            g.MutableReport().SetExternalReport(std::move(report));
        }
        g.SetCode(HTTP_OK);
        return;
    }

    auto subscriptionTag = MakeAtomicShared<TUserSubscriptionTag>(*activeTag);
    auto tagDescription = Server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetDescriptionByName(subscriptionTag->GetName());
    auto subscriptionDescription = Yensured(tagDescription)->GetAs<TUserSubscriptionTag::TDescription>();

    auto maxTaxiRides = subscriptionDescription->GetMaxTaxiRides();
    return BuildReport(g, permissions, userId, dbTag, subscriptionTag, needGenerate, maxTaxiRides);
}

void TSubscriptionTaxiScreenProcessor::BuildReport(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const TString& userId, const TDBTag& subscriptionDBTag, TUserSubscriptionTag::TPtr subscriptionTag, bool needGenerate, ui32 maxTaxiRides) const {
    const auto locale = GetLocale();
    const auto localization = Server->GetLocalization();

    auto tx = BuildTx<NSQL::ReadOnly>();
    TDBTags promocodeTags;
    TString promocodeTagName = Server->GetSettings().GetValue<TString>("subscription.taxi_promocode_tag_name").GetOrElse("");
    R_ENSURE(promocodeTagName, HTTP_INTERNAL_SERVER_ERROR, "subscription taxi screen: cannot get promocode tag name");
    R_ENSURE(Server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTagsRobust({ userId }, { promocodeTagName }, promocodeTags, tx), HTTP_INTERNAL_SERVER_ERROR, "cannot restore tags with type " + TTaxiPromocodeUserTag::TypeName + ". UserId: " + userId, tx);

    for (const auto& tag : promocodeTags) {
        auto promocodeTag = tag.GetTagAs<TTaxiPromocodeUserTag>();
        if (promocodeTag && promocodeTag->GetSLAInstant() < TInstant::Now()) {
            NJson::TJsonValue report = Server->GetSettings().GetJsonValue("handlers.subscription.taxi_screen.has_promocode");
            R_ENSURE(report.IsDefined(), HTTP_INTERNAL_SERVER_ERROR, "subscription taxi screen with active promocode is not defined");
            report["available"] = true;
            report["generate"] = false;
            report["deadline"] = promocodeTag->GetSLAInstant().Seconds();
            report["server_time"] = TInstant::Now().Seconds();
            auto link = promocodeTag->GetDeepLink();
            R_ENSURE(link, HTTP_INTERNAL_SERVER_ERROR, "subscription taxi promocode tag doesn't have promocode coupon");
            report["button"]["link"] = link;
            g.MutableReport().SetExternalReport(std::move(report));
            g.SetCode(HTTP_OK);
            return;
        }
    }

    if (CheckCarsAvailability(g, permissions) || !Yensured(subscriptionTag)->IsTaxiAvailableByFilter(GetUserLocation(), Server, tx) || !subscriptionTag->HasTaxiRides(Server, tx)) {
        NJson::TJsonValue report = Server->GetSettings().GetJsonValue("handlers.subscription.taxi_screen.no_rides");
        R_ENSURE(report.IsDefined(), HTTP_INTERNAL_SERVER_ERROR, "subscription taxi screen with no rides is not defined");
        report["available"] = false;
        g.MutableReport().SetExternalReport(std::move(report));
        g.SetCode(HTTP_OK);
        return;
    }

    if (needGenerate) {
        return Generate(g, permissions, userId, subscriptionDBTag, subscriptionTag);
    }

    NJson::TJsonValue report = Server->GetSettings().GetJsonValue("handlers.subscription.taxi_screen");
    R_ENSURE(report.IsDefined(), HTTP_INTERNAL_SERVER_ERROR, "subscription active taxi screen is not defined");
    auto subtitle = localization->GetLocalString(locale, "subscription.taxi_screen.subtitle");
    SubstGlobal(subtitle, "_MaxTaxiRides_", ToString(maxTaxiRides));
    SubstGlobal(subtitle, "_RemainingTaxiRides_", ToString(maxTaxiRides - subscriptionTag->GetUsedTaxiRidesCount()));
    report["subtitle"] = subtitle;
    report["available"] = true;
    report["generate"] = true;
    g.MutableReport().SetExternalReport(std::move(report));
    g.SetCode(HTTP_OK);
    return;
}

void TSubscriptionTaxiScreenProcessor::Generate(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const TString& userId, const TDBTag& subscriptionDBTag, TUserSubscriptionTag::TPtr subscriptionTag) const {
    TString promocodeTagName = Server->GetSettings().GetValue<TString>("subscription.taxi_promocode_tag_name").GetOrElse("");
    auto tx = BuildTx<NSQL::Writable | NSQL::RepeatableRead>();
    R_ENSURE(promocodeTagName, HTTP_INTERNAL_SERVER_ERROR, "subscription taxi screen: cannot get promocode tag name", tx);

    NDrive::ITag::TPtr tag = Server->GetDriveAPI()->GetTagsManager().GetTagsMeta().CreateTag(promocodeTagName);
    auto promocodeTag = std::dynamic_pointer_cast<TTaxiPromocodeUserTag>(tag);
    R_ENSURE(promocodeTag, HTTP_INTERNAL_SERVER_ERROR, "Incorrect tag type: " +  ::ToString(TTaxiPromocodeUserTag::TypeName), tx);

    const auto& promocodeDescription = *Yensured(promocodeTag->GetDescriptionAs<TTaxiPromocodeUserTag::TDescription>(*Yensured(Server), tx));
    NDrive::TTaxiPromocodesClient::TRequestData data;
    data.Token = Yensured(subscriptionTag)->GetUniqueTaxiRideToken(subscriptionDBTag);
    data.YandexUid = permissions->GetUid();
    data.SeriesId = promocodeDescription.GetSeriesId();
    data.ExpireAt = TInstant::Now() + promocodeDescription.GetSLADurationRef();
    R_ENSURE(promocodeTag->GeneratePromocode(Server, data), HTTP_INTERNAL_SERVER_ERROR, "cannot generate promocode");

    auto dbTags = Server->GetDriveAPI()->GetTagsManager().GetUserTags().AddTag(promocodeTag, userId, userId, Server, tx, EUniquePolicy::SkipIfExists);
    R_ENSURE(dbTags, HTTP_INTERNAL_SERVER_ERROR, "couldn't add TTaxiPromocodeUserTag. UserId: " + userId, tx);
    R_ENSURE(dbTags->size() == 1, HTTP_INTERNAL_SERVER_ERROR, "Add tag TTaxiPromocodeUserTag, Added more than 1 tag. UserId: " + userId, tx);

    Yensured(subscriptionTag)->MutableUsedTaxiRidesCount()++;
    R_ENSURE(Server->GetDriveAPI()->GetTagsManager().GetUserTags().UpdateTagData(subscriptionDBTag, subscriptionTag, userId, Server, tx), HTTP_INTERNAL_SERVER_ERROR, "Couldn't increase used taxi rides count. Tag id: " + subscriptionDBTag.GetTagId() + ". User Id: " + userId, tx);
    R_ENSURE(tx.Commit(), HTTP_INTERNAL_SERVER_ERROR, "couldn't commit tx. Error: " + tx.GetStringReport(), tx);

    NJson::TJsonValue report = Server->GetSettings().GetJsonValue("handlers.subscription.taxi_screen.has_promocode");
    R_ENSURE(report.IsDefined(), HTTP_INTERNAL_SERVER_ERROR, "subscription taxi screen with active promocode is not defined", tx);
    report["available"] = true;
    report["generate"] = false;
    report["deadline"] = promocodeTag->GetSLAInstant().Seconds();
    report["server_time"] = TInstant::Now().Seconds();
    auto link = promocodeTag->GetDeepLink();
    R_ENSURE(link, HTTP_INTERNAL_SERVER_ERROR, "subscription taxi promocode tag doesn't have promocode coupon");
    report["button"]["link"] = link;
    g.MutableReport().SetExternalReport(std::move(report));
    g.SetCode(HTTP_OK);
    return;
}

bool TSubscriptionTaxiScreenProcessor::CheckCarsAvailability(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions) const {
    TMaybe<TGeoCoord> userLocation = GetUserLocation();
    if (!userLocation) {
        return true;
    }

    TEventsGuard egFetch(g.MutableReport(), "fetch");
    TCarsFetcher fetcher(*Server, NDeviceReport::NoTraits, Context->GetRequestDeadline(), NDriveModelReport::NoTraits);

    ::TRect<TGeoCoord> searchRect(*userLocation);
    auto bboxGrowDistance = GetHandlerSetting<double>("search_distance").GetOrElse(1000);
    searchRect.GrowDistance(bboxGrowDistance);
    fetcher.SetRect(searchRect);

    fetcher.SetCheckVisibility(GetHandlerSetting<bool>("fetcher.check_visiability").GetOrElse(true));
    fetcher.SetUserLocation(*userLocation);
    fetcher.SetIsRealtime(true);

    R_ENSURE(fetcher.FetchData(permissions, nullptr), ConfigHttpStatus.UnknownErrorStatus, "cannot fetch cars for taxi screen");
    auto carIds = fetcher.GetCarsIds();
    if (carIds.empty()) {
        return false;
    }
    return true;
}
