#include "processor.h"

#include <drive/backend/chat_robots/abstract.h>
#include <drive/backend/data/container_tag.h>
#include <drive/backend/data/notifications_tags.h>
#include <drive/backend/data/support_tags.h>
#include <drive/backend/device_snapshot/snapshots/tag.h>
#include <drive/backend/logging/events.h>
#include <drive/backend/support_center/manager.h>
#include <drive/backend/support_center/telephony/entities.h>
#include <drive/backend/support_center/telephony/manager.h>

#include <drive/library/cpp/taxi/chat/client.h>

#include <rtline/library/json/builder.h>

#include <util/generic/queue.h>
#include <util/string/cast.h>

class TCallTagData {
public:
    R_FIELD(TInstant, CallConnect, TInstant::Zero());
    R_FIELD(TInstant, CallEnter, TInstant::Max());
    R_FIELD(TInstant, CallExit, TInstant::Zero());
    R_OPTIONAL(size_t, HistoryPosition);
    R_READONLY(TString, CallUrl);
    R_FIELD(TString, MdsFile);
    R_FIELD(TString, Bucket);

public:
    TCallTagData() = default;
    TCallTagData(const TPhoneCall& call) {
        if (call.GetStartTS()) {
            SetCallEnter(call.GetStartTS());
        }
        if (call.GetAnswerTS()) {
            SetCallConnect(call.GetAnswerTS());
        }
        if (call.GetEndTS()) {
            SetCallExit(call.GetEndTS());
        }
    }

    bool HasCallDuration() const {
        return CallConnect != TInstant::Zero() && CallEnter != TInstant::Max() && CallExit != TInstant::Zero();
    }

    NJson::TJsonValue SerializeToJson() {
        NJson::TJsonValue callJson = NJson::JSON_MAP;
        callJson["url"] = CallUrl;
        callJson["connect"] = CallConnect.Seconds();
        callJson["enter"] = CallEnter == TInstant::Max() ? 0 : CallEnter.Seconds();
        callJson["exit"] = CallExit.Seconds();
        if (HasCallDuration() && Bucket && MdsFile) {
            NJson::InsertNonNull(callJson, "bucket", Bucket);
            NJson::InsertNonNull(callJson, "key", MdsFile);
        }
        return callJson;
    }

    template<class T>
    void SetCallUrlIfEmpty(const TConstDBTag& tag, const NDrive::IServer* server, const TString& callId) {
        if (!CallUrl.empty())
            return;
        const T* phoneCall = tag.GetTagAs<T>();
        if (phoneCall && !callId.empty()) {
            auto tagDescription = server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetDescriptionByName(tag->GetName());
            if (!tagDescription) {
                return;
            }
            auto description = dynamic_cast<const typename T::TDescription*>(tagDescription.Get());
            if (description) {
                CallUrl = description->GetUrlTemplate();
                SubstGlobal(CallUrl, "[call_id]", callId);
            }
        }
    }

    void SetCallUrlIfEmpty(const TConstDBTag& tag, const NDrive::IServer* server) {
        if (tag.Is<TSupportPhoneCallTag>()) {
            SetCallUrlIfEmpty<TSupportPhoneCallTag>(tag, server, tag.GetTagAs<TSupportPhoneCallTag>()->GetInternalCallId());
        } else if (tag.Is<TSupportOutgoingCallTag>()) {
            SetCallUrlIfEmpty<TSupportOutgoingCallTag>(tag, server, tag.GetTagAs<TSupportOutgoingCallTag>()->GetCallId());
        } else {
            ERROR_LOG << "Unimplemented url setting for '" << tag->GetName() << "'" << Endl;
        }
    }
};

bool UndeferChat(const TString& tagId, const TString& chatId, const NDrive::IServer* server, NDrive::TEntitySession& session) {
    TVector<TDBTag> dbTags;
    if (!server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags(TSet<TString>({tagId}), dbTags, session)) {
        session.AddErrorMessage("UndeferChat", "can't restore tags");
        return false;
    }
    if (dbTags.size() != 1) {
        session.SetErrorInfo("UndeferChat", "there is no such tag", EDriveSessionResult::IncorrectRequest);
        return false;
    }
    if (!IChatScriptAction::MatchingUndefer(dbTags.front(), chatId, TSupportChatTag::TypeName, false, server, session, true)) {
        session.AddErrorMessage("UndeferChat", "can't undefer tag");
        return false;
    }
    return true;
}

void TSupportCenterRequestsProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::ChatMessage);
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::ChatsList);

    const TCgiParameters& cgi = Context->GetCgiParameters();
    auto userId = GetString(cgi, "user_id", false);
    auto performerId = GetString(cgi, "performer_id", false);
    auto sinceStr = GetString(cgi, "since", false);
    auto untilStr = GetString(cgi, "until", false);
    auto supportRequestTypes = GetStrings(cgi, "tags", false);
    auto isReverse = GetString(cgi, "rev", false);
    auto limitStr = GetString(cgi, "limit", false);

    if (!supportRequestTypes) {
        supportRequestTypes.emplace_back("@" + TSupportOutgoingCallTag::TypeName);
        supportRequestTypes.emplace_back("@" + TSupportAITag::TypeName);
        supportRequestTypes.emplace_back("@user_support_call_tag");
        supportRequestTypes.emplace_back("@user_support_chat_tag");
        supportRequestTypes.emplace_back("@user_support_mail_tag");
        supportRequestTypes.emplace_back("@user_push");
    }

    TSet<TString> requestedTags;
    for (auto&& tag : supportRequestTypes) {
        if (tag.StartsWith("@")) {
            auto tagDescrs = Server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetTagsByType(tag.substr(1, tag.size() - 1));
            for (auto&& descr : tagDescrs) {
                requestedTags.insert(descr->GetName());
            }
        } else {
            requestedTags.insert(tag);
        }
    }

    size_t limit = 333;
    if (limitStr) {
        R_ENSURE(TryFromString(limitStr, limit), ConfigHttpStatus.SyntaxErrorStatus, "'limit' is specified and not int");
    }

    TInstant since = TInstant::Zero();
    if (sinceStr) {
        ui64 sinceInt;
        R_ENSURE(TryFromString(sinceStr, sinceInt), ConfigHttpStatus.SyntaxErrorStatus, "'since' is specified and not int");
        since = TInstant::Seconds(sinceInt);
    }

    TInstant until = TInstant::Max();
    if (untilStr) {
        ui64 untilInt;
        R_ENSURE(TryFromString(untilStr, untilInt), ConfigHttpStatus.SyntaxErrorStatus, "'until' is specified and not int");
        until = TInstant::Seconds(untilInt);
    }

    const auto& userTagsManager = Server->GetDriveAPI()->GetTagsManager().GetUserTags();
    auto events = TUserTagsHistoryManager::TOptionalEvents();
    auto session = BuildTx<NSQL::ReadOnly>();
    if (userId) {
        events = userTagsManager.GetEventsByObject(userId, session, 0, since);
    } else {
        auto queryOptions = TTagEventsManager::TQueryOptions(100 * limit, !!isReverse);
        queryOptions.SetTags(requestedTags);
        events = userTagsManager.GetEvents({since, until}, session, queryOptions);
    }
    R_ENSURE(events, {}, "cannot GetEvents", session);

    if (userId && !!isReverse) {
        Reverse(events->begin(), events->end());
    }

    TSet<TString> excludedTagIds;
    TSet<TString> hasCallsTagIds;
    TVector<TString> participantIds;
    TVector<NJson::TJsonValue> historyReport;
    TMap<TString, TCallTagData> callsData;
    size_t reportSize = 0;

    TSupportRequestCategorizationDB* categorizer = nullptr;

    TSet<TString> callTypeTags;
    {
        auto callTagDescriptions = Server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetTagsByType("user_support_call_tag");
        for (auto&& descr : callTagDescriptions) {
            callTypeTags.insert(descr->GetName());
        }
    }
    TSet<TString> outgoingCallTypeTags;
    {
        auto callTagDescriptions = Server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetTagsByType(TSupportOutgoingCallTag::TypeName);
        for (auto&& descr : callTagDescriptions) {
            outgoingCallTypeTags.insert(descr->GetName());
        }
    }
    TMap<TString, TVector<TWebPhoneCall>> webphoneCalls;
    auto manager = Server->GetSupportCenterManager();
    if (manager && Server->GetChatEngine()) {
        const auto& webphoneManager = manager->GetWebPhoneCallManager();
        auto options = NSQL::TQueryOptions().SetGenericCondition("start_ts", MakeRange<ui64>(since.Seconds(), until.Seconds())).SetOrderBy({"start_ts"}).SetDescending(true).SetLimit(limit);
        if (userId) {
            options.AddGenericCondition("user_id", userId);
        }
        auto supportSession = BuildChatSession(/* readOnly = */ true);
        auto calls = webphoneManager.GetObjects(supportSession, options);
        R_ENSURE(calls, ConfigHttpStatus.UnknownErrorStatus, ToString(EDriveLocalizationCodes::InternalServerError), supportSession);
        for (auto&& call : *calls) {
            webphoneCalls[call.GetUserId()].emplace_back(std::move(call));
        }
    }

    TSet<TString> supportAITags;
    TMap<TString, TSupportAICall> supportAICalls;
    TSet<TString> audioteleTags;
    TMaybe<TMap<TString, TString>> audioteleTrackKeys;
    if (manager) {
        auto getTagNames = [&](const TString& callCenter, const TString& defaultValue, TSet<TString>& callTagNames){
            TSet<TString> centerSettings;
            auto callCenterTagNames = GetHandlerSettingDef<TString>(callCenter + ".tag_names", defaultValue);
            StringSplitter(callCenterTagNames).SplitBySet(",").SkipEmpty().Collect(&centerSettings);
            CopyIf(centerSettings.begin(), centerSettings.end(), std::inserter(callTagNames, callTagNames.begin()), [&requestedTags](const auto& item) { return requestedTags.contains(item); });
        };
        getTagNames("audiotele", "cc_audiotele_incoming,cc_audiotele_outgoing", audioteleTags);
        if (!audioteleTags.empty()) {
            TSet<TString> callIds;
            for (auto&& event : *events) {
                if (!audioteleTags.contains(event->GetName())) {
                    continue;
                }
                if (auto tag = event.GetTagAs<TSupportPhoneCallTag>(); tag && tag->GetInternalCallId()) {
                    callIds.insert(tag->GetInternalCallId());
                }
            }
            auto audioteleTx = manager->GetAudioteleCallsManager().BuildSession(NSQL::ETransactionTraits::ReadOnly);
            audioteleTrackKeys = manager->GetAudioteleCallsManager().GetCallTracksKeys(audioteleTx, callIds, limit);
            R_ENSURE(audioteleTrackKeys, ConfigHttpStatus.UnknownErrorStatus, ToString(EDriveLocalizationCodes::InternalServerError), audioteleTx);
        }
        if (userId) {
            getTagNames("support_ai", "support_ai_speed_test", supportAITags);
            if (!supportAITags.empty()) {
                NSQL::TQueryOptions options(limit, /* descending = */ true);
                options.SetOrderBy({ "id" });
                options.AddGenericCondition("user_id", userId);
                options.SetGenericCondition("status", TSet<TString>{ ToString(TSupportAICall::EStatus::Servised), ToString(TSupportAICall::EStatus::NotServised) });
                auto supportAITx = manager->GetSupportAICallManager().GetHistoryManager().BuildSession(NSQL::ETransactionTraits::ReadOnly);
                auto optionalCalls = manager->GetSupportAICallManager().GetObjects(supportAITx, options);
                R_ENSURE(optionalCalls, ConfigHttpStatus.UnknownErrorStatus, ToString(EDriveLocalizationCodes::InternalServerError), supportAITx);
                Transform(optionalCalls->begin(), optionalCalls->end(), std::inserter(supportAICalls, supportAICalls.begin()), [](auto& call) {
                    TString externalId = call.GetExternalId();
                    return std::make_pair(externalId, std::move(call));
                });
            }
        }
    }
    const TString audioteleBucket = GetHandlerSettingDef<TString>("audiotele.bucket", "carsharing-support-audiotele");;
    const TString supportAIBucket = GetHandlerSettingDef<TString>("support_ai.bucket", "carsharing-support-ai");;

    if (Server->GetSupportCenterManager()) {
        categorizer = Server->GetSupportCenterManager()->GetSupportRequestCategorizer();
    }

    TMap<TString, NJson::TJsonValue> taskReports;
    for (auto&& historyItem : *events) {
        if (historyItem.GetHistoryInstant() > until) {
            continue;
        }
        const TConstDBTag& tag = historyItem;

        if (excludedTagIds.contains(tag.GetTagId())
            && !callTypeTags.contains(tag->GetName())
            && !supportAITags.contains(tag->GetName())
            && !outgoingCallTypeTags.contains(tag->GetName()))
        {
            continue;
        }
        if (hasCallsTagIds.contains(tag.GetTagId())) {
            continue;
        }
        if (userId && tag.GetObjectId() != userId) {
            continue;
        }
        if (!requestedTags.contains(tag->GetName())) {
            continue;
        }

        if (supportAITags.contains(tag->GetName())) {
            if (historyItem.GetHistoryAction() != EObjectHistoryAction::Remove) {
                continue;
            }
            if (auto snapshotPtr = historyItem->GetObjectSnapshot()) {
                if (auto tagSnapshot = dynamic_cast<TTagSnapshot*>(snapshotPtr.Get())) {
                    TVector<TString> callIds;
                    R_ENSURE(NJson::ParseField(tagSnapshot->GetMeta(), "calls", callIds), ConfigHttpStatus.UnknownErrorStatus, "fail to parse calls from snapshots");
                    TSupportAICall* callInfo = callIds.empty() ? nullptr : supportAICalls.FindPtr(callIds.front());
                    if (callInfo) {
                        TCallTagData call(*callInfo);
                        if (call.GetCallConnect() && call.GetCallExit()) {
                            call.SetBucket(supportAIBucket);
                            call.SetMdsFile(callInfo->GetExternalId() + ".wav");
                            callsData[tag.GetTagId()] = std::move(call);
                        }
                    }
                }
            }
        }

        TString taskId;
        if (outgoingCallTypeTags.contains(tag->GetName())) {
            auto historyAction = historyItem.GetHistoryAction();
            TCallTagData& call = callsData[tag.GetTagId()];
            if (historyAction == EObjectHistoryAction::Add) {
                call.SetCallConnect(historyItem.GetHistoryInstant());
            } else if (historyAction == EObjectHistoryAction::Remove) {
                call.SetCallExit(historyItem.GetHistoryInstant());
            }
            if (excludedTagIds.contains(tag.GetTagId())) {
                continue;
            }
            auto it = webphoneCalls.find(tag.GetObjectId());
            if (it != webphoneCalls.end() && manager) {
                auto socTag = tag.GetTagAs<TSupportOutgoingCallTag>();
                if (!socTag) {
                    ERROR_LOG << "Wrong tag type " << tag.GetTagId() << Endl;
                    continue;
                }
                taskId = socTag->GetCallTaskTag();
                const auto& webphoneManager = manager->GetWebPhoneCallManager();
                auto callIt = FindIf(it->second.begin(), it->second.end(), [socTag, &webphoneManager](auto& call) {
                    return webphoneManager.ConvertOriginCallId(call.GetId()) == socTag->GetCallId();
                });
                if (callIt != it->second.end()) {
                    TCallTagData call(*callIt);
                    if (call.GetCallConnect() && call.GetCallExit()) {
                        hasCallsTagIds.insert(tag.GetTagId());
                        call.SetCallUrlIfEmpty(tag, Server);
                        call.SetBucket(webphoneManager.GetBucketName());
                        call.SetMdsFile(webphoneManager.ConvertOriginCallId(callIt->GetId()) + "." + callIt->GetCodec());
                        callsData[tag.GetTagId()] = std::move(call);
                    }
                }
            }
        } else if (callTypeTags.contains(tag->GetName())) {
            auto historyAction = historyItem.GetHistoryAction();
            TCallTagData& call = callsData[tag.GetTagId()];
            if (historyAction == EObjectHistoryAction::Add) {
                call.SetCallConnect(historyItem.GetHistoryInstant());
            } else if (historyAction == EObjectHistoryAction::Remove) {
                call.SetCallExit(historyItem.GetHistoryInstant());
            } else if (historyAction == EObjectHistoryAction::SetTagPerformer || historyAction == EObjectHistoryAction::ForceTagPerformer) {
                call.SetCallEnter(std::min(historyItem.GetHistoryInstant(), call.GetCallEnter()));
            }
            call.SetCallUrlIfEmpty(tag, Server);
            if (audioteleTags.contains(tag->GetName())) {
                if (auto phoneTag = tag.GetTagAs<TSupportPhoneCallTag>()) {
                    auto audioteleData = audioteleTrackKeys ? audioteleTrackKeys->FindPtr(phoneTag->GetInternalCallId()) : nullptr;
                    if (audioteleData) {
                        call.SetMdsFile(*audioteleData);
                        call.SetBucket(audioteleBucket);
                    }
                }
            }
            if (call.HasHistoryPosition() && ((isReverse && historyAction == EObjectHistoryAction::Add) || (!isReverse && historyAction == EObjectHistoryAction::Remove))) {
                const auto& reportPosition = call.GetHistoryPositionRef();
                if (call.HasCallDuration()) {
                    historyReport[reportPosition].InsertValue("call", call.SerializeToJson());
                    callsData.erase(tag.GetTagId());
                }
            }
            if (excludedTagIds.contains(tag.GetTagId())) {
                continue;
            }
        }

        if (performerId && tag->GetPerformer() != performerId) {
            continue;
        }

        excludedTagIds.insert(tag.GetTagId());
        NJson::TJsonValue report = historyItem.BuildReportItem();
        if (categorizer) {
            auto cat = categorizer->GetActualCategorization(tag.GetTagId());
            if (!cat.Empty()) {
                report["categorization"] = cat.BuildReport();
            }
        }
        participantIds.push_back(tag.GetObjectId());
        if (tag->GetPerformer()) {
            participantIds.push_back(tag->GetPerformer());
        }

        if (taskId) {
            if (auto taskReportPtr = taskReports.FindPtr(taskId)) {
                report["task_tag"] = *taskReportPtr;
            } else {
                NJson::TJsonValue taskReport;
                auto taskEvents = userTagsManager.GetEventsByTag(taskId, session);
                R_ENSURE(taskEvents, ConfigHttpStatus.UnknownErrorStatus, ToString(EDriveLocalizationCodes::InternalServerError), session);
                if (!taskEvents->empty()) {
                    taskReport = taskEvents->front().BuildReportItem();
                    if (categorizer) {
                        auto cat = categorizer->GetActualCategorization(taskId);
                        if (!cat.Empty()) {
                            taskReport["categorization"] = cat.BuildReport();
                        }
                    }
                }
                taskReports.emplace(taskId, taskReport);
                report["task_tag"] = taskReport;
            }
        }

        historyReport.push_back(std::move(report));
        auto call = callsData.find(tag.GetTagId());
        if (call != callsData.end()) {
            if (hasCallsTagIds.contains(tag.GetTagId())) {
                historyReport.back().InsertValue("call", call->second.SerializeToJson());
                callsData.erase(tag.GetTagId());
            } else {
                call->second.SetHistoryPosition(historyReport.size() - 1);
            }
        }

        ++reportSize;
        if (reportSize == limit) {
            break;
        }
    }

    for (auto&& [tagId, call] : callsData) {
        if (!call.HasHistoryPosition()) {
            continue;
        }
        if (call.GetCallExit() == TInstant::Zero()) {
            auto tagHistory = userTagsManager.GetEventsByTag(tagId, session, 0, until);
            R_ENSURE(tagHistory, ConfigHttpStatus.UnknownErrorStatus, "cannot acquire history for tag " << tagId, session);

            for (auto&& tagHistoryItem : *tagHistory) {
                if (tagHistoryItem.GetHistoryAction() == EObjectHistoryAction::Remove) {
                    call.SetCallExit(tagHistoryItem.GetHistoryInstant());
                }
                call.SetCallUrlIfEmpty(tagHistoryItem, Server);
            }
        }
        if (call.GetCallConnect() == TInstant::Zero() || call.GetCallEnter() == TInstant::Max()) {
            auto tagHistory = userTagsManager.GetEventsByTag(tagId, session);
            R_ENSURE(tagHistory, ConfigHttpStatus.UnknownErrorStatus, "cannot acquire history for tag " << tagId, session);

            for (auto&& tagHistoryItem : *tagHistory) {
                if (tagHistoryItem.GetHistoryAction() == EObjectHistoryAction::SetTagPerformer || tagHistoryItem.GetHistoryAction() == EObjectHistoryAction::ForceTagPerformer) {
                    call.SetCallEnter(std::min(tagHistoryItem.GetHistoryInstant(), call.GetCallEnter()));
                } else if (tagHistoryItem.GetHistoryAction() == EObjectHistoryAction::Add) {
                    call.SetCallConnect(tagHistoryItem.GetHistoryInstant());
                }
                call.SetCallUrlIfEmpty(tagHistoryItem, Server);
            }
        }
        const auto& reportPosition = call.GetHistoryPositionRef();
        historyReport[reportPosition].InsertValue("call", call.SerializeToJson());
    }

    NJson::TJsonValue usersReport;
    auto usersFetchResult = Server->GetDriveAPI()->GetUsersData()->FetchInfo(participantIds, session);
    for (auto&& userIt : usersFetchResult) {
        usersReport.AppendValue(userIt.second.GetChatReport());
    }

    NJson::TJsonValue historyReportJson;
    for (auto&& reportItem : historyReport) {
        historyReportJson.AppendValue(std::move(reportItem));
    }

    g.MutableReport().AddReportElement("tags", std::move(historyReportJson));
    g.MutableReport().AddReportElement("users", std::move(usersReport));
    g.SetCode(HTTP_OK);
}

void TSupportCenterDeferRequestProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    const TCgiParameters& cgi = Context->GetCgiParameters();
    auto tagId = requestData.Has("tag_id") ? GetString(requestData, "tag_id") : GetString(cgi, "tag_id");
    auto isFinishing = requestData.Has("is_finishing") ? GetString(requestData, "is_finishing") : GetString(cgi, "is_finishing", false);
    if (isFinishing) {
        R_ENSURE(!requestData.Has("defer_until"), ConfigHttpStatus.SyntaxErrorStatus, "can't close and defer chat at the same time");
    }

    TVector<TDBTag> dbTags;
    {
        auto session = BuildTx<NSQL::ReadOnly>();
        R_ENSURE(Server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags(TSet<TString>({tagId}), dbTags, session), ConfigHttpStatus.UnknownErrorStatus, "could not restore tags");
        R_ENSURE(dbTags.size() == 1, ConfigHttpStatus.SyntaxErrorStatus, "there is no such tag");
    }

    auto tagOriginal = dbTags.front();
    auto deferrableTag = tagOriginal.MutableTagAs<IDeferrableTag>();
    R_ENSURE(deferrableTag, ConfigHttpStatus.SyntaxErrorStatus, "tag type is not deferrable");
    TString comment = GetString(requestData, "comment", false);
    TString message = GetString(requestData, "message", false);
    TVector<NDrive::NChat::TMessage> messages;
    if (comment || message) {
        R_ENSURE(Server->GetChatEngine(), ConfigHttpStatus.ServiceUnavailable, "ChatEngine is not configured");
    }
    auto supportChatTag = tagOriginal.MutableTagAs<TSupportChatTag>();
    bool ignoreExternal = GetValue<bool>(cgi, "ignore_external", false).GetOrElse(false);

    if (comment) {
        messages.push_back(NDrive::NChat::TMessage(comment, (ui32)NDrive::NChat::IMessage::EMessageTraits::StaffOnly, NDrive::NChat::IMessage::EMessageType::Plaintext));
    }
    if (message) {
        NDrive::NChat::TMessage chatMessage(message, 0, NDrive::NChat::IMessage::EMessageType::Plaintext);
        if (supportChatTag && supportChatTag->GetExternalChatInfo().GetId()) {
            chatMessage.SetExternalStatus(NDrive::NChat::TMessage::EExternalStatus::Undefined);
        }
        messages.push_back(std::move(chatMessage));
    }

    if (requestData.Has("defer_until")) {
        auto deferUntil = GetTimestamp(requestData, "defer_until", true).GetRef();
        R_ENSURE(deferUntil > Context->GetRequestStartTime(), ConfigHttpStatus.SyntaxErrorStatus, "defer_until is in the past");
        auto session = BuildTx<NSQL::Writable>();
        if (!deferrableTag->OnDeferUntil(tagOriginal, messages, deferUntil, *permissions, session, Server) || !session.Commit()) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
    } else if (isFinishing) {
        if (!ignoreExternal && supportChatTag && supportChatTag->GetExternalChatInfo().GetId()) {
            auto taxiChatClient = Server->GetTaxiChatClient();
            R_ENSURE(taxiChatClient, ConfigHttpStatus.SyntaxErrorStatus, "Taxi client not configured");
            TString closeMessage = GetHandlerSettingDef<TString>(supportChatTag->GetExternalChatInfo().GetProvider() + ".close_message", "");
            NThreading::TFuture<bool> replyFuture;
            if (closeMessage) {
                replyFuture = taxiChatClient->CloseChatWithMessage(supportChatTag->GetExternalChatInfo().GetId(), closeMessage);
            } else {
                replyFuture = taxiChatClient->CloseChat(supportChatTag->GetExternalChatInfo().GetId(), comment);
            }
            replyFuture.Wait();
            R_ENSURE(!replyFuture.HasException() && replyFuture.HasValue() && replyFuture.GetValue(), ConfigHttpStatus.SyntaxErrorStatus, replyFuture.HasException() ? NThreading::GetExceptionMessage(replyFuture) : "No value in reply from external service");
        }
        auto session = BuildTx<NSQL::Writable>();
        if (!deferrableTag->OnClose(tagOriginal, messages, *permissions, session, Server) || !session.Commit()) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
    } else {
        TString chatId, topic;
        IChatRobot::TPtr chatRobot;
        if (supportChatTag && supportChatTag->GetExternalChatInfo().GetId() && !ignoreExternal) {
            R_ENSURE(message, ConfigHttpStatus.SyntaxErrorStatus, "Message must have body to be deferred in external chat");
            IChatRobot::ParseTopicLink(supportChatTag->GetTopicLink(), chatId, topic);
            chatRobot = Server->GetChatRobot(chatId);
            R_ENSURE(chatRobot, ConfigHttpStatus.SyntaxErrorStatus, "Chat robot not configured");
        }
        {
            auto session = BuildTx<NSQL::Writable>();
            if (!deferrableTag->OnDefer(tagOriginal, messages, *permissions, session, Server) || !session.Commit()) {
                session.DoExceptionOnFail(ConfigHttpStatus);
            }
        }
        if (supportChatTag && supportChatTag->GetExternalChatInfo().GetId() && !ignoreExternal) {
            auto taxiChatClient = Server->GetTaxiChatClient();
            R_ENSURE(taxiChatClient, ConfigHttpStatus.SyntaxErrorStatus, "Taxi client not configured");

            auto replyFuture = taxiChatClient->DeferWithMessage(supportChatTag->GetExternalChatInfo().GetId(), message);
            replyFuture.Wait();
            chatRobot->Refresh(Now());
            auto session = BuildChatSession(false);
            TMaybe<NDrive::NChat::TMessageEvent> lastMessage;
            if (!chatRobot->GetLastMessage(tagOriginal.GetObjectId(), topic, session, 0, lastMessage, permissions->GetUserId())) {
                session.DoExceptionOnFail(ConfigHttpStatus);
            }
            if (replyFuture.HasException() || !replyFuture.HasValue() || !replyFuture.GetValue()) {
                TString error = replyFuture.HasException() ? NThreading::GetExceptionMessage(replyFuture) : "No value in reply from external service";
                if ((!lastMessage || chatRobot->DeleteMessage(permissions->GetUserId(), lastMessage->GetHistoryEventId(), session))
                    && UndeferChat(tagOriginal.GetTagId(), supportChatTag->GetTopicLink(), Server, session)
                    && session.Commit()) {
                    session.SetErrorInfo("ExternalDefer", error);
                }
                session.DoExceptionOnFail(ConfigHttpStatus);
            } else {
                if (lastMessage && (!chatRobot->EditMessage(permissions->GetUserId(), lastMessage->GetHistoryEventId(), NDrive::NChat::TMessageEdit({}, {}, NDrive::NChat::TMessage::EExternalStatus::Sent), session) || !session.Commit())) {
                    g.MutableReport().AddReportElement("debug", "Failed to change message status");
                }
            }
        }
    }
    g.SetCode(HTTP_OK);
}

void TSupportCenterUndeferRequestProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr /*permissions*/, const NJson::TJsonValue& /*requestData*/) {
    const TCgiParameters& cgi = Context->GetCgiParameters();
    auto tagId = GetString(cgi, "tag_id");
    auto chatId = GetString(cgi, "chat_id");
    auto session = BuildTx<NSQL::Writable>();
    if (!UndeferChat(tagId, chatId, Server, session) || !session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }
    g.SetCode(HTTP_OK);
}

void TSupportCenterGetCategorizationProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::SupportRequestCategorization);
    R_ENSURE(Server->GetSupportCenterManager(), ConfigHttpStatus.ServiceUnavailable, "support center manager not configured");

    auto categorizer = Server->GetSupportCenterManager()->GetSupportRequestCategorizer();
    R_ENSURE(categorizer, ConfigHttpStatus.ServiceUnavailable, "support request categorizer not configured");

    const TCgiParameters& cgi = Context->GetCgiParameters();
    auto tagIds = GetStrings(cgi, "tag_id");

    categorizer->RefreshCache(Context->GetRequestStartTime());

    NJson::TJsonValue result;

    auto categorizations = categorizer->GetActualCategorizations(tagIds);
    for (auto&& [tagId, tagCategorizations]: categorizations) {
        if (tagCategorizations.Empty()) {
            continue;
        }
        result.AppendValue(tagCategorizations.BuildReport());
    }

    g.MutableReport().AddReportElement("categorizations", std::move(result));
    g.SetCode(HTTP_OK);
}

void TSupportCenterAddCategorizationProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Add, TAdministrativeAction::EEntity::SupportRequestCategorization);
    R_ENSURE(Server->GetSupportCenterManager(), ConfigHttpStatus.ServiceUnavailable, "support center manager not configured");

    auto categorizer = Server->GetSupportCenterManager()->GetSupportRequestCategorizer();
    R_ENSURE(categorizer, ConfigHttpStatus.ServiceUnavailable, "support request categorizer not configured");

    TSupportRequestCategorization cat;
    R_ENSURE(cat.DeserializeFromJson(requestData), ConfigHttpStatus.SyntaxErrorStatus, "could not deserialize categorization from json");

    auto session = Server->GetSupportCenterManager()->BuildSession();
    R_ENSURE(categorizer->AddCategorization(cat, permissions->GetUserId(), session), ConfigHttpStatus.UnknownErrorStatus, "could not add history", session);
    if (!session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }

    g.SetCode(HTTP_OK);
}

void TSupportCenterRemoveCategorizationProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Remove, TAdministrativeAction::EEntity::SupportRequestCategorization);
    R_ENSURE(Server->GetSupportCenterManager(), ConfigHttpStatus.ServiceUnavailable, "support center manager not configured");

    auto categorizer = Server->GetSupportCenterManager()->GetSupportRequestCategorizer();
    R_ENSURE(categorizer, ConfigHttpStatus.ServiceUnavailable, "support request categorizer not configured");

    const TCgiParameters& cgi = Context->GetCgiParameters();
    auto tagId = GetString(cgi, "tag_id");
    auto catId = GetValue<ui64>(cgi, "id", /* required = */ true);

    auto categorization = categorizer->GetActualCategorization(tagId, *catId);
    R_ENSURE(categorization, HTTP_NOT_FOUND, "cannot get categorization " << tagId << "/" << catId);
    R_ENSURE(categorization->GetTagId() == tagId, HTTP_INTERNAL_SERVER_ERROR, "tag_id mismatch: " << categorization->GetTagId());
    R_ENSURE(categorization->GetOperatedId() == catId, HTTP_INTERNAL_SERVER_ERROR, "id mismatch: " << categorization->GetOperatedId());

    auto session = Server->GetSupportCenterManager()->BuildSession();
    R_ENSURE(categorizer->RemoveCategorization(*categorization, permissions->GetUserId(), session), ConfigHttpStatus.UnknownErrorStatus, "could not write history", session);
    if (!session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }

    g.SetCode(HTTP_OK);
}

void TSupportCenterCategorizationHistoryProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::SupportRequestCategorization);
    R_ENSURE(Server->GetSupportCenterManager(), ConfigHttpStatus.ServiceUnavailable, "support center manager not configured");

    auto categorizer = Server->GetSupportCenterManager()->GetSupportRequestCategorizer();
    R_ENSURE(categorizer, ConfigHttpStatus.ServiceUnavailable, "support request categorizer not configured");

    const TCgiParameters& cgi = Context->GetCgiParameters();
    auto tagId = GetString(cgi, "tag_id");

    categorizer->RefreshCache(Context->GetRequestStartTime());

    auto history = categorizer->GetCategorizationHistory(tagId);

    g.MutableReport().SetExternalReport(history.BuildReport(true));
    g.SetCode(HTTP_OK);
}

void TSupportCenterCategorizationTreeProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::SupportCategorizationTree);
    R_ENSURE(Server->GetSupportCenterManager() && Server->GetSupportCenterManager()->GetSupportRequestCategorizer(), ConfigHttpStatus.ServiceUnavailable, "support center manager or catgegorizer not configured");
    auto categorizer = Server->GetSupportCenterManager()->GetSupportRequestCategorizer()->GetTreeManager();
    R_ENSURE(categorizer, ConfigHttpStatus.ServiceUnavailable, "tree manager not configured");
    const TCgiParameters& cgi = Context->GetCgiParameters();
    auto filter = GetString(cgi, "filter", false);
    auto report = categorizer->GetTreeReport(filter);
    g.MutableReport().AddReportElement("tree", std::move(report));
    g.MutableReport().AddReportElement("node_scheme", TSupportCategorizerTreeNode::GetScheme().SerializeToJson());
    g.SetCode(HTTP_OK);
}

void TSupportCenterCategorizationTreeEditProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::SupportCategorizationTree);
    R_ENSURE(Server->GetSupportCenterManager() && Server->GetSupportCenterManager()->GetSupportRequestCategorizer(), ConfigHttpStatus.ServiceUnavailable, "support center manager or catgegorizer not configured");
    auto tree = Server->GetSupportCenterManager()->GetSupportRequestCategorizer()->GetTreeManager();
    R_ENSURE(tree, ConfigHttpStatus.ServiceUnavailable, "tree manager not configured");
    R_ENSURE(requestData.Has("action") && requestData["action"].IsString(), ConfigHttpStatus.SyntaxErrorStatus, "'action' not specified or it is not string");
    auto action = requestData["action"].GetString();

    auto session = Server->GetSupportCenterManager()->BuildSession();
    if (action == "add") {
        R_ENSURE(requestData.Has("parent_id") && (requestData["parent_id"].IsString() || requestData["parent_id"].IsNull()), ConfigHttpStatus.SyntaxErrorStatus, "'parent_id' not specified or it is neither string not null");
        TSupportCategorizerTreeNode node;
        R_ENSURE(node.DeserializeMeta(requestData["meta"]), ConfigHttpStatus.SyntaxErrorStatus, "could not deserialize meta");

        NStorage::TObjectRecordsSet<TSupportCategorizerTreeNode> upsertedNodes;
        R_ENSURE(tree->UpsertNode(node, permissions->GetUserId(), session, &upsertedNodes), ConfigHttpStatus.UnknownErrorStatus, "unable to upsert node");
        if (requestData["parent_id"].IsString()) {
            tree->RefreshAll(Now());
            R_ENSURE(upsertedNodes.size() == 1, ConfigHttpStatus.UnknownErrorStatus, "unable to upsert node");
            auto parentId = requestData["parent_id"].GetString();
            auto maybeObject = tree->GetNode(parentId);
            R_ENSURE(maybeObject.Defined(), ConfigHttpStatus.SyntaxErrorStatus, "could not find parent node");
            R_ENSURE(tree->AddEdge(parentId, upsertedNodes.back().GetId(), permissions->GetUserId(), session, true), ConfigHttpStatus.UnknownErrorStatus, "could not add edge");
        }
    } else if (action == "modify") {
        R_ENSURE(requestData.Has("id") && requestData["id"].IsString(), ConfigHttpStatus.SyntaxErrorStatus, "could not deserialize id");
        auto id = requestData["id"].GetString();
        auto maybeObject = tree->GetNode(id);
        R_ENSURE(maybeObject.Defined(), ConfigHttpStatus.SyntaxErrorStatus, "could not find node");
        TSupportCategorizerTreeNode node;
        node.SetId(id);
        R_ENSURE(node.DeserializeMeta(requestData["meta"]), ConfigHttpStatus.SyntaxErrorStatus, "could not deserialize meta");
        R_ENSURE(tree->UpsertNode(node, permissions->GetUserId(), session, nullptr), ConfigHttpStatus.UnknownErrorStatus, "unable to upsert node");
    } else if (action == "remove_node") {
        R_ENSURE(requestData.Has("id") && requestData["id"].IsString(), ConfigHttpStatus.SyntaxErrorStatus, "could not deserialize id");
        auto id = requestData["id"].GetString();
        auto maybeObject = tree->GetNode(id);
        R_ENSURE(maybeObject.Defined(), ConfigHttpStatus.SyntaxErrorStatus, "could not find node");
        R_ENSURE(tree->RemoveNode(id, permissions->GetUserId(), session), ConfigHttpStatus.UnknownErrorStatus, "unable to remove node");
    } else if (action == "move") {
        R_ENSURE(requestData.Has("id") && requestData["id"].IsString(), ConfigHttpStatus.SyntaxErrorStatus, "could not deserialize id");
        R_ENSURE(requestData.Has("old_parent_id") && (requestData["old_parent_id"].IsString() || requestData["old_parent_id"].IsNull()), ConfigHttpStatus.SyntaxErrorStatus, "could not deserialize old_parent_id");
        R_ENSURE(requestData.Has("new_parent_id") && (requestData["new_parent_id"].IsString() || requestData["new_parent_id"].IsNull()), ConfigHttpStatus.SyntaxErrorStatus, "could not deserialize new_parent_id");
        auto id = requestData["id"].GetString();
        auto oldParentId = requestData["old_parent_id"].IsString() ? requestData["old_parent_id"].GetString() : "";
        auto newParentId = requestData["new_parent_id"].IsString() ? requestData["new_parent_id"].GetString() : "";
        if (oldParentId) {
            R_ENSURE(tree->RemoveEdge(oldParentId, id, permissions->GetUserId(), session), ConfigHttpStatus.UnknownErrorStatus, "unable to remove edge");
        }
        if (newParentId) {
            R_ENSURE(tree->AddEdge(newParentId, id, permissions->GetUserId(), session), ConfigHttpStatus.UnknownErrorStatus, "unable to add edge");
        }
    } else if (action == "copy") {
        R_ENSURE(requestData.Has("id") && requestData["id"].IsString(), ConfigHttpStatus.SyntaxErrorStatus, "could not deserialize id");
        R_ENSURE(requestData.Has("new_parent_id") && requestData["new_parent_id"].IsString(), ConfigHttpStatus.SyntaxErrorStatus, "could not deserialize new_parent_id");
        auto id = requestData["id"].GetString();
        auto newParentId = requestData["new_parent_id"].GetString();
        R_ENSURE(tree->AddEdge(newParentId, id, permissions->GetUserId(), session), ConfigHttpStatus.UnknownErrorStatus, "unable to add edge");
    } else {
        R_ENSURE(false, ConfigHttpStatus.SyntaxErrorStatus, "unknown command name");
    }

    if (!session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }

    g.SetCode(HTTP_OK);
}

void TSupportCenterCallEventProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    R_ENSURE(requestData.Has("cc") && requestData["cc"].IsString(), ConfigHttpStatus.SyntaxErrorStatus, "cc not specified");
    R_ENSURE(requestData.Has("phone") && requestData["phone"].IsString(), ConfigHttpStatus.SyntaxErrorStatus, "phone not specified");
    TString originCCStr = requestData["cc"].GetString();

    auto session = BuildTx<NSQL::Writable>();
    TString phone = requestData["phone"].GetString();
    TString userId;
    {
        TVector<TString> userIds;
        R_ENSURE(Server->GetDriveAPI()->GetUsersData()->SelectUsers("phone", {phone}, userIds, session, false), ConfigHttpStatus.SyntaxErrorStatus, "could not select users");
        if (userIds.size() == 0) {
            NDrive::TExternalUser externalUser;
            externalUser.SetPhone(phone);
            auto user = Server->GetDriveAPI()->GetUsersData()->RegisterExternalUser(permissions->GetUserId(), session, externalUser);
            R_ENSURE(user, ConfigHttpStatus.UnknownErrorStatus, "could not register mock user for request");
            userId = user->GetUserId();
        } else {
            auto userFetchResult = Server->GetDriveAPI()->GetUsersData()->FetchInfo(userIds, session);
            R_ENSURE(userFetchResult, {}, "cannot FetchInfo", session);
            for (auto&& [id, user] : userFetchResult) {
                if (user.IsPhoneVerified()) {
                    userId = user.GetUserId();
                }
            }
            if (!userId) {
                userId = userIds.front();
            }
        }
    }

    TSupportTelephonyEvent request;
    R_ENSURE(request.Construct(userId, requestData), ConfigHttpStatus.SyntaxErrorStatus, "could not construct");
    if (!request.Apply(Server, session) || !session.Commit()) {
        NJson::TJsonValue errorInfo = NJson::TMapBuilder("call_id", request.GetInternalCallId())
                                                        ("event_type", ::ToString(request.GetType()))
                                                        ("error_details", session.GetMessages().GetStringReport());
        NDrive::TEventLog::Log("SupportTelephonyCallEventError", errorInfo);
        ERROR_LOG << "Error performing telephony call event: " << errorInfo.GetStringRobust() << Endl;
        session.DoExceptionOnFail(ConfigHttpStatus, nullptr, "could not apply request");
    }

    g.SetCode(HTTP_OK);
}

void TSupportCenterWebphoneAuthProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    R_ENSURE(Server->GetSupportCenterManager(), ConfigHttpStatus.ServiceUnavailable, "support center manager not configured");
    auto webphoneClient = Server->GetSupportCenterManager()->GetWebphoneClient();
    R_ENSURE(webphoneClient, ConfigHttpStatus.ServiceUnavailable, "webphone client not configured");

    TString login(permissions->GetLogin());
    R_ENSURE(login, ConfigHttpStatus.UnknownErrorStatus, "we don't know login for this user");

    TString cookie;
    if (!GetHandlerSettingDef<bool>("use_tvm", false)) {
        cookie = Context->GetRequestData().HeaderInOrEmpty("Cookie");
        R_ENSURE(cookie, ConfigHttpStatus.SyntaxErrorStatus, "you haven't passed a cookie");
    }

    const TString processorName = GetTypeName();
    auto fToken = cookie
        ? webphoneClient->GetWebphoneAuthData(login, cookie, requestData)
        : webphoneClient->GetWebphoneAuthData(login, requestData);
    auto report = g.GetReport();

    g.Release();
    fToken.Subscribe(
        [report, processorName](const NThreading::TFuture<TWebphoneClient::TResponse>& tokenReport) {
            TWebphoneClient::TResponse reportValue = tokenReport.GetValue();

            if (reportValue.GetCode() / 100 > 2) {
                auto errorInfo = reportValue.SerializeToJson();
                NDrive::TEventLog::Log("SupportWebphoneAuthError", errorInfo);
                ERROR_LOG << "Error performing webphone auth: " << errorInfo.GetStringRobust() << Endl;
            }

            TJsonReport::TGuard g(report, reportValue.GetCode());
            g.MutableReport().SetExternalReport(reportValue.GetJsonContent());
        }
    );
}

void TSupportCenterGetFilteredUserCountProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    R_ENSURE(requestData.Has("roles") || requestData.Has("assigned_tags") || requestData.Has("performing_tags"), ConfigHttpStatus.SyntaxErrorStatus, "no filters provided (roles, assigned_tags, performing_tags)");
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::User);

    const auto& userTagManager = DriveApi->GetTagsManager().GetUserTags();
    auto session = BuildTx<NSQL::ReadOnly>();

    TSet<TString> filteredUsers;

    if (requestData.Has("roles")) {
        R_ENSURE(requestData["roles"].IsMap(), ConfigHttpStatus.SyntaxErrorStatus, "roles: not a map");
        bool checkRoleActive = false;
        if (requestData["roles"].Has("check_active")) {
            R_ENSURE(requestData["roles"]["check_active"].IsBoolean(), ConfigHttpStatus.SyntaxErrorStatus, "roles: check_active is not a boolean");
            checkRoleActive = requestData["roles"]["check_active"].GetBoolean();
        }
        TSet<TString> filterRoles = MakeSet(GetStrings(requestData["roles"], "list"));
        TString filterMode = GetString(requestData["roles"], "filter_mode", false);
        if (filterMode == "all_of") {
            TVector<TString> filteredUsersVector;
            R_ENSURE(Server->GetDriveAPI()->GetUsersData()->GetRoles().GetUsersWithRoles(filterRoles, filteredUsersVector, checkRoleActive), ConfigHttpStatus.UnknownErrorStatus, "failed to acquire users with provided roles");
            filteredUsers = MakeSet(filteredUsersVector);
        } else {
            R_ENSURE(Server->GetDriveAPI()->GetUsersData()->GetRoles().GetUsersWithAtLeastOneRoles(filterRoles, filteredUsers, checkRoleActive), ConfigHttpStatus.UnknownErrorStatus, "failed to acquire users with provided roles");
        }
    }

    if (requestData.Has("assigned_tags")) {
        R_ENSURE(requestData["assigned_tags"].IsMap(), ConfigHttpStatus.SyntaxErrorStatus, "assigned_tags: not a map");
        TSet<TString> filterTags = MakeSet(GetStrings(requestData["assigned_tags"], "list"));
        TString filterMode = GetString(requestData["assigned_tags"], "filter_mode", false);
        if (filterMode == "all_of") {
            TVector<TTaggedUser> taggedUsers;
            R_ENSURE(userTagManager.GetObjects(filteredUsers, filterTags, false, taggedUsers, session), ConfigHttpStatus.UnknownErrorStatus, "failed to acquire users with provided tags", session);
            filteredUsers.clear();
            for (auto&& taggedUser : taggedUsers) {
                if (taggedUser.GetTags().size() < filterTags.size()) {
                    continue;
                }
                TSet<TString> tagsOnUser;
                for (auto&& tag : taggedUser.GetTags()) {
                    tagsOnUser.insert(tag.GetTypeId(tag));
                }
                if (tagsOnUser.size() == filterTags.size()) {
                    filteredUsers.insert(taggedUser.GetId());
                }
            }
        } else {
            auto optionalTags = userTagManager.RestoreTags(filteredUsers, MakeVector(filterTags), session);
            R_ENSURE(optionalTags, {}, "cannot RestoreTags", session);
            filteredUsers.clear();
            for (auto&& tag : *optionalTags) {
                filteredUsers.insert(tag.GetObjectId());
            }
        }
    }

    if (requestData.Has("performing_tags")) {
        R_ENSURE(requestData["performing_tags"].IsMap(), ConfigHttpStatus.SyntaxErrorStatus, "performing_tags: not a map");
        auto filterPerformingTags = GetStrings(requestData["performing_tags"], "list");
        TVector<TDBTag> userTags;
        R_ENSURE(
            userTagManager.RestorePerformerTags(filterPerformingTags, MakeVector(filteredUsers), userTags, session),
            {},
            "cannot RestorePerformerTags",
            session
        );
        filteredUsers.clear();
        for (auto&& tag : userTags) {
            TString user = tag->GetPerformer();
            filteredUsers.insert(user);
        }
    }

    g.MutableReport().AddReportElement("users_count", filteredUsers.size());
    g.SetCode(HTTP_OK);
}

void TSupportCenterForwardExternalMessagesProcessor::Parse(const NJson::TJsonValue& requestData) {
    UserId = GetString(requestData, "user_id", true);
    ChatId = GetString(requestData, "chat_id", true);
    ChatType = GetString(requestData, "new_chat_type", true);
    Comment = GetString(requestData, "comment", false);
    R_ENSURE(TryFromJson(requestData["message_ids"], MessageIds), ConfigHttpStatus.SyntaxErrorStatus, "can't read messages ids");
}

void TSupportCenterForwardExternalMessagesProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    TVector<TDBTag> dbTags;
    TString externalChatId, chatId, topic;
    {
        auto session = BuildTx<NSQL::ReadOnly>();
        TSet<TString> tagNames = Server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetRegisteredTagNames({TSupportChatTag::TypeName});
        if (tagNames.size()) {
            R_ENSURE(Server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({UserId}, MakeVector(tagNames), dbTags, session), ConfigHttpStatus.SyntaxErrorStatus, "Can't restore tags");
            for (auto&& tag : dbTags) {
                auto tagImpl = tag.GetTagAs<TSupportChatTag>();
                if (tagImpl && tagImpl->GetExternalChatInfo().GetId() && tagImpl->GetTopicLink() == ChatId) {
                    externalChatId = tagImpl->GetExternalChatInfo().GetId();
                    IChatRobot::ParseTopicLink(tagImpl->GetTopicLink(), chatId, topic);
                    break;
                }
            }
        }
    }
    R_ENSURE(externalChatId, ConfigHttpStatus.SyntaxErrorStatus, "Can't get external chat tag data");
    auto chatRobot = Server->GetChatRobot(chatId);
    R_ENSURE(chatRobot, ConfigHttpStatus.ServiceUnavailable, "Chat robot not configured");

    NDrive::NChat::TOptionalMessageEvents messages;
    {
        auto session = BuildChatSession(true);
        messages = chatRobot->GetChatMessagesRange(UserId, topic, NDrive::NChat::TMessage::AllKnownTraits, *MessageIds.begin(), *MessageIds.rbegin()+1, session);
        if (!messages) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
    };
    TVector<TString> externalMessagesIds;
    for (auto&& message : *messages) {
        if (message.GetExternalId() && MessageIds.contains(message.GetHistoryEventId())) {
            externalMessagesIds.push_back(message.GetExternalId());
        }
    }
    R_ENSURE(externalMessagesIds.size(), ConfigHttpStatus.SyntaxErrorStatus, "Can't find messages to change status");

    auto taxiClient = Server->GetTaxiChatClient();
    R_ENSURE(taxiClient, ConfigHttpStatus.ServiceUnavailable, "Taxi client not configured");
    auto replyFuture = taxiClient->ForwardMessages(externalChatId, ChatType, externalMessagesIds, Comment);
    replyFuture.Wait();
    R_ENSURE(!replyFuture.HasException() && replyFuture.HasValue() && replyFuture.GetValue(), ConfigHttpStatus.ServiceUnavailable, replyFuture.HasException() ? NThreading::GetExceptionMessage(replyFuture) : "No reply value from external client");

    NJson::TJsonValue report = NJson::JSON_ARRAY;
    if (messages) {
        auto session = BuildChatSession(false);
        for (auto&& message : *messages) {
            if (message.GetExternalId() && MessageIds.contains(message.GetHistoryEventId())) {
                if (!chatRobot->EditMessage(permissions->GetUserId(), message.GetHistoryEventId(), NDrive::NChat::TMessageEdit({}, {}, NDrive::NChat::TMessage::EExternalStatus::Forwarded), session)) {
                    report.AppendValue(message.GetExternalId() + " message was forwarded, but message status was not changed " + session.GetStringReport());
                    break;
                }
            }
        }
        if (!session.Commit()) {
            report.AppendValue("Message was forwarded, but message status was not changed " + session.GetStringReport());
        }
    }
    g.MutableReport().AddReportElement("errors", std::move(report));
    g.SetCode(HTTP_OK);
}

void TSupportCenterGetLoadBalanceProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /* requestData */) {
    R_ENSURE(Server->GetSupportCenterManager(), ConfigHttpStatus.ServiceUnavailable, "support center manager is not configured");
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::Settings, Server->GetSupportCenterManager()->LoadBalanceSettingKey);

    const TCgiParameters& cgi = Context->GetCgiParameters();
    auto applicationName = GetString(cgi, "application_name", true);

    auto calendar = Server->GetSupportCenterManager()->GetLoadBalanceCalendar(applicationName);
    R_ENSURE(calendar, ConfigHttpStatus.SyntaxErrorStatus, "no valid calendar for application " + applicationName);

    g.MutableReport().AddReportElement("application_name", applicationName);
    g.MutableReport().AddReportElement("data", calendar->SerializeToJson());

    g.SetCode(HTTP_OK);
}

void TSupportCenterUpdateLoadBalanceProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    const auto* supportManagerPtr = Server->GetSupportCenterManager();

    R_ENSURE(supportManagerPtr, ConfigHttpStatus.ServiceUnavailable, "support center manager is not configured");
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::Settings, supportManagerPtr->LoadBalanceSettingKey);

    const TCgiParameters& cgi = Context->GetCgiParameters();
    auto applicationName = GetString(cgi, "application_name", /* required = */ true);

    const auto& loadBalanceCalendarData = requestData["data"];

    auto calendarHelper = TLoadBalanceCalendarHelper(applicationName);
    R_ENSURE(calendarHelper.DeserializeFromJson(loadBalanceCalendarData), ConfigHttpStatus.SyntaxErrorStatus, "invalid callcenter load balance data provided");

    // update current value in telephony
    auto currentLoadValue = calendarHelper.GetCurrentValue();
    R_ENSURE(currentLoadValue, ConfigHttpStatus.SyntaxErrorStatus, "cannot parse current load value");

    TMessagesCollector errors;
    R_ENSURE(supportManagerPtr->UpdateCurrentLoadBalance(*currentLoadValue, permissions->GetUserId(), errors, /* update_calendar = */ false), ConfigHttpStatus.UnknownErrorStatus, "Cannot update callcenter load balance in telephony");

    // update all calendar values (multiple calendar updates are allowed)
    R_ENSURE(supportManagerPtr->UpdateLoadBalanceCalendar(calendarHelper, permissions->GetUserId()), ConfigHttpStatus.UnknownErrorStatus, "Error saving load balance calendar");

    g.MutableReport().AddReportElement("application_name", applicationName);
    g.MutableReport().AddReportElement("data", NJson::TJsonValue(loadBalanceCalendarData));

    g.SetCode(HTTP_OK);
}

TSet<TString> GetApplications(const NDrive::IServer& server, const TString& setting) {
    TString applicationNames;
    const THttpStatusManagerConfig& httpStatuses = server.GetHttpStatusManagerConfig();
    R_ENSURE(server.GetSettings().GetValueStr(setting, applicationNames), httpStatuses.ServiceUnavailable, setting + " is undefined");
    TSet<TString> applications;
    StringSplitter(applicationNames).SplitBySet(",").SkipEmpty().Collect(&applications);
    R_ENSURE(!applications.empty(), httpStatuses.ServiceUnavailable, "Fail to parse" + setting);
    return applications;
}

void TSupportCenterPlabackInfoProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /* requestData */) {
    const auto* supportManagerPtr = Server->GetSupportCenterManager();
    R_ENSURE(supportManagerPtr, ConfigHttpStatus.ServiceUnavailable, "support center manager is not configured");
    R_ENSURE(supportManagerPtr->GetCallCenterYandexClient(), ConfigHttpStatus.ServiceUnavailable, "support call center client is not configured");
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::Settings, supportManagerPtr->IVRSettingKey);

    NJson::TJsonValue apps;
    TMessagesCollector errors;
    for (auto&& application : GetApplications(*Server, supportManagerPtr->IVRSettingKey + ".application_names")) {
        NCallCenterYandex::TAppPlayback app;
        R_ENSURE(supportManagerPtr->GetCallCenterYandexClient()->GetAppPlaybackInfo(application, app, errors), ConfigHttpStatus.UnknownErrorStatus, "Cannot load data from telephony. " + errors.GetStringReport());
        apps.AppendValue(app.GetReport());
    }

    NJson::TJsonValue report;
    report.InsertValue("applications", apps);
    g.MutableReport().SetExternalReport(std::move(report));

    g.SetCode(HTTP_OK);
}

TMaybe<TVector<NCallCenterYandex::TAppPlayback::TActionPlayback>> GetPlaybackActionsFromSettings(const TString& settings, TMessagesCollector& errors) {
    NJson::TJsonValue json;
    if (!ReadJsonTree(settings, &json)) {
        return {};
    }
    NCallCenterYandex::TAppPlayback app;
    if (!app.DeserializeFromJson(NJson::TMapBuilder("CONTENT", json), errors)) {
        return {};
    }
    return app.GetActions();
}

TVector<NCallCenterYandex::TAppPlayback::TActionPlayback> GetPlaybackActions(const NDrive::IServer& server, const TString& setting) {
    TString records;
    const THttpStatusManagerConfig& httpStatuses = server.GetHttpStatusManagerConfig();
    R_ENSURE(server.GetSettings().GetValueStr(setting, records), httpStatuses.ServiceUnavailable, setting + " is undefined");
    TMessagesCollector errors;
    auto actions = GetPlaybackActionsFromSettings(records, errors);
    R_ENSURE(actions && !actions->empty(), httpStatuses.SyntaxErrorStatus, "fail to parse settings " + errors.GetStringReport());
    return *actions;
}

void TSupportCenterPlaybackUpdateProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    const auto* supportManagerPtr = Server->GetSupportCenterManager();
    R_ENSURE(supportManagerPtr, ConfigHttpStatus.ServiceUnavailable, "support center manager is not configured");
    R_ENSURE(supportManagerPtr->GetCallCenterYandexClient(), ConfigHttpStatus.ServiceUnavailable, "support call center client is not configured");
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::Settings, supportManagerPtr->IVRSettingKey);

    TString applicationName;
    {
        applicationName = GetString(requestData, "application_name", /* required = */ true);
        auto applications = GetApplications(*Server, supportManagerPtr->IVRSettingKey + ".application_names");
        R_ENSURE(applications.contains(applicationName), ConfigHttpStatus.ServiceUnavailable, applicationName + "is unknown");
    }
    NCallCenterYandex::TAppPlayback app(applicationName);
    {
        const TString actionName = GetString(requestData, "record", /* required = */ true);
        auto actions = GetPlaybackActions(*Server, supportManagerPtr->IVRSettingKey + ".records");
        auto it = FindIf(actions, [&actionName](const auto& action) { return action.GetActionName() == actionName; });
        R_ENSURE(it != actions.end(), ConfigHttpStatus.SyntaxErrorStatus, "action '" + actionName + "' is undefined");
        app.MutableActions() = {std::move(*it)};
    }
    TMessagesCollector errors;
    R_ENSURE(supportManagerPtr->GetCallCenterYandexClient()->UpdateAppPlayback(app, errors), ConfigHttpStatus.UnknownErrorStatus, "Cannot update IVR in fromtelephony. " + errors.GetStringReport());
    g.SetCode(HTTP_OK);
}

NDrive::TScheme TSupportCenterPlaybackUpdateProcessor::GetRequestDataScheme(const IServerBase* server, const TCgiParameters& /* schemeCgi */) {
    const NDrive::IServer& serverImpl = server->GetAsSafe<NDrive::IServer>();
    NDrive::TScheme scheme;
    auto applications = GetApplications(serverImpl, ISupportCenterManager::IVRSettingKey + ".application_names");
    scheme.Add<TFSVariants>("application_name", "Имя приложения").SetVariants(applications).SetRequired(true);
    TSet<TString> keys;
    {
        auto actions = GetPlaybackActions(serverImpl, ISupportCenterManager::IVRSettingKey + ".records");
        Transform(actions.begin(), actions.end(), std::inserter(keys, keys.begin()), [](auto& action) { return action.GetActionName(); });
    }
    scheme.Add<TFSVariants>("record", "IVR").SetVariants(keys).SetRequired(true);
    return scheme;
}

void TSupportCenterDynamicDistributionProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr /* permissions */, const NJson::TJsonValue& /* requestData */) {
    auto defAppReport = [&g, this](){
        TUnistatSignalsCache::SignalAdd("dynamic-distribution", "failure", 1);
        NJson::TJsonValue report;
        TString defaultAnswer = GetHandlerSettingDef<TString>("default_answer", "");
        R_ENSURE(NJson::ReadJsonFastTree(defaultAnswer, &report), ConfigHttpStatus.UnknownErrorStatus, "Undefined default answer");
        g.SetExternalReport(std::move(report));
        g.SetCode(HTTP_OK);
    };

    TString reportTemplateStr = GetHandlerSettingDef<TString>("report_template", "");
    if (!reportTemplateStr) {
        NDrive::TEventLog::Log("DynamicDistributionFailure", NJson::TMapBuilder
            ("error", "Fail to get user report template")
        );
        defAppReport();
        return;
    }

    const TString phoneArg = GetHandlerSettingDef<TString>("phone_param", "cid");
    const TString phone = GetString(Context->GetCgiParameters(), phoneArg, /* required = */ false);
    if (!phone) {
        NDrive::TEventLog::Log("DynamicDistributionFailure", NJson::TMapBuilder
            ("error", "Fail to get user report template")
            ("phone_arg", phoneArg)
        );
        defAppReport();
        return;
    }

    TVector<TString> allowedStatuses = SplitString(GetHandlerSettingDef<TString>("allowed_statuses", ""), ",");
    auto session = BuildTx<NSQL::ReadOnly>();
    auto usersData = Server->GetDriveAPI()->GetUsersData()->FindUsersByPhone(phone, MakeSet(allowedStatuses), /* phoneVerified = */ true, session);
    if (!usersData) {
        NDrive::TEventLog::Log("DynamicDistributionFailure", NJson::TMapBuilder
            ("session_error", session.GetReport())
            ("error", "Fail to get user by phone")
            ("user_phone", phone)
        );
        defAppReport();
        return;
    }
    if (usersData->size() != 1) {
        NDrive::TEventLog::Log("DynamicDistributionFailure", NJson::TMapBuilder
            ("error", "Got wrong users count: " + ToString(usersData->size()))
            ("user_phone", phone)
        );
        defAppReport();
        return;
    }

    auto userId = usersData->front().GetUserId();
    TVector<TString> supportRequestTypes = SplitString(GetHandlerSettingDef<TString>("tags", ""), ",");
    if (!supportRequestTypes) {
        supportRequestTypes.emplace_back("@" + TSupportOutgoingCallTag::TypeName);
        supportRequestTypes.emplace_back("@" + TSupportPhoneCallTag::TypeName);
    }
    TSet<TString> requestedTags;
    for (auto&& tag : supportRequestTypes) {
        if (tag.StartsWith("@")) {
            auto tagDescrs = Server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetTagsByType(tag.substr(1, tag.size() - 1));
            for (auto&& descr : tagDescrs) {
                requestedTags.insert(descr->GetName());
            }
        } else {
            requestedTags.insert(tag);
        }
    }
    const auto& userTagsManager = Server->GetDriveAPI()->GetTagsManager().GetUserTags();
    auto queryOptions = TTagEventsManager::TQueryOptions().SetTags(requestedTags).SetObjectIds({userId});
    auto since = Now() - GetHandlerSettingDef<TDuration>("search_age", TDuration::Hours(1));
    auto events = userTagsManager.GetEvents({since}, session, queryOptions);
    if (!events) {
        NDrive::TEventLog::Log("DynamicDistributionFailure", NJson::TMapBuilder
            ("session_error", session.GetReport())
            ("error", "Fail to get user events")
            ("user_id", userId)
        );
        defAppReport();
        return;
    }
    TString operatorId;
    for (auto& ev : *events) {
        if (ev->GetPerformer()) {
            operatorId = ev->GetPerformer();
        }
    }
    if (!operatorId) {
        NDrive::TEventLog::Log("DynamicDistributionFailure", NJson::TMapBuilder
            ("error", "Fail to get performer")
            ("user_id", userId)
        );
        defAppReport();
        return;
    }

    auto userRoles = DriveApi->GetUsersData()->GetRoles().RestoreUserRoles(operatorId, session);
    if (!userRoles) {
        NDrive::TEventLog::Log("DynamicDistributionFailure", NJson::TMapBuilder
            ("session_error", session.GetReport())
            ("error", "Fail to get operator roles")
            ("operator_id", operatorId)
            ("user_id", userId)
        );
        defAppReport();
        return;
    }
    const TString defaultOperatorRole = "interface-internal-callcenter-operator";
    auto operatorRoles = MakeSet(SplitString(GetHandlerSettingDef<TString>("operator_roles", defaultOperatorRole), ","));
    auto it = FindIf(userRoles->GetRoles(), [&operatorRoles](const auto& role) { return role.GetActive() && operatorRoles.contains(role.GetRoleId()); });
    if (it == userRoles->GetRoles().end()) {
        NDrive::TEventLog::Log("DynamicDistributionFailure", NJson::TMapBuilder
            ("error", "Fail to find operator active role")
            ("operator_id", operatorId)
            ("user_id", userId)
        );
        defAppReport();
        return;
    }
    {
        TVector<TDBTag> tags;
        if (!userTagsManager.RestorePerformerTags({operatorId}, tags, session)) {
            NDrive::TEventLog::Log("DynamicDistributionFailure", NJson::TMapBuilder
                ("session_error", session.GetReport())
                ("error", "Fail to get operator tags")
                ("operator_id", operatorId)
                ("user_id", userId)
            );
            defAppReport();
            return;
        }
        if (!tags.empty()) {
            NDrive::TEventLog::Log("DynamicDistributionFailure", NJson::TMapBuilder
                ("error", "Operator performs tag")
                ("operator_id", operatorId)
                ("user_id", userId)
            );
            defAppReport();
            return;
        }
    }

    auto operatorPhone = TUserAdminSettings(operatorId).GetInternalPhone(session);
    if (!operatorPhone) {
        NDrive::TEventLog::Log("DynamicDistributionFailure", NJson::TMapBuilder
            ("session_error", session.GetReport())
            ("error", "Fail to get operator settings")
            ("operator_id", operatorId)
            ("user_id", userId)
        );
        defAppReport();
        return;
    }
    if (!*operatorPhone) {
        NDrive::TEventLog::Log("DynamicDistributionFailure", NJson::TMapBuilder
            ("error", "Operator has wrong phone")
            ("operator_id", operatorId)
            ("user_id", userId)
        );
        defAppReport();
        return;
    }

    TString phoneTemplate = "_PHONE_";
    SubstGlobal(reportTemplateStr, phoneTemplate, *operatorPhone);
    NJson::TJsonValue report;
    if (!NJson::ReadJsonFastTree(reportTemplateStr, &report)) {
        NDrive::TEventLog::Log("DynamicDistributionFailure", NJson::TMapBuilder
            ("error", "Fail to build report")
            ("operator_id", operatorId)
            ("user_id", userId)
            ("report_template", reportTemplateStr)
            ("operator_phone", *operatorPhone)
        );
        defAppReport();
        return;
    }
    g.SetExternalReport(std::move(report));
    g.SetCode(HTTP_OK);

    TUnistatSignalsCache::SignalAdd("dynamic-distribution", "success", 1);
}

void TSupportCenterStartCallProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /* requestData */) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Add, TAdministrativeAction::EEntity::WebPhoneCalls);
    const auto* supportManagerPtr = Server->GetSupportCenterManager();
    R_ENSURE(supportManagerPtr, ConfigHttpStatus.ServiceUnavailable, "support center manager is not configured");

    auto userId = GetString(Context->GetCgiParameters(), "user_id", /* required = */ false);
    auto phone = GetString(Context->GetCgiParameters(), "phone", /* required = */ false);
    R_ENSURE(userId || phone, ConfigHttpStatus.SyntaxErrorStatus, "required user id or phone");


    auto session = BuildTx<NSQL::Writable>();
    if (!userId) {
        NDrive::TExternalUser externalUser;
        externalUser.SetPhone(phone);
        auto user = Server->GetDriveAPI()->GetUsersData()->FindOrRegisterExternal(permissions->GetUserId(), session, externalUser);
        R_ENSURE(user, {}, "cannot FindOrRegisterExternal", session);
        userId = user->GetUserId();
    } else {
        auto usersInfo = Server->GetDriveAPI()->GetUsersData()->FetchInfo(userId, session);
        R_ENSURE(usersInfo, {}, "cannot FetchInfo", session);
        auto userInfo = usersInfo.GetResultPtr(userId);
        R_ENSURE(userInfo, ConfigHttpStatus.UnknownErrorStatus, ToString(EDriveLocalizationCodes::InternalServerError) + ": fail to find user " + userId);
        R_ENSURE(!phone || phone == userInfo->GetPhone(), ConfigHttpStatus.UnknownErrorStatus, ToString(EDriveLocalizationCodes::InternalServerError) + ": user has another phone number");
        R_ENSURE(userInfo->GetPhone(), ConfigHttpStatus.UnknownErrorStatus, ToString(EDriveLocalizationCodes::InternalServerError) + ": user has no phone number");
        phone = userInfo->GetPhone();
    }

    TWebPhoneCall call;
    call.SetStatus(TWebPhoneCall::EStatus::Initialized);
    call.SetUserId(userId);
    call.SetPhone(phone);
    NStorage::TObjectRecordsSet<TWebPhoneCall> records;
    auto supportSession = BuildChatSession(/* readOnly = */ false);
    if (!supportManagerPtr->GetWebPhoneCallManager().AddObjects({call}, permissions->GetUserId(), supportSession, &records) || records.empty() || !supportSession.Commit()) {
        supportSession.DoExceptionOnFail(ConfigHttpStatus);
    }
    const TString callId = supportManagerPtr->GetWebPhoneCallManager().ConvertOriginCallId(records.front().GetId());

    const TString callTag = GetHandlerSetting<TString>("call_tag").GetOrElse(TSupportOutgoingCallTag::TypeName);
    auto tag = Server->GetDriveAPI()->GetTagsManager().GetTagsMeta().CreateTag(callTag);
    R_ENSURE(tag, ConfigHttpStatus.UnknownErrorStatus, "cannot create call tag");
    auto socTag = std::dynamic_pointer_cast<TSupportOutgoingCallTag>(tag);
    R_ENSURE(socTag, ConfigHttpStatus.SyntaxErrorStatus, "incorrect call tag name");
    socTag->SetCallId(callId);
    socTag->SetPerformer(permissions->GetUserId());

    const auto tagId = GetString(Context->GetCgiParameters(), "tag_id", /* required = */ false);
    if (tagId) {
        socTag->SetCallTaskTag(tagId);
    }

    auto addedTags = Server->GetDriveAPI()->GetTagsManager().GetUserTags().AddTag(tag, permissions->GetUserId(), userId, Server, session);
    R_ENSURE(addedTags && !addedTags->empty(), ConfigHttpStatus.UnknownErrorStatus, "cannot AddTag", session);

    if (!session.Commit()) {
        auto supportRemoveSession = BuildChatSession(/* readOnly = */ false);
        if (!supportManagerPtr->GetWebPhoneCallManager().RemoveObjects({call.GetId()}, permissions->GetUserId(), supportRemoveSession) || !supportRemoveSession.Commit()) {
            supportRemoveSession.DoExceptionOnFail(ConfigHttpStatus);
        }
        session.DoExceptionOnFail(ConfigHttpStatus);
    }

    g.MutableReport().AddReportElement("call_id", callId);
    g.MutableReport().AddReportElement("tag_id", addedTags->front().GetTagId());
    g.SetCode(HTTP_OK);
}

void TSupportCenterUpdateCallProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Add, TAdministrativeAction::EEntity::WebPhoneCalls);
    const auto* supportManagerPtr = Server->GetSupportCenterManager();
    R_ENSURE(supportManagerPtr, ConfigHttpStatus.ServiceUnavailable, "support center manager is not configured");
    auto& tagsManager = Server->GetDriveAPI()->GetTagsManager();
    const TString comment = GetString(requestData, "comment", /* required = */ false);
    const auto callStatus = GetValue<TWebPhoneCall::EStatus>(requestData, "call_status", /* required = */ true);
    R_ENSURE(callStatus, ConfigHttpStatus.UnknownErrorStatus, "Call status has wrong format.");
    TWebPhoneCall call;
    {
        ui64 originId = 0;
        const TString callId = GetString(requestData, "call_id", /* required = */ true);
        R_ENSURE(supportManagerPtr->GetWebPhoneCallManager().TryParseOriginCallId(callId, originId), ConfigHttpStatus.UnknownErrorStatus, "Call id has wrong format.");
        auto supportSession = BuildChatSession(/* readOnly = */ true);
        auto calls = supportManagerPtr->GetWebPhoneCallManager().GetObjects({originId}, supportSession);
        R_ENSURE(!!calls && !calls->empty(), ConfigHttpStatus.UnknownErrorStatus, "Unknown call.", supportSession);
        call = std::move(calls->front());
    }
    switch (*callStatus) {
    case TWebPhoneCall::EStatus::Initialized:
        break;
    case TWebPhoneCall::EStatus::Completed:
    case TWebPhoneCall::EStatus::NotServised: {
        TVector<TDBTag> allTags;
        {
            auto session = BuildTx<NSQL::ReadOnly>();
            const TSet<TString> tagNames = tagsManager.GetTagsMeta().GetRegisteredTagNames({TSupportOutgoingCallTag::TypeName});
            R_ENSURE(tagsManager.GetUserTags().RestoreTags({}, MakeVector(tagNames), allTags, session), ConfigHttpStatus.UnknownErrorStatus, "Fail to restore tags.", session);
        }
        TVector<TDBTag> removeTags;
        for (auto&& dbTag : allTags) {
            auto tag = dbTag.MutableTagAs<TSupportOutgoingCallTag>();
            R_ENSURE(tag, ConfigHttpStatus.UnknownErrorStatus, "Tag has wrong type. " + dbTag->GetName());
            ui64 originId = 0;
            if (!supportManagerPtr->GetWebPhoneCallManager().TryParseOriginCallId(tag->GetCallId(), originId)) {
                continue;
            }
            if (originId == call.GetId()) {
                removeTags.push_back(std::move(dbTag));
            }
        }
        if (!removeTags.empty()) {
            const bool force = GetValue<bool>(requestData, "force_tag_remove", /* required = */ false).GetOrElse(false);
            auto session = BuildTx<NSQL::Writable>();
            session.SetComment(comment);
            if (!tagsManager.GetUserTags().RemoveTags(removeTags, permissions->GetUserId(), Server, session, force) || !session.Commit()) {
                session.DoExceptionOnFail(ConfigHttpStatus);
            }
        }
        call.SetComment(comment);
        break;
    }
    default:
        g.MutableReport().AddReportElement("error", "Unsupported status");
        g.SetCode(HTTP_BAD_REQUEST);
        return;
    }
    call.SetStatus(*callStatus);
    auto supportSession = BuildChatSession(/* readOnly = */ false);
    if (!supportManagerPtr->GetWebPhoneCallManager().UpsertObject(call, permissions->GetUserId(), supportSession) || !supportSession.Commit()) {
        supportSession.DoExceptionOnFail(ConfigHttpStatus);
    }
    g.SetCode(HTTP_OK);
}

void TSupportCenterSyncCallDataProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /* requestData */) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::WebPhoneCalls);
    const auto* supportManagerPtr = Server->GetSupportCenterManager();
    R_ENSURE(supportManagerPtr, ConfigHttpStatus.ServiceUnavailable, "Support center manager is not configured.");
    R_ENSURE(Server->GetDriveAPI(), ConfigHttpStatus.ServiceUnavailable, "DriveApi problem.");
    R_ENSURE(Server->GetDriveAPI()->HasMDSClient(), ConfigHttpStatus.ServiceUnavailable, "MDS client is not configured.");
    TWebPhoneCall call;
    {
        ui64 originId = 0;
        const TString callId = GetString(Context->GetCgiParameters(), "call_id", /* required = */ true);
        R_ENSURE(supportManagerPtr->GetWebPhoneCallManager().TryParseOriginCallId(callId, originId), ConfigHttpStatus.UnknownErrorStatus, "Call id has wrong format.");
        auto supportSession = BuildChatSession(/* readOnly = */ true);
        auto calls = supportManagerPtr->GetWebPhoneCallManager().GetObjects({originId}, supportSession);
        R_ENSURE(calls && !calls->empty(), ConfigHttpStatus.UnknownErrorStatus, "cannot GetObjects", supportSession);
        call = std::move(calls->front());
    }
    TMessagesCollector errors;
    R_ENSURE(supportManagerPtr->GetWebPhoneCallManager().GetCallData(*Server, call.GetId(), call, errors) == TWebPhoneCallManager::ECallDataError::Ok, ConfigHttpStatus.UnknownErrorStatus, "Fail to get call data. " + errors.GetStringReport());
    {
        auto supportSession = BuildChatSession(/* readOnly = */ false);
        R_ENSURE(supportManagerPtr->GetWebPhoneCallManager().UpsertObject(call, permissions->GetUserId(), supportSession), {}, "cannot UpsertObject", supportSession);
        R_ENSURE(supportSession.Commit(), {}, "cannot Commit", supportSession);
        g.MutableReport().AddReportElement("call", call.SerializeToJsonReport());
    }
    g.SetCode(HTTP_OK);
}

void TSupportCenterLoadCiptTrackProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr /* permissions */, const NJson::TJsonValue& requestData) {
    const auto* supportManagerPtr = Server->GetSupportCenterManager();
    R_ENSURE(supportManagerPtr, ConfigHttpStatus.ServiceUnavailable, "Support center manager is not configured.");
    R_ENSURE(supportManagerPtr->GetMDSClient(), ConfigHttpStatus.ServiceUnavailable, "Support MDS problem.");

    const TString& bucketName = GetString(requestData, "bucket", /* required = */ true);
    auto bucket = supportManagerPtr->GetMDSClient()->GetBucket(bucketName);
    R_ENSURE(bucket, ConfigHttpStatus.ServiceUnavailable, "MDS bucket not found.");

    const TString& callId = GetString(requestData, "call_id", /* required = */ true);
    supportManagerPtr->ProcessInternalCall(*bucket, callId).Subscribe([report = g.GetReport()](const auto& result) {
        auto res = result.GetValue();
        TJsonReport::TGuard g(report, HttpCodes::HTTP_INTERNAL_SERVER_ERROR);
        if (res.first == TSupportCenterManager::EProcessCallResult::Ok) {
            g.SetCode(HTTP_OK);
        } else {
            g.MutableReport().AddReportElement("error_type", ToString(res.first));
            g.MutableReport().AddReportElement("error", res.second);
        }
    });
    g.Release();
}

void TSupportCenterBindAudioteleCallProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::AudiotelePhoneCalls);
    const auto* supportManagerPtr = Server->GetSupportCenterManager();
    R_ENSURE(supportManagerPtr, ConfigHttpStatus.ServiceUnavailable, "Support center manager is not configured.");
    const TString& callId = GetString(requestData, "call_id", /* required = */ true);
    const auto& manager = supportManagerPtr->GetAudioteleCallsManager();
    auto session = manager.BuildSession(/* readOnly = */ false);
    TAudioteleCallEvent::TId eventId;
    {
        auto events = manager.GetEvents(session, { TAudioteleCallEvent::EAction::Finish }, { callId });
        R_ENSURE(events, ConfigHttpStatus.UnknownErrorStatus, "Fail to restore call events.", session);
        R_ENSURE(!events->empty(), ConfigHttpStatus.EmptySetStatus, "Fail to find any finishing events.");
        R_ENSURE(GetValue<bool>(requestData, "use_first_finish", /* required = */ false).GetOrElse(false) || events->size() == 1, ConfigHttpStatus.EmptySetStatus, "Fail to find finishing call event. Current events count: " << events->size());
        eventId = events->front().GetId();
    }
    if (!GetValue<bool>(requestData, "force", /* required = */ false).GetOrElse(false)) {
        auto tracks = manager.GetTracksByEvents(session, { eventId });
        R_ENSURE(tracks, ConfigHttpStatus.UnknownErrorStatus, "Fail to restore call track info.", session);
        R_ENSURE(tracks->empty(), ConfigHttpStatus.SyntaxErrorStatus, "Call is already bound.");
    }
    TAudioteleCallTrack::TId trackId;
    {
        auto tracks = manager.GetTracksByCallId(session, callId);
        R_ENSURE(tracks, ConfigHttpStatus.UnknownErrorStatus, "Fail to restore call tacks.", session);
        R_ENSURE(tracks->size() == 1, ConfigHttpStatus.EmptySetStatus, "Fail to find call tracks. Current tracks count: " << tracks->size());
        trackId = tracks->front().GetId();
    }
    R_ENSURE(manager.BindCallTrack(eventId, trackId, session) && session.Commit(), ConfigHttpStatus.UnknownErrorStatus, "Fail to bind call event to track.", session);
    g.SetCode(HTTP_OK);
}

void TUpdateSupportAICallProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    const auto* supportManagerPtr = Server->GetSupportCenterManager();
    R_ENSURE(supportManagerPtr, ConfigHttpStatus.ServiceUnavailable, "Support center manager is not configured.");

    const TString& callId = GetString(requestData, "call_id", /* required = */ true);
    const TString& callStatus = GetString(requestData, "call_status", /* required = */ true);
    TSupportAICall::EStatus status = TSupportAICall::EStatus::Initialized;
    R_ENSURE(TryFromString(callStatus, status), ConfigHttpStatus.SyntaxErrorStatus, "Fail to parse new call status.");
    auto tx = BuildChatSession(/* readOnly = */ false);
    auto optionalCalls = supportManagerPtr->GetSupportAICallManager().GetObjects(tx, NSQL::TQueryOptions().AddGenericCondition("external_id", callId));
    R_ENSURE(optionalCalls, ConfigHttpStatus.ServiceUnavailable, "Fail to fetch call.", tx);
    R_ENSURE(!optionalCalls->empty(), ConfigHttpStatus.EmptySetStatus, "Fail to find call.");
    auto& call = optionalCalls->front();
    call.SetStatus(status);
    R_ENSURE(supportManagerPtr->GetSupportAICallManager().UpsertObject(call, permissions->GetUserId(), tx) && tx.Commit(), ConfigHttpStatus.ServiceUnavailable, "Fail to update call.", tx);
    g.SetCode(HTTP_OK);
}
