#include "processor.h"

#include <drive/backend/processors/sessions/base_impl.h>

#include <drive/backend/data/notifications_tags.h>
#include <drive/backend/promo_codes/common/action.h>
#include <drive/backend/promo_codes/common/entities.h>
#include <drive/backend/promo_codes/common/manager.h>
#include <drive/backend/roles/manager.h>

#include <drive/library/cpp/clck/client.h>
#include <drive/library/cpp/taxi/request.h>

void TTaxiInfoProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    Y_UNUSED(requestData);
    R_ENSURE(permissions, ConfigHttpStatus.UnauthorizedStatus, "UserPermissions are missing");
    const auto& offers = permissions->GetOfferBuilders();
    R_ENSURE(!offers.empty(), ConfigHttpStatus.PermissionDeniedStatus, "insufficient permissions");

    const TCgiParameters& cgi = Context->GetCgiParameters();
    const auto from = ParseValue<TGeoCoord>(cgi.Get("from"));
    const auto to = ParseValue<TGeoCoord>(cgi.Get("to"));
    TString language = GetString(cgi, "lang", false);
    if (!language) {
        language = "ru";
    }

    auto taxiRouteInfoClient = Server->GetTaxiRouteInfoClient();
    R_ENSURE(taxiRouteInfoClient, ConfigHttpStatus.UnknownErrorStatus, "TaxiRouteInfoClient is missing");

    auto deadline = Context->GetRequestDeadline();
    auto timeout = deadline - Now();

    TTaxiRequest request(from, to);
    request.SetLanguage(language);
    TSimpleTaxiReply::TPtr response = taxiRouteInfoClient->SendRequest(request, timeout);
    R_ENSURE(response, ConfigHttpStatus.UnknownErrorStatus, "TaxiAPI response is missing");
    R_ENSURE(response->IsSucceeded(), ConfigHttpStatus.IncompleteStatus, "TaxiAPI request is unsuccessful: " << response->GetCode());

    auto report = response->GetReport().GetOriginalReport();
    g.MutableReport().SetExternalReport(std::move(report));
    g.SetCode(HTTP_OK);
}

void TTaxiSessionItemProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    auto locale = GetLocale();
    auto sessionId = GetString(requestData, "order_id");
    SetLocale(locale);
    SetTrackStatus(NDrive::ECarStatus::csRide);

    auto tx = BuildTx<NSQL::ReadOnly>();

    THistoryRidesContext context(*TBase::Server, TInstant::Zero(), false, true);
    {
        auto eg = g.BuildEventGuard("InitializeHistoryRidesContext");
        auto ydbTx = BuildYdbTx<NSQL::ReadOnly>("taxi_session_item");
        R_ENSURE(context.InitializeSession(sessionId, tx, ydbTx), TBase::ConfigHttpStatus.UnknownErrorStatus, "cannot initialize HistoryRidesContext");
    }
    auto session = context.GetSession(sessionId);
    R_ENSURE(session, ConfigHttpStatus.EmptySetStatus, "cannot find order_id " << sessionId);
    R_ENSURE(session->GetUserId() == permissions->GetUserId(), ConfigHttpStatus.PermissionDeniedStatus, "no permissions to view " << sessionId);

    bool useRawTracks = GetHandlerSetting<bool>("use_raw_tracks").GetOrElse(false);

    auto sessions = NContainer::Scalar(*session);
    PrefetchTracks(sessions, g, false, useRawTracks);
    PrefetchPayments(sessions, tx, g);
    PrefetchFullCompiledRiding(sessions, g);
    PrefetchGeocodedLocations(sessions, g);
    PrefetchFineAttachments(sessions, tx, g);

    const auto& fetcher = GetFetcher(sessions, g, permissions);
    {
        NJson::TJsonValue data = GetReport(*session, g, permissions, /*detailed=*/true);
        g.AddReportElement("data", std::move(data));
    }
    {
        NJson::TJsonValue meta = NJson::JSON_MAP;
        g.AddReportElement("meta", std::move(meta));
    }
    fetcher.BuildReportMeta(g.MutableReport());
    g.SetCode(HTTP_OK);
}

void TTaxiSessionListProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    const TCgiParameters& cgi = Context->GetCgiParameters();
    const auto locale = GetLocale();
    const bool skipEmpty = GetValue<bool>(cgi, "skip_empty", false).GetOrElse(false);

    const auto count = GetValue<ui64>(requestData, "range.results", false).GetOrElse(10);
    const auto until = GetTimestamp(requestData, "range.older_than.created_at", TInstant::Max());
    SetLocale(locale);

    auto session = BuildTx<NSQL::ReadOnly>();

    THistoryRidesContext context(*Server, TInstant::Zero(), false, true);
    {
        auto eg = g.BuildEventGuard("InitializeHistoryRidesContext");
        auto ydbTx = BuildYdbTx<NSQL::ReadOnly>("taxi_session_list");
        R_ENSURE(context.InitializeUser(permissions->GetUserId(), session, ydbTx, until, count), ConfigHttpStatus.UnknownErrorStatus, "cannot initialize HistoryRidesContext");
    }

    bool hasMore = false;
    auto latest = context.GetIterator().GetAndNext();
    auto sessions = context.GetSessions(until, count, &hasMore, skipEmpty);
    PrefetchFullCompiledRiding(sessions, g);
    PrefetchGeocodedLocations(sessions, g);

    const auto& fetcher = GetFetcher(sessions, g, permissions);
    {
        NJson::TJsonValue orders = NJson::JSON_ARRAY;
        for (auto&& session : sessions) {
            orders.AppendValue(GetReport(session, g, permissions, /*detailed=*/false));
        }
        g.AddReportElement("orders", std::move(orders));
    }
    if (hasMore && !sessions.empty()) {
        const auto& session = sessions.back();
        NJson::TJsonValue cursor;
        cursor.InsertValue("created_at", session.GetStartTS().Seconds());
        cursor.InsertValue("order_id", session.GetSessionId());
        g.AddReportElement("cursor", std::move(cursor));
    }
    if (latest) {
        NJson::TJsonValue serviceMetadata;
        serviceMetadata["name"] = GetHandlerLocalization("service_metadata.name", "Yandex.Drive", locale);
        serviceMetadata["service"] = "drive";
        serviceMetadata["last_order_id"] = latest->GetSessionId();
        g.AddReportElement("service_metadata", std::move(serviceMetadata));
    }
    fetcher.BuildReportMeta(g.MutableReport());
    g.SetCode(HTTP_OK);
}

void TCreateAccountWithPromocodeProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    ReqCheckCondition(Server->GetPromoCodesManager(), ConfigHttpStatus.ServiceUnavailable, EDriveLocalizationCodes::InternalServerError);

    const TString generatorName = GetString(Context->GetCgiParameters(), "generator", true);
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Add, TAdministrativeAction::EEntity::PromoCodes, generatorName);
    auto action = Server->GetDriveAPI()->GetRolesManager()->GetAction(generatorName);
    ReqCheckCondition(action.Defined(), ConfigHttpStatus.UnknownErrorStatus, EDriveLocalizationCodes::InternalServerError);
    const IPromoCodeGenerator* generatorAction = (*action).GetAs<IPromoCodeGenerator>();
    ReqCheckCondition(generatorAction, ConfigHttpStatus.UnknownErrorStatus, EDriveLocalizationCodes::InternalServerError);

    auto sendSmsTag = GetString(Context->GetCgiParameters(), "send_sms", false);

    auto codesCount = GetValue<ui32>(Context->GetCgiParameters(), "count", false);
    auto phones = GetStrings(requestData, "phones", false);
    ReqCheckCondition(codesCount.Defined() || !phones.empty(), ConfigHttpStatus.SyntaxErrorStatus, EDriveLocalizationCodes::SyntaxUserError);
    codesCount = codesCount.Defined() ? codesCount : phones.size();

    TGeneratorContext context;
    auto stantartContext = GetHandlerSetting<TString>(generatorName + "_standart_context");
    if (stantartContext.Defined()) {
        NJson::TJsonValue contextArray;
        for (const auto& param : requestData.GetMap()) {
            contextArray.AppendValue(NJson::TMapBuilder("key", param.first)("value", param.second.GetStringRobust()));
        }
        NJson::TJsonValue jsonContext;
        ReqCheckCondition(ReadJsonTree(*stantartContext, &jsonContext) && context.DeserializeFromJson(jsonContext) && context.SetTagDictionaryContext(contextArray), ConfigHttpStatus.UnknownErrorStatus, EDriveLocalizationCodes::InternalServerError);
    } else {
        ReqCheckCondition(context.DeserializeFromJson(requestData), ConfigHttpStatus.SyntaxErrorStatus, EDriveLocalizationCodes::SyntaxUserError);
    }
    ReqCheckCondition(!!context.GetGivenOut(), ConfigHttpStatus.SyntaxErrorStatus, EDriveLocalizationCodes::SyntaxUserError);

    const NDrive::NBilling::TAccountsManager& accountsManager = Server->GetDriveAPI()->GetBillingManager().GetAccountsManager();
    const TString accountName = GetString(requestData, "name", true);
    auto registeredAccounts = accountsManager.GetRegisteredAccounts(TInstant::Zero());
    TMaybe<NDrive::NBilling::TAccountDescriptionRecord> description;
    for (auto&& acc : registeredAccounts) {
        if (acc.GetName() == accountName) {
            description = acc;
        }
    }
    R_ENSURE(description.Defined(), ConfigHttpStatus.UserErrorState, "unknown type " + accountName, EDriveSessionResult::IncorrectRequest);
    R_ENSURE(description->GetIsPersonal(), ConfigHttpStatus.UserErrorState, "can't create not personal account " + accountName, EDriveSessionResult::IncorrectRequest);

    TVector<TPromoCodeMeta> codes;
    auto session = BuildTx<NSQL::Writable | NSQL::RepeatableRead>();
    if (!Server->GetPromoCodesManager()->GenerateCodes(*codesCount, *generatorAction, context, permissions->GetUserId(), codes, session)) {
        session.DoExceptionOnFail(ConfigHttpStatus, Server->GetLocalization());
    }

    TVector<ui32> accountIds = CreateAccounts(permissions, *description, requestData, session, codes.size());
    R_ENSURE(codes.size() == accountIds.size(), ConfigHttpStatus.UserErrorState, "can't create accounts");

    TString smsTextTemplate;
    if (sendSmsTag) {
        auto desc = Server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetDescriptionByName(sendSmsTag);
        auto messageDesc = dynamic_cast<const TUserMessageTagBase::TDescription*>(desc.Get());
        R_ENSURE(messageDesc, ConfigHttpStatus.UserErrorState, "incorrect notification tag", EDriveSessionResult::IncorrectRequest);

        smsTextTemplate = messageDesc->GetMessageText();
    }

    NJson::TJsonValue report = NJson::JSON_ARRAY;
    for (size_t i = 0 ; i < codes.size(); ++i) {
        auto& code = codes[i];
        auto& accountId = accountIds[i];
        auto deeplinkTemplate = GetHandlerSettingDef<TString>("deeplink_template", "yandexdrive://promo/_PROMOCODE_");
        SubstGlobal(deeplinkTemplate, "_PROMOCODE_", code.GetCode());
        if (Server->GetClckClient() && GetHandlerSettingDef<bool>("short_deeplink", true)) {
            TMessagesCollector errors;
            Y_UNUSED(Server->GetClckClient()->Compress(deeplinkTemplate, deeplinkTemplate, errors));
        }
        report.AppendValue(NJson::TMapBuilder("code", code.GetCode())("account_id", accountId)("deeplink", deeplinkTemplate));
        if (sendSmsTag && i < phones.size()) {
            TString messageText = smsTextTemplate;
            SubstGlobal(messageText, "_CODE_", code.GetCode());
            SubstGlobal(messageText, "_DEEPLINK_", deeplinkTemplate);

            TString userId;
            auto users = Server->GetDriveAPI()->GetUsersData()->SelectUsers("phone", { phones[i] }, session, false);
            if (!users) {
                session.DoExceptionOnFail(ConfigHttpStatus);
            }

            if (!users->empty()) {
                for (const auto& user : *users) {
                    if (user.GetPublicStatus() == NDrive::UserStatusActive) {
                        userId = user.GetUserId();
                        break;
                    }
                }
                if (!userId) {
                    userId = users->front().GetUserId();
                }
            } else {
                NDrive::TExternalUser externalUser;
                externalUser.SetPhone(phones[i]);
                auto user = Server->GetDriveAPI()->GetUsersData()->RegisterExternalUser(permissions->GetUserId(), session, externalUser);
                if (!user) {
                    session.DoExceptionOnFail(ConfigHttpStatus);
                }
                userId = user->GetUserId();
            }

            R_ENSURE(userId, ConfigHttpStatus.SyntaxErrorStatus, "could not find user with phone " + phones[i]);

            auto tag = Server->GetDriveAPI()->GetTagsManager().GetTagsMeta().CreateTag(sendSmsTag);
            auto smsTag = std::dynamic_pointer_cast<TUserMessageTagBase>(tag);
            R_ENSURE(smsTag, ConfigHttpStatus.SyntaxErrorStatus, "incorrect notification tag name", EDriveSessionResult::IncorrectRequest);
            smsTag->SetMessageText(messageText);
            if (!Server->GetDriveAPI()->GetTagsManager().GetUserTags().AddTag(tag, permissions->GetUserId(), userId, Server, session)) {
                session.DoExceptionOnFail(ConfigHttpStatus);
            }
        }
        TPromocodeAccountLink promoLink;
        promoLink.SetPromocode(code.GetCode()).SetAccountId(accountId);
        R_ENSURE(
            Server->GetDriveAPI()->GetBillingManager().GetPromocodeAccountLinks().AddHistory(promoLink, permissions->GetUserId(), EObjectHistoryAction::Add, session, nullptr),
            ConfigHttpStatus.UserErrorState,
            "can't link promocode",
            EDriveSessionResult::InconsistencyUser
        );
    }
    if (!session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }

    g.MutableReport().AddReportElement("accounts", std::move(report));
    g.SetCode(HTTP_OK);
}
