#include "processor.h"

#include <drive/backend/cars/car.h>
#include <drive/backend/cars/car_model.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/data/scoring/scoring.h>
#include <drive/backend/data/telematics.h>
#include <drive/backend/device_snapshot/manager.h>
#include <drive/backend/offers/offers/standart.h>
#include <drive/backend/processors/leasing/session_process.h>
#include <drive/backend/sessions/matcher/session_matcher.h>

#include <drive/telematics/api/sensor/history.h>
#include <drive/telematics/server/data/events.h>
#include <drive/telematics/server/sensors/cache.h>
#include <drive/telematics/server/tasks/lite.h>

#include <rtline/library/json/cast.h>

namespace {
    const TVector<NDrive::TSensorId> CarStateSensors = {
        VEGA_MCU_FIRMWARE_VERSION,
        CAN_ODOMETER_KM,
        CAN_ENGINE_TEMP,
        CAN_FUEL_LEVEL_P,
        CAN_FUEL_DISTANCE_KM,
        VEGA_ACC_VOLTAGE,
        VEGA_POWER_VOLTAGE,
        CAN_DRIVER_DOOR,
        CAN_PASS_DOOR,
        CAN_L_REAR_DOOR,
        CAN_R_REAR_DOOR,
        CAN_PARKING,
        CAN_IGNITION,
        CAN_ENGINE_IS_ON,
        CAN_HOOD,
        CAN_TRUNK,
        CAN_DIPPED_BEAM,
        CAN_HIGH_BEAM,
        CAN_HAND_BREAK,
        CAN_PEDAL_BREAK,
        CAN_AIRBAG,
        CAN_BATTERY,
        CAN_CHECK_BRAKE_PADS,
        CAN_FUEL_LEVEL,
        CAN_DRIVER_SAFE_BELT,
        CAN_CHECK_HAND_BREAK,
        CAN_INFLATION_PRESSURE,
        CAN_CHECK_OIL,
        CAN_CHECK_ENGINE,
        SERVICE_MAINT,
        VEGA_MCC,
        VEGA_MNC,
        VEGA_LAC,
        VEGA_CELLID,
        VEGA_GSM_SIGNAL_LEVEL,
        VEGA_EXT_SERVING_CELL_INF,
        VEGA_GSM_USED_SIM,
        VEGA_GPS_JAMMED,
        VEGA_GPS_SPOOF_SENSOR,
        VEGA_GPS_IS_ACTIVE,
        VEGA_INT_TEMP,
        NDrive::NVega::DigitalInput<1>(),
        NDrive::NVega::DigitalInput<2>(),
        NDrive::NVega::DigitalOutput<3>(),
    };

    const TVector<NDrive::TSensorId> ActivitySensors = {
        CAN_ENGINE_IS_ON,
    };

    void CheckVisibility(const TString& carId, TUserPermissions::TPtr permissions, const NDrive::IServer& server) {
        auto object = server.GetDriveDatabase().GetTagsManager().GetDeviceTags().GetObject(carId);
        R_ENSURE(object, HTTP_NOT_FOUND, "cannot find object" << carId);

        auto invisibilityInfo = TUserPermissions::TUnvisibilityInfoSet(0);
        auto visibility = permissions->GetVisibility(*object, NEntityTagsManager::EEntityType::Car, &invisibilityInfo);
        R_ENSURE(visibility != TUserPermissions::EVisibility::NoVisible, HTTP_FORBIDDEN, "cannot access " << carId << ": " << TUserPermissions::ExplainInvisibility(invisibilityInfo));
    }

    TString GetModel(const TString& carId, const NDrive::IServer& server) {
        const auto deviceInfos = server.GetDriveAPI()->GetCarsData()->FetchInfo({ carId }, TInstant::Zero());
        auto deviceInfo = deviceInfos.GetResultPtr(carId);
        R_ENSURE(deviceInfo, HTTP_INTERNAL_SERVER_ERROR, "device " << carId << " is not found");
        return deviceInfo->GetModel();
    }
}

void TTelematicsStatusProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    Y_UNUSED(permissions);
    Y_UNUSED(requestData);
    const NDrive::TTelematicsClient& client = Server->GetTelematicsClient();
    const TCgiParameters& cgi = Context->GetCgiParameters();
    const TString& id = GetString(cgi, "id");

    TJsonReport& report = g.MutableReport();
    NDrive::TTelematicsClient::THandler handler = client.FetchInfo(id);

    R_ENSURE(handler.GetFuture().Wait(Context->GetRequestDeadline()), ConfigHttpStatus.TimeoutStatus, "FetchInfo timeout");
    report.AddReportElement("handler", NJson::ToJson(handler));

    g.SetCode(HTTP_OK);
}

void TTelematicsCarPasswordProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    Y_UNUSED(requestData);
    auto& report = g.MutableReport();
    const TCgiParameters& cgi = Context->GetCgiParameters();
    const auto& carId = GetString(cgi, "car_id", false);
    const auto& imei = GetString(cgi, "imei", false);
    const auto serverId = GetValue<ui16>(cgi, "server_id", false).GetOrElse(1);
    R_ENSURE(carId || imei, ConfigHttpStatus.UserErrorState, "either car_id or imei is required");

    bool useAdmPermissions = GetHandlerSetting<bool>("use_adm_permissions").GetOrElse(true);
    if (useAdmPermissions) {
        ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::ObserveStructure, TAdministrativeAction::EEntity::Car, carId);
    } else {
        auto session = BuildTx<NSQL::ReadOnly>();
        bool found = NDrive::ContainsActionInTag(permissions->GetUserId(), DriveApi, ConfigHttpStatus, carId, session, "vega-password");
        R_ENSURE(found, ConfigHttpStatus.PermissionDeniedStatus, "action vega-password not found amoung performed tags for " << carId);
    }

    auto expectedPinInfo = carId
        ? TTelematicsConfigurationTag::GetPin(carId, Server, serverId, Context->GetRequestDeadline())
        : TTelematicsConfigurationTag::GetPinByIMEI(imei, Server, serverId, Context->GetRequestDeadline());
    R_ENSURE(expectedPinInfo, ConfigHttpStatus.ServiceUnavailable, expectedPinInfo.GetError().what());
    report.AddReportElement("imei", expectedPinInfo->Imei);
    report.AddReportElement("method", expectedPinInfo->Method);
    report.AddReportElement("password", expectedPinInfo->Value);
    g.SetCode(HTTP_OK);
}

void TTelematicsCarStateProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    Y_UNUSED(requestData);
    const TCgiParameters& cgi = Context->GetCgiParameters();

    TString carId = GetUUID(cgi, "car_id", false);
    TString imei = GetString(cgi, "imei", false);
    R_ENSURE(carId || imei, ConfigHttpStatus.SyntaxErrorStatus, "either car_id or imei should be present");
    R_ENSURE(carId.empty() || imei.empty(), ConfigHttpStatus.SyntaxErrorStatus, "only one of car_id or imei should be present");
    auto sensorIds = GetValues<NDrive::TSensorId>(cgi, "sensor_id", false);
    auto onlyCritical = GetValue<bool>(cgi, "report_critical_sensors", false).GetOrElse(false);
    auto needSensorMeta = GetValue<bool>(cgi, "need_sensor_meta", false).GetOrElse(false);
    bool useAdmPermissions = GetHandlerSetting<bool>("use_adm_permissions").GetOrElse(true);
    if (!imei) {
        imei = DriveApi->GetIMEI(carId);
    }
    if (!carId) {
        carId = DriveApi->GetCarIdByIMEI(imei); // need for permissions
    }
    if (useAdmPermissions) {
        ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::ObserveStructure, TAdministrativeAction::EEntity::Car, carId);
    } else {
        auto session = BuildTx<NSQL::ReadOnly>();
        bool found = NDrive::ContainsActionInTag(permissions->GetUserId(), DriveApi, ConfigHttpStatus, carId, session, "view-telematics");
        R_ENSURE(found, ConfigHttpStatus.PermissionDeniedStatus, "action view-telematics not found amoung performed tags for " << carId);
        if (!session.Commit()) {
            g.AddEvent(session.GetReport());
        }
    }
    R_ENSURE(imei, ConfigHttpStatus.EmptySetStatus, "IMEI is missing");

    const NDrive::ISensorApi* api = Server->GetSensorApi();
    R_ENSURE(api, ConfigHttpStatus.UnknownErrorStatus, "Sensor API is missing");
    auto sensors = api->GetSensors(imei).ExtractValue(Context->GetRequestDeadline() - Now());

    auto configurationTag = DriveApi->GetTagsManager().GetDeviceTags().GetTagFromCache(carId, TTelematicsConfigurationTag::Type(), TInstant::Zero());
    auto configuration = configurationTag ? configurationTag->GetTagAs<TTelematicsConfigurationTag>() : nullptr;

    auto description = configuration ? DriveApi->GetTagsManager().GetTagsMeta().GetDescriptionByName(configuration->GetName()) : nullptr;
    auto metaDescription = description ? description->GetAs<TTelematicsConfigurationTag::TDescription>() : nullptr;

    auto localization = Server->GetLocalization();
    auto locale = GetLocale();

    NJson::TJsonValue snsrs = NJson::JSON_ARRAY;
    TSet<NDrive::TSensorId> ids;
    if (!sensorIds.empty()) {
        ids = MakeSet(sensorIds);
    } else {
        ids.insert(CarStateSensors.begin(), CarStateSensors.end());
        ids.insert(sensors.begin(), sensors.end());
    }
    for (auto&& id : ids) {
        auto name = id.GetName();
        auto sensor = api->FindSensor(sensors, id);

        if (onlyCritical && metaDescription) {
            if (!metaDescription->IsSensorValueCritical(id, sensor)) {
                continue;
            }
        }

        NJson::TJsonValue element;
        element["id"] = id.Id;
        if (id.SubId) {
            element["subid"] = id.SubId;
        }
        element["name"] = name;

        if (localization) {
            auto title = localization->GetLocalString(locale, TStringBuilder() << "telematics.sensors." << name << ".title", TString{});
            if (title) {
                element["title"] = std::move(title);
            }
            auto message = localization->GetLocalString(locale, TStringBuilder() << "telematics.sensors." << name << ".message", TString{});
            if (message) {
                element["message"] = std::move(message);
            }
        }
        if (sensor) {
            if (std::holds_alternative<double>(sensor->Value)) {
                R_ENSURE(IsValidFloat(std::get<double>(sensor->Value)), ConfigHttpStatus.UnknownErrorStatus,
                "value of sensor \"" << name << "\" in car with imei=" << imei << " is not a valid float");
            }
            element["value"] = sensor->GetJsonValue();
            element["since"] = sensor->Since.Seconds();
            element["updated"] = sensor->Timestamp.Seconds();
        } else {
            element["value"] = NJson::JSON_NULL;
        }
        if (sensor && configuration && configuration->GetCalibrators().contains(id)) {
            const auto& calibrator = configuration->GetCalibrators().at(id);
            element["calibrated"] = calibrator.Get(*sensor);
        }

        if (needSensorMeta && metaDescription && metaDescription->GetSensorMeta().contains(id)) {
            element["sensor_meta"] = NJson::ToJson(metaDescription->GetSensorMeta().at(id));
        }

        snsrs.AppendValue(std::move(element));
    }
    g.MutableReport().AddReportElement("sensors", std::move(snsrs));
    g.MutableReport().SetSortKeys(true);
    g.SetCode(HTTP_OK);
}

void TTelematicsFirmwareAddProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    Y_UNUSED(permissions);
    const TCgiParameters& cgi = Context->GetCgiParameters();
    const TBlob& post = Context->GetBuf();
    R_ENSURE(!post.Empty(), ConfigHttpStatus.UserErrorState, "POST data is missing");

    auto description = NJson::ReadJsonFastTree(cgi.Get("info"));
    auto info = NJson::FromJson<NDrive::NVega::TFirmwareInfo>(description);
    const TString& name = info.GetName();
    R_ENSURE(name, ConfigHttpStatus.UserErrorState, "full_name is empty or missing");

    auto parsedInfo = NDrive::NVega::ParseFirmwareInfo(name);
    info.Model = info.Model ? info.Model : parsedInfo.Model;
    info.Revision = info.Revision ? info.Revision : parsedInfo.Revision;
    info.Tag = info.Tag ? info.Tag : parsedInfo.Tag;
    info.Type = info.Type != NDrive::NVega::TFirmwareInfo::Unknown ? info.Type : parsedInfo.Type;
    R_ENSURE(info.Model, ConfigHttpStatus.UserErrorState, "cannot determine model for " << name);
    R_ENSURE(info.Revision, ConfigHttpStatus.UserErrorState, "cannot determine revision for " << name);
    R_ENSURE(info.Type != NDrive::NVega::TFirmwareInfo::Unknown, ConfigHttpStatus.UserErrorState, "cannot determine type for " << name);

    auto models = DriveApi->GetModelsData()->FetchInfo(TInstant::Zero());
    R_ENSURE(models.GetResult().contains(info.Model), ConfigHttpStatus.UserErrorState, "model " << info.Model << " is unknown");

    auto data = TBuffer(post.AsCharPtr(), post.Size());
    R_ENSURE(!data.Empty(), ConfigHttpStatus.UserErrorState, "POST data is empty");
    R_ENSURE(TTelematicsFirmwareTag::Add(info, std::move(data), *Server), ConfigHttpStatus.UnknownErrorStatus, "cannot add " << name);
    g.SetCode(HTTP_OK);
}

void TTelematicsFirmwareRemoveProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    Y_UNUSED(permissions);
    auto info = NJson::FromJson<NDrive::NVega::TFirmwareInfo>(requestData);
    const auto& name = info.GetName();
    R_ENSURE(name, ConfigHttpStatus.UserErrorState, "full_name is empty or missing");
    R_ENSURE(TTelematicsFirmwareTag::Remove(name, *Server), ConfigHttpStatus.UnknownErrorStatus, "cannot remove " << name);
    g.SetCode(HTTP_OK);
}

void TTelematicsFirmwareListProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    Y_UNUSED(permissions);
    auto names = TTelematicsFirmwareTag::List(*Server);
    auto firmware = NJson::TJsonValue(NJson::JSON_ARRAY);
    for (auto&& name : names) {
        auto fw = TTelematicsFirmwareTag::GetInfo(name, *Server);
        if (fw) {
            firmware.AppendValue(NJson::ToJson(*fw));
        }
    }
    g.MutableReport().AddReportElement("firmware", std::move(firmware));
    g.SetCode(HTTP_OK);
}

void TTelematicsHistoryProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    const TCgiParameters& cgi = Context->GetCgiParameters();
    const TString& source = GetString(cgi, "source", false);
    if (source == "ch") {
        Process2(g, permissions);
    } else if (source == "saas" || source.empty()) {
        Process1(g, permissions);
    } else {
        R_ENSURE(false, HTTP_BAD_REQUEST, "unknown source:" << source);
    }
}

void TTelematicsHistoryProcessor::Process1(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::ObserveStructure, TAdministrativeAction::EEntity::Car);

    const TCgiParameters& cgi = Context->GetCgiParameters();
    ui32 sensorId = 0;
    TryFromString(cgi.Get("sensor_id"), sensorId);
    TString carId = GetUUID(cgi, "car_id", false);
    TString imei = GetString(cgi, "imei", false);
    R_ENSURE(carId || imei || sensorId, ConfigHttpStatus.SyntaxErrorStatus, "either car_id or imei or sensorId should be present");

    if (!Server->GetSensorHistoryApi()) {
        g.SetCode(ConfigHttpStatus.ServiceUnavailable);
        return;
    }

    auto carReportBuilder = [](const NDrive::ISensorHistoryApi::TSensorHistory& history, NJson::TJsonValue& carReport) {
        for (auto sensorHistory : history) {
            auto sReport = sensorHistory.first.ToJson();
            sReport["last_update"] = sensorHistory.second.GetLastUpdate().Seconds();
            NJson::TJsonValue& timeline = sReport.InsertValue("timeline", NJson::JSON_ARRAY);
            for (auto sEvent : sensorHistory.second.GetTimeline()) {
                NJson::TJsonValue& vReport = timeline.AppendValue(NJson::JSON_MAP);
                vReport["value"] = NDrive::TSensorValueOperator<NDrive::TSensorValue>::ToJson(sEvent.second);
                vReport["ts"] = sEvent.first.Seconds();
            }
            carReport.AppendValue(sReport);
        }
    };

    NJson::TJsonValue report(NJson::JSON_ARRAY);
    if (carId || imei) {
        if (!imei) {
            imei = DriveApi->GetIMEI(carId);
        }
        R_ENSURE(imei, ConfigHttpStatus.EmptySetStatus, "IMEI is missing");
        NJson::TJsonValue& carReport = report.InsertValue(imei, NJson::JSON_ARRAY);
        auto history = Server->GetSensorHistoryApi()->GetSensorsHistory(imei).ExtractValueSync();
        carReportBuilder(history, carReport);

    } else {
        auto carsHistory = Server->GetSensorHistoryApi()->GetSensorHistory(sensorId).ExtractValueSync();
        for (auto&& history : carsHistory) {
            NJson::TJsonValue& carReport = report.InsertValue(history.first, NJson::JSON_ARRAY);
            carReportBuilder(history.second, carReport);
        }
    }
    g.MutableReport().SetExternalReport(std::move(report));
    g.SetCode(HTTP_OK);
}

void TTelematicsHistoryProcessor::Process2(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions) {
    const TCgiParameters& cgi = Context->GetCgiParameters();
    auto objectId = GetString(cgi, "car_id", false);
    auto imei = GetString(cgi, "imei", false);
    auto now = Context->GetRequestStartTime();
    auto sensorIds = GetValues<NDrive::TSensorId>(cgi, "sensor_id", false);
    auto since = GetTimestamp(cgi, "since", now - TDuration::Days(3));
    auto until = GetTimestamp(cgi, "until", now);
    auto needSensorMeta = GetValue<bool>(cgi, "need_sensor_meta", false).GetOrElse(false);
    R_ENSURE(objectId || imei, HTTP_BAD_REQUEST, "either car_id or imei should be present");

    auto deadline = Context->GetRequestDeadline();
    auto lastActivity = TInstant::Zero();

    R_ENSURE(since < until, HTTP_BAD_REQUEST, "?since must not be greater than or equal to ?until"); // server drops in testing (timeout 3s, sensors.HasValue() - true but

    if (objectId) {
        ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::ObserveStructure, TAdministrativeAction::EEntity::Car, objectId);
        auto taggedObject = Server->GetDriveDatabase().GetTagsManager().GetDeviceTags().GetObject(objectId);
        R_ENSURE(taggedObject, HTTP_NOT_FOUND, "cannot find " << objectId);

        auto eg = g.BuildEventGuard("last_activity_from_snapshot");
        auto snapshot = Server->GetSnapshotsManager().GetSnapshot(objectId);
        auto heartbeat = snapshot.GetHeartbeat();
        if (heartbeat) {
            lastActivity = heartbeat->Timestamp;
        } else {
            g.AddEvent(NJson::TMapBuilder
                ("event", "NoHeartbeatInSnapshot")
                ("object_id", objectId)
            );
        }
    } else {
        ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::ObserveStructure, TAdministrativeAction::EEntity::Car);

        auto eg = g.BuildEventGuard("last_activity_from_sensors");
        auto sensorClient = Server->GetSensorApi();
        R_ENSURE(sensorClient, HTTP_INTERNAL_SERVER_ERROR, "cannot GetSensorApi");
        auto asyncHeartbeat = sensorClient->GetHeartbeat(imei);
        R_ENSURE(asyncHeartbeat.Wait(deadline), HTTP_GATEWAY_TIME_OUT, "cannot Wait for heartbeat");
        auto heartbeat = asyncHeartbeat.GetValue();
        if (heartbeat) {
            lastActivity = heartbeat->Timestamp;
        } else {
            g.AddEvent(NJson::TMapBuilder
                ("event", "NoHeartbeatInSensors")
                ("imei", imei)
            );
        }
    }

    if (!imei) {
        imei = DriveApi->GetIMEI(objectId);
    }
    if (!objectId) {
        objectId = DriveApi->GetCarIdByIMEI(imei);
    }

    auto configurationTag = DriveApi->GetTagsManager().GetDeviceTags().GetTagFromCache(objectId, TTelematicsConfigurationTag::Type(), TInstant::Zero());
    auto configuration = needSensorMeta && configurationTag ? configurationTag->GetTagAs<TTelematicsConfigurationTag>() : nullptr;

    auto description = configuration ? DriveApi->GetTagsManager().GetTagsMeta().GetDescriptionByName(configuration->GetName()) : nullptr;
    auto metaDescription = description ? description->GetAs<TTelematicsConfigurationTag::TDescription>() : nullptr;

    const NDrive::ISensorApi* api = Server->GetSensorApi();
    R_ENSURE(api, ConfigHttpStatus.UnknownErrorStatus, "Sensor API is missing");
    auto carSensorsFuture = api->GetSensors(imei);
    R_ENSURE(carSensorsFuture.Wait(deadline), HTTP_GATEWAY_TIME_OUT, "wait timeout");
    auto carSensors = carSensorsFuture.GetValue();
    R_ENSURE(carSensors, ConfigHttpStatus.UnknownErrorStatus, "Car sensors is not found"); // in some cases need 4 tries

    TVector<NDrive::TSensorId> carSensorIds;
    for (auto&& sensor : carSensors) {
        carSensorIds.push_back(NDrive::TSensorId(sensor.Id, sensor.SubId));
    }
    if (sensorIds) {
        for (const auto& sensorId : sensorIds) {
            bool sensorIsFound = std::find(carSensorIds.begin(), carSensorIds.end(), sensorId) != carSensorIds.end();
            R_ENSURE(sensorIsFound, HTTP_NOT_FOUND, "all sensor_id should be present in car");
        }
    } else {
        sensorIds = carSensorIds;
    }

    R_ENSURE(sensorIds, HTTP_BAD_REQUEST, "sensor_id should be present");
    auto sensorHistoryClient = Server->GetSensorHistoryClient();
    R_ENSURE(sensorHistoryClient, HTTP_INTERNAL_SERVER_ERROR, "SensorHistoryClient is not configured");
    auto sensors = sensorHistoryClient->Get(objectId, imei, since, until, sensorIds);
    R_ENSURE(sensors.Wait(deadline), HTTP_GATEWAY_TIME_OUT, "wait timeout"); // deadline for more errors Now() + TDuration::Seconds(3)
    auto sensorCache = sensors.GetValue();
    {
        TVector<NDrive::TSensorId> cachedSensorIds;
        NJson::TJsonValue report;
        TMap<NDrive::TSensorId, NJson::TJsonValue> meta;
        if (needSensorMeta && metaDescription) {
            for(auto& sensorId : sensorIds) {
                if (metaDescription->GetSensorMeta().contains(sensorId)) {
                    meta[sensorId] = NJson::ToJson(metaDescription->GetSensorMeta().at(sensorId));
                }
            }
        }
        if (sensorCache) {
            cachedSensorIds = sensorCache->Fill(report, &meta);
        }
        for (auto& sensorId : sensorIds) { // no cached sensors fill zeros. cachedSensorIds is empty when sensorCache is empty
            if (std::find(cachedSensorIds.begin(), cachedSensorIds.end(), sensorId) != cachedSensorIds.end()) {
                continue;
            }
            NJson::TJsonValue& s = report.AppendValue(NJson::JSON_MAP);
            s["id"] = sensorId.Id;
            s["subid"] = sensorId.SubId;
            s["values"] = NJson::TJsonValue(NJson::JSON_ARRAY);
            if (meta && meta.contains(sensorId)) {
                s["sensor_meta"] = meta.at(sensorId);
            }
        }
        g.AddReportElement("sensors", std::move(report));
        if (!sensorCache) {
            g.SetCode(HTTP_NO_CONTENT);
            return;
        }
    }
    if (!lastActivity) {
        lastActivity = sensorCache->GetTimestamp();
        g.AddEvent(NJson::TMapBuilder
            ("event", "LastActivityFromSensorHistory")
        );
    }
    if (lastActivity) {
        g.AddReportElement("last_activity", lastActivity.Seconds());
    }
    {
        g.AddEvent(NJson::TMapBuilder
            ("event", "CorrectUntil")
            ("last_activity", lastActivity.Seconds())
            ("until", until.Seconds())
        );
        until = std::min(until, lastActivity);
    }
    {
        TDuration quantum = TDuration::Hours(1);
        NJson::TJsonValue activities = NJson::JSON_ARRAY;
        for (auto&& sensor : ActivitySensors) {
            if (!sensorIds.empty()) {
                auto p = std::find(sensorIds.begin(), sensorIds.end(), sensor);
                if (p == sensorIds.end()) {
                    continue;
                }
            }

            NJson::TJsonValue activity;
            activity["id"] = sensor.Id;
            activity["subid"] = sensor.SubId;
            NJson::TJsonValue values;
            if (auto last = sensors.GetValue()->GetTimestamp()) {
                values = NJson::JSON_ARRAY;
            }
            for (auto timestamp = since; timestamp < until; timestamp += quantum) {
                auto left = TInstant::Seconds((timestamp.Seconds() / quantum.Seconds()) * quantum.Seconds());
                auto right = left + quantum;
                auto start = std::max(left, since);
                auto finish = std::min(right, until);
                auto v = sensors.GetValue()->GetDuration(sensor, start, finish);
                if (!v) {
                    continue;
                }

                NJson::TJsonValue value;
                value["since"] = start.Seconds();
                value["timestamp"] = finish.Seconds();
                value["value"] = v.Seconds();
                values.AppendValue(std::move(value));
            }
            activity["values"] = std::move(values);
            activities.AppendValue(std::move(activity));
        }
        g.AddReportElement("activities", std::move(activities));
    }
    g.SetCode(HTTP_OK);
}

using namespace NDrivematics;

void TTelematicsPusherProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    const auto imei = GetString(requestData, "imei");
    const auto carId = DriveApi->GetCarIdByIMEI(imei);

    NDrive::TMultiSensor sensors;
    R_ENSURE(NJson::ParseField(requestData["sensors"], sensors), HTTP_BAD_REQUEST, "cannot parse sensors");

    g.AddEvent(NJson::TMapBuilder
        ("event", "input")
        ("id", carId)
        ("imei", imei)
        ("sensors", NJson::ToJson(sensors))
    );

    if (!carId) {
        g.SetCode(HTTP_OK);
        return;
    }

    const auto& settings = Server->GetSettings();
    auto tx = BuildTx<NSQL::Writable | NSQL::Deferred | NSQL::RepeatableRead>();
    bool changedSessionState = false;

    const auto& tagManager = DriveApi->GetTagsManager().GetDeviceTags();
    auto cachedObject = tagManager.GetObject(carId);
    R_ENSURE(cachedObject, HTTP_NOT_FOUND, "cannot GetObject " << carId, tx);

    TContext::TPtr contextSession = MakeAtomicShared<TContext>(carId, cachedObject->GetFirstTagByClass<TChargableTag>());
    for (auto&& sensor : sensors) {
        switch (sensor.Id) {
        case CAN_ENGINE_IS_ON:
            if (auto enableEngineSessions = settings.GetValue<bool>("telematics.engine_session.enable"); enableEngineSessions.GetOrElse(false)) {
                auto enableTaxiSessionsTagName = settings.GetValue<TString>("taxi.taxi_session.tag");
                auto enableEngineSessionsTagName = settings.GetValue<TString>("telematics.engine_session.tag");
                bool isEngineSession = (enableEngineSessionsTagName && cachedObject->HasTag(*enableEngineSessionsTagName));
                bool isTaxiSession = (enableTaxiSessionsTagName && cachedObject->HasTag(*enableTaxiSessionsTagName));
                if (!isEngineSession && !isTaxiSession) {
                    break;
                }

                if (!contextSession->GetChargableTag()) {
                    g.AddEvent(NJson::TMapBuilder
                        ("event", "NoChargableTag")
                        ("object_id", carId)
                    );
                    g.SetCode(HTTP_BAD_REQUEST);
                    return;
                }

                bool engineOn = sensor.ConvertTo<bool>();
                if (!engineOn && !contextSession->GetChargableTag()->GetPerformer()) {
                    g.AddEvent("FastSkip0");
                    continue;
                }

                if (engineOn && contextSession->GetChargableTag()->GetName() == TChargableTag::Riding) {
                    g.AddEvent("FastSkip1");
                    continue;
                }
                contextSession->SetChargableTag(
                    GetRestoredTag(carId, {
                        TChargableTag::Reservation,
                        TChargableTag::Riding,
                        TChargableTag::Parking
                    }, tagManager, tx)
                );

                if (IsOldTimestamp(contextSession, sensor.Timestamp.Get(), tagManager, tx)) {
                    g.AddEvent(NJson::TMapBuilder
                        ("event", "OldEngineSensor")
                        ("last_update", NJson::ToJson(contextSession->GetLastUpdatedAt()))
                        ("timestamp", NJson::ToJson(sensor.Timestamp))
                    );
                    continue;
                }

                const auto offer = DriveApi->GetCurrentCarOffer(contextSession->GetCarId());
                {
                    TString offerName;
                    if (offer) {
                        offerName = offer->GetName();
                    }

                    if (offerName.empty()) {
                        NDrive::TEventLog::Log("EmptyOfferName", NJson::TMapBuilder
                            ("id", carId)
                            ("tag_id", contextSession->GetChargableTag().GetTagId())
                        );
                    } else if (offerName == settings.GetValue<TString>("taxi.taxi_session.offer_name").GetOrElse("taxi_session")) {
                        g.AddEvent(NJson::TMapBuilder
                            ("event", "TelematicsPusherProcessorContinueWithTaxiOffer")
                            ("id", carId)
                            ("tag_id", contextSession->GetChargableTag().GetTagId())
                        );
                        if (contextSession->GetChargableTag()->GetPerformer() != permissions->GetUserId()) {
                            const auto performerPermissions = DriveApi->GetUserPermissions(contextSession->GetChargableTag()->GetPerformer());
                            R_ENSURE(performerPermissions, {}, "performerPermissions are missing", tx);
                            if (ProcessTaxiOffer(contextSession, performerPermissions, *Server, tx, contextSession->GetChargableTag(), isTaxiSession, engineOn)) {
                                changedSessionState = true;
                                continue;
                            }
                        } else {
                            g.AddEvent("NoMatchedUserPerformer");
                        }
                        continue;
                    }
                }

                if (!isEngineSession) {
                    break;
                }

                if (engineOn && contextSession->GetChargableTag()->GetPerformer().empty()) {
                    auto operatorPermissions = GetLinkedOperatorPermissions(carId, *Server, tx);
                    auto sessionOwnerPermissions = operatorPermissions ? operatorPermissions : permissions;
                    R_ENSURE(
                        CreateSession(contextSession, sessionOwnerPermissions, *Server, tx, "telematics.engine_session"),
                        {},
                        contextSession->GetErrorReport(),
                        tx
                    );
                    changedSessionState = true;
                    continue;
                }
                auto timeToRiding = settings.GetValue<TDuration>("telematics.engine_session.time_to_riding").GetOrElse(TDuration::Minutes(2));
                if (
                    engineOn &&
                    contextSession->GetChargableTag()->GetPerformer() == permissions->GetUserId() &&
                    contextSession->GetChargableTag()->GetName() == TChargableTag::Parking &&
                    sensor.GetSinceDelta(Now()) > timeToRiding
                ) {
                    R_ENSURE(
                        SwitchToRiding(contextSession, permissions, *Server, tx),
                        {},
                        contextSession->GetErrorReport(),
                        tx
                    );
                    changedSessionState = true;
                    continue;
                }
                auto timeToParking = settings.GetValue<TDuration>("telematics.engine_session.time_to_parking").GetOrElse(TDuration::Minutes(3));
                auto timeToFinish = settings.GetValue<TDuration>("telematics.engine_session.time_to_finish_session").GetOrElse(TDuration::Minutes(5));
                if (!engineOn && contextSession->GetChargableTag()->GetPerformer() == permissions->GetUserId()) {
                    if (contextSession->GetChargableTag()->GetName() == TChargableTag::Riding && sensor.GetSinceDelta(Now()) > timeToParking) {
                        R_ENSURE(
                            SwitchToParking(contextSession, permissions, *Server, tx),
                            {},
                            contextSession->GetErrorReport(),
                            tx
                        );
                        changedSessionState = true;
                        continue;
                    }
                    if (contextSession->GetChargableTag()->GetName() == TChargableTag::Parking && contextSession->GetLastUpdatedAt() < sensor.Timestamp - timeToFinish) {
                        R_ENSURE(
                            SwitchToReservation(contextSession, permissions, *Server, tx),
                            {},
                            contextSession->GetErrorReport(),
                            tx
                        );
                        changedSessionState = true;
                    }
                    continue;
                }
            }
            break;
        case VEGA_SPEED:
            if (auto enableGPSSessions = settings.GetValue<bool>("telematics.gps_session.enable"); enableGPSSessions.GetOrElse(false)) {
                auto enableGPSSessionsTagName = settings.GetValue<TString>("telematics.gps_session.tag");
                bool isGPSSession = (enableGPSSessionsTagName && cachedObject->HasTag(*enableGPSSessionsTagName));
                if (!isGPSSession) {
                    break;
                }
                bool liarEngineOn = sensor.SubId == CAN_ENGINE_IS_ON;
                if (!liarEngineOn) {
                    g.AddEvent("FastSkip3");
                    continue;
                }

                if (!contextSession->GetChargableTag()) {
                    g.AddEvent(NJson::TMapBuilder
                        ("event", "NoChargableTag")
                        ("object_id", carId)
                    );
                    g.SetCode(HTTP_BAD_REQUEST);
                    return;
                }

                if (liarEngineOn && contextSession->GetChargableTag()->GetName() == TChargableTag::Riding) {
                    g.AddEvent("FastSkip4");
                    continue;
                }

                contextSession->SetChargableTag(
                    GetRestoredTag(carId, {
                        TChargableTag::Reservation,
                        TChargableTag::Riding
                    }, tagManager, tx)
                );

                if (IsOldTimestamp(contextSession, sensor.Timestamp, tagManager, tx)) {
                    g.AddEvent(NJson::TMapBuilder
                        ("event", "OldGPSSensor")
                        ("last_update", NJson::ToJson(contextSession->GetLastUpdatedAt()))
                        ("timestamp", NJson::ToJson(sensor.Timestamp))
                    );
                    break;
                }

                if (liarEngineOn) {
                    if (contextSession->GetChargableTag()->GetPerformer().empty()) {
                        auto operatorPermissions = GetLinkedOperatorPermissions(carId, *Server, tx);
                        auto sessionOwnerPermissions = operatorPermissions ? operatorPermissions : permissions;
                        R_ENSURE(
                            CreateSession(contextSession, sessionOwnerPermissions, *Server, tx, "telematics.gps_session"),
                            {},
                            contextSession->GetErrorReport(),
                            tx
                        );
                        changedSessionState = true;
                        continue;
                    }
                }
            }
            break;
        default:
            break;
        }
    }
    R_ENSURE(tx.Commit(), {}, "cannot Commit", tx);
    g.SetCode(changedSessionState ? HTTP_ACCEPTED : HTTP_OK);
}

void TTelematicsEventPusherProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    const auto imei = GetString(requestData, "imei");
    const auto id = DriveApi->GetCarIdByIMEI(imei);

    NDrive::TTelematicsEvents events;
    R_ENSURE(NJson::ParseField(requestData["events"], events), HTTP_BAD_REQUEST, "cannot parse events");

    g.AddEvent(NJson::TMapBuilder
        ("event", "input")
        ("id", id)
        ("imei", imei)
        ("events", NJson::ToJson(events))
    );

    if (!id) {
        g.SetCode(HTTP_OK);
        return;
    }

    const auto& settings = Server->GetSettings();
    auto tx = BuildTx<NSQL::Writable | NSQL::Deferred | NSQL::RepeatableRead>();
    bool changed = false;
    for (auto&& event : events) {
        switch (event.Type) {
            case NDrive::TTelematicsEvent::EType::Aggression:
            if (auto enableRealtimeAgression = settings.GetValue<bool>("telematics.realtime_agression.enable"); enableRealtimeAgression.GetOrElse(false)) {
                auto matcher = NDrive::NSession::TMatchingConstraintsGroup::Construct(Server, {}, { "violation_during_session" });
                R_ENSURE(matcher, HTTP_INTERNAL_SERVER_ERROR, "cannot create matcher", tx);
                auto matched = matcher->MatchSession(id, event.Timestamp);
                if (!matched) {
                    g.AddEvent(NJson::TMapBuilder
                        ("event", "UnmatchedAggressionEvent")
                        ("timestamp", NJson::ToJson(event.Timestamp))
                    );
                    continue;
                }
                const auto& sessionId = matched->GetSessionId();
                if (!sessionId) {
                    g.AddEvent(NJson::TMapBuilder
                        ("event", "OrphanAggressionEvent")
                        ("matched", NJson::ToJson(matched))
                        ("timestamp", NJson::ToJson(event.Timestamp))
                    );
                    continue;
                }
                auto realtimeAgressionTagName = settings.GetValue<TString>("telematics.realtime_agression.tag");
                if (!realtimeAgressionTagName) {
                    continue;
                }
                const auto& tagManager = DriveApi->GetTagsManager().GetTraceTags();
                auto optionalTags = tagManager.RestoreTags(TVector{ sessionId }, { *realtimeAgressionTagName }, tx);
                R_ENSURE(optionalTags, {}, "cannot RestoreTags for " << sessionId, tx);
                auto tag = optionalTags->empty()
                    ? DriveApi->GetTagsManager().GetTagsMeta().CreateTag(*realtimeAgressionTagName)
                    : optionalTags->at(0).GetData();
                R_ENSURE(tag, HTTP_INTERNAL_SERVER_ERROR, "cannot CreateTag " << realtimeAgressionTagName, tx);
                auto impl = std::dynamic_pointer_cast<TScoringTraceTag>(tag);
                R_ENSURE(impl, HTTP_INTERNAL_SERVER_ERROR, "cannot cast " << tag->GetName() << " as ScoringTraceTag", tx);
                TScoringTraceTag::TEvent ev;
                ev.Timestamp = event.Timestamp;
                if (event.Location) {
                    ev.Location = event.Location->GetCoord();
                }
                if (impl->AddEvent(std::move(ev))) {
                    auto added = tagManager.AddTag(tag, permissions->GetUserId(), sessionId, Server, tx);
                    R_ENSURE(added, {}, "cannot AddTag", tx);
                    changed = true;
                }
            }
            case NDrive::TTelematicsEvent::EType::Unknown:
                break;
        }
    }
    R_ENSURE(tx.Commit(), {}, "cannot Commit", tx);
    g.SetCode(changed ? HTTP_ACCEPTED : HTTP_OK);
}

void TTelematicsConfigurationGetProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    const TCgiParameters& cgi = Context->GetCgiParameters();

    TString carId = GetString(cgi, "car_id", false);
    const auto& imei = GetString(cgi, "imei", false);
    auto timeout = GetDuration(cgi, "timeout", TDuration::Seconds(20));
    R_ENSURE(carId || imei, HTTP_BAD_REQUEST, "either car_id or imei should be present");
    R_ENSURE(carId.empty() || imei.empty(), HTTP_BAD_REQUEST, "only one of car_id or imei should be present");

    if (!carId) {
        carId = DriveApi->GetCarIdByIMEI(imei);
    }
    R_ENSURE(carId, ConfigHttpStatus.EmptySetStatus, "car_id is missing");

    CheckVisibility(carId, permissions, *Server);
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::ModifyStructure, TAdministrativeAction::EEntity::Car, carId);

    auto sensorIds = GetValues<NDrive::TSensorId>(requestData, "sensors", false);



    bool withValues = false;
    if (cgi.Has("with_values")) {
        withValues = IsTrue(cgi.Get("with_values"));
    }

    auto filter = MakeSet(GetValues<ui16>(requestData, "filter", false));
    TTelematicsConfigurationTag::TCustom result;

    {
        auto handler = TTelematicsConfigurationTag::GetActualConfiguration(carId, Server, timeout);
        handler.GetFuture().Wait();

        auto settingFile = handler.GetResponse<NDrive::TTelematicsClient::TDownloadFileResponse>();
        R_ENSURE(
            handler.GetStatus() == NDrive::TTelematicsClient::EStatus::Success && settingFile,
            HTTP_INTERNAL_SERVER_ERROR,
            "cannot download setting file"
        );

        NDrive::NVega::TSettings settings;

        settings.Parse(settingFile->Data);
        for (const auto& setting : settings) {
            if (!filter.empty() && !filter.contains(setting.Id)) {
                continue;
            }
            result.Apply(setting);
        }
    }

    if (withValues) {
        TSet<NDrive::TSensorId> ids;
        if (!sensorIds.empty()) {
            ids.insert(sensorIds.begin(), sensorIds.end());
        }

        for (const auto& id : result.GetSensors()) {
            ids.insert(id.first);
        }

        auto handlers = TTelematicsConfigurationTag::GetActualValues(carId, ids, Server, timeout);

        NThreading::TFutures<void> waiters;
        waiters.reserve(handlers.size());
        for (auto&& handler : handlers) {
            waiters.push_back(handler.GetFuture());
        }

        NThreading::WaitAll(waiters).Wait(timeout);
        for (auto&& handler : handlers) {
            if (handler.GetStatus() != NDrive::TTelematicsClient::EStatus::Success) {
                continue;
            }

            auto value = handler.GetResponse<NDrive::TTelematicsClient::TGetParameterResponse>();
            if (!value) {
                continue;
            }

            result.Apply(*value);
        }
        result.PostLoad();
    }

    if (!result.GetModel()) {
        result.MutableModel() = GetModel(carId, *Server);
    }

    g.MutableReport().AddReportElement("settings", NJson::ToJson(result));
    g.SetCode(HTTP_OK);
}

void TTelematicsConfigurationSetProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    const TCgiParameters& cgi = Context->GetCgiParameters();

    TString carId = GetString(cgi, "car_id", false);
    const auto& imei = GetString(cgi, "imei", false);
    R_ENSURE(carId || imei, HTTP_BAD_REQUEST, "either car_id or imei should be present");
    R_ENSURE(carId.empty() || imei.empty(), HTTP_BAD_REQUEST, "only one of car_id or imei should be present");

    if (!carId) {
        carId = DriveApi->GetCarIdByIMEI(imei);
    }
    R_ENSURE(carId, ConfigHttpStatus.EmptySetStatus, "car_id is missing");

    CheckVisibility(carId, permissions, *Server);
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::ModifyStructure, TAdministrativeAction::EEntity::Car, carId);

    TTelematicsConfigurationTag::TCustom settings;
    bool isGlobal = false;

    R_ENSURE(NJson::ParseField(requestData["is_global"], isGlobal), HTTP_BAD_REQUEST, "cannot parse is_global");
    R_ENSURE(NJson::ParseField(requestData["settings"], settings), HTTP_BAD_REQUEST, "cannot parse settings");

    if (!settings.GetModel()) {
        settings.MutableModel() = GetModel(carId, *Server);
    }

    const auto& tagsManager = Server->GetDriveAPI()->GetTagsManager();
    const auto& deviceTagsManager = tagsManager.GetDeviceTags();
    const auto& tagsMeta = tagsManager.GetTagsMeta();

    auto taggedObject = Server->GetDriveDatabase().GetTagsManager().GetDeviceTags().GetObject(carId);
    auto tag = taggedObject->GetFirstTagByClass<TTelematicsConfigurationTag>();
    auto tx = Server->GetDriveAPI()->template BuildTx<NSQL::Writable>();

    if (isGlobal) {
        TString tagName = tag ? tag->GetName() : TTelematicsConfigurationTag::Type();
        auto descriptionPtr = tagsMeta.GetDescriptionByName(tagName);
        R_ENSURE(descriptionPtr, HTTP_INTERNAL_SERVER_ERROR, "cannot get description from " << tagName);
        auto description = std::dynamic_pointer_cast<const TTelematicsConfigurationTag::TDescription>(descriptionPtr);
        if (!description) {
            description = MakeAtomicShared<TTelematicsConfigurationTag::TDescription>();
        }

        auto copy = description->Clone();
        auto descriptionCopy = std::dynamic_pointer_cast<TTelematicsConfigurationTag::TDescription>(copy);
        R_ENSURE(descriptionCopy, HTTP_INTERNAL_SERVER_ERROR, "cannot cast copy of TTelematicsConfigurationTag");
        descriptionCopy->Merge(settings);

        R_ENSURE(tagsMeta.RegisterTag(descriptionCopy, permissions->GetUserId(), tx), HTTP_INTERNAL_SERVER_ERROR, "cannot register tag ", tx);
        R_ENSURE(tx.Commit(), HTTP_INTERNAL_SERVER_ERROR, "cannot regiter tag ", tx);
    } else {
        auto configurationTag = std::dynamic_pointer_cast<TTelematicsConfigurationTag>(tag.GetData());
        if (!configurationTag) {
            configurationTag = MakeAtomicShared<TTelematicsConfigurationTag>();
        }
        configurationTag->Merge(settings);
        configurationTag->SetApplyOnAdd(true);

        R_ENSURE(deviceTagsManager.AddTag(configurationTag, permissions->GetUserId(), carId, Server, tx), HTTP_INTERNAL_SERVER_ERROR, "cannot update tag data ", tx);
        R_ENSURE(tx.Commit(), HTTP_INTERNAL_SERVER_ERROR, "cannot regiter tag ", tx);
    }

    g.SetCode(HTTP_OK);
}
