#include "traces_info.h"

#include <drive/backend/data/alerts/traces.h>

#include <drive/backend/saas/api.h>

#include <library/cpp/string_utils/quote/quote.h>

#include <rtline/protos/report_accessor.h>

#include <util/string/type.h>

IRequestProcessor::TPtr TTracesInfoProcessorConfig::DoConstructAuthProcessor(IReplyContext::TPtr context, IAuthModule::TPtr authModule, const IServerBase* server) const {
    return new TTracesInfoProcessor(*this, context, authModule, server);
}

void TTracesInfoProcessorConfig::DoInit(const TYandexConfig::Section* section) {
    TBase::DoInit(section);
    TracksApiName = section->GetDirectives().Value("TracksApiName", TracksApiName);
    AssertCorrectConfig(!!TracksApiName, "Incorrect 'TracksApiName' field in configuration '%s'", Name.data());
}

void TTracesInfoProcessorConfig::ToString(IOutputStream& os) const {
    TBase::ToString(os);
    os << "TracksApiName: " << TracksApiName << Endl;
}

void TTracesInfoProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    const TCgiParameters& cgi = Context->GetCgiParameters();
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::Car);
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::User);

    const TString DeviceId = cgi.Get("car");
    const TString RideId = cgi.Get("ride");
    const TString UserId = cgi.Get("user");
    const TString SessionId = cgi.Get("session");
    R_ENSURE(DeviceId || RideId || UserId || SessionId, HTTP_BAD_REQUEST, "at least one parameter of car,ride,user,session is required");

    auto tracksApiName = GetHandlerSetting<TString>("tracks_api").GetOrElse(Config.GetTracksApiName());

    TJsonReport& report = g.MutableReport();

    TTracesAccessor ta(Server, tracksApiName);
    TString text;
    if (DeviceId) {
        text += Sprintf("s_device_id:\"%s\" ", DeviceId.data());
    }
    if (RideId) {
        text += Sprintf("url:\"%s\" ", RideId.data());
    }
    if (UserId) {
        text += Sprintf("s_user_id:\"%s\" ", UserId.data());
    }
    if (SessionId) {
        text += Sprintf("s_session_id:\"%s\" ", SessionId.data());
    }
    ta.SetOverrideText(text).SetNeedTrace(true);
    TMessagesCollector errors;
    if (!ta.Execute(errors, TDuration::Zero())) {
        report.AddReportElement("error", errors.GetReport());
        g.SetCode(ConfigHttpStatus.UnknownErrorStatus);
        return;
    }

    NJson::TJsonValue reportTraces;
    for (auto&& i : ta.GetTraces()) {
        reportTraces.AppendValue(i.GetReport());
    }
    report.AddReportElement("tracks", std::move(reportTraces));
    g.SetCode(HTTP_OK);
}

IRequestProcessorConfig::TFactory::TRegistrator<TTracesInfoProcessorConfig> TTracesInfoProcessorConfig::Registrator("traces_info");
