#include "traces_search.h"

#include <drive/backend/data/alerts/traces.h>

#include <drive/backend/compiled_riding/manager.h>
#include <drive/backend/saas/api.h>

#include <library/cpp/string_utils/quote/quote.h>

#include <rtline/library/geometry/coord.h>
#include <rtline/library/geometry/rect.h>
#include <rtline/protos/report_accessor.h>
#include <rtline/util/types/interval.h>

#include <util/string/type.h>

IRequestProcessor::TPtr TTracesSearchProcessorConfig::DoConstructAuthProcessor(IReplyContext::TPtr context, IAuthModule::TPtr authModule, const IServerBase* server) const {
    return new TTracesSearchProcessor(*this, context, authModule, server);
}

void TTracesSearchProcessorConfig::DoInit(const TYandexConfig::Section* section) {
    TBase::DoInit(section);
    TracksApiName = section->GetDirectives().Value("TracksApiName", TracksApiName);
    AssertCorrectConfig(!!TracksApiName, "Incorrect 'TracksApiName' field in configuration '%s'", Name.data());
}

void TTracesSearchProcessorConfig::ToString(IOutputStream& os) const {
    TBase::ToString(os);
    os << "TracksApiName: " << TracksApiName << Endl;
}

void TTracesSearchProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    const TCgiParameters& cgi = Context->GetCgiParameters();
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::Car);
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::User);

    const TString& model = cgi.Get("model");
    const TString& sTimeSince = cgi.Has("since") ? cgi.Get("since") : cgi.Get("time");
    const TString& sTimeUntil = cgi.Get("until");
    const TDuration gTimeout = cgi.Has("g_timeout") ? FromString<TDuration>(cgi.Get("g_timeout")) : TDuration::Seconds(20);
    const TDuration rTimeout = cgi.Has("r_timeout") ? FromString<TDuration>(cgi.Get("r_timeout")) : TDuration::Seconds(2);
    const ui32 packSize = cgi.Has("pack_size") ? FromString<ui32>(cgi.Get("pack_size")) : 300;
    const TString sessionId = cgi.Get("session_id");
    TMap<TString, TString> orders;
    TSet<TString> cars;
    TInterval<ui32> intervalTime;
    auto session = BuildTx<NSQL::ReadOnly>();
    {
        const TInstant tsSince = TInstant::ParseIso8601(sTimeSince);

        TInstant tsUntil = tsSince + TDuration::Minutes(1);
        if (!!sTimeUntil) {
            tsUntil = TInstant::ParseIso8601(sTimeUntil);
        }
        intervalTime.SetMin(tsSince.Seconds());
        intervalTime.SetMax(tsUntil.Seconds());

        auto queryOptions = NSQL::TQueryOptions();
        if (sessionId) {
            queryOptions.AddGenericCondition("session_id", sessionId);
        }
        auto compiledSessions = DriveApi->GetMinimalCompiledRides().GetEvents<TMinimalCompiledRiding>({}, tsSince, session, queryOptions);
        R_ENSURE(compiledSessions, {}, "cannot GetMinimalCompiledRidings", session);

        for (auto&& i : *compiledSessions) {
            if (sessionId && i.GetSessionId() != sessionId) {
                continue;
            }
            if (i.GetStartInstant() > tsUntil) {
                continue;
            } else {
                orders.emplace(i.GetSessionId(), i.TCompiledRiding::GetObjectId());
                cars.emplace(i.TCompiledRiding::GetObjectId());
            }
        }
    }
    TVector<TGeoCoord> coords;
    R_ENSURE(TGeoCoord::DeserializeVector(cgi.Get("coords"), coords) && !coords.empty(), HTTP_BAD_REQUEST, "cgi parameter 'coords' is incorrect");
    ::TRect<TGeoCoord> rect(coords.front());
    rect.ExpandTo(coords);

    TJsonReport& report = g.MutableReport();

    NJson::TJsonValue interestOrders = NJson::JSON_ARRAY;
    NJson::TJsonValue problemOrders = NJson::JSON_ARRAY;
    TMap<TString, NRTLine::TSearchReply> results;
    const TRTLineAPI* configApi =  Server->GetRTLineAPI(GetHandlerSettingDef<TString>("tracks_api", Config.GetTracksApiName()));
    R_ENSURE(configApi, HTTP_INTERNAL_SERVER_ERROR, "Incorrect configuration (api name)");

    TInstant start = Now();
    while (results.size() != orders.size() && Now() - start < gTimeout) {
        ui32 idx = 0;
        for (auto&& i : orders) {
            if (!results.contains(i.first)) {
                NRTLine::TQuery query;
                query.SetText(Sprintf("s_session_id:\"%s\" ", i.first.data())).SetTimeout(rTimeout).SetNumDoc(10000);
                results.emplace(i.first, configApi->GetSearchClient().SendAsyncQuery(query, rTimeout));
                ++idx;
            }
            if (idx > packSize) {
                break;
            }
        }

        for (auto&& i : orders) {
            auto it = results.find(i.first);
            if (it != results.end()) {
                if (it->second.GetCode() != 200) {
                    results.erase(i.first);
                }
            }
        }
    }

    TVector<TTraceInfo> infos;
    {
        for (auto&& i : orders) {
            auto it = results.find(i.first);
            if (it == results.end()) {
                problemOrders.AppendValue("code:" + i.first);
            } else {
                if (!TTraceInfo::ParseFromReply(it->second, infos)) {
                    problemOrders.AppendValue("parse:" + i.first);
                }
            }
        }
    }

    auto gCars = Server->GetDriveAPI()->GetCarsData()->FetchInfo(cars, session);
    TSet<TString> duplicationsChecker;
    for (auto&& t : infos) {
        if (duplicationsChecker.contains(t.GetSessionId())) {
            continue;
        }
        auto* carInfo = gCars.GetResultPtr(t.GetDeviceId());
        if (!carInfo) {
            problemOrders.AppendValue("c:" + t.GetSessionId());
            continue;
        }
        if (!!model && carInfo && carInfo->GetModel() != model) {
            continue;
        }

        for (ui32 p = 0; p + 1 < t.GetCoords().size(); ++p) {
            ::TRect<TGeoCoord> rLocal(t.GetCoords()[p], t.GetCoords()[p + 1]);
            TInterval<ui32> intervalSegment(t.GetTimestamps()[p], t.GetTimestamps()[p + 1]);
            if (rLocal.Cross(rect) && intervalTime.Intersection(intervalSegment)) {
                if (intervalSegment.GetLength() && (t.GetCoords()[p].GetLengthTo(t.GetCoords()[p + 1]) / intervalSegment.GetLength()) > 250 / 3.6) {
                    continue;
                }
                interestOrders.AppendValue(TCarsharingUrl().TrackPage(t.GetSessionId()));
                duplicationsChecker.emplace(t.GetSessionId());
                break;
            }
        }
    }

    report.AddReportElement("interest", std::move(interestOrders));
    report.AddReportElement("problem", std::move(problemOrders));
    report.AddReportElement("traces_count", orders.size());
    report.AddReportElement("traces_count_result", results.size());
    g.SetCode(HTTP_OK);
}

IRequestProcessorConfig::TFactory::TRegistrator<TTracesSearchProcessorConfig> TTracesSearchProcessorConfig::Registrator("traces_search");
