#include "analyzer.h"

#include <drive/backend/processors/common_app/fetcher.h>

#include <drive/backend/compiled_riding/manager.h>
#include <drive/backend/data/area_tags.h>
#include <drive/backend/logging/events.h>
#include <drive/backend/processors/sessions/violations_helper.h>
#include <drive/backend/saas/api.h>
#include <drive/backend/tracks/client.h>
#include <drive/backend/tracks/report.h>

#include <drive/library/cpp/maps_router/linker.h>
#include <drive/library/cpp/tracks/client.h>

#include <library/cpp/string_utils/quote/quote.h>

#include <util/string/type.h>

using namespace NDrive;

IRequestProcessor::TPtr TAnalyzerProcessorConfig::DoConstructAuthProcessor(IReplyContext::TPtr context, IAuthModule::TPtr authModule, const IServerBase* server) const {
    return new TAnalyzerProcessor(*this, context, authModule, server);
}

void TAnalyzerProcessorConfig::DoInit(const TYandexConfig::Section* section) {
    TBase::DoInit(section);
    TracksApiName = section->GetDirectives().Value("TracksApiName", TracksApiName);
    AssertCorrectConfig(!!TracksApiName, "Incorrect 'TracksApiName' field in configuration '%s'", Name.data());

    LinkerApiName = section->GetDirectives().Value("LinkerApiName", LinkerApiName);
    AssertCorrectConfig(!!LinkerApiName, "Incorrect 'LinkerApiName' field in configuration '%s'", Name.data());

    ShouldForceUserId = section->GetDirectives().Value("ForceUserId", ShouldForceUserId);
}

void TAnalyzerProcessorConfig::ToString(IOutputStream& os) const {
    TBase::ToString(os);
    os << "TracksApiName: " << TracksApiName << Endl;
    os << "LinkerApiName: " << LinkerApiName << Endl;
    os << "ForceUserId: " << ShouldForceUserId << Endl;
}

void TAnalyzerProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    const TCgiParameters& cgi = Context->GetCgiParameters();
    const TString DeviceId = cgi.Has("car_id") ? cgi.Get("car_id") : cgi.Get("car");
    const TString RideId = cgi.Get("ride");
    const TString UserId = cgi.Get("user_id");
    const TString SessionId = cgi.Has("session_id") ? cgi.Get("session_id") : cgi.Get("session");
    R_ENSURE(DeviceId || RideId || SessionId || UserId, ConfigHttpStatus.SyntaxErrorStatus, "at least one parameter of car_id, ride, user_id, session_id is required");

    const TString format = GetString(cgi, "format", false);
    TInstant since = GetTimestamp(cgi, "since", TInstant::Zero());
    const TInstant until = GetTimestamp(cgi, "until", TInstant::Max());

    const TString verbosity = GetString(cgi, "verbose", false);  // presence of the field means that extra information about order shall be reported; used in chats

    TMaybe<NDrive::ECarStatus> status;
    if (!cgi.Has("status")) {
        status = NDrive::ECarStatus::csRide;
    } else if (const TString& s = cgi.Get("status")) {
        status = ParseValue<NDrive::ECarStatus>(s);
    }

    auto lengthThreshold = GetValue<double>(cgi, "length_threshold", false);
    auto speedThreshold = GetValue<double>(cgi, "speed_threshold", false);
    auto speedLimitLow = GetValue<double>(cgi, "speed_limit_low", false);
    auto filterThreshold = GetValue<double>(cgi, "filter_threshold", false);
    auto splitThreshold = GetValue<double>(cgi, "split_threshold", false);
    auto filterSpan = GetValue<double>(cgi, "filter_span", false);
    auto numdocDefault = SessionId ? 1000 : 10;
    auto numdoc = GetValue<ui32>(cgi, "numdoc", false).GetOrElse(numdocDefault);
    auto segmentSize = GetValue<ui32>(cgi, "segment_size", false);
    auto d = GetValue<ui32>(cgi, "d", false);

    auto speedThresholdsString = GetValue<TString>(cgi, "speed_thresholds", false).OrElse(GetHandlerSetting<TString>("analyzer.speed_thresholds"));
    auto speedThresholdsJson = speedThresholdsString ? NJson::ToJson(NJson::JsonString(*speedThresholdsString)) : NJson::JSON_NULL;
    auto speedThresholds = NJson::TryFromJson<TSpeedThresholds>(speedThresholdsJson);

    auto deadline = Context->GetRequestDeadline();
    auto locale = GetLocale();
    auto tracksService = GetString(cgi, "tracks-service", false);
    auto linkerService = GetString(cgi, "linker-service", false);

    bool needGeocodedLocation = GetValue<bool>(cgi, "geocoded", false).GetOrElse(false);

    bool useRawTracks =
        GetValue<bool>(cgi, "use_raw_tracks", false)
            .GetOrElse(GetHandlerSetting<bool>("use_raw_tracks")
                           .GetOrElse(false));

    TJsonReport& report = g.MutableReport();

    const auto handlerTracksApiName = GetHandlerSetting<TString>("tracks_api");
    const TString& tracksApiName = tracksService ? tracksService : handlerTracksApiName.GetOrElse(Config.GetTracksApiName());
    auto trackClient = CreateTrackClient(tracksApiName, *Server);
    R_ENSURE(trackClient, HTTP_INTERNAL_SERVER_ERROR, "cannot CreateTrackClient");

    const auto handlerLinkerApiName = GetHandlerSetting<TString>("linker_api");
    const TString& linkerApiName = linkerService ? linkerService : handlerLinkerApiName.GetOrElse(Config.GetLinkerApiName());
    const auto linker = Server->GetLinker(linkerApiName);
    R_ENSURE(linker, ConfigHttpStatus.UserErrorState, "cannot find linker-service " << linkerApiName);

    NDrive::TTracksLinker::TOptions tracksLinkerOptions;
    if (filterSpan) {
        tracksLinkerOptions.FilterSpan = *filterSpan;
    }
    if (filterThreshold) {
        tracksLinkerOptions.FilterThreshold = *filterThreshold;
    }
    if (d) {
        tracksLinkerOptions.Precision = *d;
    }
    if (segmentSize) {
        tracksLinkerOptions.SegmentSize = *segmentSize;
    }
    if (splitThreshold) {
        tracksLinkerOptions.SplitThreshold = *splitThreshold;
    }
    if (lengthThreshold) {
        tracksLinkerOptions.ViolationOptions.CriticalLength = *lengthThreshold;
    }
    if (speedThreshold) {
        tracksLinkerOptions.ViolationOptions.SpeedThreshold = *speedThreshold;
    }
    if (speedThresholds) {
        tracksLinkerOptions.ViolationOptions.SpeedThresholds = *speedThresholds;
    }
    if (speedLimitLow) {
        tracksLinkerOptions.ViolationOptions.SpeedLimitLow = *speedLimitLow;
    }

    NDrive::TTrackQuery query;
    query.DeviceId = DeviceId;
    query.RideId = RideId;
    query.UserId = UserId;
    query.SessionId = SessionId;
    query.Since = since;
    query.Until = until;
    query.NumDoc = numdoc;
    query.Status = status;

    /* Here we are going to get linked tracks from YDB */
    TTracksLinker::TResults linkedTracks;
    auto useYDB = GetHandlerSetting<bool>("tracks.use_ydb_client").GetOrElse(false);
    if (useYDB) {
        TYDBTracksClient ydbClient(*Server, tracksLinkerOptions);
        linkedTracks = ydbClient.GetYDBTracks(query);
    }

    if (auto prelinkedRange = TTrackOps::GetLinkedTracksRange(linkedTracks); prelinkedRange) {
        if (prelinkedRange->To.GetRef() > since) {
            query.Since = prelinkedRange->To.GetRef();
        }
    }

    g.AddEvent(NJson::TMapBuilder
        ("event", "TrackRequest")
        ("query", NJson::ToJson(query))
    );
    auto asyncTracks = trackClient->GetTracks(query, deadline - Now());
    R_ENSURE(asyncTracks.Wait(deadline), ConfigHttpStatus.TimeoutStatus, "track request timeout");
    R_ENSURE(asyncTracks.HasValue(), ConfigHttpStatus.IncompleteStatus, "track request exception: " << NThreading::GetExceptionMessage(asyncTracks));
    TTracks tracks = asyncTracks.GetValue();

    g.AddEvent(NJson::TMapBuilder
        ("event", "TrackResponse")
        ("size", tracks.size())
    );

    if (Config.ForceUserId()) {
        for (auto&& track : tracks) {
            if (track.UserId != permissions->GetUserId()) {
                ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::ObserveStructure, TAdministrativeAction::EEntity::User, track.UserId);
            }
        }
    }
    if (useRawTracks) {
        g.AddEvent("RawTracks");

        auto result = ToLegacyFormat(tracks);
        report.AddReportElement("tracks", std::move(result));

        g.SetCode(HTTP_OK);
        return;
    }

    auto r = NDrive::CreateAnalyzerReport(format);
    R_ENSURE(r, ConfigHttpStatus.UserErrorState, "unknown format: " << format);

    if (r->UseDirectOrder()) {
        std::reverse(tracks.begin(), tracks.end());
    }

    if (!!verbosity) {
        const TMinimalRidingHistoryManager& compiledRides = DriveApi->GetMinimalCompiledRides();
        TOptionalObjectEvents<TFullCompiledRiding> compiledSessions;

        auto tx = BuildTx<NSQL::ReadOnly>();
        auto ydbTx = BuildYdbTx<NSQL::ReadOnly>("analyzer");

        if (SessionId) {
            compiledSessions = compiledRides.Get<TFullCompiledRiding>({ SessionId }, tx, ydbTx);
        } else if (UserId) {
            compiledSessions = compiledRides.GetUser<TFullCompiledRiding>(UserId, tx, ydbTx);
        } else if (DeviceId) {
            compiledSessions = compiledRides.GetObject<TFullCompiledRiding>({ UserId }, tx, ydbTx);
        }
        R_ENSURE(compiledSessions, {}, "cannot GetFullCompiledRiding", tx);

        TSet<TString> foundTrackIds;
        for (auto&& track : tracks) {
            foundTrackIds.insert(track.SessionId);
        }

        TSet<TString> carIds;
        TVector<TTrack> emptyTracks;
        for (auto&& ride : *compiledSessions) {
            if (SessionId && ride.GetSessionId() != SessionId) {
                continue;
            }
            if (UserId && ride.GetHistoryUserId() != UserId) {
                continue;
            }

            auto carId = ride.TFullCompiledRiding::GetObjectId();
            if (DeviceId && carId != DeviceId) {
                continue;
            }
            carIds.insert(carId);

            if (!foundTrackIds.contains(ride.GetSessionId())) {
                TTrack emptyTrack;
                emptyTrack.SessionId = ride.GetSessionId();
                emptyTrack.DeviceId = carId;
                emptyTrack.UserId = ride.GetHistoryUserId();
                foundTrackIds.insert(emptyTrack.SessionId);
                emptyTracks.push_back(std::move(emptyTrack));
            }
        }

        TCarsFetcher fetcher(*Server, NDeviceReport::ReportSession);
        fetcher.SetIsRealtime(false);
        fetcher.SetLocale(locale);
        {
            TTimeGuard tgFetch("Fetcher");
            if (!fetcher.FetchData(permissions, carIds)) {
                g.AddEvent("FetchDataError");
            }
        }

        NJson::TJsonValue sessions;
        for (auto&& ride : *compiledSessions) {
            auto sessionId = ride.GetSessionId();
            if (!foundTrackIds.contains(sessionId)) {
                continue;
            }
            auto carId = ride.TFullCompiledRiding::GetObjectId();

            NJson::TJsonValue sessionReport;
            NJson::TJsonValue carReport;
            if (fetcher.BuildOneCarInfoReport(carId, permissions->GetFilterActions(), carReport)) {
                g.AddEvent(NJson::TMapBuilder
                    ("event", "BuildOneCarInfoReportError")
                    ("object_id", carId)
                    ("session_id", sessionId)
                );
            }
            sessionReport.InsertValue("car", std::move(carReport));
            sessionReport.InsertValue("segment", ride.GetReport(locale, NDriveSession::ReportUserApp, *Server));

            sessions.AppendValue(std::move(sessionReport));
        }

        report.AddReportElement("views", fetcher.GetViewsReportSafe());
        report.AddReportElement("models", fetcher.GetModelsReportSafe(/*asMap=*/true));
        report.AddReportElement("sessions", std::move(sessions));

        for (auto&& track : emptyTracks) {
            r->OnStartTrack(track);
            r->OnFinishTrack(track);
        }
    }

    auto helper = MakeHolder<NDrive::TViolationsHelper>(*Server);
    auto linkedFutures = TVector<std::pair<TString, NThreading::TFuture<NDrive::TTracksLinker::TResult>>>();
    NDrive::TTracksLinker tracksLinker(linker, tracksLinkerOptions);

    for (auto&& track : tracks) {
        auto name = track.Name;
        g.AddEvent(NJson::TMapBuilder
            ("event", "LinkingTrack")
            ("name", name)
        );
        auto linked = tracksLinker.Link(std::move(track)).Apply([server = Server, &locale, &deadline, &helper, &needGeocodedLocation](const NThreading::TFuture<NDrive::TTracksLinker::TResult> result) -> NDrive::TTracksLinker::TResult {
            auto correctedTrack = result.GetValue();
            TSpeedLimitCorrectionAreaTag::CorrectSpeedLimitRange(correctedTrack, *server);

            if (needGeocodedLocation) {
                helper->FetchGeocodedRangeCenter(correctedTrack, locale, deadline);
            }
            return correctedTrack;
        });
        linkedFutures.emplace_back(name, linked);
    }

    bool complete = true;
    {
        for (auto&& [name, linked] : linkedFutures) {
            if (!linked.Wait(deadline)) {
                complete = false;
                g.AddEvent(NJson::TMapBuilder
                    ("event", "LinkingWaitTimeout")
                    ("name", name)
                );
                continue;
            }
            if (!linked.HasValue()) {
                complete = false;
                g.AddEvent(NJson::TMapBuilder
                    ("event", "LinkingWaitError")
                    ("exception", NThreading::GetExceptionInfo(linked))
                    ("name", name)
                );
                continue;
            }
            const auto& l = linked.ExtractValue();
            linkedTracks.push_back(std::move(l));
        }

        NDrive::TTrackOps::SortLinkedTracks(linkedTracks);
    }

    for (const auto& linked : linkedTracks) {
        g.AddEvent(NJson::TMapBuilder
            ("event", "ProcessingTrack")
            ("name", linked.Track.Name)
        );
        r->OnStartTrack(linked.Track);
        for (size_t i = 0; i < linked.Segments.size(); ++i) {
            g.AddEvent(NJson::TMapBuilder
                ("event", "ProcessingSegment")
                ("index", i)
            );

            const NDrive::TTracksLinker::TSegment& segment = linked.Segments[i];
            const NGraph::TRouter::TTimedGeoCoordinates& coordinates = segment.Coordinates;
            const NGraph::TRouter::TMatch& match = segment.Linked;

            r->OnStartSegment(match, coordinates);
            for (auto&& range : segment.Processed) {
                r->OnSegmentRange(range, coordinates);
            }
            r->OnFinishSegment(match, coordinates);
        }
        r->OnFinishTrack(linked.Track);
    }
    r->Fill(report);
    if (complete) {
        g.SetCode(HTTP_OK);
    } else {
        g.SetCode(ConfigHttpStatus.IncompleteStatus);
    }
}

IRequestProcessorConfig::TFactory::TRegistrator<TAnalyzerProcessorConfig> TAnalyzerProcessorConfig::Registrator("analyzer");
