#include "tail.h"

#include <drive/backend/saas/api.h>
#include <drive/backend/tracks/client.h>

#include <drive/library/cpp/tracks/client.h>
#include <drive/library/cpp/tracks/tio.h>

#include <rtline/util/algorithm/ptr.h>

IRequestProcessor::TPtr TTailProcessorConfig::DoConstructAuthProcessor(IReplyContext::TPtr context, IAuthModule::TPtr authModule, const IServerBase* server) const {
    return new TTailProcessor(*this, context, authModule, server);
}

void TTailProcessorConfig::DoInit(const TYandexConfig::Section* section) {
    TBase::DoInit(section);
    const auto& directives = section->GetDirectives();
    TracksApiName = directives.Value("TracksApiName", TracksApiName);
    AssertCorrectConfig(!!TracksApiName, "Incorrect 'TracksApiName' field in configuration '%s'", Name.data());

    LinkerApiName = directives.Value("LinkerApiName", LinkerApiName);
    AssertCorrectConfig(!!LinkerApiName, "Incorrect 'LinkerApiName' field in configuration '%s'", Name.data());
}

void TTailProcessorConfig::ToString(IOutputStream& os) const {
    TBase::ToString(os);
    os << "TracksApiName: " << TracksApiName << Endl;
    os << "LinkerApiName: " << LinkerApiName << Endl;
}

void TTailProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr /*permissions*/, const NJson::TJsonValue& /*requestData*/) {
    const TCgiParameters& cgi = Context->GetCgiParameters();
    const TString& deviceId = cgi.Has("car_id") ? cgi.Get("car_id") : cgi.Get("car");
    R_ENSURE(deviceId, ConfigHttpStatus.SyntaxErrorStatus, "car_id is required");

    auto d = GetValue<ui32>(cgi, "d", false);
    auto debug = GetValue<bool>(cgi, "debug", false).GetOrElse(false);
    auto length = GetValue<double>(cgi, "length", false);

    auto deadline = Context->GetRequestDeadline();
    auto linkerApiName = GetHandlerSetting<TString>("linker_api").GetOrElse(Config.GetLinkerApiName());
    auto tracksApiName = GetHandlerSetting<TString>("tracks_api").GetOrElse(Config.GetTracksApiName());

    auto trackClient = CreateTrackClient(tracksApiName, *Server);
    R_ENSURE(trackClient, HTTP_INTERNAL_SERVER_ERROR, "cannot CreateTrackClient");

    NDrive::TTracksLinker::TOptions tracksLinkerOptions;
    if (d) {
        tracksLinkerOptions.Precision = *d;
    }
    NDrive::TTracksLinker tracksLinker(Yensured(Server->GetLinker(linkerApiName)));
    NDrive::TTailQuery tailQuery;
    tailQuery.DeviceId = deviceId;
    if (length) {
        tailQuery.Length = *length;
    }

    auto asyncTail = trackClient->GetTail(tailQuery, deadline);
    auto asyncLinked = tracksLinker.Link(std::move(asyncTail));
    R_ENSURE(asyncLinked.Wait(deadline), ConfigHttpStatus.TimeoutStatus, "wait timeout");
    R_ENSURE(asyncLinked.HasValue(), ConfigHttpStatus.IncompleteStatus, "error: " << NThreading::GetExceptionMessage(asyncLinked));

    auto linked = asyncLinked.ExtractValue();
    for (auto&& segment : linked.Segments) {
        const auto& match = segment.Linked;
        const auto& tail = linked.Track;
        NJson::TJsonValue tracks(NJson::JSON_ARRAY);
        if (!match.Elements.empty()) {
            NJson::TJsonValue t;
            NDrive::SerializeMetaInfo(t, tail);

            auto& trace = t["track"].SetType(NJson::JSON_ARRAY);
            for (auto&& element : match.Elements) {
                NDrive::SerializeTrace(trace, element.PolyLine.GetCoords());
            }

            auto& summary = t["summary"].SetType(NJson::JSON_MAP);
            summary["length"] = CalcLength(match);

            if (debug) {
                t["segment"].AppendValue(TGeoCoord::SerializeVector(tail.Coordinates));
            }
            tracks.AppendValue(std::move(t));
        }

        g.AddReportElement("tracks", std::move(tracks));
        g.SetCode(HTTP_OK);
    }
}

IRequestProcessorConfig::TFactory::TRegistrator<TTailProcessorConfig> TailProcessor("tail");
