#include "favourite_address_advisor.h"

#include <drive/backend/abstract/frontend.h>
#include <drive/backend/database/transaction/assert.h>
#include <drive/backend/history_iterator/history_iterator.h>
#include <drive/backend/roles/permissions.h>

#include <drive/library/cpp/datasync/client.h>
#include <drive/library/cpp/scheme/scheme.h>

#include <library/cpp/cache/cache.h>
#include <library/cpp/json/json_value.h>

#include <rtline/library/json/adapters.h>
#include <rtline/library/json/builder.h>
#include <rtline/library/json/cast.h>
#include <rtline/library/json/parse.h>

#include <util/generic/algorithm.h>

namespace {
    template <typename T>
    TMaybe<T> GetSettingValue(const IServerBase& server, const TString& key, const TString& prefix, TUserPermissions::TPtr permissions) {
        return TUserPermissions::GetSetting<T>(JoinSeq(".", { prefix, key }), server.GetSettings(), permissions);
    }
}

namespace NJson {
    template <>
    TJsonValue ToJson(const NFavouriteAddressAdvisor::IAdvisor& object) {
        return object.SerializeToJson();
    }

    template <>
    bool TryFromJson(const TJsonValue& value, NFavouriteAddressAdvisor::IAdvisor& result) {
        return result.DeserializeFromJson(value);
    }
}

namespace NFavouriteAddressAdvisor {
    bool TFavourites::DeserializeFromDatasyncResponse(const NJson::TJsonValue& data, TMessagesCollector& errors) {
        RequestInstant = Now();

        const auto& items = data["items"];
        if (!items.IsArray()) {
            return false;
        }

        for (const auto& item : items.GetArraySafe()) {
            // https://wiki.yandex-team.ru/disk/personality/addresses/
            TGeoCoord coord;
            if (!NJson::ParseField(item["longitude"], coord.X, /* required = */ true, errors) ||
                !NJson::ParseField(item["latitude"], coord.Y, /* required = */ true, errors)
            ) {
                return false;
            }

            Points.push_back(std::move(coord));
        }

        return true;
    }

    const TDuration IAdvisor::DefaultMaxWaitTimeout = TDuration::MilliSeconds(100);

    TInstant IAdvisor::GetDeadline() const {
        return ReqActuality + MaxWaitTimeout;
    }

    bool IAdvisor::IsSuggestAvailable() const {
        return GetEnabled() && AreFavouritesAvailable() && GetHistoryFinishPoints().Defined();
    }

    bool IAdvisor::AreFavouritesAvailable() const {
        return (Favourites.HasValue()) ? Favourites.GetValueSync().Defined() : false;
    }

    bool IAdvisor::WaitFavouritesInitialized() const {
        return Favourites.Wait(GetDeadline());
    }

    NDrive::TScheme IAdvisor::GetScheme(const IServerBase& /* server */) {
        NDrive::TScheme scheme;
        scheme.Add<TFSBoolean>("enabled", "Включен ли саджест").SetDefault(false);
        scheme.Add<TFSDuration>("max_wait_timeout", "Макс. время ожидания ответа").SetDefault(DefaultMaxWaitTimeout);
        return scheme;
    }

    bool IAdvisor::DeserializeFromJson(const NJson::TJsonValue& data) {
        return NJson::ParseField(data["enabled"], Enabled) &&
               NJson::ParseField(data["max_wait_timeout"], NJson::Stringify(MaxWaitTimeout)) &&
               DoDeserializeFromJson(data);
    }

    NJson::TJsonValue IAdvisor::SerializeToJson() const {
        NJson::TJsonValue result;
        NJson::InsertField(result, "enabled", Enabled);
        NJson::InsertField(result, "max_wait_timeout", NJson::Stringify(MaxWaitTimeout));
        DoSerializeToJson(result);
        return result;
    }

    class TFavouritesCache {
        using TCache = TLRUCache<ui64, TFavourites>;

    public:
        TFavouritesCache(size_t size, TDuration age)
            : Cache(size)
            , Age(age)
        {
        }

        bool Find(const ui64& uid, TFavourites& favourites) {
            TReadGuard g(Mutex);
            auto cached = Cache.Find(uid);
            if (cached != Cache.End() && cached.Value().GetRequestInstant() + Age >= Now()) {
                favourites = cached.Value();
                return true;
            }
            return false;
        }

        void Update(const ui64& uid, const TFavourites& favourites) {
            TWriteGuard g(Mutex);
            Cache.Update(uid, favourites);
        }

    private:
        mutable TRWMutex Mutex;
        TCache Cache;
        const TDuration Age;
    };

    const TString TAdvisor::SettingPrefix = "session.feedback.favourite_address_suggest";

    const TString TAdvisor::DefaultDatasyncCollection = "v2/personality/profile/addresses";  // alias: v1/personality/profile/profile/common_addresses

    const size_t TAdvisor::DefaultCacheSize = 1000;
    const TDuration TAdvisor::DefaultCacheMaxAge = TDuration::Seconds(30);

    const ui32 TAdvisor::DefaultMaxSessionCount = 1000;
    const TDuration TAdvisor::DefaultMaxTimePassed = TDuration::Days(365);
    const ui32 TAdvisor::DefaultMinClusterSize = 3;
    const ui32 TAdvisor::DefaultMaxCentroidDistanceMeters = 500;

    TAdvisor::TAdvisor(const NDrive::IServer& server)
        : Server(server)
    {
    }

    void TAdvisor::InitializeFavourites(const ui64 uid, const TInstant reqActuality) {
        if (!!GetReqActuality()) {
            return;  // already requested
        }

        SetProtectedReqActuality((!!reqActuality) ? reqActuality : Now());

        if (!GetEnabled()) {
            SetProtectedFavourites(NThreading::MakeFuture<TMaybe<TFavourites>>());
            return;
        }

        auto cache = (CacheEnabled) ? Singleton<TFavouritesCache>(CacheSize, CacheMaxAge) : nullptr;
        {
            TFavourites cachedFavourites;
            if (cache && cache->Find(uid, cachedFavourites)) {
                SetProtectedFavourites(NThreading::MakeFuture<TMaybe<TFavourites>>(std::move(cachedFavourites)));
                return;
            }
        }

        auto datasyncClientPtr = Server.GetDatasyncClient();
        if (!datasyncClientPtr) {
            SetProtectedFavourites(NThreading::MakeFuture<TMaybe<TFavourites>>());
            return;
        }

        auto asyncResponse = datasyncClientPtr->Get(GetDatasyncCollection(), "", uid);
        SetProtectedFavourites(
            asyncResponse.Apply(
                [cache, uid](const NThreading::TFuture<TDatasyncClient::TResponse>& r) {
                    auto report = r.GetValue();
                    if (!report) {
                        ERROR_LOG << "Cannot fetch favourites for user " << uid << ": " << report.GetCode() << " " << report.GetError() << Endl;
                        return NThreading::MakeFuture<TMaybe<TFavourites>>();
                    }

                    TFavourites favourites;
                    TMessagesCollector errors;
                    const bool parseResult = favourites.DeserializeFromDatasyncResponse(report.GetValue(), errors);
                    if (!parseResult) {
                        ERROR_LOG << errors.GetStringReport() << Endl;
                        return NThreading::MakeFuture<TMaybe<TFavourites>>();
                    }

                    if (cache) {
                        cache->Update(uid, favourites);
                    }
                    return NThreading::MakeFuture<TMaybe<TFavourites>>(std::move(favourites));
                }
            )
        );
    }

    void TAdvisor::InitializeHistoryFinishCoords(const TString& userId, NDrive::TEntitySession& session) {
        THistoryRidesContext context(Server, TInstant::Zero());
        auto ydbTx = Server.GetDriveAPI()->BuildYdbTx<NSQL::ReadOnly>("rtroad_charges_watcher", &Server);
        R_ENSURE(context.InitializeUser(userId, session, ydbTx, {}, MaxSessionCount), {}, "cannot InitializeUser " << userId, session);

        TVector<THistoryRideObject> sessions = context.GetSessions(TInstant::Max(), MaxSessionCount);
        THistoryRideObject::FetchFullRiding(&Server, sessions);

        TFinishPoints historyFinishPoints;

        for (auto&& i : sessions) {
            if (i.GetLastLocation().Defined() && GetReqActuality() - i.GetLastTS() < MaxTimePassed) {
                historyFinishPoints.push_back({ i.GetLastLocation()->GetCoord(), i.GetLastTS() });
            }
        }

        SetProtectedHistoryFinishPoints(std::move(historyFinishPoints));
    }

    bool TAdvisor::Suggest(TFinishPoint point) const {
        if (!IsSuggestAvailable()) {
            return false;
        }

        if (GetReqActuality() - point.Instant >= MaxTimePassed) {
            return false;
        }

        // NB. Actually sessions are compared by pairs deriving O(n^2) complexity
        // Smth like library/cpp/knn_index/distance.h could be used, but each point must be interpreted as a vector
        std::function<double(const TGeoCoord&)> getDistanceTo;

        switch (DistanceType) {
        case EDistanceType::SPHERE:
            getDistanceTo = [&point](const TGeoCoord& coord) { return point.Location.GetLengthTo(coord); };
            break;
        case EDistanceType::DISTANCE_L1:
            getDistanceTo = [&point](const TGeoCoord& coord) { return Abs(point.Location.X - coord.X) + Abs(point.Location.Y - coord.Y); };
            break;
        case EDistanceType::DISTANCE_L2:
            getDistanceTo = [&point](const TGeoCoord& coord) { return (point.Location - coord).SimpleLength(); };
            break;
        }

        for (const auto& coord : GetFavourites().GetValueSync()->GetPoints()) {
            bool isSimilarToAnotherFavourite = (getDistanceTo(coord) < MaxCentroidDistanceMeters);
            if (isSimilarToAnotherFavourite) {
                return false;
            }
        }

        ui32 clusterSize = 1;  // point itself is a centroid center
        for (const auto& historyPoint : *GetHistoryFinishPoints()) {
            if (GetReqActuality() - historyPoint.Instant < MaxTimePassed && getDistanceTo(historyPoint.Location) < MaxCentroidDistanceMeters) {
                clusterSize++;
            }
        }

        return clusterSize >= MinClusterSize;
    }

    TAdvisor::TPtr TAdvisor::Construct(const NDrive::IServer& server, TUserPermissionsPtr permissions) {
        auto defaults = GetSettingDefaults(server, permissions);

        auto instance = MakeAtomicShared<TAdvisor>(server);
        if (!instance->DeserializeFromJson(defaults)) {
            return nullptr;
        }

        return instance;
    }

    NDrive::TScheme TAdvisor::GetScheme(const IServerBase& server) {
        NDrive::TScheme scheme = TBase::GetScheme(server);
        scheme.Add<TFSString>("datasync_collection", "Uri коллекции в datasync").SetDefault(DefaultDatasyncCollection);
        scheme.Add<TFSBoolean>("cache_enabled", "Использовать кэш для ответов datasync").SetDefault(true);
        scheme.Add<TFSNumeric>("cache_size", "Размер кэша").SetDefault(DefaultCacheSize);
        scheme.Add<TFSDuration>("cache_max_age", "Максимально допустимый возраст кэша").SetDefault(DefaultCacheMaxAge);
        scheme.Add<TFSNumeric>("max_session_count", "Брать столько последних сессий").SetDefault(DefaultMaxSessionCount);
        scheme.Add<TFSDuration>("max_time_passed", "Исключать если прошло больше времени").SetDefault(DefaultMaxTimePassed);
        scheme.Add<TFSNumeric>("min_cluster_size", "Минимальный размер кластера").SetDefault(DefaultMinClusterSize);
        scheme.Add<TFSVariants>("distance_type", "Способ подсчета расстояния").InitVariants<EDistanceType>().SetDefault(::ToString(EDistanceType::SPHERE));
        scheme.Add<TFSNumeric>("max_centroid_distance_meters", "Максимально допустимое расстояние до центра кластера").SetDefault(DefaultMaxCentroidDistanceMeters);
        return scheme;
    }

    NJson::TJsonValue TAdvisor::GetSettingDefaults(const IServerBase& server, TUserPermissionsPtr permissions) {
        return NJson::TMapBuilder
            ("enabled", GetSettingValue<bool>(server, "enabled", SettingPrefix, permissions).GetOrElse(false))
            ("max_wait_timeout", ::ToString(GetSettingValue<TDuration>(server, "max_wait_timeout", SettingPrefix, permissions).GetOrElse(DefaultMaxWaitTimeout)))
            ("datasync_collection", GetSettingValue<TString>(server, "datasync_collection", SettingPrefix, permissions).GetOrElse(DefaultDatasyncCollection))
            ("cache_enabled", GetSettingValue<bool>(server, "cache_enabled", SettingPrefix, permissions).GetOrElse(true))
            ("cache_size", GetSettingValue<size_t>(server, "cache_size", SettingPrefix, permissions).GetOrElse(DefaultCacheSize))
            ("cache_max_age", ::ToString(GetSettingValue<TDuration>(server, "cache_max_age", SettingPrefix, permissions).GetOrElse(DefaultCacheMaxAge)))
            ("max_session_count", GetSettingValue<ui32>(server, "max_session_count", SettingPrefix, permissions).GetOrElse(DefaultMaxSessionCount))
            ("max_time_passed", ::ToString(GetSettingValue<TDuration>(server, "max_time_passed", SettingPrefix, permissions).GetOrElse(DefaultMaxTimePassed)))
            ("min_cluster_size", GetSettingValue<ui32>(server, "min_cluster_size", SettingPrefix, permissions).GetOrElse(DefaultMinClusterSize))
            ("distance_type", GetSettingValue<TString>(server, "distance_type", SettingPrefix, permissions).GetOrElse(""))
            ("max_centroid_distance_meters", GetSettingValue<ui32>(server, "max_centroid_distance_meters", SettingPrefix, permissions).GetOrElse(DefaultMaxCentroidDistanceMeters));
    }

    bool TAdvisor::DoDeserializeFromJson(const NJson::TJsonValue& data) {
        return NJson::ParseField(data["datasync_collection"], DatasyncCollection) &&
               NJson::ParseField(data["cache_enabled"], CacheEnabled) &&
               NJson::ParseField(data["cache_size"], CacheSize) &&
               NJson::ParseField(data["cache_max_age"], NJson::Stringify(CacheMaxAge)) &&
               NJson::ParseField(data["max_session_count"], MaxSessionCount) &&
               NJson::ParseField(data["max_time_passed"], NJson::Stringify(MaxTimePassed)) &&
               NJson::ParseField(data["min_cluster_size"], MinClusterSize) &&
               NJson::ParseField(data["distance_type"], NJson::Stringify(DistanceType)) &&
               NJson::ParseField(data["max_centroid_distance_meters"], MaxCentroidDistanceMeters);
    }

    void TAdvisor::DoSerializeToJson(NJson::TJsonValue& result) const {
        NJson::InsertField(result, "datasync_collection", DatasyncCollection);
        NJson::InsertField(result, "cache_enabled", CacheEnabled);
        NJson::InsertField(result, "cache_size", CacheSize);
        NJson::InsertField(result, "cache_max_age", NJson::Stringify(CacheMaxAge));
        NJson::InsertField(result, "max_session_count", MaxSessionCount);
        NJson::InsertField(result, "max_time_passed", NJson::Stringify(MaxTimePassed));
        NJson::InsertField(result, "min_cluster_size", MinClusterSize);
        NJson::InsertField(result, "distance_type", NJson::Stringify(DistanceType));
        NJson::InsertField(result, "max_centroid_distance_meters", MaxCentroidDistanceMeters);
    }
}
