#pragma once

#include <drive/backend/abstract/base.h>
#include <drive/backend/database/transaction/tx.h>

#include <library/cpp/threading/future/future.h>

#include <rtline/library/geometry/coord.h>
#include <rtline/util/types/accessor.h>

#include <util/string/join.h>

class TGeoCoord;
class TMessagesCollector;

namespace NDrive {
    class IServer;
    class TScheme;
}

namespace NJson {
    class TJsonValue;
}

class TUserPermissions;

namespace NFavouriteAddressAdvisor {
    enum class EDistanceType {
        DISTANCE_L1 /* "distance_l1" */,
        DISTANCE_L2 /* "distance_l2" */,
        SPHERE /* "sphere" */,
    };

    struct TFinishPoint {
        TGeoCoord Location;
        TInstant Instant;
    };

    using TFinishPoints = TVector<TFinishPoint>;

    class TFavourites {
    public:
        bool Empty() const noexcept {
            return Points.empty();
        }

        size_t Size() const {
            return Points.size();
        }

        bool DeserializeFromDatasyncResponse(const NJson::TJsonValue& data, TMessagesCollector& errors);

    private:
        R_READONLY(TVector<TGeoCoord>, Points);
        R_READONLY(TInstant, RequestInstant);
    };

    class IAdvisor {
    public:
        using TPtr = TAtomicSharedPtr<IAdvisor>;

        virtual ~IAdvisor() = default;

        virtual void InitializeFavourites(const ui64 uid, const TInstant reqActuality) = 0;
        virtual void InitializeHistoryFinishCoords(const TString& userId, NDrive::TEntitySession& session) = 0;

        virtual bool Suggest(TFinishPoint point) const = 0;

        TInstant GetDeadline() const;

        bool IsSuggestAvailable() const;

        bool AreFavouritesAvailable() const;
        bool WaitFavouritesInitialized() const;

        static NDrive::TScheme GetScheme(const IServerBase& server);
        virtual bool DeserializeFromJson(const NJson::TJsonValue& data);
        virtual NJson::TJsonValue SerializeToJson() const;

        virtual bool DoDeserializeFromJson(const NJson::TJsonValue& data) = 0;
        virtual void DoSerializeToJson(NJson::TJsonValue& result) const = 0;

    protected:
        static const TDuration DefaultMaxWaitTimeout;

    private:
        R_READONLY(bool, Enabled, false);

        R_READONLY(NThreading::TFuture<TMaybe<TFavourites>>, Favourites);
        R_READONLY(TMaybe<TFinishPoints>, HistoryFinishPoints);

        R_READONLY(TInstant, ReqActuality);
        R_READONLY(TDuration, MaxWaitTimeout, DefaultMaxWaitTimeout);
    };

    class TAdvisor : public IAdvisor {
        using TBase = IAdvisor;
        using TUserPermissionsPtr = TIntrusiveConstPtr<TUserPermissions>;

    public:
        explicit TAdvisor(const NDrive::IServer& server);

        virtual void InitializeFavourites(const ui64 uid, const TInstant reqActuality) override;
        virtual void InitializeHistoryFinishCoords(const TString& userId, NDrive::TEntitySession& session) override;

        virtual bool Suggest(TFinishPoint point) const override;

        static TPtr Construct(const NDrive::IServer& server, TUserPermissionsPtr permissions = nullptr);

        static NDrive::TScheme GetScheme(const IServerBase& server);
        virtual bool DoDeserializeFromJson(const NJson::TJsonValue& data) override;
        virtual void DoSerializeToJson(NJson::TJsonValue& result) const override;

    private:
        static NJson::TJsonValue GetSettingDefaults(const IServerBase& server, TUserPermissionsPtr permissions = nullptr);

    private:
        static const TString SettingPrefix;

        static const TString DefaultDatasyncCollection;

        static const size_t DefaultCacheSize;
        static const TDuration DefaultCacheMaxAge;

        static const ui32 DefaultMaxSessionCount;
        static const TDuration DefaultMaxTimePassed;
        static const ui32 DefaultMinClusterSize;
        static const ui32 DefaultMaxCentroidDistanceMeters;

        const NDrive::IServer& Server;

        R_READONLY(TString, DatasyncCollection, DefaultDatasyncCollection);

        R_READONLY(bool, CacheEnabled, true);
        R_READONLY(size_t, CacheSize, DefaultCacheSize);
        R_READONLY(TDuration, CacheMaxAge, DefaultCacheMaxAge);

        R_READONLY(ui32, MaxSessionCount, DefaultMaxSessionCount);
        R_READONLY(TDuration, MaxTimePassed, DefaultMaxTimePassed);

        R_READONLY(ui32, MinClusterSize, DefaultMinClusterSize);

        R_READONLY(EDistanceType, DistanceType, EDistanceType::SPHERE);
        R_READONLY(ui32, MaxCentroidDistanceMeters, DefaultMaxCentroidDistanceMeters);
    };
}
