#include "insurance.h"


#include <drive/backend/data/scoring/scoring.h>
#include <drive/backend/data/user_insurance.h>
#include <drive/backend/database/drive/private_data.h>
#include <drive/backend/tags/tags_manager.h>
#include <drive/backend/user_insurance/entities.h>

#include <drive/library/cpp/car/number.h>
#include <drive/library/cpp/mds/client.h>
#include <drive/library/cpp/renins/client.h>

#include <rtline/util/algorithm/datetime.h>


namespace {
    const TString KaskoSettingsPrefix = "user_insurance.kasko.";
    const ui32 KaskoScoreCoefficient = 1000;

    NDrive::TScheme GetDriverScheme(NJson::TJsonValue defaultJson = NJson::JSON_UNDEFINED, const bool readOnly = false) {
        auto setDefault = [&defaultJson, readOnly](NDrive::IBaseDefaultSchemeElement& field) {
            TString key = field.GetFieldName();
            if (defaultJson.Has(key)) {
                field.SetDefaultValueView(defaultJson[key]);
            }
            if (readOnly) {
                field.SetReadOnly(true);
            } else {
                field.SetRequired(true);
            }
        };
        NDrive::TScheme scheme;
        setDefault(scheme.Add<TFSString>("name", "Имя"));
        setDefault(scheme.Add<TFSString>("middle_name", "Отчество"));
        setDefault(scheme.Add<TFSString>("last_name", "Фамилия"));
        setDefault(scheme.Add<TFSNumeric>("date_of_birth", "Дата рождения").SetVisual(TFSNumeric::DateTime));

        if (!readOnly) {
            setDefault(scheme.Add<TFSVariants>("sex_code", "Пол").SetCompoundVariants({ TFSVariants::TCompoundVariant("F", "Ж"), TFSVariants::TCompoundVariant("M", "М") }));
            setDefault(scheme.Add<TFSBoolean>("maried", "Семейное положение"));
        }

        setDefault(scheme.Add<TFSString>("driver_license", "Номер водительского удостоверения"));
        setDefault(scheme.Add<TFSNumeric>("driving_experience_date_start", "Дата начала стажа").SetVisual(TFSNumeric::DateTime));
        return scheme;
    }

    NDrive::NRenins::TAddressData ParseAddress(const TUserPassportData& pass) {
        NDrive::NRenins::TAddressData addres;
        addres.SetCountry(pass.GetRegistrationCountry());
        addres.SetRegion(pass.GetRegistration().GetRegionDef(""));
        addres.SetCity(pass.GetRegistration().GetAreaDef(""));
        addres.SetStreet(pass.GetRegistration().GetStreetDef(""));
        addres.SetHouse(pass.GetRegistration().GetHouseDef(""));
        return addres;
    }

    template <class T>
    NDrive::NRenins::TUserMainData ParseUserMainData(const T& data) {
        NDrive::NRenins::TUserMainData mainData;
        mainData.SetName(data.GetFirstName());
        mainData.SetMiddleName(data.GetMiddleName());
        mainData.SetLastName(data.GetLastName());
        mainData.SetDateOfBirth(data.GetUserBirthDate());
        return mainData;
    }

    NDrive::NRenins::NKasko::TDriverOrderStartData ParseDriver(const TUserDrivingLicenseData& license) {
        NDrive::NRenins::NKasko::TDriverOrderStartData result;
        if (auto number = license.GetNumber(); number.size() > NDrive::NRenins::LicenseSeriesSize) {
            result.SetLicenseSeries(number.substr(0, NDrive::NRenins::LicenseSeriesSize));
            result.SetLicenseNumber(number.substr(NDrive::NRenins::LicenseSeriesSize));
        }
        result.SetExperienceDateStart(license.GetExperienceFrom());
        result.SetUserData(ParseUserMainData(license));
        return result;
    }

    NThreading::TFuture<NDrive::NRenins::NKasko::TCalculateOrder> BuildCalculateParams(const IPrivateDataStorage& privateDataClient, const TDriveUserData& user, const NDrive::NRenins::NKasko::TAutoData&& car) {
        if (!user.GetPassportDatasyncRevision() || !user.GetDrivingLicenseDatasyncRevision()) {
            return NThreading::MakeErrorFuture<NDrive::NRenins::NKasko::TCalculateOrder>(std::make_exception_ptr(yexception() << "no user data"));
        }
        auto passportFuture = privateDataClient.GetPassport(user, user.GetPassportDatasyncRevision());
        auto licenseFuture = privateDataClient.GetDrivingLicense(user, user.GetDrivingLicenseDatasyncRevision());
        auto initializedDrivingLicense = NThreading::Initialize(passportFuture).IgnoreResult();
        auto initializedPassport = NThreading::Initialize(licenseFuture).IgnoreResult();
        return NThreading::WaitAll(initializedDrivingLicense, initializedPassport).Apply([
            passportFuture = std::move(passportFuture),
            licenseFuture = std::move(licenseFuture),
            car = std::move(car)
        ](const NThreading::TFuture<void>&) mutable {
            auto pass = passportFuture.GetValue();
            NDrive::NRenins::NKasko::TCalculateOrder result;
            result.SetAutoData(car);
            {
                auto owner = NDrive::NRenins::NKasko::TPersonData()
                    .SetAddresses(TVector{ ParseAddress(pass) })
                    .SetDateOfBirth(pass.GetUserBirthDate());
                result.MutableOwners().emplace_back(std::move(owner));
            }
            {
                auto license = licenseFuture.GetValue();
                NDrive::NRenins::NKasko::TDriverData driver;
                driver.SetUserData(ParseUserMainData(license));
                driver.SetExperienceDateStart(license.GetExperienceFrom());
                result.MutableDrivers().emplace_back(std::move(driver));
            }
            return result;
        });
    }

    NDrive::NRenins::TPersonPassport ParsePassport(const TUserPassportData& pass) {
        NDrive::NRenins::TPersonPassport result;
        if (auto number = pass.GetNumber(); number.size() > NDrive::NRenins::PassportSeriesSize) {
            result.Series = pass.GetNumber().substr(0, NDrive::NRenins::PassportSeriesSize);
            result.Number = pass.GetNumber().substr(NDrive::NRenins::PassportSeriesSize);
        }
        result.DateIssue = pass.GetIssueDate();
        result.IssuedBy = pass.GetSubdivisionCodeDef("");
        return result;
    }

    NDrive::NRenins::NKasko::TDriverOrderStartData ParseDriver(const TUserPassportData& pass, const TUserDrivingLicenseData& license) {
        auto result = ParseDriver(license);
        if (ToUpperUTF8(pass.GetGenderDef("")).StartsWith("М")) {
            result.SetSexCode("M");
        } else if (ToUpperUTF8(pass.GetGenderDef("")).StartsWith("Ж")) {
            result.SetSexCode("F");
        }
        return result;
    }

    NDrive::NRenins::NKasko::TPersonOrderStartData ParseOwner(const TUserPassportData& pass, const TString& phone, const TString& email) {
        NDrive::NRenins::NKasko::TPersonOrderStartData owner;
        owner.MutableAddresses().emplace_back(ParseAddress(pass));
        owner.SetPhone(phone ? phone.substr(1) : "");
        owner.SetEmail(email);
        owner.SetUserData(ParseUserMainData(pass));
        owner.SetPassport(ParsePassport(pass));
        return owner;
    }

    NThreading::TFuture<NDrive::NRenins::NKasko::TStartOrderData> BuildStartOrederParams(const IPrivateDataStorage& privateDataClient, const TDriveUserData& user) {
        if (!user.GetPassportDatasyncRevision() || !user.GetDrivingLicenseDatasyncRevision()) {
            return NThreading::MakeErrorFuture<NDrive::NRenins::NKasko::TStartOrderData>(std::make_exception_ptr(yexception() << "no user data"));
        }
        auto passportFuture = privateDataClient.GetPassport(user, user.GetPassportDatasyncRevision());
        auto licenseFuture = privateDataClient.GetDrivingLicense(user, user.GetDrivingLicenseDatasyncRevision());
        auto initializedDrivingLicense = NThreading::Initialize(passportFuture).IgnoreResult();
        auto initializedPassport = NThreading::Initialize(licenseFuture).IgnoreResult();
        return NThreading::WaitAll(initializedDrivingLicense, initializedPassport).Apply([
            passportFuture = std::move(passportFuture),
            licenseFuture = std::move(licenseFuture),
            phone = user.GetPhone(),
            email = user.GetEmail()
        ](const NThreading::TFuture<void>&) mutable {
            auto pass = passportFuture.GetValue();
            NDrive::NRenins::NKasko::TStartOrderData result;
            result.MutableDrivers().emplace_back(ParseDriver(pass, licenseFuture.GetValue()));
            result.MutableOwners().emplace_back(ParseOwner(pass, phone, email));
            return result;
        });
    }

    TString GetVariantTitle(const NDrive::IServer& server, const TString& setting, const TString& key, const TString& subKey = "") {
        auto findTitle = [](const NJson::TJsonValue& array, const TString& code) {
            for (auto& item : array.GetArray()) {
                if (item["code"] == code) {
                    return item["name"].GetString();
                }
            }
            return code;
        };
        if (auto strParam = server.GetSettings().GetValueDef<TString>(KaskoSettingsPrefix + setting, "")) {
            NJson::TJsonValue json;
            if (NJson::ReadJsonFastTree(strParam, &json)) {
                return subKey ? findTitle(json[key], subKey) : findTitle(json, key);
            }
        }
        return subKey ? subKey : key;
    }

    NJson::TJsonValue GetKaskoReport(const TVector<NDrive::NRenins::TKaskoData>& infos, ELocalization locale, const NDrive::IServer& server) {
        NJson::TJsonValue insurances = NJson::JSON_ARRAY;
        bool canMore = true;
        for (const auto& info : infos) {
            const auto status = info.GetStatus();
            canMore &= status == NDrive::NRenins::EKaskoStatus::Signed;
            auto& report = insurances.AppendValue(info.SerializeToJson());
            if (auto localization = server.GetLocalization()) {
                auto statusKey = ToString(info.GetStatus());
                report.InsertValue("status_title", localization->GetLocalString(locale, KaskoSettingsPrefix + "status_title." + statusKey));
                report.InsertValue("status_description", localization->GetLocalString(locale, KaskoSettingsPrefix + "status_description." + statusKey));
                report.InsertValue("status_hint", localization->GetLocalString(locale, KaskoSettingsPrefix + "status_hint." + statusKey));
                if (info.GetStatus() == NDrive::NRenins::EKaskoStatus::Error) {
                    auto error = report["error_type"].GetString();
                    if (!error) {
                        error = "unknown";
                    }
                    report.InsertValue("error_ui", localization->GetLocalString(locale, KaskoSettingsPrefix + "error." + error));
                }
                if (report.Has("price") && report["price"].Has("discounts")) {
                    for (auto& discount : report["price"]["discounts"].GetArraySafe()) {
                        TString title = discount.Has("description_key")
                            ? discount["description_key"].GetString()
                            : "unknown";
                        discount.InsertValue("description", localization->GetLocalString(locale, KaskoSettingsPrefix + "discount." + title));
                    }
                }
            }
            NJson::TJsonValue buttons = NJson::JSON_ARRAY;
            if (auto supButton = server.GetSettings().GetValue<TString>(KaskoSettingsPrefix + "support_button")) {
                NJson::TJsonValue supJson;
                if (NJson::ReadJsonFastTree(*supButton, &supJson)) {
                    buttons.AppendValue(supJson);
                }
            }
            if (status == NDrive::NRenins::EKaskoStatus::NeedPso) {
                if (auto psoButton = server.GetSettings().GetValue<TString>(KaskoSettingsPrefix + "pso_button")) {
                    NJson::TJsonValue psoJson;
                    if (NJson::ReadJsonFastTree(*psoButton, &psoJson)) {
                        buttons.AppendValue(psoJson);
                    }
                }
            }
            if (!buttons.GetArray().empty()) {
                report.InsertValue("buttons", buttons);
            }
            auto& carReport = report["car"];
            if (info.GetCarReportData().GetBrand()) {
                carReport.InsertValue("brand", GetVariantTitle(server, "auto_brand_code", info.GetCarReportData().GetBrand()));
                if (info.GetCarReportData().GetModel()) {
                    carReport.InsertValue("model", GetVariantTitle(server, "auto_brand_code.auto_model_code", info.GetCarReportData().GetBrand(), info.GetCarReportData().GetModel()));
                }
            }
            if (auto path = report["document_path"].GetString()) {
                report.InsertValue("document_link", server.GetSettings().GetValueDef<TString>(KaskoSettingsPrefix + "document_link", "") + path);
            }
        }
        NJson::TJsonValue result;
        result.InsertValue("insurances", insurances);
        result.InsertValue("can_create_more", canMore);
        result.InsertValue("details_link", server.GetSettings().GetValueDef<TString>(KaskoSettingsPrefix + "details_link", ""));
        result.InsertValue("order_terms", server.GetSettings().GetValueDef<TString>(KaskoSettingsPrefix + "order_terms", ""));
        result.InsertValue("payment_terms", server.GetSettings().GetValueDef<TString>(KaskoSettingsPrefix + "payment_terms", ""));
        result.InsertValue("success_payment_link", server.GetSettings().GetValueDef<TString>(KaskoSettingsPrefix + "success_payment_link", ""));
        result.InsertValue("failure_payment_link", server.GetSettings().GetValueDef<TString>(KaskoSettingsPrefix + "failure_payment_link", ""));
        return result;
    }

    void SetVariantsByJson(const NJson::TJsonValue& json, TFSVariants& field, const TString& def = "") {
        TVector<TFSVariants::TCompoundVariant> variants;
        bool needDef = false;
        for (auto&& item : json.GetArray()) {
            if (item.IsString() && item.GetString()) {
                variants.emplace_back(item.GetString(), item.GetString());
                needDef |= def == item.GetString();
                continue;
            }
            const TString text = item["name"].GetString();
            const TString value = item["code"].GetString();
            if (text && value) {
                needDef |= def == value;
                variants.emplace_back(value, text);
            }
        }
        field.SetCompoundVariants(variants);
        if (def && needDef) {
            field.SetDefault(def);
        }
    }

    TFSNumeric& SetDefault(const TCgiParameters& schemeCgi, TFSNumeric& field, TMaybe<double> def = {}) {
        double value = 0;
        if (auto val = schemeCgi.Get(field.GetFieldName()); val && TryFromString(val, value)) {
            if (field.GetMinDef(value) <= value && field.GetMaxDef(value) >= value) {
                field.SetDefault(value);
            }
        } else if (def) {
            field.SetDefault(*def);
        }
        return field;
    }

    TFSString& SetDefault(const TCgiParameters& schemeCgi, TFSString& field) {
        if (auto val = schemeCgi.Get(field.GetFieldName())) {
            field.SetDefault(val);
        }
        return field;
    }

    TFSBoolean& SetDefault(const TCgiParameters& schemeCgi, TFSBoolean& field) {
        bool value = false;
        if (auto val = schemeCgi.Get(field.GetFieldName()); val && TryFromString(val, value)) {
            field.SetDefault(value);
        }
        return field;
    }

    class TSchemePresets {
        R_FIELD(TSet<TString>, Years);
        R_FIELD(TSet<TString>, EnginePowers);
        R_FIELD(NJson::TJsonValue, Bodies);
        R_OPTIONAL(ui32, DefPrice);
        R_OPTIONAL(ui32, MaxPrice);
        R_OPTIONAL(ui32, MinPrice);

    public:
        bool Parse(const NJson::TJsonValue& json) {
            return NJson::ParseField(json, "auto_year_of_issue_code", Years)
                && NJson::ParseField(json, "engine_power_code", EnginePowers)
                && NJson::ParseField(json, "auto_body_type_code", Bodies)
                && NJson::ParseField(json["auto_cost"], "def", DefPrice)
                && NJson::ParseField(json["auto_cost"], "max", MaxPrice)
                && NJson::ParseField(json["auto_cost"], "min", MinPrice);
        }
    };

    TMap<TString, TString> GetKaskoErrorMap(const NDrive::IServer& server) {
        NJson::TJsonValue json;
        auto strParam = server.GetSettings().GetValueDef<TString>(KaskoSettingsPrefix + "errors", "");
        if (!NJson::ReadJsonFastTree(strParam, &json)) {
            return {};
        }
        TMap<TString, TString> result;
        for (auto item : json.GetArray()) {
            TString key;
            if (!NJson::ParseField(item, "key", key, /* required = */ true)) {
                continue;
            }
            for (auto&& code : item["codes"].GetArray()) {
                if (const TString& codeStr = code.GetString()) {
                    result[codeStr] = key;
                }
            }
        }
        return result;
    }
}

void TUserInsuranceCalculateProcessor::Parse(const NJson::TJsonValue& requestData) {
    {
        auto carData = NDrive::NRenins::ParseKaskoCarData(requestData["car"]);
        R_ENSURE(carData, ConfigHttpStatus.SyntaxErrorStatus, "fail to parse car data");
        CarData = std::move(*carData);
    }
    {
        auto carData = NDrive::NRenins::ParseCarKaskoOrderData(requestData["car"]);
        R_ENSURE(carData, ConfigHttpStatus.SyntaxErrorStatus, "fail to parse car order data");
        KaskoData.SetCar(std::move(*carData));
        KaskoData.MutableCarReportData().SetBrand(CarData.GetMainAutoData().GetBrandCode());
        KaskoData.MutableCarReportData().SetModel(CarData.GetMainAutoData().GetModelCode());
    }
    {
        const bool parsedInsuranceParams = NJson::ParseField(requestData["insurance"], "date_start", KaskoData.MutableInsuranceStart(), /* required = */ true)
            && NJson::ParseField(requestData["insurance"], "date_end", KaskoData.MutableInsuranceEnd(), /* required = */ false)
            && NJson::ParseField(requestData["insurance"], "franchise", KaskoData.MutableFranchise(), /* required = */ true);
        R_ENSURE(parsedInsuranceParams, ConfigHttpStatus.SyntaxErrorStatus, "fail to parse insurance data");
    }
}

void TUserInsuranceCalculateProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /* requestData */) {
    auto kaskoClient = Yensured(Server)->GetReninsKaskoClient();
    R_ENSURE(kaskoClient, ConfigHttpStatus.UnknownErrorStatus, "unknown insurance client");
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Control, TAdministrativeAction::EEntity::UserInsurance, { "kasko" });
    const auto& tagsManager = Yensured(Server->GetDriveAPI())->GetTagsManager();

    TUsersDB::TOptionalUser user;
    const auto tagName = GetHandlerSettingDef<TString>("tag_name", TUserKaskoTag::Type());
    {
        auto tx = BuildTx<NSQL::ReadOnly>();
        user = DriveApi->GetUserManager().RestoreUser(permissions->GetUserId(), tx);
        R_ENSURE(user, ConfigHttpStatus.IncompleteStatus, "unknown user");
        auto dbTags = tagsManager.GetUserTags().RestoreEntityTags({ permissions->GetUserId() }, { tagName }, tx);
        R_ENSURE(dbTags, ConfigHttpStatus.IncompleteStatus, "fail to get user tags", tx);
        auto hasOrderInProcess = AnyOf(dbTags->begin(), dbTags->end(), [](const TDBTag& dbTag) {
            auto tag = dbTag.GetTagAs<TUserKaskoTag>();
            if (!tag || !tag->HasKaskoData() || tag->GetKaskoDataRef().GetStatus() == NDrive::NRenins::EKaskoStatus::Signed) {
                return false;
            }
            return true;
        });
        R_ENSURE(!hasOrderInProcess, ConfigHttpStatus.SyntaxErrorStatus, "has order in process");
    }
    TString packageName;
    NJson::TJsonValue packageMeta;
    TMaybe<double> userScore;
    {
        const auto tagName = GetHandlerSettingDef<TString>("tag_name", TUserKaskoTag::Type());
        auto descPtr = Server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetDescriptionByName(tagName);
        const TUserKaskoTag::TDescription* description = descPtr
            ? dynamic_cast<const TUserKaskoTag::TDescription*>(descPtr.Get())
            : nullptr;
        R_ENSURE(description, ConfigHttpStatus.UnknownErrorStatus, "fail to get tag description: " + tagName);
        packageName = description->GetKaskoPackageName();
        R_ENSURE(packageName, ConfigHttpStatus.UnknownErrorStatus, "fail to get pack name: " + tagName);
        KaskoData.SetPackageName(packageName);
        {
            TString packageParams = description->GetKaskoPackageParams().GetStringRobust();
            SubstGlobal(packageParams, "__franchise__", KaskoData.GetFranchise());
            R_ENSURE(NJson::ReadJsonFastTree(packageParams, &packageMeta), ConfigHttpStatus.UnknownErrorStatus, "fail to parse kasko meta: " + packageParams);
            KaskoData.SetConditions(packageMeta);
        }

        ui32 discount = 0;
        auto tx = BuildTx<NSQL::ReadOnly>();
        auto dbScoreTags = tagsManager.GetUserTags().RestoreEntityTags({ permissions->GetUserId() }, { description->GetScoreTagName() }, tx);
        R_ENSURE(dbScoreTags, ConfigHttpStatus.IncompleteStatus, "fail to get user scoring tags", tx);
        const TScoringUserTag* scoreTag = dbScoreTags->empty() ? nullptr : dbScoreTags->front().GetTagAs<TScoringUserTag>();
        if (scoreTag && !description->GetDiscountMap().empty()) {
            discount = description->GetDiscount((ui32)scoreTag->GetValue());
            userScore = scoreTag->GetValue() / KaskoScoreCoefficient;
        }
        KaskoData.SetDiscount(discount);
    }

    TString tagId;
    {
        auto tag = tagsManager.GetTagsMeta().CreateTagAs<TUserKaskoTag>(tagName);
        R_ENSURE(tag, ConfigHttpStatus.IncompleteStatus, "wrong tag data: " + tagName);
        tag->SetKaskoData(KaskoData);
        auto tx = BuildTx<NSQL::Writable>();
        auto added = tagsManager.GetUserTags().AddTag(tag, permissions->GetUserId(), permissions->GetUserId(), Server, tx);
        R_ENSURE(added && !added->empty() && tx.Commit(), ConfigHttpStatus.UnknownErrorStatus, "unknown user", tx);
        tagId = added->front().GetTagId();
    }

    BuildCalculateParams(DriveApi->GetPrivateDataClient(), *user, std::move(CarData)).Apply([
        &server = *Server,
        kaskoClient,
        tagId,
        userScore,
        packageName = std::move(packageName),
        packageMeta = std::move(packageMeta)
    ](const NThreading::TFuture<NDrive::NRenins::NKasko::TCalculateOrder>& result) {
        auto order = result.GetValue();
        order.SetConditions(packageMeta);
        order.SetUserScore(userScore);
        NDrive::TEventLog::Log("KaskoCalculateRequest", NJson::TMapBuilder
            ("tag_id", tagId)
            ("request", order.GetReport())
        );
        return kaskoClient->Calculate(order, packageName).Apply([tagId, &server](const auto& result) {
            const auto& userTagsManager = Yensured(server.GetDriveAPI())->GetTagsManager().GetUserTags();
            auto tx = userTagsManager.BuildTx<NSQL::Writable>();
            auto tags = userTagsManager.RestoreTags({ tagId }, tx);
            if (!tags) {
                ythrow yexception() << "fail to restore tag: " << tagId << ", error: "  << tx.GetStringReport() ;
            }
            if (tags->empty()) {
                ythrow yexception() << "empty tags fetch: " << tagId;
            }
            auto tag = tags->front().MutableTagAs<TUserKaskoTag>();
            if (!tag || !tag->HasKaskoData() || tag->GetKaskoDataRef().GetStatus() != NDrive::NRenins::EKaskoStatus::NoData) {
                ythrow yexception() << "wrong tag data: " << tagId;
            }
            NDrive::NRenins::TKaskoData report;
            auto& kasko = tag->MutableKaskoDataRef();
            kasko.SetTagId(tagId);
            try {
                auto calcResult = result.GetValue();
                kasko.SetCalcFinalPrice(calcResult.GetPrice());
                kasko.SetOrderId(calcResult.GetAccountNumber());
            } catch (const NDrive::NRenins::TKaskoException& e) {
                kasko.SetErrorType(GetKaskoErrorMap(server).Value(e.GetCode(), "unknown"));
                kasko.SetErrorText(e.GetCode() + ":" + e.GetMessage());
            }
            report = kasko;
            const auto userId = tags->front().GetObjectId();
            if (!userTagsManager.UpdateTagData(tags->front(), userId, tx) || !tx.Commit()) {
                ythrow yexception() << "fail to update tag: " << tagId << ", error: " << tx.GetStringReport();
            }
            return report;
        });
    }).Subscribe([tagId, userId = permissions->GetUserId(), locale = GetLocale(), &server = *Server, report = g.GetReport()](const auto& result) {
        TJsonReport::TGuard g(report, HTTP_OK);
        TJsonReport& r = g.MutableReport();
        if (!result.HasValue() || result.HasException()) {
            r.AddReportElement("status", "error");
            r.AddReportElement("error", NThreading::GetExceptionInfo(result));
            if (auto api = server.GetDriveAPI()) {
                const auto& tagsManager = api->GetTagsManager();
                auto tx = tagsManager.GetUserTags().BuildTx<NSQL::Writable>();
                auto dbTags = tagsManager.GetUserTags().RestoreTags({ tagId }, tx);
                if (!dbTags) {
                    r.AddReportElement("sub_error", "Fail to fetch tags to remove");
                    r.AddReportElement("tx_error", tx.GetReport());
                } else if (!dbTags->empty() && !tagsManager.GetUserTags().RemoveTags(*dbTags, userId, &server, tx, /* force = */ true) || !tx.Commit()) {
                    r.AddReportElement("sub_error", "Fail to remove tags");
                    r.AddReportElement("tx_error", tx.GetReport());
                }
            }
            g.SetCode(HttpCodes::HTTP_INTERNAL_SERVER_ERROR);
        } else {
            r.SetExternalReport(GetKaskoReport({ result.GetValue() }, locale, server));
            g.SetCode(HTTP_OK);
        }
    });

    g.Release();
}

NDrive::TScheme TUserInsuranceCalculateProcessor::GetRequestDataScheme(const IServerBase* server, const TCgiParameters& schemeCgi) {
    const bool needFull = schemeCgi.Get("report") == "full";
    const TString brand = schemeCgi.Get("auto_brand_code");
    auto buildVariants = [&server = *Yensured(server), &schemeCgi](const TString& key, const TString& desc, NDrive::TScheme& scheme, const bool refresh = false) {
        auto& field = scheme.Add<TFSVariants>(key, desc);
        if (auto strParam = server.GetSettings().GetValueDef<TString>(KaskoSettingsPrefix + key, "")) {
            NJson::TJsonValue json;
            R_ENSURE(NJson::ReadJsonFastTree(strParam, &json), HTTP_INTERNAL_SERVER_ERROR, "fail to parse data");
            SetVariantsByJson(json, field, schemeCgi.Get(key));
        }
        field.SetRequired(true);
        field.SetRefreshOnUpdate(refresh);
    };
    auto getJsonParam = [&server = *server](const TString& key) {
        auto strParam = server.GetSettings().GetValueDef<TString>(KaskoSettingsPrefix + key, "");
        R_ENSURE(strParam, HTTP_INTERNAL_SERVER_ERROR, "empty " + key + " data");
        NJson::TJsonValue json;
        R_ENSURE(NJson::ReadJsonFastTree(strParam, &json), HTTP_INTERNAL_SERVER_ERROR, "fail to parse " + key + " data");
        return json;
    };
    NDrive::TScheme scheme;

    auto& insScheme = scheme.Add<TFSStructure>("insurance").SetStructure<NDrive::TScheme>();
    buildVariants("franchise", "Франшиза", insScheme);
    auto minStart = Now() + server->GetSettings().GetValueDef<TDuration>(KaskoSettingsPrefix + "start_from_now.min", TDuration::Hours(25));
    auto maxStart = Now() + server->GetSettings().GetValueDef<TDuration>(KaskoSettingsPrefix + "start_from_now.max", TDuration::Days(59));
    SetDefault(schemeCgi, insScheme.Add<TFSNumeric>("date_start", "Дата старта страховки")).SetVisual(TFSNumeric::DateTime).SetMin(minStart.Seconds()).SetMax(maxStart.Seconds()).SetRequired(true);
    auto& dateEndScheme = SetDefault(schemeCgi, insScheme.Add<TFSNumeric>("date_end", "Дата завершения"), AddMonths(minStart, 12).Seconds()).SetVisual(TFSNumeric::DateTime);
    if (auto setting = server->GetSettings().GetValue<TDuration>(KaskoSettingsPrefix + "end_from_now.min")) {
        dateEndScheme.SetMin((Now() + *setting).Seconds());
    }
    if (auto setting = server->GetSettings().GetValue<TDuration>(KaskoSettingsPrefix + "end_from_now.max")) {
        dateEndScheme.SetMax((Now() + *setting).Seconds());
    }

    auto& carScheme = scheme.Add<TFSStructure>("car").SetStructure<NDrive::TScheme>();
    buildVariants("auto_brand_code", "Марка", carScheme, true);
    if (!needFull && !brand) {
        return scheme;
    }
    const TString model = schemeCgi.Get("auto_model_code");
    if (brand) {
        R_ENSURE(brand, HTTP_INTERNAL_SERVER_ERROR, "brand param is required");
        auto json = getJsonParam("auto_brand_code.auto_model_code");
        if (json.Has(brand)) {
            auto& field = carScheme.Add<TFSVariants>("auto_model_code", "Модель");
            SetVariantsByJson(json[brand], field, model);
            field.SetRefreshOnUpdate(true);
        } else {
            buildVariants("auto_model_code", "Модель", carScheme, true);
        }
    } else {
        buildVariants("auto_model_code", "Модель", carScheme, true);
    }
    if (!needFull && !model) {
        return scheme;
    }
    TMaybe<TSchemePresets> presets;
    if (model) {
        auto json = getJsonParam("auto_model_code.meta");
        if (json.Has(brand)) {
            if (json[brand].Has(model)) {
                presets.ConstructInPlace();
                R_ENSURE(presets->Parse(json[brand][model]), HTTP_INTERNAL_SERVER_ERROR, "fail to parse presets");
            } else {
                return scheme;
            }
        }
    }
    if (presets) {
        auto& bodies = carScheme.Add<TFSVariants>("auto_body_type_code", "Тип кузова");
        SetVariantsByJson(presets->GetBodies(), bodies);
        auto& years = carScheme.Add<TFSVariants>("auto_year_of_issue_code", "Год выпуска").SetVariants(presets->GetYears());
        if (auto val = schemeCgi.Get("auto_year_of_issue_code"); val && presets->GetYears().contains(val)) {
            years.SetDefault(val);
        }
        auto& enginePowers = carScheme.Add<TFSVariants>("engine_power_code", "Сила двигателя л.с").SetVariants(presets->GetEnginePowers());
        if (auto val = schemeCgi.Get("engine_power_code"); val && presets->GetEnginePowers().contains(val)) {
            enginePowers.SetDefault(val);
        }
        auto& cost = carScheme.Add<TFSNumeric>("auto_cost", "Стоимость автомобиля").SetVisual(TFSNumeric::Money);
        cost.SetMin(presets->GetMinPriceDef(0));
        if (presets->HasDefPrice()) {
            cost.SetDefault(presets->GetDefPriceRef());
        }
        if (presets->HasMaxPrice()) {
            cost.SetMax(presets->GetMaxPriceRef());
        }
        SetDefault(schemeCgi, cost);
        bodies.SetRequired(true);
        years.SetRequired(true);
        enginePowers.SetRequired(true);
        cost.SetRequired(true);
    } else {
        buildVariants("auto_body_type_code", "Тип кузова", carScheme);
        SetDefault(schemeCgi, carScheme.Add<TFSNumeric>("auto_year_of_issue_code", "Год выпуска").SetMin(2000)).SetRequired(true);
        SetDefault(schemeCgi, carScheme.Add<TFSNumeric>("engine_power_code", "Сила двигателя л.с")).SetRequired(true);
        SetDefault(schemeCgi, carScheme.Add<TFSNumeric>("auto_cost", "Стоимость автомобиля").SetVisual(TFSNumeric::Money).SetMin(0)).SetRequired(true);
    }
    SetDefault(schemeCgi, carScheme.Add<TFSBoolean>("in_credit", "Находится ли машина в кредите")).AddHidingRule(false, "bank_of_credit").SetRequired(true);
    buildVariants("bank_of_credit", "Название банка", carScheme);
    SetDefault(schemeCgi, carScheme.Add<TFSString>("registration_number", "Гос. номер")).SetRequired(true);
    SetDefault(schemeCgi, carScheme.Add<TFSString>("vin", "VIN")).SetRequired(true);
    SetDefault(schemeCgi, carScheme.Add<TFSString>("sts_number", "Серия и номер СТС")).SetRequired(true);
    SetDefault(schemeCgi, carScheme.Add<TFSNumeric>("sts_issue_date", "Дата выпуска СТС")).SetVisual(TFSNumeric::DateTime).SetRequired(true);
    SetDefault(schemeCgi, carScheme.Add<TFSNumeric>("mileage", "Пробег").SetMin(0)).SetRequired(true);

    return scheme;
}

void TUserInsuranceOrderProcessor::Parse(const NJson::TJsonValue& requestData) {
    for (auto&& driver : requestData["drivers"].GetArray()) {
        auto driverData = NDrive::NRenins::ParseDriverKaskoOrderData(driver);
        R_ENSURE(driverData, ConfigHttpStatus.SyntaxErrorStatus, "fail to parse driver data");
        AdditionalDrivers.emplace_back(std::move(*driverData));
    }
}

void TUserInsuranceOrderProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    auto kaskoClient = Yensured(Server)->GetReninsKaskoClient();
    R_ENSURE(kaskoClient, ConfigHttpStatus.UnknownErrorStatus, "unknown insurance client");
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Control, TAdministrativeAction::EEntity::UserInsurance, { "kasko" });
    const auto& tagsManager = Yensured(Server->GetDriveAPI())->GetTagsManager();
    auto id = GetUUID(Context->GetCgiParameters(), "id", /* required = */ false);
    if (!id) {
        id = GetUUID(requestData, "id", /* required = */ false);
    }
    const auto tagName = GetHandlerSettingDef<TString>("tag_name", TUserKaskoTag::Type());

    TString succeedPaymentUrl = Server->GetSettings().GetValueDef<TString>(KaskoSettingsPrefix + "success_payment_link", "");
    TString failedPaymentUrl = Server->GetSettings().GetValueDef<TString>(KaskoSettingsPrefix + "failure_payment_link", "");
    R_ENSURE(succeedPaymentUrl && failedPaymentUrl, ConfigHttpStatus.UnknownErrorStatus, "unknown payment link");

    TUsersDB::TOptionalUser user;
    NDrive::NRenins::TKaskoData kaskoData;
    TString tagId;
    {
        auto tx = BuildTx<NSQL::ReadOnly>();
        user = DriveApi->GetUserManager().RestoreUser(permissions->GetUserId(), tx);
        R_ENSURE(user, ConfigHttpStatus.IncompleteStatus, "unknown user");
        auto dbTags = id
            ? tagsManager.GetUserTags().RestoreTags(id, tx)
            : tagsManager.GetUserTags().RestoreEntityTags({ permissions->GetUserId() }, { tagName }, tx);
        R_ENSURE(dbTags, ConfigHttpStatus.IncompleteStatus, "fail to get user tags", tx);
        for(auto&& dbTag : *dbTags) {
            auto tag = dbTag.GetTagAs<TUserKaskoTag>();
            if (!tag || !tag->HasKaskoData() || tag->GetKaskoDataRef().GetStatus() == NDrive::NRenins::EKaskoStatus::Signed || !tag->GetKaskoDataRef().GetOrderId()) {
                continue;
            }
            kaskoData = std::move(tag->GetKaskoDataRef());
            tagId = dbTag.GetTagId();
            break;
        }
        R_ENSURE(kaskoData.GetOrderId(), ConfigHttpStatus.SyntaxErrorStatus, "fail to get insurance data");
    }
    BuildStartOrederParams( DriveApi->GetPrivateDataClient(), *user).Apply([
        &server = *Server,
        kaskoClient,
        tagId = std::move(tagId),
        kaskoData = std::move(kaskoData),
        additionalDrivers = std::move(AdditionalDrivers),
        succeedPaymentUrl = std::move(succeedPaymentUrl),
        failedPaymentUrl = std::move(failedPaymentUrl)
    ](const NThreading::TFuture<NDrive::NRenins::NKasko::TStartOrderData>& result) mutable {
        auto order = result.GetValue();
        for (auto&& driver : additionalDrivers) {
            order.MutableDrivers().emplace_back(std::move(driver));
        }
        TVector<TString> drivers;
        for(const auto& driver : order.GetDrivers()) {
            const auto& user = driver.GetUserData();
            auto name = TStringBuilder()
                << user.GetLastName()
                << " " << user.GetName();
            if (user.HasMiddleName()) {
                name << " " << user.GetMiddleNameRef();
            }
            drivers.emplace_back(name);
        }
        order.SetConditions(std::move(kaskoData.MutableConditions()));
        order.SetCar(std::move(kaskoData.MutableCar()));
        order.SetPackName(kaskoData.GetPackageName());
        order.SetDateStart(kaskoData.GetInsuranceStart());
        order.SetDateEnd(kaskoData.GetInsuranceEnd());
        order.MutableOrderData().OrderId = kaskoData.GetOrderId();
        order.MutableOrderData().SucceedPaymentUrl = succeedPaymentUrl;
        order.MutableOrderData().FailedPaymentUrl = failedPaymentUrl;
        NDrive::TEventLog::Log("KaskoOrderRequest", NJson::TMapBuilder
            ("tag_id", tagId)
            ("request", order.GetReport())
        );
        return kaskoClient->Order(order).Apply([&server, tagId, drivers = std::move(drivers)](const auto& result) {
            const auto& userTagsManager = Yensured(server.GetDriveAPI())->GetTagsManager().GetUserTags();
            auto tx = userTagsManager.BuildTx<NSQL::Writable>();
            auto tags = userTagsManager.RestoreTags({ tagId }, tx);
            if (!tags) {
                ythrow yexception() << "fail to restore tag: " << tagId << ", error: "  << tx.GetStringReport() ;
            }
            if (tags->empty()) {
                ythrow yexception() << "empty tags fetch: " << tagId;
            }
            auto tag = tags->front().MutableTagAs<TUserKaskoTag>();
            if (!tag || !tag->HasKaskoData()) {
                ythrow yexception() << "wrong tag data: " << tagId;
            }
            auto& kasko = tag->MutableKaskoDataRef();
            kasko.SetDrivers(drivers);
            kasko.SetTagId(tagId);
            try {
                auto status = result.GetValue();
                kasko.SetStatus(status);
                if (!status.HasPrice()) {
                    ythrow yexception() << "fail to get price: " << tagId;
                }
                kasko.SetOrderFinalPrice(status.GetPriceRef());
                kasko.SetPaymentLink(status.GetPaymentLink());
            } catch (const NDrive::NRenins::TKaskoException& e) {
                kasko.SetErrorType(GetKaskoErrorMap(server).Value(e.GetCode(), "unknown"));
                kasko.SetErrorText(e.GetCode() + ":" + e.GetMessage());
            }
            const auto userId = tags->front().GetObjectId();
            if (!userTagsManager.UpdateTagData(tags->front(), userId, tx) || !tx.Commit()) {
                ythrow yexception() << "fail to update tag: " << tagId << ", error: " << tx.GetStringReport();
            }
            return std::move(kasko);
        });
    }).Subscribe([locale = GetLocale(), &server = *Server, report = g.GetReport()](const auto& result) {
        TJsonReport::TGuard g(report, HTTP_OK);
        TJsonReport& r = g.MutableReport();
        if (!result.HasValue() || result.HasException()) {
            r.AddReportElement("status", "error");
            r.AddReportElement("error", NThreading::GetExceptionInfo(result));
            g.SetCode(HttpCodes::HTTP_INTERNAL_SERVER_ERROR);
        } else {
            r.SetExternalReport(GetKaskoReport({ result.GetValue() }, locale, server));
            g.SetCode(HTTP_OK);
        }
    });
    g.Release();
}

NDrive::TScheme TUserInsuranceOrderProcessor::GetRequestDataScheme(const IServerBase* /* server */, const TCgiParameters& /* schemeCgi */) {
    NDrive::TScheme scheme;
    scheme.Add<TFSArray>("drivers").SetElement(GetDriverScheme());
    return scheme;
}

TVector<NDrive::NRenins::TKaskoData> GetKaskoInfos(TVector<TDBTag>& dbTags, const TString& skipId = "") {
    TVector<NDrive::NRenins::TKaskoData> infos;
    for (auto&& dbTag : dbTags) {
        if (skipId && dbTag.GetTagId() == skipId) {
            continue;
        }
        if (auto tag = dbTag.MutableTagAs<TUserKaskoTag>(); tag && tag->HasKaskoData()) {
            auto& data = tag->MutableKaskoDataRef();
            data.SetTagId(dbTag.GetTagId());
            infos.emplace_back(std::move(data));
        }
    }
    return infos;
}

void TUserInsuranceStatusProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    auto kaskoClient = Yensured(Server)->GetReninsKaskoClient();
    R_ENSURE(kaskoClient, ConfigHttpStatus.UnknownErrorStatus, "unknown insurance client");
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Control, TAdministrativeAction::EEntity::UserInsurance, { "kasko" });

    bool force = true;
    if (auto param = GetValue<bool>(requestData, "force", /* required = */ false)) {
        force = *param;
    } else if (auto param = GetValue<bool>(Context->GetCgiParameters(), "force", /* required = */ false)) {
        force = *param;
    }
    const TString id = GetUUID(Context->GetCgiParameters(), "id", /* required = */ false);

    IEntityTagsManager::TOptionalTags dbTags;
    const auto& tagsManager = Yensured(Server->GetDriveAPI())->GetTagsManager();
    {
        const auto tagName = GetHandlerSettingDef<TString>("tag_name", TUserKaskoTag::Type());
        auto tx = BuildTx<NSQL::ReadOnly>();
        dbTags = id
            ? tagsManager.GetUserTags().RestoreTags(id, tx)
            : tagsManager.GetUserTags().RestoreEntityTags({ permissions->GetUserId() }, { tagName }, tx);
        R_ENSURE(dbTags, ConfigHttpStatus.IncompleteStatus, "fail to get user tags", tx);
    }

    if (!force) {
        auto infos = GetKaskoInfos(*dbTags);
        g.SetExternalReport(GetKaskoReport(infos, GetLocale(), *Server));
        g.SetCode(HTTP_OK);
        return;
    }

    TString orderId;
    TString tagId;
    TString dealId;
    bool needPaymentLink = true;
    bool needDocument = true;
    TVector<NDrive::NRenins::TKaskoData> asyncReportInfos;
    {
        for(auto&& dbTag : *dbTags) {
            auto tag = dbTag.GetTagAs<TUserKaskoTag>();
            if (!tag || !tag->HasKaskoData() || tag->GetKaskoDataRef().GetStatus() == NDrive::NRenins::EKaskoStatus::Signed || !tag->GetKaskoDataRef().GetOrderId()) {
                continue;
            }
            orderId = tag->GetKaskoDataRef().GetOrderId();
            needPaymentLink = tag->GetKaskoDataRef().GetStatus() == NDrive::NRenins::EKaskoStatus::Processing && !tag->GetKaskoDataRef().GetPaymentLink();
            if (tag->GetKaskoDataRef().HasStatus()) {
                dealId = tag->GetKaskoDataRef().GetStatusRef().GetDealId();
            }
            needDocument = dealId && tag->GetKaskoDataRef().GetStatusRef().GetPaymentStatus() == NDrive::NRenins::NKasko::EPaymentStatus::Paid;
            tagId = dbTag.GetTagId();
            break;
        }
        if (!id && !orderId) {
            auto infos = GetKaskoInfos(*dbTags);
            g.SetExternalReport(GetKaskoReport(infos, GetLocale(), *Server));
            g.SetCode(HTTP_OK);
            return;
        } else {
            asyncReportInfos = GetKaskoInfos(*dbTags, tagId);
        }
        R_ENSURE(orderId, ConfigHttpStatus.SyntaxErrorStatus, "fail to get insurance data");
    }

    TVector<NThreading::TFuture<void>> futures;
    auto statusFuture = kaskoClient->GetStatus(orderId);
    futures.emplace_back(NThreading::Initialize(statusFuture).IgnoreResult());
    TMaybe<NThreading::TFuture<TString>> paymentFuture;
    if (needPaymentLink) {
        TString succeedPaymentUrl = Server->GetSettings().GetValueDef<TString>(KaskoSettingsPrefix + "success_payment_link", "");
        TString failedPaymentUrl = Server->GetSettings().GetValueDef<TString>(KaskoSettingsPrefix + "failure_payment_link", "");
        R_ENSURE(succeedPaymentUrl && failedPaymentUrl, ConfigHttpStatus.UnknownErrorStatus, "unknown payment link");
        paymentFuture = kaskoClient->GetPaymentLink({ orderId, succeedPaymentUrl, failedPaymentUrl });
        futures.emplace_back(NThreading::Initialize(*paymentFuture).IgnoreResult());
    }
    TMaybe<NThreading::TFuture<TString>> documentFuture;
    if (needDocument) {
        TString bucketName = Server->GetSettings().GetValueDef<TString>(KaskoSettingsPrefix + "bucket", "");
        R_ENSURE(bucketName, ConfigHttpStatus.UnknownErrorStatus, "unknown bucket");
        R_ENSURE(Server->GetMDSClient(), ConfigHttpStatus.UnknownErrorStatus, "unknown bucket");
        auto bucket = Server->GetMDSClient()->GetBucket(bucketName);
        R_ENSURE(bucket, ConfigHttpStatus.UnknownErrorStatus, "wrong bucket");
        const TString path = TStringBuilder() << "kasko" << "/" << permissions->GetUserId() << "/" << tagId << ".pdf";
        documentFuture = bucket->HasKey(path).Apply([kaskoClient, bucket, orderId, dealId, path](const auto& report) {
            if (report.GetValue()) {
                return NThreading::MakeFuture(path);
            }
            return Yensured(kaskoClient)->GetDocument(orderId, dealId).Apply([bucket, path](const NThreading::TFuture<TString>& report) {
                return NS3::TBucket::CheckReply(Yensured(bucket)->PutKey(path, report.GetValue())).Apply([path](const auto&) { return path; });
            });
        });
        futures.emplace_back(NThreading::Initialize(*documentFuture).IgnoreResult());
    }

    NThreading::WaitAll(futures).Apply([
        statusFuture = std::move(statusFuture),
        paymentFuture = std::move(paymentFuture),
        documentFuture = std::move(documentFuture),
        tagId,
        &server = *Server
    ](const NThreading::TFuture<void>&) mutable {
        const auto& userTagsManager = Yensured(server.GetDriveAPI())->GetTagsManager().GetUserTags();
        auto tx = userTagsManager.BuildTx<NSQL::Writable>();
        auto tags = userTagsManager.RestoreTags({ tagId }, tx);
        if (!tags) {
            ythrow yexception() << "fail to restore tag: " << tagId << ", error: "  << tx.GetStringReport() ;
        }
        if (tags->empty()) {
            ythrow yexception() << "empty tags fetch: " << tagId;
        }
        auto tag = tags->front().MutableTagAs<TUserKaskoTag>();
        if (!tag || !tag->HasKaskoData()) {
            ythrow yexception() << "wrong tag data: " << tagId;
        }
        auto& kasko = tag->MutableKaskoDataRef();
        kasko.SetTagId(tagId);
        bool needUpdate = false;
        try {
            auto status = statusFuture.GetValue();
            if (auto currentStatus = kasko.OptionalStatus()) {
                needUpdate |= currentStatus->GetAccountStatus() != status.GetAccountStatus()
                    || currentStatus->GetPsoStatus() != status.GetPsoStatus()
                    || currentStatus->GetPaymentStatus() != status.GetPaymentStatus();
            }
            kasko.SetStatus(status);
        } catch (const NDrive::NRenins::TKaskoException& e) {
            if (TString errorType = GetKaskoErrorMap(server).Value(e.GetCode(), "unknown"); kasko.GetErrorType() != errorType) {
                kasko.SetErrorType(errorType);
                kasko.SetErrorText(e.GetCode() + ":" + e.GetMessage());
                needUpdate = true;
            }
        }
        if (paymentFuture) {
            if (paymentFuture->HasValue()) {
                kasko.SetPaymentLink(paymentFuture->GetValue());
                needUpdate = true;
            } else {
                NDrive::TEventLog::Log("KaskoPaymentLinkError", NJson::TMapBuilder
                    ("error", NThreading::GetExceptionMessage(*paymentFuture))
                    ("tag_id", tagId)
                );
            }
        }
        if (documentFuture) {
            if (documentFuture->HasValue()) {
                kasko.SetDocumentPath(documentFuture->GetValue());
                needUpdate = true;
            } else {
                NDrive::TEventLog::Log("KaskoDocumentError", NJson::TMapBuilder
                    ("error", NThreading::GetExceptionMessage(*documentFuture))
                    ("tag_id", tagId)
                );
            }
        }
        if (!needUpdate) {
            return std::move(kasko);
        }
        const auto userId = tags->front().GetObjectId();
        if (!userTagsManager.UpdateTagData(tags->front(), userId, tx) || !tx.Commit()) {
            ythrow yexception() << "fail to update tag: " << tagId << ", error: " << tx.GetStringReport();
        }
        return std::move(kasko);
    }).Subscribe([locale = GetLocale(), &server = *Server, report = g.GetReport(), infos = std::move(asyncReportInfos)](const auto& result) mutable {
        TJsonReport::TGuard g(report, HTTP_OK);
        TJsonReport& r = g.MutableReport();
        if (!result.HasValue() || result.HasException()) {
            r.AddReportElement("status", "error");
            r.AddReportElement("error", NThreading::GetExceptionInfo(result));
            g.SetCode(HttpCodes::HTTP_INTERNAL_SERVER_ERROR);
        } else {
            infos.emplace_back(result.GetValue());
            r.SetExternalReport(GetKaskoReport(infos, locale, server));
            g.SetCode(HTTP_OK);
        }
    });
    g.Release();
}

void TUserInsuranceCarProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /* requestData */) {
    auto kaskoClient = Yensured(Server)->GetReninsKaskoClient();
    R_ENSURE(kaskoClient, ConfigHttpStatus.UnknownErrorStatus, "unknown insurance client");
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Control, TAdministrativeAction::EEntity::UserInsurance, { "kasko" });

    const TString grz = GetString(Context->GetCgiParameters(), "registration_number", /* required = */ true);
    kaskoClient->GetAutocodeData(NDrive::ToCyrillic(grz)).Subscribe([report = g.GetReport()](const auto& result) {
        TJsonReport::TGuard g(report, HTTP_OK);
        TJsonReport& r = g.MutableReport();
        if (!result.HasValue() || result.HasException()) {
            r.AddReportElement("status", "error");
            r.AddReportElement("error", NThreading::GetExceptionInfo(result));
            g.SetCode(HttpCodes::HTTP_INTERNAL_SERVER_ERROR);
        } else {
            r.SetExternalReport(result.GetValue().GetReport());
        }
    });
    g.Release();
}

void TUserInsuranceDriverProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /* requestData */) {
    TUsersDB::TOptionalUser user;
    {
        auto tx = BuildTx<NSQL::ReadOnly>();
        user = DriveApi->GetUserManager().RestoreUser(permissions->GetUserId(), tx);
        R_ENSURE(user, ConfigHttpStatus.IncompleteStatus, "unknown user");
    }

    auto dlFuture = DriveApi->GetPrivateDataClient().GetDrivingLicense(*user, user->GetDrivingLicenseDatasyncRevision()).Apply(
        [](const NThreading::TFuture<TUserDrivingLicenseData>& dlFuture)
    {
        return ParseDriver(dlFuture.GetValue());
    }).Subscribe([report = g.GetReport()](const auto& result) {
        TJsonReport::TGuard g(report, HTTP_OK);
        TJsonReport& r = g.MutableReport();
        if (!result.HasValue() || result.HasException()) {
            r.AddReportElement("status", "error");
            r.AddReportElement("error", NThreading::GetExceptionInfo(result));
            g.SetCode(HttpCodes::HTTP_INTERNAL_SERVER_ERROR);
        } else {
            r.SetExternalReport(GetDriverScheme(result.GetValue().GetReport(), /* readOnly = */ true).SerializeToJson());
            g.SetCode(HTTP_OK);
        }
    });
    g.Release();
}

void TUserInsurancePresetsProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr /* permissions */, const NJson::TJsonValue& /* requestData */) {
    auto getJsonParam = [&server = *Yensured(Server)](const TString& key) {
        auto strParam = server.GetSettings().GetValueDef<TString>(KaskoSettingsPrefix + key, "");
        R_ENSURE(strParam, HTTP_INTERNAL_SERVER_ERROR, "empty " + key + " data");
        NJson::TJsonValue json;
        R_ENSURE(NJson::ReadJsonFastTree(strParam, &json), HTTP_INTERNAL_SERVER_ERROR, "fail to parse " + key + " data");
        return json;
    };
    if (GetValue<bool>(Context->GetCgiParameters(), "full", /* required = */ false).GetOrElse(false)) {
        g.SetExternalReport(getJsonParam("auto_model_code.all"));
        g.SetCode(HTTP_OK);
        return;
    }
    const TString brand = GetString(Context->GetCgiParameters(), "auto_brand_code", /* required = */ false);
    const TString model = GetString(Context->GetCgiParameters(), "auto_model_code", /* required = */ false);
    if (!brand && !model) {
        g.AddReportElement("auto_brand_code", getJsonParam("auto_brand_code"));
    } else if (!model) {
        R_ENSURE(brand, HTTP_INTERNAL_SERVER_ERROR, "brand param is required");
        auto json = getJsonParam("auto_brand_code.auto_model_code");
        R_ENSURE(json.Has(brand), HTTP_INTERNAL_SERVER_ERROR, "unknown brand");
        g.AddReportElement("auto_model_code", std::move(json[brand]));
    } else {
        auto json = getJsonParam("auto_model_code.meta");
        R_ENSURE(json.Has(model), HTTP_INTERNAL_SERVER_ERROR, "unknown model");
        g.SetExternalReport(std::move(json[model]));
    }
    g.SetCode(HTTP_OK);
}

void TUserInsuranceFixDataProcessor::Parse(const NJson::TJsonValue& requestData) {
    {
        auto carData = NDrive::NRenins::ParseCarKaskoOrderData(requestData["car"]);
        R_ENSURE(carData, ConfigHttpStatus.SyntaxErrorStatus, "fail to parse car order data");
        KaskoData.SetCar(std::move(*carData));
        const bool parseCarParams = NJson::ParseField(requestData["car"], "auto_brand_code", KaskoData.MutableCarReportData().MutableBrand(), /* required = */ false)
            && NJson::ParseField(requestData["car"], "auto_model_code", KaskoData.MutableCarReportData().MutableModel(), /* required = */ false);
        R_ENSURE(parseCarParams, ConfigHttpStatus.SyntaxErrorStatus, "fail to parse insurance data");
    }
    {
        const bool parsedInsuranceParams = NJson::ParseField(requestData["insurance"], "date_start", KaskoData.MutableInsuranceStart(), /* required = */ true)
            && NJson::ParseField(requestData["insurance"], "date_end", KaskoData.MutableInsuranceEnd(), /* required = */ false)
            && NJson::ParseField(requestData["insurance"], "franchise", KaskoData.MutableFranchise(), /* required = */ true);
        R_ENSURE(parsedInsuranceParams, ConfigHttpStatus.SyntaxErrorStatus, "fail to parse insurance data");
    }
    if (requestData.Has("state")) {
        {
            const bool parsedStateParams = NJson::ParseField(requestData["state"], "drivers", KaskoData.MutableDrivers(), /* required = */ false)
                && NJson::ParseField(requestData["state"], "package_name", KaskoData.MutablePackageName(), /* required = */ false)
                && NJson::ParseField(requestData["state"], "order_id", KaskoData.MutableOrderId(), /* required = */ false)
                && NJson::ParseField(requestData["state"], "calc_final_price", KaskoData.MutableCalcFinalPrice(), /* required = */ false)
                && NJson::ParseField(requestData["state"], "order_final_price", KaskoData.MutableOrderFinalPrice(), /* required = */ false)
                && NJson::ParseField(requestData["state"], "discount", KaskoData.MutableDiscount(), /* required = */ false)
                && NJson::ParseField(requestData["state"], "payment_link", KaskoData.MutablePaymentLink(), /* required = */ false)
                && NJson::ParseField(requestData["state"], "document_link", KaskoData.MutableDocumentPath(), /* required = */ false);
            R_ENSURE(parsedStateParams, ConfigHttpStatus.SyntaxErrorStatus, "fail to parse state data");
        }
        if (requestData["state"].Has("state")) {
            auto& state = KaskoData.MutableStatus();
            state.ConstructInPlace();
            const bool parsedStateParams = NJson::ParseField(requestData["state"]["state"], "account_status", NJson::Stringify(state->MutableAccountStatus()), /* required = */ false)
                && NJson::ParseField(requestData["state"]["state"], "pso_status", NJson::Stringify(state->MutablePsoStatus()), /* required = */ false)
                && NJson::ParseField(requestData["state"]["state"], "payment_status", NJson::Stringify(state->MutablePaymentStatus()), /* required = */ false)
                && NJson::ParseField(requestData["state"]["state"], "price", state->MutablePrice(), /* required = */ false);
            R_ENSURE(parsedStateParams, ConfigHttpStatus.SyntaxErrorStatus, "fail to parse state data");
        }
    }
}

void TUserInsuranceFixDataProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    const auto& tagsManager = Yensured(DriveApi)->GetTagsManager();
    if (auto id = GetString(requestData, "id", /* required = */ false)) {
        auto tx = BuildTx<NSQL::Writable>();
        auto tags = tagsManager.GetUserTags().RestoreTags({ id }, tx);
        R_ENSURE(tags && !tags->empty(), ConfigHttpStatus.UnknownErrorStatus, "fail to restore tag", tx);
        auto tag = tags->front().MutableTagAs<TUserKaskoTag>();
        R_ENSURE(tag, ConfigHttpStatus.UnknownErrorStatus, "tag has wrong type");
        tag->SetKaskoData(KaskoData);
        R_ENSURE(tagsManager.GetUserTags().UpdateTagData(tags->front(), permissions->GetUserId(), tx) && tx.Commit(), ConfigHttpStatus.UnknownErrorStatus, "fail to update tag", tx);
        KaskoData.SetTagId(id);
    } else {
        const auto tagName = GetHandlerSettingDef<TString>("tag_name", TUserKaskoTag::Type());
        R_ENSURE(tagName, ConfigHttpStatus.UnknownErrorStatus, "empty tag name");
        auto tag = tagsManager.GetTagsMeta().CreateTagAs<TUserKaskoTag>(tagName);
        R_ENSURE(tag, ConfigHttpStatus.IncompleteStatus, "wrong tag data: " + tagName);
        tag->SetKaskoData(KaskoData);
        auto tx = BuildTx<NSQL::Writable>();
        auto added = tagsManager.GetUserTags().AddTag(tag, permissions->GetUserId(), permissions->GetUserId(), Yensured(Server), tx);
        R_ENSURE(added && !added->empty() && tx.Commit(), ConfigHttpStatus.UnknownErrorStatus, "fail to add tag", tx);
        KaskoData.SetTagId(added->front().GetTagId());
    }
    g.SetExternalReport(GetKaskoReport({ KaskoData }, GetLocale(), *Yensured(Server)));
    g.SetCode(HTTP_OK);
}

