#include "processor.h"

#include "sessions_context.h"
#include "user_context.h"

#include <drive/backend/processors/car_scanner/processor.h>
#include <drive/backend/processors/common_app/fetcher.h>
#include <drive/backend/processors/leasing/get_signals.h>
#include <drive/backend/processors/sessions/aggression_helper.h>
#include <drive/backend/processors/sessions/base_impl.h>

#include <drive/backend/abstract/notifier.h>
#include <drive/backend/actions/evolution_policy.h>
#include <drive/backend/actions/session_photo_screen.h>
#include <drive/backend/areas/areas.h>
#include <drive/backend/common/localization.h>
#include <drive/backend/data/achievement.h>
#include <drive/backend/data/additional_service.h>
#include <drive/backend/data/dedicated_fleet.h>
#include <drive/backend/data/delegation.h>
#include <drive/backend/data/dictionary_tags.h>
#include <drive/backend/data/sharing.h>
#include <drive/backend/data/telematics.h>
#include <drive/backend/data/transformation.h>
#include <drive/backend/data/scoring/scoring.h>
#include <drive/backend/database/drive/private_data.h>
#include <drive/backend/device_snapshot/image.h>
#include <drive/backend/history_iterator/history_iterator.h>
#include <drive/backend/insurance/task/history.h>
#include <drive/backend/logging/events.h>
#include <drive/backend/notifications/mail/binder.h>
#include <drive/backend/notifications/push/push.h>
#include <drive/backend/offers/action.h>
#include <drive/backend/offers/context.h>
#include <drive/backend/offers/manager.h>
#include <drive/backend/offers/actions/flexipack.h>
#include <drive/backend/offers/actions/long_term.h>
#include <drive/backend/offers/actions/pack.h>
#include <drive/backend/offers/actions/rental_offer.h>
#include <drive/backend/offers/offers/additional_service.h>
#include <drive/backend/offers/offers/dedicated_fleet.h>
#include <drive/backend/roles/manager.h>
#include <drive/backend/saas/api.h>
#include <drive/backend/sessions/manager/billing.h>
#include <drive/backend/signalq/signals/helpers.h>
#include <drive/backend/signalq/signals/tag.h>
#include <drive/backend/tracks/report.h>
#include <drive/backend/user_document_photos/manager.h>
#include <drive/backend/users/user.h>

#include <drive/library/cpp/raw_text/phone_number.h>
#include <drive/library/cpp/taxi/antifraud/client.h>
#include <drive/library/cpp/threading/container.h>
#include <drive/telematics/protocol/errors.h>
#include <drive/telematics/server/tasks/lite.h>

#include <kernel/daemon/common/time_guard.h>

#include <library/cpp/json/json_writer.h>
#include <library/cpp/string_utils/parse_vector/vector_parser.h>

#include <rtline/library/unistat/signals.h>
#include <rtline/util/blob_with_header.h>
#include <rtline/util/algorithm/container.h>
#include <rtline/util/algorithm/ptr.h>
#include <rtline/util/algorithm/type_traits.h>

#include <util/generic/guid.h>
#include <util/string/vector.h>

#include <ranges>


namespace {
    auto MakeCommandCallback(IServerReportBuilder::TPtr report, TUserPermissions::TPtr permissions, const NDrive::IServer* server, ELocalization locale, bool verbose = false) {
        return [report, permissions, server, locale, verbose](const NThreading::TFuture<NDrive::TCommonCommandResponse>& r) {
            if (report) {
                NDrive::TEventLog::TUserIdGuard userIdGuard(permissions->GetUserId());
                TJsonReport::TGuard g(report, HTTP_OK);
                const auto& response = r.GetValue();
                g.AddReportElement("telematic_status", ToString(response.Status));
                if (response.Status != NDrive::TTelematicsClient::EStatus::Success) {
                    const auto code = NDrive::ParseError(response.Message);
                    const auto localizedMessage = TTelematicsCommandTag::GetUserErrorDescription(code, locale, *permissions, *server);
                    const auto localizedTitle = TTelematicsCommandTag::GetUserErrorTitle(code, locale, *permissions, *server);
                    g.AddReportElement("telematic_message_code", ToString(code));
                    g.MutableErrors().SetLocalizedMessage(localizedMessage);
                    g.MutableErrors().SetLocalizedTitle(localizedTitle);
                }
                if (response.Message) {
                    g.AddReportElement("telematics_message", std::move(response.Message));
                }
                if (response.Sensor) {
                    g.AddReportElement("imei", response.Imei);
                    g.AddReportElement("id", response.Sensor.Id);
                    g.AddReportElement("subid", response.Sensor.SubId);
                    g.AddReportElement("value", response.Sensor.GetJsonValue());
                    g.AddReportElement("formatted_value", response.Sensor.FormatValue());
                }
                if (verbose) {
                    g.AddReportElement("telematic_task", MakeCopy(response.DebugInfo));
                }
                switch (response.Status) {
                case NDrive::TTelematicsClient::EStatus::Success:
                    break;
                case NDrive::TTelematicsClient::EStatus::Timeouted:
                    g.SetCode(server->GetHttpStatusManagerConfig().TimeoutStatus);
                    break;
                case NDrive::TTelematicsClient::EStatus::Incorrect:
                    g.SetCode(server->GetHttpStatusManagerConfig().UserErrorState);
                    break;
                default:
                    g.SetCode(server->GetHttpStatusManagerConfig().UnknownErrorStatus);
                    break;
                }
            }
        };
    }

    NJson::TJsonValue GetPhotoScreenReport(const NDrive::IServer& server, const TUserCurrentContext& context, const THistoryRideObject& ride) {
        if (context.GetSessionPhotoScreenActions().empty()) {
            return {};
        }
        auto&& compiledRide = ride.GetFullCompiledRiding();
        if (!compiledRide) {
            return {};
        }
        bool photoRequired = false;
        i32 photoCount = 0;
        for (auto&& action : context.GetSessionPhotoScreenActions()) {
            auto screenAction = action.GetAs<NDrive::TSessionPhotoScreenAction>();
            if (!screenAction) {
                continue;
            }
            if (!screenAction->CheckSession(server, context.GetPermissions(), *compiledRide, ride.GetLastLocation())) {
                continue;
            }
            photoRequired = photoRequired || screenAction->GetPhotoRequired();
            photoCount = std::max(photoCount, screenAction->GetPhotoCount());
        }
        NJson::TJsonValue report;
        report.InsertValue("required", photoRequired);
        report.InsertValue("count", photoCount);
        return report;
    }

    void AddTrackEndPointsToReport(NJson::TJsonValue& trackReport, const NDrive::TTracksLinker::TResults& trackValue, const THistoryRideObject& session) {
        auto startCoordinate = [&]() -> const NGraph::TTimedGeoCoordinate * {
            for (const auto& trackResult : trackValue) {
                for (const auto& segment: trackResult.Segments) {
                    if (!segment.Coordinates.empty()) {
                        return &segment.Coordinates.front();
                    }
                }
            }
            return nullptr;
        }();

        if (startCoordinate) {
            trackReport.InsertValue("track_start", startCoordinate->SerializeToJson());

            auto lastCoordinate = [&]() -> const NGraph::TTimedGeoCoordinate * {
                for (const auto& trackResult : trackValue | std::views::reverse) {
                    for (const auto& segment: trackResult.Segments | std::views::reverse) {
                        if (!segment.Coordinates.empty()) {
                            return &segment.Coordinates.back();
                        }
                    }
                }
                return nullptr;
            }();
            if (lastCoordinate) {
                auto value = lastCoordinate->SerializeToJson();
                if (session.IsActive()) {
                    trackReport.InsertValue("track_current", std::move(value));
                } else {
                    trackReport.InsertValue("track_finish", std::move(value));
                }
            }
        }
    }
}  //namespace

void TCarControlUserProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    const TCgiParameters& cgi = Context->GetCgiParameters();

    TString action = GetString(requestData, "action");

    auto userSession = GetCurrentUserSession(permissions, Context->GetRequestStartTime());
    TString carId = userSession ? userSession->GetObjectId() : GetString(requestData, "car_id", false);
    TVector<TDBTag> perfTags;
    {
        auto session = BuildTx<NSQL::ReadOnly>();
        if (!Server->GetDriveAPI()->GetTagsManager().GetDeviceTags().RestorePerformerTags({ permissions->GetUserId() }, perfTags, session)) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
        R_ENSURE(session.Commit(), {}, "cannot Commit", session);

        if (!carId) {
            for (auto&& i : perfTags) {
                if (dynamic_cast<const TChargableTag*>(i.GetData().Get())) {
                    R_ENSURE(!carId, ConfigHttpStatus.UserErrorState, "incorrect_cars_selected", NDrive::MakeError("incorrect_cars_for_user"), session);
                    carId = i.GetObjectId();
                }
            }
            if (!carId) {
                for (auto&& i : perfTags) {
                    R_ENSURE(!carId, ConfigHttpStatus.UserErrorState, "incorrect_cars_selected", NDrive::MakeError("incorrect_cars_for_user"), session);
                    carId = i.GetObjectId();
                }
            }
        }
        R_ENSURE(carId, ConfigHttpStatus.UserErrorState, "cannot_detect_car", NDrive::MakeError("incorrect_cars_for_user"), session);

        bool found = false;
        for (auto&& tag : perfTags) {
            if (tag.GetObjectId() != carId) {
                continue;
            }
            auto description = DriveApi->GetTagsManager().GetTagsMeta().GetDescriptionByName(tag->GetName());
            if (!description) {
                continue;
            }
            if (description->GetAvailableCarActions().contains(action)) {
                found = true;
                break;
            }
        }
        R_ENSURE(found, ConfigHttpStatus.PermissionDeniedStatus, "no permissions to execute " << action, EDriveSessionResult::NoUserPermissions);
    }

    auto command = TTelematicsCommandTag::ParseCommand(requestData, carId, *Server);
    R_ENSURE(
        command,
        ConfigHttpStatus.UserErrorState,
        "cannot parse command from " << action << ": " << command.GetError().what(),
        EDriveSessionResult::IncorrectRequest
    );
    auto locale = GetLocale();

    auto timeout = GetDuration(cgi, "timeout", Config.GetDefaultTimeout());

    auto session = BuildTx<NSQL::Writable>();
    auto response = TTelematicsCommandTag::Command(carId, *command, timeout * Config.GetTaskTimeoutFraction(), *permissions, Server, session);
    R_ENSURE(response.Initialized(), {}, "cannot Command", session);
    R_ENSURE(session.Commit(), {}, "cannot Commit", session);
    response.Subscribe(MakeCommandCallback(g.GetReport(), permissions, Server, locale));
    g.Release();
}

void TCarControlRootProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    const TCgiParameters& cgi = Context->GetCgiParameters();
    TString carId;
    if (requestData.IsDefined()) {
        carId = GetUUID(requestData, "car_id");
    } else {
        carId = GetUUID(cgi, "car_id");
    }

    NDrive::NVega::TCommand command;
    if (requestData.IsDefined()) {
        auto cmd = NDrive::ParseCommand(requestData);
        R_ENSURE(
            cmd,
            ConfigHttpStatus.UserErrorState,
            "cannot parse telematics command from " << requestData.GetStringRobust() << ": " << cmd.GetError().what()
        );
        command = *cmd;
    } else {
        command.Code = GetValue<NDrive::NVega::ECommandCode>(cgi, "command").GetRef();
    }
    auto locale = GetLocale();
    auto timeout = GetDuration(cgi, "timeout", Config.GetDefaultTimeout());
    auto verbose = GetValue<bool>(cgi, "verbose", false).GetOrElse(false);

    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Control, TAdministrativeAction::EEntity::Car, ToString(command.Code));

    auto session = BuildTx<NSQL::Writable>();
    auto response = TTelematicsCommandTag::Command(carId, command, timeout * Config.GetTaskTimeoutFraction(), *permissions, Server, session);
    R_ENSURE(response.Initialized(), {}, "cannot Command", session);
    R_ENSURE(session.Commit(), {}, "cannot Commit", session);
    response.Subscribe(MakeCommandCallback(g.GetReport(), permissions, Server, locale, verbose));
    g.Release();
}

void TTagEvolveProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr originalPermissions, const NJson::TJsonValue& requestData) {
    auto dryRun = GetValue<bool>(requestData, "dry_run", false).GetOrElse(false);

    const TCgiParameters& cgi = Context->GetCgiParameters();
    const TString& userId = originalPermissions->GetUserId();
    const auto locale = GetLocale();

    ITag::TPtr newTag;
    {
        TMessagesCollector errors;
        newTag = IJsonSerializableTag::BuildFromJson(Server->GetDriveAPI()->GetTagsManager(), requestData, &errors);
        R_ENSURE(newTag, ConfigHttpStatus.SyntaxErrorStatus, errors.GetReport());
    }
    auto session = BuildTx<NSQL::Writable>();
    TString targetTagName = newTag->GetName();

    auto tagId = Coalesce(GetString(cgi, "tag_id", false), GetString(requestData, "tag_id", false));
    auto sessionId = Coalesce(GetString(cgi, "session_id", false), GetString(requestData, "session_id", false));
    auto objectId = GetString(requestData, "object_id", false);

    auto restoreByObjectId = [&](const TString& objectId, const IEntityTagsManager* manager) {
        Y_ENSURE(manager);
        auto optionalObject = manager->RestoreObject(objectId, session);
        R_ENSURE(optionalObject, {}, "cannot RestoreObject " << objectId, session);

        TDBTags result;
        for (auto&& tag : optionalObject->GetTags()) {
            auto evolution = originalPermissions ? originalPermissions->GetEvolutionPtr(tag->GetName(), targetTagName) : nullptr;
            if (evolution) {
                result.push_back(std::move(tag));
            }
        }
        return result;
    };
    auto restoreByTagId = [&](const TString& tagId, const IEntityTagsManager* manager) {
        Y_ENSURE(manager);
        auto optionalTags = manager->RestoreTags(tagId, session);
        R_ENSURE(optionalTags, {}, "cannot RestoreTag " << tagId, session);
        return std::move(*optionalTags);
    };

    TVector<TDBTag> tags;
    const IEntityTagsManager* tagsManager = &Server->GetDriveAPI()->GetTagsManager().GetDeviceTags();
    if (tags.empty() && tagId) {
        auto eg = g.BuildEventGuard("DeviceTags::RestoreTags");
        tags = restoreByTagId(tagId, tagsManager);
    }
    if (tags.empty() && sessionId) {
        auto eg = g.BuildEventGuard("DeviceTags::RestoreBySession");
        auto optionalSession = DriveApi->GetSessionManager().GetSession(sessionId, session);
        R_ENSURE(optionalSession, {}, "cannot GetSession", session);
        auto billingSession = *optionalSession;
        if (billingSession) {
            R_ENSURE(billingSession->Compile(), HTTP_INTERNAL_SERVER_ERROR, "cannot Compile", session);
            auto ev = billingSession->GetLastEvent();
            if (ev) {
                R_ENSURE(tagsManager->RestoreTags({ ev->GetTagId() }, tags, session), {}, "cannot RestoreTags", session);
            }
        }
    }
    if (tags.empty() && originalPermissions) {
        auto eg = g.BuildEventGuard("DeviceTags::RestoreEvolutionTagsByUser");
        R_ENSURE(tagsManager->RestoreEvolutionTagsByUser(*originalPermissions, targetTagName, tags, session), {}, "cannot RestoreEvolutionTagsByUser", session);
    }
    if (tags.empty() && objectId) {
        auto eg = g.BuildEventGuard("DeviceTags::RestoreByObjectId");
        tags = restoreByObjectId(objectId, tagsManager);
    }
    if (tags.empty() && tagId) {
        auto eg = g.BuildEventGuard("UserTags::RestoreEvolutionTags");
        tagsManager = &Server->GetDriveAPI()->GetTagsManager().GetUserTags();
        tags = restoreByTagId(tagId, tagsManager);
    }
    if (tags.empty() && objectId) {
        auto eg = g.BuildEventGuard("UserTags::RestoreByObjectId");
        tagsManager = &Server->GetDriveAPI()->GetTagsManager().GetUserTags();
        tags = restoreByObjectId(objectId, tagsManager);
    }
    R_ENSURE(tags.size() < 2, ConfigHttpStatus.UserErrorState, "Incorrect tags for this user " << userId, NDrive::MakeError("incorrect_cars_for_user"), session);
    R_ENSURE(tags.size() != 0, ConfigHttpStatus.UserErrorState, "No tags for this user " << userId, NDrive::MakeError("incorrect_evolution"), session);
    R_ENSURE(tags.front()->GetName() != TTransformationTag::TypeName, ConfigHttpStatus.UserErrorState, "car is in transformation", NDrive::MakeError("car_in_transformation"), session);

    TDBTag tagCurrent = tags.front();

    TUserPermissions::TPtr permissions;
    if (tagCurrent->GetPerformer() != userId && !!tagCurrent->GetPerformer()) {
        bool checkPermissions = false;
        if (sessionId) {
            auto optionalSession = DriveApi->GetSessionManager().GetSession(sessionId, session);
            R_ENSURE(optionalSession, {}, "cannot GetSession", session);
            auto billingSession = *optionalSession;
            R_ENSURE(billingSession, HTTP_NOT_FOUND, "cannot find session " << sessionId, session);
            R_ENSURE(!billingSession->GetClosed(), HTTP_BAD_REQUEST, "session " << sessionId << " is already closed", session);
            auto billingSessionImpl = dynamic_cast<const TBillingSession*>(billingSession.Get());
            R_ENSURE(billingSessionImpl, HTTP_NOT_FOUND, "cannot find session " << sessionId, session);

            auto offer = Yensured(billingSessionImpl->GetCurrentOffer());
            checkPermissions = !!CheckWalletsAccess(originalPermissions, TAdministrativeAction::EAction::Delegation, { offer->GetSelectedCharge() });
        }
        if (!checkPermissions) {
            ReqCheckAdmActions(originalPermissions, TAdministrativeAction::EAction::Delegation, TAdministrativeAction::EEntity::Tag);
        }
        permissions = DriveApi->GetUserPermissions(tagCurrent->GetPerformer(), TUserPermissionsFeatures());
        if (!GetOriginatorId()) {
            SetOriginatorId(userId);
        }
    } else {
        permissions = originalPermissions;
    }
    auto timeout = GetDuration(cgi, "timeout", Config.GetDefaultTimeout());
    auto taskTimeout = timeout * Config.GetTaskTimeoutFraction();
    auto rollbackTimeout = std::max(timeout * Config.GetKfTimeoutRollbackByRobot() * Config.GetTaskTimeoutFraction(), TDuration::Minutes(1));
    const EEvolutionMode eMode = GetValue<EEvolutionMode>(cgi, "evolution_mode", false).GetOrElse(EEvolutionMode::Default);
    ProcessEvolveTag(
        Self(),
        g,
        session,
        tagCurrent,
        newTag,
        permissions,
        locale,
        dryRun,
        requestData,
        taskTimeout,
        rollbackTimeout,
        eMode,
        tagsManager,
        ConfigHttpStatus,
        Server
    );
}

void ProcessEvolveTag(
    IRequestProcessor::TPtr self,
    TJsonReport::TGuard& g,
    NDrive::TEntitySession& session,
    TDBTag tagCurrent,
    ITag::TPtr newTag,
    TUserPermissions::TPtr permissions,
    ELocalization locale,
    bool dryRun,
    const NJson::TJsonValue& requestData,
    TDuration taskTimeout,
    TDuration rollbackTimeout,
    EEvolutionMode eMode,
    const IEntityTagsManager* tagsManager,
    const THttpStatusManagerConfig& ConfigHttpStatus,
    const NDrive::IServer* Server
) {
    TString objectId = tagCurrent.GetObjectId();

    auto applicablePolicy = tagCurrent->BuildCustomEvolutionPolicy(tagCurrent, newTag, Server, *permissions, requestData, eMode, session);
    if (!applicablePolicy) {
        if (applicablePolicy.HasError()) {
            session.SetErrorInfo("BuildCustomEvolutionPolicy", applicablePolicy.GetErrorMessage(), MakeCopy(applicablePolicy.GetErrorRef()));
        } else if (applicablePolicy.HasResult()) {
            session.SetErrorInfo("BuildCustomEvolutionPolicy", applicablePolicy.GetErrorMessage(), applicablePolicy.GetResultRef());
        } else {
            session.SetErrorInfo("BuildCustomEvolutionPolicy", applicablePolicy.GetErrorMessage());
        }
        session.SetCode(applicablePolicy.OptionalErrorCode());
        session.SetLocalizedMessageKey(applicablePolicy.GetUIErrorMessage());
        session.SetLocalizedTitleKey(applicablePolicy.GetUIErrorTitle());
        session.DoExceptionOnFail(ConfigHttpStatus, Server->GetLocalization(), TStringBuf("bad evolution policy"));
    }

    TString originalTagName = newTag->GetName();
    if (!!applicablePolicy.NewTag) {
        newTag = applicablePolicy.NewTag;
    }
    TString targetTagName = newTag->GetName();

    NDrive::ITag::TEvolutionContext eContext;
    eContext.SetPolicies(applicablePolicy.Policies);
    eContext.SetMode(eMode);

    g.AddEvent(NJson::TMapBuilder
        ("event", "GetEvolution")
        ("from", tagCurrent->GetName())
        ("to", targetTagName)
    );

    auto evolution = permissions->GetEvolutionPtr(tagCurrent->GetName(), targetTagName);
    if (!evolution) {
        TString localizedMessage =
            Server->GetLocalization()->GetLocalString(locale, TStringBuilder() << "error.incorrect_evolution.from." << tagCurrent->GetName() << ".to." << targetTagName, "");
        if (localizedMessage) {
            session.SetLocalizedMessageKey(localizedMessage);
        }

        session.SetCode(HTTP_FORBIDDEN);
        session.SetErrorInfo(
            "TTagEvolveProcessor",
            "assertion evolution failed",
            NDrive::MakeError("incorrect_evolution_2"),
            EDriveSessionResult::IncorrectRequest);
        session.DoExceptionOnFail(ConfigHttpStatus, Server->GetLocalization(), TStringBuilder() << "no permissions to evolve " << tagCurrent->GetName() << " into " << targetTagName);
    }

    R_ENSURE(
        !evolution->GetOnlyPerformed() || tagCurrent->GetPerformer(),
        HTTP_FORBIDDEN,
        "no permissions to evolve " << tagCurrent->GetName() << " without performer",
        NDrive::MakeError("incorrect_evolution_without_performer"),
        session
    );

    if (evolution && !evolution->CheckEvolution(*tagsManager, tagCurrent, session)) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }
    auto optionalCommand = TMaybe<NDrive::NVega::TCommand>();
    if (evolution && evolution->HasDeviceOperation(eMode) && Server->GetSettings().GetValueDef("emergency.evolution.telematic_usage", true)) {
        optionalCommand = evolution->GetDeviceOperation(eMode);
    }
    if (applicablePolicy.Command) {
        g.AddEvent(NJson::TMapBuilder
            ("event", "OverrideCommand")
            ("before", NJson::ToJson(optionalCommand))
            ("after", NJson::ToJson(applicablePolicy.Command))
        );
        optionalCommand = *applicablePolicy.Command;
    }

    auto externalCommandResult = TMaybe<NDrive::TExternalCommandResult>();
    R_ENSURE(NJson::TryFromJson(requestData["external_command_result"], externalCommandResult), ConfigHttpStatus.SyntaxErrorStatus, "cannot parse external_command_result");
    if (externalCommandResult && optionalCommand) {
        auto validationResult = TTelematicsCommandTag::ValidateExternalCommand(objectId, *optionalCommand, *externalCommandResult, *permissions, *Server);
        if (validationResult && *validationResult) {
            NDrive::TEventLog::Log("SkipTelematicsCommand", NJson::TMapBuilder
                ("object_id", objectId)
                ("command", NJson::ToJson(optionalCommand))
                ("external_command_result", NJson::ToJson(externalCommandResult))
            );
            g.AddEvent("SkipTelematicsCommand");
            optionalCommand.Clear();

            auto chargableTag = std::dynamic_pointer_cast<TChargableTag>(newTag);
            if (chargableTag) {
                chargableTag->SetTransformationSkippedByExternalCommand(true);
            }
        } else {
            NJson::TJsonValue ev = NJson::TMapBuilder
                ("object_id", objectId)
                ("command", NJson::ToJson(optionalCommand))
                ("external_command_result", NJson::ToJson(externalCommandResult))
                ("validation_result", ToString(validationResult))
            ;
            NDrive::TEventLog::Log("SkipTelematicsCommandFailure", ev);
            g.AddEvent(NJson::TMapBuilder
                ("event", "SkipTelematicsCommandFailure")
                ("data", std::move(ev))
            );
        }
    }

    auto snapshot = NDrive::IObjectSnapshot::ConstructFromJson(requestData["snapshot"]);
    if (snapshot) {
        auto object = Server->GetDriveDatabase().GetCarManager().GetObject(objectId);
        R_ENSURE(!object || object->GetIMEI().empty(), HTTP_FORBIDDEN, "cannot set snapshot for " << objectId, session);
        if (object && optionalCommand) {
            NDrive::TEventLog::Log("SkipTelematicsCommand", NJson::TMapBuilder
                ("object_id", objectId)
                ("command", NJson::ToJson(optionalCommand))
                ("snapshot", snapshot->SerializeToJson())
            );
            g.AddEvent("SkipTelematicsCommandBySnapshot");
            optionalCommand.Clear();
        }
    }
    newTag->SetObjectSnapshot(std::move(snapshot));

    TDBTag oldTag = tagCurrent;
    if (optionalCommand) {
        const auto& command = *optionalCommand;
        auto commandId = TTelematicsCommandTag::GenerateId(objectId, command, g.GetReqId(), *Server);
        auto transformation = MakeAtomicShared<TTransformationTag>(oldTag->GetName(), targetTagName);
        transformation->SetCommandId(commandId);
        transformation->SetRollbackTimestamp(Now() + rollbackTimeout);
        tagCurrent.SetData(transformation);
        R_ENSURE(transformation->CopyOnEvolve(*oldTag, evolution, *Server), ConfigHttpStatus.UserErrorState, EDriveLocalizationCodes::CannotCopyOnEvolution);
        R_ENSURE(newTag->CopyOnEvolve(*oldTag, evolution, *Server), ConfigHttpStatus.UserErrorState, EDriveLocalizationCodes::CannotCopyOnEvolution);
        if (!tagsManager->EvolveTag(oldTag, transformation, *permissions, Server, session, &eContext)) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
        if (dryRun) {
            g.AddEvent(NJson::TMapBuilder
                ("event", "DryRun")
                ("branch", "optionalCommandPresent")
            );
            R_ENSURE(session.Rollback(), {}, "cannot Rollback", session);
            g.MutableReport().SetExternalReport(NJson::JSON_MAP);
            g.SetCode(HTTP_OK);
            return;
        }
        if (!session.Commit()) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }

        auto options = TTelematicsCommandTag::TRuntimeOptions(taskTimeout);
        options.EvolveTag = false;
        options.ExternalCommandId = commandId;
        auto callback = [
            currentTag = tagCurrent,
            locale,
            newTag,
            oldTag = oldTag.GetData(),
            evolutionMode = eMode,
            permissions,
            server = Server,
            eContext
        ] (TJsonReport::TGuard& g, const NDrive::TCommonCommandResponse& response) {
            const auto& statuses = server->GetHttpStatusManagerConfig();
            NDrive::TEventLog::TUserIdGuard userIdGuard(permissions->GetUserId());
            g.AddReportElement("telematic_status", ToString(response.Status));
            if (response.Message) {
                g.AddReportElement("telematic_message", response.Message);
            }

            bool success =
                response.Status == NDrive::TTelematicsClient::EStatus::Success ||
                evolutionMode == EEvolutionMode::IgnoreTelematic;
            auto attempts = 2;
            auto result = success
                ? TTransformationTag::Complete(currentTag, newTag, *permissions, server, attempts, &eContext)
                : TTransformationTag::Rollback(currentTag, oldTag, *permissions, server, attempts, &eContext);
            result.GetValue();
            if (!success) {
                auto errorCode = NDrive::ParseError(response.Message);
                auto localizedMessage = TTelematicsCommandTag::GetUserErrorDescription(errorCode, locale, *permissions, *server);
                auto localizedTitle = TTelematicsCommandTag::GetUserErrorTitle(errorCode, locale, *permissions, *server);
                g.AddReportElement("telematic_message_code", ToString(errorCode));
                g.AddReportElement("telematic_message_localized", localizedMessage);

                auto code = statuses.UserErrorState;
                if (response.Status == NDrive::TTelematicsClient::EStatus::Timeouted) {
                    code = statuses.TimeoutStatus;
                }
                if (response.Status == NDrive::TTelematicsClient::EStatus::InternalError) {
                    code = statuses.UnknownErrorStatus;
                }
                throw TCodedException(code)
                    .SetLocalizedMessage(localizedMessage)
                    .SetLocalizedTitle(localizedTitle)
                    .SetErrorCode(ToString(response.Status))
                << "rolled back";
            }
            g.SetCode(HTTP_OK);
        };
        auto tx = tagsManager->BuildTx<NSQL::Writable | NSQL::Deferred>();
        auto response = TTelematicsCommandTag::Command(objectId, command, options, *permissions, Server, tx);
        R_ENSURE(response.Initialized(), {}, "cannot Command", tx);
        R_ENSURE(tx.Commit(), {}, "cannot Commit", tx);

        response.Subscribe([
            callback = std::move(callback),
            report = g.GetReport(),
            processor = self
        ] (const NThreading::TFuture<NDrive::TCommonCommandResponse>& response) {
            processor->Process(report, [&](TJsonReport::TGuard& g) {
                callback(g, response.GetValue());
            });
        });
        g.Release();
    } else {
        R_ENSURE(newTag->CopyOnEvolve(*oldTag, evolution, *Server), ConfigHttpStatus.UserErrorState, EDriveLocalizationCodes::CannotCopyOnEvolution);
        auto evolved = tagsManager->EvolveTag(oldTag, newTag, *permissions, Server, session, &eContext);
        R_ENSURE(evolved, {}, "cannot EvolveTag", session);
        bool post = evolution->PostEvolution(*tagsManager, *evolved, *permissions, Server, session);
        R_ENSURE(post, {}, "cannot PostEvolution", session);
        if (dryRun) {
            g.AddEvent(NJson::TMapBuilder
                ("event", "DryRun")
                ("branch", "optionalCommandEmpty")
            );
            R_ENSURE(session.Rollback(), {}, "cannot Rollback", session);
            g.MutableReport().SetExternalReport(NJson::JSON_MAP);
            g.SetCode(HTTP_OK);
            return;
        }
        R_ENSURE(session.Commit(), {}, "cannot Commit", session);
        g.MutableReport().SetExternalReport(NJson::JSON_MAP);
        g.SetCode(HTTP_OK);
    }
}

void TSessionAgreementProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    const auto& cgi = Context->GetCgiParameters();
    const auto& sessionId = GetString(cgi, "session_id", false);

    auto compiledSession = [&]() -> THolder<TFullCompiledRiding> {
        auto eg = g.BuildEventGuard("get_current_session");
        ISession::TConstPtr session;
        R_ENSURE(DriveApi->GetUserSession(permissions->GetUserId(), session, sessionId, Now()), HTTP_INTERNAL_SERVER_ERROR, "cannot GetRequestUserSession");
        if (!session) {
            return nullptr;
        }
        if (session->GetUserId() != permissions->GetUserId()) {
            ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::User);
        }
        auto billingSession = std::dynamic_pointer_cast<const TBillingSession>(session);
        R_ENSURE(billingSession, HTTP_INTERNAL_SERVER_ERROR, "cannot cast session " << session->GetSessionId() << " to BillingSession");
        return billingSession->BuildCompiledRiding(Server, nullptr);
    }();
    if (!compiledSession && sessionId) {
        auto eg = g.BuildEventGuard("get_compiled_session");
        auto tx = BuildTx<NSQL::ReadOnly>();
        auto ydbTx = BuildYdbTx<NSQL::ReadOnly>("session_agreement");
        auto optionalCompiledSession = Server->GetDriveDatabase().GetCompiledSessionManager().Get<TFullCompiledRiding>(NContainer::Scalar(sessionId), tx, ydbTx);
        R_ENSURE(optionalCompiledSession, {}, "cannot GetCompiledSession " << sessionId, tx);
        for (auto&& session : *optionalCompiledSession) {
            if (session.GetSessionId() == sessionId) {
                compiledSession = MakeHolder<TFullCompiledRiding>(std::move(session));
                break;
            }
        }
    }

    R_ENSURE(compiledSession, HTTP_INTERNAL_SERVER_ERROR, "cannot acquire CompiledSession " << sessionId);
    auto offer = compiledSession->GetOffer();
    R_ENSURE(offer, HTTP_INTERNAL_SERVER_ERROR, "cannot find Offer in session " << compiledSession->GetSessionId());
    auto standardOffer = std::dynamic_pointer_cast<TStandartOffer>(offer);
    R_ENSURE(standardOffer, HTTP_INTERNAL_SERVER_ERROR, "cannot cast offer " << offer->GetOfferId() << " to Standard");

    offer->BuildAgreement(GetLocale(), std::move(compiledSession), permissions, Server).Subscribe([report = g.GetReport()] (const NThreading::TFuture<TString>& future) {
            TJsonReport::TGuard g(report, HTTP_OK);
            TJsonReport& r = g.MutableReport();
            if (!future.HasValue() || future.HasException()) {
                r.AddReportElement("error", NThreading::GetExceptionInfo(future));
                g.SetCode(HttpCodes::HTTP_INTERNAL_SERVER_ERROR);
            } else {
                r.AddReportElement("agreement", std::move(future.GetValue()));
                g.SetCode(HTTP_OK);
            }
        });

    g.Release();
}

void TSessionsHistoryProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    TTimeGuard tg("build_history");
    const TCgiParameters& cgi = Context->GetCgiParameters();
    TString userId = permissions->GetUserId();
    TString carId;
    TString sessionId;
    {
        ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::Car);
        if (cgi.Has("session_id")) {
            sessionId = GetString(cgi, "session_id");
        } else if (cgi.Has("user_id")) {
            userId = GetUUID(cgi, "user_id");
        } else if (cgi.Has("car_id")) {
            carId = GetUUID(cgi, "car_id");
        }
    }

    const ui32 numdoc = sessionId ? Max<ui32>() - 100 : GetValue<ui32>(cgi, "numdoc", false).GetOrElse(Max<ui32>() - 1);
    const TInstant since = GetTimestamp(cgi, "since", GetTimestamp(cgi, "ts_since", TInstant::Zero()));
    const TInstant until = GetTimestamp(cgi, "until", GetTimestamp(cgi, "ts_supreme", TInstant::Max()));
    const TString type = GetString(cgi, "type", false);
    const auto locale = GetLocale();
    bool needGeocodedLocation = GetValue<bool>(cgi, "geocoded", false).GetOrElse(false);

    THistoryRidesContext context(*Server, since);
    auto session = BuildTx<NSQL::ReadOnly>();
    auto ydbTx = BuildYdbTx<NSQL::ReadOnly | NSQL::Deferred>("sessions_history");

    ReqCheckCondition(context.Initialize(sessionId, userId, carId, session, ydbTx, until, numdoc), ConfigHttpStatus.UnknownErrorStatus, "history_iterator.initialization.error", session.GetStringReport());
    bool hasMore = false;
    TVector<THistoryRideObject> sessions = context.GetSessions(until, numdoc, &hasMore);
    THistoryRideObject::FetchFullRiding(Server, sessions, static_cast<bool>(ydbTx));

    TSet<TString> cars;
    TSet<TString> models;
    TSet<TString> sessionIds;
    for (auto&& i : sessions) {
        cars.emplace(i.GetObjectId());
        models.emplace(i.GetObjectModel());
        sessionIds.emplace(i.GetSessionId());
    }

    TTaggedObjectsSnapshot taggedSessions;
    {
        if (!NDrive::GetFullEntityTags(DriveApi->GetTagsManager().GetTraceTags(), sessionIds, taggedSessions, session, ydbTx)) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
    }

    TCarsFetcher fetcher(*Server, NDeviceReport::ReportAdminHistory & permissions->GetDeviceReportTraits());
    fetcher.SetIsRealtime(false);
    fetcher.SetLocale(locale);
    fetcher.SetSelectedModels(models);
    {
        TTimeGuard tgFetch("Fetcher");
        R_ENSURE(fetcher.FetchData(permissions, cars), HTTP_INTERNAL_SERVER_ERROR, "unknown failure", session);
    }

    NJson::TJsonValue result(NJson::JSON_ARRAY);
    auto customization = MakeAtomicShared<TBillingReportCustomization>();
    customization->SetNeedBill(true);
    customization->SetIsSimpleReport(false);

    TSet<TString> users;
    TSet<TString> actionBuilderNames;
    TUsersDB::TFetchResult usersFetchData;
    for (auto&& i : sessions) {
        if (auto offer = i.GetOffer()) {
            actionBuilderNames.emplace(offer->GetBehaviourConstructorId());
        }
        if (users.contains(i.GetUserId())) {
            continue;
        }
        if (CheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::User, i.GetUserId())) {
            users.emplace(i.GetUserId());
        }
    }
    usersFetchData = DriveApi->GetUsersData()->FetchInfo(users, session);

    auto actionBuilders = DriveApi->GetRolesManager()->GetActions(actionBuilderNames);

    if (needGeocodedLocation) {
        PrefetchGeocodedLocations(sessions, g);
    }

    for (auto&& i : sessions) {
        const TString& objectId = i.GetObjectId();
        const TString& currentSessionId = i.GetSessionId();
        if (!users.contains(i.GetUserId())) {
            continue;
        }
        ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::User, userId);

        NJson::TJsonValue& sessionReport = result.AppendValue(NJson::JSON_MAP);

        NJson::TJsonValue carReport;
        if (objectId && fetcher.BuildOneCarInfoReport(objectId, permissions->GetFilterActions(), carReport)) {
            // no-op
        } else if (auto objectModel = i.GetObjectModel()) {
            carReport["model_id"] = objectModel;
        }
        sessionReport.InsertValue("car", std::move(carReport));

        sessionReport.InsertValue("segment", i.GetReport(locale, customization, type));
        const TDriveUserData* userData = usersFetchData.GetResultPtr(i.GetUserId());
        if (userData) {
            if (CheckUserGeneralAccessRights(i.GetUserId(), permissions, NAccessVerification::EAccessVerificationTraits::TagsAccess | NAccessVerification::EAccessVerificationTraits::ObserveAccess)) {
                sessionReport.InsertValue("user_details", userData->GetReport());
            } else {
                TDriveUserData emptyUser;
                sessionReport.InsertValue("user_details", emptyUser.GetReport());
            }
        }
        NJson::TJsonValue diffReport = i.GetDiffReport(locale, Server);
        if (!diffReport.IsNull()) {
            sessionReport.InsertValue("device_diff", diffReport);
        }
        sessionReport.InsertValue("offer_proto", i.GetOfferProtoReport());

        NJson::TJsonValue geoTags(NJson::JSON_ARRAY);
        auto location = i.GetLastLocation();
        if (location.Defined()) {
            geoTags = NJson::ToJson(GetFetchedGeoTags(location->GetCoord()));
        }
        sessionReport.InsertValue("geo_tags", std::move(geoTags));

        auto startLocation = i.GetStartLocation();
        if (startLocation) {
            sessionReport.InsertValue("start_geo_tags", NJson::ToJson(GetFetchedGeoTags(startLocation->GetCoord())));
        }

        if (needGeocodedLocation) {
            TString geocodedStart = GetGeocodedStart(i.GetSessionId(), g);
            TString geocodedFinish = GetGeocodedFinish(i.GetSessionId(), g);
            if (geocodedStart) {
                sessionReport.InsertValue("geocoded_start", geocodedStart);
            }
            if (geocodedFinish) {
                sessionReport.InsertValue("geocoded_finish", geocodedFinish);
            }
        }

        NJson::TJsonValue traceTags(NJson::JSON_ARRAY);
        const TTaggedObject* tObject = taggedSessions.Get(currentSessionId);
        if (tObject) {
            for (auto&& tag : tObject->GetTags()) {
                if (!tag) {
                    continue;
                }
                traceTags.AppendValue(tag->GetName());
            }
        }
        sessionReport.InsertValue("trace_tags", std::move(traceTags));

        auto optionalInsuranceNotifications = DriveApi->GetInsuranceHistoryManager().GetSession(currentSessionId, session);
        if (optionalInsuranceNotifications) {
            NJson::TJsonValue insuranceNotifications;
            for (auto&& notification : *optionalInsuranceNotifications) {
                NJson::TJsonValue& insuranceNotification = insuranceNotifications.AppendValue(NJson::JSON_MAP);
                NJson::InsertField(insuranceNotification, "id", notification.GetNotificationId());
                NJson::InsertField(insuranceNotification, "start", notification.GetStart().Seconds());
                NJson::InsertField(insuranceNotification, "finish", notification.GetFinish().Seconds());
                NJson::InsertField(insuranceNotification, "created", NJson::Nullable(notification.GetCreated().Seconds()));
                NJson::InsertField(insuranceNotification, "sent", NJson::Nullable(notification.GetSent().Seconds()));
            }
            sessionReport.InsertValue("insurance_notifications", std::move(insuranceNotifications));
        }
        if (auto offer = i.GetOffer()) {
            auto itAction = actionBuilders.find(offer->GetBehaviourConstructorId());
            if (itAction != actionBuilders.end()) {
                NJson::TJsonValue tags(NJson::JSON_ARRAY);
                for (auto&& tag : itAction->second->GetGrouppingTags()) {
                    tags.AppendValue(tag);
                }
                sessionReport.InsertValue("offer_groupping_tags", tags);
            }
        }
    }
    g.MutableReport().AddReportElement("has_more", hasMore);
    g.MutableReport().AddReportElement("sessions", std::move(result));
    fetcher.BuildReportMeta(g.MutableReport());

    g.MutableReport().AddReportElement("server_time", Now().Seconds());
    g.SetCode(HTTP_OK);
}

NDrive::TLocationTags TSessionsHistoryProcessor::GetFetchedGeoTags(const TGeoCoord& coordinate) const {
    NDrive::TLocationTags result;
    const auto actor = [&result, this](const TFullAreaInfo& area) {
        for (auto&& tag : area.GetArea().GetTags()) {
            if (Config.GetFetchedGeoTags().contains(tag)) {
                result.insert(tag);
            }
        }
        return true;
    };

    DriveApi->GetAreasDB()->ProcessTagsInPoint(coordinate, actor, Config.GetFetchedGeoTags());
    return result;
}

void TClientHistoryProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    const TCgiParameters& cgi = Context->GetCgiParameters();
    const TString& externalObjectId = GetString(cgi, "car_id", false);
    const TString& externalUserId = GetString(cgi, "user_id", false);
    const TString& sessionId = GetString(cgi, "session_id", false);
    const TString& report = GetString(cgi, "report", false);
    const auto locale = GetLocale();
    const auto traits = GetValues<NDriveSession::EReportTraits>(cgi, "traits", false);
    const bool includeDriverInfo = GetValue<bool>(cgi, "user_info", false).GetOrElse(false) ? true : GetValue<bool>(cgi, "users_info", false).GetOrElse(false);
    const bool includeDriverScore = GetValue<bool>(cgi, "user_score", false).GetOrElse(false);
    const bool skipEmpty = GetValue<bool>(cgi, "skip_empty", false).GetOrElse(false);
    const bool reportAggressionScore = GetHandlerSetting<bool>("report_aggression_score").GetOrElse(false);
    const bool showVisibleSessions = GetValue<bool>(cgi, "show_visible_sessions", false).GetOrElse(false);

    NDriveSession::TReportTraits reportTraits = NDriveSession::GetReportTraits(report);
    for (auto&& trait : traits) {
        reportTraits |= trait;
    }
    if (!GetHandlerSetting<bool>("fetch_payment_methods").GetOrElse(true)) {
        reportTraits &= (~NDriveSession::EReportTraits::ReportPaymentMethods);
    }
    reportTraits |= NDriveSession::EReportTraits::ReportFavouriteAddressSuggest;  // favourite addr advisor controls itself whether it's enabled or not

    const auto needBill = GetValue<bool>(cgi, "need_bill", false).GetOrElse(!sessionId.empty());
    const ui32 numdoc = GetValue<ui32>(cgi, "numdoc", false).GetOrElse(Max<ui32>());
    const TInstant until = GetTimestamp(cgi, "until", GetTimestamp(cgi, "ts_supreme", TInstant::Max()));
    const TInstant since = GetTimestamp(cgi, "since", GetTimestamp(cgi, "ts_since", TInstant::Zero()));
    const TInstant deadline = Context->GetRequestDeadline();
    const TInstant requestTimestamp = Context->GetRequestStartTime();
    bool needGeocodedLocation = GetValue<bool>(cgi, "geocoded", false).GetOrElse(false);

    auto checkObjectVisibility = GetHandlerSetting<bool>("session.check_object_visibility").GetOrElse(false);
    auto maxHistoryDepth = GetHandlerSetting<TDuration>("session.history.max_depth").GetOrElse(TDuration::Days(90));
    auto historySince = std::max(since, requestTimestamp - maxHistoryDepth);

    TVector<TString> externalObjectsIds;
    if (showVisibleSessions) {
        TVector<TTaggedObject> taggedObjects;
        R_ENSURE(Server->GetDriveDatabase().GetTagsManager().GetDeviceTags().GetObjectsFromCacheByIds({}, taggedObjects, TInstant::Zero()), HTTP_INTERNAL_SERVER_ERROR, "can't fetch objects from cache");
        TVector<TString> visibleObjectsIds;
        visibleObjectsIds.reserve(taggedObjects.size());
        for (const auto& taggedObject : taggedObjects) {
            if (permissions->GetVisibility(taggedObject, NEntityTagsManager::EEntityType::Car) == TUserPermissions::EVisibility::Visible) {
                visibleObjectsIds.push_back(taggedObject.GetId());
            }
        }
        externalObjectsIds = std::move(visibleObjectsIds);
    } else if (externalObjectId) {
        auto optionalObject = Server->GetDriveDatabase().GetTagsManager().GetDeviceTags().GetObject(externalObjectId);
        R_ENSURE(optionalObject, HTTP_NOT_FOUND, "cannot GetObject: " << externalObjectId);
        TUserPermissions::TUnvisibilityInfoSet invisibilityInfo = 0;
        auto visibility = permissions->GetVisibility(*optionalObject, NEntityTagsManager::EEntityType::Car, &invisibilityInfo);
        R_ENSURE(
            visibility == TUserPermissions::EVisibility::Visible,
            HTTP_FORBIDDEN,
            "invisible object " << externalObjectId << ": " << TUserPermissions::ExplainInvisibility(invisibilityInfo)
        );
        externalObjectsIds = { externalObjectId };
    }

    for (const auto& objectId : externalObjectsIds) {
        ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::Car, objectId);
    }

    if (externalUserId && externalUserId != permissions->GetUserId()) {
        ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::User, externalUserId);
    }
    const TString& userId = externalUserId ? externalUserId : permissions->GetUserId();

    auto session = BuildTx<NSQL::ReadOnly>();
    auto ydbTx = BuildYdbTx<NSQL::ReadOnly | NSQL::Deferred>("client_history");

    const TString filter = GetString(cgi, "filter", false);
    TSet<TString> filterNames = StringSplitter(filter).Split(',').SkipEmpty();
    bool hasUsedInScoring = filterNames.contains("used_in_scoring");
    if (hasUsedInScoring) {
        filterNames.erase("used_in_scoring");
    }
    THistoryRidesContext context(*Server, historySince, true, filterNames.contains("fined"));
    bool hasMore = false;
    TVector<THistoryRideObject> sessions;

    auto namesFilterFunction = filterNames ? ConstructFilter(filterNames) : nullptr;
    TFilter originFilterFunction = nullptr;
    const auto origin = GetOrigin();
    const bool needCheckOrigin = GetHandlerSetting<bool>("session.check_origin").GetOrElse(false);
    if (needCheckOrigin && origin) {
        originFilterFunction = [&origin](const THistoryRideObject& ride) {
            return !ride.GetOffer() || ride.GetOffer()->GetOrigin() == origin;
        };
    }
    TFilter filterFunction = nullptr;
    if (namesFilterFunction || originFilterFunction) {
        filterFunction = [&namesFilterFunction, &originFilterFunction](const THistoryRideObject& ride) {
            bool result = true;
            result &= !namesFilterFunction || namesFilterFunction(ride);
            result &= !originFilterFunction || originFilterFunction(ride);
            return result;
        };
    }

    if (filterFunction && !hasUsedInScoring) {
        ReqCheckCondition(
            context.Initialize(sessionId, userId, externalObjectsIds, session, ydbTx, until, Max<ui32>() - 100),
            ConfigHttpStatus.UnknownErrorStatus,
            "history_iterator.initialization.error",
            session.GetStringReport()
        );
        sessions = context.GetSessions(until, numdoc, &hasMore, skipEmpty, filterFunction);
    } else {
        auto correctedNumdoc = hasUsedInScoring ? Max<ui32>() - 100 : numdoc;
        ReqCheckCondition(
            context.Initialize(sessionId, userId, externalObjectsIds, session, ydbTx, until, correctedNumdoc),
            ConfigHttpStatus.UnknownErrorStatus,
            "history_iterator.initialization.error",
            session.GetStringReport()
        );
        sessions = context.GetSessions(until, correctedNumdoc, &hasMore, skipEmpty);
    }
    THistoryRideObject::FetchFullRiding(Server, sessions, static_cast<bool>(ydbTx));
    // "used_in_scoring" is special filter for only-scored sessions.
    // It can not be applied before full ridings are fetched.
    if (hasUsedInScoring) {
        auto distanceSumLimit = GetHandlerSetting<double>("distance_sum_limit").GetOrElse(200);
        double distanceSum = 0;
        auto distanceFilterFunction = [
            &distanceSum,
            distanceSumLimit,
            filterFunction
        ](const THistoryRideObject& ride) {
            if (distanceSum >= distanceSumLimit) {
                return true;
            }
            distanceSum += ride.GetDistance().GetOrElse(0);
            if (filterFunction && !filterFunction(ride)) {
                return true;
            }
            return false;
        };
        auto it = std::remove_if(sessions.begin(), sessions.end(), distanceFilterFunction);
        sessions.erase(it, sessions.end());
    }
    PrefetchAggressiveEvents(sessions, g);
    if (reportTraits & NDriveSession::ReportTrack) {
        PrefetchTracks(sessions, g, needGeocodedLocation);
    }
    TSet<TString> users;
    TSet<TString> cars;
    TSet<TString> models;
    for (auto&& i : sessions) {
        cars.emplace(i.GetObjectId());
        models.emplace(i.GetObjectModel());
        if (includeDriverInfo) {
            users.emplace(i.GetUserId());
        }
    }
    TUsersDB::TFetchResult usersFetchData;
    if (includeDriverInfo) {
        usersFetchData = DriveApi->GetUsersData()->FetchInfo(users, session);
    }
    TCarsFetcher fetcher(*Server, NDeviceReport::ReportHistory);
    fetcher.SetIsRealtime(false);
    fetcher.SetLocale(locale);
    fetcher.SetSelectedModels(models);
    if (!fetcher.FetchData(permissions, cars)) {
        g.AddEvent("FetchDataFailure");
    }
    if (needGeocodedLocation) {
        PrefetchGeocodedLocations(sessions, g);
    }

    NJson::TJsonValue result(NJson::JSON_ARRAY);
    TMultiBill mBill;
    ui32 totalCashback = 0;
    const bool isPlusUser = permissions->GetUserFeatures().GetIsPlusUser();

    TSet<TString> visibleCameraEvents;
    auto signalqSignalsDescriptions = NDrive::NSignalq::GetSignalDescriptions();
    for (const auto& signalqSignalDesc: signalqSignalsDescriptions) {
        visibleCameraEvents.insert(signalqSignalDesc.GetName());
    }

    for (auto&& i : sessions) {
        if (filterFunction && !filterFunction(i)) {
            g.AddEvent(NJson::TMapBuilder
                ("event", "SecondStageFilteredOut")
                ("session_id", i.GetSessionId())
            );
            continue;
        }

        const TString& objectId = i.GetObjectId();
        if (checkObjectVisibility || externalObjectId) {
            auto optionalObject = Server->GetDriveDatabase().GetTagsManager().GetDeviceTags().RestoreObject(objectId, session);
            R_ENSURE(optionalObject, {}, "cannot RestoreObject: " << objectId, session);
            TUserPermissions::TUnvisibilityInfoSet invisibilityInfo = 0;
            auto visibility = permissions->GetVisibility(*optionalObject, NEntityTagsManager::EEntityType::Car, &invisibilityInfo);
            R_ENSURE(
                visibility == TUserPermissions::EVisibility::Visible,
                HTTP_FORBIDDEN,
                "invisible object " << objectId << ": " << TUserPermissions::ExplainInvisibility(invisibilityInfo),
                session
            );
        }
        if (i.GetUserId() != userId) {
            ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::User, i.GetUserId());
        }

        auto fullCompiledRiding = i.GetFullCompiledRiding(static_cast<bool>(ydbTx));
        if (!fullCompiledRiding) {
            continue;
        }
        const auto& cr = *fullCompiledRiding;
        if (cr.HasBill()) {
            auto bill = cr.GetBillWithTariffRecord();
            if (!isPlusUser) {
                mBill.AddBill(TBill::GetBillForNonPlusUser(std::move(bill)));
            } else {
                mBill.AddBill(std::move(bill));
            }
            totalCashback += cr.GetBillUnsafe().GetCashbackSum();
        }
        NJson::TJsonValue& sessionReport = result.AppendValue(NJson::JSON_MAP);

        if (includeDriverInfo) {
            const TDriveUserData* userData = usersFetchData.GetResultPtr(i.GetUserId());
            if (userData) {
                sessionReport.InsertValue("user_details", userData->GetReport(NUserReport::EReportTraits::ReportNames | NUserReport::EReportTraits::ReportId));
            } else {
                TDriveUserData emptyUser;
                sessionReport.InsertValue("user_details", emptyUser.GetReport(NUserReport::EReportTraits::ReportNames));
            }
            if (includeDriverScore) {
                auto& userDetails = sessionReport["user_details"];
                auto optionalObject = DriveApi->GetTagsManager().GetUserTags().GetCachedOrRestoreObject(i.GetUserId(), session);
                auto scoringTagContainer = optionalObject->GetTagsByClass<TScoringUserTag>();
                if (scoringTagContainer && scoringTagContainer.size() == 1) {
                    userDetails.InsertValue("user_score", scoringTagContainer.begin()->SerializeToJson());
                }
            }
        }
        sessionReport.InsertValue("segment", cr.GetReport(locale, reportTraits, *Server));
        if (auto stage = i.GetStage()) {
            sessionReport.InsertValue("stage", std::move(stage));
        }

        NJson::TJsonValue carReport;
        if (objectId && fetcher.BuildOneCarInfoReport(objectId, permissions->GetFilterActions(), carReport)) {
            // do nothing
        } else if (auto objectModel = i.GetObjectModel()) {
            carReport["model_id"] = objectModel;
        }
        sessionReport.InsertValue("car", std::move(carReport));

        bool isSpeeding = IsSpeedLimitExceeded(i);
        auto& sessionTracks = GetTracks();
        auto track = sessionTracks.find(i.GetSessionId());
        TMaybe<NDrive::TTrackProjector> trackProjector;
        if (track != sessionTracks.end()) {
            NJson::TJsonValue trackReport = NJson::JSON_NULL;
            if (track->second.Wait(deadline) && track->second.HasValue()) {
                NDrive::TTracksLinker::TResults trackValue = track->second.GetValue();
                trackValue = FilterTrack(std::move(trackValue));
                isSpeeding &= ::IsSpeedLimitExceeded(trackValue);
                trackReport = NDrive::GetAnalyzerReport<NJson::TJsonValue>(trackValue, "geojson");
                trackProjector = NDrive::TTrackProjector(trackValue);
                if (!DriveApi->GetIMEI(objectId) && !trackValue.empty()){
                    AddTrackEndPointsToReport(trackReport, trackValue, i);
                }
            } else {
                g.AddEvent(NJson::TMapBuilder
                    ("event", "FetchTrackFailure")
                    ("exception", NThreading::GetExceptionInfo(track->second))
                    ("session_id", i.GetSessionId())
                );
            }
            sessionReport.InsertValue("track", std::move(trackReport));
        }

        auto& realtimeRideAggressions = GetRealtimeRideAggressions();
        auto& rideAggressions = GetRideAggressions();
        auto aggressiveEventsIt = rideAggressions.find(i.GetSessionId());
        if (aggressiveEventsIt != rideAggressions.end()) {
            sessionReport.InsertValue("is_aggressive", true);
            if (reportAggressionScore) {
                NJson::InsertNonNull(sessionReport, "aggressive_score", GetRideAggressionScore(i));
            }
            NJson::TJsonValue events = NJson::JSON_ARRAY;
            for (auto&& event : aggressiveEventsIt->second) {
                events.AppendValue(event.GetReport(deadline, trackProjector));
            }
            sessionReport.InsertValue("aggressive_events", std::move(events));
        } else if (auto rra = realtimeRideAggressions.find(i.GetSessionId()); rra != realtimeRideAggressions.end()) {
            sessionReport.InsertValue("is_aggressive", true);
            sessionReport.InsertValue("is_realtime_aggressive", true);
            NJson::TJsonValue events = NJson::JSON_ARRAY;
            for (auto&& event : rra->second) {
                events.AppendValue(event.GetReport(deadline, trackProjector));
            }
            sessionReport.InsertValue("aggressive_events", std::move(events));
        } else {
            sessionReport.InsertValue("is_aggressive", false);
        }

        {
            NJson::TJsonValue camera_events = NJson::JSON_ARRAY;
            auto taggedSession = DriveApi->GetTagsManager().GetTraceTags().GetCachedOrRestoreObject(i.GetSessionId(), session);
            if (taggedSession) {
                auto signalqTags = taggedSession->GetTagsByClass<TSignalqEventTraceTag>();
                for (auto tag: signalqTags) {
                    auto signalqTag = tag.MutableTagAs<TSignalqEventTraceTag>();
                    if (signalqTag && visibleCameraEvents.find(signalqTag->GetName()) != visibleCameraEvents.end()) {
                        const auto& event = signalqTag->GetEvent();
                        NJson::TJsonValue json;
                        NJson::InsertField(json, "ts", event.GetAt().Seconds());
                        auto optionalGnss = event.OptionalGnss();
                        if (optionalGnss && optionalGnss->OptionalLat() && optionalGnss->OptionalLon()) {
                            if (trackProjector) {
                                NJson::InsertField(json, "center", trackProjector->Project({*optionalGnss->OptionalLat(), *optionalGnss->OptionalLon()}, event.GetAt()));
                            } else {
                                NJson::InsertField(json, "center", TGeoCoord{*optionalGnss->OptionalLat(), *optionalGnss->OptionalLon()});
                            }
                        }
                        NJson::InsertField(json, "tag_id", tag.GetTagId());
                        NJson::InsertField(json, "kind", signalqTag->GetName());
                        camera_events.AppendValue(std::move(json));
                    }

                }
                if (!camera_events.GetArray().empty()) {
                    sessionReport.InsertValue("camera_events", std::move(camera_events));
                }
            }
        }

        sessionReport.InsertValue("is_speeding", isSpeeding);

        if (needGeocodedLocation) {
            TString geocodedStart = GetGeocodedStart(i.GetSessionId(), g);
            TString geocodedFinish = GetGeocodedFinish(i.GetSessionId(), g);
            if (geocodedStart) {
                sessionReport.InsertValue("geocoded_start", geocodedStart);
            }
            if (geocodedFinish) {
                sessionReport.InsertValue("geocoded_finish", geocodedFinish);
            }
        }
    }
    if (sessionId) {
        TUserSessionsContext userSessionsContext(Server, permissions, locale, permissions->GetDeviceReportTraits());
        TUserCurrentContext userCurrentContext(Server, permissions, reportTraits, GetOrigin(), GetUserLocation());
        if (!userCurrentContext.Initialize(session, g.MutableReport())) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
        if (!userSessionsContext.Initialize(session, g.MutableReport(), Context->GetRequestStartTime(), false, sessionId, &userCurrentContext)) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
        if (!sessions.empty()) {
            auto&& session = sessions.front();
            g.MutableReport().AddReportElement("photo_screen", GetPhotoScreenReport(*Server, userCurrentContext, session), /*allowUndefined=*/false);
        }
        if (userSessionsContext.GetSessions().size()) {
            auto&& session = userSessionsContext.GetSessions().front();
            g.MutableReport().AddReportElement("feedback", session.GetReportFeedback(locale, &userCurrentContext, needBill, reportTraits));
        }
        if (permissions->GetUserFeatures().GetIsPlusUser() && totalCashback > 0 && userCurrentContext.GetSettingsTag().HasData()) {
            auto countStr = userCurrentContext.GetSettingsTag().GetTagAs<TUserDictionaryTag>()->GetField("cashback_count");
            ui32 count = 0;
            if (!countStr || TryFromString(*countStr, count) && Server->GetSettings().GetValueDef<ui32>("billing.max_cashback_promo_count", 0) > count) {
                g.MutableReport().AddReportElement("show_cashback_promo", true);
                i64 balance = ((userCurrentContext.GetYandexPaymentMethod() ? userCurrentContext.GetYandexPaymentMethod()->GetBalance() : 0) / 100) * 100;
                g.MutableReport().AddReportElement("start_yandex_account_balance", balance);
            }
        }
        g.MutableReport().AddReportElement("multi_bill", mBill.GetReport(locale, NDriveSession::ReportUserApp, *Server));
    }
    g.MutableReport().AddReportElement("has_more", hasMore);
    g.MutableReport().AddReportElement("sessions", std::move(result));
    fetcher.BuildReportMeta(g.MutableReport());

    g.MutableReport().AddReportElement("server_time", Now().Seconds());
    g.SetCode(HTTP_OK);
}

void TCurrentSessionProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    TEventsGuard egProcess(g.MutableReport(), "current_session");

    const auto& cgi = Context->GetCgiParameters();
    const EApp app = GetUserApp();
    const auto multiSessions = GetValue<bool>(cgi, "multi_sessions", false).GetOrElse(false);
    const auto needBill = GetValue<bool>(cgi, "need_bill", false).GetOrElse(false);
    const auto report = GetString(cgi, "report", false);
    const auto sessionId = GetString(cgi, "session_id", false);
    const auto traits = GetValues<NDriveSession::EReportTraits>(cgi, "traits", false);
    const auto antitraits = GetValues<NDriveSession::EReportTraits>(cgi, "antitraits", false);
    const auto locale = GetLocale();
    const auto userLocation = GetUserLocation();
    const auto requestTimestamp = Context->GetRequestStartTime();

    NDriveSession::TReportTraits reportTraits = NDriveSession::GetReportTraits(report);
    for (auto&& trait : traits) {
        reportTraits |= trait;
    }
    if (permissions->GetSetting<bool>("offers.switching.enabled", false)) {
        reportTraits |= NDriveSession::EReportTraits::ReportSwitchable;
    }
    if (!GetHandlerSetting<bool>("fetch_payment_methods").GetOrElse(true)) {
        reportTraits &= ~NDriveSession::EReportTraits::ReportPaymentMethods;
    }
    for (auto&& trait : antitraits) {
        reportTraits &= ~trait;
    }

    TString externalUserId = GetUUID(cgi, "user_id", false);
    TString userId;
    TUserPermissions::TPtr userPermissions;
    if (externalUserId) {
        auto eg = g.BuildEventGuard("external_user_id_permissions");
        ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::User, externalUserId);
        TUserPermissionsFeatures upf;
        userId = externalUserId;
        userPermissions = DriveApi->GetUserPermissions(userId, upf);
    } else {
        userId = permissions->GetUserId();
        userPermissions = permissions;
    }
    ReqCheckCondition(!!userId, ConfigHttpStatus.EmptyRequestStatus, EDriveLocalizationCodes::UserNotFound);
    ReqCheckCondition(!!userPermissions, ConfigHttpStatus.EmptyRequestStatus, EDriveLocalizationCodes::NoPermissions);

    NDeviceReport::TReportTraits deviceReportTraits = permissions->GetDeviceReportTraits();
    if (!GetHandlerSetting<bool>("fetch_photos").GetOrElse(true)) {
        deviceReportTraits &= ~NDeviceReport::ReportValidatedPhotos;
        deviceReportTraits &= ~NDeviceReport::ReportSupportFilterPhotos;
        reportTraits &= ~NDriveSession::ReportPhotos;
    }

    TUserCurrentContext userCurrentContext(Server, userPermissions, reportTraits, GetOrigin(), userLocation);
    THolder<TUserSessionsContext> userSessionsContextPtr = MakeHolder<TUserSessionsContext>(Server, userPermissions, locale, deviceReportTraits);

    {
        auto session = BuildTx<NSQL::ReadOnly>();
        if (!userCurrentContext.Initialize(session, g.MutableReport())) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
        if (deviceReportTraits & EReportTraits::ReportBluetoothInfo
            && Server->GetDriveAPI()->HasBillingManager()
            && Server->GetDriveAPI()->GetBillingManager().GetDebt(userCurrentContext.GetActivePayments()) > 0) {
            deviceReportTraits &= ~EReportTraits::ReportBluetoothInfo;
            userSessionsContextPtr = MakeHolder<TUserSessionsContext>(Server, userPermissions, locale, deviceReportTraits);
        }
        if (!userSessionsContextPtr->Initialize(session, g.MutableReport(), requestTimestamp, !multiSessions, sessionId, &userCurrentContext)) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
    }
    const auto& userSessionsContext = *userSessionsContextPtr;

    const TSet<TString> supportVerdicts = StringSplitter(GetHandlerSettingDef<TString>("car.damage.allowed_support_verdicts", "")).Split(',').SkipEmpty();
    if (multiSessions) {
        {
            TEventsGuard eg(g.MutableReport(), "sessions");
            g.MutableReport().AddReportElement("sessions", userSessionsContext.GetSessionsReport(locale, needBill, reportTraits, app, &userCurrentContext));
        }
        if (userSessionsContext.HasSharedSessions()) {
            TEventsGuard eg(g.MutableReport(), "shared_sessions");
            g.MutableReport().AddReportElement("shared_sessions", userSessionsContext.GetSharedSessionsReport(locale, needBill, reportTraits, app, &userCurrentContext));
        }
        if (userSessionsContext.HasAdditionalServiceSessions()) {
            TEventsGuard eg(g.MutableReport(), "additional_service_sessions");
            g.MutableReport().AddReportElement("additional_service_sessions", userSessionsContext.GetAdditionalServiceSessionsReport(locale, needBill, reportTraits, app, &userCurrentContext));
        }
        {
            TEventsGuard eg(g.MutableReport(), "states");
            g.MutableReport().AddReportElement("states", userSessionsContext.GetStatesReport(locale, &userCurrentContext));
        }
        {
            TEventsGuard eg(g.MutableReport(), "futures");
            g.MutableReport().AddReportElement("futures", userSessionsContext.GetFuturesReport(locale, &userCurrentContext));
        }
        {
            TEventsGuard eg(g.MutableReport(), "delegations");
            g.MutableReport().AddReportElement("delegations", userSessionsContext.GetDelegationsReport());
        }
        {
            TEventsGuard eg(g.MutableReport(), "incoming_delegations");
            g.MutableReport().AddReportElement("incoming_delegations", userCurrentContext.GetIncomingDelegationsReport());
        }
        {
            TEventsGuard eg(g.MutableReport(), "cars");
            g.MutableReport().AddReportElementString("cars", userSessionsContext.GetCarsReport(supportVerdicts));
        }
        {
            TEventsGuard eg(g.MutableReport(), "cars_meta");
            g.MutableReport().AddReportElement("cars_meta", userSessionsContext.GetCarsMetaReport());
        }
        {
            TEventsGuard eg(g.MutableReport(), "extra_sessions_available");
            g.MutableReport().AddReportElement("extra_sessions_available", userSessionsContext.IsExtraSessionsAvailable());
        }
    } else {
        TDBTag futuresTag;
        const TUserSessionsContext::TSessionContext* currentSessionContext = nullptr;
        {
            if (userSessionsContext.GetFutures().size()) {
                futuresTag = userSessionsContext.GetFutures().front();
            } else if (userCurrentContext.GetFutures().size()) {
                futuresTag = userCurrentContext.GetFutures().front();
            } else if (userSessionsContext.GetSessions().size()) {
                currentSessionContext = &userSessionsContext.GetSessions().front();
            }
        }
        if (currentSessionContext) {
            TEventsGuard eg(g.MutableReport(), "context");
            TStringStream ss;
            if (currentSessionContext->ShouldReportObject()) {
                NJson::TJsonWriter jWriter(&ss, false);
                currentSessionContext->BuildCarReport(jWriter, userSessionsContext.GetFetcher(!currentSessionContext->GetContextClosed()), userPermissions, supportVerdicts);
            }
            TString carReport = ss.Str();
            if (carReport) {
                TEventsGuard egInt(g.MutableReport(), "car");
                g.MutableReport().AddReportElementString("car", std::move(carReport));
            } else {
                g.AddReportElement("car", NJson::JSON_NULL);
            }
            userSessionsContext.GetFetcher(!currentSessionContext->GetContextClosed()).BuildReportMeta(g.MutableReport());

            {
                TEventsGuard egInt(g.MutableReport(), "segment");
                g.MutableReport().AddReportElement("segment", currentSessionContext->GetReportSession(locale, needBill, reportTraits));
            }
            {
                TEventsGuard egInt(g.MutableReport(), "notification");
                g.MutableReport().AddReportElement("notification", currentSessionContext->GetReportNotification(locale), false);
            }
            {
                TEventsGuard egInt(g.MutableReport(), "poi");
                g.MutableReport().AddReportElement("poi", currentSessionContext->GetReportPoi(locale), false);
            }
            {
                TEventsGuard egInt(g.MutableReport(), "device_diff");
                g.MutableReport().AddReportElement("device_diff", currentSessionContext->GetReportObjectDiff(locale), false);
            }
            {
                TEventsGuard egInt(g.MutableReport(), "feedback");
                g.MutableReport().AddReportElement("feedback", currentSessionContext->GetReportFeedback(locale, &userCurrentContext, needBill, reportTraits));
            }
            {
                TEventsGuard egInt(g.MutableReport(), "transportation");
                g.MutableReport().AddReportElement("transportation", currentSessionContext->GetReportTransportation(userLocation.Get(), app, locale), false);
            }
            {
                TEventsGuard egInt(g.MutableReport(), "fueling_ability");
                g.MutableReport().AddReportElement("fueling_ability", currentSessionContext->GetFuelingAbility());
            }
            {
                TEventsGuard egInt(g.MutableReport(), "tanker_fueling_ability");
                g.MutableReport().AddReportElement("tanker_fueling_ability", currentSessionContext->GetTankerFuelingAbility());
            }
            {
                TEventsGuard egInt(g.MutableReport(), "additional_service_offers");
                g.MutableReport().AddReportElement("additional_service_offers", currentSessionContext->GetAdditionalServiceOffersReport(locale, &userCurrentContext, userSessionsContext));
            }
        } else if (futuresTag) {
            auto futuresTagImpl = futuresTag.GetTagAs<TTagReservationFutures>();
            R_ENSURE(futuresTagImpl, HTTP_INTERNAL_SERVER_ERROR, "cannot cast tag " << futuresTag.GetTagId() << " to TagReservationFutures");

            if (!futuresTagImpl->IsFailed()) {
                TEventsGuard egInt(g.MutableReport(), "future");
                TStringStream ss;
                {
                    NJson::TJsonWriter jWriter(&ss, false);
                    TUserSessionsContext::TSessionContext::BuildCarReport(jWriter, futuresTagImpl->IsFailed(), futuresTag.GetObjectId(), false, false, userSessionsContext.GetFetcher(!futuresTagImpl->IsFailed()), userPermissions);
                }
                TString carReport = ss.Str();
                g.MutableReport().AddReportElementString("car", std::move(carReport));
                userSessionsContext.GetFetcher(!futuresTagImpl->IsFailed()).BuildReportMeta(g.MutableReport());
            } else {
                TEventsGuard egInt(g.MutableReport(), "future_fail");
                TString carReport = "{\"number\" : \"\", \"model_id\" : \"\"}";
                g.MutableReport().AddReportElementString("car", std::move(carReport));
            }

            ICommonOffer::TReportOptions reportOptions(locale, NDriveSession::ReportUserApp);
            g.AddReportElement("futures_offer", futuresTagImpl->GetOffer()->BuildJsonReport(reportOptions, *Server));
            g.AddReportElement("futures_offer_failed", futuresTagImpl->IsFailed());
            g.AddReportElement("futures_offer_tag_id", futuresTag.GetTagId());

            g.MutableReport().AddReportElement("fueling_ability", false);
            g.MutableReport().AddReportElement("tanker_fueling_ability", false);
        }
        {
            TEventsGuard eg(g.MutableReport(), "delegation");
            if (currentSessionContext && !!currentSessionContext->GetDelegationTag()) {
                g.MutableReport().AddReportElement("delegation", currentSessionContext->GetReportDelegation());
            } else if (userCurrentContext.GetDelegations().size()) {
                g.MutableReport().AddReportElement("delegation", userCurrentContext.GetDelegations().front().GetTagAs<TDelegationUserTag>()->GetReport(userCurrentContext.GetDelegations().front(), Server));
            }
        }
        if (userCurrentContext.GetIncomingDelegations().size()) {
            TEventsGuard eg(g.MutableReport(), "incoming_delegations");
            g.MutableReport().AddReportElement("incoming_delegations", userCurrentContext.GetIncomingDelegationsReport());
        }
    }
    {
        TEventsGuard eg(g.MutableReport(), "flags");
        g.MutableReport().AddReportElement("flags", userCurrentContext.GetFlagsReport());
    }
    {
        TEventsGuard eg(g.MutableReport(), "area_common_user_info");
        g.MutableReport().AddReportElement("area_common_user_info", userCurrentContext.GetAreaUserReport());
    }
    {
        TEventsGuard eg(g.MutableReport(), "user");
        g.MutableReport().AddReportElement("user", userCurrentContext.GetReport(locale, requestTimestamp, userSessionsContext.GetSessionCount()));
    }
    {
        TEventsGuard eg(g.MutableReport(), "server_time");
        g.MutableReport().AddReportElement("server_time", Now().Seconds());
    }
    {
        TEventsGuard eg(g.MutableReport(), "device_status");
        g.MutableReport().AddReportElement("device_status", ::ToString(GetNewDeviceStatus()));
    }
    g.SetCode(HTTP_OK);
}

void TDropSessionProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    auto sessionId = GetString(requestData, "session_id");
    auto userId = GetString(requestData, "user_id", false);

    auto session = BuildTx<NSQL::Writable>();
    if (!userId) {
        userId = permissions->GetUserId();
    }
    if (!TChargableTag::DropSession(sessionId, userId, *permissions, *Server, session)) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }
    if (!session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }
    g.SetCode(HTTP_OK);
}

void TSharedSessionBookProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    auto sessionId = GetString(requestData, "session_id");

    auto tx = BuildTx<NSQL::Writable>();
    auto sharingTag = TSessionSharingTag::Get(sessionId, permissions->GetUserId(), *Server, tx);
    R_ENSURE(sharingTag, {}, "cannot get SessionSharingTag for session " << sessionId, tx);
    R_ENSURE(sharingTag->GetPerformer() == permissions->GetUserId(), HTTP_INTERNAL_SERVER_ERROR, "wrong performer for tag " << sharingTag.GetTagId(), tx);
    auto bookingTag = TSessionSharingTag::Book(std::move(sharingTag), *permissions, *Server, tx);
    R_ENSURE(bookingTag, {}, "cannot Book shared session " << sessionId, tx);
    R_ENSURE(tx.Commit(), {}, "cannot Commit", tx);
    g.SetCode(HTTP_OK);
}

void TSharedSessionInviteProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    auto sessionId = GetString(requestData, "session_id");
    auto userId = GetString(requestData, "user_id");

    auto tx = BuildTx<NSQL::Writable>();
    auto optionalSession = DriveApi->GetSessionManager().GetSession(sessionId, tx);
    R_ENSURE(optionalSession, {}, "cannot GetSession", tx);
    auto session = *optionalSession;
    R_ENSURE(session, HTTP_NOT_FOUND, "cannot find session " << sessionId, tx);
    auto compilation = session->GetCompilationAs<TBillingSession::TBillingCompilation>();
    R_ENSURE(compilation, HTTP_INTERNAL_SERVER_ERROR, "could not get billing compilation", tx);
    R_ENSURE(session->GetUserId() == permissions->GetUserId(), HTTP_FORBIDDEN, "cannot share someone elses session", tx);
    R_ENSURE(!session->GetClosed(), HTTP_BAD_REQUEST, "session " << sessionId << " is already closed", tx);
    R_ENSURE(compilation->GetCurrentOffer()->CheckSession(*permissions, tx, Server, false) == EDriveSessionResult::Success, HTTP_FORBIDDEN, "session " << sessionId << " has debt", tx);

    auto userFetchResult = DriveApi->GetUsersData()->FetchInfo(userId, tx);
    R_ENSURE(userFetchResult, {}, "cannot FetchInfo " << userId, tx);

    auto user = userFetchResult.GetResultPtr(userId);
    R_ENSURE(user, HTTP_NOT_FOUND, "cannot find user " << userId, tx);
    R_ENSURE(user->GetStatus() == NDrive::UserStatusActive, HTTP_BAD_REQUEST, "user " << userId << " is not active", tx);

    auto sharingTag = TSessionSharingTag::Invite(session, userId, *permissions, *Server, tx);
    if (sharingTag) {
        g.AddEvent(NJson::TMapBuilder
            ("event", "session_sharing_invite")
            ("tag", NJson::ToJson(sharingTag))
        );
    }
    R_ENSURE(tx.Commit(), {}, "cannot Commit", tx);
    g.SetCode(HTTP_OK);
}

void TSessionsProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr /*permissions*/, const NJson::TJsonValue& /*requestData*/) {
    const TCgiParameters& cgi = Context->GetCgiParameters();

    TVector<TString> objects;
    NJson::TJsonValue report(NJson::JSON_ARRAY);
    if (cgi.Has("session_id")) {
        auto builder = Server->GetDriveAPI()->GetTagsManager().GetDeviceTags().GetHistoryManager().GetSessionsBuilderSafe("billing");
        R_ENSURE(builder, ConfigHttpStatus.UnknownErrorStatus, "cannot receive sessions info");
        auto sessionId = GetString(cgi, "session_id");
        auto session = builder->GetSession(sessionId);
        if (session) {
            R_ENSURE(session->Compile(), ConfigHttpStatus.UnknownErrorStatus, "cannot compile session " << sessionId);
            report.AppendValue(session->GetEventsReport<TShortSessionReportBuilder>());
        }
    } else {
        auto since = GetTimestamp(cgi, "since").GetRef();
        auto until = GetTimestamp(cgi, "until", TInstant::Max());
        bool reportActiveSessions = GetValue<bool>(cgi, "report_active_sessions", false).GetOrElse(false);

        auto builder = Server->GetDriveAPI()->GetTagsManager().GetDeviceTags().GetHistoryManager().GetSessionsBuilderSafe("billing", until);
        R_ENSURE(builder, HTTP_INTERNAL_SERVER_ERROR, "cannot receive sessions info in required interval");

        TMap<TString, TVector<IEventsSession<TCarTagHistoryEvent>::TPtr>> sessions;
        if (cgi.Has("car_id")) {
            objects = GetUUIDs(cgi, "car_id");
            sessions = builder->GetSessionsActualByObjects(since, until, objects);
        } else if (cgi.Has("imei")) {
            const TVector<TString> imeis = GetStrings(cgi, "imei");
            auto rCars = Server->GetDriveAPI()->GetCarsImei()->GetCachedOrFetch(imeis);
            for (auto&& i : rCars.GetResult()) {
                if (!!i.second.GetIMEI()) {
                    objects.emplace_back(i.second.GetId());
                }
            }
            sessions = builder->GetSessionsActualByObjects(since, until, objects);
        } else {
            sessions = builder->GetSessionsActual(since, until);
        }
        for (auto&& sessionsList : sessions) {
            for (auto&& s : sessionsList.second) {
                if (s->HasEvents(since, until)) {
                    report.AppendValue(s->GetEventsReport<TShortSessionReportBuilder>(since, until));
                    continue;
                }
                if (!s->GetClosed() && reportActiveSessions) {
                    report.AppendValue(s->GetEventsReport<TShortSessionReportBuilder>(TInstant::Zero(), until));
                    continue;
                }
            }
        }
    }

    g.MutableReport().AddReportElement("sessions", std::move(report));
    g.SetCode(HTTP_OK);
}

void TOfferBookProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    auto locale = GetLocale();
    auto session = BuildTx<NSQL::Writable | NSQL::RepeatableRead>();
    if (!TUserProblemTag::EnsureNotBlocked(permissions->GetUserId(), *Server, session)) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }

    TOffersBuildingContext offersBuildingContext(Server);
    if (!offersBuildingContext.Parse(Context, *permissions, requestData, session)) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }
    {
        TUserOfferContext uoc(Server, permissions, Context);
        ReqCheckCondition(uoc.FillData(this, requestData), ConfigHttpStatus.UnknownErrorStatus, "user_offer_context_fill_data_error");
        uoc.SetNeedDefaultAccount(false);
        if (!uoc.GetRequestAccountIds()) {
            uoc.SetFilterAccounts(false);
        }
        uoc.SetExternalUserId(GetExternalUserId());
        uoc.SetLocale(locale);
        uoc.SetOrigin(GetOrigin());
        ReqCheckCondition(uoc.Prefetch(), ConfigHttpStatus.UnknownErrorStatus, "user_offer_context_prefetch_error");
        offersBuildingContext.SetUserHistoryContext(std::move(uoc));
    }
    TVector<IOfferReport::TPtr> offersAll;

    auto correctors = permissions->GetOfferCorrections();
    {
        auto additionalActions = DriveApi->GetRolesManager()->GetUserAdditionalActions(permissions->GetUserId(), DriveApi->GetTagsManager(), false, session);
        R_ENSURE(additionalActions, ConfigHttpStatus.UnknownErrorStatus, "cannot GetUserAdditionalActions", session);
        auto filtered = TAdditionalActions::Filter<IOfferCorrectorAction>(std::move(*additionalActions));
        correctors.insert(correctors.end(), filtered.begin(), filtered.end());
    }
    if (offersBuildingContext.HasUserHistoryContext()) {
        auto walletAdditionalActions = DriveApi->GetRolesManager()->GetAccountsAdditionalActions(offersBuildingContext.GetUserHistoryContextRef().GetRequestUserAccounts(), DriveApi->GetTagsManager(), false, session);
        R_ENSURE(walletAdditionalActions, ConfigHttpStatus.UnknownErrorStatus, "cannot GetAccountsAdditionalActions", session);
        auto filtered = TAdditionalActions::Filter<IOfferCorrectorAction>(std::move(*walletAdditionalActions));
        correctors.insert(correctors.end(), filtered.begin(), filtered.end());
    }

    for (auto&& i : permissions->GetOfferBuilders()) {
        auto action = dynamic_cast<const IOfferBuilderAction*>(i.Get());
        if (!action || (!offersBuildingContext.GetUserHistoryContextUnsafe().GetRequestAccountIds() && !Config.AcceptAction(action->GetName()))) {
            continue;
        }

        TVector<IOfferReport::TPtr> offersCurrent;
        if (action->BuildOffers(*permissions, correctors, offersCurrent, offersBuildingContext, Server, session) == EOfferCorrectorResult::Problems) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }

        const auto pred = [this](IOfferReport::TPtr offer) {
            return !Yensured(offer->GetOffer())->NeedStore(*Yensured(Server));
        };
        offersCurrent.erase(std::remove_if(offersCurrent.begin(), offersCurrent.end(), pred), offersCurrent.end());

        offersAll.insert(offersAll.end(), offersCurrent.begin(), offersCurrent.end());
        if (!offersAll.empty()) {
            break;
        }
    }
    ReqCheckCondition(!offersAll.empty(), ConfigHttpStatus.EmptySetStatus, "cannot_build_offers");

    TChargableTag::TBookOptions bookOptions;
    bookOptions.CheckBlocked = false;
    bookOptions.DeviceId = GetDeviceId();
    bookOptions.MultiRent = GetHandlerSettingDef("multi_rent", false);
    auto offer = offersAll.front()->GetOfferPtrAs<IOffer>();
    auto booked = TChargableTag::Book(offer, *permissions, *Server, session, bookOptions);
    R_ENSURE(booked, {}, "cannot book offer", session);
    R_ENSURE(session.Commit(), {}, "cannot Commit", session);
    g.SetCode(HTTP_OK);
}

void TReplaceCarPrepareProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /* requestData */) {
    /* session_id in cgi */

    auto locale = GetLocale();
    auto tx = BuildTx<NSQL::Writable>();
    R_ENSURE(TUserProblemTag::EnsureNotBlocked(permissions->GetUserId(), *Server, tx), {}, "user blocked", tx);

    auto userSession = GetCurrentUserSession(permissions, Context->GetRequestStartTime());
    R_ENSURE(userSession, {}, "no current session");

    auto deviceTags = DriveApi->GetTagsManager().GetDeviceTags().RestoreObject(userSession->GetObjectId(), tx);
    R_ENSURE(deviceTags, {}, "cannot get device tags", tx);
    auto replacingTagRaw = deviceTags->GetTag(TReplaceCarTag::TypeName);
    R_ENSURE(replacingTagRaw, ConfigHttpStatus.PermissionDeniedStatus, "no replacing tag on car");
    auto replacingTag = replacingTagRaw->MutableTagAs<TReplaceCarTag>();
    R_ENSURE(replacingTag, ConfigHttpStatus.PermissionDeniedStatus, "replacing tag has wrong type");

    auto billingSession = std::dynamic_pointer_cast<const TBillingSession>(userSession);
    R_ENSURE(billingSession, {}, "cannot get billing tx");
    auto currentOffer = billingSession->GetCurrentOffer();
    R_ENSURE(currentOffer, {}, "no current offer");
    const auto builderAction = DriveApi->GetRolesManager()->GetAction(currentOffer->GetBehaviourConstructorId());
    R_ENSURE(builderAction, {}, "no current offer builder");
    const auto longTermBuilderAction = builderAction->GetAs<TLongTermOfferBuilder>();
    R_ENSURE(longTermBuilderAction, {}, "current offer builder not long term");
    const auto replaceBuilder = longTermBuilderAction->GetReplaceOfferBuildingAction();
    R_ENSURE(replaceBuilder, {}, "no replace builder specified");
    const auto tmpBuilderAction = DriveApi->GetRolesManager()->GetAction(replaceBuilder);
    R_ENSURE(tmpBuilderAction, {}, "no replace builder");
    const auto tmpOfferBuilderAction = tmpBuilderAction->GetAs<IOfferBuilderAction>();
    R_ENSURE(tmpOfferBuilderAction, {}, "replace builder not a builder");

    TOffersBuildingContext offersBuildingContext(Server);
    offersBuildingContext.SetCarId(userSession->GetObjectId());
    offersBuildingContext.SetSwitching(true);
    TUserOfferContext uoc(Server, permissions);
    uoc.SetLocale(locale);
    offersBuildingContext.SetUserHistoryContext(std::move(uoc));
    auto offers = tmpOfferBuilderAction->BuildOffers(offersBuildingContext, tx);
    R_ENSURE(offers.size() == 1, {}, "one offer expected, found " + ToString(offers.size()), tx);
    auto tmpOfferReport = offers[0];
    auto storedOffer = Server->GetOffersStorage()->StoreOffers({tmpOfferReport}, tx);
    R_ENSURE(storedOffer.Initialized(), {}, "cannot store offer", tx);

    replacingTag->SetTmpOfferId(tmpOfferReport->GetOffer()->GetOfferId());
    auto updated = DriveApi->GetTagsManager().GetDeviceTags().UpdateTagData(*replacingTagRaw, permissions->GetUserId(), tx);
    R_ENSURE(updated, {}, "cannot update replacing tag", tx);

    R_ENSURE(tx.Commit(), {}, "cannot commit tx", tx);

    g.SetCode(HTTP_OK);
}

void TBookProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    TEventsGuard egProcess(g.MutableReport(), "book");
    const TString offerId = GetString(requestData, "offer_id", true);
    const auto futures = GetValue<bool>(requestData, "is_future_car", false);
    const auto targetUserTags = GetStrings(requestData, "target_user_tags", false);
    const TString mobilePaymethodId = GetString(requestData, "mobile_paymethod_id", false);

    auto session = BuildTx<NSQL::Writable | NSQL::RepeatableRead>();
    if (!TUserProblemTag::EnsureNotBlocked(permissions->GetUserId(), *Server, session)) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    };
    auto gUser = Server->GetDriveAPI()->GetUsersData()->FetchInfo(permissions->GetUserId(), session);
    egProcess.AddEvent("fetch_user_info");
    const TDriveUserData* userData = gUser.GetResultPtr(permissions->GetUserId());
    Y_ENSURE_EX(!userData || userData->IsPhoneVerified(), TCodedException(ConfigHttpStatus.UserErrorState)
        .SetErrorCode("phone_not_verified")
        .SetLocalizedMessage(NDrive::TLocalization::PhoneNotVerified())
    );
    const bool skipEmailVerification = permissions->GetSetting<bool>("user.skip_email_verification", false);
    Y_ENSURE_EX(skipEmailVerification || !userData || !!userData->GetEmail(), TCodedException(ConfigHttpStatus.UserErrorState)
        .SetErrorCode("email_not_verified")
        .SetLocalizedMessage(NDrive::TLocalization::EmailNotVerified())
    );

    auto asyncOffer = Server->GetOffersStorage()->RestoreOffer(offerId, permissions->GetUserId(), session);
    if (!asyncOffer.Initialized()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }
    {
        TEventsGuard eg(g.MutableReport(), "WaitRestoreOffer");
        R_ENSURE(asyncOffer.Wait(Context->GetRequestDeadline()), ConfigHttpStatus.TimeoutStatus, "RestoreOffer timeout");
    }
    auto baseOffer = asyncOffer.GetValue();
    R_ENSURE(baseOffer, ConfigHttpStatus.EmptySetStatus, "null offer restored");
    egProcess.AddEvent("check_user_cards");
    auto checkCards = CheckUserCards(permissions, baseOffer);

    if (GetHandlerSettingDef<bool>("agreement.check.enabled", false)) {
        auto checkResult = baseOffer->CheckAgreementRequirements(GetLocale(), permissions, Server);
        if (!checkResult) {
            auto errorReport = checkResult.GetError();
            g.MutableReport().AddReportElement("error_details", std::move(errorReport));
            g.SetCode(ConfigHttpStatus.PermissionDeniedStatus);
            return;
        }
    }

    if (auto additionalServiceOffer = std::dynamic_pointer_cast<TAdditionalServiceOffer>(baseOffer)) {
        TAdditionalServiceOfferHolderTag::TBookOptions bookOptions;
        bookOptions.CheckBlocked = false;
        auto booked = TAdditionalServiceOfferHolderTag::Book(baseOffer, *permissions, *Server, session, bookOptions);
        R_ENSURE(booked, {}, "cannot book offer", session);
    } else if (auto dedicatedFleetOffer = std::dynamic_pointer_cast<TDedicatedFleetOffer>(baseOffer)) {
        TDedicatedFleetOfferHolderTag::TBookOptions bookOptions;
        bookOptions.CheckBlocked = false;
        auto booked = TDedicatedFleetOfferHolderTag::Book(baseOffer, *permissions, *Server, session, bookOptions);
        R_ENSURE(booked, {}, "cannot book offer", session);
    } else {
        auto offer = std::dynamic_pointer_cast<IOffer>(baseOffer);
        R_ENSURE(offer, ConfigHttpStatus.EmptySetStatus, "wrong-typed offer restored");
        if (!targetUserTags.empty()) {
            offer->SetTargetUserTags(MakeSet(targetUserTags));
        }

        TChargableTag::TBookOptions bookOptions;
        bookOptions.DeviceId = GetDeviceId();
        bookOptions.Futures = futures;
        bookOptions.CheckBlocked = false;
        bookOptions.MultiRent = GetHandlerSettingDef("multi_rent", false);
        bookOptions.MobilePaymethodId = mobilePaymethodId;
        auto booked = TChargableTag::Book(offer, *permissions, *Server, session, bookOptions);
        R_ENSURE(booked, {}, "cannot book offer", session);
    }

    if (DriveApi->HasBillingManager() && baseOffer->GetSelectedCreditCard()) {
        auto eg = g.BuildEventGuard("SetDefaultCreditCard");
        bool set = DriveApi->GetBillingManager().SetDefaultCreditCard(baseOffer->GetSelectedCreditCard(), permissions->GetUserId(), !!checkCards, session);
        R_ENSURE(set, {}, "cannot SetDefaultCreditCard", session);
    }
    if (!session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }
    g.SetCode(HTTP_OK);
}


void TCarAdminPortalInfoProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::Car);

    const auto& cgi = Context->GetCgiParameters();
    const auto& carId = GetUUID(cgi, "car_id", true);
    const auto locale = GetLocale();
    // get basic car information
    TCarsFetcher fetcher(*Server, permissions->GetDeviceReportTraits());
    fetcher.SetIsRealtime(true);
    fetcher.SetLocale(locale);
    fetcher.SetSensorIds({
        CAN_FUEL_LEVEL_P,
        CAN_ODOMETER_KM,
        CAN_DRIVER_DOOR,
        CAN_PASS_DOOR,
        CAN_L_REAR_DOOR,
        CAN_R_REAR_DOOR,
        CAN_HOOD,
        CAN_TRUNK,
        CAN_ENGINE_IS_ON,
        VEGA_ACC_VOLTAGE,
        VEGA_POWER_VOLTAGE,
        VEGA_SPEED,
        VEGA_NRF_VISIBLE_MARKS_BF,
        VEGA_NRF_BATTLOW_MARKS_BF,
        VEGA_NRF_MARK_ID_1,
        NDrive::NVega::DigitalOutput<3>(),
        NDrive::NVega::SvrRawState,
        CAN_ACCELERATOR,
    });
    {
        TTimeGuard tgFetch("Fetcher");
        R_ENSURE(fetcher.FetchData(permissions, {carId}), HTTP_INTERNAL_SERVER_ERROR, "unknown failure");
    }
    TString carReportStr;
    {
        TStringOutput so(carReportStr);
        NJson::TJsonWriter carReport(&so, false);
        carReport.OpenMap();
        Y_ENSURE_EX(fetcher.BuildOneCarInfoReport(carId, permissions->GetFilterActions(), carReport), TCodedException(ConfigHttpStatus.SyntaxErrorStatus) << "cannot build report about car");

        carReport.Write("models", fetcher.GetModelsReportSafe());
        carReport.Write("sf", fetcher.GetShortCarPropertiesReportSafe());
        carReport.Write("views", fetcher.GetViewsReportSafe());

        // get current order (i.e. session) and user in this order
        auto sessionBuilder = Server->GetDriveAPI()->GetTagsManager().GetDeviceTags().GetHistoryManager().GetSessionsBuilder("billing");
        Y_ENSURE_EX(!!sessionBuilder, TCodedException(ConfigHttpStatus.UnknownErrorStatus) << "incorrect builder for 'billing' cgi field");
        auto session = sessionBuilder->GetLastObjectSession(carId);
        if (session) {
            auto dbSession = BuildTx<NSQL::ReadOnly>();
            auto payments = Server->GetDriveAPI()->GetBillingManager().GetPaymentsManager().GetFinishedPayments(NContainer::Scalar(session->GetSessionId()), dbSession);
            Y_ENSURE_EX(!!payments, TCodedException(ConfigHttpStatus.UnknownErrorStatus) << dbSession.GetStringReport());

            auto reportCustomization = MakeAtomicShared<TBillingReportCustomization>();
            reportCustomization->SetNeedBill(true);
            reportCustomization->OptionalPaymentsData() = payments->size() ? payments->begin()->second : TMaybe<TPaymentsData>();
            carReport.Write("session_info", session->GetReport(locale, Server, reportCustomization));

            auto userId = session->GetUserId();
            NJson::TJsonValue userReport;
            auto fetchResult = Server->GetDriveAPI()->GetUsersData()->RestoreUser(userId, dbSession);
            R_ENSURE(fetchResult, ConfigHttpStatus.UnknownErrorStatus, "user " << userId << " is not found", dbSession);
            userReport = fetchResult->GetReport();
            carReport.Write("user", std::move(userReport));
        } else {
            carReport.Write("session", NJson::JSON_NULL);
            carReport.Write("user", NJson::JSON_NULL);
        }
        carReport.CloseMap();
    }
    g.MutableReport().SetExternalReportString(std::move(carReportStr));
    g.SetCode(HTTP_OK);
}

void TCarWebInfoProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    const auto& cgi = Context->GetCgiParameters();
    auto carId = GetString(cgi, "car_id", false);
    auto carNumber = GetString(cgi, "car_number", false);
    auto carQrCode = GetString(cgi, "car_qr_code", false);
    if (!carId && carNumber) {
        carId = DriveApi->GetCarIdByNumber(carNumber);
        R_ENSURE(carId, HTTP_NOT_FOUND, "cannot find car by number " << carNumber);
    }
    if (!carId && carQrCode) {
        carId = DriveApi->GetCarAttachmentAssignments().GetAttachmentOwnerByServiceAppSlug(carQrCode);
        R_ENSURE(carId, HTTP_NOT_FOUND, "cannot find car by QR code " << carQrCode);
    }
    R_ENSURE(carId, HTTP_BAD_REQUEST, "no car_* parameters specified");

    NDeviceReport::TReportTraits reportTraits =
        NDeviceReport::ReportAvailability |
        NDeviceReport::ReportCarId |
        NDeviceReport::DeepLink |
        NDeviceReport::ReportViews;
    TCarsFetcher fetcher(*Server, reportTraits);
    fetcher.SetCheckVisibility(false);
    fetcher.SetIsRealtime(false);
    fetcher.SetLocale(GetLocale());
    {
        auto eg = g.BuildEventGuard("FetchData");
        R_ENSURE(fetcher.FetchData(permissions, {carId}), HTTP_INTERNAL_SERVER_ERROR, "unknown failure");
    }
    {
        NJson::TJsonValue carReport;
        R_ENSURE(fetcher.BuildOneCarInfoReport(carId, {}, carReport), HTTP_INTERNAL_SERVER_ERROR, "cannot build report for " << carId);
        g.AddReportElement("car", std::move(carReport));
    }
    g.AddReportElement("config", Server->GetSettings().GetJsonValue("car_complaints.config"));
    g.AddReportElement("views", fetcher.GetViewsReportSafe());
    g.SetCode(HTTP_OK);
}

void TUserEmailBindProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    auto userId = permissions->GetUserId();
    auto email = ToLowerUTF8(GetString(requestData, "email"));

    auto session = BuildTx<NSQL::Writable>();
    auto userFetchResult = DriveApi->GetUsersData()->FetchInfo(userId, session);
    auto userPtr = userFetchResult.GetResultPtr(userId);

    bool isBadEmail = false;
    {
        auto emailTokens = SplitString(email, "@");
        if (emailTokens.size() != 2) {
            isBadEmail = true;
        } else if (emailTokens[1].find(".") == TString::npos) {
            isBadEmail = true;
        } else if (emailTokens[1].find(" ") != TString::npos) {
            isBadEmail = true;
        }
    }

    Y_ENSURE_EX(
        !isBadEmail,
        TCodedException(ConfigHttpStatus.SyntaxErrorStatus)
            .SetErrorCode("email.invalid")
            .SetLocalizedMessage(NDrive::TLocalization::EmailInvalid())
    );

    Y_ENSURE_EX(
        !DriveApi->GetUsersData()->CheckEmailExists(email) || userPtr->GetEmail() == email,
        TCodedException(ConfigHttpStatus.SyntaxErrorStatus)
            .SetErrorCode("email.exists")
            .SetLocalizedMessage(NDrive::TLocalization::EmailExists())
    );

    auto user = *userPtr;
    user.SetEmail(email);
    user.SetEMailVerified(true);

    {
        if (!DriveApi->GetUsersData()->UpdateUser(user, userId, session) || !session.Commit()) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
    }

    g.MutableReport().AddReportElement("status", "success");
    g.SetCode(HTTP_OK);

    bool sendConfirmationEmail = (userPtr->GetEmail() != email && !permissions->GetUserFeatures().GetDefaultEmails().contains(email) && permissions->HasAction(GetHandlerSettingDef<TString>("send_confirmation_email_action", "send_confirmation_email_flag")));
    if (sendConfirmationEmail) {
        TString userAuthHeader(Context->GetRequestData().HeaderInOrEmpty("Authorization"));
        auto session = DriveApi->BuildTx<NSQL::Writable>();
        if (DriveApi->BindUserMail(email, userId, userAuthHeader, GetClientIp(), *Server, session) && session.Commit()) {
            TUnistatSignalsCache::SignalAdd("frontend", "send_confirmation_email-success", 1);
        } else {
            TUnistatSignalsCache::SignalAdd("frontend", "send_confirmation_email-fail", 1);
        }
    }
}

void TSendVerificationEmailProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    auto userId = permissions->GetUserId();

    auto session = BuildTx<NSQL::Writable>();
    auto userFetchResult = DriveApi->GetUsersData()->FetchInfo(permissions->GetUserId(), session);
    auto userPtr = userFetchResult.GetResultPtr(userId);
    Y_ENSURE_EX(!!userPtr, TCodedException(ConfigHttpStatus.UnknownErrorStatus) << "unknown user");

    TString userAuthHeader(Context->GetRequestData().HeaderInOrEmpty("Authorization"));
    if (!DriveApi->BindUserMail(userPtr->GetEmail(), userId, userAuthHeader, GetClientIp(), *Server, session) && session.Commit()) {
        TUnistatSignalsCache::SignalAdd("frontend", "send_confirmation_email-fail", 1);
        session.DoExceptionOnFail(ConfigHttpStatus);
    }
    TUnistatSignalsCache::SignalAdd("frontend", "send_confirmation_email-success", 1);
    g.SetCode(HTTP_OK);
}

void TUserEmailConfirmProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    auto userId = permissions->GetUserId();
    auto key = GetString(requestData, "key", true);

    TString userAuthHeader(Context->GetRequestData().HeaderInOrEmpty("Authorization"));
    auto session = BuildTx<NSQL::ReadOnly>();
    if (!DriveApi->ConfirmUserMail(key, userId, userAuthHeader, GetClientIp(), *Server, session)) {
        TUnistatSignalsCache::SignalAdd("frontend", "confirmation_email-fail", 1);
        session.DoExceptionOnFail(ConfigHttpStatus);
    } else {
        TUnistatSignalsCache::SignalAdd("frontend", "confirmation_email-success", 1);
    }
    g.SetCode(HTTP_OK);
}

void TUserPhoneBindSubmitProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    Y_ENSURE_EX(Server->GetUserDevicesManager(), TCodedException(ConfigHttpStatus.ServiceUnavailable) << "incorrect server configuration");
    auto locale = GetLocale();
    auto phone = GetString(requestData, "phone", false);

    if (phone) {
        TPhoneNormalizer normalizer;
        auto normalized = normalizer.TryNormalize(phone);
        phone = normalized;
    }
    auto verificationMethod = GetValue<IUserDevicesManager::EVerificationMethod>(requestData, "verification_method", false).GetOrElse(GetHandlerSettingDef<IUserDevicesManager::EVerificationMethod>("verification_method", IUserDevicesManager::EVerificationMethod::Sms));
    auto session = BuildTx<NSQL::Writable>();
    auto userFetchResult = DriveApi->GetUsersData()->FetchInfo(permissions->GetUserId(), session);
    R_ENSURE(!userFetchResult.empty(), ConfigHttpStatus.UnknownErrorStatus, "no such user in table");
    auto p = userFetchResult.MutableResult().begin();
    auto user = std::move(p->second);

    if (!phone) {
        phone = user.GetPhone();
    }

    TDeviceIdVerificationContext divc;
    divc.ApplicationId = GetApplicationId();
    divc.ClientIp = GetClientIp();
    divc.EnableGpsPackageName = IRequestProcessor::IsAndroid(GetUserApp());
    TSet<TString> errorCodes;
    auto verificationResult = Server->GetUserDevicesManager()->StartDeviceIdVerification(permissions->GetUserId(), GetDeviceId(), phone, divc, Context, session, verificationMethod, errorCodes, GetNewDeviceStatus() != ENewDeviceStatus::Verified);
    if (verificationResult) {
        TMaybe<TObjectEvent<TUserDevice>> lastDeviceEvent;
        if (!Server->GetUserDevicesManager()->GetLastDeviceEvent({ permissions->GetUserId() }, {}, { EObjectHistoryAction::Proposition }, session, lastDeviceEvent) ||
            !AddTagOnDeviceChange(permissions->GetUserId(), *verificationResult, lastDeviceEvent, session)) {
            errorCodes.emplace("antifraud_failed");
            verificationResult = Nothing();
        }
    }
    if (!verificationResult) {
        TString message;
        TString title;
        TString primaryError;
        if (errorCodes.size()) {
            primaryError = *errorCodes.begin();
            message = Server->GetLocalization()->GetLocalString(locale, "verification.device.messages." + *errorCodes.begin(), message);
            title = Server->GetLocalization()->GetLocalString(locale, "verification.device.messages." + *errorCodes.begin() + ".title", title);
        } else {
            primaryError = "unknown_phone";
            message = Server->GetLocalization()->GetLocalString(locale, "verification.device.messages.unknown_phone");
        }
        Y_ENSURE_EX(
            false,
            TCodedException(ConfigHttpStatus.UserErrorState)
                .AddPublicInfo("error_codes", NJson::ToJson(errorCodes))
                .AddPublicInfo("error_message", message)
                .AddPublicInfo("error_primary", primaryError)
                .SetErrorCode("phone.bind.failed")
                .SetLocalizedMessage(message)
                .SetLocalizedTitle(title)
        );
    }

    if (!user.GetPhone() || !user.IsPhoneVerified()) {
        user.SetPhone(phone);
    }
    if (!DriveApi->GetUsersData()->UpdateUser(user, permissions->GetUserId(), session) || !session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }

    g.MutableReport().AddReportElement("seconds_to_retry", Server->GetUserDevicesManager()->GetMinPropositionsInterval().Seconds());
    g.SetCode(HTTP_OK);
}

void TUserPhoneBindCommitProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    Y_ENSURE_EX(Server->GetUserDevicesManager(), TCodedException(ConfigHttpStatus.ServiceUnavailable) << "incorrect server configuration");

    const TString& code = GetString(requestData, "code");
    const auto deadline = Context->GetRequestDeadline();
    const auto locale = GetLocale();

    auto session = BuildTx<NSQL::Writable>();
    auto userFetchResult = DriveApi->GetUsersData()->FetchInfo(permissions->GetUserId(), session);
    R_ENSURE(!userFetchResult.empty(), ConfigHttpStatus.UnknownErrorStatus, "no such user in table");
    auto p = userFetchResult.MutableResult().begin();
    auto user = std::move(p->second);

    TSet<TString> errorCodes;
    TString phone;
    if (!Server->GetUserDevicesManager()->FinishDeviceIdVerification(permissions->GetUserId(), GetDeviceId(), code, GetClientIp(), deadline, errorCodes, session, &phone, true)) {
        TString message;
        TString title;
        TString primaryError;
        if (errorCodes.size()) {
            primaryError = *errorCodes.begin();
            message = Server->GetLocalization()->GetLocalString(locale, "verification.device.messages." + *errorCodes.begin(), message);
            title = Server->GetLocalization()->GetLocalString(locale, "verification.device.messages." + *errorCodes.begin() + ".title", title);
        } else {
            primaryError = "unknown_code";
            message = Server->GetLocalization()->GetLocalString(locale, "verification.device.messages.unknown_code");
        }
        Y_ENSURE_EX(
            false,
            TCodedException(ConfigHttpStatus.UserErrorState)
                .AddPublicInfo("error_codes", NJson::ToJson(errorCodes))
                .AddPublicInfo("error_message", message)
                .AddPublicInfo("error_primary", primaryError)
                .SetErrorCode("phone.commit.failed")
                .SetLocalizedMessage(message)
                .SetLocalizedTitle(title)
        );
    } else {
        user.SetPhone(phone);
    }

    bool twinHadRide = false;
    TVector<TString> twinIds;
    if (user.GetPhone() && DriveApi->GetUsersData()->SelectUsers("phone", {user.GetPhone()}, twinIds, session)) {
        auto usersFetchResult = DriveApi->GetUsersData()->FetchInfo(twinIds, session);
        for (auto&& userIt : usersFetchResult) {
            if (!userIt.second.IsFirstRiding()) {
                twinHadRide = true;
            }
            if (!userIt.second.IsPhoneVerified() || userIt.first == user.GetUserId()) {
                continue;
            }
            auto userData = userIt.second;
            userData.SetPhoneVerified(false);
            R_ENSURE(DriveApi->GetUsersData()->UpdateUser(userData, permissions->GetUserId(), session), ConfigHttpStatus.UnknownErrorStatus, "can't drop twin phone verification status");
        }
    }

    if (twinHadRide) {
        user.SetFirstRiding(false);
    }
    user.SetPhoneVerified(true);
    if (!DriveApi->GetUsersData()->UpdateUser(user, permissions->GetUserId(), session) || !session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }

    g.SetCode(HTTP_OK);
}

class TPushSubscriptionCallback : public NNeh::THttpAsyncReport::ICallback {
public:
    TPushSubscriptionCallback(IServerReportBuilder::TPtr report, const TString& userId)
        : Report(report)
        , UserId(userId)
    {
    }

    void OnResponse(const std::deque<NNeh::THttpAsyncReport>& reports) override {
        NDrive::TEventLog::TUserIdGuard userIdGuard(UserId);
        TJsonReport::TGuard g(Report, HTTP_INTERNAL_SERVER_ERROR);

        ui32 code = 0;
        TVector<NJson::TJsonValue> rr;
        for (auto&& report : reports) {
            code = std::max(code, report.GetHttpCode());
            auto r = report.GetReport();
            if (r) {
                rr.push_back(NJson::ToJson(NJson::Stringify(*r)));
            } else {
                rr.push_back(NJson::JSON_NULL);
            }
        }

        if (rr.size() != 1) {
            g.SetExternalReport(NJson::ToJson(rr));
            g.SetCode(code ? code : HTTP_INTERNAL_SERVER_ERROR);
            NDrive::TEventLog::Log("PushSubscriptionCallbackError", NJson::TMapBuilder
                ("reports", NJson::ToJson(rr))
            );
        } else {
            g.SetExternalReport(std::move(rr.at(0)));
            g.SetCode(code);
        }
    }

private:
    IServerReportBuilder::TPtr Report;
    TString UserId;
};

void TUserPushSubscriptionProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    auto platform = GetString(requestData, "platform");
    auto apnToken = GetString(requestData, "apn_token");
    auto uuid = Context->GetRequestData().HeaderInOrEmpty("UUID");

    Y_ENSURE_EX(platform == "i" || platform == "a" || platform == "as" || platform == "hms", TCodedException(ConfigHttpStatus.UnknownErrorStatus)
        .SetErrorCode("push.subscription.platform.unknown")
    );

    TString appName;
    if (platform == "i") {
        appName = "ru.yandex.mobile.drive";
    } else if (platform == "a" || platform == "hms") {
        appName = "com.yandex.mobile.drive";
    } else {
        appName = "com.yandex.mobile.drive.service";
    }

    if (requestData.Has("app_name") && requestData["app_name"].IsString()) {
        appName = requestData["app_name"].GetString();
    }

    NDrive::INotifier::TPtr notifier = Server->GetNotifier("drive_push");

    auto* pushSender = dynamic_cast<const TPushNotifier*>(notifier.Get());
    R_ENSURE(pushSender, ConfigHttpStatus.UnknownErrorStatus, "no 'drive_push' or 'drive_push_service_app' notifier configured");

    auto callback = MakeHolder<TPushSubscriptionCallback>(g.GetReport(), permissions->GetUserId());
    g.Release();
    pushSender->SubscribeUser(appName, platform, permissions->GetUid(), TString{uuid}, apnToken, std::move(callback));
}

void TSwitchOfferProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    const TCgiParameters& cgi = Context->GetCgiParameters();
    TString offerId;
    if (!offerId) {
        offerId = GetString(requestData, "offer_id", /*required=*/false);
    }
    if (!offerId) {
        offerId = GetString(cgi, "offer_id", /*required=*/false);
    }

    TString externalTagName = GetString(requestData, "tag", /*required=*/false);
    if (externalTagName) {
        auto allowedTagNames = TChargableTag::GetTagNames(DriveApi->GetTagsManager().GetTagsMeta());
        R_ENSURE(
            std::find(allowedTagNames.begin(), allowedTagNames.end(), externalTagName) != allowedTagNames.end(),
            ConfigHttpStatus.UserErrorState,
            "unrecognized target tag name: " << externalTagName
        );
    }

    auto userSession = GetCurrentUserSession(permissions, Context->GetRequestStartTime());
    auto session = BuildTx<NSQL::Writable | NSQL::RepeatableRead>();
    if (!TUserProblemTag::EnsureNotBlocked(permissions->GetUserId(), *Server, session)) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    };

    IOffer::TPtr offerCopy;
    if (!!offerId) {
        auto asyncOffer = Server->GetOffersStorage()->RestoreOffer(offerId, permissions->GetUserId(), session);
        if (!asyncOffer.Initialized()) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
        R_ENSURE(asyncOffer.Wait(Context->GetRequestDeadline()), ConfigHttpStatus.TimeoutStatus, "RestoreOffer timeout");
        R_ENSURE(asyncOffer.GetValue(), ConfigHttpStatus.UnknownErrorStatus, "null offer restored for " << offerId);
        offerCopy = std::dynamic_pointer_cast<IOffer>(asyncOffer.GetValue());
        R_ENSURE(offerCopy, ConfigHttpStatus.UnknownErrorStatus, "no-vehicle offer restored for " << offerId);
    }
    TChargableTag::TSwitchOptions switchOptions;
    if (!offerCopy) {
        switchOptions.CopiedOfferId = ICommonOffer::CreateOfferId();
    }
    switchOptions.DeviceId = GetDeviceId();
    switchOptions.MultiRent = GetHandlerSettingDef("multi_rent", false);
    switchOptions.TargetTagName = externalTagName;
    bool switched = TChargableTag::Switch(userSession, offerCopy, permissions, *Server, session, switchOptions);
    if (!switched) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }
    if (!session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }
    g.AddReportElement("offer_id", offerCopy ? offerCopy->GetOfferId() : switchOptions.CopiedOfferId);
    g.SetCode(HTTP_OK);
}

void TReplaceCarProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /* requestData */) {
    const TCgiParameters& cgi = Context->GetCgiParameters();
    TString newOfferId = GetString(cgi, "offer_id", /*required=*/true);
    /* session_id in cgi */

    auto tx = BuildTx<NSQL::Writable>();
    if (!TUserProblemTag::EnsureNotBlocked(permissions->GetUserId(), *Server, tx)) {
        tx.DoExceptionOnFail(ConfigHttpStatus);
    }

    auto userSession = GetCurrentUserSession(permissions, Context->GetRequestStartTime());
    R_ENSURE(userSession, {}, "no current session");
    auto deviceTags = DriveApi->GetTagsManager().GetDeviceTags().RestoreObject(userSession->GetObjectId(), tx);
    R_ENSURE(deviceTags, {}, "cannot get device tags", tx);
    auto replacingTagRaw = deviceTags->GetTag(TReplaceCarTag::TypeName);
    R_ENSURE(replacingTagRaw, ConfigHttpStatus.PermissionDeniedStatus, "no replacing tag on car");
    auto replacingTag = replacingTagRaw->MutableTagAs<TReplaceCarTag>();
    R_ENSURE(replacingTag, ConfigHttpStatus.PermissionDeniedStatus, "replacing tag has wrong type");
    auto tmpOfferId = replacingTag->GetTmpOfferId();
    R_ENSURE(tmpOfferId, {}, "no offer id in replacing tag");

    auto newOffer = std::dynamic_pointer_cast<IOffer>(Server->GetOffersStorage()->RestoreOffer(newOfferId, permissions->GetUserId(), tx).GetValueSync());
    R_ENSURE(newOffer, {}, "new offer not found", tx);
    auto tmpOffer = std::dynamic_pointer_cast<IOffer>(Server->GetOffersStorage()->RestoreOffer(tmpOfferId, permissions->GetUserId(), tx).GetValueSync());
    R_ENSURE(newOffer, {}, "tmp offer not found", tx);

    TChargableTag::TSwitchOptions switchOptions;
    switchOptions.DeviceId = GetDeviceId();
    switchOptions.MultiRent = true;
    switchOptions.Replacing = true;
    bool switched = TChargableTag::Switch(userSession, tmpOffer, permissions, *Server, tx, switchOptions);
    R_ENSURE(switched, {}, "cannot switch", tx);

    newOffer->SetParentId(tmpOffer->GetOfferId());
    TChargableTag::TBookOptions bookOptions;
    bookOptions.DeviceId = GetDeviceId();
    bookOptions.MultiRent = true;
    auto booked = TChargableTag::Book(newOffer, *permissions, *Server, tx, bookOptions);
    R_ENSURE(booked, {}, "cannot book new offer", tx);

    deviceTags = DriveApi->GetTagsManager().GetDeviceTags().RestoreObject(userSession->GetObjectId(), tx);
    R_ENSURE(deviceTags, {}, "cannot get device tags (2)", tx);
    R_ENSURE(!deviceTags->GetTag(TReplaceCarTag::TypeName), {}, "replacing tag not removed by switch", tx);

    R_ENSURE(tx.Commit(), {}, "cannot commit tx", tx);

    g.SetCode(HTTP_OK);
}

void TVideoUploadProcessor::Process(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions) {
    const auto& cgi = Context->GetCgiParameters();
    const auto type = GetValue<NUserDocument::EType>(cgi, "type", true).GetRef();
    const auto bvData = TString{Context->GetBuf().AsStringBuf()};
    const auto& userId = permissions->GetUserId();

    auto typesMapping = Server->GetDriveAPI()->GetDocumentPhotosManager().GetUserPhotosDB().GetTypeToRecentPhotoMapping(userId, {type});
    if (typesMapping.size() != 1) {
        g.MutableReport().AddReportElement("status", "error");
        g.MutableReport().AddReportElement("error", "no photo of type " + ToString(type));
        g.SetCode(HTTP_BAD_REQUEST);
        return;
    }
    auto photoId = typesMapping[type].GetId();
    auto report = g.GetReport();

    const auto& videoDB = Server->GetDriveAPI()->GetDocumentPhotosManager().GetUserBackgroundVideosDB();

    {
        auto session = Server->GetDriveAPI()->BuildTx<NSQL::Writable>();
        auto videoFetchResult = videoDB.FetchInfo(photoId, session);
        if (videoFetchResult.empty()) {
            auto evlog = NDrive::GetThreadEventLogger();
            if (evlog) {
                evlog->AddEvent(NJson::TMapBuilder
                    ("error", "Missing bv template entry. Will create one")
                    ("photo_id", photoId)
                    ("type", type)
                    ("user_id", userId)
                    ("topic", "registrations")
                );
            }
            TUserDocumentVideo videoDBObject(photoId, type);
            if (!videoDB.Insert(videoDBObject, session) || !session.Commit()) {
                g.MutableReport().AddReportElement("status", "error");
                g.MutableReport().AddReportElement(
                    "error",
                    "failed to create bv template for " + photoId + ": " + session.GetStringReport()
                );
                g.SetCode(HTTP_INTERNAL_SERVER_ERROR);
                return;
            }
        }
    }


    auto videoUpdateFuture = Server->GetDriveAPI()->GetDocumentPhotosManager().UpdateBackgroundVideo(userId, photoId, bvData, *Server);
    videoUpdateFuture.Subscribe([report](const auto& result) {
        TJsonReport::TGuard g(report, HTTP_OK);
        TJsonReport& r = g.MutableReport();
        if (!result.HasValue() || result.HasException()) {
            r.AddReportElement("status", "error");
            r.AddReportElement("error", NThreading::GetExceptionInfo(result));
            g.SetCode(HttpCodes::HTTP_INTERNAL_SERVER_ERROR);
        } else {
            r.AddReportElement("status", "ok");
            g.SetCode(HTTP_OK);
        }
    });

    g.Release();
}

void SetErrorInfo(TJsonReport& r, TJsonReport::TGuard& g, NJson::TJsonValue error) {
    r.AddReportElement("status", "error");
    r.AddReportElement("error", std::move(error));
    g.SetCode(HttpCodes::HTTP_INTERNAL_SERVER_ERROR);
}

THolder<TSelfieVerificationResultTag> BuildSelfieResultTag(const TTaxiAntifraudClient::TFacesSimilarityInfo& facesSimilarityInfo, const TString& selfieResultTagName) {
    THolder<TSelfieVerificationResultTag> selfieVerificationResultTag = MakeHolder<TSelfieVerificationResultTag>();
    selfieVerificationResultTag->SetName(selfieResultTagName);
    selfieVerificationResultTag->SetFoundFaceOnSelfie(facesSimilarityInfo.GetFoundFaceOnFirstPhoto());
    selfieVerificationResultTag->SetFoundFaceOnPassport(facesSimilarityInfo.GetFoundFaceOnSecondPhoto());
    selfieVerificationResultTag->SetSimilarity(facesSimilarityInfo.GetSimilarity());
    return selfieVerificationResultTag;
}

bool TDocumentPhotoUploadProcessor::TSelfieUploadTagActions::OnSelfieVerified(const NDrive::IServer& server, NDrive::TEntitySession& session) const {
    if (RemoveSelfieVerificationTagNames.size()) {
        TVector<TDBTag> tagsToRemove;
        if (!server.GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ UserId }, MakeVector(RemoveSelfieVerificationTagNames), tagsToRemove, session)) {
            session.AddErrorMessage("TDocumentPhotoUploadProcessor::TSelfieUploadTagActions", "could not restore tags to remove");
            return false;
        }
        if (tagsToRemove.size() && !server.GetDriveAPI()->GetTagsManager().GetUserTags().RemoveTags(tagsToRemove, UserId, &server, session)) {
            session.AddErrorMessage("TDocumentPhotoUploadProcessor::TSelfieUploadTagActions", "could not remove tags");
            return false;
        }
    }

    if (EvolveSelfieVerificationTagNames.size()) {
        auto permissions = server.GetDriveAPI()->GetUserPermissions(UserId, TUserPermissionsFeatures());
        if (!permissions) {
            session.SetErrorInfo("TDocumentPhotoUploadProcessor::TSelfieUploadTagActions", "could not restore permissions");
            return false;
        }
        TVector<TString> tagsToRestore = MakeVector(NContainer::Keys(EvolveSelfieVerificationTagNames));
        TVector<TDBTag> tagsToEvolve;
        if (!server.GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ UserId }, tagsToRestore, tagsToEvolve, session)) {
            session.AddErrorMessage("TDocumentPhotoUploadProcessor::TSelfieUploadTagActions", "could not restore tags");
            return false;
        }

        for (auto&& tag : tagsToEvolve) {
            auto evolveNameIt = EvolveSelfieVerificationTagNames.find(tag->GetName());
            if (evolveNameIt != EvolveSelfieVerificationTagNames.end()) {
                auto evolveTag = server.GetDriveAPI()->GetTagsManager().GetTagsMeta().CreateTag(evolveNameIt->second);
                if (!evolveTag || !server.GetDriveAPI()->GetTagsManager().GetUserTags().EvolveTag(tag, evolveTag, *permissions, &server, session)) {
                    session.AddErrorMessage("TDocumentPhotoUploadProcessor::TSelfieUploadTagActions", "could not evolve tags");
                    return false;
                }
            }
        }
    }

    return true;
}

bool TDocumentPhotoUploadProcessor::TSelfieUploadTagActions::OnSelfieUploaded(const NDrive::IServer& server, NDrive::TEntitySession& session) const {
    if (AddOnUploadTagNames.size()) {
        TMessagesCollector errors;
        TVector<ITag::TPtr> tagPtrsToAdd;
        for (auto&& tagStr : AddOnUploadTagNames) {
            ITag::TPtr tagToAdd = IJsonSerializableTag::BuildFromString(server.GetDriveAPI()->GetTagsManager(), tagStr, &errors);
            if (tagToAdd) {
                tagPtrsToAdd.emplace_back(tagToAdd);
            } else {
                ERROR_LOG << "TDocumentPhotoUploadProcessor incorrect tag data " << tagStr << " errors: " << errors.GetStringReport() << Endl;
            }
        }
        if (tagPtrsToAdd.size() && !server.GetDriveAPI()->GetTagsManager().GetUserTags().AddTags(tagPtrsToAdd, UserId, UserId, &server, session)) {
            session.AddErrorMessage("TDocumentPhotoUploadProcessor::TSelfieUploadTagActions", "could not add tags");
            return false;
        }
    }

    if (UpdateEventTagNames.size()) {
        auto tagsToUpdate = server.GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags(TVector<TString>{ UserId }, UpdateEventTagNames, session);
        if (!tagsToUpdate) {
            session.AddErrorMessage("TDocumentPhotoUploadProcessor::TSelfieUploadTagActions", "could not restore tags to update");
            return false;
        }
        for (auto&& tag : *tagsToUpdate) {
            auto eventTag = tag.MutableTagAs<IEventTag>();
            if (eventTag && EventName) {
                eventTag->AddEvent(EventName, Now(), EventMeta);
            }
        }
        if (tagsToUpdate->size() && !server.GetDriveAPI()->GetTagsManager().GetUserTags().UpdateTagsData(*tagsToUpdate, UserId, session)) {
            session.AddErrorMessage("TDocumentPhotoUploadProcessor::TSelfieUploadTagActions", "could not update tags");
            return false;
        }
    }
    return true;
}

bool TDocumentPhotoUploadProcessor::TSelfieUploadTagActions::MergeTagActions(const TSelfieVerificationEvolutionPolicyAction& selfieAction) {
    auto evolveIt = EvolveSelfieVerificationTagNames.find(selfieAction.GetVerificationTagName());
    if (evolveIt != EvolveSelfieVerificationTagNames.end() && selfieAction.GetRecheckDocumentsTagName() != evolveIt->second) {
        return false;
    }
    if (!selfieAction.GetRecheckDocumentsTagName().empty() && RemoveSelfieVerificationTagNames.contains(selfieAction.GetVerificationTagName())) {
        return false;
    }

    if (selfieAction.GetRecheckDocumentsTagName()) {
        EvolveSelfieVerificationTagNames[selfieAction.GetVerificationTagName()] = selfieAction.GetRecheckDocumentsTagName();
    } else {
        RemoveSelfieVerificationTagNames.insert(selfieAction.GetVerificationTagName());
    }
    if (selfieAction.GetEventTagName()) {
        UpdateEventTagNames.push_back(selfieAction.GetEventTagName());
    }
    if (selfieAction.GetTagsToAddOnSubmit()) {
        std::copy(selfieAction.GetTagsToAddOnSubmit().begin(), selfieAction.GetTagsToAddOnSubmit().end(), std::back_inserter(AddOnUploadTagNames));
    }
    return true;
}

void TDocumentPhotoUploadProcessor::Process(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions) {
    const auto& cgi = Context->GetCgiParameters();
    const auto additionalLength = GetValue<size_t>(cgi, "additional_length", false).GetOrElse(0);
    const auto type = GetValue<NUserDocument::EType>(cgi, "type", true).GetRef();
    const auto data = TString{Context->GetBuf().AsStringBuf()};
    const auto& userId = permissions->GetUserId();

    TSelfieUploadTagActions selfieTagActions(userId);
    selfieTagActions.SetEventName(GetHandlerSettingDef<TString>("event_name", "user_document_photo_upload"));
    selfieTagActions.SetEventMeta(TUserDevice::GenerateDescriptionJson(Context));

    bool useAntifraud = false;
    bool waitForAntifraud = false;
    if (type == NUserDocument::EType::Selfie) {
        auto permissions = Server->GetDriveAPI()->GetUserPermissions(userId, TUserPermissionsFeatures());
        if (permissions) {
            for (auto&& action : permissions->GetActionsActual()) {
                if (!action->CheckRequest(Context)) {
                    continue;
                }
                auto vAction = action.GetAs<TSelfieVerificationEvolutionPolicyAction>();
                if (vAction) {
                    R_ENSURE(selfieTagActions.MergeTagActions(*vAction), ConfigHttpStatus.UnknownErrorStatus, "conflicting tags in selfie verification action");
                    useAntifraud |= vAction->GetUseAntifraud();
                    waitForAntifraud |= vAction->GetWaitForAntifraud();
                }
            }
        }
    }

    if (additionalLength > data.size()) {
        g.MutableReport().AddReportElement("status", "error");
        g.MutableReport().AddReportElement("error", "additional length is greater than payload size");
        g.SetCode(HTTP_BAD_REQUEST);
        return;
    }

    TString photoContent;
    TString videoContent;
    if (additionalLength) {
        photoContent = data.substr(0, data.Size() - additionalLength);
        videoContent = data.substr(data.Size() - additionalLength, additionalLength);
    } else {
        photoContent = data;
    }

    auto locale = GetLocale();
    auto report = g.GetReport();
    auto documentUploadFuture = photoContent
        ? Server->GetDriveAPI()->GetDocumentPhotosManager().AddDocumentPhoto(userId, "", type, photoContent, videoContent, *Server, false)
        : Server->GetDriveAPI()->GetDocumentPhotosManager().AddDocumentVideo(userId, "", type, videoContent, *Server, false);

    NThreading::TFuture<TTaxiAntifraudClient::TFacesSimilarityInfo> compareResultFuture;
    TString selfieResultTagName = GetHandlerSettingDef("antifraud_tag_name", TSelfieVerificationResultTag::TypeName);
    if (useAntifraud && type == NUserDocument::EType::Selfie) {
        R_ENSURE(Server->GetTaxiAntifraudClient(), ConfigHttpStatus.UnknownErrorStatus, "TaxiAntifraudClient is not configured");
        auto recentPassportSelfie = Server->GetDriveAPI()->GetDocumentPhotosManager().GetUserPhotosDB().GetTypeToRecentPhotoMapping(userId, { NUserDocument::EType::PassportSelfie });
        R_ENSURE(recentPassportSelfie.contains(NUserDocument::EType::PassportSelfie), ConfigHttpStatus.UnknownErrorStatus, "could not find passport selfie for user " + userId);
        TString passporSelfieContent;
        if (!Server->GetDriveAPI()->GetDocumentPhotosManager().GetDocumentPhoto(recentPassportSelfie[NUserDocument::EType::PassportSelfie].GetId(), passporSelfieContent, *Server) || passporSelfieContent.empty()) {
            g.MutableReport().AddReportElement("status", "error");
            g.MutableReport().AddReportElement("error", "could not get passport photo for user " + userId);
            g.SetCode(HTTP_BAD_REQUEST);
            return;
        }
        NThreading::TFuture<TTaxiAntifraudClient::TFacesSimilarityInfo> compareFuture = Server->GetTaxiAntifraudClient()->CalculateFacesSimilarity(userId, photoContent, passporSelfieContent);
        if (waitForAntifraud) {
            compareResultFuture = std::move(compareFuture);
        } else {
            auto eventLogState = NDrive::TEventLog::CaptureState();
            compareFuture.Subscribe([userId, server = Server, selfieResultTagName, eventLogState = std::move(eventLogState)](const auto& result) {
                NDrive::TEventLog::TStateGuard stateGuard(eventLogState);
                if (result.HasException() || !result.HasValue()) {
                    NDrive::TEventLog::Log("TDocumentPhotoUploadProcessor", NJson::TMapBuilder
                        ("user_id", userId)
                        ("status", "taxi antifraud request error")
                        ("error", NThreading::GetExceptionInfo(result))
                    );
                    return;
                }
                auto selfieVerificationTag = BuildSelfieResultTag(result.GetValue(), selfieResultTagName);
                auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
                if (!server->GetDriveAPI()->GetTagsManager().GetUserTags().AddTag(std::move(selfieVerificationTag), userId, userId, server, session) || !session.Commit()) {
                    NDrive::TEventLog::Log("TDocumentPhotoUploadProcessor", NJson::TMapBuilder
                        ("user_id", userId)
                        ("status", "could not add antifraud tag")
                        ("error", session.GetReport())
                    );
                }
            });
        }
    }

    TVector<NThreading::TFuture<void>> futures = { documentUploadFuture.IgnoreResult() };
    if (compareResultFuture.Initialized()) {
        futures.push_back(compareResultFuture.IgnoreResult());
    }
    auto eventLogState = NDrive::TEventLog::CaptureState();
    NThreading::WaitExceptionOrAll(futures).Subscribe(
        [
            docUploadFuture = std::move(documentUploadFuture),
            docCompareFuture = std::move(compareResultFuture),
            server = Server,
            selfieTagActions,
            userId,
            locale,
            selfieResultTagName,
            report,
            eventLogState
        ]
        (const auto& /*w*/) {
            NDrive::TEventLog::TStateGuard stateGuard(eventLogState);
            TJsonReport::TGuard g(report, HTTP_OK);
            TJsonReport& r = g.MutableReport();

            if (docCompareFuture.Initialized() && (docCompareFuture.HasException() || !docCompareFuture.HasValue())) {
                SetErrorInfo(r, g, NThreading::GetExceptionMessage(docCompareFuture));
                return;
            }
            if (docUploadFuture.HasException() || !docUploadFuture.HasValue()) {
                SetErrorInfo(r, g, NThreading::GetExceptionMessage(docUploadFuture));
                return;
            }

            auto addedDocuments = docUploadFuture.GetValue();
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            const auto& addedPhoto = addedDocuments.GetPhoto();
            if (addedPhoto && !server->GetDriveAPI()->GetDocumentPhotosManager().GetUserPhotosDB().Insert(addedPhoto, session)) {
                session.AddErrorMessage("TDocumentPhotoUploadProcessor::TCallback", "cannot Insert into PhotoDB");
                SetErrorInfo(r, g, session.GetStringReport());
                return;
            }
            const auto& addedVideo = addedDocuments.GetVideo();
            if (addedVideo && !server->GetDriveAPI()->GetDocumentPhotosManager().GetUserBackgroundVideosDB().Insert(addedVideo, session)) {
                session.AddErrorMessage("TDocumentPhotoUploadProcessor::TCallback", "cannot Insert into VideoDB");
                SetErrorInfo(r, g, session.GetStringReport());
                return;
            }
            if (!selfieTagActions.OnSelfieUploaded(*server, session)) {
                SetErrorInfo(r, g, session.GetStringReport());
                return;
            }

            bool entryAllowed = true;
            if (docCompareFuture.Initialized()) {
                THolder<TSelfieVerificationResultTag> selfieVerificationResultTag = BuildSelfieResultTag(docCompareFuture.GetValue(), selfieResultTagName);
                entryAllowed = selfieVerificationResultTag->IsEntryAllowed(*server);
                if (!server->GetDriveAPI()->GetTagsManager().GetUserTags().AddTag(std::move(selfieVerificationResultTag), userId, userId, server, session)) {
                    SetErrorInfo(r, g, session.GetStringReport());
                    return;
                }
            }

            if (entryAllowed && !selfieTagActions.OnSelfieVerified(*server, session)) {
                SetErrorInfo(r, g, session.GetStringReport());
                return;
            }

            if (!session.Commit()) {
                SetErrorInfo(r, g, session.GetStringReport());
                return;
            }

            if (!entryAllowed) {
                TString message = server->GetLocalization()->GetLocalString(locale, "selfie_upload.bad_photo");
                TString title = server->GetLocalization()->GetLocalString(locale, "selfie_upload.bad_photo.title");
                TCodedException ex(HTTP_BAD_REQUEST);
                ex.SetErrorCode("selfie_upload.bad_photo");
                ex.SetLocalizedMessage(message);
                ex.SetLocalizedTitle(title);
                g.SetCode(ex);
                return;
            }

            r.AddReportElement("status", "ok");
            g.SetCode(HTTP_OK);
        }
    );
    g.Release();
}

void TUserP2PDelegationProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    const TCgiParameters& cgi = Context->GetCgiParameters();
    auto phone = GetString(cgi, "phone", false);
    auto objectId = GetString(cgi, "object_id", true);
    auto locale = GetLocale();
    auto session = BuildTx<NSQL::ReadOnly>();

    TString targetUserId;
    TDriveUserData targetUser;
    if (phone) {
        if (phone.StartsWith("8")) {
            phone[0] = '7';
        }
        if (!phone.StartsWith("+")) {
            phone = "+" + phone;
        }

        TMaybe<TDriveUserData> targetUserMaybe;
        TDriveAPI::EDelegationAbility carDelegationPossibility;
        if (phone == permissions->GetPhone()) {
            carDelegationPossibility = TDriveAPI::EDelegationAbility::SameUser;
        } else {
            targetUserMaybe = Server->GetDriveAPI()->GetUsersData()->GetUserByPhone(phone, session);
            R_ENSURE(targetUserMaybe, {}, "cannot GetUserByPhone", session);
            targetUser = std::move(*targetUserMaybe);
            targetUserId = targetUser.GetUserId();
            carDelegationPossibility = targetUser ? Server->GetDriveAPI()->GetP2PAbility(targetUser, objectId, session) : TDriveAPI::EDelegationAbility::UserNotRegistered;
        }
        auto verdict = ToString(carDelegationPossibility);
        Y_ENSURE_EX(
            carDelegationPossibility == TDriveAPI::EDelegationAbility::Possible,
            TCodedException(ConfigHttpStatus.UserErrorState)
                .SetErrorCode(verdict)
                .SetLocalizedTitle(Server->GetLocalization()->GetLocalString(locale, "delegation.p2p.errors.delegator.title." + verdict))
                .SetLocalizedMessage(Server->GetLocalization()->GetLocalString(locale, "delegation.p2p.errors.delegator.message." + verdict))
        );
        g.MutableReport().AddReportElement("name", targetUser.GetShortName());
    }

    NJson::TJsonValue typesReport = NJson::JSON_ARRAY;

    {
        auto bSessions = DriveApi->GetTagsManager().GetDeviceTags().GetHistoryManager().GetSessionsBuilder("billing", TInstant::Zero());
        R_ENSURE(bSessions, ConfigHttpStatus.UnknownErrorStatus, "no sessions builder");
        auto objectSession = bSessions->GetLastObjectSession(objectId);
        R_ENSURE(objectSession, ConfigHttpStatus.UnknownErrorStatus, "no object session");
        const TBillingSession* bSession = dynamic_cast<const TBillingSession*>(objectSession.Get());
        if (bSession && !bSession->GetClosed() && !!bSession->GetCurrentOffer()) {
            auto currentOffer = bSession->GetCurrentOffer();
            if (!!currentOffer) {
                auto billingCompilation = bSession->GetCompilationAs<TBillingSession::TBillingCompilation>();
                R_ENSURE(billingCompilation, ConfigHttpStatus.UnknownErrorStatus, "no billing compilation");
                auto packOfferState = billingCompilation->GetCurrentOfferStateAs<TPackOfferState>();
                bool packOfferTraits = currentOffer->GetTypeName() == TPackOffer::GetTypeNameStatic() || currentOffer->GetTypeName() == TFlexiblePackOffer::GetTypeNameStatic();
                auto passOfferAbility = targetUserId ? Server->GetDriveAPI()->GetP2PAbility(targetUser, objectId, session, true) : TDriveAPI::EDelegationAbility::Possible;
                if (currentOffer->GetTransferable()) {
                    if (packOfferState && packOfferTraits) {
                        auto remainingDistance = packOfferState->GetRemainingDistance();
                        auto remainingDuration = packOfferState->GetPackRemainingTime();
                        auto offerName = currentOffer->GetName();
                        { // TODO: should block this too if P2PAbility is false?
                            NJson::TJsonValue p2pReport;
                            p2pReport["type"] = ToString(ECarDelegationType::P2P);
                            p2pReport["name"] = Server->GetLocalization()->GetLocalString(locale, "delegation.p2p.pack_no_offer.title", "Не передавать тариф");
                            p2pReport["details"] = Server->GetLocalization()->FormatDelegationBodyWithoutPackOffer(locale, remainingDistance, remainingDuration);
                            typesReport.AppendValue(std::move(p2pReport));
                        }
                        if (passOfferAbility == TDriveAPI::EDelegationAbility::Possible) {
                            NJson::TJsonValue p2pReport;
                            p2pReport["type"] = ToString(ECarDelegationType::P2PPassOffer);
                            p2pReport["name"] = Server->GetLocalization()->GetLocalString(locale, "delegation.p2p.pack_with_offer.title", "Передать вместе с тарифом");
                            p2pReport["details"] = Server->GetLocalization()->FormatDelegationBodyWithPackOffer(locale, offerName, remainingDistance, remainingDuration);
                            typesReport.AppendValue(std::move(p2pReport));
                        }
                    } else {
                        NJson::TJsonValue p2pReport;
                        p2pReport["type"] = ToString(ECarDelegationType::P2P);
                        p2pReport["name"] = Server->GetLocalization()->GetLocalString(locale, "delegation.p2p.title", "Передать без сохранения тарифа");
                        p2pReport["details"] = Server->GetLocalization()->GetLocalString(locale, "delegation.p2p.body", "Неиспользованные минуты и километры сгорят");
                        typesReport.AppendValue(std::move(p2pReport));
                    }
                }
            } else {
                Y_ENSURE_EX(
                    false,
                    TCodedException(ConfigHttpStatus.UserErrorState)
                        .SetErrorCode("no_offer")
                        .SetLocalizedTitle(Server->GetLocalization()->GetLocalString(locale, "delegation.p2p.errors.delegator.title.unknown_error"))
                        .SetLocalizedMessage(Server->GetLocalization()->GetLocalString(locale, "delegation.p2p.errors.delegator.message.unknown_error"))
                );
            }
        }
    }

    g.MutableReport().AddReportElement("types", std::move(typesReport));
    g.MutableReport().AddReportElement("can_delegate", true);
    g.MutableReport().AddReportElement("ui_message", NDrive::TLocalization::DelegationPossible());
    g.MutableReport().AddReportElement("id", targetUserId);
    g.SetCode(HTTP_OK);
}

void TUserP2PDelegationRejectProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    const TCgiParameters& cgi = Context->GetCgiParameters();
    auto tagId = GetString(cgi, "tag_id", false);
    auto carTagId = GetString(cgi, "car_tag_id", false);
    auto byDelegator = GetValue<bool>(cgi, "by_delegator", false).GetOrElse(false);

    auto session = BuildTx<NSQL::Writable>();
    TVector<TDBTag> tags;

    if (byDelegator) {
        R_ENSURE(tagId, ConfigHttpStatus.SyntaxErrorStatus, "tag_id not specified");
        R_ENSURE(
            Server->GetDriveAPI()->GetTagsManager().GetDeviceTags().RestoreTags(tagId, tags, session),
            ConfigHttpStatus.UnknownErrorStatus,
            "could not restore tags",
            session
        );
        R_ENSURE(tags.size() == 1, ConfigHttpStatus.SyntaxErrorStatus, "there is no such tag");

        auto carDelegationTag = tags.front();
        auto carDelegationTagImpl = carDelegationTag.GetTagAs<TP2PDelegationTag>();
        R_ENSURE(carDelegationTagImpl, ConfigHttpStatus.SyntaxErrorStatus, "this is not p2p delegation tag");

        R_ENSURE(
            Server->GetDriveAPI()->GetTagsManager().GetDeviceTags().RemoveTag(carDelegationTag, permissions->GetUserId(), Server, session),
            ConfigHttpStatus.UnknownErrorStatus,
            "could not update tag data",
            session
        );
    } else {
        R_ENSURE(!tagId.empty() ^ !carTagId.empty(), ConfigHttpStatus.SyntaxErrorStatus, "exactly one of {tag_id, car_tag_id} should be specified");
        TDBTag carTag;

        if (tagId) {
            R_ENSURE(Server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags(tagId, tags, session), ConfigHttpStatus.UnknownErrorStatus, "could not restore tags", session);
            R_ENSURE(tags.size() == 1, ConfigHttpStatus.SyntaxErrorStatus, "there is no such tag");

            auto incomingTag = tags.front();
            auto incomingTagImpl = incomingTag.GetTagAs<TIncomingDelegationUserTag>();
            R_ENSURE(incomingTagImpl, ConfigHttpStatus.SyntaxErrorStatus, "this is not incoming delegation tag");

            auto carId = incomingTagImpl->GetObjectId();
            TVector<TDBTag> carTags;
            R_ENSURE(Server->GetDriveAPI()->GetTagsManager().GetDeviceTags().RestoreTags({carId}, {TP2PDelegationTag::TypeName}, carTags, session), ConfigHttpStatus.UnknownErrorStatus, "no p2p tag on car", session);
            R_ENSURE(carTags.size() == 1, ConfigHttpStatus.UnknownErrorStatus, "there is no delegation tag on car");
            carTag = carTags.front();

            R_ENSURE(
                Server->GetDriveAPI()->GetTagsManager().GetUserTags().RemoveTag(incomingTag, permissions->GetUserId(), Server, session),
                ConfigHttpStatus.UnknownErrorStatus,
                "could not remove incoming delegation tag",
                session
            );
        } else {
            R_ENSURE(Server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({carTagId}, tags, session), ConfigHttpStatus.UnknownErrorStatus, "could not restore tags", session);
            R_ENSURE(tags.size() == 1, ConfigHttpStatus.SyntaxErrorStatus, "there is no such tag");
            carTag = tags.front();
        }

        auto carTagImpl = carTag.MutableTagAs<TP2PDelegationTag>();
        R_ENSURE(carTagImpl, ConfigHttpStatus.SyntaxErrorStatus, "car doesn't have p2p tag");
        R_ENSURE(!carTagImpl->GetP2PUserId() || carTagImpl->GetP2PUserId() == permissions->GetUserId(), ConfigHttpStatus.PermissionDeniedStatus, "you're not the target user of this delegation");

        carTagImpl->SetRejected(true);
        R_ENSURE(
            Server->GetDriveAPI()->GetTagsManager().GetDeviceTags().UpdateTagData(carTag, permissions->GetUserId(), session),
            ConfigHttpStatus.UnknownErrorStatus,
            "could not update tag data",
            session
        );
    }

    R_ENSURE(session.Commit(), ConfigHttpStatus.UnknownErrorStatus, "could not commit session", session);

    g.MutableReport().AddReportElement("status", "ok");
    g.SetCode(HTTP_OK);
}

TString TUserProfileProcessor::GetTypeName() {
    return "user_profile";
}

void TUserProfileProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    const TCgiParameters& cgi = Context->GetCgiParameters();
    ELocalization locale = GetLocale();
    TString userId = GetString(cgi, "user_id", false);
    TUserPermissions::TPtr userPermissions;
    if (userId) {
        auto eg = g.BuildEventGuard("user_id_permissions");
        ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::User, userId);
        userPermissions = DriveApi->GetUserPermissions(userId);
    } else {
        userId = permissions->GetUserId();
        userPermissions = permissions;
    }
    ReqCheckCondition(!!userId, ConfigHttpStatus.EmptyRequestStatus, EDriveLocalizationCodes::UserNotFound);
    ReqCheckCondition(!!userPermissions, ConfigHttpStatus.EmptyRequestStatus, EDriveLocalizationCodes::NoPermissions);
    NDriveSession::TReportTraits reportTraits = 0;
    TUserCurrentContext userCurrentContext(Server, userPermissions, reportTraits, GetOrigin(), GetUserLocation());
    auto session = BuildTx<NSQL::ReadOnly>();
    if (!userCurrentContext.Initialize(session, g.MutableReport())) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }
    g.MutableReport().AddReportElement("driving_style", userCurrentContext.GetAggressionReport(locale));
    {
        Y_ENSURE(Server);
        const auto& tagsManager = Server->GetDriveAPI()->GetTagsManager();
        std::map<TString, TAchievementUserTag> tags;
        for (auto&& dbTag : userCurrentContext.GetAchievementTags()) {
            const auto& tag = *Yensured(dbTag.GetTagAs<TAchievementUserTag>());
            tags[tag.GetName()] = tag;
        }
        TVector<TAchievementUserTag> achievementTags;
        for (auto&& name : userCurrentContext.GetAchievementTagNames()) {
            if (auto it = tags.find(name); it != tags.end()) {
                auto&& tag = it->second;
                if (tag.IsEnabled(*Server, permissions)) {
                    achievementTags.push_back(tag);
                }
                continue;
            }
            auto description = tagsManager.GetTagsMeta().GetDescriptionByName(name);
            if (!description) {
                continue;
            }
            auto tagDescription = std::dynamic_pointer_cast<const TAchievementUserTag::TDescription>(description);
            if (!tagDescription) {
                continue;
            }
            // Construct empty tag.
            TAchievementUserTag tag(name);
            if (tag.IsEmptyEnabled(*Server, permissions)) {
                achievementTags.push_back(tag);
            }
        }
        TAchievementUserTag::Sort(achievementTags, *Server, permissions);
        NJson::TJsonValue achievements = NJson::JSON_ARRAY;
        for (auto&& tag : achievementTags) {
            achievements.AppendValue(tag.GetPublicReport(*Server, locale));
        }
        g.MutableReport().AddReportElement("achievements", std::move(achievements));
    }
    g.SetCode(HTTP_OK);
}
