#pragma once

#include <drive/backend/processors/common_app/config.h>
#include <drive/backend/processors/common_app/data_check.h>
#include <drive/backend/processors/common_app/processor.h>
#include <drive/backend/processors/sessions/base.h>

#include <drive/backend/data/chargable.h>
#include <drive/backend/data/user_tags.h>
#include <drive/backend/offers/offers/abstract.h>
#include <drive/backend/users/user_documents.h>
#include <drive/backend/user_document_photos/manager.h>

class TEvolveConfig {
    R_READONLY(double, TaskTimeoutFraction, 0.8);
    R_READONLY(TDuration, DefaultTimeout, TDuration::Seconds(60));
    R_READONLY(double, KfTimeoutRollbackByRobot, 2);
public:
    void InitFeatures(const TYandexConfig::Section* section) {
        TaskTimeoutFraction = section->GetDirectives().Value("TaskTimeoutFraction", TaskTimeoutFraction);
        AssertCorrectConfig(TaskTimeoutFraction <= 1 && TaskTimeoutFraction > 0, "Incorrect TaskTimeoutFraction for evolve config");

        DefaultTimeout = section->GetDirectives().Value("DefaultTimeout", DefaultTimeout);
        KfTimeoutRollbackByRobot = section->GetDirectives().Value("KfTimeoutRollbackByRobot", KfTimeoutRollbackByRobot);
    }

    void ToStringFeatures(IOutputStream& os) const {
        os << "TaskTimeoutFraction: " << TaskTimeoutFraction << Endl;

        os << "DefaultTimeout: " << DefaultTimeout << Endl;
        os << "KfTimeoutRollbackByRobot: " << KfTimeoutRollbackByRobot << Endl;
    }
};

class TCarControlUserProcessor: public TAppCommonProcessor<TCarControlUserProcessor, TEvolveConfig> {
private:
    using TBase = TAppCommonProcessor<TCarControlUserProcessor, TEvolveConfig>;

public:
    using TBase::TBase;

    static TString GetTypeName() {
        return "car_user_control";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TSwitchOfferProcessor: public TAppCommonProcessor<TSwitchOfferProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TSwitchOfferProcessor, TEmptyConfig>;

public:
    using TBase::TBase;

    static TString GetTypeName() {
        return "switch_offer";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TCarControlRootProcessor: public TAppCommonProcessor<TCarControlRootProcessor, TEvolveConfig> {
private:
    using TBase = TAppCommonProcessor<TCarControlRootProcessor, TEvolveConfig>;

public:
    using TBase::TBase;

    static TString GetTypeName() {
        return "car_root_control";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TTagEvolveProcessor: public TAppCommonProcessor<TTagEvolveProcessor, TEvolveConfig> {
private:
    using TBase = TAppCommonProcessor<TTagEvolveProcessor, TEvolveConfig>;

public:
    using TBase::TBase;

    static TString GetTypeName() {
        return "tag_evolve";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

void ProcessEvolveTag(
    IRequestProcessor::TPtr self,
    TJsonReport::TGuard& g,
    NDrive::TEntitySession& session,
    TDBTag tagCurrent,
    ITag::TPtr newTag,
    TUserPermissions::TPtr permissions,
    ELocalization locale,
    bool dryRun,
    const NJson::TJsonValue& requestData,
    TDuration taskTimeout,
    TDuration rollbackTimeout,
    EEvolutionMode eMode,
    const IEntityTagsManager* tagsManager,
    const THttpStatusManagerConfig& ConfigHttpStatus,
    const NDrive::IServer* Server
);

class TCurrentSessionProcessor: public TAppCommonProcessor<TCurrentSessionProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TCurrentSessionProcessor, TEmptyConfig>;

public:
    using TBase::TBase;

    static TString GetTypeName() {
        return "current_session";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TDropSessionProcessor: public TAppCommonProcessor<TDropSessionProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TDropSessionProcessor, TEmptyConfig>;

public:
    using TBase::TBase;

    static TString GetTypeName() {
        return "drop_session";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TSharedSessionBookProcessor: public TAppCommonProcessor<TSharedSessionBookProcessor> {
private:
    using TBase = TAppCommonProcessor<TSharedSessionBookProcessor>;

public:
    using TBase::TBase;

    static TString GetTypeName() {
        return "shared_session_book";
    }

    void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TSharedSessionInviteProcessor: public TAppCommonProcessor<TSharedSessionInviteProcessor> {
private:
    using TBase = TAppCommonProcessor<TSharedSessionInviteProcessor>;

public:
    using TBase::TBase;

    static TString GetTypeName() {
        return "shared_session_invite";
    }

    void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TSessionAgreementProcessor: public TAppCommonProcessor<TSessionAgreementProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TSessionAgreementProcessor, TEmptyConfig>;

public:
    using TBase::TBase;

    static TString GetTypeName() {
        return "session_agreement";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TSessionsHistoryProcessorConfig {
    R_READONLY(TSet<TString>, FetchedGeoTags, {});

public:
    void InitFeatures(const TYandexConfig::Section* section) {
        TString fetchedGeoTags = section->GetDirectives().Value("FetchedGeoTags", TString());
        StringSplitter(fetchedGeoTags).Split(',').SkipEmpty().Collect(&FetchedGeoTags);
    }

    void ToStringFeatures(IOutputStream& os) const {
        os << "FetchedGeoTags: " << JoinSeq(",", FetchedGeoTags) << Endl;
    }
};

class TSessionsHistoryProcessor
    : public TAppCommonProcessor<TSessionsHistoryProcessor, TSessionsHistoryProcessorConfig>
    , public TSessionProcessorTraits<TSessionsHistoryProcessor>
{
private:
    using TBase = TAppCommonProcessor<TSessionsHistoryProcessor, TSessionsHistoryProcessorConfig>;

public:
    TSessionsHistoryProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server)
        , TSessionProcessorTraits<TSessionsHistoryProcessor>(context, server)
    {
    }

    static TString GetTypeName() {
        return "sessions_history";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;

private:
    NDrive::TLocationTags GetFetchedGeoTags(const TGeoCoord& coordinate) const;
};

class TClientHistoryProcessor
    : public TAppCommonProcessor<TClientHistoryProcessor, TEmptyConfig>
    , public TSessionProcessorTraits<TClientHistoryProcessor>
{
private:
    using TBase = TAppCommonProcessor<TClientHistoryProcessor, TEmptyConfig>;

public:
    TClientHistoryProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server)
        , TSessionProcessorTraits<TClientHistoryProcessor>(context, server)
    {
    }

    static TString GetTypeName() {
        return "client_history";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TSessionsProcessor: public TAppCommonProcessor<TSessionsProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TSessionsProcessor, TEmptyConfig>;

public:
    using TBase::TBase;

    static TString GetTypeName() {
        return "sessions";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TReplaceCarPrepareProcessor: public TAppCommonProcessor<TReplaceCarPrepareProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TReplaceCarPrepareProcessor, TEmptyConfig>;

public:
    using TBase::TBase;

    static TString GetTypeName() {
        return "replace_car_prepare";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TReplaceCarProcessor: public TAppCommonProcessor<TReplaceCarProcessor> {
private:
    using TBase = TAppCommonProcessor<TReplaceCarProcessor>;

public:
    using TBase::TBase;

    static TString GetTypeName() {
        return "replace_car";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TBookProcessor: public TAppCommonProcessor<TBookProcessor> {
private:
    using TBase = TAppCommonProcessor<TBookProcessor>;

public:
    using TBase::TBase;

    static TString GetTypeName() {
        return "book";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TOfferBookConfig {
private:
    R_READONLY(TSet<TString>, OfferNames);
    R_READONLY(TString, OfferPrefix);

public:
    bool AcceptAction(const TString& name) const {
        if (OfferNames.contains(name)) {
            return true;
        }
        if (OfferPrefix) {
            return name.StartsWith(OfferPrefix);
        }
        return OfferNames.empty() && OfferPrefix.empty();
    }

    void InitFeatures(const TYandexConfig::Section* section) {
        const TString& offerNames = section->GetDirectives().Value("OfferNames", Default<TString>());
        StringSplitter(offerNames).SplitBySet(", ").SkipEmpty().Collect(&OfferNames);
        const TString& offerName = section->GetDirectives().Value("OfferName", Default<TString>());
        if (!!offerName) {
            OfferNames.emplace(offerName);
        }
        OfferPrefix = section->GetDirectives().Value("OfferPrefix", OfferPrefix);
    }

    void ToStringFeatures(IOutputStream& os) const {
        os << "OfferNames: " << JoinSeq(", ", OfferNames) << Endl;
        os << "OfferPrefix: " << OfferPrefix << Endl;
    }
};

class TOfferBookProcessor: public TAppCommonProcessor<TOfferBookProcessor, TOfferBookConfig> {
private:
    using TBase = TAppCommonProcessor<TOfferBookProcessor, TOfferBookConfig>;

public:
    using TBase::TBase;

    static TString GetTypeName() {
        return "offer_book";
    }

    virtual bool AddCustomActionsToPermissions() const override {
        return false;
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TCarAdminPortalInfoProcessor: public TAppCommonProcessor<TCarAdminPortalInfoProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TCarAdminPortalInfoProcessor, TEmptyConfig>;

public:
    using TBase::TBase;

    static TString GetTypeName() {
        return "car_admin_portal_info";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TCarWebInfoProcessor: public TAppCommonProcessor<TCarWebInfoProcessor> {
private:
    using TBase = TAppCommonProcessor<TCarWebInfoProcessor>;

public:
    using TBase::TBase;

    static TString GetTypeName() {
        return "car_web_info";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TUserEmailBindProcessor : public TAppCommonProcessor<TUserEmailBindProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TUserEmailBindProcessor, TEmptyConfig>;

public:
    using TBase::TBase;

    static TString GetTypeName() {
        return "user_email_bind";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TSendVerificationEmailProcessor : public TAppCommonProcessor<TSendVerificationEmailProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TSendVerificationEmailProcessor, TEmptyConfig>;

public:
    using TBase::TBase;

    static TString GetTypeName() {
        return "send_verification_email";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TUserEmailConfirmProcessor : public TAppCommonProcessor<TUserEmailConfirmProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TUserEmailConfirmProcessor, TEmptyConfig>;

public:
    using TBase::TBase;

    static TString GetTypeName() {
        return "user_email_confirm";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TUserPhoneBindSubmitProcessor : public TDeviceChangeCheckProcessor<TUserPhoneBindSubmitProcessor, TEmptyConfig> {
private:
    using TBase = TDeviceChangeCheckProcessor<TUserPhoneBindSubmitProcessor, TEmptyConfig>;

public:
    using TBase::TBase;

    static TString GetTypeName() {
        return "user_phone_bind_submit";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TUserPhoneBindCommitProcessor : public TAppCommonProcessor<TUserPhoneBindCommitProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TUserPhoneBindCommitProcessor, TEmptyConfig>;

public:
    using TBase::TBase;

    static TString GetTypeName() {
        return "user_phone_bind_commit";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TUserPushSubscriptionProcessor : public TAppCommonProcessor<TUserPushSubscriptionProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TUserPushSubscriptionProcessor, TEmptyConfig>;

public:
    using TBase::TBase;

    static TString GetTypeName() {
        return "user_push_subscription";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TSelfieVerificationEvolutionPolicyAction;

class TDocumentPhotoUploadProcessor : public TAppCommonProcessor<TDocumentPhotoUploadProcessor, TEmptyConfig, TBaseProcessor> {
private:
    using TBase = TAppCommonProcessor<TDocumentPhotoUploadProcessor, TEmptyConfig, TBaseProcessor>;

public:
    class TSelfieUploadTagActions {
    public:
        using TStringMap = TMap<TString, TString>;

    private:
        R_FIELD(TSet<TString>, RemoveSelfieVerificationTagNames);
        R_FIELD(TStringMap, EvolveSelfieVerificationTagNames);
        R_FIELD(TVector<TString>, UpdateEventTagNames);
        R_FIELD(TVector<TString>, AddOnUploadTagNames);
        R_FIELD(TString, EventName);
        R_FIELD(NJson::TJsonValue, EventMeta);
        R_FIELD(TString, UserId);

    public:
        TSelfieUploadTagActions(const TString& userId)
           : UserId(userId)
        {
        }
        bool OnSelfieVerified(const NDrive::IServer& server, NDrive::TEntitySession& session) const;
        bool OnSelfieUploaded(const NDrive::IServer& server, NDrive::TEntitySession& session) const;
        bool MergeTagActions(const TSelfieVerificationEvolutionPolicyAction& selfieAction);
    };

public:
    using TBase::TBase;

    static TString GetTypeName() {
        return "user_document_photo_upload";
    }

protected:
    void Process(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions) override;
};

class TVideoUploadProcessor : public TAppCommonProcessor<TVideoUploadProcessor, TEmptyConfig, TBaseProcessor> {
private:
    using TBase = TAppCommonProcessor<TVideoUploadProcessor, TEmptyConfig, TBaseProcessor>;

public:
    using TBase::TBase;

    static TString GetTypeName() {
        return "user_background_video";
    }

protected:
    void Process(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions) override;
};

class TUserP2PDelegationProcessor : public TAppCommonProcessor<TUserP2PDelegationProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TUserP2PDelegationProcessor, TEmptyConfig>;

public:
    using TBase::TBase;

    static TString GetTypeName() {
        return "user_p2p_delegation";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TUserP2PDelegationRejectProcessor : public TAppCommonProcessor<TUserP2PDelegationRejectProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TUserP2PDelegationRejectProcessor, TEmptyConfig>;

public:
    using TBase::TBase;

    static TString GetTypeName() {
        return "user_p2p_delegation_reject";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

// TUserProfileProcessor represents processor for driving style profile.
class TUserProfileProcessor: public TAppCommonProcessor<TUserProfileProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TUserProfileProcessor, TEmptyConfig>;

public:
    using TBase::TBase;

    static TString GetTypeName();

    void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};
