#include "sessions_context.h"

#include <drive/backend/actions/session_photo_screen.h>

#include <drive/backend/data/chargable.h>
#include <drive/backend/data/delegation.h>
#include <drive/backend/data/dictionary_tags.h>
#include <drive/backend/data/feedback.h>
#include <drive/backend/data/model.h>
#include <drive/backend/data/offer.h>
#include <drive/backend/data/sharing.h>
#include <drive/backend/data/state.h>
#include <drive/backend/data/transformation.h>
#include <drive/backend/history_iterator/history_iterator.h>
#include <drive/backend/images/database.h>
#include <drive/backend/offers/actions/additional_service.h>
#include <drive/backend/offers/actions/fix_point.h>
#include <drive/backend/offers/actions/long_term.h>
#include <drive/backend/sessions/manager/additional_service.h>
#include <drive/backend/sessions/manager/billing.h>

#include <rtline/util/algorithm/ptr.h>

bool TUserSessionsContext::TSessionContext::GetCurrentMultiBill(TMultiBill& mBill) const {
    THistoryRidesContext context(*Server, TInstant::Zero());
    auto tx = Server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
    auto ydbTx = Server->GetDriveAPI()->BuildYdbTx<NSQL::ReadOnly>("user_sessions_context", Server);
    if (!context.InitializeSession(Session->GetSessionId(), tx, ydbTx)) {
        return false;
    }
    TVector<THistoryRideObject> sessions = context.GetSessions(TInstant::Max(), 1000);
    THistoryRideObject::FetchFullRiding(Server, sessions);

    for (auto&& i : sessions) {
        auto fullCompiledRiding = i.GetFullCompiledRiding(tx);
        if (!fullCompiledRiding) {
            continue;
        }
        if (fullCompiledRiding->HasBill()) {
            auto b = fullCompiledRiding->GetBillWithTariffRecord();
            mBill.AddBill(std::move(b));
        }
    }
    return true;
}

const TBillingSession::TBillingCompilation* TUserSessionsContext::TSessionContext::GetBillingCompilation() const {
    if (!BillingCompilation) {
        BillingCompilation = GetEventsSession()->GetCompilationAs<TBillingSession::TBillingCompilation>();
    }
    return BillingCompilation.Get();
}

IOffer::TPtr TUserSessionsContext::TSessionContext::GetOffer() const {
    auto compilation = GetBillingCompilation();
    if (!compilation) {
        return {};
    }
    return compilation->GetCurrentOffer();
}

void TUserSessionsContext::TSessionContext::BuildCarReport(NJson::TJsonWriter& carReport, const bool closed, const TString& objectId, const bool fuelingAbility, const bool tankerFuelingAbility, const TCarsFetcher& fetcher, TUserPermissions::TPtr permissions, const TSet<TString>& supportAllowedPhotos) {
    carReport.OpenMap();
    if (!fetcher.BuildOneCarInfoReport(objectId, permissions->GetFilterActions(), carReport, supportAllowedPhotos)) {
        carReport.Write("corrupted", true);
        carReport.Write("object_id", objectId);
    }
    if (!closed) {
        auto it = fetcher.GetObservableTagsData().find(objectId);
        if (it != fetcher.GetObservableTagsData().end()) {
            carReport.Write("fueling_ability", fuelingAbility);
            carReport.Write("tanker_fueling_ability", tankerFuelingAbility);
            carReport.OpenArray("session_features");
            for (auto&& tag : it->second) {
                const TDeviceAdditionalFeature* tagAdditionalFeature = dynamic_cast<const TDeviceAdditionalFeature*>(tag->GetData().Get());
                if (tagAdditionalFeature) {
                    carReport.Write(tagAdditionalFeature->SerializeToJson());
                }
            }
            carReport.CloseArray();
        }
    }
    carReport.CloseMap();
}

bool TUserSessionsContext::TSessionContext::Initialize(NDrive::TEntitySession& session, const TCarsFetcher& fetcher, const TMap<TString, TDBTag>& delegationCarTags, NFavouriteAddressAdvisor::IAdvisor::TPtr favouriteAddressAdvisorPtr, bool fetchPhotos) {
    if (!Session->Compile()) {
        session.SetErrorInfo("SessionContext::Initialize", "cannot compile session");
        return false;
    }
    if (!Server->GetDriveAPI()->GetTagsManager().GetTraceTags().RestoreEntityTags(Session->GetSessionId(), {}, SessionTags, session)) {
        return false;
    }
    const TString& objectId = Session->GetObjectId();
    if (objectId) {
        TagsSnapshot = Server->GetDriveAPI()->GetTagsManager().GetDeviceTags().RestoreObject(objectId, session);
    }
    if (TagsSnapshot) {
        ModelTag = TagsSnapshot->GetFirstTagByClass<TModelTag>();
        ServicingEnabled = TagsSnapshot->GetTag("enable_servicing").Defined();
    } else if (objectId) {
        return false;
    }
    auto delegationIt = delegationCarTags.find(Session->GetObjectId());
    if (delegationIt != delegationCarTags.end()) {
        DelegationTag = delegationIt->second;
    }

    auto eventSession = GetEventsSession();
    if (!BillingCompilation) {
        BillingCompilation = eventSession->GetCompilationAs<TBillingSession::TBillingCompilation>();
    }
    if (BillingCompilation) {
        auto offer = BillingCompilation->GetCurrentOffer();
        if (offer && offer->GetSharedSessionId()) {
            auto optionalSharingSession = Server->GetDriveDatabase().GetSessionManager().GetSession(offer->GetSharedSessionId(), session);
            if (!optionalSharingSession) {
                return false;
            }
            auto sharedSession = *optionalSharingSession;
            auto sharedSessionImpl = std::dynamic_pointer_cast<const TBillingSession>(sharedSession);
            if (sharedSessionImpl) {
                SharedSessionOffer = sharedSessionImpl->GetCurrentOffer();
            }
            if (sharedSession) {
                Session = MakeAtomicShared<TSharingSession>(std::move(Session), std::move(sharedSession));
            }
        }
    }

    TBillingSession::TBillingEventsCompilation eventsCompilation;
    if (!eventSession->FillCompilation(eventsCompilation)) {
        return false;
    }
    const auto [acceptanceStartedSnapshot, acceptanceFinishedSnapshot] = eventsCompilation.GetAcceptanceSnapshots();
    AcceptanceFinished = acceptanceFinishedSnapshot.OptionalDate();
    AcceptanceStarted = acceptanceStartedSnapshot.OptionalDate();

    auto offer = GetOffer();
    FuelingAbility = fetcher.GetNeedFueling(Session->GetObjectId());
    if (FuelingAbility && offer && !offer->HasFueling()) {
        FuelingAbility = false;
    }
    if (FuelingAbility && SharedSessionOffer && !SharedSessionOffer->HasFueling()) {
        FuelingAbility = false;
    }
    const auto& sessionId = Session->GetSessionId();
    if (fetchPhotos && objectId) {
        const auto& manager = Server->GetDriveDatabase().GetImageStorage();
        bool fetchPhotosBySessionId = Server->GetSettings().GetValue<bool>("session.fetch_photos_by_session_id").GetOrElse(false);
        auto since = Session->GetStartTS();
        auto until = Session->GetClosed() ? MakeMaybe(Session->GetLastTS()) : Nothing();
        auto optionalImages = fetchPhotosBySessionId
            ? manager.Fetch(session, NSQL::TQueryOptions()
                .AddGenericCondition("session_id", sessionId)
                .SetOrderBy({ "image_id" })
            )
            : manager.Get(objectId, { since, until }, session);
        if (!optionalImages) {
            return false;
        }
        for (auto&& image : *optionalImages) {
            if (image.GetSessionId() != sessionId) {
                continue;
            }
            Images.push_back(std::move(image));
        }
    }
    TankerFuelingAbility = fetcher.GetNeedTankerFueling(Session->GetObjectId());
    FavouriteAddressAdvisorPtr = favouriteAddressAdvisorPtr;
    if (TagsSnapshot && TagsSnapshot->GetTag(TReplaceCarTag::TypeName).Defined()) {
        CarReplaceAbility = true;
    }
    return true;
}

IEventsSession<TCarTagHistoryEvent>::TConstPtr TUserSessionsContext::TSessionContext::GetEventsSession() const {
    auto result = std::dynamic_pointer_cast<const IEventsSession<TCarTagHistoryEvent>>(Session);
    if (result) {
        return result;
    }

    auto sharingSession = std::dynamic_pointer_cast<const TSharingSession>(Session);
    result = sharingSession
        ? std::dynamic_pointer_cast<const IEventsSession<TCarTagHistoryEvent>>(sharingSession->GetReal())
        : nullptr;
    if (result) {
        return result;
    }

    auto sessionId = Session ? Session->GetSessionId() : "NullSession";
    R_ENSURE(result, HTTP_INTERNAL_SERVER_ERROR, "cannot GetEventsSession from session " << sessionId);
    return result;
}

NJson::TJsonValue TUserSessionsContext::TSessionContext::GetReportPoi(ELocalization locale) const {
    if (ContextClosed) {
        return NJson::JSON_NULL;
    }
    const NDrive::TLocationTags& poiLocationTags = TModelTag::GetPoiLocationTags(ModelTag, *Server);

    NDrive::TAreaIds areaIds;
    if (ShouldReportObject()) {
        areaIds = Snapshot.GetAreaIds();
    }

    auto offer = GetOffer();
    if (auto fixPointOffer = std::dynamic_pointer_cast<TFixPointOffer>(offer)) {
        const auto& destinationAreaIds = Server->GetDriveAPI()->GetAreaIds(fixPointOffer->GetFinish());
        areaIds.insert(destinationAreaIds.begin(), destinationAreaIds.end());
    }
    TVector<TArea> areasFetchResult;
    R_ENSURE(
        Server->GetDriveAPI()->GetAreasDB()->GetCustomObjectsFromCache(areasFetchResult, areaIds),
        HTTP_INTERNAL_SERVER_ERROR,
        "cannot GetCustomObjectsFromCache from AreasDB"
    );
    const TMaybe<TString> beaconParkingPlaceNumber = Snapshot.GetBeaconParkingPlaceNumber();
    return NDrive::GetPoiReport(areasFetchResult, poiLocationTags, beaconParkingPlaceNumber, locale, *Server);
}

NJson::TJsonValue TUserSessionsContext::TSessionContext::GetReport(
    ELocalization locale,
    const TUserCurrentContext* userCurrentContext,
    const TUserSessionsContext& userSessionsContext,
    bool needBill,
    NDriveSession::TReportTraits reportTraits,
    IRequestProcessor::EApp app
) const {
    auto userData = userCurrentContext ? userCurrentContext->GetUserData() : nullptr;
    auto userLocation = userCurrentContext ? userCurrentContext->GetUserLocationPtr() : nullptr;
    NJson::TJsonValue result;
    result.InsertValue("segment", GetReportSession(locale, needBill, reportTraits));
    result.InsertValue("delegation", GetReportDelegation());
    result.InsertValue("notification", GetReportNotification(locale));
    result.InsertValue("poi", GetReportPoi(locale));
    result.InsertValue("device_diff", GetReportObjectDiff(locale));
    result.InsertValue("feedback", GetReportFeedback(locale, userCurrentContext, needBill, reportTraits));
    result.InsertValue("transportation", GetReportTransportation(userLocation, app, locale));
    result.InsertValue("fueling_ability", GetFuelingAbility());
    result.InsertValue("tanker_fueling_ability", GetTankerFuelingAbility());
    result.InsertValue("car_replace_ability", GetCarReplaceAbility());
    result.InsertValue("additional_service_offers", GetAdditionalServiceOffersReport(locale, userCurrentContext, userSessionsContext));
    if (reportTraits & NDriveSession::ReportPaymentMethods) {
        result.InsertValue("payment_methods", GetReportPayments(locale, userCurrentContext));
    }
    if (userCurrentContext && (reportTraits & NDriveSession::ReportPhotos)) {
        result.InsertValue("photos", GetReportPhotos(userCurrentContext->GetImageHostBySourceMapping()));
    }
    if (IsServiceFuelingPossible(userCurrentContext)) {
        result.InsertValue("service_fueling", GetServiceFuelingReport(locale));
    }
    if (SharedSessionOffer) {
        result.InsertValue("shared_session_id", SharedSessionOffer->GetOfferId());
    }
    if (userData && userData->GetUserId() != Session->GetUserId()) {
        result.InsertValue("user", userData->GetPublicReport());
    }
    return result;
}

NJson::TJsonValue TUserSessionsContext::TSessionContext::GetReportDelegation() const {
    if (!DelegationTag) {
        return NJson::JSON_NULL;
    }
    return DelegationTag.GetTagAs<IDelegationTag>()->GetReport(DelegationTag, Server);
}

NJson::TJsonValue TUserSessionsContext::TSessionContext::GetReportSession(ELocalization locale, bool needBill, NDriveSession::TReportTraits reportTraits) const {
    auto api = Server ? Server->GetDriveAPI() : nullptr;
    auto carData = api ? api->GetCarsData() : nullptr;
    auto customization = MakeAtomicShared<TBillingReportCustomization>();
    customization->SetNeedBill(ContextClosed || needBill);
    customization->SetReportTraits(reportTraits);
    customization->OptionalPaymentsData() = PaymentsData;

    auto localization = Server ? Server->GetLocalization() : nullptr;

    auto segment = Session->GetReport(locale, Server, customization);
    if (needBill) {
        const TBillingSession* bSession = dynamic_cast<const TBillingSession*>(Session.Get());
        if (bSession && BillingCompilation) {
            TMultiBill mBill;
            if (GetCurrentMultiBill(mBill)) {
                segment.InsertValue("multi_bill", mBill.GetReport(locale, NDriveSession::ReportUserApp, *Server));
            }
        }
    }

    if (AcceptanceFinished) {
        segment.InsertValue("session_acceptance_finish", AcceptanceFinished->Seconds());
    }

    if (AcceptanceStarted) {
        segment.InsertValue("session_acceptance_start", AcceptanceStarted->Seconds());
    }

    auto last = GetEventsSession()->GetLastEvent();
    if (!!last && !ContextClosed) {
        const auto offerContainer = last->GetTagAs<IOfferContainer>();
        const auto& stageName = offerContainer ? offerContainer->GetStage(TagsSnapshot.Get()) : last.GetRef()->GetName();
        const auto& stageDescription = localization
            ? localization->GetLocalString(locale, "session." + stageName + ".description")
            : stageName;
        segment["session"]["current_performing"] = stageName;
        segment["session"]["current_stage_description"] = stageDescription;

        auto stageReport = (offerContainer && Server) ? offerContainer->GetStageReport(TagsSnapshot.Get(), *Server) : NJson::JSON_NULL;
        if (stageReport.IsDefined()) {
            segment["session"]["current_stage_info"] = std::move(stageReport);
        }
    }

    const auto& objectId = Session->GetObjectId();
    const auto object = carData ? carData->GetObject(objectId) : Nothing();
    if (object && ShouldReportObject()) {
        segment["car_number"] = object->GetNumber();
    }

    if (ServicingEnabled && !SharedSessionOffer) {
        segment["servicing"]["enabled"] = ServicingEnabled;
    }

    return segment;
}

NJson::TJsonValue TUserSessionsContext::TSessionContext::GetReportObjectDiff(ELocalization locale) const {
    TSnapshotsDiffCompilation snapshotsDiff;
    if (GetEventsSession()->FillCompilation(snapshotsDiff)) {
        return snapshotsDiff.GetReport(locale, Server, nullptr);
    }
    return NJson::JSON_NULL;
}

NJson::TJsonValue TUserSessionsContext::TSessionContext::GetReportFeedback(ELocalization locale, const TUserCurrentContext* userCurrentContext, bool needBill, NDriveSession::TReportTraits reportTraits) const {
    NJson::TJsonValue feedback(NJson::JSON_MAP);
    auto shortRideDuration = Server->GetSettings().GetValue<TDuration>("session.short_ride_duration_threshold").GetOrElse(TDuration::Minutes(3));
    bool shortRide = BillingCompilation ? BillingCompilation->GetRideDuration() < shortRideDuration : false;
    if (ContextClosed) {
        feedback.InsertValue("is_short_ride", shortRide);
    }

    bool successfulSession = ContextClosed && IsAccepted() && !shortRide;
    if (successfulSession) {
        {
            TFeedbackButtons buttons;

            if (userCurrentContext) {
                auto externalPromoButtons = userCurrentContext->GetExternalPromoContext().GetButtons(locale);
                auto areaButtons = userCurrentContext->GetAreaFeedbackButtons();
                buttons = std::move(externalPromoButtons);
                buttons.insert(buttons.end(), areaButtons.begin(), areaButtons.end());
            }

            std::sort(buttons.begin(), buttons.end(), [](const TFeedbackButton& left, const TFeedbackButton& right) {
                return std::tie(left.Priority, left.Title) > std::tie(right.Priority, right.Title);
            });
            feedback.InsertValue("buttons", NJson::ToJson(buttons));
        }
        if (reportTraits & NDriveSession::EReportTraits::ReportFavouriteAddressSuggest) {
            bool isLocationSuggested = false;

            if (FavouriteAddressAdvisorPtr) {
                FavouriteAddressAdvisorPtr->WaitFavouritesInitialized();

                NDrive::TLocation finishLocation;
                if (FavouriteAddressAdvisorPtr->IsSuggestAvailable() && Snapshot.GetLocation(finishLocation)) {
                    isLocationSuggested = FavouriteAddressAdvisorPtr->Suggest({ finishLocation.GetCoord(), Session->GetLastTS() });
                }
            }

            feedback.InsertValue("save_destination_prompt_required", isLocationSuggested);
        }
    }

    auto ratingEnabled = userCurrentContext ? userCurrentContext->GetSetting<bool>("session.feedback.app_rating_enabled", false) : false;
    if (ratingEnabled) {
        bool rated = false;
        auto rateParking = userCurrentContext->GetSetting<bool>("session.feedback.app_rating_parking", false);
        auto ratePost = userCurrentContext->GetSetting<bool>("session.feedback.app_rating_post", false);
        auto flags = userCurrentContext->GetFlagsTag() ? userCurrentContext->GetFlagsTag().GetTagAs<TUserDictionaryTag>() : nullptr;
        if (flags) {
            auto ratingStatus = flags->GetField("app_rating_status");
            rated = ratingStatus == "rated" || ratingStatus == "ignored";
        }
        bool required = false;
        if (rateParking) {
            auto currentTag = GetCurrentSessionTag();
            required = !rated && currentTag && currentTag->GetName() == "old_state_parking";
        }
        if (ratePost) {
            required = !rated && successfulSession && needBill;
        }
        if (required) {
            feedback.InsertValue("app_rating_required", required);
        }
    }

    auto offer = GetOffer();
    auto offerType = offer ? offer->GetTypeName() : "default";
    bool offerFeedbackEnabled = userCurrentContext ? userCurrentContext->GetSetting<bool>("offers." + offerType + ".feedback.enabled").GetOrElse(true) : true;
    bool sessionFeedbackEnabled = userCurrentContext ? userCurrentContext->GetSetting<bool>("session.feedback.enabled").GetOrElse(true) : true;
    bool feedbackEnabled = offerFeedbackEnabled && sessionFeedbackEnabled;
    TString setId;
    if (feedbackEnabled && ContextClosed && needBill) {
        if (AcceptanceFinished) {
            setId = shortRide ? "feedback_post" : "feedback_post_long";
        } else if (AcceptanceStarted) {
            setId = "feedback_acceptance";
        } else {
            setId = "feedback_reservation";
        }
    }
    if (feedbackEnabled && ContextClosed && !needBill) {
        setId = "feedback_history";
    }
    if (feedbackEnabled && AcceptanceStarted && !AcceptanceFinished && !ContextClosed) {
        setId = "feedback_accepting";
    }
    if (feedbackEnabled && AcceptanceFinished && !ContextClosed) {
        setId = "feedback_riding";
    }
    if (setId) {
        feedback.InsertValue("set_id", setId);
    }

    NJson::TJsonValue feedbackTags = NJson::JSON_NULL;
    for (auto&& tag : SessionTags) {
        auto feedbackTag = tag.GetTagAs<TFeedbackTraceTag>();
        if (feedbackTag) {
            feedbackTags.AppendValue(feedbackTag->GetName());
        }
    }
    if (feedbackTags.IsDefined()) {
        feedback.InsertValue("tags", std::move(feedbackTags));
    }

    return feedback;
}

NJson::TJsonValue TUserSessionsContext::TSessionContext::GetReportPayments(ELocalization locale, const TUserCurrentContext* userCurrentContext) const {
    NJson::TJsonValue payments(NJson::JSON_ARRAY);

    auto offer = GetOffer();
    if (userCurrentContext && offer) {
        payments = userCurrentContext->GetPaymentReport(locale, offer->GetTimestamp(), TSet<TString>({offer->GetSelectedCharge()}), offer->GetSelectedCreditCard());
    }
    return payments;
}

NJson::TJsonValue TUserSessionsContext::TSessionContext::GetReportPhotos(const THostByImageSourceMapping& hostBySourceMapping) const {
    NJson::TJsonValue result = NJson::JSON_ARRAY;
    for (auto&& image : Images) {
        result.AppendValue(image.BuildReport(hostBySourceMapping));
    }
    return result;
}

NJson::TJsonValue TUserSessionsContext::TSessionContext::GetReportTransportation(const TGeoCoord* userLocation, IRequestProcessor::EApp app, ELocalization locale) const {
    if (ContextClosed) {
        return NJson::JSON_NULL;
    }

    const auto& settings = Server->GetSettings();
    auto locationOptions = Server->GetSnapshotsManager().GetLocationOptions();
    auto carLocation = Snapshot.GetLocation(TDuration::Max(), locationOptions);
    auto beaconsLocation = Snapshot.GetBeaconsLocation(TDuration::Max(), locationOptions);
    double haversine = carLocation && userLocation ? userLocation->GetLengthTo(carLocation->GetCoord()) : 0;
    NJson::TJsonValue transportationJson = NJson::JSON_MAP;

    bool firstMileEnabled = settings.GetValueDef<bool>("transportation.first_mile.enabled", true);
    auto minimalHaversine = settings.GetValueDef<double>("transportation.first_mile.minimal_haversine", 1000);
    if (firstMileEnabled && carLocation && (haversine >= minimalHaversine || beaconsLocation) && !IsAccepted()) {
        TTransportationTag::TFirstMileReportOptions fmro;
        fmro.Locale = locale;
        fmro.Location = carLocation->GetCoord();
        fmro.App = app;
        fmro.WalkingDuration = TDuration::Seconds(haversine);
        fmro.SetBeaconsInfo(settings, Snapshot);
        bool geocodedEnabled = settings.GetValueDef<bool>("transportation.first_mile.geocoded_enabled", true);
        if (geocodedEnabled) {
            auto geocoded = Snapshot.GetGeocoded(TDuration::Max(), locationOptions);
            if (geocoded && geocoded->Content) {
                fmro.Geocoded = geocoded->Content;
            }
        }
        transportationJson.InsertValue("first_mile", TTransportationTag::GetFirstMileReport(fmro, *Server));
    }

    bool destinationEnabled = settings.GetValue<bool>("transportation.destination.enabled").GetOrElse(false);
    if (destinationEnabled && IsAccepted()) {
        auto offer = GetOffer();
        auto fixPointOffer = std::dynamic_pointer_cast<TFixPointOffer>(offer);
        if (fixPointOffer) {
            TTransportationTag::TFirstMileReportOptions reportOptions;
            reportOptions.Locale = locale;
            reportOptions.Location = fixPointOffer->GetFinish();
            reportOptions.App = app;
            reportOptions.Description = fixPointOffer->GetDestinationDescription();
            reportOptions.Geocoded = fixPointOffer->GetDestinationName();
            transportationJson.InsertValue("destination", TTransportationTag::GetFirstMileReport(reportOptions, *Server));
        }
    }
    return transportationJson;
}

NJson::TJsonValue TUserSessionsContext::TSessionContext::GetServiceFuelingReport(ELocalization locale) const {
    // TODO: replace with actual values when the are present
    auto offer = GetOffer();

    NJson::TJsonValue result;
    result["possible"] = offer ? offer->GetServiceFuelingPossibility(Server) : false;
    result["price"] = 42;
    result["price_hr"] = Server->GetLocalization()->FormatServiceFuelingCost(locale, 42);

    return result;
}

NJson::TJsonValue TUserSessionsContext::TSessionContext::GetReportNotification(ELocalization locale) const {
    if (ContextClosed || !IsAccepted()) {
        return NJson::JSON_NULL;
    }

    auto offer = GetOffer();
    TCarLocationContext context("", Snapshot);
    TCarLocationFeatures features = context.GetCarAreaFeatures(false, offer.Get(), Server);

    NJson::TJsonValue notification(NJson::JSON_NULL);
    TMaybe<TString> optionalText;
    const auto& settings = Server->GetSettings();
    switch (features.GetAllowDropDef(EDropAbility::NotAllow)) {
        case EDropAbility::Allow:
            optionalText = settings.GetValue<TString>("mobile.notifications.allow_drop_off");
            break;
        default:
            if (features.GetAllowRiding()) {
                optionalText = settings.GetValue<TString>("mobile.notifications.allow_parking");
            } else {
                optionalText = settings.GetValue<TString>("mobile.notifications.deny_riding");
            }
            break;
    }
    if (optionalText) {
        auto text = std::move(*optionalText);
        auto localization = Server->GetLocalization();
        if (localization) {
            text = localization->ApplyResources(text, locale);
        }
        notification = std::move(text);
    }
    return notification;
}

NJson::TJsonValue TUserSessionsContext::TSessionContext::GetAdditionalServiceOffersReport(ELocalization locale, const TUserCurrentContext* userCurrentContext, const TUserSessionsContext& userSessionsContext) const {
    // if (ContextClosed || !IsAccepted()) {
    //     return NJson::JSON_NULL;
    // }
    if (!userCurrentContext) {
        return NJson::JSON_ARRAY;
    }
    TSet<TString> disabledBuilders;
    for (const auto& session : userSessionsContext.GetAdditionalServiceSessions()) {
        auto compilation = session->GetCompilationAs<TAdditionalServiceSession::TCompilation>();
        if (!compilation) {
            continue;
        }
        auto serviceOffer = std::dynamic_pointer_cast<TAdditionalServiceOffer>(compilation->GetOffer());
        if (!serviceOffer) {
            continue;
        }
        // TODO: Replace this with offer builder field (requirements are unknown).
        //
        // Maybe in the future we should make more generic filter using offer builder field with concrete specification.
        auto builderName = serviceOffer->GetBehaviourConstructorId();
        if (Server->GetSettings().GetValue<bool>("session.disable_duplicate_offer." + builderName).GetOrElse(false)) {
            disabledBuilders.insert(builderName);
        }
    }
    const auto &permissions = userCurrentContext->GetPermissions();
    const auto &builders = permissions.GetOfferBuilders();
    //const auto &correctors = permissions.GetOfferCorrections();
    const auto &billingCompilation = *Yensured(GetBillingCompilation());
    const auto correctors = TUserActions();
    NDrive::TInfoEntitySession session;
    TOffersBuildingContext offersBuildingContext(TUserOfferContext(Server, permissions.Self()/*, Context*/));
    offersBuildingContext.SetBillingSessionId(billingCompilation.GetSessionId());
    TVector<IOfferReport::TPtr> offersAll;
    for (auto &builder : builders) {
        auto additionalServiceBuilder = std::dynamic_pointer_cast<const TAdditionalServiceOfferBuilder>(builder);
        if (!additionalServiceBuilder) {
            continue;
        }
        if (disabledBuilders.contains(additionalServiceBuilder->GetName())) {
            continue;
        }
        TVector<IOfferReport::TPtr> offers;
        additionalServiceBuilder->BuildOffers(permissions, correctors, offers, offersBuildingContext, Server, session);
        offersAll.insert(offersAll.end(), offers.begin(), offers.end());
    }

    NJson::TJsonValue offerReports = NJson::JSON_ARRAY;
    for (auto&& offerReport : offersAll) {
        auto offerReportJson = offerReport->BuildJsonReport(locale, NDriveSession::ReportAll, *Server, permissions);
        offerReports.AppendValue(std::move(offerReportJson));
    }

    return offerReports;
}

TMaybe<NDrive::TLocation> GetLastSessionLocation(const TSnapshotsDiffCompilation& snapshotsDiff, const TRTDeviceSnapshot& snapshot) {
    if (auto snapshot = snapshotsDiff.GetSnapshotFinish()) {
        if (auto deviceSnapshot = std::dynamic_pointer_cast<THistoryDeviceSnapshot>(snapshot)) {
            if (auto location = deviceSnapshot->GetHistoryLocation()) {
                return location;
            }
        }
    }
    NDrive::TLocation currentLocation;
    if (!snapshot.GetLocation(currentLocation)) {
        return {};
    }
    return currentLocation;
}

TVector<TUserSessionsContext::TSessionContext> TUserSessionsContext::FetchCurrentSessions(const TVector<TDBTag>& tags, const TString& userId, NDrive::TEntitySession& tx) const {
    TVector<TSessionContext> sessions;
    for (auto&& tag : tags) {
        auto optionalSession = Server->GetDriveDatabase().GetSessionManager().GetTagSession(tag.GetTagId(), tx);
        R_ENSURE(optionalSession, {}, "cannot GetTagSession", tx);
        auto session = *optionalSession;
        if (session && session->GetUserId() == userId) {
            sessions.emplace_back(session, Nothing(), Server);
        }
    }
    return sessions;
}

NJson::TJsonValue TUserSessionsContext::GetDelegationsReport() const {
    NJson::TJsonValue result = NJson::JSON_MAP;
    NJson::TJsonValue& inProgressJson = result.InsertValue("actual", NJson::JSON_ARRAY);
    NJson::TJsonValue& finishedJson = result.InsertValue("finished", NJson::JSON_ARRAY);
    for (auto&& i : DelegationsInProgress) {
        const IDelegationTag* delegationTagImpl = i.GetTagAs<IDelegationTag>();
        inProgressJson.AppendValue(delegationTagImpl->GetReport(i, Server));
    }
    for (auto&& i : DelegationsFinished) {
        const TDelegationUserTag* delegationTagImpl = i.GetTagAs<TDelegationUserTag>();
        finishedJson.AppendValue(delegationTagImpl->GetReport(i, Server));
    }
    return result;
}

NJson::TJsonValue TUserSessionsContext::GetFuturesReport(ELocalization locale, const TUserCurrentContext* userCurrentContext) const {
    ICommonOffer::TReportOptions reportOptions(locale, NDriveSession::ReportUserApp);
    NJson::TJsonValue result = NJson::JSON_ARRAY;
    for (auto&& i : Futures) {
        const TTagReservationFutures* futuresTagImpl = i.GetTagAs<TTagReservationFutures>();
        NJson::TJsonValue& futureInfoJson = result.AppendValue(NJson::JSON_MAP);
        futureInfoJson.InsertValue("car_id", i.GetObjectId());
        futureInfoJson.InsertValue("futures_offer", futuresTagImpl->GetOffer()->BuildJsonReport(reportOptions, *Server));
        futureInfoJson.InsertValue("futures_offer_failed", futuresTagImpl->IsFailed());
        if (futuresTagImpl->IsFailed()) {
            futureInfoJson.InsertValue("futures_offer_failed_message", futuresTagImpl->GetErrorMessage());
        }
        futureInfoJson.InsertValue("futures_offer_tag_id", i.GetTagId());
        if (userCurrentContext) {
            futureInfoJson.InsertValue("payment_methods", userCurrentContext->GetPaymentReport(locale, futuresTagImpl->GetOffer()->GetTimestamp(), TSet<TString>({futuresTagImpl->GetOffer()->GetSelectedCharge()}), futuresTagImpl->GetOffer()->GetSelectedCreditCard()));
        }
    }
    if (userCurrentContext) {
        for (auto&& i : userCurrentContext->GetFutures()) {
            const TTagReservationFutures* futuresTagImpl = i.GetTagAs<TTagReservationFutures>();
            auto vehicleOffer = std::dynamic_pointer_cast<IOffer>(futuresTagImpl->GetOffer());
            if (!vehicleOffer) {
                continue;
            }
            NJson::TJsonValue& futureInfoJson = result.AppendValue(NJson::JSON_MAP);
            futureInfoJson.InsertValue("car_id", vehicleOffer->GetObjectId());
            futureInfoJson.InsertValue("futures_offer", futuresTagImpl->GetOffer()->BuildJsonReport(reportOptions, *Server));
            futureInfoJson.InsertValue("futures_offer_failed", futuresTagImpl->IsFailed());
            if (futuresTagImpl->IsFailed()) {
                futureInfoJson.InsertValue("futures_offer_failed_message", futuresTagImpl->GetErrorMessage());
            }
            futureInfoJson.InsertValue("futures_offer_tag_id", i.GetTagId());
            futureInfoJson.InsertValue("payment_methods", userCurrentContext->GetPaymentReport(locale, futuresTagImpl->GetOffer()->GetTimestamp(), TSet<TString>({futuresTagImpl->GetOffer()->GetSelectedCharge()}), futuresTagImpl->GetOffer()->GetSelectedCreditCard()));
        }
    }
    return result;
}

NJson::TJsonValue TUserSessionsContext::GetSessionsReport(ELocalization locale, bool needBill, NDriveSession::TReportTraits reportTraits, IRequestProcessor::EApp app, const TUserCurrentContext* userCurrentContext) const {
    NJson::TJsonValue result = NJson::JSON_ARRAY;
    for (auto&& i : Sessions) {
        result.AppendValue(i.GetReport(locale, userCurrentContext, *this, needBill, reportTraits, app));
    }
    return result;
}

NJson::TJsonValue TUserSessionsContext::GetSharedSessionsReport(ELocalization locale, bool needBill, NDriveSession::TReportTraits reportTraits, IRequestProcessor::EApp app, const TUserCurrentContext* userCurrentContext) const {
    NJson::TJsonValue result = NJson::JSON_ARRAY;
    for (auto&& i : SharedSessions) {
        auto traits = reportTraits;
        traits &= ~NDriveSession::ReportPaymentMethods;
        result.AppendValue(i.GetReport(locale, userCurrentContext, *this, needBill, traits, app));
    }
    return result;
}

NJson::TJsonValue TUserSessionsContext::GetAdditionalServiceSessionsReport(ELocalization locale, bool needBill, NDriveSession::TReportTraits reportTraits, IRequestProcessor::EApp app, const TUserCurrentContext* userCurrentContext) const {
    Y_UNUSED(needBill);
    Y_UNUSED(reportTraits);
    Y_UNUSED(app);
    Y_UNUSED(userCurrentContext);
    NJson::TJsonValue result = NJson::JSON_ARRAY;
    for (auto&& i : AdditionalServiceSessions) {
        result.AppendValue(Yensured(i)->GetReport(locale, Server, {}));
    }
    return result;
}

NJson::TJsonValue TUserSessionsContext::GetStatesReport(ELocalization locale, const TUserCurrentContext* userCurrentContext) const {
    NJson::TJsonValue result = NJson::JSON_ARRAY;
    auto add = [&](auto&& states) {
        for (auto&& tag : states) {
            auto state = tag.template GetTagAs<NDrive::IUserState>();
            result.AppendValue(Yensured(state)->GetStateReport(locale, tag, *Permissions, *Server));
        }
    };
    {
        add(States);
    }
    if (userCurrentContext) {
        add(userCurrentContext->GetStates());
    }
    return result;
}

bool TUserSessionsContext::TSessionContext::IsServiceFuelingPossible(const TUserCurrentContext* userCurrentContext) const {
    auto featureFlagName = userCurrentContext
        ? userCurrentContext->GetSetting<TString>("delegation.service.fueling.feature_flag_name")
        : Nothing();
    if (!featureFlagName) {
        return false;
    }

    bool fuelingRolledOut = userCurrentContext->GetPermissions().HasAction(*featureFlagName);
    if (!fuelingRolledOut) {
        return false;
    }

    auto offer = GetOffer();
    return offer ? offer->GetServiceFuelingPossibility(Server) : false;
}

TString TUserSessionsContext::GetCarsReport(const TSet<TString>& supportAllowedPhotos) const {
    TStringStream ss;
    {
        NJson::TJsonWriter jWriter(&ss, false);
        jWriter.OpenArray();
        const TCarsFetcher* fetcher = nullptr;
        const TSet<TString>* objectIds = nullptr;
        if (!CurrentFetcher.Empty()) {
            fetcher = &CurrentFetcher;
            objectIds = &CurrentObjectIds;
        } else if (!HistoryFetcher.Empty()) {
            fetcher = &HistoryFetcher;
            objectIds = &HistoryObjectIds;
        }
        if (!!fetcher) {
            TSet<TString> excludeCarIds;
            for (auto&& i : Sessions) {
                if (!i.ShouldReportObject()) {
                    continue;
                }
                i.BuildCarReport(jWriter, *fetcher, Permissions, supportAllowedPhotos);
                excludeCarIds.insert(i->GetObjectId());
            }
            for (auto&& i : SharedSessions) {
                if (!i.ShouldReportObject()) {
                    continue;
                }
                i.BuildCarReport(jWriter, *fetcher, Permissions, supportAllowedPhotos);
                excludeCarIds.insert(i->GetObjectId());
            }
            for (auto&& i : Futures) {
                TSessionContext::BuildCarReport(jWriter, i.GetTagAs<TTagReservationFutures>()->IsFailed(), i.GetObjectId(), false, false, *fetcher, Permissions, supportAllowedPhotos);
                excludeCarIds.insert(i.GetObjectId());
            }
            for (auto&& i : DelegationsInProgress) {
                if (!excludeCarIds.emplace(i.GetObjectId()).second) {
                    continue;
                }
                TSessionContext::BuildCarReport(jWriter, false, i.GetObjectId(), false, false, *fetcher, Permissions, supportAllowedPhotos);
            }
            Y_ENSURE(objectIds);
            for (auto&& objectId : *objectIds) {
                if (!excludeCarIds.insert(objectId).second) {
                    continue;
                }
                TSessionContext::BuildCarReport(jWriter, false, objectId, false, false, *fetcher, Permissions, supportAllowedPhotos);
            }
        }
        jWriter.CloseArray();
    }
    return ss.Str();
}

NJson::TJsonValue TUserSessionsContext::GetCarsMetaReport() const {
    NJson::TJsonValue result = NJson::JSON_MAP;
    const TCarsFetcher* fetcher = nullptr;
    if (!CurrentFetcher.Empty()) {
        fetcher = &CurrentFetcher;
    } else if (!HistoryFetcher.Empty()) {
        fetcher = &HistoryFetcher;
    }
    if (!!fetcher) {
        result = fetcher->BuildReportMeta();
    }
    return result;
}

ui64 TUserSessionsContext::GetSessionCount() const {
    return Futures.size() + Sessions.size();
}

bool TUserSessionsContext::IsExtraSessionsAvailable() const {
    if (!Permissions) {
        return false;
    }
    auto lockedLimit = Permissions->LockedResourcesLimit();
    auto takenLimit = Futures.size() + Sessions.size();
    return takenLimit < lockedLimit;
}

bool TUserSessionsContext::Initialize(NDrive::TEntitySession& session, TJsonReport& report, const TInstant reqActuality, const bool needClosedSession, const TString& sessionId, const TUserCurrentContext* userCurrentContext) {
    const auto api = Yensured(Server)->GetDriveAPI();
    if (FavouriteAddressAdvisorPtr) {
        auto eg = report.BuildEventGuard("InitializeFavourites");
        FavouriteAddressAdvisorPtr->InitializeFavourites(Permissions->GetPassportUid(), reqActuality);
    }

    const auto& tagsManager = Server->GetDriveDatabase().GetTagsManager();
    TVector<TDBTag> actualTags;
    {
        TEventsGuard eg(report, "RestorePerformerTags");
        if (!Yensured(api)->GetTagsManager().GetDeviceTags().RestorePerformerTags({Permissions->GetUserId()}, actualTags, session)) {
            return false;
        }
    }
    TVector<TDBTag> sessionTags;
    TVector<TString> sharedSessionIds;
    for (auto&& i : actualTags) {
        if (i.GetTagAs<TTagReservationFutures>() && !sessionId) {
            if (!i.GetTagAs<TTagReservationFutures>()->IsFailed()) {
                CurrentObjectIds.emplace(i.GetObjectId());
            }
            Futures.emplace_back(i);
        }
        if ((i.GetTagAs<TChargableTag>() || i.GetTagAs<TTransformationTag>())) {
            sessionTags.emplace_back(i);
        }
    }

    TVector<TDBTag> performedUserTags;
    {
        TEventsGuard eg(report, "RestorePerformedUserTags");
        if (!tagsManager.GetUserTags().RestorePerformerTags({Permissions->GetUserId()}, performedUserTags, session)) {
            return false;
        }
    }
    for (auto&& i : performedUserTags) {
        if (i.Is<NDrive::IUserState>()) {
            States.push_back(i);
        }
        auto sessionSharingTag = i.GetTagAs<TSessionSharingTag>();
        if (sessionSharingTag && sessionSharingTag->GetUserId() == Permissions->GetUserId()) {
            sharedSessionIds.push_back(sessionSharingTag->GetSessionId());
        }
    }
    auto parseStates = [&](const TDBTags& states) {
        for (auto&& state : states) {
            auto genericUserState = state.GetTagAs<TGenericUserStateTag>();
            if (genericUserState) {
                const auto& objectId = genericUserState->GetCarId();
                if (objectId) {
                    CurrentObjectIds.insert(objectId);
                }
            }
        }
    };
    {
        parseStates(States);
    }
    if (userCurrentContext) {
        parseStates(userCurrentContext->GetStates());
    }

    {
        TEventsGuard eg(report, "FetchCurrentSessions");
        Sessions = FetchCurrentSessions(sessionTags, Permissions->GetUserId(), session);
    }
    if (userCurrentContext) {
        TEventsGuard eg(report, "FetchOfferSessions");
        for (auto&& tag : userCurrentContext->GetOfferHolders()) {
            auto offerHolderSession = CreateOfferHolderSession(
                tag,
                TOfferHolderSessionOptions(),
                Yensured(api)->GetTagsManager().GetUserTags(),
                session
            );
            if (!offerHolderSession) {
                report.AddEvent(NJson::TMapBuilder
                    ("event", "CreateOfferHolderSessionError")
                    ("tag_id", tag.GetTagId())
                );
                return false;
            }
            Sessions.emplace_back(std::move(offerHolderSession), TMaybe<TPaymentsData>(), Server);
            Sessions.back().SetReportObject(false);
        }
    }
    {
        TEventsGuard eg(report, "FetchAdditionalOfferSessions");
        auto sessions = api->GetAdditionalServiceSessionManager().GetUserSessions(Permissions->GetUserId(), session);
        if (!sessions) {
            session.SetErrorInfo("TUserSessionsContext::Initialize", "can not get additional service sessions for user " + Permissions->GetUserId());
        } else {
            AdditionalServiceSessions = *sessions;
        }
    }
    {
        TEventsGuard eg(report, "FetchSharedSessions");
        for (auto&& sharedSessionId : sharedSessionIds) {
            auto optionalSharedSession = api->GetSessionManager().GetSession(sharedSessionId, session);
            if (!optionalSharedSession) {
                return false;
            }
            auto sharedSession = *optionalSharedSession;
            if (!sharedSession) {
                eg.AddEvent(NJson::TMapBuilder
                    ("event", "BadSharedSessionId")
                    ("session_id", sharedSessionId)
                );
            }
            if (sharedSession->GetClosed()) {
                continue;
            }
            SharedSessions.emplace_back(std::move(sharedSession), Nothing(), Server);
        }
    }

    TSet<TString> delegationObjectIds;
    TSet<TString> delegationHistoryObjectIds;
    TMap<TString, TDBTag> delegationCarTags;
    if (userCurrentContext) {
        delegationObjectIds = userCurrentContext->GetDelegationObjectIds();
        TVector<TDBTag> delegationDBTags;
        TVector<TString> delegationTagNames = {
            TFreeDelegationTag::TypeName,
            TP2PDelegationTag::TypeName
        };
        if (delegationObjectIds && !Server->GetDriveAPI()->GetTagsManager().GetDeviceTags().RestoreTags(delegationObjectIds, delegationTagNames, delegationDBTags, session)) {
            return false;
        }
        for (auto&& tag : delegationDBTags) {
            delegationCarTags.emplace(tag.GetObjectId(), tag);
        }
        for (auto&& d : userCurrentContext->GetDelegations()) {
            const TString& objectId = d.GetTagAs<TDelegationUserTag>()->GetObjectId();
            if (delegationObjectIds.contains(objectId)) {
                auto delegationTagIt = delegationCarTags.find(objectId);
                if (delegationTagIt == delegationCarTags.end()) {
                    delegationHistoryObjectIds.emplace(objectId);
                    DelegationsFinished.emplace_back(d);
                    continue;
                }
                const auto& delegationTag = delegationTagIt->second;
                if (!delegationTag || !delegationTag.GetTagAs<IDelegationTag>() || delegationTag.GetTagAs<IDelegationTag>()->GetBaseUserId() != Permissions->GetUserId()) {
                    delegationHistoryObjectIds.emplace(objectId);
                    DelegationsFinished.emplace_back(d);
                    continue;
                }
                DelegationsInProgress.emplace_back(delegationTag);
            } else {
                delegationHistoryObjectIds.emplace(objectId);
                DelegationsFinished.emplace_back(d);
            }
        }
    }

    if (!!sessionId) {
        const auto predRemove = [&sessionId](const TSessionContext& context) {
            return context->GetSessionId() != sessionId;
        };
        Sessions.erase(std::remove_if(Sessions.begin(), Sessions.end(), predRemove), Sessions.end());
        if (Sessions.empty()) {
            auto eg = report.BuildEventGuard("GetSessionById");
            auto optionalSession = Server->GetDriveDatabase().GetSessionManager().GetSession(sessionId, session);
            if (!optionalSession) {
                return false;
            }
            auto billingSession = *optionalSession;
            if (billingSession) {
                auto payments = Server->GetDriveAPI()->GetBillingManager().GetPaymentsManager().GetFinishedPayments(NContainer::Scalar(sessionId), session);
                if (!payments) {
                    return false;
                }
                TMaybe<TPaymentsData> finishedPayments;
                if (payments->size()) {
                    finishedPayments = payments->begin()->second;
                }
                Sessions.emplace_back(billingSession, std::move(finishedPayments), Server);
            }
        }
    } else if (Sessions.empty() && needClosedSession) {
        auto eg = report.BuildEventGuard("GetSessionByUserId");
        auto userSessionDepth = Server->GetSettings().GetValue<TDuration>("session.user_sessions.depth").GetOrElse(TDuration::Seconds(100));
        auto optionalUserSessions = Server->GetDriveDatabase().GetSessionManager().GetUserSessions(Permissions->GetUserId(), session, userSessionDepth);
        if (!optionalUserSessions) {
            return false;
        }
        auto billingSession = !optionalUserSessions->empty() ? optionalUserSessions->back() : nullptr;
        if (billingSession) {
            auto payments = Server->GetDriveAPI()->GetBillingManager().GetPaymentsManager().GetFinishedPayments(NContainer::Scalar(billingSession->GetSessionId()), session);
            if (!payments) {
                return false;
            }
            Sessions.emplace_back(billingSession, payments->size() ? std::move(payments->begin()->second) : TMaybe<TPaymentsData>(), Server);
        }
    }

    for (auto&& i : Sessions) {
        auto offer = i.GetOffer();
        if (i.GetContextClosed()) {
            if (i.ShouldReportObject()) {
                HistoryObjectIds.emplace(i->GetObjectId());
            }
            if (offer && offer->GetObjectModel()) {
                HistoryObjectModels.insert(offer->GetObjectModel());
            }
        } else {
            if (i.ShouldReportObject()) {
                CurrentObjectIds.emplace(i->GetObjectId());
            }
            if (offer && offer->GetObjectModel()) {
                CurrentObjectModels.insert(offer->GetObjectModel());
            }
        }
    }
    for (auto&& i : SharedSessions) {
        CurrentObjectIds.insert(i->GetObjectId());
    }

    CurrentObjectIds.insert(delegationObjectIds.begin(), delegationObjectIds.end());
    if (CurrentObjectIds.size() || CurrentObjectModels.size()) {
        CurrentFetcher.SetCheckVisibility(false);
        CurrentFetcher.SetIsRealtime(true);
        TSet<NDrive::TSensorId> sensorIds = {
            CAN_FUEL_LEVEL_P,
            CAN_FUEL_DISTANCE_KM,
            CAN_DRIVER_DOOR,
            CAN_PASS_DOOR,
            CAN_L_REAR_DOOR,
            CAN_R_REAR_DOOR,
            CAN_HOOD,
            CAN_TRUNK,
            CAN_ENGINE_IS_ON,
            NDrive::NVega::BlePasskey,
            NDrive::NVega::BleSessionKey,
            NDrive::NVega::SvrRawState,
            BLE_EXT_BOARD_MAC,
        };
        TSet<NDrive::TSensorId> extraSensorIds;
        if (NJson::TryFromJson(Server->GetSettings().GetJsonValue("session.telematics.extra_sensors"), extraSensorIds)) {
            sensorIds.insert(extraSensorIds.begin(), extraSensorIds.end());
        }
        CurrentFetcher.SetSensorIds(sensorIds);
        CurrentFetcher.SetSelectedModels(CurrentObjectModels);
        CurrentFetcher.SetIsParkingCorrectFuturesState(true);
        TEventsGuard eg(report, "FetchDataCurrent");
        bool fetched = CurrentFetcher.FetchData(Permissions, CurrentObjectIds);
        if (!fetched) {
            report.AddEvent("CurrentFetchDataError");
        }
    }
    if (HistoryObjectIds.size() || HistoryObjectModels.size()) {
        HistoryFetcher.SetIsRealtime(false);
        HistoryFetcher.SetIsParkingCorrectFuturesState(false);
        HistoryFetcher.SetSensorIds({});
        HistoryFetcher.SetSelectedModels(CurrentObjectModels);
        TEventsGuard eg(report, "FetchDataHistory");
        bool fetched = HistoryFetcher.FetchData(Permissions, HistoryObjectIds);
        if (!fetched) {
            report.AddEvent("HistoryFetchDataError");
        }
    }

    bool fetchPhotos = userCurrentContext ? (userCurrentContext->GetReportTraits() & NDriveSession::ReportPhotos) : false;
    for (auto&& i : Sessions) {
        TEventsGuard eg(report, "Sessions:" + i->GetSessionId());
        if (!i.Initialize(session, i.GetContextClosed() ? HistoryFetcher : CurrentFetcher, delegationCarTags, FavouriteAddressAdvisorPtr, fetchPhotos)) {
            return false;
        }
    }
    for (auto&& i : SharedSessions) {
        TEventsGuard eg(report, "SharedSession:" + i->GetSessionId());
        if (!i.Initialize(session, CurrentFetcher, delegationCarTags, FavouriteAddressAdvisorPtr, fetchPhotos)) {
            return false;
        }
    }

    if (FavouriteAddressAdvisorPtr) {
        auto eg = report.BuildEventGuard("InitializeHistoryFinishCoords");
        FavouriteAddressAdvisorPtr->InitializeHistoryFinishCoords(Permissions->GetUserId(), session);
    }

    return true;
}
